/* ==================================================== ======== ======= *
 *
 *  uustr.cpp
 *  Ubit Project  [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uustr.cpp	ubit:03.06.04"
#include <cstring>
#include <iostream>
#include <fstream>
#include <stdio.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <uprop.hpp>
#include <ugroup.hpp>
#include <ucontext.hpp>
#include <ugraph.hpp>
#include <uctrl.hpp>
#include <uview.hpp>
#include <ustr.hpp>
#include <uappli.hpp>
#include <uflow.hpp>
#include <uevent.hpp>
#include <ucolor.hpp>
#include <update.hpp>
#include <unumber.hpp>
#include <uedit.hpp>
#include <utextsel.hpp>
using namespace std;

UStr UStr::none((char*)null, UMode::UCONST);
UStr UStr::newline("\n", UMode::UCONST);
//UStr UStr::goBOL("\15", UMode::UCONST); // retour a la ligne si necessaire

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStr::UStr(const char* chs, u_modes m) : UElem(m) {
  s = null; //!dont forget: checked by _set() for freeing memory!
  //  model = null;
  setImpl(chs, (chs ? strlen(chs) : 0), false);
}

UStr::UStr() {
  s = null; //!dont forget: checked by _set() for freeing memory!
  //  model = null;
  setImpl(null, 0, false);
}

UStr::UStr(const char* chs) {
  s = null; //!dont forget: checked by _set() for freeing memory!
  //  model = null;
  setImpl(chs, (chs ? strlen(chs) : 0), false);
}

UStr::UStr(const string& str) {
  s = null; //!!
  setImpl(str.c_str(), str.length(), false);  //pas optimal mais tant pis!
}

UStr::UStr(const UStr& str) {
  s = null; //!!
  //model = null;
  setImpl(str.s, str.len, false);
}

UStr::UStr(const class UIntgBase& _n) {
  s = null; //!!
  //model = null;
  char _s[50] = {0}; 
  sprintf(_s, "%d", _n.getValue());
  setImpl(_s, strlen(_s), false);
}

UStr::UStr(const class UFloatBase& _n) {
  s = null; //!!
  //model = null;
  char _s[50] = {0}; 
  sprintf(_s, "%f", _n.getValue());
  setImpl(_s, strlen(_s), false);
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

UStr::~UStr() {
  destructs();     // necessaire car removingFrom specifique
  if (s) free(s);
  //model = null;
  s = null; len = 0;
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

void UStr::addingTo(ULink *selflink, UGroup *parent) {
  UElem::addingTo(selflink, parent);

  if (parent->isDef(0, UMode::CAN_EDIT_TEXT)) {
    UEdit* ed = static_cast<UEdit*>
      (parent->getAnyChild( &UBrick::isInstance<UEdit> ));

    if (ed && !ed->getCaretStr()) ed->setCaretStr(this, 0);
  }
}

void UStr::removingFrom(ULink *selflink, UGroup *parent) {
  if (parent->isDef(0, UMode::CAN_EDIT_TEXT)) {
    UEdit* ed = static_cast<UEdit*>
      (parent->getAnyChild( &UBrick::isInstance<UEdit> ));

    if (ed && ed->getCaretStr() == this) ed->setCaretStr(null, 0);
  }
UElem::removingFrom(selflink, parent);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */
// NB: duplicates the char string (!ATT: length MUST be the EXACT length)

bool UStr::setImpl(const char *_s, int _len, bool call_callbacks) {
  if (s) free(s);
  if (!_s) syncVals(null, 0);
  else {
    s = (char*)malloc((_len + 1) * sizeof(char));
    //if (s) {strcpy(s, _s); syncVals(s, _len);}
    if (s) {strncpy(s, _s, _len); s[_len] = 0; syncVals(s, _len);}
    else {
      syncVals(null, 0);
      error("setImpl", UError::No_more_memory);
      return false;
    }
  }

  if (call_callbacks) changed(true);
  return true;
}

/* ==================================================== ======== ======= */

void UStr::clear() {
  if (!s) return;
  setImpl((char*)null, 0, true);
}

void UStr::set(const char *_s) {
  if (equals(_s)) return;
  setImpl(_s, (_s ? strlen(_s) : 0), true);
}

void UStr::set(const UStr& _s) {
  if (equals(_s)) return;
  setImpl(_s.s, _s.len, true);
}

void UStr::set(const string& _s) {
  set(_s.c_str());
}

/* ==================================================== ======== ======= */

void UStr::set(const class UIntgBase& _n) {
  setNum(_n.getValue());
}

void UStr::set(const class UFloatBase& _n) {
  setNum(_n.getValue());
}

void UStr::setNum(int _n) {
  char _s[50] = {0}; 
  sprintf(_s, "%d", _n);
  set(_s); //cf autoupdate
}

void UStr::setNum(double _n) {
  char _s[50] = {0}; 
  sprintf(_s, "%f", _n);
  set(_s); //cf autoupdate
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

ostream& operator<<(ostream& os, const UStr& s) {
  if (s.s) return (os << s.s);
  else return os;
}

//istream& operator>>(istream& is, UStr& s) {
// ...
//  return is;
//}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UStr::equals(const char* _s) const  {
  return (_s == s || (_s && s && strcmp(_s, s)==0));
}

bool UStr::equals(const UStr& str) const {
  return equals(str.s);
}

bool UStr::equals(const string& str) const {
  return equals(str.c_str());
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int UStr::compare(const char *str) const {
  if (!s) {
    if (!str) return 0;
    else return -1;
  }
  else if (!str) return 1;
  else return strcmp(s, str);
}

int UStr::compare(const UStr &str) const {
  return compare(str.s);
}

int UStr::compare(const string& str) const {
  return compare(str.c_str());
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

int UStr::compareTo(const char *str, bool ignore_case) const {
  if (!s) {
    if (!str) return 0;
    else return -1;
  }
  else if (!str) return 1;
  else if (ignore_case) return strcasecmp(s, str);
  else return strcmp(s, str);
}

int UStr::compareTo(const UStr &str, bool ignore_case) const {
  return compareTo(str.s, ignore_case);
}

int UStr::compareTo(const string& str, bool ignore_case) const {
  return compareTo(str.c_str(), ignore_case);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

bool UStr::checkFormat(int pos, int _c) {
  if (_c == '\0') return false;      //don't insert the NUL char!
  //else if (model) return model->checkFormat(pos, _c);
  else return true;
}

bool UStr::checkFormat(int pos, const char *_s) {
  if (_s == null) return false;      //don't insert the NULL string!
  //else if (model) return model->checkFormat(pos, _s);
  else return true;
}

void UStr::syncVals(char *news, int newlen) {
  s = news;
  len = newlen;
  /*
  if (model) model->set(this);
  void UStr::setStrModel(class UEditable *m) {
  model = m;
  chainage et cas null et enlever existant;
  */
}

void UStr::update() {
  // size changed horizontally ??          // !!!CFCFCF
  // parents.updateParents(UUpdate::layout);

  // mis en doublebuf pour eviter flicking. pbm c'est trop lent ?
  UUpdate upd(UUpdate::ALL);
  upd.paintDoubleBuffered();
  parents.updateParents(upd);
}

// calls UOn::change callbacks of this object when its value is changed
// then UOn::changeElem AND UOn::changeStr callbacks of its parents

void UStr::changed(bool update_now) {
  if (update_now && (bmodes & UMode::NO_AUTO_UPDATE) == 0)
    update();
  
  // faux: parent pas notifie si pas de CB sur ustr !!!
  //  if (cache) {  !!!

  UEvent e(UEvent::change,  null, null, NULL);
  e.setAux(this);

  if (cache) fire(e, UOn::change);

  // ensuite on lance les callbacks des parents
  parents.fireParents(e, UOn::elemChange);
  // NE PAS OUBLIER changeStr !!!
  parents.fireParents(e, UOn::strChange);
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

char UStr::charAt(int pos) const {
  if (!s || pos >= len) return 0;
  if (pos < 0) pos = len-1; // last char
  return s[pos];
}
 
char UStr::setCharAt(int pos, char newchar, bool upd) {
  if (!s || pos >= len) return 0;
  if (!checkFormat(pos, newchar)) return false; // !!

  if (pos < 0) pos = len-1; // last char
  s[pos] = newchar;

  syncVals(s, len);
  changed(upd);
  return true; 
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

UStr operator&(const UStr& s1, const UStr& s2) {
  UStr str = s1; str.append(s2); return str;
}

UStr operator&(const UStr& s1, const char* s2) {
  UStr str = s1; str.append(s2); return str;
}

UStr operator&(const char* s1, const UStr& s2) {
  UStr str = s1; str.append(s2); return str;
}

UStr operator&(const UStr& s1, const string& s2) {
  UStr str = s1; str.append(s2); return str;
}

UStr operator&(const string& s1, const UStr& s2) {
  UStr str = s1; str.append(s2); return str;
}

UStr operator&(const UStr& s1, int v2) {
  UStr str = s1; str.append(v2); return str;
}

UStr operator&(const UStr& s1, float v2) {
  UStr str = s1; str.append(v2); return str;
}

UStr operator&(const UStr& s1, const UIntgBase& v2) {
  UStr str = s1; str.append(v2); return str;
}

UStr operator&(const UStr& s1, const UFloatBase& v2) {
  UStr str = s1; str.append(v2); return str;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UStr::append(const char c) {
  insertImpl(-1, c, true);
}

void UStr::append(const char *s2) {
  insertImpl(-1, s2, 0, npos, true);
}

void UStr::append(const char *s2, unsigned int nbchars) {
  insertImpl(-1, s2, 0, nbchars, true);
}

void UStr::append(const UStr &s2) {
  insertImpl(-1, s2.s, 0, npos, true);
}

void UStr::append(const UStr &s2, unsigned int nbchars) {
  insertImpl(-1, s2.s, 0, nbchars, true);
}

void UStr::append(const string& s2) {
  insertImpl(-1, s2.c_str(), 0, npos, true);
}

void UStr::append(const string& s2, unsigned int nbchars) {
  insertImpl(-1, s2.c_str(), 0, nbchars, true);
}

void UStr::append(int v2) {
  char _s2[50] = {0}; 
  sprintf(_s2, "%d", v2);
  insertImpl(-1, _s2, 0, npos, true);
}

void UStr::append(float v2) {
  char _s2[50] = {0}; 
  sprintf(_s2, "%f", v2);
  insertImpl(-1, _s2, 0, npos, true);
}

void UStr::append(const UIntgBase& v2) {
  append(v2.getValue());
}

void UStr::append(const UFloatBase& v2) {
  append(v2.getValue());
}

// - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

bool UStr::insert(int to_pos, char c) {
  return insertImpl(to_pos, c, true);
}

bool UStr::insert(int to_pos, const UStr &s2, int from_pos,
                  unsigned int from_nbchars) {
  return insertImpl(to_pos, s2.s, from_pos, from_nbchars, true);
}

bool UStr::insert(int to_pos, const char *s2, int from_pos,
                  unsigned int from_nbchars) {
  return insertImpl(to_pos, s2, from_pos, from_nbchars, true);
}

bool UStr::insert(int to_pos, const std::string& s2, int from_pos,
                  unsigned int from_nbchars) {
  return insertImpl(to_pos, s2.c_str(), from_pos, from_nbchars, true);
}

/* ==================================================== ======== ======= */
// from_nbchars < 0 means whole string s2
// return : true if changed

bool UStr::insertImpl(int to_pos, const char *s2, int from_pos,
                      unsigned int from_nbchars, bool upd) {
  
  if (!checkFormat(to_pos, s2) || to_pos > len) return false; // !!

  if (to_pos < 0) to_pos = len; // append

  int len_s2 = strlen(s2);
  if (from_pos < 0) from_pos = len_s2 -1;
  
  int nbc = (from_nbchars == npos) ? len_s2 : from_nbchars;
  if (from_pos + nbc > len_s2) nbc = len_s2 - from_pos;
  if (nbc <= 0) return false;

  char* news = (char*) malloc(len + nbc + 1); // 0 final
  if (!news) {
    error("insertImpl", UError::No_more_memory);
    return false;		// str et strLen inchanges !
  }

  if (!s) strncpy(news, s2+from_pos, nbc);
  else {
    if (to_pos > 0) strncpy(news, s, to_pos);
    strncpy(news+to_pos, s2+from_pos, nbc);
    strcpy(news+to_pos+nbc, s+to_pos);
    free(s);
  }
  news[len+nbc] = 0;  // 0 final

  syncVals(news, len+nbc);
  changed(upd);
  return true;
}

/* ==================================================== ======== ======= */

bool UStr::insertImpl(int pos, char c, bool upd) {
  if (!checkFormat(pos, c)) return false; //!!
  if (pos < 0 || pos > len) pos = len; // append

  char *news = (char*) malloc(len + 1 + 1); // 0 final
  if (!news) {
    error("insertImpl", UError::No_more_memory);
    return false;		// str et strLen inchanges !
  }

  if (!s) {
    news[0] = c;
    news[1] = '\0';
  }
  else {
    if (pos>0) strncpy(news, s, pos);
    news[pos] = c;
    strcpy(news+pos+1, s+pos);
    free(s);
  }

  syncVals(news, len+1);
  changed(upd);
  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UStr::remove(int pos, unsigned int nbchars) {
  replaceImpl(pos, nbchars, (char*)null, true);
}

void UStr::replace(int pos, unsigned int nbchars, const UStr& s2) {
  replaceImpl(pos, nbchars, s2.s, true);
}

void UStr::replace(int pos, unsigned int nbchars, const char *s2) {
  replaceImpl(pos, nbchars, s2, true);
}

void UStr::replace(int pos, unsigned int nbchars, const std::string& s2) {
  replaceImpl(pos, nbchars, s2.c_str(), true);
}

/* ==================================================== ======== ======= */

bool UStr::replaceImpl(int pos, unsigned int nbchars, const char *str, bool upd) {
  if (pos < 0) pos = len-1;  // last char (new 26jt)

  int nbc = (nbchars == npos) ? len : nbchars;
  
  // revient a un insert() ou un append() dans ce cas
  if (pos >= len) return insertImpl(pos, str, 0, npos, upd);

  // ne pas depasser taille actuelle de str
  if (pos + nbc > len) nbc = len - pos;

  int nadd = str ? strlen(str) : 0;
  int newlen = len - nbc + nadd;

  if (newlen <= 0) {	  // theoriquement jamais < 0 mais == 0
    syncVals(null, 0);
  }
  
  else {
    char *news = (char*)malloc(newlen + 1); // 0 final
    if (!news) {
      error("replaceImpl", UError::No_more_memory);
      return false;		  // str et strLen inchanges !
    }

    // cas (!s ou !*s)  ==>  equivalent a remove()
    if (nadd==0) {	
      if (!s) return false;	  // securite: deja teste par: newlen <= 0
      if (pos>0) strncpy(news, s, pos);
      strcpy(news+pos, s+pos+nbc);
      free(s); s = null;
    }

    // il y a qq chose a ajouter
    else {
      if (!checkFormat(pos, str)) return false; //!!

      if (!s) strcpy(news, str);
      else {
	if (pos>0) strncpy(news, s, pos);
	strncpy(news+pos, str, nadd);
	strcpy(news+pos+nadd, s+pos+nbc);
	free(s); s = null;
      }
    }

    syncVals(news, newlen);
  }

  changed(upd);
  return true;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */


bool UStr::copyTo(UStr& to, int pos, unsigned int nbchars) const {
  return to.insert(-1, *this, pos, nbchars);
}

bool UStr::copyTo(string& str, int from_pos, unsigned int nbchars) const {
  if (!s || from_pos >= len) // hors bornes
    return false;
  else {
    if (from_pos < 0) from_pos = len-1;
    int nbc = (nbchars == npos) ? len : nbchars;
  
    // ne pas depasser taille actuelle de str
    if (from_pos + nbc > len) nbc = len - from_pos;
    if (nbc <= 0) return false;
    else {
      str.append(s + from_pos, nbc);
      return true;
    }
  }
}

UStr UStr::substring(int pos, unsigned int nbchars) const {
  UStr substr;
  copyTo(substr, pos, nbchars);
  return substr;
}

std::string UStr::toString(int pos, unsigned int nbchars) const {
  std::string substr;
  copyTo(substr, pos, nbchars);
  return substr;
}

/*
void UStr::copyTo(string& str) const {
  str = s;
}
string UStr::getString() const {
  string str = s;
  return str;
}
string UStr::getString(int from_pos, int nbc) const {
  string str;
  copyTo(str, from_pos, nbc);
  return str;
}
void UStr::appendTo(string& str) const {
  str.append(s);
}
void UStr::insertInto(string& str, int to_pos) const {
  if (to_pos == -1) to_pos = str.length();
  str.insert(to_pos, s);
}

char *UStr::getChars() const {
  if (!s) return null;
  char* p = (char*) malloc((len+1)*sizeof(char));
  strcpy(p, s);
  return p;
}

char *UStr::getChars(int from_pos, int nbc) const {
  if (!s || nbc <= 0 || from_pos < 0 || from_pos >= len)  // hors bornes
    return null;
  else {
    // ne pas depasser taille actuelle de str
    if (from_pos + nbc > len) nbc = len - from_pos;

    char *p = (char*) malloc((nbc+1)*sizeof(char));
    strncpy(p, s + from_pos, nbc);
    p[nbc] = '\0';	  // 0 final
    return p;
  }
}
*/
/* ==================================================== ======== ======= */

void UStr::trim(bool strip_beginning, bool strip_end) {
  strip(strip_beginning, strip_end);
}

void UStr::strip(bool strip_beginning, bool strip_end) {
  if (!s || !*s) return;

  char* p1 = s;
  if (strip_beginning) {	// enlever les \n \t et spaces au debut
    // virer tout les caracteres speciaux jusuq'a ' ' inclus
    // attention ,cast necessaire sinon suppression des accentues
    while (*p1 && (unsigned char)*p1 <= ' ') p1++;
    if (!*p1) {
      free(s);
      syncVals(null, 0);
      return;
    }
  }

  char* p2 = s + len - 1;
  if (strip_end) {		// enlever les \n \t et spaces a la fin
    //while (p2 >= p1 && (*p2 == ' ' || *p2 == '\n' || *p2 == '\t')) p2--;
    while (p2 >= p1 && (unsigned char)*p2 <= ' ') p2--;
    *(p2+1) = 0;
  }

  char* s2 = CStr::strdup(p1);
  free(s);
  syncVals(s2, p2 - p1 + 1);
}

/* ==================================================== ======== ======= */

UStr UStr::split(int pos, bool delete_char_at_pos) {
  UStr res;

  if (!s || pos<0 || pos>=len) return res;

  if (delete_char_at_pos) res.set(s+1+pos);
  else res.set(s+pos);

  if (pos == 0) {
    free(s);
    syncVals(null, 0);
  }
  else {
    len = pos;
    s = (char*)realloc(s, sizeof(char)*(pos+1));
    s[pos] = '\0';
    syncVals(s, len);
  }

  changed(true);
  return res;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

int UStr::find(char c) {
  if (empty()) return -1;
  char* p = ::strchr(s, c);
  return (p ? p-s : -1);
}

int UStr::rfind(char c) {
  if (empty()) return -1;
  char* p = ::strrchr(s, c);
  return (p ? p-s : -1);
}

int UStr::find(const char* s2) {
  if (empty() || !s2) return -1;
  char* p = ::strstr(s, s2);
  return (p ? p-s : -1);
}

int UStr::find(const UStr& s2) {
  return find(s2.chars());
}

int UStr::find(const std::string& s2) {
  return find(s2.c_str());
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UStr::getSize(UContext *props, u_dim *w, u_dim *h) const {
  if (s && *s) {
    props->winview->wg().getTextSize(props->fontdesc, s, len, *w, *h);
    // sinon il n'y aura pas de place pour afficher le caret en fin de ligne
    // ex: *w += 1; plus necessaire car modif de l'affichage du caret
    // dans UEdit (une position en moins)
  }
  else {
    //NOTES: 
    // -1- conserver une hauteur stable
    // -2- faire en sorte que la surface ne soit jamais de taille nulle 
    //     (sinon on ne pourra plus la selecionner en cliquant)
    *h = props->winview->wg().getTextHeight(props->fontdesc);
    // de plus si w=0 le caret ne s'affiche pas
    *w = 1;
  }
}

// prototype for warped text (UFlowView)
void UStr::getSize(class UContext* props, u_dim *w, u_dim *h, 
		   u_dim available_width, int offset, 
		   int *sublen, int *change_line) const {
  if (s && *s) {
    props->winview->wg().getSubTextSize(props->fontdesc, 
					s + offset, len - offset,
					*w, *h, available_width, 
					*sublen, *change_line);

    // sinon il n'y aura pas de place pour afficher le caret en fin de ligne
    // *w += 1; // $$$$
  }
  else { 
    // voir NOTES ci-dessus
    *h = props->winview->wg().getTextHeight(props->fontdesc);
    *w = 1; // $$$$
    *sublen = 0;
  }
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void UStr::paint(UWinGraph &g, UContext *props, const URegion &r) const {
  paint(g, props, r, 0, len);
  // >>> deplace dans uuview
  //if (props->local.edit && props->local.edit->getCaretStr() == this) {
  //  props->local.edit->drawCaret(this, props, r, null, g);
  // }
}

void UStr::paint(UWinGraph &g, UContext *props, const URegion &r,
		 int offset, int cellen) const {
  if (!s || !*s) return;

  // !! don't confuse cellen and len!
  //UTextsel& ts = g.getAppli()->getDefaultFlow().getTextsel();
  UTextsel& ts = g.getDisp()->openFlow(0)->getTextsel();

  //ce qui serait cool c'est d'economiser les setColor et setFont!!!
  g.setFont(props->fontdesc);
  g.setColor(props->color);

  // cas ou il n'y a pas de TextSel, ou l'objet pointe est null
  // ou ce n'est pas le parent de str 
  // -> affichage de base sans selection
  
  if ((bmodes & UMode::IN_TEXTSEL) == 0
      || !ts.inObj
      // permettre l'affichage des objets inclus dans ts.inObj
      // || ts.inObj != props->obj
      || !ts.fromLink
      || !ts.toLink
      ) {
    g.drawString(s + offset, cellen, r.x, r.y);
  }

  // -> sinon affichage avec selection
  else {
    int deb = 0, fin = cellen;
    int ww = 0;

    // IL FAUDRAIT tester LEs LIENs (pas les UStr)
    // sinon les strings partagees vont s'allumer a plusieurs endroits!!

    if (this == dynamic_cast<const UStr*>(ts.fromLink->getChild())) {
      deb = ts.fromPos - offset;
      if (deb < 0) deb = 0;  //inclus dans selection
      else if (deb >= cellen) deb = cellen;   //exclus (avant) selection
    }

    // pas de else: this peut etre a la fois fromLink et toLink
    // (c'est meme le cas le plus frequent!)

    if (this == dynamic_cast<const UStr*>(ts.toLink->getChild())) {
      fin = ts.toPos - offset;
      if (fin < 0) fin = 0;   //exclus (apres) selection
      else if (fin >= cellen) fin = cellen; //inclus dans selection
    }

    if (deb > 0) {
      g.drawString(s+offset, deb, r.x, r.y);
      ww = props->winview->wg().getTextWidth(props->fontdesc, s+offset, deb);
    }
    
    if (fin > deb) {
      if (ts.color)   g.setColor(ts.color);
      if (ts.bgcolor) g.setBgcolor(ts.bgcolor);
      //UFontDesc *fd = &props->fontdesc;
      //if (ts.fontdesc) {fd = ts.fontdesc; g.setFont(fd);}
      if (ts.font) {
	UFontDesc fd = props->fontdesc;     // NEW aug03
	fd.merge(*ts.font,0);
	g.setFont(fd);
	g.drawString(s+deb+offset, fin-deb, r.x + ww, r.y);
	ww += props->winview->wg().getTextWidth(fd,
						s+deb+offset, fin-deb);
      }
      else {
	g.drawString(s+deb+offset, fin-deb, r.x + ww, r.y);
	ww += props->winview->wg().getTextWidth(props->fontdesc, 
						s+deb+offset, fin-deb);
      }
    }

    if (fin < cellen) {
      g.setFont(props->fontdesc);
      g.setColor(props->color);
      g.drawString(s+fin+offset, cellen-fin, r.x + ww, r.y);
    }
  }
} 

/* ==================================================== ======== ======= */
/* ==================================================== [Elc:03] ======= */
/* === useful functions =============================== ======== ======= */

// creates a duplicate of s1 (similar to strdup when this function exists)
// - note: argument can be null

char *CStr::strdup(const char *s1) {
  if (!s1) return null;
  else {
    char *p = (char*)malloc((strlen(s1) + 1) * sizeof(char));
    if (p) strcpy(p, s1);
    return(p);
  }
}

// creates a duplicate of s1 + s2
// - note: arguments can be null

char *CStr::strdupcat(const char *s1, const char *s2) {
  if (!s1 || !*s1) return CStr::strdup(s2);
  else if (!s2 || !*s2) return CStr::strdup(s1);
  else {
    int l1 = strlen(s1);
    char *p = (char*)malloc((l1 + strlen(s2) + 1) * sizeof(char));
    if (p) {
      strcpy(p, s1);
      strcpy(p+l1, s2);
    }
    return(p);
  }
}

char* CStr::strdupcat(const char *s1, char sep, const char *s2) {
  if (!s1 || !*s1) return CStr::strdup(s2);
  else if (!s2 || !*s2) return CStr::strdup(s1);
  else {
    int l1 = strlen(s1);
    char *p = (char*)malloc((l1 + strlen(s2) + 2) * sizeof(char));
    if (p) {
      strcpy(p, s1);
      p[l1] = sep;
      strcpy(p + l1 + 1, s2);
    }
    return(p);
  }
}

const char *CStr::strext(const char *path) {
  return CStr::strext((char*) path);
}

char *CStr::strext(char *path) {
  char *p = path + strlen(path);
  //probleme: etre sur que le . est apres le dernier /
  while (p >= path) {
    if (*p == '.' || *p == '/') break;
    p--;
  }
  // be sure its a . and eliminate . and .xxxrc and ..
  if (p <= path || *p != '.') return null;
  else if (p == path+1 && *path == '.') return null;
  else return p+1;
}

/* ==================================================== [Elc:03] ======= */
/* ==================================================== ======== ======= */

void CStr::strparse(char* path, int len,
		char*& dir_name, int& dir_len,
		char*& base_name, int& base_len,
		char*& suffix, int& suffix_len) {
  dir_name = null;
  dir_len = 0;
  base_name = null;
  base_len = 0;
  suffix = null;
  suffix_len = 0;

  if (len == 0 || !path || !*path) return;

  char *p = path + len;

  // chercher le /
  while (p > path) {
    if (*p == '/') break;
    p--;
  }

  if (*p == '/') {
    dir_name = path;
    if (p == path) dir_len = 1; // CP du "/"
    else dir_len = p - path;
    base_name = p+1;
    base_len = len - dir_len -1;
  }
  else {
    base_name = path;
    base_len = len;
  }

  if (!*base_name) {
    base_name = null;
    base_len = 0;
    return;
  }

  // chercher le . (be sure its a . and eliminate . and .xxxrc and ..)
  char *f = base_name;
  bool normal_char_found = false;
  while (*f != 0) {
    if (*f != '.') normal_char_found = true;
    else if (normal_char_found) break;
    f++;
  }

  if (*f == '.' && normal_char_found) {
    suffix = f+1;
    suffix_len = len - (f+1 - path);
    base_len -= (suffix_len + 1);
  }
}

/* ==================================================== ======== ======= */

UStr UStr::getFileName(bool with_suffix) const {
  UStr res;
  if (len == 0) return res;

  char* dir_name, *base_name, *suffix;
  int dir_len, base_len, suffix_len;
  CStr::strparse(s, len, dir_name, dir_len, base_name, base_len, suffix, suffix_len);

  if (with_suffix) res.set(base_name);
  else res.insert(0, base_name, 0, base_len);
  return res;
}


UStr UStr::getFileDir(bool with_slash) const {
  UStr res;
  if (len == 0) res = ".";
  else {
    char* dir_name, *base_name, *suffix; 
    int dir_len, base_len, suffix_len;
    CStr::strparse(s, len, dir_name, dir_len, base_name, base_len, suffix, suffix_len);

    if (dir_len == 0) res = ".";
    else res.insert(0, dir_name, 0, dir_len);
  }

  if (with_slash && res.charAt(-1) != '/') res.insert(-1, '/');
  return res;
}


UStr UStr::getFilePath(const UStr& new_suffix) const {
  UStr res;
  if (len == 0) res = ".";
  else {
    char* dir_name, *base_name, *suffix; 
    int dir_len, base_len, suffix_len;
    CStr::strparse(s, len, dir_name, dir_len, base_name, base_len, suffix, suffix_len);
    if (suffix) res.insert(0, s, 0, suffix-s);
    else if (s[len-1] != '.') res.insert(-1, '.');
  }

  res.insert(-1, new_suffix);
  return res;
}


UStr UStr::getFileSuffix() const {
  UStr res;
  if (len == 0) return res;

  char* dir_name, *base_name, *suffix;
  int dir_len, base_len, suffix_len;
  CStr::strparse(s, len, dir_name, dir_len, base_name, base_len, suffix, suffix_len);

  res.insert(0, suffix, 0, suffix_len);
  return res;
}


long UStr::getFileType() const {
  if (!s || !*s) return 0;  // null or empty filename
  else {
    struct stat filestat;
    if (stat(s, &filestat) != 0) return 0;   // not found
    else return (filestat.st_mode & S_IFMT); // returns the type
  }
}


long UStr::getFileType(bool& is_regular_file, bool& is_directory) const {
  is_regular_file = is_directory = false;
  long t = getFileType() ;

  if (t == 0) return false;
  if ((t & S_IFMT) == S_IFREG) is_regular_file = true;
  if ((t & S_IFMT) == S_IFDIR) is_directory = true;
  return t;
}


bool UStr::isFileFound(bool regular_file) const {
  long t = getFileType() ;

  if (t == 0) return false;
  else if (regular_file) {
    return ((t & S_IFMT) == S_IFREG); 
  }
  else return true;
}

/* ==================================================== ======== ======= */

int UStr::readFile(const UStr& filename) {
  if (filename.empty()) return UFilestat::CannotOpen;

  UStr fname = filename;
  if (fname.charAt(0) == '~' && fname.charAt(1) == '/') {
    //NB: ne PAS faire de free() sur un getenv!
    char *home = getenv("HOME");
    if (home) fname.replace(0, 1, home); // virer ~mais pas /
  }
  
  char *buffer = null;
  struct stat finfo;
  int fd = -1;
  int res = UFilestat::NotOpened;
  int ll = 0;

  if ((fd = open(fname.chars(), O_RDONLY)) == -1) {
    res = UFilestat::CannotOpen;
  }
  else if (fstat(fd, &finfo) == -1
	   || (finfo.st_mode & S_IFMT) != S_IFREG) {
    res = UFilestat::CannotOpen;
  }
  else if (!(buffer = (char*)malloc((finfo.st_size + 1) * sizeof(char)))) {
    res = UFilestat::NoMemory;
  }
  else if (read(fd, buffer, finfo.st_size) <= 0) {
    res = UFilestat::InvalidData;
    if (buffer) free(buffer);
    buffer = null;
  }
  else {
    // trouver le 1et \0 et s'arreter la pour eviter plantages avec
    // les executables, etc.
    for (ll = 0; ll < finfo.st_size; ll++)
      if (buffer[ll] == 0) break;

    // be sure it's NULL terminated
    if (ll >= finfo.st_size) {
      ll = finfo.st_size;
      buffer[ll-1] = '\0';
    }

    res = UFilestat::Opened;
  }

  if (fd != -1) close(fd);

  setImpl(buffer, ll, true);

  return res;
}

/* ==================================================== ======== ======= */

int UStr::readFile(const UStr& filename, std::vector<UStr*>& slist,
                   bool strip_beginning, bool strip_end) {

  if (filename.empty()) return UFilestat::CannotOpen;

  UStr fname = filename;
  if (fname.charAt(0) == '~' && fname.charAt(1) == '/') {
    //NB: ne PAS faire de free() sur un getenv!
    char *home = getenv("HOME");
    if (home) fname.replace(0, 1, home); // virer ~mais pas /
  }

  ifstream in(fname.chars());
  if (!in) return UFilestat::CannotOpen;

  string line;
  while ((getline(in, line))) {
    UStr* s = new UStr(line);
    if (strip_beginning || strip_end) s->strip(strip_beginning, strip_end);
    slist.push_back(s);
  }

  return UFilestat::Opened;
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:03] ======= */

