/* ==================================================== ======== ======= *
 *
 *  uutextsel.cpp
 *  Ubit Project [Elc][2003]
 *  Author: Eric Lecolinet
 *
 *  Part of the Ubit Toolkit: A Brick Construction Game Model for Creating GUIs
 *
 *  (C) 1999-2003 Eric Lecolinet @ ENST Paris
 *  WWW: http://www.enst.fr/~elc/ubit   Email: elc@enst.fr (subject: ubit)
 *
 * ***********************************************************************
 * COPYRIGHT NOTICE : 
 * THIS PROGRAM IS DISTRIBUTED WITHOUT ANY WARRANTY AND WITHOUT EVEN THE 
 * IMPLIED WARRANTY OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. 
 * YOU CAN REDISTRIBUTE IT AND/OR MODIFY IT UNDER THE TERMS OF THE GNU 
 * GENERAL PUBLIC LICENSE AS PUBLISHED BY THE FREE SOFTWARE FOUNDATION; 
 * EITHER VERSION 2 OF THE LICENSE, OR (AT YOUR OPTION) ANY LATER VERSION.
 * SEE FILES 'COPYRIGHT' AND 'COPYING' FOR MORE DETAILS.
 * ***********************************************************************
 *
 * ==================================================== [Elc:03] ======= *
 * ==================================================== ======== ======= */

//pragma ident	"@(#)uutextsel.cpp	ubit:03.04.04"
#include <udefs.hpp>
#include <ubrick.hpp>
#include <ucall.hpp>
#include <uerror.hpp>
#include <ucolor.hpp>
#include <ustr.hpp>
#include <ufont.hpp>
#include <ugroup.hpp>
#include <update.hpp>
#include <uevent.hpp>
#include <utextsel.hpp>


UTextsel::UTextsel(const UColor* _color, const UColor* _bgcolor, 
		   const UFont* _font, bool recursive_selection) {
  recursiveSelection = recursive_selection;
  color    = _color;
  bgcolor  = _bgcolor;
  font     = _font;
  inObj = null;
  reset(false);
}

void UTextsel::reset(bool upd) {
  pressLink = oldLink = null;
  pressPos = oldPos = 0;
  fromLink = toLink = null;
  fromPos  = toPos = 0;
  if (upd) update(null, 0);
  inObj = null;
}

/* ==================================================== ======== ======= */

void UTextsel::update(ULink *newLink, int newPos) {
  if (!inObj) return;
  int refreshFromPos = 0, refreshToPos = 0;

  if (pressLink && newLink) {

    if (newLink == pressLink) {
      fromLink = toLink = pressLink; 
      if (newPos > pressPos) {fromPos = pressPos; toPos = newPos;}
      else {fromPos = newPos; toPos = pressPos;}
    }

    else {
      bool found = false;

      for (ULink *l = pressLink; l!=null; l = l->getNext())
	if (l == newLink) {
	  found = true;
	  fromLink = pressLink;
	  fromPos  = pressPos;
	  toLink   = newLink;
	  toPos    = newPos;
	  break;
	}
  
      if (!found) {
	for (ULink *l = newLink; l!=null; l = l->getNext())
	  if (l == pressLink) {
	    fromLink = newLink;
	    fromPos  = newPos;
	    toLink   = pressLink;
	    toPos    = pressPos;
	    break;
	  }
      }
      //NB: dans certains cas newLink appartient a un autre objet
      //et on n'arrive donc pas a trouver fromLink et toLink
      // c'est pas grave on garde les anciens
    }

    refreshFromPos = fromPos, refreshToPos = toPos;

    if (oldLink == fromLink && newPos == fromPos) {
      if (oldPos < fromPos) {
	refreshFromPos = oldPos;
	refreshToPos   = fromPos;
      }
      else {
	refreshFromPos = fromPos;
	refreshToPos   = oldPos;
      }
    }

    else if (oldLink == toLink && newPos == toPos) {
      if (oldPos < toPos) {
	refreshFromPos = oldPos;
	refreshToPos = toPos;
      }
      else {
	refreshFromPos = toPos;
	refreshToPos   = oldPos;
      }
    }

    oldLink = newLink;
    oldPos  = newPos;
  }

  paint(refreshFromPos, refreshToPos, recursiveSelection);
}

/* ==================================================== ======== ======= */
// NB: cette fonction change egalement le mode IN_TEXTSEL des elems

void UTextsel::paint(int refreshFromPos, int refreshToPos, bool recursive) {

  bool initial_state = false;
  paintImpl(inObj, initial_state, refreshFromPos, refreshToPos, recursive);
}

void UTextsel::paintImpl(UGroup* obj, bool state,
			 int refreshFromPos, int refreshToPos,
			 bool recursive) {

  for (ULink *l = obj->getChildLinks(); l!=null; l = l->getNext()) {

    if (l == fromLink) state = true;

    UElem*elem = null;
    UGroup* chgrp = null;

    if ((elem = l->getChild()->elemCast())) {
      u_modes bmodes = elem->getBmodes();

      if (state || (!state && (bmodes & UMode::IN_TEXTSEL) != 0)) {

	// allume ou eteint le mode IN_TEXTSEL de l'elem suivant le cas
	elem->setBmodes(UMode::IN_TEXTSEL, state);
	UUpdate upd(UUpdate::PAINT);

	// ATT: il faut egalement reafficher les zones qui redeviennent 
	// non selectionneees et repassent en noir

	UStr* str = elem->strCast();

	if (!str) upd.paintElem(elem);
	else {
	  if (l == fromLink && l == toLink)
	    upd.paintStr(str, refreshFromPos, refreshToPos);	 
	  else if (l == fromLink)
	    upd.paintStr(str, refreshFromPos, str->length());
	  else if (l == toLink)
	    upd.paintStr(str, 0, refreshToPos);
	  else 
	    upd.paintStr(str, 0, str->length());
	  
	}
	// a defaut de faire un traitement plus optimise
	// on reaffiche en double buffering pour eviter le flicking

	/////	upd.paintDoubleBuffered();

	// update all parents (not only the inBox being selected)
	obj->update(upd); 
      }

    } // endif(item)

    else if (recursive && (chgrp = l->getChild()->groupCast())) {
      // (initial) state = false ou true suivant le cas
      paintImpl(chgrp, state, refreshFromPos, refreshToPos, recursive);
    }
    
    //!att: a mettre a la fin du corps de boucle!
    if (l == toLink) state = false;
  }
}

/* ==================================================== ======== ======= */

UStr UTextsel::copyText() const {
  UStr res;
  copyText(res, recursiveSelection);
  return res;
}

void UTextsel::copyText(UStr& res) const {
  copyText(res, recursiveSelection);
}

void UTextsel::copyText(UStr& selection, bool recursive) const {
  bool state = false;
  int refreshFromPos = fromPos, refreshToPos = toPos;
  //char *ps = null;

  selection = null;

  if (!inObj) return;

  const UStr* sep = inObj->getTextSeparator();
  if (sep && sep->length() <= 0) sep = null;

  for (ULink *l = inObj->getChildLinks(); l!=null; l = l->getNext()) {

    if (l == fromLink) state = true;

    if (state) {

      UBrick* ch = l->getChild();
      UGroup* chgrp = null;

      if (ch->elemCast()) {

	//inutile dans le cas print
	//u_modes bmodes = ch->getBmodes();
	//if (state || (!state && (bmodes & UMode::IN_TEXTSEL) != 0)) {
	
	UStr *str = ch->strCast();

	//nb: cas (elem && !str) pas traite!

	if (str) {
          if (l == fromLink && l == toLink) {

            //if ((ps = str->getChars(refreshFromPos, refreshToPos-refreshFromPos))){
            //selection.append(ps);
            //delete ps;
            if (refreshToPos > refreshFromPos)
              selection.insert(-1, *str, refreshFromPos,
                               refreshToPos-refreshFromPos);
            }
          else if (l == fromLink) {
            //if ((ps = str->getChars(refreshFromPos, str->length()))){
	    //  selection.append(ps);
	    //  delete ps;
	    //}
            selection.insert(-1, *str, refreshFromPos, str->length());
          }
          else if (l == toLink) {
	    //if ((ps = str->getChars(0, refreshToPos))){
	    //  selection.append(ps);
	    //  delete ps;
	    //}
            if (0 < refreshToPos)
              selection.insert(-1, *str, 0, refreshToPos);
	  }
	  else selection.append(str->chars());
	} // endif(str)
      } // endif(elem)


      else if (recursive && (chgrp = ch->groupCast())) {
	if (sep /*&& chgrp_found*/) selection.append(*sep);
	// chgrp_found = true;
	selection.append(chgrp->copyText(true));
      }

    } // endif(state)
    
    //!att: a mettre a la fin du corps de boucle!
    if (l == toLink) state = false;
  }
}

/* ==================================================== [Elc:02] ======= */
/* ==================================================== ======== ======= */

void UTextsel::start(UEvent& e) {
  if (e.getButtons() == UEvent::MButton1    // PARAMETRER!!
      && e.getSource()) {
    
    //effacer selection courante (ce qui necessite ancien obj)
    reset(true);

    // nouvelle selection
    inObj = e.getSource();
    if (!inObj) return;
    being_selected = true;

    UElemProps itd;
    if (!e.getStr(itd)) inObj = null;
    else {
      pressLink = itd.elemLink;
      pressPos  = itd.strpos;
    }
  }
}

/* ==================================================== ======== ======= */

void UTextsel::extend(UEvent& e) {
  if (!inObj || e.getSource() != inObj) return;

  // si on a localise une string, changer la selection
  // (sinon on ne fait rien pour ne pas abimer inutilement 
  // la selection precedente)

  UElemProps itd;
  //printf("\n$ textsel:extend --> getStr()\n");
  if (e.getStr(itd)) {
    update(itd.elemLink, itd.strpos);
  }
  /*
  UView *view = e.getView();
  if (view) {
    // on ne reaffiche que la vue courante
    UUpdate upd;
    upd.paintRegion(&itd.region);
    e.getSource()->updateView(e, view, upd); 
  }
  */
}

/* ==================================================== ======== ======= */

bool UTextsel::complete(UEvent& e) {
  being_selected = false;

  // si on a *effectivement* selectionne qq chose 
  // ==> obtenir la XSelection (pour desactiver la section dans les
  //     autres applis et leur permettre ensuite de savoir que qq chose
  //     a ete selectione)

  return (fromLink && toLink && (fromLink != toLink || fromPos != toPos));
}

/* ==================================================== [TheEnd] ======= */
/* ==================================================== [Elc:02] ======= */

