// This may look like C code, but it is really -*- C++ -*-

//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        faceattr.h
//
// Purpose:     Interface to class FaceAttr
//
// Created:     7 Feb 1996  Georg Meszaros
//
// Modified:    13 Jun 1996 Georg Meszaros    
//
//
// Description:
//
// Definition of the class FaceAttr which holds the vertex list and
// the normal vector list of an indexed face set. In addition
// pointers to the corresponding attributes for the indexed faceset
// (material, lighting, backface culling) are held.
//
// All in a scene existing material and lighting appearances are
// stored on the free store and never deleted while the scene
// exists. Pointers to elements of this globally stored settings are the
// values maintained in this class. Therefore several facesets can use
// the same material or light settings, these settings do not need to be
// stored with every faceset.
//
// In this version it is assumed that one indexed faceset only
// consists of one material, it is not supported that every face holds
// its own material setting.
//
//
//</file>


#ifndef harmony_scene_faceattr_h
#define harmony_scene_faceattr_h


//<class>
//
// Name:    FaceAttr
//
// Purpose: Indexed faceset and attributes of this faceset.
//
// Public Interface:
/*

   - FaceAttr();

   - FaceAttr(int vertex_num,               // number of vertices
              QvMFVec3f* vertex_list,       // vertices
              int normal_num,               // number of normals
              QvMFVec3f* normal_list);      // normals
     > the vertex- and the normal-list are not copied, just linked into the class
     > attributes (material, etc. ) are not set with the constructor

   - FaceAttr(const FaceAttr& faceattr);
     > copy constructor - copies the lists

   - ~FaceAttr();       
     > frees the vertexlists

   - FaceAttr& operator=(const FaceAttr& faceattr);

   - void print();
   - void printVertices();
   - void printNormals();
     > all print to cerr

   - void setAttributes(materialsGE3D* material,  // one material structure
                        unsigned light_num,       // number of lights set
                        AnyLight** light,         // list of some light structures
                        int backface_culling);    // flag 
   
   - unsigned long id() const;
     > unique id for this instance

   - int vertexNum() const;
   - int normalNum() const;

   - void setVertexListSize(int num);
   - void setNormalListSize(int num);
     > allocates num places for vertices or normals
               
   - point3D* vertexList() const;
     > returns the pointer to the vertexlist
   - vector3D* normalList() const;
     > returns the pointer to the normallist
  
   - int extendVertices(point3D& new_vertex,
                        vector3D& new_normal);
     > allocates new memory, if necessary, and extends the lists with
       the values for new_vertex and new_normal - if the normal is not (0,0,0)

   - void reset();
     > resets the counter that is used for building the unique id

   - void newFaceRef();
     > a counter is increased with each face that belongs to this faceattr

   - void unFaceRef();
     > the counter is decreased until it is zero - in this case this
       instance is deleted

   - materialsGE3D* material() const;    
   - AnyLight** light() const;
   - int backfaceCulling() const;
     > the attributes for all faces that belong to this faceattr

   - int lightNum() const;
     > the number of active lights
    
   - void setBoundBox(const point3D& min, const point3D& max);
     > sets the bounding box for all vertices held in this faceset

   - void drawBoundBox();

*/
// Description:
//
// Building a class of attributes to faces(a faceset) has
// the advantage that more than one face can belong to the same attribute 
// class.
// The vertex list and normal vector list are stored in this class. 
// Furthermore the principles of indexed facesets are implemented to
// save memory for faces sharing some of their vertices (every split
// of a convex polygon only extends the vertex list by 2 vertices).
// A face holds indices to this lists(arrays).
// This saves memory storage and complexity. Even more than one
// faceset can hold(point to) the same attributes light material- and
// lighting settings.
//
//
//</class>



#include <ge3d/vectors.h>
#include <ge3d/mtl.h>

class Face;
class AnyLight;
class QvMFVec3f;

//maximum number of lights at the same time
#define NUM_LIGHTS 20


class FaceAttr
{
  public:

    FaceAttr();

    FaceAttr(                           
      int vertex_num,               // number of vertices
      QvMFVec3f* vertex_list,       // vertices
      int normal_num,               // number of normals
      QvMFVec3f* normal_list);      // normals

    FaceAttr(const FaceAttr& faceattr);

    ~FaceAttr();       

    FaceAttr& operator=(const FaceAttr& faceattr);

    void print();
    void printVertices();
    void printNormals();

    void setAttributes(
      materialsGE3D* material,      // one material structure
      unsigned light_num,
      AnyLight** light,             // list of some light structures
      int backface_culling
    );
   
    unsigned long id() const;
    int vertexNum() const;
    int normalNum() const;
    void setVertexListSize(int num);
    void setNormalListSize(int num);
               
    point3D* vertexList() const;
    vector3D* normalList() const;
  
    int extendVertices(
           point3D& new_vertex,
           vector3D& new_normal
    );

    void reset();

    void newFaceRef();
    void unFaceRef();

    materialsGE3D* material() const;      // one material structure
    AnyLight** light() const;             // list of some light structures
    int backfaceCulling() const;

    int lightNum() const;
  
    void setBoundBox(const point3D& min, const point3D& max);
   
    void drawBoundBox();

  private:
    void free();
    void copy(const FaceAttr& faceattr);
   
    static unsigned long attr_number_;  // count the existing instances
    static unsigned long counter_;      // count for a unique id
    static int alloc_ahead_;            // is set in the .C file
    // every time a new vertex or normal is going to be added at the
    // end of the lists - not only one but some more (alloc_ahead) memory
    // will be allocated at once     
                                
    unsigned long id_;            // the unique id     
    int face_ref_count_;          // for ref/unref with faces
    // used to delete the whole tree - if a face is deleted, the
    // corrsponding faceattr gets the unFaceRef call which decrements this
    // counter which is incremented if a new face is linked to this faceattr

    QvMFVec3f* vertex_list_;      // list of all vertices  
    QvMFVec3f* normal_list_;      // list of normal vectors
    int vertex_num_;              // number of all vertices
    int normal_num_;              // number of all normals
      
    materialsGE3D* material_;     // one material structure
    AnyLight** light_;            // list of some light structures
    int backface_culling_;        // 0 turned off, 1 activated 

    int light_num_;
 
    // bounding box is axes-aligned
    point3D min_;  // minimal point of the bounding box
    point3D max_;  // maximal point of the bounding box of object


}; // FaceAttr

inline unsigned long FaceAttr::id() const { return id_; } 

inline int FaceAttr::vertexNum() const { return vertex_num_; }
inline int FaceAttr::normalNum() const { return normal_num_; }

inline void FaceAttr::newFaceRef() { face_ref_count_++; }
inline void FaceAttr::unFaceRef() { if (--face_ref_count_ == 0) delete this; }

inline materialsGE3D* FaceAttr::material() const { return material_; }    
inline AnyLight** FaceAttr::light() const { return light_; }
inline int FaceAttr::backfaceCulling() const { return backface_culling_; } 

inline int FaceAttr::lightNum() const { return light_num_; }

inline void FaceAttr::setBoundBox(const point3D& min, const point3D& max) 
                      { min_ = min; max_ = max; }   

inline void FaceAttr::reset() { counter_ = 0; }

#endif
