//<copyright>
//
// Copyright (c) 1996
// Institute for Information Processing and Computer Supported New Media (IICM),
// Graz University of Technology, Austria.
//
// This file is part of VRweb.
//
// VRweb is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2, or (at your option)
// any later version.
//
// VRweb is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with VRweb; see the file LICENCE. If not, write to the
// Free Software Foundation, Inc., 59 Temple Place - Suite 330,
// Boston, MA 02111-1307, USA.
//
// Note that the GNU General Public License does not permit incorporating
// the Software into proprietary or commercial programs. Such usage
// requires a separate license from IICM.
//
//</copyright>

//<file>
//
// Name:        wrlpass.C
//
// Purpose:     VRML scene graph traversal
//
// Created:      7 Jun 96   Michael Pichler
//
// Changed:     13 Jun 96   Michael Pichler
//
// $Id: wrlpass.C,v 1.6 1997/02/25 17:03:58 mpichler Exp $
//
//</file>


// this is indented for various (small) purpose scene graph traversal
// see enum below for operations on the the scene graph


#if defined(PMAX) || defined (HPUX9)
enum Part { goofyPart };        // cfront confused about QvCone::Part and QvCylinder::Part
enum Binding { goofyBinding };  // cfront confused about QvMaterialBinding/QvNormalBinding::Binding
#endif


#include <vrml/QvElement.h>
#include <vrml/QvNodes.h>
#include <vrml/QvChildList.h>
#include <vrml/QvExtensions.h>
#include <vrml/QvUnknownNode.h>
#include <vrml/QvDebugError.h>
#include <vrml/QvState.h>

#include "scene3d.h"
#include "vrmlscene.h"

#include <hyperg/hyperg/message.h>
#include <hyperg/utils/verbose.h>

#include <stdio.h>
#include <iostream.h>
#include <math.h>


enum
{
  pass_null,
  pass_clearhganchor,  // clear Hyper-G anchors in scene graph
  pass_getnodebyname,  // find node by name
  pass_findtexture     // find texture by filename field
};
static int pass_op = 0;  // what to do

static const char* node_name = 0;
static QvNode* node_location = 0;
static QvGroup* node_parent = 0;
static int node_childindex = 0;


/***** clearAnchors *****/

void VRMLScene::clearAnchors ()
{
  if (!root_)
    return;

  pass_op = pass_clearhganchor;
  root_->pass (0, 0);  // args: parent, childindex

} // clearAnchors


/***** getNodeByName *****/

QvNode* VRMLScene::getNodeByName (const char* name, QvGroup*& parent, int& childindex)
{
  if (!name || !root_)
    return 0;

  node_name = name;  // non-NULL
  node_location = 0;
  node_parent = 0;
  node_childindex = 0;

  pass_op = pass_getnodebyname;
  root_->pass (0, 0);

  QvNode* location = node_location;
  parent = node_parent;
  childindex = node_childindex;

  node_name = 0;
  node_location = 0;
  node_parent = 0;
  node_childindex = 0;

  return location;

} // getNodeByName


/***** findTexture *****/

QvTexture2* VRMLScene::findTexture (const char* fnamefield)
{
  if (!fnamefield || !root_)
    return 0;

  node_name = fnamefield;  // non-NULL
  node_location = 0;

  pass_op = pass_findtexture;
  root_->pass (0, 0);

  QvTexture2* texnode = (QvTexture2*) node_location;
  if (texnode && texnode->nodeType () != QvNodeType::QvTexture2)
  { cerr << "VRMLScene::findTexture: internal program error. "
            "Type of node " << (void*) texnode << ": " << texnode->nodeType () << endl;
    texnode = 0;
  }

  node_name = 0;
  node_location = 0;

  return texnode;

} // findTexture



/***** groups *****/


void QvGroup::pass (QvGroup* parent, int childindex)
{
  QvNode::pass (parent, childindex);
  if (node_location)  // only need to find first location
    return;

  int n = getNumChildren ();
  QvNode* child;

  for (int i = 0; i < n; i++)
  {
    child = getChild (i);
    child->pass (this, i);
  } // for all children

} // QvGroup


void QvLOD::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


void QvSeparator::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


void QvSwitch::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


void QvTransformSeparator::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


void QvUnknownNode::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


void QvWWWInline::pass (QvGroup* parent, int childindex)
{
  QvGroup::pass (parent, childindex);
}


/***** anchor *****/


void QvWWWAnchor::pass (QvGroup* parent, int childindex)
{
  // first remove any anchors from children
  QvGroup::pass (parent, childindex);

  if (pass_op != pass_clearhganchor || !hganchorid_)  // WWW anchor or different task
    return;

  if (!parent || childindex < 0 || childindex >= parent->getNumChildren () || getNumChildren () != 1)
  {
    HgMessage::error ("unable to remove Hyper-G anchor from internal data structure.");
    // not sure wheter it is better to mark this object as non-Hyper-G-anchor or not
    // (creation/insertion of new ones vs. using old ones)
//   hganchorid_ = 0;
//   parentURL_ = "";
    return;
  }

  DEBUGNL ("anchor being removed: child " << childindex << " of parent");

  QvNode* origchild = getChild (0);
  origchild->ref ();
  QvChildList* childlist = parent->getChildren ();  // non-NULL
  childlist->remove (childindex);  // will also unref/delete this (!), which unrefs origchild
  // all member vars invalid from now on
  childlist->insert (childindex, origchild);  // ref
  origchild->unref ();

} // QvWWWAnchor



/***** nodes *****/


void QvNode::pass (QvGroup* parent, int childindex)
{
  if (pass_op == pass_getnodebyname && !node_location)
  {
    const char* myname = getName ().getString ();
    // cerr << "comparing node name '" << node_name << "' with my node name '" << myname << "'" << endl;
    if (myname && !strcmp (myname, node_name))
    {
      node_location = this;
      node_parent = parent;
      node_childindex = childindex;
    }
  }
}


void QvTexture2::pass (QvGroup* parent, int childindex)
{
  QvNode::pass (parent, childindex);
  if (pass_op == pass_findtexture && !node_location)
  {
    const char* texname = filename.value.getString ();
    // cerr << "searching texture named '" << node_name << "'; this texture's filename: '" << texname << "'" << endl;
    if (texname && !strcmp (texname, node_name))
    {
      node_location = this;
      hgtexture_ = 1;  // see Hg3dViewer::storeanchor and QvTexture2::draw
    }
  }
  // hgtexture_ not needed to be reset on clearAnchors,
  // because texture is requested only once (won't be requested again on reload)
}


#define PASS(className)  \
void className::pass (QvGroup* parent, int childindex)  \
{ QvNode::pass (parent, childindex);  \
}


PASS(QvCoordinate3)
PASS(QvNormal)
PASS(QvTextureCoordinate2)
PASS(QvFontStyle)
PASS(QvMaterial)
PASS(QvMaterialBinding)
PASS(QvNormalBinding)
PASS(QvShapeHints)
PASS(QvTexture2Transform)
PASS(QvDirectionalLight)
PASS(QvPointLight)
PASS(QvSpotLight)
PASS(QvOrthographicCamera)
PASS(QvPerspectiveCamera)
PASS(QvTransform)
PASS(QvRotation)
PASS(QvMatrixTransform)
PASS(QvTranslation)
PASS(QvScale)
PASS(QvAsciiText)
PASS(QvCube)
PASS(QvCone)
PASS(QvCylinder)
PASS(QvSphere)
PASS(QvIndexedFaceSet)
PASS(QvIndexedLineSet)
PASS(QvPointSet)
PASS(QvInfo)
PASS(QvLabel)
PASS(QvLightModel)
