;; Routines for "what you see is HTML".
;; copyright 2003 Junichi Uekawa.

;; This file is free software; you can redistribute it and/or modify
;; it under the terms of the GNU General Public License as published by
;; the Free Software Foundation; either version 2, or (at your option)
;; any later version.
;;
;; readme-debian.el is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY; without even the implied warranty of
;; MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
;; GNU General Public License for more details.
;;
;; You should have received a copy of the GNU General Public License
;; along with your Debian installation, in /usr/share/common-licenses/GPL
;; If not, write to the Free Software Foundation, 675 Mass Ave,
;; Cambridge, MA 02139, USA.

(defgroup wysihtml nil 
  "WYSIHTML"
  :group 'html
  :prefix "wysihtml-")
(defcustom wysihtml-daemon-name "@LIBEXECDIR@/wysihtmldaemon" 
  "The default value for wysihtml daemon."
  :group 'wysihtml
  :type 'file)
(defconst wysihtml-daemon-process-name "wysihtmldaemon-interactive")
(defvar wysihtml-daemon-buffer nil 
  "The buffer name for wysihtml daemon, this value should not be modified by user.")
(defvar wysihtml-active-modes-alist 
  '((html-mode . t)
    (dancer-diary-english-mode . t)
    (dancer-diary-japanese-mode . t))
  "*List of major modes that wysihtml is activated."
  )
(defconst wysihtml-version "@VERSION@"
  "The version number of wysihtml.")
(defcustom wysihtml-elserv-port 8089 "Elserv port used for wysihtml."
  :group 'wysihtml
  :type 'number)
(defcustom wysihtml-mozilla-command-line "mozilla -remote openurl\\(%s\\)"
  "The command-line used to invoke mozilla-remote interface.")
(defvar wysihtml-elserv-process nil "The elserv process identifier for the elserv process being used for wysihtml.")
(defvar wysihtml-current-buffer nil 
  "The buffer which is going to be displayed through elserv, for preview.")
(defcustom wysihtml-interactive-time 1
  "The number of seconds to wait before renew.
slow machines may need several seconds, but fast machines may work with  less, for better response time."
  :group 'wysihtml
  :type 'number)
(require 'comint)
(require 'elserv)
(require 'mcharset)

(defun wysihtml-start-daemon ()
  "Start up daemon for wysihtml, and assign a buffer for the running process."
  (setq wysihtml-daemon-buffer 
	(make-comint wysihtml-daemon-process-name
		     wysihtml-daemon-name
		     nil
		     "-m" wysihtml-mozilla-command-line
		     "-s" (number-to-string wysihtml-interactive-time))))

(defun wysihtml-restart-daemon ()
  "Restart the daemon after reconfiguration."
  (interactive)
  (kill-buffer wysihtml-daemon-buffer)
  (wysihtml-start-daemon))

(defun wysihtml-start-elserv ()
  "Internal routine to start up elserv process.
wysihtml uses this process for displaying HTML through MOZILLA."
  (setq wysihtml-elserv-process (elserv-start wysihtml-elserv-port))
  (elserv-publish wysihtml-elserv-process
		  "/preview"
		  :allow '("localhost")
		  :function 'wysihtml-elserv-preview
		  :description "Preview of current HTML buffer"))

(defun wysihtml-elserv-preview (result path ppath request) 
  "View HTML from buffer.

RESULT is the resulting value
PATH is relative path from the published path
PPATH is the published path
REQUEST is the request data."
  (if (and (>= (length path) 11)
	   (string= (substring path 0 11) "/index.html"))
      (save-excursion
	(let* ((charset nil))
	  (set-buffer wysihtml-current-buffer)
	  ;; publish the related files.
	  (elserv-publish wysihtml-elserv-process
			  "/files"
			  :allow '("localhost")
			  :directory default-directory
			  :description "Files in current directory")
	  (setq charset (detect-mime-charset-region (point-min)(point-max)))
	  (elserv-set-result-header 
	   result
	   (list 'content-type (concat "text/html; charset=" (symbol-name charset))))
	  (elserv-set-result-body result 
				  (encode-mime-charset-string (buffer-string) charset))))
    ;; other file requests, redirect them
    (elserv-make-redirect result (concat "http://localhost:" (number-to-string wysihtml-elserv-port) "/files/" path))))

(define-minor-mode wysihtml-mode 
  "WYSIhtml minor-mode for auto-previewing of HTML through mozilla
remote interface"
  nil
  " WYSIHTML"
  nil
  (if wysihtml-mode
      (progn
	(if (assoc major-mode wysihtml-active-modes-alist)
	    t
	  (setq wysihtml-mode nil)
	  (error "current major-mode is not in wysihtml-active-modes-alist"))
	(if (eq nil wysihtml-daemon-buffer)
	    (wysihtml-start-daemon))
	(wysihtml-daemon-post-command)	;run this once immediately
	(add-hook 'post-command-hook (function wysihtml-daemon-post-command)
		  nil t))
    (progn
      (remove-hook 'post-command-hook (function wysihtml-daemon-post-command)))))
	
(defun wysihtml-daemon-post-command ()
  "Wysihtml that is invoked after `post-command-hook'."
  (if (not wysihtml-mode)
      t
    (if (local-variable-p 'wysihtml-daemon-buffer-modified-tick)
	t
      (make-local-variable 'wysihtml-daemon-buffer-modified-tick))
    (if (assoc major-mode wysihtml-active-modes-alist)
	(let* ((curval (buffer-modified-tick)))
	  ;; check if all conditions are met.
	  (if (and (boundp 'wysihtml-daemon-buffer-modified-tick)
		   (= curval wysihtml-daemon-buffer-modified-tick)
		   (not (assoc last-command '((recenter . t)))))
	      t				;nothing should be done.
	    ;; otherwise, queue for the redrawing of the page.
	    (setq wysihtml-daemon-buffer-modified-tick curval) 
	    (wysihtml-daemon-moz-preview-page))))))

(defun wysihtml-daemon-moz-preview-page ()
  "Preview the current buffer in mozilla by sending information to the daemon.
It will send the current filename to the daemon.
If it finds <a name in the html source, it will try to jump to the 
nearest one."
  (if (eq nil wysihtml-elserv-process)
      (wysihtml-start-elserv))
  (setq wysihtml-current-buffer (current-buffer))
  (save-excursion
    (let* ((deactivate-mark nil)
	   (optional-name ""))
      (if (re-search-backward "<[aA] [nN][aA][mM][eE]=\"?\\([^>\" ]*\\)" nil t nil)
	  (setq optional-name (concat (number-to-string (buffer-modified-tick)) "#" (match-string 1))))
      (set-buffer wysihtml-daemon-buffer)
      (insert (concat "http://localhost:" (number-to-string wysihtml-elserv-port) "/preview/index.html" optional-name ))
      (comint-send-input))))

(provide 'wysihtml)
