#include <gtk/gtk.h>
#include <gdk/gdkx.h>

#include "xmms/configfile.h"
#include "bump_scope.h"

static GtkWidget *configure_win = NULL;
static GtkWidget *vbox, *options_frame, *options_vbox;
static GtkWidget *options_colorpicker;
static GtkWidget *bbox, *ok, *cancel;

static gboolean config_read = FALSE;

static BumpScopeConfig bumpscope_cfg_old;

void bumpscope_read_config(void)
{
	ConfigFile *cfg;
	gchar *filename;

	if(!config_read)
	{
		bumpscope_cfg.color = 0x7ACCFF;
		bumpscope_cfg.color_cycle = FALSE;
		bumpscope_cfg.moving_light = TRUE;
		bumpscope_cfg.diamond = FALSE;
		bumpscope_cfg.auto_fullscreen = FALSE;

		bumpscope_cfg.width = 256;
		bumpscope_cfg.height = 128;
		bumpscope_cfg.phongrad = 256;

		bumpscope_cfg.x = bumpscope_cfg.width/2;
		bumpscope_cfg.y = bumpscope_cfg.height;

		filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
		cfg = xmms_cfg_open_file(filename);

		if (cfg)
		{
			xmms_cfg_read_int(cfg, "BumpScope", "color", &bumpscope_cfg.color);

			xmms_cfg_read_int(cfg, "BumpScope", "x", &bumpscope_cfg.x);
			xmms_cfg_read_int(cfg, "BumpScope", "y", &bumpscope_cfg.y);

			xmms_cfg_read_int(cfg, "BumpScope", "width", &bumpscope_cfg.width);
			xmms_cfg_read_int(cfg, "BumpScope", "height", &bumpscope_cfg.height);
			xmms_cfg_read_int(cfg, "BumpScope", "phongrad", &bumpscope_cfg.phongrad);

			xmms_cfg_read_boolean(cfg, "BumpScope", "color_cycle", &bumpscope_cfg.color_cycle);
			xmms_cfg_read_boolean(cfg, "BumpScope", "moving_light", &bumpscope_cfg.moving_light);
			xmms_cfg_read_boolean(cfg, "BumpScope", "diamond", &bumpscope_cfg.diamond);
			xmms_cfg_read_boolean(cfg, "BumpScope", "auto_fullscreen", &bumpscope_cfg.auto_fullscreen);
			xmms_cfg_free(cfg);
		}
		g_free(filename);
		config_read = TRUE;
	}
}

void bumpscope_write_cfg(void) {
	ConfigFile *cfg;
	gchar *filename;
	gdouble color[3];

	filename = g_strconcat(g_get_home_dir(), "/.xmms/config", NULL);
	cfg = xmms_cfg_open_file(filename);
	if (!cfg) cfg = xmms_cfg_new();

	if (configure_win) {
		gtk_color_selection_get_color(GTK_COLOR_SELECTION(options_colorpicker), color);
		bumpscope_cfg.color = ((guint32)(255.0*color[0])<<16) |
			              ((guint32)(255.0*color[1])<<8) |
		        	      ((guint32)(255.0*color[2]));
	}
	xmms_cfg_write_int(cfg, "BumpScope", "color", bumpscope_cfg.color);

	xmms_cfg_write_int(cfg, "BumpScope", "x", bumpscope_cfg.x);
	xmms_cfg_write_int(cfg, "BumpScope", "y", bumpscope_cfg.y);

	xmms_cfg_write_int(cfg, "BumpScope", "width", bumpscope_cfg.width);
	xmms_cfg_write_int(cfg, "BumpScope", "height", bumpscope_cfg.height);
	xmms_cfg_write_int(cfg, "BumpScope", "phongrad", bumpscope_cfg.phongrad);

	xmms_cfg_write_boolean(cfg, "BumpScope", "color_cycle", bumpscope_cfg.color_cycle);
	xmms_cfg_write_boolean(cfg, "BumpScope", "moving_light", bumpscope_cfg.moving_light);
	xmms_cfg_write_boolean(cfg, "BumpScope", "diamond", bumpscope_cfg.diamond);
	xmms_cfg_write_boolean(cfg, "BumpScope", "auto_fullscreen", bumpscope_cfg.auto_fullscreen);
	xmms_cfg_write_file(cfg, filename);
	xmms_cfg_free(cfg);
	g_free(filename);
}

static void configure_ok(GtkWidget *w, gpointer data)
{
	bumpscope_write_cfg();
	if (!bumpscope_cfg.color_cycle) {
		BUMP_LOCK()
		bumpscope_generate_cmap(bumpscope_cfg.color);
		BUMP_UNLOCK();
	}
	gtk_widget_destroy(configure_win);
}

static void configure_cancel(GtkWidget *w, gpointer data)
{
	bumpscope_cfg = bumpscope_cfg_old;
	if (!bumpscope_cfg.color_cycle) {
		BUMP_LOCK()
		bumpscope_generate_cmap(bumpscope_cfg.color);
		BUMP_UNLOCK();
	}
	gtk_widget_destroy(configure_win);
}

static void color_changed(GtkWidget *w, gpointer data)
{
	gdouble color[3];
	gtk_color_selection_get_color(GTK_COLOR_SELECTION(options_colorpicker), color);
	bumpscope_cfg.color = ((guint32)(255.0*color[0])<<16) |
		              ((guint32)(255.0*color[1])<<8) |
		              ((guint32)(255.0*color[2]));
	if (!bumpscope_cfg.color_cycle) {
		BUMP_LOCK()
		bumpscope_generate_cmap(bumpscope_cfg.color);
		BUMP_UNLOCK();
	}
}

static void bumpscope_toggle_cb(GtkWidget *w, gint *data) {
	*data = GTK_TOGGLE_BUTTON(w)->active;
}

static void bumpscope_diamond_cb(GtkWidget *w, gpointer data) {
	bumpscope_cfg.diamond = GTK_TOGGLE_BUTTON(w)->active;
	bumpscope_generate_phongdat();
}

static void bumpscope_w_h_cb(GtkWidget *w, gint *data) {
	*data = GTK_ADJUSTMENT(w)->value;

	/* The config dialog shouldn't be accessable when in full screen,
	   but even if it is, it'll only cause a minor cosmetic bug */
	bumpscope_win_w = WIDTH; bumpscope_win_h = HEIGHT;

	bumpscope_draw_xmms_logo();
	gtk_widget_set_usize(bumpscope_window, bumpscope_win_w, bumpscope_win_h);
}

static void bumpscope_phongrad_cb(GtkWidget *w, gint *data) {
	BUMP_LOCK();
	*data = GTK_ADJUSTMENT(w)->value;
	bumpscope_generate_phongdat();
	BUMP_UNLOCK();
}

void bumpscope_configure (void)
{
	gdouble color[3];
	GtkWidget *hbox, *check;
	GtkObject *adj;
	
	if (configure_win)
		return;

	bumpscope_read_config();
	bumpscope_cfg_old = bumpscope_cfg;

	color[0]=((gdouble)(bumpscope_cfg.color /0x10000))/256;
	color[1]=((gdouble)((bumpscope_cfg.color %0x10000)/0x100))/256;
	color[2]=((gdouble)(bumpscope_cfg.color %0x100))/256;

	configure_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_container_set_border_width(GTK_CONTAINER(configure_win), 10);
	gtk_window_set_title(GTK_WINDOW(configure_win), "Bump Scope Configuration");
	gtk_window_set_policy(GTK_WINDOW(configure_win), FALSE, FALSE, FALSE);
	gtk_window_set_position(GTK_WINDOW(configure_win), GTK_WIN_POS_MOUSE);
	gtk_signal_connect(GTK_OBJECT(configure_win), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			   &configure_win);

	vbox = gtk_vbox_new(FALSE, 5);

	options_frame = gtk_frame_new("Options:");
	gtk_container_set_border_width(GTK_CONTAINER(options_frame), 5);

	options_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(options_vbox), 5);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(options_vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	check = gtk_check_button_new_with_label("Colors cycle");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), bumpscope_cfg.color_cycle);
	gtk_signal_connect(GTK_OBJECT(check), "toggled",
		GTK_SIGNAL_FUNC(bumpscope_toggle_cb), &bumpscope_cfg.color_cycle);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	check = gtk_check_button_new_with_label("Moving light");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), bumpscope_cfg.moving_light);
	gtk_signal_connect(GTK_OBJECT(check), "toggled",
		GTK_SIGNAL_FUNC(bumpscope_toggle_cb), &bumpscope_cfg.moving_light);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	check = gtk_check_button_new_with_label("Diamond light");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), bumpscope_cfg.diamond);
	gtk_signal_connect(GTK_OBJECT(check), "toggled",
		GTK_SIGNAL_FUNC(bumpscope_diamond_cb), NULL);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	check = gtk_check_button_new_with_label("Fullscreen on play");
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(check), bumpscope_cfg.auto_fullscreen);
	gtk_signal_connect(GTK_OBJECT(check), "toggled",
		GTK_SIGNAL_FUNC(bumpscope_toggle_cb), &bumpscope_cfg.auto_fullscreen);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	if (!xmms_fullscreen_available(GDK_DISPLAY()))
		gtk_widget_set_sensitive(check, FALSE);
	gtk_widget_show(check);

	hbox = gtk_hbox_new(FALSE, 5);
	gtk_box_pack_start(GTK_BOX(options_vbox), hbox, TRUE, TRUE, 0);
	gtk_widget_show(hbox);

	adj = gtk_adjustment_new(bumpscope_cfg.width, 64, MAX_WIDTH, 1, 1, 0);
	gtk_signal_connect(GTK_OBJECT(adj), "value-changed", GTK_SIGNAL_FUNC(bumpscope_w_h_cb), &bumpscope_cfg.width);
	check = gtk_label_new("Width:");
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
	check = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 8, 0);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	adj = gtk_adjustment_new(bumpscope_cfg.height, 32, MAX_HEIGHT, 1, 1, 0);
	gtk_signal_connect(GTK_OBJECT(adj), "value-changed", GTK_SIGNAL_FUNC(bumpscope_w_h_cb), &bumpscope_cfg.height);
	check = gtk_label_new("Height:");
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
	check = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 8, 0);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	adj = gtk_adjustment_new(bumpscope_cfg.phongrad, 8, MAX_PHONGRES/2, 1, 1, 0);
	gtk_signal_connect(GTK_OBJECT(adj), "value-changed", GTK_SIGNAL_FUNC(bumpscope_phongrad_cb), &bumpscope_cfg.phongrad);
	check = gtk_label_new("Light radius:");
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);
	check = gtk_spin_button_new(GTK_ADJUSTMENT(adj), 8, 0);
	gtk_box_pack_start(GTK_BOX(hbox), check, TRUE, TRUE, 0);
	gtk_widget_show(check);

	options_colorpicker = gtk_color_selection_new();
	gtk_color_selection_set_color(GTK_COLOR_SELECTION(options_colorpicker), color);
	gtk_signal_connect(GTK_OBJECT(options_colorpicker), "color_changed", GTK_SIGNAL_FUNC(color_changed), NULL);

	gtk_box_pack_start(GTK_BOX(options_vbox), options_colorpicker, FALSE, FALSE, 0);
        gtk_widget_show(options_colorpicker);


	gtk_container_add(GTK_CONTAINER(options_frame), options_vbox);
	gtk_widget_show(options_vbox);

	gtk_box_pack_start(GTK_BOX(vbox), options_frame, TRUE, TRUE, 0);
	gtk_widget_show(options_frame);

	bbox = gtk_hbutton_box_new();
	gtk_button_box_set_layout(GTK_BUTTON_BOX(bbox), GTK_BUTTONBOX_END);
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bbox), 5);
	gtk_box_pack_start(GTK_BOX(vbox), bbox, FALSE, FALSE, 0);

	ok = gtk_button_new_with_label("Ok");
	gtk_signal_connect(GTK_OBJECT(ok), "clicked",
			   GTK_SIGNAL_FUNC(configure_ok), NULL);
	GTK_WIDGET_SET_FLAGS(ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), ok, TRUE, TRUE, 0);
	gtk_widget_show(ok);


	cancel = gtk_button_new_with_label("Cancel");
	gtk_signal_connect(GTK_OBJECT(cancel), "clicked",
			   GTK_SIGNAL_FUNC(configure_cancel), NULL);
	GTK_WIDGET_SET_FLAGS(cancel, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bbox), cancel, TRUE, TRUE, 0);
	gtk_widget_show(cancel);
	gtk_widget_show(bbox);

	gtk_container_add(GTK_CONTAINER(configure_win), vbox);
	gtk_widget_show(vbox);
	gtk_widget_show(configure_win);
	gtk_widget_grab_default(ok);
}
