
#ifndef __XPTRLIST_H__
#define __XPTRLIST_H__

// By Andrew O'Meara


// Post:	Returns a num >= 0 iff the first arg is less than or equal to the second arg.
//			Returns a num < 0 iff the first arg is greater than the second arg.
typedef int (*CompFunctionT)(const void*, const void*, const void*  );
typedef int (*ANSI_QSCompFcnT)(const void*, const void*);

enum ListOrderingT {
	cOrderImportant,
	cOrderNotImportant,
	cSorted
};

class XPtrList  {

		friend class XFloatList;
		friend class XLongList;
		friend class XStrList;

	public:
								XPtrList( const XPtrList& inList );
								XPtrList( ListOrderingT inOrdering = cOrderNotImportant );

		virtual					~XPtrList();

		//	This determines how all items in this list will be ordered.
		//	Note:	If you turn on sorting, all items in this list will be resorted.
		//	Note:	SetCompFcn() must be called before you turn on sorting.
		virtual void			SetListOrdering( ListOrderingT inOrdering );

		//	Post:	Assigns this from <inList>.
		void					Assign( const XPtrList& inList );

		//	Post:	Adds the ptr to this' list
		//	Order:	O( 1 ), O( log n ) if sorting is on.
		//	Note:	Returns what index the new element occupies (1-based indexing).
		long					Add( const void* inPtrToAdd );

		//	Post:	Adds the ptr to this' list after (existing) ptr number inN--make inN 0 if you want your ptr to be the 1st element
		//	Order:	O( 1 )
		//	Note:	If inN is invalid, the closest element is used
		void					Add( const void* inPtrToAdd, long inN );

		//	Post:	Adds each element of the incoming list to this list.  This is functionally the same as
		//			looping thru the size of inPtrList and calling this.Add() for each element.
		void					Add( const XPtrList& inPtrList );

		//	Post:	Searches for the ptr in the list and removes it if found.
		//	Note:	<true> is returned if the ptr was found (and removed)
		//	Order:	O( N ), O( log n + alpha n ) if sorting is on.
		bool					Remove( const void* inPtrToRemove );

		//	Note:	<true> is returned if the element was found (and removed)
		//	Order:	O( N ), O( log n ) if sorting is on.
		//	Note:	When order is set to be preserved (via cOrderImportant or a specificed sort fcn), elements
		//			at the end of the list are removed faster
		bool					RemoveElement( long inIndex );
		bool					RemoveLast();

		//	Post:	Empties the ptr list
		void					RemoveAll();

		//	Post:	Used to fetch any given ptr. If the index exists, <true> is returned and the ptr is copied to *inPtrDest.
		//	Note:	One based indexing is used
		//	Order:	O( 1 )
		void*					Fetch( long inIndex ) const;
		bool					Fetch( long inIndex, void** ioPtrDest ) const;
		inline bool				FetchLast( void** ioPtrDest ) const									{ return Fetch( Count(), ioPtrDest );			}

		//	Same as Fetch() except that an element can be accessed by all its wrapped mods.  For example, in a 10 element
		//  array, the 2nd element can be accessed using: ... -38, -28, -18, -8,  2, 12, 22, 33, 42, ...
		void*					FetchWrapped( long inIndex ) const;

		// 	Allows easy dynamic array usage.  Simply use any index and XPtrList will expand to meet that size.
		//	Impt:	Zero based indexing is used here! (In contrast to Fetch())
		//	Note:	Elements that are newly accessed are initialized to NULL
		//	Note:	Indexs below 0 lead to sDummy;
		//	Note:	Since caller has access to changes values, any current sorting fcn is not used
		void*&					operator[] ( const long inIndex );

		//	Post:	Moves element at <inIndex> so that it has index 1.
		// 	Note:	If sorting is on, it will be turned off (b/c this will move items out of sorted order) and ordering will be set to cOrderImportant
		void					MoveToHead( long inIndex );

		//	Post:	Searches for the given ptr in its ptr list and returns the index of the match
		//			If the item cannot be found, 0 is returned
		//	Note:	One based indexing is used
		//	Order:	O( N ), or O( log n ) if sorting is on.
		long					FindIndexOf( const void* inMatchPtr ) const;

		//	Post:	Returns the number of items in this list.
		//	Order:	O( 1 )
		inline long				Count() const														{ return mNumElements;		}

		//	Post:	All ptrs in this list are randomly reordered.
		void					Randomize();

		//	Allows elements to be compared.  Allows you use Rank(), enable sorting, and more rapidly find elements.
		//  Pass NULL for inFcn if you wish to deinstall the comp fcn.
		void					SetCompFcn( CompFunctionT inFcn, void* inCompParam );

		//	Pre:  mCompFcn is assigned
		//	Post: Ranks all the values in this list.
		//	Post: Fetch( outRank[ i ] ) is the ith largest value in this list.
		//	Post: outRank.Count() == inNumToRank  (ie, only inNumToRank values of the ranking are returned)
		//	Note: If inNumToRank is invalid, the full ranking is returned
		//	Note: O( N log N ) running time if ordering isn't cSorted, O( N ) is ordering is cSorted.
		void					Rank( XLongList& outRank, long inNumToRank = -1 ) const;

		inline ListOrderingT	GetOrder() const													{ return mOrdering;			}

		//	Prepares this list for the given number of elements to avoid repeated resizing
		void					Dim( long inNumElements );

	protected:

		void**					mBuf;
		long					mDimElements;
		long					mNumElements;

		ListOrderingT			mOrdering;
		CompFunctionT			mCompFcn;
		void*					mCompParam;

		//	Pre:	The list of nums must be sorted
		//	Post:	Returns the (one based) index of the predicessor of <inNum>.  If zero is returned,
		//			then <inNum> is <= than all the elements in the list.
		long					FetchPredIndex( const void* inNum ) const;

		static void*			sDummy;

		static int				sRankComparitor( const void* inA, const void* inB, const void* inThis );

		static void				QuickSort( void** inBuf, long inNumElements, CompFunctionT inCompFcn, const void* inCompPtr );
};

#endif // __XPTRLIST_H__
