/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#ifndef __EDITOR_QUERY_HTTP_H__
#define __EDITOR_QUERY_HTTP_H__

#include <glib.h>

#include "editor_query_parser.h"

typedef struct _HTTPProxy {

	gchar* name;
	gint   port;
	gchar* user;
	gchar* passwd;
}
HTTPProxy;

typedef struct _LyrixServer {

	gchar *name;
	gint   port;

	// The pages on the servers, which handle the requests
	gchar *search_uri;
	gchar *get_uri;
	gchar *add_uri;
}
LyrixServer;

typedef struct _UserAgent {

	gchar *program;
	gchar *version;
}
UserAgent;

// * The request http sruct collects all "static" data about
// * the http connection to a lyrix server
typedef struct _LyrixRequestHTTP {

	gint         is_set;
	gint         ref;

	LyrixServer  server;
	HTTPProxy    proxy;
	UserAgent    agent;
}
LyrixRequestHTTP;

typedef enum _SearchMatch {

	MATCH_PERFECT = 1,
	MATCH_FIRST   = 2,
	MATCH_ALL     = 3,
	MATCH_ANY     = 4,

	MATCH_LAST    = 5, // Don't use this
}
SearchMatch;

// This struct descripes a lyrix server request
// The possible request types are: SEARCH / GET / ADD
// For better overview their specific data is grouped inside a union
typedef struct _LyrixRequest {

	// Request
	gint request_type;

	gchar *artist;
	gchar *album;
	gchar *title;

	union {
		struct _search {

			SearchMatch artist_match;
			SearchMatch title_match;
			gint        page;
		}
		search;

		struct _get {

			gint   db_id;
		}
		get;

		struct _add {

			gchar *author;
			gchar *lyric_stream;
		}
		add;
	} u;

	// Server data
	gboolean          use_proxy;
	gchar            *redirect;
	LyrixRequestHTTP *http_data;
}
LyrixRequest;

// * The following funtions are used to handle the server data object.
// * The programming policy is to create an object for every server
// * and exchange the server object of the request via the link funtions
// * lyrix_request_(un)link_data
LyrixRequestHTTP* lyrix_request_data_new  (void);
void              lyrix_request_data_free (LyrixRequestHTTP* http_data);

void lyrix_request_data_set_server (LyrixRequestHTTP *http_data,
	const gchar *name, gint port, const gchar *search_uri,
	const gchar *get_uri, const gchar *add_uri);
void lyrix_request_data_set_proxy  (LyrixRequestHTTP *http_data,
	const gchar *name, gint port, const gchar *user, const gchar *passwd);
void lyrix_request_data_set_agent  (LyrixRequestHTTP *http_data,
	const gchar *program, const gchar *version);

gboolean lyrix_request_data_is_valid(LyrixRequestHTTP *http_data);

// * The following functions handle the real request
// * The link functions are used to assign a http server object to a request.
// * In this way it is easy to do a request to several servers
// * Be aware that the server objects have to be freed at program shutdown!
// * For more information see the comment for lyrix_request_data_new
LyrixRequest*     lyrix_request_new  (void);
void              lyrix_request_free (LyrixRequest* req);

void lyrix_request_use_proxy
	(LyrixRequest *req, gboolean use);
void lyrix_request_set_redirect
	(LyrixRequest *req, const gchar *redirect);
void lyrix_request_link_http_data
	(LyrixRequest *req, LyrixRequestHTTP *data);
void lyrix_request_unlink_http_data
	(LyrixRequest *req);

// * Prepare the request structure for a request type
void lyrix_request_set_lyrics_info(LyrixRequest *req,
	const gchar *artist, const gchar *album, const gchar* title);

void lyrix_request_set_search (LyrixRequest *req,
	SearchMatch artist, SearchMatch title);

void lyrix_request_set_get    (LyrixRequest *req, gint db_id);

void lyrix_request_set_add
	(LyrixRequest *req, const gchar *author, const gchar *stream);

void lyrix_request_set_add_from_search
	(LyrixRequest *req, const gchar *artist, const gchar *album,
	const gchar* title, const gchar *author, const gchar *stream);

// Most importamt function, which actually executes the query
gboolean lyrix_request_do_query
	(LyrixRequest *req, LyrixQueryResult **res, HTTPProxy *proxy);

gboolean lyrix_request_is_valid(LyrixRequest *req);

#endif // __EDITOR_QUERY_HTTP_H__
