/*
 * Copyright (C) 2003 Olivier Chapuis
 *
 *
 */

/*
 * Generic rootless to Aqua specific glue code
 *
 * This code acts as a glue between the generic rootless X server code
 * and the Aqua specific implementation, which includes definitions that
 * conflict with stardard X types.
 *
 * Greg Parker     gparker@cs.stanford.edu
 */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/types.h>
#include <sys/ipc.h> 
#include <sys/shm.h>
#include <errno.h>

#include "rootlessConfig.h"
#include "rootlessCommon.h"
#include "rootless.h"

#include "regionstr.h"
#include "scrnintstr.h"
#include "servermd.h"
#include "globals.h" // dixScreenOrigins[]
#include "propertyst.h"

#include <X11/X.h>
#include <X11/Xatom.h>

#include "rfb.h"
#include "mipointer.h"

#define TRACE_RFBROOTLESS 0

/*
 * Private data of the rootless glue.
 */

typedef struct _WindowClients {
	struct _WindowClients *next;

	rfbClientPtr cl;

	RegionRec modifiedRegion;
	int shapeChanged;

	Bool firstTime;
	Bool updateRequested;

} WindowClients;

struct _rfbWindowRec {
	rfbWindowRec *next;      /* doubly linked list */

	WindowClients *clients;

	char *pixelData;
	int bytesPerRow;

	RegionRec bShape;  /* bounding shape for shaped window */

	RootlessWindowPtr frame;     /* backpointer to frame */
};


/*
 * List of all Windows created by rfb.
 */
static rfbWindowRec *Windows = NULL;


/*
 * manage window client
 */
static
WindowClients *add_client(WindowClients *clients, rfbClientPtr cl)
{
	WindowClients *t,*prev = NULL;
	WindowClients *new;
	//BoxRec box;

	t = clients;
	while(t != NULL)
	{
		if (t->cl == cl)
		{
			/* it's already there, do nothing */
			return clients;
		}
		prev = t;
		t = t->next;
	}

	new = (WindowClients *)malloc(sizeof(WindowClients));
	new->cl = cl;
	new->next = NULL;

	REGION_INIT(pScreen,&new->modifiedRegion,NullBox,0);

	new->firstTime = TRUE;
	new->shapeChanged = 0;
	new->updateRequested = FALSE;

	if (clients == NULL)
	{
		return new;
	}
	prev->next = new;
	return clients;
}

static
WindowClients *remove_client(WindowClients *clients, rfbClientPtr cl) {
	WindowClients  *tl = clients;
	WindowClients *prev = NULL;

	while (tl && tl->cl != cl)
	{
		prev = tl;
		tl = tl->next;
	}

	if (tl == NULL)
	{
		return clients;
	}

	if (prev)
	{
		prev->next = tl->next;
	}
	if (clients == tl)
	{
		clients = clients->next;
	}
	free(tl);

	return clients;
}

static
WindowClients *find_client(rfbWindowRec *rw, rfbClientPtr cl)
{
	WindowClients *t = rw->clients;

	while(t!= NULL && t->cl != cl)
	{
		t = t->next;
	}
	return t;
}

/* ************************************************************************* */

static
rfbWindowRec *add_window(RootlessWindowPtr pFrame) {
	rfbWindowRec *t,**prev;
	rfbClientPtr cl;

	t = Windows;
	prev = &Windows;

	while(t!= NULL)
	{
		if (t->frame == pFrame)
		{
			/* it's already there, do nothing */
			return NULL;
		}
		prev = &(t->next);
		t = t->next;
	}
	*prev = (rfbWindowRec *)malloc(sizeof(rfbWindowRec));
	(*prev)->frame = pFrame;
	(*prev)->clients = NULL;
	(*prev)->pixelData = NULL;
	(*prev)->next = NULL;
	for (cl = rfbClientHead; cl; cl = cl->next)
	{
		(*prev)->clients = add_client((*prev)->clients, cl);
	}
	return (*prev);
}

static
void remove_window(rfbWindowRec *rw)
{
	rfbWindowRec *t,**prev;

	t = Windows;
	prev = &Windows;
	while((t!= NULL)&&(t != rw))
	{
		prev = &(t->next);
		t = t->next;
	}
	if(t!= NULL)
	{
		if(prev != NULL)
		{
			*prev = t->next;
		}

		while(t->clients)
		{
			t->clients = remove_client(t->clients, t->clients->cl);
		}
		free(t);
	}
}

static
unsigned long getrfbFlags(WindowPtr w, Window *transientFor)
{
	unsigned long flags = 0;
	PropertyPtr pProp;
	static Atom netCheckingAtom = None;
	static Atom netWindowTypeAtom = None;
	static Atom netEwmhDesktopAtom = None;

	if (w->overrideRedirect)
	{
		flags |= rfbWindowFlagsOverrideRedirect;
		flags |= rfbWindowFlagsUnmanaged;
	}
	if (w->drawable.class == InputOnly)
	{
		flags |= rfbWindowFlagsInputOnly;
	}
	/* FIXME ? */
	if (!w->firstChild)
	{
		flags |= rfbWindowFlagsUnmanaged;
	}

	if (netCheckingAtom == None)
	{
		netCheckingAtom = MakeAtom(
			"_NET_SUPPORTING_WM_CHECK", 24, TRUE);
	}
	if (netWindowTypeAtom == None)
	{
		netWindowTypeAtom = MakeAtom(
			"_NET_WM_WINDOW_TYPE", 19, TRUE);
	}
	if (netEwmhDesktopAtom == None)
	{
		netEwmhDesktopAtom = MakeAtom(
			"_NET_WM_WINDOW_TYPE_DESKTOP", 27, TRUE);
	}

	if (w->firstChild)
	{
		if (w->firstChild->firstChild)
		{
			pProp = wUserProps(w->firstChild->firstChild);
		}
		else
		{
				pProp = wUserProps(w->firstChild);
		}
	}
	else
	{
		pProp = wUserProps(w);
	}

	while(pProp)
	{
		if (pProp->propertyName == netCheckingAtom)
		{
			flags |= rfbWindowFlagsNetChecking;
		}
		if (pProp->propertyName == netWindowTypeAtom)
		{
			int i = 0;
			int found = 0;
			int size = (pProp->format/8) * pProp->size;
			Atom *data = (Atom *)pProp->data;
			while(i < size && !found)
			{
				if (data[i] == netEwmhDesktopAtom)
				{
					found = 1;
				}
				i++;
			}
			if (found)
			{
				flags |= rfbWindowFlagsEwmhDesktop;
			}
		}
		if (pProp->propertyName == XA_WM_TRANSIENT_FOR && pProp->size)
		{
			Window *data = (Window *)pProp->data;
			Window w;
			WindowPtr transWin;
			WindowPtr topTransWin;

			w = data[0];
			transWin = (WindowPtr)SecurityLookupIDByType(
				NullClient, w, RT_WINDOW, SecurityReadAccess);
			if (transWin)
			{
				flags |= rfbWindowFlagsTransient;
				topTransWin = TopLevelParent(transWin);
				*transientFor = topTransWin->drawable.id;
			}
		}
		pProp = pProp->next;
	}
	return flags;
}

/* ************************************************************************** 
 *
 *
 *
 * ************************************************************************** */

void window_add_client(rfbClientPtr cl) {
	rfbWindowRec *w;

	for(w = Windows; w != NULL; w = w->next) {
		w->clients = add_client(w->clients, cl);
	}
}

void window_remove_client(rfbClientPtr cl) {
	rfbWindowRec *w;

	for(w = Windows; w != NULL; w = w->next) {
		w->clients = remove_client(w->clients, cl);
	}
}

void window_set_updateRequested(rfbClientPtr cl)
{
	rfbWindowRec *w;
	WindowClients *c;

	for(w = Windows; w != NULL; w = w->next) {
		if ((c = find_client(w, cl)) != NULL)
		{
			c->updateRequested = TRUE;
		}
	}
}

void window_union_modifiedRegion(
	rfbClientPtr cl, rfbWindowRec *w, RegionRec *reg)
{
	WindowClients *c;

	if ((c = find_client(w, cl)) == NULL)
	{
		return;
	}
	REGION_UNION(pScreen,&c->modifiedRegion,&c->modifiedRegion, reg);
}

void window_union_modifiedRegion_allwin(rfbClientPtr cl, RegionRec *reg)
{
	rfbWindowRec *w;

	for(w = Windows; w != NULL; w = w->next)
	{
		window_union_modifiedRegion(cl, w, reg);
	}
}

void
window_region_uninit(rfbClientPtr cl)
{
	rfbWindowRec *w;
	WindowClients *c;

	for(w = Windows; w != NULL; w = w->next) {
		if ((c = find_client(w, cl)) != NULL)
		{
			REGION_UNINIT(pScreen, &c->modifiedRegion);
		}
	}
}

void
window_region_init(rfbClientPtr cl, BoxRec *box)
{
	rfbWindowRec *w;
	WindowClients *c;

	for(w = Windows; w != NULL; w = w->next) {
		if ((c = find_client(w, cl)) != NULL)
		{
			REGION_INIT(pScreen, &c->modifiedRegion,box,0);
		}
	}
}

/* ************************************************************************** 
 *
 *
 *
 * ************************************************************************** */

static
Bool window_MakeWindowShapeUpdate(
	rfbClientPtr cl, ScreenPtr pScreen,  rfbWindowRec *rw)
{
	rfbFramebufferUpdateRectHeader rect;
	rfbWindowShapeHeader sh;
	unsigned int nRects;
	int i,size;
	char *buf;
	CARD16 *data;

	rect.encoding = Swap32IfLE(rfbEncodingWindowShape);

	/* not useful any way ? */
	rect.r.x = Swap16IfLE(0);
	rect.r.y = Swap16IfLE(0);
	rect.r.w = Swap16IfLE(rw->frame->width);
	rect.r.h = Swap16IfLE(rw->frame->height);

	memcpy(
		&updateBuf[ublen],(char *)&rect,
		sz_rfbFramebufferUpdateRectHeader);

	ublen += sz_rfbFramebufferUpdateRectHeader;

	cl->rfbWindowShapeUpdatesSent++;

	nRects = REGION_NUM_RECTS(&rw->bShape);
	
	sh.nrects = Swap32IfLE(nRects);

	memcpy(&updateBuf[ublen],(char *)&sh, sz_rfbWindowShapeHeader);

	ublen += sz_rfbWindowShapeHeader;

	if (!rfbSendUpdateBuf(cl))
	{
		return FALSE;
	}

	if (nRects == 0)
	{
		return TRUE;
	}

	size = nRects * 4 * sizeof(CARD16);

	buf = (char *)malloc(size);
	data = (CARD16 *)buf;
	
	/* FIXME: size > UPDATE_BUF_SIZE */
	for (i = 0; i < REGION_NUM_RECTS(&rw->bShape); i++)
	{
		CARD16 x = (CARD16)REGION_RECTS(&rw->bShape)[i].x1;
		CARD16 y = (CARD16)REGION_RECTS(&rw->bShape)[i].y1;
		CARD16 w = (CARD16)REGION_RECTS(&rw->bShape)[i].x2 - x;
		CARD16 h = (CARD16)REGION_RECTS(&rw->bShape)[i].y2 - y;
		
		if (cl->format.bigEndian != rfbServerFormat.bigEndian)
		{
			data[i*4] = Swap16(x);
			data[i*4+1] = Swap16(y);
			data[i*4+2] = Swap16(w);
			data[i*4+3] = Swap16(h);
		}
		else
		{
			data[i*4] = x;
			data[i*4+1] = y;
			data[i*4+2] = w;
			data[i*4+3] = h;
		}
	}
	
	memcpy(&updateBuf[ublen],(char *)buf, size);
	ublen += size;

	cl->rfbWindowShapeBytesSent += size;

	if (!rfbSendUpdateBuf(cl))
	{
		return FALSE;
	}

	return TRUE;
}


static
Bool window_SendWindowShapeUpdate(WindowClients *c, rfbWindowRec *rw)
{
	ScreenPtr pScreen = screenInfo.screens[0];
	rfbFramebufferUpdateMsg *fu = (rfbFramebufferUpdateMsg *)updateBuf;

	if (!((c->cl)->enableShapeUpdate && c->shapeChanged))
	{
		return TRUE;
	}

	if ((c->cl)->enableShapeUpdate && c->shapeChanged)
	{
		/* ok */
	}
	else
	{
		return TRUE;
	}

	c->shapeChanged = 0;
	c->cl->rfbFramebufferUpdateMessagesSent++;
	
	fu->type = rfbFramebufferUpdate;
	fu->window = Swap32IfLE((CARD32)rw->frame->win->drawable.id);
	if (IsRoot(rw->frame->win))
		fu->topWindow = 0;
	else
		fu->topWindow = Swap32IfLE((CARD32)rw->frame->win->drawable.id);

	fu->nRects = Swap16IfLE(1);
	fu->shmid = 0;
	fu->pad = 0;

	ublen = sz_rfbFramebufferUpdateMsg;

	if (!window_MakeWindowShapeUpdate(c->cl, pScreen,  rw))
	{
		return FALSE;
	}

	return TRUE;
}

static
void window_shapeChanged(rfbWindowRec *w)
{
	WindowClients *c;
	rfbClientPtr cl;

	for (cl = rfbClientHead; cl; cl = cl->next)
	{
		if ((c = find_client(w, cl)) == NULL)
		{
			continue;
		}
		c->shapeChanged = 1;
		window_SendWindowShapeUpdate(c, w);
	}
}


/* ************************************************************************** 
 *
 *
 *
 * ************************************************************************** */

static
Bool my_rfbSendFramebufferUpdate(WindowClients *c, rfbWindowRec *rw)
{
    ScreenPtr pScreen = screenInfo.screens[0];
    int i;
    int nUpdateRegionRects;
    rfbFramebufferUpdateMsg *fu = (rfbFramebufferUpdateMsg *)updateBuf;
    RegionRec updateRegion;
    Bool sendCursorShape = FALSE;
    Bool sendCursorPos = FALSE;
    Bool sendWindowShape = FALSE;
    int shmid = 0;
    char *shm = NULL;
    unsigned int shmsize = 0;

    if (c->cl->enableCursorShapeUpdates)
    {
	    if (!rfbScreen.cursorIsDrawn && c->cl->cursorWasChanged)
	    {
		    sendCursorShape = TRUE;
	    }
    }


    if (c->cl->enableCursorPosUpdates && c->cl->cursorWasMoved)
    {
	    sendCursorPos = TRUE;
    }

    if ((c->cl)->enableShapeUpdate && c->shapeChanged)
    {
	    sendWindowShape = TRUE;
    }

    REGION_INIT(pScreen,&updateRegion,NullBox,0);
    REGION_UNION(pScreen, &updateRegion, &updateRegion, &c->modifiedRegion);

    if (!(REGION_NOTEMPTY(pScreen,&updateRegion) && c->updateRequested) &&
	!sendCursorShape && !sendCursorPos)
    {
	    REGION_UNINIT(pScreen,&updateRegion);
	    return TRUE;
    }

    /*
     * Now send the update.
     */

    c->cl->rfbFramebufferUpdateMessagesSent++;

    nUpdateRegionRects = 0;

    if (REGION_NOTEMPTY(pScreen,&updateRegion) && c->updateRequested)
    {
	    if (c->cl->preferredEncoding == rfbEncodingCoRRE) {

		    for (i = 0; i < REGION_NUM_RECTS(&updateRegion); i++) {
			    int x = REGION_RECTS(&updateRegion)[i].x1;
			    int y = REGION_RECTS(&updateRegion)[i].y1;
			    int w = REGION_RECTS(&updateRegion)[i].x2 - x;
			    int h = REGION_RECTS(&updateRegion)[i].y2 - y;
			    nUpdateRegionRects += 
				    (((w-1) / c->cl->correMaxWidth + 1)
				     * ((h-1) / c->cl->correMaxHeight + 1));
		    }
	    } else {
		    nUpdateRegionRects = REGION_NUM_RECTS(&updateRegion);
	    }
    }

    fu->type = rfbFramebufferUpdate;
    fu->window = Swap32IfLE((CARD32)rw->frame->win->drawable.id);
    if (IsRoot(rw->frame->win))
    {
	    fu->topWindow = 0;
    }
    else
    {
	    fu->topWindow = Swap32IfLE((CARD32)rw->frame->win->drawable.id);
    }
    if (nUpdateRegionRects != 0xFFFF)
    {
	    fu->nRects = Swap16IfLE(
		    nUpdateRegionRects + !!sendCursorShape + !!sendCursorPos +
		    !!sendWindowShape);
    }
    else
    {
	    fu->nRects = 0xFFFF;
    }

    fu->pad = 0;
    fu->shmid = 0;

    if (c->cl->preferredEncoding == rfbEncodingRawShm &&
	REGION_NOTEMPTY(pScreen,&updateRegion) && c->updateRequested)
    {
	    for (i = 0; i < REGION_NUM_RECTS(&updateRegion); i++)
	    {
		    int x = REGION_RECTS(&updateRegion)[i].x1;
		    int y = REGION_RECTS(&updateRegion)[i].y1;
		    int w = REGION_RECTS(&updateRegion)[i].x2 - x;
		    int h = REGION_RECTS(&updateRegion)[i].y2 - y;
		    
		    shmsize += (sz_rfbFramebufferUpdateRectHeader
			     + w * (c->cl->format.bitsPerPixel / 8) * h);
	    }

	    shmsize += sizeof(CARD16);

	    if ((shmid = shmget(IPC_PRIVATE, shmsize, IPC_CREAT | 0600)) < 0)
	    {
		    perror("shmget");
		    shmid = 0;
	    }

	    if (shmid > 0 && (shm = shmat(shmid, NULL, 0)) == (char *)(-1))
	    {
		perror("shmat");
		shmctl(shmid, IPC_RMID, 0);
		shmid = 0;
	    }

	    if (shmid > 0)
	    {
		    fu->nRects = Swap16IfLE(
			    !!sendCursorShape + !!sendCursorPos +
			    !!sendWindowShape);
		    fu->shmid = Swap32IfLE(shmid);
		    
		    c->cl->rfbRectanglesSent[rfbEncodingRawShm]
			    += nUpdateRegionRects;
		    c->cl->rfbBytesSent[rfbEncodingRawShm] += shmsize;
		    c->cl->rfbRawBytesEquivalent += shmsize;
	    }
    }

    ublen = sz_rfbFramebufferUpdateMsg;

    if (shmid > 0)
    {
	    char *s = shm;
	    rfbFramebufferUpdateRectHeader rect;

	    nUpdateRegionRects = Swap16IfLE(nUpdateRegionRects);
	    memcpy(s, &nUpdateRegionRects, sizeof(CARD16));
	    s += sizeof(CARD16);
	    
	    for (i = 0; i < REGION_NUM_RECTS(&updateRegion); i++)
	    {
		    int x = REGION_RECTS(&updateRegion)[i].x1;
		    int y = REGION_RECTS(&updateRegion)[i].y1;
		    int w = REGION_RECTS(&updateRegion)[i].x2 - x;
		    int h = REGION_RECTS(&updateRegion)[i].y2 - y;
		    int bytesPerLine = w * (c->cl->format.bitsPerPixel / 8);
		    char *fbptr =
			    (rw->pixelData + (rw->bytesPerRow * y) +
			     (x * (rfbScreen.bitsPerPixel / 8)));

#if 0
		    fprintf(stderr,"%i %i %ix%i\n", x,y,w,h);
#endif
		    rect.r.x = Swap16IfLE(x);
		    rect.r.y = Swap16IfLE(y);
		    rect.r.w = Swap16IfLE(w);
		    rect.r.h = Swap16IfLE(h);
		    rect.encoding = Swap32IfLE(rfbEncodingRawShm);

		    memcpy(s, (char *)&rect, sz_rfbFramebufferUpdateRectHeader);

		    s += sz_rfbFramebufferUpdateRectHeader;

		    (*c->cl->translateFn)(
			    c->cl->translateLookupTable,
			    &rfbServerFormat,
			    &c->cl->format, fbptr, s, rw->bytesPerRow,
			    w, h);
		    s += bytesPerLine * h;
	    }

	    /* force shm to be resident (cygwin do not have SHM_LOCK) */
#ifdef SHM_LOCK
	    shmctl(shmid,SHM_LOCK,0);
#endif
	    /* now the viewer is responsible of the segment */
	    shmdt(shm);
    }

    if (sendCursorShape) {
	c->cl->cursorWasChanged = FALSE;
	if (!rfbSendCursorShape(c->cl, pScreen))
	    return FALSE;
    }

    if (sendCursorPos) {
	    c->cl->cursorWasMoved = FALSE;
	    if (!rfbSendCursorPos(c->cl, pScreen))
		    return FALSE;
    }

    if (sendWindowShape) {
	    c->shapeChanged = 0;
	    window_MakeWindowShapeUpdate(c->cl, pScreen, rw);
    }

#if TRACE_RFBROOTLESS
    fprintf(stderr, "Send 0x%lx (%lu) / %p (%u)\n",
	    rw->frame->win->drawable.id, REGION_NUM_RECTS(&updateRegion),
	    rw->pixelData, rw->bytesPerRow);
#endif

    if (!(REGION_NOTEMPTY(pScreen,&updateRegion) && c->updateRequested))
    {
	    REGION_UNINIT(pScreen,&updateRegion);
	    if (!rfbSendUpdateBuf(c->cl))
		    return FALSE;
	    return TRUE;
    }
    
    c->updateRequested = FALSE;
    REGION_SUBTRACT(
	    pScreen, &c->modifiedRegion, &c->modifiedRegion, &updateRegion);

    if (shmid > 0)
    {
	    REGION_UNINIT(pScreen,&updateRegion);
	    if (!rfbSendUpdateBuf(c->cl))
		    return FALSE;
	    return TRUE;
    }

    for (i = 0; i < REGION_NUM_RECTS(&updateRegion); i++) {
	int x = REGION_RECTS(&updateRegion)[i].x1;
	int y = REGION_RECTS(&updateRegion)[i].y1;
	int w = REGION_RECTS(&updateRegion)[i].x2 - x;
	int h = REGION_RECTS(&updateRegion)[i].y2 - y;

	c->cl->rfbRawBytesEquivalent +=
		(sz_rfbFramebufferUpdateRectHeader
		 + w * (c->cl->format.bitsPerPixel / 8) * h);

	switch (c->cl->preferredEncoding) {
	case rfbEncodingRawShm:
	case rfbEncodingRaw:
	    if (!wncSendRectEncodingRaw(
			c->cl, rw->pixelData, rfbScreen.bitsPerPixel,
			rw->bytesPerRow, x, y, w, h)) {
		REGION_UNINIT(pScreen,&updateRegion);
		return FALSE;
	    }
	    break;
	case rfbEncodingCoRRE:
	    if (!rfbSendRectEncodingCoRRE(
			c->cl, rw->pixelData, rfbScreen.bitsPerPixel,
			rw->bytesPerRow, x, y, w, h)) {
		REGION_UNINIT(pScreen,&updateRegion);
		return FALSE;
	    }
	    break;
	default:
		/* must not happen */
		break;
	}
    }

    REGION_UNINIT(pScreen,&updateRegion);

    if (nUpdateRegionRects == 0xFFFF && !rfbSendLastRectMarker(c->cl)) {
	    fprintf(stderr, "NUMBER TO BIG\n");
	    return FALSE;
    }

    if (!rfbSendUpdateBuf(c->cl))
	return FALSE;

    return TRUE;
}


Bool window_rfbSendFramebufferUpdate(rfbClientPtr cl) {
	rfbWindowRec *w;
	WindowClients *c;

	for(w = Windows; w != NULL; w = w->next) {
		if ((c = find_client(w, cl)) != NULL)
		{
			if (FB_UPDATE_PENDING(c))
			{
				return my_rfbSendFramebufferUpdate(c, w);
			}
		}
	}
	return FALSE;
}

void window_schedule_fb_update_all(rfbClientPtr cl)
{
	rfbWindowRec *w;
	WindowClients *c;

	for(w = Windows; w != NULL; w = w->next) {
		if ((c = find_client(w, cl)) != NULL)
		{
			if (c->firstTime)
			{
				BoxRec box;
				RegionRec reg;

				box.x1 = 0;
				box.y1 = 0;
				box.x2 = w->frame->width;
				box.y2 = w->frame->height;
				REGION_INIT(pScreen,&reg,&box,0);
				REGION_UNION(
					pScreen,&c->modifiedRegion,
					&c->modifiedRegion, &reg);
				rfbSendConfigureWindow(
					w->frame->win->drawable.id,
					w->frame->x, w->frame->y,
					w->frame->width,
					w->frame->height);
				if (REGION_NOTEMPTY(pScreen,&w->bShape))
				{
					c->shapeChanged = 1;
				}
				my_rfbSendFramebufferUpdate(c, w);
				if (w->frame->win->mapped)
				{
					/* FIXME should send in stacking
					 * order */
					unsigned long flags;
					Window tf = None;

					flags = getrfbFlags(w->frame->win, &tf);
					rfbSendRestackWindow(
						w->frame->win->drawable.id, 0,
						tf, flags);
				}
				c->firstTime = 0;
			}
			else if (FB_UPDATE_PENDING(c))
			{
				my_rfbSendFramebufferUpdate(c, w);
			}
		}
	}
}

/* ****************************************************************************
 *
 * Rootless Call Back
 *
 * ****************************************************************************/

#define DEBUG_rfbCreateFrame 0
static
Bool rfbCreateFrame(
	RootlessWindowPtr pFrame, ScreenPtr pScreen,
	int newX, int newY, RegionPtr pShape)
{
	rfbWindowRec *rw;

	rw = add_window(pFrame);

	/* Store the implementation private frame ID */
	pFrame->wid = (RootlessFrameID *) rw;

	rw->bytesPerRow = PixmapBytePad(pFrame->width, rfbScreen.depth);
	rw->pixelData = (char *)xcalloc(1, rw->bytesPerRow * pFrame->height);

#if TRACE_RFBROOTLESS || DEBUG_rfbCreateFrame
	fprintf(
		stderr,"rfbCreateFrame for 0x%lx %i %i %ix%i %i %ix%i 0x%lx\n",
		pFrame->win->drawable.id, pFrame->x, pFrame->y,
		pFrame->width, pFrame->height, pFrame->borderWidth,
		pFrame->win->drawable.width, pFrame->win->drawable.height,
		(rw->pixelData)? rw->pixelData:0);
#endif

	rfbSendConfigureWindow(
		pFrame->win->drawable.id, newX, newY, pFrame->width,
		pFrame->height);

	REGION_INIT(pScreen,&rw->bShape,NullBox,0);
	if (pShape)
	{
		REGION_UNION(pScreen, &rw->bShape, pShape, pShape);
		window_shapeChanged(rw);
	}

	return TRUE;
}

#define DEBUG_rfbDestroyFrame 0
static
void rfbDestroyFrame(
	RootlessFrameID wid)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;

#if TRACE_RFBROOTLESS || DEBUG_rfbDestroyFrame
	fprintf(stderr,"rfbDestroyFrame for 0x%lx\n",
		rw->frame->win->drawable.id);
#endif

	if (rw->pixelData) {
		xfree(rw->pixelData);
		rw->pixelData = NULL;
	}

	REGION_UNINIT(pScreen,&rw->bShape);

	rfbSendDestroyWindow(rw->frame->win->drawable.id);
	remove_window(rw);
}

#define DEBUG_rfbMoveFrame 0
static
void rfbMoveFrame(RootlessFrameID wid, ScreenPtr pScreen, int newX, int newY)
{

	rfbWindowRec *rw = (rfbWindowRec *)wid;

#if TRACE_RFBROOTLESS || DEBUG_rfbMoveFrame
	fprintf(
		stderr,"rfbMoveFrame for 0x%lx: %i,%i (%ix%i)\n",
		rw->frame->win->drawable.id,
		newX, newY, rw->frame->width, rw->frame->height);
#endif

	rfbSendConfigureWindow(
		rw->frame->win->drawable.id, newX, newY, rw->frame->width,
		rw->frame->height);

}

#define DEBUG_rfbRestackFrame 0
static
void rfbRestackFrame(RootlessFrameID wid, RootlessFrameID nextWid)
{
	unsigned long flags = 0;
	rfbWindowRec *rw = (rfbWindowRec *)wid;
	rfbWindowRec *nextRw = (rfbWindowRec *)nextWid;
	Window tf = None;

#if TRACE_RFBROOTLESS || DEBUG_rfbRestackFrame
	fprintf(
		stderr,"rfbRestackFrame for 0x%lx (above 0x%lx)\n",
		rw->frame->win->drawable.id,
		(nextRw)? nextRw->frame->win->drawable.id : 0);
#endif

	if (rw->frame->win == 0)
	{
		/* FIXME: it seems that this can happen */
		return;
	}

	flags = getrfbFlags(rw->frame->win, &tf);
	rfbSendRestackWindow(
		rw->frame->win->drawable.id,
		(nextRw)? nextRw->frame->win->drawable.id : 0,
		tf, flags);


	{
		WindowClients *c;
		rfbClientPtr cl;

		for (cl = rfbClientHead; cl; cl = cl->next)
		{
			if ((c = find_client(rw, cl)) != NULL)
			{
				c->firstTime = 0;
			}
		}
	}
}


#define DEBUG_rfbResizeFrame 0
static
void rfbResizeFrame(
	RootlessFrameID wid, ScreenPtr pScreen,
	int newX, int newY, unsigned int newW, unsigned int newH,
	unsigned int gravity)
{
	rfbClientPtr cl;
	rfbWindowRec *rw = (rfbWindowRec *)wid;

	if (rw->pixelData) {
		xfree(rw->pixelData);
	}

	rw->bytesPerRow = PixmapBytePad(newW, rfbScreen.depth);
	rw->pixelData = (char *)xcalloc(1, rw->bytesPerRow * newH);

#if TRACE_RFBROOTLESS || DEBUG_rfbResizeFrame
	fprintf(
		stderr,"rfbResizeFrame for 0x%lx: %i,%i %ix%i %i %i %i,%i "
		"0x%lx\n",
		rw->frame->win->drawable.id, newX, newY, newW, newH, gravity,
		rw->frame->borderWidth,
		rw->frame->win->drawable.width, rw->frame->win->drawable.height,
		(long unsigned)rw->pixelData);
#endif
	rfbSendConfigureWindow(
		rw->frame->win->drawable.id, newX, newY, newW, newH);

	for (cl = rfbClientHead; cl; cl = cl->next)
	{
		WindowClients *c;
		if ((c = find_client(rw, cl)) == NULL)
		{
			continue;
		}
		c->updateRequested = FALSE;
		REGION_SUBTRACT(
			pScreen, &c->modifiedRegion, &c->modifiedRegion,
			&c->modifiedRegion);
	}
}

#define DEBUG_rfbReshapeFrame 0
void
rfbReshapeFrame(RootlessFrameID wid, RegionPtr pShape)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;

#if TRACE_RFBROOTLESS || DEBUG_rfbReshapeFrame
	fprintf(
		stderr,"rfbReshapeFrame for 0x%lx\n",
		rw->frame->win->drawable.id);
#endif

	REGION_UNINIT(pScreen,&rw->bShape);
	REGION_INIT(pScreen,&rw->bShape,NullBox,0);

	if (pShape && REGION_NUM_RECTS(pShape) > 0)
	{
		REGION_UNION(pScreen, &rw->bShape, pShape, pShape);
	}
	window_shapeChanged(rw);

}

#define DEBUG_rfbUnmapFrame 0
static
void rfbUnmapFrame(RootlessFrameID wid)
{
	rfbClientPtr cl;
	rfbWindowRec *rw = (rfbWindowRec *)wid;

#if TRACE_RFBROOTLESS || DEBUG_rfbUnmapFrame
	fprintf(
		stderr,"rfbUnmapFrame for 0x%lx\n",
		rw->frame->win->drawable.id);
#endif
	rfbSendUnmapWindow(rw->frame->win->drawable.id);

	for (cl = rfbClientHead; cl; cl = cl->next)
	{
		WindowClients *c;
		if ((c = find_client(rw, cl)) == NULL)
		{
			continue;
		}
		c->updateRequested = FALSE;
		REGION_SUBTRACT(
			pScreen, &c->modifiedRegion, &c->modifiedRegion,
			&c->modifiedRegion);
	}
}


#define DEBUG_rfbUpdateRegion 0

static
void rfbUpdateRegion(
	RootlessFrameID wid, RegionPtr pDamage)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;
	rfbClientPtr cl;

#if DEBUG_rfbUpdateRegion || TRACE_RFBROOTLESS
	fprintf(
		stderr,"rfbUpdateRegion for 0x%lx with damage (%i)",
		rw->frame->win->drawable.id, rw->frame->is_drawing);
#if !defined(ROOTLESS_TRACK_DAMAGE) || !DEBUG_rfbUpdateRegion
	fprintf(stderr,"\n");
#endif
#endif

#ifdef ROOTLESS_TRACK_DAMAGE
	if (!pDamage)
	{
		return;
	}
#endif /* ROOTLESS_TRACK_DAMAGE */

	for (cl = rfbClientHead; cl; cl = cl->next) {
		WindowClients *c;
		if ((c = find_client(rw, cl)) == NULL)
		{
			continue;
		}
#ifdef ROOTLESS_TRACK_DAMAGE
		REGION_UNION(
			pScreen,&c->modifiedRegion,&c->modifiedRegion, pDamage);
#endif
		if (!rw->frame->is_drawing && FB_UPDATE_PENDING(c))
		{
			my_rfbSendFramebufferUpdate(c, rw);
		}
	}
}

#ifndef ROOTLESS_TRACK_DAMAGE
#define DEBUG_rfbDamageRects 0
#define DEBUG_Limits 0

static void debug_print_limits(
	char *str, rfbWindowRec *rw, int shift_x, int shift_y, BoxRec sr)
{
#if DEBUG_Limits
	fprintf(stderr,
		"%s for 0x%lx with "
		"(%i, %i):\n   ww: %i, wh: %i //  x1: %i y1: %i x2: %i y2: %i\n",
		str, rw->frame->win->drawable.id, shift_x, shift_y,
		rw->frame->win->drawable.width, rw->frame->win->drawable.height,
		sr.x1,sr.y1,sr.x2,sr.y2);
#endif
}

static
void rfbDamageRects(
	RootlessFrameID wid, int count, const BoxRec *rects,
	int shift_x, int shift_y)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;
	int i;
	RegionRec pDamage;
	rfbClientPtr cl;

	REGION_INIT(pScreen, &pDamage, NullBox, 0);

#if DEBUG_rfbDamageRects || TRACE_RFBROOTLESS
	fprintf(
		stderr,"rfbDamageRects for 0x%lx with rectangles (%i, %i,%i): ",
		rw->frame->win->drawable.id, count, shift_x, shift_y);
#endif

	for (i = 0; i < count; i++)
	{
		RegionRec reg;
		BoxRec sr;

		sr.x1 = rects->x1 + shift_x;
		sr.y1 = rects->y1 + shift_y;
		sr.x2 = rects->x2 + shift_x;
		sr.y2 = rects->y2 + shift_y;
		/* It seems that this may happen (thanks to Tony J. White) */
		if ((int)sr.x1 < 0)
		{
			debug_print_limits(
				"x1 negative", rw, shift_x, shift_y, sr);
			sr.x1 = 0;
			sr.x2 = (int)rw->frame->width;
		}
		if ((int)sr.y1 < 0)
		{
			debug_print_limits(
				"y1 negative", rw, shift_x, shift_y, sr);
			sr.y1 = 0;
			sr.y2 = (int)rw->frame->height;
		}
		/* safty check */
		if ((int)sr.x2 > (int)rw->frame->width)
		{
			debug_print_limits(
				"x2 to big", rw, shift_x, shift_y, sr);
			sr.x1 = 0;
			sr.x2 = (int)rw->frame->width;
		}
		if ((int)sr.y2 > (int)rw->frame->height)
		{
			debug_print_limits(
				"y2 to big", rw, shift_x, shift_y, sr);
			sr.y1 = 0;
			sr.y2 = (int)rw->frame->height;
		}
		if ((int)sr.x1 >= (int)sr.x2)
		{
			rects++;
			debug_print_limits(
				"x1 >= x2", rw, shift_x, shift_y, sr);
			/*continue;*/
			sr.x1 = 0;
			sr.x2 = (int)rw->frame->width;
		}
		if ((int)sr.y1 >= (int)sr.y2)
		{
			rects++;
			debug_print_limits(
				"y1 >= y2", rw, shift_x, shift_y, sr);
			/*continue;*/
			sr.y1 = 0;
			sr.y2 = (int)rw->frame->height;
		}
		REGION_INIT(pScreen, &reg, &sr, 0);
		REGION_UNION(pScreen,&pDamage,&pDamage,&reg);
		REGION_UNINIT(pScreen, &reg);
#if DEBUG_rfbDamageRects
		{
			int x = sr.x1;
			int y = sr.y1;
			int w = sr.x2 - x;
			int h = sr.y2 - y;
			fprintf(stderr, " r %i %i %i %i /",
				sr.x1,sr.y1,sr.x2,sr.y2);
			fprintf(stderr, " d %i,%i %ix%i /",x,y,w,h);
		}
#endif
		rects++;
	}

#if  DEBUG_rfbDamageRects
	fprintf(stderr, "\n");
#endif

	for (cl = rfbClientHead; cl; cl = cl->next) {
		WindowClients *c;
		
		if ((c = find_client(rw, cl)) == NULL)
		{
			continue;
		}
		REGION_UNION(
			pScreen,&c->modifiedRegion,&c->modifiedRegion, &pDamage);
	}
	REGION_UNINIT(pScreen, &pDamage);
}
#endif /* ndef ROOTLESS_TRACK_DAMAGE */

#define DEBUG_rfbStartStopDrawing 0

static
void rfbStartDrawing(
	RootlessFrameID wid, char **pixelData, int *bytesPerRow)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;
	
#if TRACE_RFBROOTLESS || DEBUG_rfbStartStopDrawing
	fprintf(
		stderr,"rfbStartDrawing for 0x%lx 0x%lx\n",
		rw->frame->win->drawable.id, (rw->pixelData)? rw->pixelData:0);
#endif

	*pixelData = rw->pixelData;
	*bytesPerRow = rw->bytesPerRow;
}

static
void rfbStopDrawing(RootlessFrameID wid, Bool flush)
{
	rfbWindowRec *rw = (rfbWindowRec *)wid;
	rfbClientPtr cl;

#if TRACE_RFBROOTLESS ||  DEBUG_rfbStartStopDrawing
	fprintf(
		stderr,"rfbStopDrawing for 0x%lx 0x%lx flush: %i\n",
		rw->frame->win->drawable.id,
		(rw->pixelData)? (long unsigned)rw->pixelData:0,
		flush);
#endif

	if (!flush)
	{
		return;
	}

	for (cl = rfbClientHead; cl; cl = cl->next)
	{
		WindowClients *c;

		if ((c = find_client(rw, cl)) == NULL)
		{
			continue;
		}
		
		if (FB_UPDATE_PENDING(c))
		{
			my_rfbSendFramebufferUpdate(c, rw);
		}
	}
}

#define DEBUG_rfbSwitchWindow 0

static
void rfbSwitchWindow(RootlessWindowPtr pFrame, WindowPtr oldWin)
{
#if TRACE_RFBROOTLESS || DEBUG_rfbSwitchWindow
	fprintf(stderr, "rfbSwitchWindow 0x%lx, 0x%lx\n",
		pFrame->win->drawable.id, oldWin->drawable.id);
#endif
}

/*
 * Function pointer table for rootless support.
 */

static RootlessFrameProcsRec rfbRootlessProcs = {
	rfbCreateFrame,
	rfbDestroyFrame,

	rfbMoveFrame,
	rfbResizeFrame,
	rfbRestackFrame,
	rfbReshapeFrame,
	rfbUnmapFrame,

	rfbStartDrawing,
	rfbStopDrawing,
	rfbUpdateRegion,

#ifndef ROOTLESS_TRACK_DAMAGE
	rfbDamageRects,
#endif

	rfbSwitchWindow,
	/* Optional acceleration functions */
	NULL, // CopyBytes
	NULL, // FillBytes
	NULL, // CompositePixels
	NULL, // CopyWindow
};

/* ***************************************************************************
 *
 * *************************************************************************** */

Bool
rfbRootlessSetupScreen (int index, ScreenPtr pScreen)
{
	/* do not use acceleration functions */
	rootless_CopyBytes_threshold = 0;
	rootless_FillBytes_threshold = 0;
	rootless_CompositePixels_threshold = 0;
	rootless_CopyWindow_threshold = 0;

	rootless_HaveRoot = 1;

	/* Setup generic rooless support. */
	return RootlessInit (pScreen, &rfbRootlessProcs);
}
