#
#   Install    This script can be run *inside* a normal CMF Site and will 
#              make all necessary changes to convert membership to an 
#              LDAP-based setup
#

from Products.CMFLDAP import cmfldap_globals
from Products.CMFCore.DirectoryView import addDirectoryViews
from Products.CMFCore.utils import getToolByName
from Products.CMFLDAP.LDAPMembershipTool import LDAPMembershipTool, \
                                    manage_addLDAPMembershipTool
from Products.CMFLDAP.LDAPMemberDataTool import LDAPMemberDataTool, \
                                    manage_addLDAPMemberDataTool

try:
    # Only expose Plone tools if Plone is installed
    from Products.CMFLDAP.PloneLDAPMembershipTool import PloneLDAPMembershipTool, \
                                             manage_addPloneLDAPMembershipTool
    PLONE_TOOLS = (
            ('portal_membership', PloneLDAPMembershipTool),
            ('portal_memberdata', LDAPMemberDataTool),
        )
except ImportError:
    PLONE_TOOLS = ()

#
# List all specially initialized components
########################################################

CMF_TOOLS = (
        ('portal_membership', LDAPMembershipTool),
        ('portal_memberdata', LDAPMemberDataTool),
    )


#
# Initialize the log
#########################################################

log = []


#
# The main conversion method. This is the method name showing in 
# the external method object to be instantiated and run in Zope
#########################################################

def install(self):
    """ The central conversion method """
    log.append('Starting LDAP Membership conversion\n')

    # Find the portal to be modified. It is assumed that the external method
    # is instantiated at the root of the portal.
    portal = self
    if portal.meta_type not in ('CMF Site', 'Plone Site'):
        return 'You are not running this method inside a CMF Site Instance!'
    else:
        log.append('Found CMF Site at %s' % portal.absolute_url())


    # Instantiate tools that are not in the standard CMF site
    log.append('\n * Instantiating Tools in the portal:')
    if portal.__class__.__name__ == 'PloneSite':
        TOOL_CLASSES = PLONE_TOOLS
    else:
        TOOL_CLASSES = CMF_TOOLS

    for tool_tuple in TOOL_CLASSES:
        registerTool(portal, tool_tuple)


    # Register the skins directories as Filesystem Directory View
    # in the skins tool (portal_skins)
    log.append('\n * Registering skins directories with the Skins Tool:')
    registerSkins(portal)
        
    log.append('\nFinished conversion process!\n\n')
    
    return '\n'.join(log)


#
# Helper methods to register varous components with the CMF machinery
###############################################################

def registerTool(portal_obj, tool_tuple):
    """ Register tools that do not come with the CMF """
    tool_id  = tool_tuple[0]
    tool_obj = tool_tuple[1]()

    if hasattr(portal_obj, tool_id):
        old_tool = getattr(portal_obj, tool_id)
        if hasattr(old_tool, '_actions'):
            old_actions = old_tool.listActions()
        else:
            old_actions = None
        portal_obj._delObject(tool_id)

    if tool_id == 'portal_membership':
        if tool_obj.meta_type == 'Plone LDAP Membership Tool':
            manage_addPloneLDAPMembershipTool(portal_obj)
        else:
            manage_addLDAPMembershipTool(portal_obj)
    elif tool_id == 'portal_memberdata':
        manage_addLDAPMemberDataTool(portal_obj)
        md = getattr(portal_obj, tool_id)
        md._properties = old_tool._properties
        for property in old_tool.propertyMap():
            setattr(md, property['id'], old_tool.getProperty(property['id']))
        md._members = old_tool._members

    log.append('   - Inserted tool %s' % tool_id)


def registerSkins(portal_obj):
    """ Register skins as Filesystem Directory Views"""
    skins_tool = getToolByName(portal_obj, 'portal_skins')
    existing_skins = skins_tool.objectIds()

    if 'cmfldap' in existing_skins:
        skins_tool._delObject('cmfldap')

    addDirectoryViews(skins_tool, 'skins', cmfldap_globals)
    log.append('   - Registered Filesystem Directory Views')

    skins = skins_tool.getSkinPaths()

    for skin_name, skin_path in skins:
        path_elems = map(lambda x: x.strip(), skin_path.split(','))
        
        if 'cmfldap' not in path_elems:
            for i in range(len(path_elems)):
                if path_elems[i] == 'custom':
                    path_elems.insert(i+1, 'cmfldap')
                    break

        new_path = ', '.join(path_elems)
        skins_tool.addSkinSelection(skin_name, new_path)
        log.append('   - Adjusted skin path for skin "%s"' % skin_name)


