#############################################################################
#
# Copyright (c) 2001 Zope Corporation and Contributors. All Rights Reserved.
#
# This software is subject to the provisions of the Zope Public License,
# Version 2.0 (ZPL).  A copy of the ZPL should accompany this distribution.
# THIS SOFTWARE IS PROVIDED "AS IS" AND ANY AND ALL EXPRESS OR IMPLIED
# WARRANTIES ARE DISCLAIMED, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
# WARRANTIES OF TITLE, MERCHANTABILITY, AGAINST INFRINGEMENT, AND FITNESS
# FOR A PARTICULAR PURPOSE
#
##############################################################################
from __future__ import nested_scopes
  
__version__='$Revision: 1.12.4.6 $'[11:-2]
  
import sys
from types import DictType, ListType, StringType, SliceType

import RestrictedPython
from RestrictedPython.Guards import safe_builtins, full_write_guard
from RestrictedPython.Utilities import utility_builtins
from RestrictedPython.Eval import RestrictionCapableEval
from SecurityManagement import getSecurityManager
from SecurityInfo import secureModule
from SimpleObjectPolicies import Containers, ContainerAssertions
from zExceptions import Unauthorized

_marker = []  # Create a new marker object.

RP_safe_builtins, safe_builtins = safe_builtins, safe_builtins.copy()
safe_builtins.update(utility_builtins)

# This is always created do that we can test it independently of whether
# we are using the C version.
def guarded_getattr(inst, name, default=_marker):
    """Retrieves an attribute, checking security in the process.

    Raises Unauthorized if the attribute is found but the user is
    not allowed to access the attribute.
    """
    if name[:1] != '_':
        # Try to get the attribute normally so that unusual
        # exceptions are caught early.
        try:
            v = getattr(inst, name)
        except AttributeError:
            if default is not _marker:
                return default
            raise

        assertion = Containers(type(inst))
        if assertion:
            if callable(assertion):
                factory = assertion(name, v)
                if callable(factory):
                    return factory(inst, name)
                assert factory == 1
            else:
                assert assertion == 1
            return v

        validate = getSecurityManager().validate
        # Filter out the objects we can't access.
        if hasattr(inst, 'aq_acquire'):
            return inst.aq_acquire(name, aq_validate, validate)
        # Or just try to get the attribute directly.
        if validate(inst, inst, name, v):
            return v
    raise Unauthorized, name

guarded_getattr_py = guarded_getattr

try:

    #raise ImportError
    import os
    if os.environ.get("ZOPE_SECURITY_POLICY", None) == "PYTHON":
        raise ImportError # :)
    from cAccessControl import aq_validate, guarded_getattr

except ImportError:

    def aq_validate(inst, obj, name, v, validate):
        return validate(inst, obj, name, v)

safe_builtins['getattr'] = guarded_getattr

def guarded_hasattr(object, name):
    try:
        guarded_getattr(object, name)
    except (AttributeError, Unauthorized):
        return 0
    return 1
safe_builtins['hasattr'] = guarded_hasattr

def guarded_getitem(object, index):
    if type(index) is SliceType:
        if index.step is not None:
            v = object[index]
        else:
            start = index.start
            stop = index.stop
            if start is None:
                start = 0
            if stop is None:
                v = object[start:]
            else:
                v = object[start:stop]
        # We don't guard slices.
        return v
    v = object[index]
    if Containers(type(object)) and Containers(type(v)):
        # Simple type.  Short circuit.
        return v
    if getSecurityManager().validate(object, object, index, v):
        return v
    raise Unauthorized, 'unauthorized access to element %s' % `i`

def get_dict_get(d, name):
    def guarded_get(key, default=None):
        try:
            return guarded_getitem(d, key)
        except KeyError:
            return default
    return guarded_get

def get_dict_pop(d, name):
    def guarded_pop(key, default=_marker):
        try:
            v = guarded_getitem(d, key)
        except KeyError:
            if default is not _marker:
                return default
            raise
        else:
            del d[key]
            return v        
    return guarded_pop

def get_iter(c, name):
    iter = getattr(c, name)
    def guarded_iter():
        return SafeIter(iter(), c)
    return guarded_iter

def get_list_pop(lst, name):
    def guarded_pop(index=-1):
        # XXX This is not thread safe, but we don't expect
        # XXX thread interactions between python scripts <wink>
        v = guarded_getitem(lst, index)
        del lst[index]
        return v        
    return guarded_pop


# Provide special handling for some dict and list methods.

dict_handlers = {
    'get':get_dict_get, 'has_key':1,
    'iterkeys': get_iter,  'itervalues':get_iter,
    'pop':get_dict_pop, }
def dict_attr_factory(attr, v, handlers = dict_handlers):
    return handlers.get(attr, 1)

ContainerAssertions[type({})] = dict_attr_factory

list_handlers = {'pop':get_list_pop, }
def list_attr_factory(attr, v, handlers = list_handlers):
    return handlers.get(attr, 1)

ContainerAssertions[type([])] = list_attr_factory

# This implementation of a "safe" iterator uses a global guard()
# function to implement the actual guard.  This check is defined as a
# global so that it can delay imports of some module to avoid circular
# dependencies while also making it possible to use a faster
# implementation once the imports are done (by avoiding the import
# machinery on subsequent calls).  Use of a method on the SafeIter
# class is avoided to ensure the best performance of the resulting
# function.


if sys.version_info < (2, 2):

    class SafeIter:
        def __init__(self, sequence, container=None):
            self.sequence = sequence
            if container is None:
                container = sequence
            self.container = container
            self.next_index = 0

        def __getitem__(self, index):
            ob = self.sequence[self.next_index]
            self.next_index += 1
            guard(self.container, ob, self.next_index - 1)
            return ob
            
        def __len__(self):
            return len(self.sequence)

    def _error(index):
        raise Unauthorized, 'unauthorized access to element %s' % `index`

else:
    class SafeIter(object):
        #__slots__ = '_next', 'container'

        def __init__(self, ob, container=None):
            self._next = iter(ob).next
            if container is None:
                container = ob
            self.container = container

        def __iter__(self):
            return self

        def next(self):
            ob = self._next()
            guard(self.container, ob)
            return ob

    def _error(index):
        raise Unauthorized, 'unauthorized access to element'

    safe_builtins['iter'] = SafeIter


def guard(container, value, index=None):
    if Containers(type(container)) and Containers(type(value)):
        # Simple type.  Short circuit.
        return
    if getSecurityManager().validate(container, container, index, value):
        return
    _error(index)


# More replacement built-ins.


def guarded_filter(f, seq, skip_unauthorized=0):
    if type(seq) is StringType:
        return filter(f, seq)
    if f is None:
        def f(x): return x
    v = getSecurityManager().validate
    result = []
    a = result.append
    for el in seq:
        if v(seq, seq, None, el):
            if f(el): a(el)
        elif not skip_unauthorized:
            raise Unauthorized, 'unauthorized access to element'
    return result
safe_builtins['filter'] = guarded_filter

def guarded_reduce(f, seq, initial=_marker):
    if initial is _marker:
        return reduce(f, SafeIter(seq))
    else:
        return reduce(f, SafeIter(seq), initial)
safe_builtins['reduce'] = guarded_reduce

def guarded_max(item, *items):
    if items:
        item = [item]
        item.extend(items)
    return max(SafeIter(item))
safe_builtins['max'] = guarded_max

def guarded_min(item, *items):
    if items:
        item = [item]
        item.extend(items)
    return min(SafeIter(item))
safe_builtins['min'] = guarded_min

def guarded_map(f, *seqs):
    safe_seqs = []
    for seqno in range(len(seqs)):
        seq = guarded_getitem(seqs, seqno)
        safe_seqs.append(seq)
    return map(f, *safe_seqs)
safe_builtins['map'] = guarded_map

def guarded_import(mname, globals={}, locals={}, fromlist=None):
    mnameparts = mname.split('.')
    firstmname = mnameparts[0]
    validate = getSecurityManager().validate
    module = load_module(None, None, mnameparts, validate, globals, locals)
    if module is not None:
        if fromlist is None:
            fromlist = ()
        try:
            for name in fromlist:
                if name == '*':
                    raise ImportError, ('"from %s import *" is not allowed'
                                        % mname)
                v = getattr(module, name, None)
                if v is None:
                    v = load_module(module, mname, [name], validate,
                                    globals, locals)
                if not validate(module, module, name, v):
                    raise Unauthorized
            else:
                return __import__(mname, globals, locals, fromlist)
        except Unauthorized, why:
            raise ImportError, ('import of "%s" from "%s" is unauthorized. %s'
                                % (name, mname, why))
    raise ImportError, 'import of "%s" is unauthorized' % mname
safe_builtins['__import__'] = guarded_import

if sys.version_info >= (2, 2):
    class GuardedListType:
        def __call__(self, *args, **kwargs):
            return list(*args)

        if sys.version_info >= (2, 4):
            def sorted(self, iterable, cmp=None, key=None, reverse=False):
                return list.sorted(iterable, cmp=None, key=None, reverse=False)
    safe_builtins['list'] = GuardedListType()

    class GuardedDictType:
        def __call__(self, *args, **kwargs):
            return dict(*args, **kwargs)

        if sys.version_info >= (2, 3):
            # dict.fromkeys() was introduced in 2.3.
            def fromkeys(self, S, v=None):
                return dict.fromkeys(S,v)
    safe_builtins['dict'] = GuardedDictType()
else:
    # In python 2.1 or before, list is not a type (there was no dict)
    safe_builtins['list'] = list
    
def load_module(module, mname, mnameparts, validate, globals, locals):
    modules = sys.modules
    while mnameparts:
        nextname = mnameparts.pop(0)
        if mname is None:
            mname = nextname
        else:
            mname = '%s.%s' % (mname, nextname)
        nextmodule = modules.get(mname, None)
        if nextmodule is None:
            nextmodule = secureModule(mname, globals, locals)
            if nextmodule is None:
                return
        else:
            secureModule(mname)
        if module and not validate(module, module, nextname, nextmodule):
            return
        module = nextmodule
    return module

# This version of apply is used by restricted Python, which transforms
# extended call syntax into a call of _apply_(), after tucking the callable
# into the first element of args.  For example,
#     f(3, eggs=1, spam=False)
# is changed to
#     _apply_(f, 3, eggs=1, spam=False)
def guarded_apply(func, *args, **kws):
    return builtin_guarded_apply(func, args, kws)

# This version is the safe_builtins apply() replacement, so needs to match the
# signature of __builtin__.apply.
def builtin_guarded_apply(func, args=(), kws={}):
    # Check the args elements.  args may be an arbitrary iterable, and
    # iterating over it may consume it, so we also need to save away
    # the arguments in a new list to pass on to the real apply().
    i, arglist = 0, []
    for elt in args:
        guard(args, elt, i)
        arglist.append(elt)
        i += 1
    # Check kws similarly.  Checking the keys may not be strictly necessary,
    # but better safe than sorry.  A new argument dict is created just in
    # case kws is a hostile user-defined instance that may do horrid things
    # as a side-effect of calling items().
    argdict = {}
    for k, v in kws.items():
        guard(kws, k)
        guard(kws, v, k)
        argdict[k] = v
    return func(*arglist, **argdict)

safe_builtins['apply'] = builtin_guarded_apply


# AccessControl clients generally need to set up a safe globals dict for
# use by restricted code.  The get_safe_globals() function returns such
# a dict, containing '__builtins__' mapped to our safe bulitins, and
# bindings for all the special functions inserted into Python code by
# RestrictionMutator transformations.  A client may wish to add more
# bindings to this dict.  It's generally safe to do so, as
# get_safe_globals returns a (shallow) copy of a canonical safe globals
# dict.
# Exception:  For obscure technical reasons, clients have to import
# guarded_getattr from this module (ZopeGuards) and plug it into the
# dict themselves, with key '_getattr_'.

_safe_globals = {'__builtins__': safe_builtins,
                 '_apply_':      guarded_apply,
                 '_getitem_':    guarded_getitem,
                 '_getiter_':    SafeIter,
                 '_print_':      RestrictedPython.PrintCollector,
                 '_write_':      full_write_guard,
                 # The correct implementation of _getattr_, aka
                 # guarded_getattr, isn't known until
                 # AccessControl.Implementation figures that out, then
                 # stuffs it into *this* module's globals bound to
                 # 'guarded_getattr'.  We can't know what that is at
                 ## '_getattr_':   guarded_getattr,
                }

get_safe_globals = _safe_globals.copy

RestrictionCapableEval.globals.update(_safe_globals)

#
#   Force our version of 'safe_builtins' into RestrctedPython.Guards.
#
RP_safe_builtins.update(safe_builtins)
