/*
 * Copyright (c) 1995 - 2000 Kungliga Tekniska Hgskolan
 * (Royal Institute of Technology, Stockholm, Sweden).
 * All rights reserved.
 * 
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 
 * 3. Neither the name of the Institute nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 * 
 * THIS SOFTWARE IS PROVIDED BY THE INSTITUTE AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE INSTITUTE OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */

/* $Id: xfs_message.c,v 1.3 2000/10/02 23:59:58 lha Exp $ */

#include <xfs_locl.h>

/*
 *
 */

int
xfs_message_installroot(struct xfs_channel *chan,
			struct xfs_message_installroot * message,
			u_int size)
{
    int error = 0;
    
    XFSDEB(XDEBMSG, ("xfs_message_installroot (%d,%d,%d,%d)\n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));
    
    if (chan->root != NULL) {
	DbgPrint ("XFS PANIC WARNING! "
		  "xfs_message_installroot: called again!\n");
	error = STATUS_DEVICE_BUSY;
    } else {
	error = xfs_new_node(chan, &message->node, &chan->root);
	if (error)
	    return error;
#if 0
	chan->root->vn->v_flag |= VROOT;
#endif
    }
    return error;
}

/*
 *
 */


int
xfs_message_installnode(struct xfs_channel *chan,
			struct xfs_message_installnode * message,
			u_int size)
{
    int error = 0;
    struct xfs_node *n, *dp;

    XFSDEB(XDEBMSG, ("xfs_message_installnode (%d,%d,%d,%d)\n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));

    dp = xfs_node_find(chan, &message->parent_handle);
    if (dp) {
	
	error = xfs_new_node(chan, &message->node, &n);
	if (error) {
	    xfs_vrele (dp);
	    return error;
	}

	xfs_dnlc_enter_name(dp, message->name, n);
	xfs_vrele (n);
	xfs_vrele (dp);
    } else {
	DbgPrint ("XFS PANIC WARNING! xfs_message_installnode: no parent\n");
	error = STATUS_NO_SUCH_FILE;
    }
    XFSDEB(XDEBMSG, ("return: xfs_message_installnode: %d\n", error));

    return error;
}

/*
 *
 */

int
xfs_message_installattr(struct xfs_channel *chan,
			struct xfs_message_installattr * message,
			u_int size)
{
    int error = STATUS_SUCCESS;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_installattr (%d,%d,%d,%d) \n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));

    t = xfs_node_find(chan, &message->node.handle);
    if (t != 0) {
	t->tokens = message->node.tokens;
	if ((t->tokens & XFS_DATA_MASK)&& !XFS_VALID_DATAHANDLE(t)) {
	    DbgPrint ("xfs_message_installattr: tokens and no data\n");
	    t->tokens &= ~XFS_DATA_MASK;
	}
	xfs_attr2vattr(&message->node.attr, t);
#if 0
	xfs_set_vp_size(XNODE_TO_VNODE(t), t->attr.va_size);
#endif
	bcopy(message->node.id, t->id, sizeof(t->id));
	bcopy(message->node.rights, t->rights, sizeof(t->rights));
	t->anonrights = message->node.anonrights;
	xfs_vrele (t);
    } else {
	XFSDEB(XDEBMSG, ("xfs_message_installattr: no such node\n"));
    }

    return error;
}

/*
 *
 */

int
xfs_message_installdata(struct xfs_channel *chan,
			struct xfs_message_installdata * message,
			u_int size)
{
    struct xfs_node *t;
    int error = 0;

    XFSDEB(XDEBMSG, ("xfs_message_installdata (%d,%d,%d,%d)\n",
		     message->node.handle.a,
		     message->node.handle.b,
		     message->node.handle.c,
		     message->node.handle.d));

    t = xfs_node_find(chan, &message->node.handle);
    if (t != NULL) {
	struct xfs_fhandle_t *fh =
	    (struct xfs_fhandle_t *)&message->cache_handle;
	FILE_OBJECT *cache_obj;

	XFSDEB(XDEBMSG, ("cache_name = '%s'\n", message->cache_name));

	error = xfs_open_file (message->cache_name, NULL,
			       FILE_OPEN, FILE_NON_DIRECTORY_FILE,
			       NULL, &cache_obj);
	
	
	if (NT_SUCCESS(error)) {
	    if (XFS_VALID_DATAHANDLE(t)) {
		xfs_close_data_handle (t);
	    }
	    DATA_FROM_XNODE(t) = cache_obj;

	    XFSDEB(XDEBMSG, ("xfs_message_installdata: t = %lx;"
			     " tokens = %x\n",
			     (unsigned long)t, message->node.tokens));

	    t->tokens = message->node.tokens;
	    xfs_attr2vattr(&message->node.attr, t);
#if 0
	    xfs_set_vp_size(XNODE_TO_VNODE(t), t->attr.va_size);
	    if (XNODE_TO_VNODE(t)->v_type == VDIR
		&& (message->flag & XFS_ID_INVALID_DNLC))
		dnlc_cache_purge (XNODE_TO_VNODE(t));
#endif
	    bcopy(message->node.id, t->id, sizeof(t->id));
	    bcopy(message->node.rights, t->rights, sizeof(t->rights));
	    t->anonrights = message->node.anonrights;
	} else {
	    DbgPrint("XFS PANIC WARNING! xfs_message_installdata failed!\n");
	    DbgPrint("Reason: lookup failed on cache file '%s', error = %d\n",
		     message->cache_name, error);
	}
	xfs_vrele (t);
    } else {
	DbgPrint("XFS PANIC WARNING! xfs_message_installdata failed\n");
	DbgPrint("Reason: No node to install the data into!\n");
	error = STATUS_NO_SUCH_FILE;
    }

    return error;
}

/*
 *
 */

int
xfs_message_invalidnode(struct xfs_channel *chan,
			struct xfs_message_invalidnode * message,
			u_int size)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_invalidnode (%d,%d,%d,%d)\n",
		     message->handle.a,
		     message->handle.b,
		     message->handle.c,
		     message->handle.d));

    t = xfs_node_find(chan, &message->handle);
    if (t != 0) {
#if 0
        /* If open for writing, return immediately. Last close:er wins! */
	if (vp->v_usecount >= 0 && vp->v_writecount >= 1) {
	    xfs_trele (t);
            return 0;
	}
#endif

	if (XFS_VALID_DATAHANDLE(t)) {
	    xfs_close_data_handle (t);
	}
	XFS_TOKEN_CLEAR(t, ~0,
			XFS_OPEN_MASK | XFS_ATTR_MASK |
			XFS_DATA_MASK | XFS_LOCK_MASK);
	/* Dir changed, must invalidate DNLC. */
#if 0
	if (vp->v_type == VDIR)
	    xfs_dnlc_purge(vp);
	if (vp->v_usecount == 0) {
	    XFSDEB(XDEBVNOPS, ("xfs_message_invalidnode: vrecycle\n"));
	    xfs_vrecycle(vp, 0);
	}
	xfs_trele (t);
#endif
    } else {
	DbgPrint("XFS PANIC WARNING! xfs_message_invalidnode: no node!\n");
	error = STATUS_NO_SUCH_FILE;
    }

    return error;
}

/*
 *
 */

int
xfs_message_updatefid(struct xfs_channel *chan,
		      struct xfs_message_updatefid * message,
		      u_int size)
{
    int error = 0;
    struct xfs_node *t;

    XFSDEB(XDEBMSG, ("xfs_message_updatefid (%d,%d,%d,%d)\n",
		     message->old_handle.a,
		     message->old_handle.b,
		     message->old_handle.c,
		     message->old_handle.d));

    t = xfs_node_find (chan, &message->old_handle);
    if (t != NULL) {
	t->handle = message->new_handle;
	xfs_vrele (t);
    } else {
	DbgPrint ("XFS PANIC WARNING! xfs_message_updatefid: no node!\n");
	error = STATUS_NO_SUCH_FILE;
    }
    return error;
}

/*
 *
 */

static void
gc_vnode (struct xfs_node *node)
{
    /* This node is on the freelist */
    if (node->ReferenceCount <= 0) {
	
	/*  DIAGNOSTIC */
	ASSERT (node->ReferenceCount >= 0);
	
	XFSDEB(XDEBMSG, ("xfs_message_gc: success\n"));
	
#if 0
	xfs_vgone(node);
#endif
    } else {
	XFSDEB(XDEBMSG, ("xfs_message_gc: used\n"));
    }
}

int
xfs_message_gc_nodes(struct xfs_channel *chan,
		     struct xfs_message_gc_nodes * message,
		     u_int size)
{
    XFSDEB(XDEBMSG, ("xfs_message_gc\n"));

#if 0
    if (message->len == 0) {
	struct vnode *vp;

	/* XXX see comment in xfs_node_find */

	for(vp = XFS_TO_VFS(chan)->m_mounth;
	    vp != NULL; 
	    vp = vp->v_mountf) {
	    gc_vnode (vp, p);
	}

    } else {
	struct xfs_node *t;
	int i;

	for (i = 0; i < message->len; i++) {
	    t = xfs_node_find (chan, &message->handle[i]);
	    if (t == NULL)
		continue;

	    gc_vnode(XNODE_TO_VNODE(t), p);
	}
    }
#endif

    return 0;
}

/*
 *
 */

int
xfs_message_version(struct xfs_channel *chan,
		    struct xfs_message_version *message,
		    u_int size)
{
    return STATUS_SUCCESS;
}


