#include "DBThreadCtlMO.hh"
#include "DBThreadPool.hh"


DBThreadPool::DBThreadPool(int initThreads = 4, 
			   int minThreads = 4, 
			   int maxThreads = 20, 
			   int upperQueueLimit = 2,
			   int lowerQueueLimit = 0) {

  ACE_DEBUG((LM_DEBUG,"(%t) Thread Pool created\n"));

  // Thread parameters:
  // We'll create initThreads+1 and augment the other parameters cos 1 thread is used
  // full time by the management Method Object.

  _nThreads = initThreads+1;
  _initThreads = initThreads;
  _minThreads = minThreads+1;
  _maxThreads = maxThreads+1;
  _upperQueueLimit = upperQueueLimit;
  _lowerQueueLimit = lowerQueueLimit;

  // Initialize the moving average data.
  // This is used to know when to grow or shrink the number of threads
  for (unsigned int i=0;i<MOVING_AVERAGE_SIZE;i++)
    _movingAverage[i]=0;

  this->open(_nThreads);

  // Accounting
  _growCount = 0;
  _shrinkCount = 0;

  // Memory accounting
  /*
  _initialMemory = 0;
  _lastMemory = 0;
  _initialMemory = checkmem("DBThreadPool end constructor");
  _lastMemory = _initialMemory;
  */

}

void DBThreadPool::open(int nthreads) {

  this->activate(THR_NEW_LWP | THR_DETACHED, nthreads, 1);

}


DBThreadPool::~DBThreadPool() {

  ACE_DEBUG((LM_DEBUG,"(%t) Thread Pool destroyed\n"));

}

void DBThreadPool::submit(ACE_Method_Object *t) {

  this->activation_queue_.enqueue(t);

}


int DBThreadPool::svc(void) {

  // The auto pointer deletes the Method_Object on next iteration.
  
  while(1) {

    auto_ptr<ACE_Method_Object> mo (this->activation_queue_.dequeue());
    if (!mo.get() || (mo->call() == -1)) break;

  }
  
  return 0;

}




// This method will be called by a "manteinance" thread, and will create and
// destroy threads to adapt to the load.
// The parameter informs us about how many threads we can grow or shrink at once.
// This parameter usually will be equal to the number of Connections Pools, so that
// we can grow or shrink one thread for every connection grown or shrinked.

int DBThreadPool::reviewThreads(int threadSet) {

  int objectsInQueue = this->getQueueSize();
  int mAvg = updateMovingAverage(objectsInQueue);
	    
  ACE_DEBUG((LM_DEBUG,"(%t %T) DBThreadPool::reviewThreads. Queue: %d, M.Average:%d. Rng:(%d,%d)\n", objectsInQueue,mAvg,_minThreads,_maxThreads));

  if (mAvg>=_upperQueueLimit && (_nThreads+threadSet)<=_maxThreads) {
    return growThreads(threadSet);
  }

  ACE_OS::sleep(ACE_Time_Value(1));

  if (mAvg<=_lowerQueueLimit && (_nThreads-threadSet)>=_minThreads) {
    return shrinkThreads(threadSet);
  }

  return 0;

}


// PRIVATE ******************************************************************

// Update the moving average with this value
// returning the current value

int DBThreadPool::updateMovingAverage(int value) {

  int total = 0;

  for (unsigned int i=0;i<MOVING_AVERAGE_SIZE-1;i++) {
    _movingAverage[i] = _movingAverage[i+1];
    total += _movingAverage[i];
  }

  _movingAverage[MOVING_AVERAGE_SIZE-1] = value;
  total += value;

  return (int)(total / MOVING_AVERAGE_SIZE);


}

int DBThreadPool::growThreads(int growth) {

    ACE_DEBUG((LM_NOTICE,"(%t %T) DBThreadPool: >> Grow by %d, %d growths so far \n",growth,++_growCount)); 
    //checkmem("Before growing!!");
    this->open(growth);
    _nThreads += growth;
    //checkmem("After growing!!");

    return 1;

}

int DBThreadPool::shrinkThreads(int shrink) {

    ACE_DEBUG((LM_NOTICE,"(%t %T) DBThreadPool: << Shrink by %d. %d shrinks so far \n",shrink, --_shrinkCount));

    //checkmem("Before shrinking!!");
    for (int i=0;i<shrink;i++)
      this->submit(new DBThreadCtlMO());
    _nThreads -= shrink;
    //checkmem("After shinking!!");

    return -1;

}



/*

int DBThreadPool::checkmem(string message) {

  char file[127];
  string tmp;
  int ncampo = 23;

  sprintf(file,"/proc/%d/stat",getpid());
  ifstream f(file);
  for (int i=0;i<ncampo;i++) f >> tmp;
  f.close();

  int mem = atoi(tmp.c_str());

  ACE_DEBUG((LM_NOTICE,"DBBalancerMEM: (%s) (pid= %d, mem=#%d#,%d,%d) \n",message.c_str(),getpid(),mem,mem-_lastMemory,mem-_initialMemory));
  _lastMemory = mem;

  return mem;

}

*/
