/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "embed.h"
#include "window.h"
#include "misc_callbacks.h"
#include "misc_general.h"
#include "misc_gui.h"
#include "misc_string.h"
#include "history.h"
#include "prefs.h"
#include "mozilla_prefs.h"
#include "mozilla.h"
#include "mozcallbacks.h"
#include "downloader.h"
#include "MozRegisterComponents.h"
#include "gfilepicker.h"
#include "dialog.h"
#include "mozilla_notifiers.h"
#include "themes.h"
#include "tabbutton.h"
#include "bookmarks.h"
#include "session.h"
#include "stylesheets.h"
#include "favicon.h"

#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <libgnome/gnome-config.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-i18n.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-preferences.h>
#include <libgnomeui/gtkpixmapmenuitem.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/* local function prototypes */
static void embed_connect_signals (GaleonEmbed *embed);
static void embed_single_init (void);
static void embed_view_source_external (GaleonEmbed *embed,
					const gchar *url);
static void add_bf_menu (GtkWidget *menu, char* label, int index, 
			 GaleonEmbed *embed, int level);
static GtkWidget * new_num_accel_menu_item (gint num, gchar *origtext, 
					    gboolean lettersok, 
					    GtkWidget *menu,
					    const PixmapData *ico);
static gboolean embed_save_with_content (GaleonEmbed *embed,
					 char *path, gboolean main);
static void embed_set_save_message (GaleonEmbed *embed, char *path,
				    gboolean result);
static void embed_select_new_page (GaleonEmbed *embed);
static gint check_mozilla_version (void);
static gint read_mozilla_version (void);

/** The global list of all GaleonEmbed structures */
GList *all_embeds = NULL;

/* global character set hash and sorted title list*/
GHashTable *charsets = NULL;
GList *sorted_charset_titles = NULL;

/* tab close button global pixmap data */
PixmapData *close_pix = NULL;

/* global var to indicate whether xpcom has been initialised */
gboolean pushed_startup = FALSE;

typedef enum
{
	STRING,
	COMPOUND_TEXT
} SelType;

/* DnD drop types for embeds and tabs -- NOTE NOTE NOTE dont mess around 
 * with the order of these unless you know what you're doing! -- MattA */
const GtkTargetEntry embed_drop_types[] =
{
	{ "GALEON_URL",      0, DND_TARGET_GALEON_URL      },
	{ "text/uri-list",   0, DND_TARGET_TEXT_URI_LIST   },
	{ "_NETSCAPE_URL",   0, DND_TARGET_NETSCAPE_URL    },
	{ "STRING",          0, DND_TARGET_STRING          }
};
const gint embed_drop_types_num_items = (sizeof (embed_drop_types) / 
				         sizeof (GtkTargetEntry));

/* selection types */
const GtkTargetEntry selection_types[] =
{
	{ "STRING",        0, STRING        },
	{ "COMPOUND_TEXT", 0, COMPOUND_TEXT }
};
const gint selection_types_num_targets = (sizeof (selection_types) /
					  sizeof (GtkTargetEntry));


/**
 * embed_create_in_window_chrome: creates a GaleonEmbed and puts it inside
 * of window, or in a new window if window is NULL.  if url is non-null, it
 * will be loaded in the embed.  flags, defined and described in
 * galeon_types.h, can contain special instructions about the creation of
 * the embed and its window.  chrome_mask contains the chrome settings for
 * the embed and its window.
 */
GaleonEmbed *
embed_create_in_window_chrome (GaleonWindow *window,
			       GaleonEmbed *previous_embed,
			       const gchar *url,
			       EmbedCreateFlags flags,
			       guint32 chrome_mask)
{
	GaleonEmbed *embed;
	GaleonWindow *previous_window;
	gchar *default_url;

	/* create the embed */
	embed = embed_create_no_window ();
	return_val_if_not_embed (embed, NULL);

	/* keep create flags */
	embed->create_flags = flags;

	if (flags & EMBED_CREATE_DONT_AUTOSAVE_INITIALLY)
		embed->dont_autosave_session = TRUE;
	else embed->dont_autosave_session = FALSE;

	/* setup chromedness */
	if ((chrome_mask & GTK_MOZ_EMBED_FLAG_OPENASCHROME) != 0)
		embed->is_chrome = TRUE;

	/* create a parent window if one wasn't given */
	if (!window)
	{
		previous_window = previous_embed ? previous_embed->parent_window
						 : NULL;
		window = window_create (chrome_mask, previous_window);
	}

	/* add embed to window */
	window_add_embed (window, embed, flags);

	if (flags & EMBED_CREATE_LOAD_DEFAULT_URL)
	{
		embed_set_visibility (embed, TRUE);

		if (previous_embed != NULL && 
		    eel_gconf_get_integer (CONF_GENERAL_NEWPAGE_TYPE) == STARTPAGE_LAST)
		{
			/* copy across session history (the wrapper must be
			 * initialized) thats why the embed_load_url */
			embed_load_url (embed, "about:blank");
			mozilla_copy_session_history (previous_embed, embed);
		} else 
		{
			default_url = embed_get_default_url (previous_embed);
			embed_load_url (embed, default_url);
			g_free (default_url);
		}
	}
	else if (url)
	{
		embed_set_visibility (embed, TRUE);
		embed_load_url (embed, url);

		/* set modified location, so the location will show up
		 * in the entry */
		embed->modified_location = g_strdup (url);
		embed->location_is_modified = TRUE;
		if (embed->is_active)
			window_update_location_entry_text (window);
	}
	else if (!(flags & EMBED_CREATE_DONT_SHOW))
	{
		embed_set_visibility (embed, TRUE);
		embed_load_url (embed, "about:blank");
	}

	/* done! */
	return embed;
}

/**
 * embed_create_in_window: calls embed_create_in_window_chrome, using the
 * default chrome style
 */
GaleonEmbed *
embed_create_in_window (GaleonWindow *window, GaleonEmbed *previous_embed,
			const gchar *url, EmbedCreateFlags flags)
{
	return embed_create_in_window_chrome (window, previous_embed, url,
					      flags, DEFAULT_CHROME);
}

/**
 * embed_create_after_embed_chrome: wraps embed_create_in_window, allowing
 * caller to supply the previous embed and a new_window boolean, rather
 * than an actual window.
 */
GaleonEmbed *
embed_create_after_embed_chrome (GaleonEmbed *previous_embed,
				 gboolean new_window,
				 const gchar *url,
				 EmbedCreateFlags flags,
				 guint32 chrome_mask)
{
	/* FIXME: do this correct */
	if (url && strncmp (url, "javascript", 10) == 0) {
		embed_load_url (previous_embed, url);
		return previous_embed;
	}

	if (new_window || (previous_embed == NULL))
		return embed_create_in_window_chrome (NULL, previous_embed,
						      url, flags, chrome_mask);
	else
		return embed_create_in_window_chrome (
					       previous_embed->parent_window,
					       previous_embed, url, flags,
					       chrome_mask);
}

/**
 * embed_create_after_embed: calls embed_create_after_embed_chrome, using
 * the default chrome style
 */
GaleonEmbed *
embed_create_after_embed (GaleonEmbed *previous_embed,
			  gboolean new_window,
			  const gchar *url,
			  EmbedCreateFlags flags)
{
	return embed_create_after_embed_chrome (previous_embed, new_window,
						url, flags, DEFAULT_CHROME);
}

/**
 * embed_create_no_window: create a GaleonEmbed without reference to
 * a parent window
 */
GaleonEmbed *
embed_create_no_window (void)
{
	GaleonEmbed *embed;

	/* build an embed structure */
	embed = g_new0 (GaleonEmbed, 1);

	/* add to global list of all embeds */
	all_embeds = g_list_prepend (all_embeds, embed);

	/* don't quit, we have an embed */
	session_server_stop_timeout ();

	/* no content for the location or title */
	embed->location = NULL;
	embed->realurl = NULL;
	embed->modified_location = NULL;
	embed->location_is_modified = FALSE;
	embed->title = g_strdup (_("Untitled"));
	embed->title_utf8 = mozilla_locale_to_utf8 (embed->title);

	embed->notebook_close_button = NULL;
	embed->notebook_close_pixmap = NULL;

	embed->usersheets = g_hash_table_new (g_str_hash, g_str_equal);
	embed->temp_statusbar_message = NULL;

	embed->print_dialog = NULL;

	embed->being_closed = FALSE;

	/* initial zoom level */
	embed->zoom = 100;
	
	/* make an embedding widget and check */
	embed->mozembed = gtk_moz_embed_new ();
	g_assert (GTK_IS_MOZ_EMBED (embed->mozembed));

	/* ref/unref widget -- is this really needed? */
	/* NB: apparently it is to get rid of some warnings, but I'm
	 * still not clear why -- MattA 13/4/2001 */
	/* as of 1/6/2001, it still doens't work... ! */
	gtk_widget_ref (GTK_WIDGET (embed->mozembed));
	gtk_object_set_data_full (GTK_OBJECT (embed->mozembed), "mozembed",
				  GTK_WIDGET (embed->mozembed),
				  (GtkDestroyNotify) gtk_widget_unref);

	/* setup data so we can always find GaleonEmbed */
	gtk_object_set_data (GTK_OBJECT (embed->mozembed), 
			     "GaleonEmbed", embed);

	/* connect appropriate signals */
	embed_connect_signals (embed);

	/* initialise autoreload data (defaults to 10 minutes) */
	embed->autoreload_info = g_new0 (AutoreloadInfo, 1);
	embed->autoreload_info->enabled = FALSE;
	embed->autoreload_info->delay = 10;
	embed->autoreload_info->delay_unit = 1;
	embed->autoreload_info->timeout_id = 0;
	
	/* set magic as this is now ready to go */
	embed->magic = GALEON_EMBED_MAGIC;

	embed->parent_window = NULL;
#ifdef ENABLE_NAUTILUS_VIEW
	embed->nautilus_view = NULL;
#endif

	/* clear gestures stuff */
	embed->gesture_motion_signal = 0;
	embed->gesture_press_signal = 0;
	embed->gesture_release_signal = 0;

	/* done! */
	return embed;
}

/**
 * embed_get_vs_url_from_string: prefixes "view-source:" to the
 * given string.  returned string must be freed by caller
 */
gchar *
embed_get_vs_url_from_string (const gchar *url)
{
	gchar *vs_url;

	/* prepend view-source: protocol */
	vs_url = g_strconcat ("view-source:", url, NULL);

	/* return url */
	return vs_url;
}

/**
 * embed_get_default_url: get the default URL on new page creation
 */
gchar *
embed_get_default_url (GaleonEmbed *previous_embed)
{
	const gchar *last_page_url;
	gchar *home_page_url;
	gint page_type;

	/* find out where we're supposed to start */
	if (previous_embed == NULL)
	{
		page_type = eel_gconf_get_integer 
			(CONF_GENERAL_STARTPAGE_TYPE);
	}
	else
	{
		page_type = eel_gconf_get_integer (CONF_GENERAL_NEWPAGE_TYPE);
	}

       	/* return the appropriate page */
	if (page_type == STARTPAGE_HOME)
	{
		/* get location of home page */
		home_page_url = eel_gconf_get_string(CONF_GENERAL_HOMEPAGE);

		/* load home page, if set */
		if (home_page_url != NULL)
		{
			return home_page_url;
		}
	}

	/* get location of last page: use previous browser if one,
	 * otherwise resort to fetching it from global history */
	if (previous_embed != NULL && previous_embed->location != NULL)
	{
		last_page_url = previous_embed->location;
	}
	else
	{
		last_page_url = history_get_last_url ();
	}
	
	if (page_type == STARTPAGE_LAST && last_page_url != NULL)
	{
		/* return last page pointed at */
		return g_strdup (last_page_url);
	}

	/* even in case of error, it's a good default */
	return g_strdup ("about:blank");
}

/**
 * embed_set_visibility: set the visiblity of a single embed
 */
void
embed_set_visibility (GaleonEmbed *embed, gboolean visibility)
{
	GaleonWindow *window;

	/* check args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	/* test for hiding */
	if (!visibility && embed->is_visible && window)
	{
		window_hide_one_embed (window);
	}

	/* test for showing */
	if (visibility && !embed->is_visible)
	{
		/* show the widget */
		/* don't ask me why this has to be done BEFORE showing
		 * the main window, it just does... */
		gtk_widget_show (embed->mozembed);

		if (window)
		{
			window_show_one_embed (window);
		}
	}

	/* new status */
	embed->is_visible = visibility;
}

/**
 * embed_load_url: interpret and load a URL into a given GaleonEmbed
 */
void
embed_load_url (GaleonEmbed *embed, const gchar *url)
{
	gchar *clean_url;

	/* check arguments */
	return_if_not_sane_embed (embed);
	g_return_if_fail (url != NULL);

	clean_url = misc_string_remove_outside_whitespace (url);

	/* initialise embed whenever a document is first loaded */
	if (embed->wrapper == NULL)
	{
		embed_wrapper_init (embed);

		/* hack to work around a bunch of problems when nothing
		 * is loaded */
		embed->realurl = g_strdup (clean_url);
		gtk_moz_embed_load_url (GTK_MOZ_EMBED (embed->mozembed),
					"about:blank");
	}
	else
	{
		/* load the URL */
		gtk_moz_embed_load_url (GTK_MOZ_EMBED(embed->mozembed),
					clean_url);
	}

	/* free allocated string */
	g_free (clean_url);
}

void
embed_grab_focus (GaleonEmbed *embed)
{
	if (!embed) return;
	gtk_widget_grab_focus (GTK_BIN (embed->mozembed)->child);
}

/**
 * embed_activate_link_mouse: handle a link click, using a mouse state
 */
gboolean
embed_activate_link_mouse (GaleonEmbed *previous, GaleonEmbed **new_embed,
			   const gchar *url, GdkEventButton *event)
{
	LinkState link_state;

	/* get link state from mouse state */
	link_state = misc_general_mouse_state_to_link_state (
						event->button, event->state);

	/* do the activation */
	return embed_activate_link (previous, new_embed, url, link_state);
}

/**
 * embed_activate_link_keyboard: handle a link click, using a keyboard state
 */
gboolean
embed_activate_link_keyboard (GaleonEmbed *previous, GaleonEmbed **new_embed,
			      const gchar *url, guint evstate)
{
	LinkState link_state;

	/* get link state from mouse state */
	link_state = misc_general_key_state_to_link_state (evstate);

	/* do the activation */
	return embed_activate_link (previous, new_embed, url, link_state);
}

/**
 * embed_activate_link: generalized URL activation function
 */
gboolean
embed_activate_link (GaleonEmbed *previous, GaleonEmbed **new_embed,
		     const gchar *url, LinkState link_state)
{
	gboolean tabbed_mode, handled;
	gboolean shifted, ctrled, alted;
	GaleonEmbed *new = NULL;
	guint button;

	/* check URL is valid */
	if (url == NULL)
	{
		return FALSE;
	}

	/* parse link state */
	handled = FALSE;
	button = (link_state & LINKSTATE_BUTTON_MASK) + 1; 
	shifted = link_state & LINKSTATE_SHIFTED;
	ctrled = link_state & LINKSTATE_CTRLED;
	alted = link_state & LINKSTATE_ALTED;
	
        /* parse button */
	switch (button)
	{
	case 1:
		/* left button (or no button) */
		if (!previous) {
			new = previous = embed_create_in_window (NULL, NULL,
				"about:blank", EMBED_CREATE_RAISE_WINDOW);
		}
		if (shifted) {
			downloader_save_link (previous, url);
		} else {
			embed_load_url (previous, url);
		}
		handled = TRUE;
		break;

	case 2:
		/* middle button */
#ifdef ENABLE_NAUTILUS_VIEW
		if (EMBED_IN_NAUTILUS (previous))
		{
			galeon_nautilus_view_open_in_new_window
				(previous->nautilus_view, url);
		}
		else
#endif
		{
			/* get tabbed mode */
			if (link_state == LINKSTATE_NEWTAB)
				tabbed_mode = TRUE;
			else if (link_state == LINKSTATE_NEWWIN)
				tabbed_mode = FALSE;
			else
				tabbed_mode = eel_gconf_get_boolean (
							CONF_TABS_TABBED);

			/* if shift was held, switch tabbed mode */
			if (shifted) tabbed_mode = !tabbed_mode;

			/* if ctrl was held, switch jump-to-tab policy */
			if (ctrled)
			{
				if (eel_gconf_get_boolean (
						CONF_TABS_TABBED_AUTOJUMP))
				{
					new = embed_create_after_embed (
						previous, !tabbed_mode, url,
						EMBED_CREATE_RAISE_WINDOW |
						EMBED_CREATE_GRAB_FOCUS |
						EMBED_CREATE_NEVER_JUMP);
				}
				else
				{
					new = embed_create_after_embed (
						previous, !tabbed_mode, url,
						EMBED_CREATE_RAISE_WINDOW |
						EMBED_CREATE_GRAB_FOCUS |
						EMBED_CREATE_FORCE_JUMP);
				}
			}
			/* otherwise, just open the embed like normal */
			else
			{
				new = embed_create_after_embed (previous,
					!tabbed_mode, url,
					EMBED_CREATE_RAISE_WINDOW |
					EMBED_CREATE_GRAB_FOCUS);
			}
		}
		handled = TRUE;
		break;

	case 3:
		/* context menu shown by other stuff */
		handled = FALSE;
		break;

	default:
		g_warning ("link selected with unusual button");
		handled = FALSE;
		break;
	}

	/* store return value (NULL if not handled) */
	if (new_embed != NULL) {
		*new_embed = new;
	}

	/* done */
	return handled;
}

/**
 * embed_switch_to_page: switch to the embed notebook page
 */
void 
embed_switch_to_page (GaleonEmbed *embed)
{
	GaleonWindow *window;
	gint page;

	/* check */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

	if (window)
	{
		/* lookup the page number */
		page = gtk_notebook_page_num (GTK_NOTEBOOK (window->notebook),
					      GTK_WIDGET (embed->mozembed));
		
		/* jump to that page */
		gtk_notebook_set_page (GTK_NOTEBOOK (window->notebook), page);
	}
}

/**
 * embed_startup_init: initialisation that needs to occur only one time
 * before any gtkmozembed object is created
 */
void
embed_startup_init ()
{
	gchar *profile, *mozprefs, *build_test;
	gboolean set_mozilla_prefs = FALSE;
	
	/* init mozilla home */
	gtk_moz_embed_set_comp_path (g_getenv ("MOZILLA_FIVE_HOME"));

	/* set a path for the profile */
	profile = g_concat_dir_and_file (g_get_home_dir (), ".galeon/mozilla");

	/* no mozilla prefs ? or new galeon build */
	mozprefs = g_concat_dir_and_file (profile, "/galeon/prefs.js"); 
	build_test = eel_gconf_get_string ("/apps/galeon/gconf_test");
	if (!g_file_exists(mozprefs) || 
	    build_test == NULL ||
	    strncmp (build_test, __TIME__, 8) != 0)
	{
		set_mozilla_prefs = TRUE;
		eel_gconf_set_string ("/apps/galeon/gconf_test", __TIME__);
	}
	g_free (mozprefs);
	g_free (build_test);

	/* initialize profile */
	gtk_moz_embed_set_profile_path (profile, "galeon");
	g_free (profile);


	/* startup done */
	gtk_moz_embed_push_startup ();
	pushed_startup = TRUE;

	embed_single_init ();

	if (!mozilla_register_components ())
	{
		GtkWidget *dialog;
		dialog = gnome_error_dialog (_("Unable to initialize mozilla libraries"));
		gnome_dialog_run_and_close (GNOME_DIALOG (dialog));
		exit (0);
	}

	/* Register the javascript console observer */
	mozilla_register_js_listener ();

	/* set mozilla prefs if necessary */
	if (set_mozilla_prefs)
	{
		mozilla_prefs_set ();
	}

	/* set the user agent, this is always neccesary because the galeon
	   version may change */
	mozilla_prefs_set_user_agent ();

	/* check mozilla version */
	if (check_mozilla_version ())
	{
		exit (1);
	}
	
	/* installs notifiers for some mozilla prefs */
	mozilla_notifiers_init();

	/* only register the external mailto handler if it isn't galeon.
	   Note that this is startup only, as gnome-config does not
	   support notifiers. If the user does the stupid thing while galeon
	   is running, they're SOL. */
	if (!prefs_galeon_used_by_gnome_for_protocol ("mailto"))
	{
		mozilla_register_MailtoProtocolHandler ();
	}

	/* optionally register galeon to handle ghelp protocols */
	if (eel_gconf_get_boolean(CONF_GHELP_USE_GALEON) == TRUE ||
	    prefs_galeon_used_by_gnome_for_protocol ("man") ||
	    prefs_galeon_used_by_gnome_for_protocol ("info") ||
	    prefs_galeon_used_by_gnome_for_protocol ("ghelp") ||
	    prefs_galeon_used_by_gnome_for_protocol ("gnome-help"))
	{
		mozilla_register_gHelpProtocolHandlers ();
	}
}

/**
 * read_mozilla_version: extract mozilla version from mozilla prefs file
 */
static gint
read_mozilla_version (void)
{
	gchar agentstr[160], *p, *endptr;
	gint ver= 0, v;
	gchar *preffile;
	FILE *f;

	preffile = g_strconcat (getenv ("MOZILLA_FIVE_HOME"),
				"/defaults/pref/all.js", NULL);
	f = fopen (preffile, "r");
	
	if (f == NULL)
	{
		g_warning ("error opening %s: %s", preffile, strerror (errno));
		g_free (preffile);
		return 0;
	}

	while ((p = fgets (agentstr, 160, f)) != NULL)
	{
		if ((p = strstr (agentstr, "useragent.misc")) != NULL &&
			(p = strstr (p, "rv:")) != NULL)
			break;
	}
	fclose(f);

	if (p == NULL)
	{
		g_warning ("Mozilla version not found in %s", preffile);
	}
	else
	{
		p += 3;
		v = strtol (p, &endptr, 10);
		ver = v << 24;
		if (*endptr == '.')
		{
			p = endptr + 1;
			v = strtol (p, &endptr, 10);
			ver += v << 16;
			if (*endptr == '.')
			{
				p = endptr + 1;
				v = strtol (p, &endptr, 10);
				ver += v << 8;
			}
		}
		if (*endptr == '+')
		{
			ver++;
		}
	}

	g_free (preffile);
	return ver;
}

/**
 * check_mozilla_version: verify that the runtime mozilla is the same version
 * as the compile time.  If not print a warning and display a confirmation
 * dialog.
 */
static gint
check_mozilla_version (void)
{
	mozilla_version = read_mozilla_version ();

	if (MOZILLA_VERSION != mozilla_version)
	{
		gchar *questionStr, *rememberStr, *verMatchStr;
		gchar *version1_str, *version2_str;
		gboolean continuev=0, rememberv=0;

		g_warning ("compiled MOZILLA_VERSION ("VERSIONFMT") !="
			" detected mozilla_version ("VERSIONFMT")",
			VERSIONFMTARGS (MOZILLA_VERSION),
			VERSIONFMTARGS (mozilla_version));

		verMatchStr = g_strdup_printf ("%xv%x", MOZILLA_VERSION,
						mozilla_version);
		rememberStr = eel_gconf_get_string 
				(CONF_MOZ_MISMATCH);

		if (rememberStr != NULL)
		{
			int ignore = strcmp (rememberStr, verMatchStr);
			g_free (rememberStr);
			if (ignore == 0)
			{
				g_free (verMatchStr);
				return eel_gconf_get_integer
				    (CONF_MOZ_MISMATCH_ACTION);
			}
		}

		version1_str = g_strdup_printf (VERSIONFMT,
					VERSIONFMTARGS (MOZILLA_VERSION));
		version2_str = g_strdup_printf (VERSIONFMT,
					VERSIONFMTARGS (mozilla_version));

		questionStr = g_strdup_printf (_(
"The version of Mozilla Galeon was compiled against (%s) "
"does not match the detected version (%s).\n\n"
"You may experience segfaults and other oddities.  Please test "
"with a matching version of Mozilla before reporting any bugs.\n\n"
"Continue anyway?"),
			version1_str, version2_str);

		g_free (version1_str);
		g_free (version2_str);

		continuev = dialog_confirm_check (NULL, _("Version mismatch!"),
					questionStr, _("Don't ask me again"),
					&rememberv);
	
		g_free (questionStr);
		if (rememberv)
		{
			eel_gconf_set_string
				(CONF_MOZ_MISMATCH,
				verMatchStr);
			eel_gconf_set_integer
				(CONF_MOZ_MISMATCH_ACTION,
				 !continuev);
		}
		g_free (verMatchStr);
		return !continuev;
	}
	return 0;
}


/**
 * embed_connect_signals: connect the necessary signals to the embed
 */
static void 
embed_connect_signals (GaleonEmbed *embed)
{
	gint i;

        /* signals to connect on each embed widget */
	static const struct
	{ 
		char *event; 
		void *func; /* should be a GtkSignalFunc or similar */
	}
	signal_connections[] =
	{
		{ "location",        mozembed_location_changed_cb  },
		{ "title",           mozembed_title_changed_cb     },
		{ "net_start",       mozembed_load_started_cb      },
		{ "net_stop",        mozembed_load_finished_cb     },
		{ "net_state_all",   mozembed_net_status_change_cb },
		{ "progress",        mozembed_progress_change_cb   },
		{ "link_message",    mozembed_link_message_cb      },
		{ "js_status",       mozembed_js_status_cb         },
		{ "visibility",      mozembed_visibility_cb        },
		{ "destroy_browser", mozembed_destroy_brsr_cb      },
		{ "dom_mouse_down",  mozembed_dom_mouse_down_cb    },	
		{ "dom_mouse_click", mozembed_dom_mouse_click_cb   },
		{ "dom_key_press",   mozembed_dom_key_press_cb     },
		{ "size_to",         mozembed_size_to_cb           },
		{ "new_window",      mozembed_new_window_cb        },
		{ "security_change", mozembed_security_change_cb   },

		/* terminator -- must be last in the list! */
		{ NULL, NULL } 
	};
	
	/* connect signals */
	for (i = 0; signal_connections[i].event != NULL; i++)
	{
		gtk_signal_connect_while_alive (GTK_OBJECT(embed->mozembed),
						signal_connections[i].event,
						signal_connections[i].func, 
						embed,
						GTK_OBJECT(embed->mozembed));
	}
	
	/* also connect destroy (but not with "while_alive") */
	gtk_signal_connect (GTK_OBJECT (embed->mozembed), "destroy",
			    GTK_SIGNAL_FUNC (mozembed_destroy_cb), embed);

	/* set gtkmozembed drag and drop destination */
	gtk_drag_dest_set (GTK_WIDGET(embed->mozembed), GTK_DEST_DEFAULT_ALL,
			   embed_drop_types, embed_drop_types_num_items,
			   GDK_ACTION_COPY | GDK_ACTION_MOVE |
			   GDK_ACTION_LINK | GDK_ACTION_ASK );

	/* set gtkmozembed drag and drop signals */
	gtk_signal_connect (GTK_OBJECT (embed->mozembed), "drag_data_received",
			    GTK_SIGNAL_FUNC (embed_drag_data_received_cb),
			    embed);

	/* set links drag signal */
	gtk_signal_connect (GTK_OBJECT (embed->mozembed), "drag_data_get",
			    GTK_SIGNAL_FUNC (window_drag_data_get_cb),
			    embed);

	/* set selection signals */
	gtk_selection_add_targets (GTK_WIDGET(embed->mozembed),
				   GDK_SELECTION_PRIMARY, 
				   selection_types, 
				   selection_types_num_targets);
	gtk_selection_add_targets (GTK_WIDGET(embed->mozembed),
				   gdk_atom_intern("CLIPBOARD", FALSE),
				   selection_types, 
				   selection_types_num_targets);
	gtk_signal_connect (GTK_OBJECT(embed->mozembed), "selection_get",
			    GTK_SIGNAL_FUNC (embed_selection_get_cb),
			    NULL);
}

/**
 * embed_single_init: init the gtkmozembed Single object
 */
static void
embed_single_init (void)
{
	GtkMozEmbedSingle *single;

	/* get single */
	single = gtk_moz_embed_single_get ();
	if (single == NULL)
	{
		g_warning ("Failed to get singleton embed object!\n");
	}

	/* allow creation of orphan windows */
	gtk_signal_connect (GTK_OBJECT (single), "new_window_orphan",
			    GTK_SIGNAL_FUNC (new_window_orphan_cb),  NULL);
}

/**
 * embed_wrapper_init: call it after the first page is loaded
 */
void 
embed_wrapper_init (GaleonEmbed *embed)
{
	embed->wrapper = mozilla_wrapper_init (embed);
}

/**
 * embed_progress_clear: clear all infos about download progress
 */
void
embed_progress_clear (GaleonEmbed *embed)
{
	/* set all progress values to 0 */ 
	embed->load_percent = 0;
	embed->bytes_loaded = 0;
	embed->max_bytes_loaded = 0;
}

/**
 * embed_save_image: save an image from an embed
 */
void 
embed_save_image (GaleonEmbed *embed, gchar *url, gboolean ask)
{
	embed_save_url (embed, url, CONF_STATE_SAVE_IMAGE_DIR, 
			ask, FALSE, FALSE, TRUE);
}

/**
 * embed_set_image_as_background: set an image as the desktop background
 */
void 
embed_set_image_as_background (GaleonEmbed *embed, gchar *url)
{
	gchar *file, *path;

	/* get the filename and check */
	file = g_basename (url);
	g_return_if_fail (file != NULL);
	g_return_if_fail (strlen (file) != 0);

	/* build a path to save it in */
	path = g_strconcat (g_get_home_dir (), "/.galeon/", file, NULL);

	/* save the image */
	mozilla_save_url (embed, url, path, ACTION_SETBACKGROUND, NULL);

	/* free */
	g_free (path);
}

/**
 * embed_save_document: save this document (or frame)
 */
void embed_save_document (GaleonEmbed *embed, gboolean main)
{
	gchar *url = NULL;
		
	/* get either frame or main document */
	url = main ? mozilla_get_main_document_url (embed) :
		mozilla_get_document_url (embed);

	if (!url)
	{
		gnome_error_dialog (_("No document is loaded"));
		return;
	}

	/* save it */
	embed_save_url (embed, url, CONF_STATE_SAVE_DIR, TRUE, TRUE, main, TRUE);

	/* free */
	g_free (url);
}

/**
 * embed_save_url: save a given URL
 * FIXME pass main here is weird
 */
void 
embed_save_url (GaleonEmbed *embed, gchar *target, 
		gchar *default_dir_pref, gboolean ask,
		gboolean ask_content, gboolean main, 
		gboolean save_dir)
{
	GnomeVFSURI *uri;
	gchar *retPath = NULL;
	gchar *fileName = NULL;
	gchar *dirName = NULL;
	gchar *retDir;
	gboolean ret;
	gboolean content;
	GtkWidget *parent = NULL;

	/* Get a filename from the target url */
	uri = gnome_vfs_uri_new (target);
	if (uri)
	{
		fileName = gnome_vfs_uri_extract_short_name (uri);
		gnome_vfs_uri_unref (uri);
	}

	/* if this is a normal download, and we have a previously saved
	 * downloading dir, use it */
	if (!strcmp (default_dir_pref, CONF_DOWNLOADING_DIR))
	{
		dirName = eel_gconf_get_string (CONF_STATE_LAST_DOWNLOAD_DIR);
		if (dirName && dirName[0] == '\0')
		{
			g_free (dirName);
			dirName = NULL;
		}
	}

	/* otherwise, get the dir from the supplied pref */
	if (strcmp (default_dir_pref, CONF_DOWNLOADING_DIR) || !dirName)
		dirName = eel_gconf_get_string (default_dir_pref);

	/* if we still don't have dir, use the home directory */
	if (!dirName || dirName[0] == '\0')
	{
		if (dirName) g_free (dirName);
		dirName = g_strdup (g_get_home_dir ());
	}

	/* find parent window */
#ifdef ENABLE_NAUTILUS_VIEW
	if (!EMBED_IN_NAUTILUS (embed))
#endif
	{
		if (embed->parent_window)
			parent = embed->parent_window->wmain;
	}

	/* get the destination filename */
	if (ask)
	{
		/* show the file picker */
		ret = show_file_picker (parent,
					_("Select the destination filename"),
					dirName, fileName, modeSave, &retPath,
					ask_content ? &content : NULL,
					NULL, NULL);
	}
	else
	{
		retPath = g_concat_dir_and_file (dirName, fileName);
		ret = dialog_check_overwrite_file (retPath, parent);
	}

	/* go ahead with the download */
	if (ret == TRUE)
	{
		uri = gnome_vfs_uri_new (retPath);

		if (uri)
		{
			retDir = gnome_vfs_uri_extract_dirname (uri);

			if (ask_content && content)
			{
				ret = embed_save_with_content (embed,
							       retPath, main);
			}
			else
			{
				ret = mozilla_save_url (embed, target, 
							retPath, ACTION_NONE,
							NULL);
			}
			embed_set_save_message (embed, retPath, ret);

			/* set default save dir in pref if requested */
			if (save_dir)
				eel_gconf_set_string (default_dir_pref, 
						      retDir);

			/* update the this-session-only download dir pref */
			if (!strcmp (default_dir_pref, CONF_DOWNLOADING_DIR))
				eel_gconf_set_string (
					CONF_STATE_LAST_DOWNLOAD_DIR, retDir);

			g_free (retDir);
			gnome_vfs_uri_unref (uri);
		}
	}

	g_free (dirName);
	g_free (fileName);
	g_free (retPath);
}

static gboolean
embed_save_with_content (GaleonEmbed *embed, char *path, gboolean main)
{
	gchar *data_dir;
	gboolean ret;

	return_val_if_not_embed (embed, FALSE);

	data_dir = g_strconcat (path, "content", NULL);

	if (mkdir (data_dir, 0755) != -1)
	{
		ret = mozilla_save_document (embed, path, data_dir,
					     ACTION_NONE, main); 
	}
	else 
	{
		ret = FALSE;
	}

	g_free (data_dir);

	return ret;
}

static void
embed_set_save_message (GaleonEmbed *embed, char *path, gboolean result)
{
	gchar *temp_message;

	if (embed == NULL) return;
	return_if_not_embed (embed);

	if (result) temp_message = g_strdup_printf(_("Saved as %s"), path);
	else temp_message = g_strdup_printf(_("Save as %s failed"), path);
 
	embed_set_temp_statusbar_message (embed, temp_message);

	g_free (temp_message);
}

/**
 * embed_set_temp_statusbar_message: set the temporary statusbar message to
 * the given string, and update the window if necessary
 */
void
embed_set_temp_statusbar_message (GaleonEmbed *embed, const gchar *message)
{
	gboolean changed = TRUE;

	return_if_not_sane_embed (embed);

	/* free old message */
	if (embed->temp_statusbar_message)
		g_free (embed->temp_statusbar_message);
	/* if the message is still null, don't bother updating the statusbar */
	else if (message == NULL) changed = FALSE;

	if (!message) embed->temp_statusbar_message = NULL;
	else
	{
		char *ptr;

		embed->temp_statusbar_message = g_strdup (message);

		/* remove \n or the statusbar height will grow */
		for (ptr = embed->temp_statusbar_message; *ptr != '\0'; ptr++)
		{
			if (*ptr == '\n') *ptr = ' ';
		}
	}

#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed)) {
		galeon_nautilus_view_set_statusbar (embed->nautilus_view, 
						    message);
	} 
	else
#endif
	{
		GaleonWindow *window = embed->parent_window;

		/* if this embed is active, and the message hasn't remained
		 * null, update the statusbar */
	    	if (embed->is_active && changed)
			window_statusbar_update_message (window);
	}
}

/**
 * embed_view_source: view web page source 
 */
void embed_view_source (GaleonEmbed *embed, gboolean main, 
			gboolean new_window)
{
	gchar *url;
	
	if (main) {
		url = mozilla_get_main_document_url (embed);
	} 
	else {
		url = mozilla_get_document_url (embed);
	}

	if (!url)
	{
		gnome_error_dialog (_("No document is loaded"));
		return;
	}

	if (eel_gconf_get_boolean (CONF_PROGRAMS_USE_EXTERNAL_SOURCE_VIEWER))
	{
		embed_view_source_external (embed, url);
	}
	else
	{
		gchar *vs_url;

		vs_url = embed_get_vs_url_from_string (url);
		embed_create_after_embed (embed, new_window, vs_url,
					  EMBED_CREATE_FORCE_JUMP |
					  EMBED_CREATE_RAISE_WINDOW);
		g_free (vs_url);
	}
	g_free (url);
}

/**
 * embed_view_source_external: view web page source with an external viewer 
 */
static void embed_view_source_external (GaleonEmbed *embed, const gchar *url)
{
	char *filename;
	char *htmlfile;
	int result;

	filename = g_strconcat(g_get_tmp_dir(),"/galeon-viewsource-XXXXXX", NULL);

	/* get a name for the temporary file */
	result = mkstemp (filename);

	if (result == -1)
	{
		gnome_error_dialog ( _("Could not create a "
				       "temporary file"));
		return;
	}
	close (result);

	htmlfile = g_strconcat (filename, ".html", NULL);
	rename (filename, htmlfile);
	mozilla_save_url (embed, url, htmlfile, ACTION_VIEWSOURCE, NULL);

	g_free (htmlfile);
	g_free (filename);
}

/**
 * embed_close: close a GaleonEmbed
 */
void
embed_close (GaleonEmbed *embed)
{
	GaleonWindow *window;
	return_if_not_sane_embed (embed);

	embed->being_closed = TRUE;

	/* stop any pending loads */
	gtk_moz_embed_stop_load (GTK_MOZ_EMBED (embed->mozembed));

#ifdef ENABLE_NAUTILUS_VIEW
	if (!EMBED_IN_NAUTILUS (embed))
#endif
	{
		window = embed->parent_window;
		
		/* select the new page before destroying the current one */
		embed_select_new_page (embed);
		
		/* Give the focus to the new active embed */
		if (window->active_embed && (window->active_embed != embed))
		{
			embed_grab_focus (window->active_embed);
		}
	}

	stylesheets_free_usersheets (embed);

	/* if we're in a gesture, clean up */
	if (embed->gesture_release_signal)
		mozembed_gesture_disconnect (embed);

	/* destroy the embedding widget -- this will also destroy 
	 * the notebook tab and its label, and remove it from 
	 * the relevant lists */
	gtk_widget_destroy (GTK_WIDGET (embed->mozembed));
}

/**
 * embed_open_frame: open the focused frame
 */
void 
embed_open_frame (GaleonEmbed *embed, gboolean same_embed, gboolean new_window)
{
	gchar *url;
	
	/* get document URL */
	url = mozilla_get_document_url (embed);
	if (url == NULL)
	{
		return;
	}

	/* load it */
	if (same_embed)
	{
		embed_load_url (embed, url);
	} 
	else 
	{
		embed_create_after_embed (embed, new_window, url,
					  EMBED_CREATE_FORCE_JUMP |
					  EMBED_CREATE_RAISE_WINDOW);
	}

	/* free */
	g_free (url);
}

/*
 * embed_reload: call gtk_moz_embed_reload.
 */
void
embed_reload (GaleonEmbed *embed, int flags)
{
	gtk_moz_embed_reload (GTK_MOZ_EMBED (embed->mozembed), 
			      flags);
}

/**
 * embed_update_location: updates the embed's location variable, and also
 * the parent window's location entry, if appropriate.
 */
void
embed_update_location (GaleonEmbed *embed)
{
	/* set the actual location */
	if (embed->location) g_free (embed->location);
	embed->location = mozilla_get_real_url (embed);

	/* clear modified location, if about:blank wasn't loaded, or if
	 * it was, but if about:blank is also in the modified location */
	if (embed->location_is_modified &&
	    (strcmp (embed->location, "about:blank") ||
	     (embed->modified_location &&
	      !strcmp (embed->modified_location, "about:blank"))))
	{
		embed->location_is_modified = FALSE;
		if (embed->modified_location)
		{
			g_free (embed->modified_location);
			embed->modified_location = NULL;
		}
	}

#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		galeon_nautilus_view_set_location (embed->nautilus_view,
						   embed->location);
	}
	else
#endif
	{
		GaleonWindow *window;

		/* get the parent window */
		window = embed->parent_window;
		return_if_not_window (window);
		
		/* update the text */
		if (embed->is_active)
			window_update_location_entry_text (window);
	}
}

/**
 * embed_update_title: updates the embed's title.  also updates the tab
 * title and the parent window's title, if appropriate.
 */
void
embed_update_title (GaleonEmbed *embed)
{
	gchar *title, *title_utf8;

	return_if_not_embed (embed);

	/* free existing titles */
	if (embed->title) g_free (embed->title);
	if (embed->title_utf8) g_free (embed->title_utf8);

	/* get both versions of the title */
	title = mozilla_get_document_title (embed, &title_utf8);

	/* use "Untitled" if we didn't get a title */
	if (title == NULL || strlen (title) == 0)
	{
		embed->title = g_strdup (_("Untitled"));
		embed->title_utf8 = mozilla_locale_to_utf8 (embed->title);

		if (title) g_free (title);
		if (title_utf8) g_free (title_utf8);
	}
	/* otherwise, set the title */
	else
	{
		embed->title = title;
		embed->title_utf8 = title_utf8;
	}

	/* set notebook label (although this might not be visible) */
	embed_update_tab_title (embed);

	/* if this page isn't being viewed, get out now */
	if (!embed->is_active)
		return;

#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		galeon_nautilus_view_set_title (embed->nautilus_view,
						embed->title);
	}
	else
#endif	
	{
		GaleonWindow *window;

		window = embed->parent_window;
		return_if_not_window (window);

		/* tell the parent window to update its title */
		window_update_title (window);
	}
}

/**
 * embed_update_tab_title: updates the tab to contain the page title
 */
void
embed_update_tab_title (GaleonEmbed *embed)
{
	GaleonWindow *window;
	gchar *shortened;
	GtkWidget *tab;
	gint length;
	gint style;

#ifdef ENABLE_NAUTILUS_VIEW
	/* no tabs whem embeded in nautilus */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	/* get the parent window */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

        /* shorten notebook's label */
	tab = embed->notebook_hbox;
	style = eel_gconf_get_integer (CONF_TABS_TABBED_SHORTEN_STYLE);
        switch (style)
        {
		case 0:
			/* set fixed size of the label widget */
			shortened = embed->title;
			break;

		case 1: 
			/* abbreviate the text label */
			length = eel_gconf_get_integer (
				CONF_TABS_TABBED_SHORTEN_CHARS);   
			shortened = misc_string_shorten_name (embed->title,
							      length);
			break;

		default:
			g_assert_not_reached ();
			return;
        }

	/* if it's different than the new shortened text, change it */
	if (strcmp (GTK_LABEL (embed->notebook_label)->label, shortened) != 0)
	{
		gtk_label_set_text (GTK_LABEL (embed->notebook_label),
				    shortened);
		embed_update_tab_size (embed);
	}

	/* free allocated strings */
	if (style == 1)
	{
		g_free (shortened);
	}
}

/**
 * embed_update_tab_size: updates tab sizing policies and size
 */
void
embed_update_tab_size (GaleonEmbed *embed)
{
	GaleonWindow *window;
	GtkWidget *tab;
	gint length;
	gint style;

#ifdef ENABLE_NAUTILUS_VIEW
	/* no tabs whem embeded in nautilus */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	/* get the parent window */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

        /* shorten notebook's label */
	tab = embed->notebook_label;
	style = eel_gconf_get_integer (CONF_TABS_TABBED_SHORTEN_STYLE);
        switch (style)
        {
        case 0:
		length = eel_gconf_get_integer 
			(CONF_TABS_TABBED_SHORTEN_POINTS);
                gtk_widget_set_usize (GTK_WIDGET (tab), length, -2);
                break;

        case 1: 
                gtk_widget_set_usize (GTK_WIDGET (tab), -1, -1);
                break;

        default:
                g_assert_not_reached ();
                return;
        }

	/* make sure everything is resized */
	gtk_widget_queue_resize (window->notebook);
}

/**
 * embed_update_tab_status: sets the status for the notebook tab
 * label belonging to embed.  this means changing the color of the tab
 * title and changing the closebutton between the active/inactive states if
 * necessary
 */
void
embed_update_tab_status (GaleonEmbed *embed)
{
	GtkWidget *label;

	return_if_not_embed (embed);

#ifdef ENABLE_NAUTILUS_VIEW
	/* no tabs whem embeded in nautilus */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	/* get the label */
	label = GTK_WIDGET (embed->notebook_label);
	g_return_if_fail (GTK_IS_LABEL (label));

	/* set the right colour */
	if (embed->load_started > 0)
	{
		/* loading... */
		gtk_widget_set_style (label, loading_text_style);
	}
	else if (!(embed->has_been_viewed))
	{
		/* loaded, with new content */
		gtk_widget_set_style (label, new_text_style);
	}
	else if (embed->is_active)
	{
		/* loaded, viewed */
		gtk_widget_set_rc_style (label);
	}

	/* update closebutton status */
	if (embed->notebook_close_button != NULL &&
	    embed->notebook_close_pixmap != NULL)
	{
		gboolean make_close_insensitive;

		make_close_insensitive = eel_gconf_get_boolean (
			CONF_TABS_TABBED_CLOSEBUTTON_INSENSITIVE);

		if (embed->is_active || !make_close_insensitive)
		{
			gtk_widget_hide (embed->notebook_close_pixmap);
			gtk_widget_show (embed->notebook_close_button);
		}
		else if (make_close_insensitive)
		{
			gtk_widget_hide (embed->notebook_close_button);
			gtk_widget_show (embed->notebook_close_pixmap);
		}
	}

	if (embed->notebook_close_button)
	{
		gtk_object_set_data (GTK_OBJECT (embed->notebook_close_button),
			    	     "tab_active",
				     GINT_TO_POINTER (embed->is_active));
	}
}

/**
 * embed_notebook_tab_update_close_button: the notebook tab belonging to
 * embed, is updated (if necessary) to reflect the user's preference in the
 * display of a close button.
 */
void
embed_update_tab_close_button (GaleonEmbed *embed)
{
	GaleonWindow *window;
	GtkWidget *button, *pixmap;
	gboolean show;

	/* check args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

#ifdef ENABLE_NAUTILUS_VIEW
	/* no tabs whem embeded in nautilus */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	/* check prefs */
	show = eel_gconf_get_boolean (CONF_TABS_TABBED_CLOSEBUTTON);

	/* check for case when this is being destroyed */
	if (!show && (embed->notebook_close_button != NULL ||
		      embed->notebook_close_pixmap != NULL))
	{
		gtk_widget_destroy (embed->notebook_close_button);
		embed->notebook_close_button = NULL;
		gtk_widget_destroy (embed->notebook_close_pixmap);
		embed->notebook_close_pixmap = NULL;
	}
	/* check for case where button is being created */
	else if (show)
	{
		GtkRequisition r;
		if (embed->notebook_close_button != NULL)
		{
			gtk_widget_destroy (embed->notebook_close_button);
		}
		
		if (embed->notebook_close_pixmap != NULL)
		{
			gtk_widget_destroy (embed->notebook_close_pixmap);
		}
		
		close_pix = themes_get_pixmap ("CloseTab.png", FALSE);

		/* build button */
		button = embed->notebook_close_button = tab_button_new ();
		tab_button_set_relief (TAB_BUTTON (button), GTK_RELIEF_NONE);
		pixmap = gtk_pixmap_new (close_pix->pixmap, close_pix->mask);
		gtk_widget_show (pixmap);
		gtk_container_add (GTK_CONTAINER (button), pixmap);

		/* pack button into tab */
		gtk_box_pack_start (GTK_BOX (embed->notebook_hbox), button,
				    FALSE, FALSE, 0);

		/* connect click handler */
		gtk_signal_connect (GTK_OBJECT (button), "clicked",
				    GTK_SIGNAL_FUNC (
				    	embed_tab_close_button_clicked_cb),
				    embed);

		/* get button size */
		gtk_widget_size_request (button, &r);
		
		/* build inactive close pixmap */
		pixmap = embed->notebook_close_pixmap = 
			gtk_pixmap_new (close_pix->pixmap, close_pix->mask);

		/* give it the same size as the button, to prevent tab
		 * resizing */
		gtk_widget_set_usize (pixmap, r.width, r.height);

		/* it's only used in inactive situations, so make it
		 * insensitive */
		gtk_widget_set_sensitive (pixmap, FALSE);

		/* pack button into tab */
		gtk_box_pack_start (GTK_BOX (embed->notebook_hbox), pixmap,
				    FALSE, FALSE, 0);
	}
}

/**
 * embed_go_prev_link: go to the "previous" link, if there is one
 */
void
embed_go_prev_link (GaleonEmbed *embed)
{
	GList *l, *items;
	gboolean found = FALSE;

	items = mozilla_get_link_interface_items (embed);

	for (l = items; l; l = g_list_next (l))
	{
		LinkInterfaceItem *lti = (LinkInterfaceItem *) l->data;

		if (!found)
		{
			switch (lti->type)
			{
				case LINK_ITEM_PREV:
				case LINK_ITEM_PREVIOUS:
					embed_load_url (embed, lti->href);
					found = TRUE;
					break;
				default:
					break;
			}

			if (lti->title) g_free (lti->title);
			if (lti->href) g_free (lti->href);
			g_free (lti);
		}
	}

	g_list_free(items);
}

/**
 * embed_go_prev_link: go to the "next" link, if there is one
 */
void
embed_go_next_link (GaleonEmbed *embed)
{
	GList *l, *items;
	gboolean found = FALSE;

	items = mozilla_get_link_interface_items (embed);

	for (l = items; l; l = g_list_next (l))
	{
		LinkInterfaceItem *lti = (LinkInterfaceItem *) l->data;

		if (!found)
	        {
			switch (lti->type)
			{
			case LINK_ITEM_NEXT:
				embed_load_url (embed, lti->href);
				found = TRUE;
				break;
			default:
				break;
			}
		}

		if (lti->title) g_free (lti->title);
		if (lti->href) g_free (lti->href);
		g_free (lti);
	}

	g_list_free (items);
}

/**
 * embed_go_contents_link: Attempt to go to the table of contents of
 * the current page, following any number of link types, and in the
 * worst case simply going "up" as far as possible.
 */
void
embed_go_contents_link (GaleonEmbed *embed)
{
	GList *l, *items;
	gboolean found = FALSE;
	gchar *uri, *up_uri;

	items = mozilla_get_link_interface_items (embed);

	for (l = items; l; l = g_list_next (l))
	{
		LinkInterfaceItem *lti = (LinkInterfaceItem *) l->data;

		if (!found)
		{
			switch (lti->type)
			{
			case LINK_ITEM_START:
			case LINK_ITEM_CONTENTS:
			case LINK_ITEM_TOC:
			case LINK_ITEM_TOP:
			case LINK_ITEM_HOME:
				embed_load_url (embed, lti->href);
				found = TRUE;
				break;
			default:
				break;
			}
		}
		
		if (lti->title) g_free (lti->title);
		if (lti->href) g_free (lti->href);
		g_free (lti);
	}

	g_list_free (items);

	if (found) return;

	uri = g_strdup (embed->location);
	if (uri == NULL)
	{
		return;
	}

	/* go upwards to the top */
	while ((up_uri = mozilla_uri_get_parent (uri)) != NULL)
	{
		g_free (uri);
		uri = up_uri;
	}

	/* visit it */
	embed_load_url (embed, uri);

	/* free */
	g_free (uri);
}

/**
 * embed_go_up: go to the nth parent directory
 */
void
embed_go_up (GaleonEmbed *embed, gint levels, LinkState state)
{
	gchar *uri, *up_uri;
	uri = g_strdup(embed->location);

	/* go upwards to find the nth level up */
	do
	{
		up_uri = mozilla_uri_get_parent (uri);
		g_free (uri);
		uri = up_uri;

		/* this can happen if Up is selected from the menu */
		if (uri == NULL)
		{
			return;
		}
	}
	while (levels--);

	/* visit it */
	embed_activate_link (embed, NULL, uri, state);

	/* free */
	g_free (uri);
}

/**
 * embed_can_go_up: test to see if we can go to a parent directory
 */
gboolean
embed_can_go_up (GaleonEmbed *embed)
{
	gboolean result;

	/* check embed location is valid */
	if (embed->location == NULL || strlen (embed->location) == 0)
	{
		return FALSE;
	}

	result = mozilla_uri_has_parent (embed->location);

	return result;
}

/**
 * embed_set_zoom: set the zoom level for a given embed
 */
void
embed_set_zoom (GaleonEmbed *embed, gint zoom, gboolean reflow)
{
	/* sanity check */
	if (zoom == 0)
	{
		g_warning ("ignoring request to set zoom to 0");
		return;
	}

	/* check wrapper exists */
	if (embed->wrapper == NULL)
	{
		return;
	}

	/* set in mozilla */
	mozilla_set_zoom (embed, (float)zoom / 100.0, reflow);
	embed->zoom_auto_set = FALSE;
	embed->zoom = zoom;

	/* set in nautilus view, if applicable.  note that we don't set the
	 * zoom value in the parent window's spinner here, in order to
	 * avoid an additional, unnecessary call to embed_set_zoom() */
#ifdef ENABLE_NAUTILUS_VIEW
	if (EMBED_IN_NAUTILUS (embed))
	{
		galeon_nautilus_view_report_zoom (embed->nautilus_view,
						  embed->zoom);
	}
#endif
}

/** 
 * Creates the back history menu 
 */
GtkMenu *
embed_create_back_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history_get_all_titles (embed, &titles,
						     &count, &index))
	{
		return NULL;
	}

	for (i = index - 1, level = 0; i >= 0; i--, level++) 
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}

	misc_string_free_array (titles, count);
	return GTK_MENU(menu);
}

/**
 * Creates the forward history menu
 */
GtkMenu *
embed_create_forward_menu (GaleonEmbed *embed)
{
	int index, count, i, level;
	char **titles;
	GtkWidget *menu = gtk_menu_new ();

	if (!mozilla_session_history_get_all_titles (embed, &titles,
						     &count, &index))
	{
		return NULL;
	}	

	for (i = index + 1, level = 0; i < count; i++, level++)
	{
		add_bf_menu (menu, titles[i], i, embed, level);
	}
	
	misc_string_free_array (titles, count);
	return GTK_MENU(menu);
}

/** 
 * Creates the multi-level up menu 
 */
GtkMenu *
embed_create_up_menu (GaleonEmbed *embed)
{
	char *uri, *up_uri;
	GtkWidget *menu, *item;
	gint level;

	/* check embed location is valid */
	if (embed->location == NULL || strlen (embed->location) == 0)
	{
		return NULL;
	}

	uri = g_strdup (embed->location);

	/* create the menu */
	menu = gtk_menu_new ();

	/* create each possible up entry */
	for (level = 0;; level++)
	{
		const PixmapData *ico;
		
		up_uri = mozilla_uri_get_parent (uri);
		g_free (uri);
		uri = up_uri;

		/* get out of loop if no parent */
		if (uri == NULL)
		{
			break;
		}

		/* create the menu entry */
		ico = favicon_get_pixmap (uri);
		item = new_num_accel_menu_item (level, uri, TRUE, menu,
						ico);
		gtk_widget_show (GTK_WIDGET (item));
		gtk_object_set_user_data (GTK_OBJECT (item), embed);
		gtk_object_set_data_full (GTK_OBJECT (item), "url", 
					  g_strdup (uri), g_free);
		gtk_object_set_data (GTK_OBJECT (item), "title", uri);
		gtk_menu_append (GTK_MENU (menu), GTK_WIDGET (item));
		gtk_signal_connect (GTK_OBJECT (item), "activate",
				    up_menu_menuitem_activate_cb, 
				    GINT_TO_POINTER (level));
		gtk_signal_connect (GTK_OBJECT(item), "button-press-event",
			            GTK_SIGNAL_FUNC
			               (generic_link_button_press_event_cb),
			            embed->parent_window);
		gtk_signal_connect (GTK_OBJECT (item), "button_release_event",
				(GtkSignalFunc) 
					(up_menu_menuitem_button_release_cb),
				GINT_TO_POINTER (level));

		gtk_widget_lock_accelerators (item);
	}

	/* the menu is completed */
	return GTK_MENU (menu);
}

/**
 * embed_shift_tab: shifts embed by amount in the notebook
 */
void
embed_shift_tab (GaleonEmbed *embed, gint amount)
{
	GaleonWindow *window;
	GtkNotebook *notebook;
	gint page;

	return_if_not_sane_embed (embed);

#ifdef ENABLE_NAUTILUS_VIEW
	/* This operation makes no sense when embeded */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	window = embed->parent_window;
	notebook = GTK_NOTEBOOK (window->notebook);

	/* reorder the tab */
	page = gtk_notebook_page_num (notebook, GTK_WIDGET (embed->mozembed));
	gtk_notebook_reorder_child (notebook, GTK_WIDGET (embed->mozembed),
				    page + amount);

	/* and update the menuitems */
	window_update_tab_controls (window);
}

/* embed_move_to_new_window: moves embed to a new window */
void
embed_move_to_new_window (GaleonEmbed *embed)
{
	GaleonWindow *window;

	return_if_not_embed (embed);

	window = window_create (DEFAULT_CHROME, embed->parent_window);
	return_if_not_window (window);

	embed_move_to_existing_window (embed, window, 0);
}

/* embed_move_to_existing_window: moves embed to window, placing it on the
 * given page in the notebook */
void
embed_move_to_existing_window (GaleonEmbed *embed, GaleonWindow *window,
			       gint page)
{
	GaleonWindow *orig_window;
	GtkNotebook *notebook, *orig_notebook;
	gint orig_page;

	return_if_not_sane_embed (embed);
	return_if_not_window (window);

	/* get the original window */
	orig_window = embed->parent_window;

	/* find where we're putting it */
	notebook = GTK_NOTEBOOK (window->notebook);

	if (orig_window == NULL)
	{
		window_add_embed (window, embed,
				  EMBED_CREATE_FORCE_JUMP |
				  EMBED_CREATE_RAISE_WINDOW);
	}
	/* simple case -- moving within the same window; just reorder
	 * the page */
	else if (window == orig_window)
	{
		gtk_notebook_reorder_child (notebook, embed->mozembed, page);
		window_update_tab_controls (window);
	}
	/* otherwise, move the embed to the other notebook */
	else
	{
		/* get original notebook */
		orig_notebook = GTK_NOTEBOOK (orig_window->notebook);
		orig_page = gtk_notebook_page_num (orig_notebook,
						   embed->mozembed);

		/* add a reference to the tab so it won't be destroyed when
		 * we remove the page (the embed has already been
		 * referenced when it was created, so it doesn't need
		 * another reference) */
		gtk_object_ref (GTK_OBJECT (embed->notebook_hbox));

		/* select the new page before moving the current one */
		embed_select_new_page (embed);

		/* remove the embed from its current page */
		orig_window->embed_list =
			g_list_remove (orig_window->embed_list, embed);
		if (embed->is_visible)
		{
			window_hide_one_embed (orig_window);
		}
		gtk_notebook_remove_page (orig_notebook, orig_page);
		window_update_tab_controls (orig_window);

		/* if that was the only embed in the window,
		 * delete the window */
		if (orig_window->embed_list == NULL)
		{
			window_close (orig_window);
		}

		/* add the embed to the new window's list */
		window->embed_list = g_list_append (window->embed_list, embed);
		if (embed->is_visible)
		{
			window_show_one_embed (window);
		}
		embed->parent_window = window;

		/* add it into the destination notebook */
		gtk_notebook_insert_page (notebook, embed->mozembed, 
					  embed->notebook_hbox, page);

		/* switch to the dropped embed's page */
		gtk_notebook_set_page (notebook, page);

		/* update tab state */
		embed_update_tab_status (embed);

		/* and remove the tab's reference */
		gtk_object_unref (GTK_OBJECT (embed->notebook_hbox));
	}
}

/**
 * Creates a menu item with a numbered/lettered accel
 */
static GtkWidget *
new_num_accel_menu_item (gint num, gchar *origtext, gboolean lettersok, 
			 GtkWidget *menu, const PixmapData *ico)
{
	gchar *text = misc_string_new_num_accel (num, origtext, lettersok);
	GtkWidget *item = gtk_pixmap_menu_item_new ();
	GtkWidget *hb = gtk_hbox_new (FALSE, 0);
	GtkWidget *label = gtk_label_new ("");

	gtk_box_pack_start (GTK_BOX (hb), label, FALSE, FALSE, 0);
	gtk_container_add (GTK_CONTAINER (item), hb);

	gtk_pixmap_menu_item_set_pixmap (GTK_PIXMAP_MENU_ITEM (item),
					 gtk_pixmap_new (ico->pixmap,
							 ico->mask));

	gtk_widget_lock_accelerators (item);
	
	if (text == NULL)
	{
		gtk_label_set_text (GTK_LABEL (label), origtext);
	}
	else
	{
		misc_gui_label_set_accel_text (text, label, menu, item);
		g_free (text);
	}

	gtk_widget_show_all (item);

	return item;
}

/**
 * Adds a menuitem to a back/forward history menu
 */
static void
add_bf_menu (GtkWidget *menu, char* label, int index, GaleonEmbed *embed,
	     int level)
{
	gchar *url = mozilla_session_history_get_url (embed, index); 
	const PixmapData *ico = favicon_get_pixmap (url);
	gchar *tmp = misc_string_escape_uline_accel (label);
	GtkWidget *item = new_num_accel_menu_item (level, tmp, TRUE, menu, ico);
	g_free (tmp);

	gtk_widget_show (item);
	gtk_object_set_user_data (GTK_OBJECT (item), embed);
	gtk_menu_append (GTK_MENU (menu), item);
	gtk_object_set_data_full (GTK_OBJECT (item), "url", url, g_free);
	gtk_object_set_data_full (GTK_OBJECT (item), "title", g_strdup (label),
				  g_free);
	gtk_signal_connect (GTK_OBJECT (item), "activate",
			    history_menu_menuitem_activate_cb, 
			    GINT_TO_POINTER (index));
	gtk_signal_connect (GTK_OBJECT(item), "button-press-event",
			    GTK_SIGNAL_FUNC
			    (generic_link_button_press_event_cb),
			    embed->parent_window);
	gtk_signal_connect (GTK_OBJECT (item), "button_release_event",
			(GtkSignalFunc)	
				(history_menu_menuitem_button_release_cb),
			GINT_TO_POINTER (index));
	gtk_widget_lock_accelerators (item);
}


static void
embed_select_new_page (GaleonEmbed *embed)
{
	GaleonWindow *window;
	GaleonEmbed *new_embed;
	gboolean last_embed = FALSE;

	/* check args */
	return_if_not_sane_embed (embed);
	window = embed->parent_window;

#ifdef ENABLE_NAUTILUS_VIEW
	/* desn't make sense */
	if (EMBED_IN_NAUTILUS (embed))
		return;
#endif

	/* check if this is the last embed in the window */
	last_embed = (g_list_length (window->embed_list) == 1);

	if (!last_embed && embed->is_active)
	{
		/* this logic implements the galeon Smart Tab Selection
		 * Technology (tm) as suggested by Matthew Aubury -- Nate */
		switch (embed->focus_type)
		{
		case FOCUS_ON_CREATE:
			/* 
			 * embed_list is in order of tabs most recently
			 * focused.  select the second element in list since
			 * the first still hasn't been destroyed
			 */
			new_embed = g_list_nth_data (window->embed_list, 1);
			embed_switch_to_page (new_embed);
			break;

		case FOCUS_ON_REQUEST:
			gtk_notebook_next_page 
				(GTK_NOTEBOOK (window->notebook));
			break;

		default:
			g_assert_not_reached ();
			break;
		}
	} 
}

void
embed_toggle_image_blocking (const gchar *url)
{
	GnomeVFSURI *vfs_uri = gnome_vfs_uri_new (url);
	const gchar *host;
	GList *permissions, *l;
	gboolean done = FALSE;

	if (vfs_uri == NULL)
	{
		return;
	}
	
	host = gnome_vfs_uri_get_host_name (vfs_uri);
	if (host == NULL)
	{
		gnome_vfs_uri_unref (vfs_uri);
		return;
	}

	permissions = mozilla_get_permissions (IMAGEPERMISSION);

	for (l = permissions; l && done == FALSE; l = g_list_next (l))
	{
		BlockedHost *b = (BlockedHost *) l->data;
		
		if (strcmp (host, b->domain) == 0)
		{
			GList *remove = NULL;
			remove = g_list_append (remove, b);
			mozilla_remove_permissions (remove, IMAGEPERMISSION);
			g_list_free (remove);
			done = TRUE;
		}

		g_free (b->type);
		g_free (b->domain);
		g_free (b);
	}
	
	if (done == FALSE)
	{
		mozilla_set_url_permission (url, IMAGEPERMISSION, FALSE);
	}

	gnome_vfs_uri_unref (vfs_uri);
	g_list_free (permissions);
}

/**
 * embed_save_modified_location: used when switching to a new tab. save the
 * old (possibly edited) site location in the embed and hack up the
 * selection to preserve it (if any).
 */
void
embed_save_modified_location (GaleonEmbed *embed)
{
	GaleonWindow *window;
	guchar *text, *stored;
	gchar *entry_text;
	int start, end, tmp;

	/* get window */
	window = embed->parent_window;
	return_if_not_window (window);
	if (!embed->is_active) return;

	/* if we don't have a location entry, we can't do anything */
	if (!window->location_entry)
		return;

	/* free old value */
	if (embed->modified_location)
		g_free (embed->modified_location);

	entry_text = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 0, -1);
	if (entry_text && embed->location && 
	    strcmp (entry_text, embed->location))
	{
		/* get location text */
		embed->modified_location = entry_text;
		embed->location_is_modified = TRUE;
	}
	else
	{
		/* otherwise, location isn't modified */
		embed->modified_location = NULL;
		embed->location_is_modified = FALSE;
		if (entry_text) g_free (entry_text);
	}
	
			
	/* return if no selection was made */
	if (!(GTK_EDITABLE (window->location_entry)->has_selection))
		return;

	/* get selected text */
	start = GTK_EDITABLE (window->location_entry)->selection_start_pos;
	end = GTK_EDITABLE (window->location_entry)->selection_end_pos;

	/* switch them when they are in the opposite order,
	   it happens if you select from right to left */
	if (start > end)
	{
		tmp = start;
		start = end;
		end = tmp;
	}

	text = gtk_editable_get_chars 
		(GTK_EDITABLE (window->location_entry), 
		 start, end);

	/* free up stored per-window selection */
	stored = gtk_object_get_data (GTK_OBJECT (embed->mozembed),
				      "selection");
	gtk_object_remove_data (GTK_OBJECT (embed->mozembed), "selection");
	g_free (stored);

        /* store in parent window */
	gtk_object_set_data (GTK_OBJECT (embed->mozembed), "selection", text);
	
	/* window takes ownership of primary selection */
	gtk_selection_owner_set (embed->mozembed, GDK_SELECTION_PRIMARY,
				 GDK_CURRENT_TIME);
}

/**
 * embed_exists: check whether an embed still exists
 */
gboolean
embed_exists (GaleonEmbed *embed)
{
	return (g_list_find (all_embeds, embed) != NULL);
}

/**
 * embed_copy_text_to_clipboard: Copies some text into the clipboard
 */
void
embed_copy_text_to_clipboard (gchar *text, GaleonEmbed *embed)
{
	gint have_selection;
	GtkWidget *window = embed->mozembed;

	/* FIXME: free previous data? */
	gtk_object_set_data (GTK_OBJECT (window),
			     "selection", g_strdup (text));
	have_selection = gtk_selection_owner_set (GTK_WIDGET (window),
					 gdk_atom_intern("CLIPBOARD",FALSE), 
					  GDK_CURRENT_TIME)&&
		         gtk_selection_owner_set (window,
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
	if (!have_selection)
	{
		g_warning("Selection not found");
	}
} 

