// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "collectionview.h"
#include "collection.h"
#include "icons.h"
#include "mediaview.h"
#include "mq3.h"
#include "organizer.h"

#include <qaction.h>
#include <qapplication.h>
#include <qcursor.h>
#include <qdir.h>
#include <qfileinfo.h>
#include <qinputdialog.h>
#include <qmessagebox.h>
#include <qpopupmenu.h>
#include <qstatusbar.h>

static CollectionView *INSTANCE = 0;


CollectionView::CollectionView(QWidget *parent, const char *name)
    : QListView(parent, name), current(0), modified(false)
{
    setShowSortIndicator(false);
    setMouseTracking(false);
    viewport()->setMouseTracking(false);
    setRootIsDecorated(false);
    setResizeMode(AllColumns);
    addColumn(tr("Collections"));

    connect(this, SIGNAL(doubleClicked(QListViewItem *)),
	    SLOT(itemSelected(QListViewItem *)));
    connect(this, SIGNAL(returnPressed(QListViewItem *)),
	    SLOT(itemSelected(QListViewItem *)));

    INSTANCE = this;
}

void CollectionView::setupActions()
{
    actionNewColl = new QAction(tr("New Collection"),
				tr("&New"),
				CTRL + Key_N,
				this);
    actionNewColl->setStatusTip(tr("Create a new Collection."));
    connect(actionNewColl, SIGNAL(activated()), SLOT(newCollection()));

    actionRenameColl = new QAction(tr("Rename Collection"),
				   tr("&Rename"),
				   0,
				   this);
    actionRenameColl->setStatusTip(tr("Rename the selected Collection."));
    connect(actionRenameColl, SIGNAL(activated()), SLOT(renameCollection()));

    actionRemColl = new QAction(tr("Remove Collection"),
				tr("Re&move"),
				0,
				this);
    actionRemColl->setStatusTip(tr("Remove the selected Collection."));
    connect(actionRemColl, SIGNAL(activated()), SLOT(removeCollection()));

    actionImportColl = new QAction(tr("Import Collection"),
				   tr("&Import"),
				   0,
				   this);
    actionImportColl->setStatusTip(tr("Import a Collection from another program."));
    connect(actionImportColl, SIGNAL(activated()), SLOT(importCollection()));
}

void CollectionView::setupMenus()
{
    mainmenu = new QPopupMenu(this);
    collmenu = new QPopupMenu(this);

    // collections menu (menubar)
    actionNewColl->addTo(mainmenu);
    actionRenameColl->addTo(mainmenu);
    actionRemColl->addTo(mainmenu);
    mainmenu->insertSeparator();
    actionImportColl->addTo(mainmenu);

    // collection context menu
    actionRenameColl->addTo(collmenu);
    actionRemColl->addTo(collmenu);
    collmenu->insertSeparator();
    actionNewColl->addTo(collmenu);
    actionImportColl->addTo(collmenu);
}

void CollectionView::setupToolbars()
{
    QToolBar *toolbar = new QToolBar(tr("Collections"), Organizer::instance());

    // collections
    actionNewColl->addTo(toolbar);
    actionRenameColl->addTo(toolbar);
    actionRemColl->addTo(toolbar);
    toolbar->addSeparator();
    actionImportColl->addTo(toolbar);
}

void CollectionView::loadIcons()
{
    setColumnText(0, IconLoader::load("collections"), tr("Collections"));
    actionNewColl->setIconSet(IconLoader::load("newcollection"));
    actionRenameColl->setIconSet(IconLoader::load("renamecollection"));
    actionRemColl->setIconSet(IconLoader::load("removecollection"));
    actionImportColl->setIconSet(IconLoader::load("importcollection"));
    iconset = IconLoader::load("collection");
}

void CollectionView::load()
{
    QString filename = QDir::homeDirPath() + "/.mq3/collections";
    QFile file(filename);
    if (! file.open(IO_ReadOnly))
	return;

    setUpdatesEnabled(false);

    QDataStream stream(&file);

    QString header("MQ3 Collections Description");
    QString s, r, g;
    uint c;
    int p, o;

    // make sure we can read the header from the stream - this tells us that we
    // are really reading the right file format...
    stream >> s;
    if (s == header) {
	// read the current collection name
	stream >> r;

	while (! stream.atEnd()) {
	    // read collection name and entry count
	    stream >> s >> c;

	    Collection *collection = new Collection(this, s);

	    uint i;
	    for (i = 0; i < c; i ++) {
		// read signature, priority and state
		stream >> g >> p >> o;

		Collection::SongState *state = new Collection::SongState(g, p, o);
		collection->songlist.append(state);
	    }
	}
    }

    // reinstate the last current collection
    if (! r.isNull() && ! r.isEmpty()) {
	QListViewItem *c = firstChild();
	Collection *collection;
	while (c) {
	    if (c->rtti() == Collection::RTTI) {
		collection = (Collection *) c;
		if (collection->name() == r) {
		    setCurrentCollection(collection);
		    break;
		}
	    }

	    c = c->nextSibling();
	}
    }

    file.close();

    setUpdatesEnabled(true);
    setModified(false);
}

void CollectionView::save()
{
    if (! modified)
	return;

    Organizer::instance()->statusBar()->message(tr("Saving collection changes..."));
    QApplication::setOverrideCursor(QCursor(WaitCursor));
    qApp->processEvents(0);

    QFileInfo fileinfo(QDir::homeDirPath() + "/.mq3");
    if (! fileinfo.exists()) {
	QDir::home().mkdir(".mq3");
    }

    QFile file(QDir::homeDirPath() + "/.mq3/collections");
    if (! file.open(IO_WriteOnly)) {
	qWarning("Organizer: error opening output file");
	return;
    }

    QDataStream ds(&file);
    dataDescription(ds);
    file.close();

    setModified(false);

    QApplication::restoreOverrideCursor();
    Organizer::instance()->statusBar()->message(tr("Saved collection changes."), 2000);
}

void CollectionView::dataDescription(QDataStream &stream) const
{
    QString header("MQ3 Collections Description");
    stream << header;

    // write out the current collection
    QString cc;
    if (current)
	cc = current->name();
    stream << cc;

    QListViewItem *c = firstChild();
    while (c) {
	if (c->rtti() == Collection::RTTI)
	    ((Collection *) c)->dataDescription(stream);

	c = c->nextSibling();
    }
}

void CollectionView::setCurrentCollection(Collection *coll)
{
    MQ3::instance()->stop();

    if (current == coll)
	return;

    setUpdatesEnabled(false);

    if (current)
	current->setCurrent(false);
    current = coll;
    current->setCurrent(true);

    setUpdatesEnabled(true);
    setModified(true);
}

void CollectionView::newCollection()
{
    QString name = QInputDialog::getText(tr("New Collection"),
					 tr("Enter the name of the new Collection:"),
					 QLineEdit::Normal, QString::null,
					 0, this);

    if (name.isNull() || name.isEmpty())
	return;

    new Collection(this, name);
    setModified(true);
}

void CollectionView::renameCollection()
{
    QListViewItem *item = currentItem();
    if (! item || item->rtti() != Collection::RTTI)
	return;

    Collection *collection = (Collection *) item;
    QString name = QInputDialog::getText(tr("Rename Collection"),
					 tr("Enter the new name of the Collection:"),
					 QLineEdit::Normal, collection->name(),
					 0, this);

    if (name.isNull() || name.isEmpty())
	return;

    collection->setName(name);
    setModified(true);
}

void CollectionView::removeCollection()
{
    QListViewItem *item = currentItem();
    if (! item || item->rtti() != Collection::RTTI)
	return;

    if (QMessageBox::warning(this,
			     tr("Confirm Remove"),
			     tr("Are you sure you want to remove this collection?"),
			     tr("&Yes"),
			     tr("&No"),
			     QString::null, 0, 1))
	return;

    Collection *c = (Collection *) item;
    if (c == current)
	current = 0;
    delete c;

    setModified(true);
}

void CollectionView::importCollection()
{
    QMessageBox::information(this,
			     tr("MQ3 Media Player"),
			     tr("This feature is not implemented."));
}

void CollectionView::itemSelected(QListViewItem *item)
{
    if (! item || item->rtti() != Collection::RTTI)
	return;
    setCurrentCollection((Collection *) item);
}

CollectionView *CollectionView::instance()
{
    return INSTANCE;
}
