// Copyright (c) 2000-2001 Brad Hughes <bhughes@trolltech.com>
//
// Use, modification and distribution is allowed without limitation,
// warranty, or liability of any kind.
//

#include "group.h"
#include "decoder.h"
#include "mediaview.h"
#include "organizer.h"
#include "song.h"

#include <qdir.h>
#include <qdom.h>
#include <qpainter.h>
#include <qregexp.h>
#include <qstyle.h>


int Group::RTTI = 141831;

Group::Group(const QString &t)
    : QCheckListItem(MediaView::instance(), t, CheckBox), coltext(t),
      totalSongs(0), enabledSongs(0)
{
    setExpandable(true);
    setSelectable(true);
}


Group::Group(Group *parent, const QString &t)
    : QCheckListItem(parent, t, CheckBox), coltext(t),
      totalSongs(0), enabledSongs(0)
{
    setExpandable(true);
    setSelectable(true);
}

void Group::setOn(bool on)
{
    if (totalSongs == 0) {
	QCheckListItem::setOn(false);
	return;
    }

    QListViewItem *item = firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI) {
	    ((Group *) item)->setOn(on);
	} else if (item->rtti() == Song::RTTI) {
	    Song *song = (Song *) item;
	    if (song->isOn() != on)
		song->setOn(on);
	}

	item = item->nextSibling();
    }
    QCheckListItem::setOn(on);
}

void Group::propogate(const QString &path)
{
    QDir dir(path);

    if (! dir.isReadable()) {
	setExpandable(false);
	return;
    }

    listView()->setUpdatesEnabled(false);

    const QFileInfoList *files = dir.entryInfoList();

    if (files) {
	QFileInfoListIterator it(*files);
	QFileInfo *fi;

	while ((fi = it.current()) != 0) {
	    ++it;

	    if (fi->fileName() == "." || fi->fileName() == "..") {
		;
	    } else if (fi->isDir()) {
		Group *g = new Group(this, fi->fileName());
		g->propogate(fi->absFilePath());
	    } else if (Decoder::supports(fi->fileName())) {
		new Song(this, fi->fileName(), fi->absFilePath(), "Default");
	    }
	}
    }

    listView()->setUpdatesEnabled(true);
}


void Group::setText(int column, const QString &text)
{
    if (column != 0 || text.isEmpty() || text.isNull())
	return;
    coltext = text;
    MediaView::instance()->setModified(true);
}


QString Group::text(int column) const
{
    if (column == 0)
	return coltext;
    return QString();
}

void Group::paintCell(QPainter *p, const QColorGroup &cg,
		      int column, int width, int align)
{
    if ( !p )
	return;

    p->fillRect( 0, 0, width, height(), cg.brush(QColorGroup::Base));

    if ( column != 0 ) {
	// The rest is text, or for subclasses to change.
	QListViewItem::paintCell( p, cg, column, width, align );
	return;
    }

    int marg = MediaView::instance()->itemMargin();
    int r = marg;

    if ( type() != CheckBox ) {
	qWarning("Groups should be checkboxes");
	return;
    }

    int x = 0;
    int y = (height() - 16) / 2;
    if ( isEnabled() ) {
	if (enabledSongs == totalSongs || enabledSongs == 0)
	    p->setPen( QPen( cg.text(), 2 ) );
	else
	    p->setPen( QPen( cg.dark(), 2 ) );
    } else
	p->setPen( QPen( MediaView::instance()->
			 palette().color( QPalette::Disabled,
					  QColorGroup::Text ), 2 ) );
    if ( isSelected() && !MediaView::instance()->rootIsDecorated() ) {
	p->fillRect( 0, 0, x + marg + 16 + 4, height(),
		     cg.brush( QColorGroup::Highlight ) );
	if ( isEnabled() )
	    p->setPen( QPen( cg.highlightedText(), 2 ) );
    }

    p->drawRect( x+marg, y+2, 16-4, 16-4 );
    /////////////////////
    x++;
    y++;
    if ( isOn() ) {
	QPointArray a( 7*2 );
	int i, xx, yy;
	xx = x+1+marg;
	yy = y+5;
	for ( i=0; i<3; i++ ) {
	    a.setPoint( 2*i,   xx, yy );
	    a.setPoint( 2*i+1, xx, yy+2 );
	    xx++; yy++;
	}
	yy -= 2;
	for ( i=3; i<7; i++ ) {
	    a.setPoint( 2*i,   xx, yy );
	    a.setPoint( 2*i+1, xx, yy+2 );
	    xx++; yy--;
	}
	p->drawLineSegments( a );
    }
    ////////////////////////
    r += 16 + 4;

    p->translate( r, 0 );
    p->setPen( QPen( cg.text() ) );
    QListViewItem::paintCell( p, cg, column, width - r, align );
}

int Group::rtti() const
{
    return Group::RTTI;
}

void Group::xmlDescription(QTextStream &stream) const
{
    stream << "<group>";

    QString s = coltext;
    s = (s.replace(QRegExp("&"), "&amp;").
	 replace(QRegExp("<"), "&lt;").
	 replace(QRegExp(">"), "&gt;"));
    stream << "<name>" << s << "</name>";

    QListViewItem *item = firstChild();
    while (item) {
	if (item->rtti() == Group::RTTI)
	    ((Group *) item)->xmlDescription(stream);
	else if (item->rtti() == Song::RTTI)
	    ((Song *) item)->xmlDescription(stream);

	item = item->nextSibling();
    }

    stream << "</group>" << endl;
}

void Group::create(Group *parent, const QDomElement &element)
{
    Group *group = 0;

    if (element.tagName() == "group") {
	if (parent)
	    group = new Group(parent, "temp");
	else
	    group = new Group("temp");

	QDomElement e = element.firstChild().toElement();
	while (! e.isNull()) {
	    if (e.tagName() == "name")
		group->setText(0, e.firstChild().toText().data());
	    else if (e.tagName() == "group")
		Group::create(group, e);
	    else if (e.tagName() == "song")
		Song::create(group, e);

	    e = e.nextSibling().toElement();
	}
    }
}

void Group::addSong()
{
    Group *g = this;
    while (g) {
	g->totalSongs++;
	g = (Group *) g->parent();
    }
}

void Group::remSong()
{
    Group *g = this;
    while (g) {
	g->totalSongs--;
	g = (Group *) g->parent();
    }
}

void Group::enableSong(bool on)
{
    Group *g = this;
    while (g) {
	g->enabledSongs += on ? 1 : -1;
	g->QCheckListItem::setOn(g->enabledSongs > 0);
	g->repaint();
	g = (Group *) g->parent();
    }
}
