/* 
 * ratHold.c --
 *
 *	Manages different holds of messages.
 *
 *
 * TkRat software and its included text is Copyright 1996-2000 by
 * Martin Forssn
 *
 * The full text of the legal notice is contained in the file called
 * COPYRIGHT, included with this distribution.
 */

#include <pwd.h>
#include <signal.h>
#include "rat.h"
#include <locale.h>


/*
 * Parts of the message and body handlers which are saved in the hold.
 */
static char *holdMessageParts[] = {"remail", "return_path", "date", "from",
	"reply_to", "subject", "to", "cc", "bcc", "in_reply_to", "message_id",
	"save_to", "request_dsn", (char *) NULL};
static char *holdBodyParts[] = {"type", "subtype", "encoding", "parameter",
	"id", "description", "charset", "filename", "removeFile", "pgp_sign",
	"pgp_encrypt", "disp_type", "disp_parm", (char *) NULL};

/*
 * Keeps track of number of held and deferred messages
 */
static int numHeld, numDeferred;

/*
 * Save bodyparts during hold
 */
static int RatHoldBody(Tcl_Interp *interp, FILE *fPtr, char *baseName,
	char *handler, char **listValue, int *listValueSize, int intId);

/*
 *----------------------------------------------------------------------
 *
 * RatHold --
 *
 *	See ../doc/interface
 *
 * Results:
 *      A standard tcl result.
 *
 * Side effects:
 *      None.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatHold(ClientData dummy, Tcl_Interp *interp, int argc, char *argv[])
{
    static int fileListInitialized = 0;
    static Tcl_DString fileList;
    Tcl_DString tildeBuffer;
    char buf[1024], *holdDir;
    int length, ret = TCL_OK;
    char c;

    if (argc < 2) {
	Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		" option ?arg?\"", (char *) NULL);
	return TCL_ERROR;
    }
    holdDir = Tcl_GetVar2(interp, "option", "hold_dir", TCL_GLOBAL_ONLY);
    holdDir = Tcl_TranslateFileName(interp, holdDir, &tildeBuffer);
    if (0 != mkdir(holdDir, DIRMODE) && EEXIST != errno) {
	Tcl_AppendResult(interp, "error creating directory \"", holdDir,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	Tcl_DStringFree(&tildeBuffer);
	return TCL_ERROR;
    }

    c = argv[1][0];
    length = strlen(argv[1]);
    if ((c == 'i') && (strncmp(argv[1], "insert", length) == 0) && (length>1)) {
	if (argc != 4) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " insert handler description\"", (char *) NULL);
	    ret = TCL_ERROR;
	} else {
	    ret = RatHoldInsert(interp, holdDir, argv[2], argv[3]);
	}

    } else if ((c == 'l') && (strncmp(argv[1], "list", length) == 0)
	    && (length > 1)) {

	if (!fileListInitialized) {
	    Tcl_DStringInit(&fileList);
	    fileListInitialized = 1;
	} else {
	    Tcl_DStringFree(&fileList);
	}
	ret = RatHoldList(interp, holdDir, &fileList);

    } else if ((c == 'e') && (strncmp(argv[1], "extract", length) == 0)
	    && (length > 1)) {
	int index;
	char *eId;

	if (argc != 3) {
	    Tcl_AppendResult(interp, "wrong # args: should be \"", argv[0],
		    " extract index\"", (char *) NULL);
	    ret = TCL_ERROR;
	} else if (!fileListInitialized) {
	    Tcl_SetResult(interp, "You must list the content of the hold first",
		    TCL_STATIC);
	    ret = TCL_ERROR;
	} else if (TCL_OK != Tcl_GetInt(interp, argv[2], &index)) {
	    Tcl_SetResult(interp, "index must be integer", TCL_STATIC);
	    ret = TCL_ERROR;
	} else {
	    eId = RatLindex(interp, Tcl_DStringValue(&fileList), index);
	    snprintf(buf, sizeof(buf), "%s/%s", holdDir, eId);
	    ret = RatHoldExtract(interp, buf, NULL, NULL);
	}

    } else {
	Tcl_AppendResult(interp, "bad option \"", argv[1],
		"\": must be insert, list or extract", (char *) NULL);
	ret = TCL_ERROR;
    }

    Tcl_DStringFree(&tildeBuffer);
    return ret;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldBody --
 *
 *	Saves a bodypart into the file specified by fPtr. The handler of
 *	the bodypart is passed in the handler argument. We also get a
 *	listValue buffer and the length of it which we may use when
 *	converting list elements. The last argument is an internal counter
 *	which use is explained below. The baseName argument is a string
 *	which contains the first part of the name of any new files. The
 *	new value of intId is returned.
 *
 *	The goal of this function is to generate code which recreates
 *	the structure we are passed, but with another set of handlers.
 *	The handlers are on the form holdX, where X is a number. To get
 *	unique X'es we use the holdId variable which is incremented
 *	every time we need a new handler. When we enter this routine
 *	holdId has a good value. The problem is that this bodypart
 *	must include references to all its children, and we don't know
 *	which id's the children are going to get (they may create their
 *	own children). This is solved by storing the name of the reference
 *	variable in another variable holdRefX where the number X is taken
 *	from the intId argument. Then we can add code so that each child
 *	adds itself to the reference list.
 *
 * Results:
 *      None.
 *
 * Side effects:
 *      The size of the space allocated in listValue may be increased,
 *	this is mirrored in the listValueSize argument.
 *
 *
 *----------------------------------------------------------------------
 */

static int
RatHoldBody(Tcl_Interp *interp, FILE *fPtr, char *baseName, char *handler,
	    char **listValue, int *listValueSize, int intId)
{
    int i, newSize, flags;
    char *value;

    /*
     * Write standard part
     */
    fprintf(fPtr, "global hold${holdId}\n");
    for (i=0; holdBodyParts[i]; i++) {
	if ((value = Tcl_GetVar2(interp, handler, holdBodyParts[i],
		TCL_GLOBAL_ONLY))) {
	    if ((newSize = Tcl_ScanElement(value, &flags)) > *listValueSize) {
		*listValueSize = newSize+1;
		*listValue = (char*) ckrealloc(*listValue, *listValueSize);
	    }
	    Tcl_ConvertElement(value, *listValue, flags);
	    fprintf(fPtr, "set hold${holdId}(%s) %s\n", holdBodyParts[i],
		    *listValue);
	}
    }
    /*
     * Handle children (if any)
     */
    if ((value = Tcl_GetVar2(interp, handler, "children", TCL_GLOBAL_ONLY))) {
	char **argv;
	int argc, myId = intId;

	Tcl_SplitList(interp, value, &argc, &argv);
	fprintf(fPtr, "set holdRef%d hold${holdId}(children)\n", intId);
	fprintf(fPtr, "incr holdId\n");
	for (i=0; i<argc; i++) {
	    fprintf(fPtr, "lappend $holdRef%d hold${holdId}\n", myId);
	    intId = RatHoldBody(interp, fPtr, baseName, argv[i], listValue,
		    listValueSize, intId+1);
	    if (0 > intId) {
		ckfree(argv);
		return -1;
	    }
	}
	ckfree(argv);
	fprintf(fPtr, "unset holdRef%d\n", myId);
    } else {
	fprintf(fPtr, "incr holdId\n");
    }

    return intId;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldInsert --
 *
 *	Inserts a message into the specified hold directory. The directory
 *	will be created if it doesn't exist.
 *
 * Results:
 *      A standard Tcl result. the result area will contain the name of the
 *	newly inserted entry (if everything went ok).
 *
 * Side effects:
 *	The hold on disk may obviously be modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatHoldInsert(Tcl_Interp *interp, char *dir, char *handler, char *description)
{
    int listValueSize = 0, newSize, flags, i, result = 0;
    char baseName[1024], buf[1024];
    char *listValue = NULL;
    struct stat sbuf;
    char *value;
    FILE *fPtr;

    i = 0;
    do {
	snprintf(baseName, sizeof(baseName), "%s/%s_%x_%xM",
		dir, currentHost, (unsigned int)getpid(), i++);
    } while(!stat(baseName, &sbuf));

    /*
     * Write description file
     */
    snprintf(buf, sizeof(buf), "%s.desc", baseName);
    if (NULL == (fPtr = fopen(buf, "w"))) {
	Tcl_AppendResult(interp, "error creating file \"", buf,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    (void)fprintf(fPtr, "%s\n", description);
    (void)fclose(fPtr);

    /*
     * Write main file
     */
    if (NULL == (fPtr = fopen(baseName, "w"))) {
	Tcl_AppendResult(interp, "error creating file \"", baseName,
		"\": ", Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    fprintf(fPtr, "global hold${holdId}\n");
    for (i=0; holdMessageParts[i]; i++) {
	if ((value = Tcl_GetVar2(interp, handler, holdMessageParts[i],
		TCL_GLOBAL_ONLY))) {
	    if ((newSize = Tcl_ScanElement(value, &flags)) > listValueSize){
		listValueSize = newSize+1;
		listValue = (char*) ckrealloc(listValue, listValueSize);
	    }
	    Tcl_ConvertElement(value, listValue, flags);
	    fprintf(fPtr, "set hold${holdId}(%s) %s\n", holdMessageParts[i],
		    listValue);
	}
    }

    snprintf(buf, sizeof(buf), "%s tag ranges noWrap",
	    Tcl_GetVar2(interp, handler, "composeBody", TCL_GLOBAL_ONLY));
    Tcl_Eval(interp, buf);
    if ((newSize = Tcl_ScanElement(Tcl_GetStringResult(interp),
	    &flags)) > listValueSize){
	listValueSize = newSize+1;
	listValue = (char*) ckrealloc(listValue, listValueSize);
    }
    Tcl_ConvertElement(Tcl_GetStringResult(interp), listValue, flags);
    fprintf(fPtr, "set hold${holdId}(tag_range) %s\n", listValue);

    if ((value = Tcl_GetVar2(interp, handler, "body", TCL_GLOBAL_ONLY))) {
	fprintf(fPtr, "set hold${holdId}(body) hold[incr holdId]\n");
	result = RatHoldBody(interp, fPtr, baseName, value, &listValue, 
		&listValueSize, 0);
    }
    ckfree(listValue);
    if ( 0 > fprintf(fPtr, "\n") || 0 != fclose(fPtr) || 0 > result) {
	struct dirent *direntPtr;
	DIR *dirPtr;
	char *cPtr;

	(void)fclose(fPtr);
	for (cPtr = baseName+strlen(baseName)-1; *cPtr != '/'; cPtr--);
	*cPtr++ = '\0';
	dirPtr = opendir(dir);
	while (0 != (direntPtr = readdir(dirPtr))) {
	    if (!strncmp(direntPtr->d_name, cPtr, strlen(cPtr))) {
		snprintf(buf, sizeof(buf),"%s/%s", baseName, direntPtr->d_name);
		(void)unlink(buf);
	    }
	}
	closedir(dirPtr);

	Tcl_AppendResult(interp, "error writing files: ",
		Tcl_PosixError(interp), (char *) NULL);
	return TCL_ERROR;
    }
    Tcl_SetResult(interp, baseName, TCL_VOLATILE);
    RatHoldUpdateVars(interp, dir, 1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldList --
 *
 *	List the content of the specified hold directory.
 *
 * Results:
 *      A standard Tcl result.
 *
 * Side effects:
 *	The fileListPtr argument will be used to hold the return list
 *
 *
 *----------------------------------------------------------------------
 */

int
RatHoldList(Tcl_Interp *interp, char *dir, Tcl_DString *fileListPtr)
{
    struct dirent *direntPtr;
    char buf[1024];
    DIR *dirPtr;
    FILE *fPtr;
    Tcl_Obj *oPtr = Tcl_NewObj();
    int l;

    if (NULL == (dirPtr = opendir(dir))) {
	snprintf(buf, sizeof(buf), "Failed to open %s: %s",
		dir, Tcl_PosixError(interp));
	Tcl_SetResult(interp, buf, TCL_VOLATILE);
	return TCL_ERROR;
    }
    while (0 != (direntPtr = readdir(dirPtr))) {
	l = strlen(direntPtr->d_name);
	if (   'd' == direntPtr->d_name[l-4]
	    && 'e' == direntPtr->d_name[l-3]
	    && 's' == direntPtr->d_name[l-2]
	    && 'c' == direntPtr->d_name[l-1]) {
	    snprintf(buf, sizeof(buf), "%s/%s", dir, direntPtr->d_name);
	    fPtr = fopen(buf, "r");
	    fgets(buf, sizeof(buf), fPtr);
	    fclose(fPtr);
	    buf[strlen(buf)-1] = '\0';
	    Tcl_ListObjAppendElement(interp, oPtr, Tcl_NewStringObj(buf, -1));
	    snprintf(buf, sizeof(buf), direntPtr->d_name);
	    if (fileListPtr) {
		buf[strlen(buf)-5] = '\0';
		Tcl_DStringAppendElement(fileListPtr, buf);
	    }
	}
    }
    closedir(dirPtr);
    Tcl_SetObjResult(interp, oPtr);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldExtract --
 *
 *	Extract a specified held message.
 *
 * Results:
 *      A standard Tcl result. The handler of the new message will be left
 *	in the result area.
 *
 * Side effects:
 *	The content of holdId will be modified.
 *
 *
 *----------------------------------------------------------------------
 */

int
RatHoldExtract(Tcl_Interp *interp, char *prefix, Tcl_DString *usedVariablesPtr,
	Tcl_DString *filesPtr)
{
    static int holdId = 0;
    Tcl_Channel ch;
    char buf[1024], *cPtr;
    int i, oldId;
    Tcl_Obj *oPtr = Tcl_NewObj();

    /*
     * We start by reading the file. This is complicated by the fact that
     * the file is encoded in utf-8.
     */
    ch = Tcl_OpenFileChannel(interp, prefix, "r", 0);
    Tcl_SetChannelOption(interp, ch, "-encoding", "utf-8");
    i = Tcl_Seek(ch, 0, SEEK_END);
    Tcl_Seek(ch, 0, SEEK_SET);
    Tcl_ReadChars(ch, oPtr, i, 0);
    Tcl_Close(interp, ch);

    /*
     * Now we should eval the data, first we set the holdId variable
     * so that the file can generate unique handlers, after the read
     * we remember a new value of holdId. Then we get the right entry
     * in the list of files.
     */
    oldId = holdId;
    sprintf(buf, "%d", holdId);
    Tcl_SetVar(interp, "holdId", buf, 0);
    Tcl_EvalObjEx(interp, oPtr, TCL_EVAL_DIRECT);
    sprintf(buf, "hold%d", holdId);
    Tcl_SetResult(interp, buf, TCL_VOLATILE);
    Tcl_GetInt(interp, Tcl_GetVar(interp, "holdId", 0), &holdId);
    if (usedVariablesPtr) {
	for (i=oldId; i<holdId; i++) {
	    sprintf(buf, "hold%d", i);
	    Tcl_DStringAppendElement(usedVariablesPtr, buf);
	}
    }
    snprintf(buf, sizeof(buf), "%s.desc", prefix);
    if (filesPtr) {
	Tcl_DStringAppendElement(filesPtr, prefix);
	Tcl_DStringAppendElement(filesPtr, buf);
    } else {
	unlink(prefix);
	unlink(buf);
    }
    fflush(stderr);

    RatStrNCpy(buf, prefix, sizeof(buf));
    if ((cPtr = strrchr(buf, '/'))) *cPtr = '\0';
    RatHoldUpdateVars(interp, buf, -1);
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldInitVars --
 *
 *	Initialize some hold variables
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	Some global variables are initialized
 *
 *----------------------------------------------------------------------
 */

void
RatHoldInitVars(Tcl_Interp *interp)
{
    char *dir;
    Tcl_DString tildeBuffer;

    Tcl_DStringInit(&tildeBuffer);
    dir = Tcl_GetVar2(interp, "option", "send_cache", TCL_GLOBAL_ONLY);
    dir = Tcl_TranslateFileName(interp, dir, &tildeBuffer);
    RatHoldList(interp, dir, NULL);
    Tcl_ListObjLength(interp, Tcl_GetObjResult(interp), &numDeferred);

    dir = Tcl_GetVar2(interp, "option", "hold_dir", TCL_GLOBAL_ONLY);
    dir = Tcl_TranslateFileName(interp, dir, &tildeBuffer);
    if (TCL_OK == RatHoldList(interp, dir, NULL)) {
	Tcl_ListObjLength(interp, Tcl_GetObjResult(interp), &numHeld);
    } else {
	numHeld = 0;
    }
    Tcl_DStringFree(&tildeBuffer);

    Tcl_SetVar2Ex(interp, "numDeferred", NULL, Tcl_NewIntObj(numDeferred),
	    TCL_GLOBAL_ONLY);
    Tcl_SetVar2Ex(interp, "numHeld", NULL, Tcl_NewIntObj(numHeld),
	    TCL_GLOBAL_ONLY);
}

/*
 *----------------------------------------------------------------------
 *
 * RatHoldUpdateVars --
 *
 *	Update the hold variables
 *
 * Results:
 *      None.
 *
 * Side effects:
 *	Some global variables may be modified
 *
 *----------------------------------------------------------------------
 */

void
RatHoldUpdateVars(Tcl_Interp *interp, char *dir, int diff)
{
    char *senddir, *varname;
    Tcl_DString tildeBuffer;
    int *intvar;

    Tcl_DStringInit(&tildeBuffer);
    senddir = Tcl_GetVar2(interp, "option", "send_cache", TCL_GLOBAL_ONLY);
    senddir = Tcl_TranslateFileName(interp, senddir, &tildeBuffer);
    if (!strcmp(dir, senddir)) {
	varname = "numDeferred";
	intvar = &numDeferred;
    } else {
	varname = "numHeld";
	intvar = &numHeld;
    }

    *intvar += diff;
    Tcl_SetVar2Ex(interp, varname, NULL,Tcl_NewIntObj(*intvar),TCL_GLOBAL_ONLY);
}
