/*****************************************************************************/
/*  protocols.c                                                              */
/*  Copyright (C) 1999 Brian Masney <masneyb@seul.org>                       */
/*                                                                           */
/*  This library is free software; you can redistribute it and/or            */
/*  modify it under the terms of the GNU Library General Public              */
/*  License as published by the Free Software Foundation; either             */
/*  version 2 of the License, or (at your option) any later version.         */
/*                                                                           */
/*  This library is distributed in the hope that it will be useful,          */
/*  but WITHOUT ANY WARRANTY; without even the implied warranty of           */
/*  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU        */
/*  Library General Public License for more details.                         */
/*                                                                           */
/*  You should have received a copy of the GNU Library General Public        */
/*  License along with this library; if not, write to the Free Software      */
/*  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111 USA      */
/*****************************************************************************/

#include "protocols.h"

supported_gftp_protocols gftp_protocols[] = {
	{"FTP",		rfc959_init},
	{"HTTP",	rfc2068_init},
	{NULL,		NULL}};

gftp_request *gftp_request_new (void) {
   gftp_request *request;
   
   request = g_malloc0 (sizeof (gftp_request));
   request->sockfd = NULL;
   request->datafd = NULL;
   request->data_type = GFTP_TYPE_BINARY;
   request->transfer_type = gftp_transfer_passive;
   return (request);
}
/*****************************************************************************/
void gftp_request_destroy (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   gftp_disconnect (request);
   if (request->hostname) g_free (request->hostname);
   if (request->username) g_free (request->username);
   if (request->password) g_free (request->password);
   if (request->account) g_free (request->account);
   if (request->directory) g_free (request->directory);
   if (request->proxy_config) g_free (request->proxy_config);
   if (request->proxy_hostname) g_free (request->proxy_hostname);
   if (request->proxy_username) g_free (request->proxy_username);
   if (request->proxy_password) g_free (request->proxy_password);
   if (request->proxy_account) g_free (request->proxy_account);
   if (request->last_ftp_response) g_free (request->last_ftp_response);
   g_free (request);
}
/*****************************************************************************/
void gftp_file_destroy (gftp_file *file) {
   g_return_if_fail (file != NULL);
   
   if (file->file) g_free (file->file);
   if (file->user) g_free (file->user);
   if (file->group) g_free (file->group);
   if (file->attribs) g_free (file->attribs);
}
/*****************************************************************************/
void gftp_logging (gftp_request *request, int logging, gftp_logging_func logging_function, void *ptr) {
   g_return_if_fail (request != NULL);

   request->logging = logging;
   request->logging_function = logging_function;
   request->user_data = ptr;
}
/*****************************************************************************/
int gftp_connect (gftp_request *request) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->connect == NULL) return (-2);
   return (request->connect (request));
}
/*****************************************************************************/
void gftp_disconnect (gftp_request *request) {
   g_return_if_fail (request != NULL);
   
   if (request->disconnect == NULL) return;
   request->disconnect (request);
}
/*****************************************************************************/
long gftp_get_file (gftp_request *request, const char *filename, long startsize) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->get_file == NULL) return (-2);
   return (request->get_file (request, filename, startsize));
}
/*****************************************************************************/
int gftp_put_file (gftp_request *request, const char *filename, long startsize) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->put_file == NULL) return (-2);
   return (request->put_file (request, filename, startsize));
}
/*****************************************************************************/
int gftp_end_transfer (gftp_request *request) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->end_transfer == NULL) return (-2);
   return (request->end_transfer (request));
}
/*****************************************************************************/
int gftp_list_files (gftp_request *request) {
   g_return_val_if_fail (request != NULL, -2);

   if (request->list_files == NULL) return (-2);   
   return (request->list_files (request));
}
/*****************************************************************************/
int gftp_get_next_file (gftp_request *request, gftp_file *fle, FILE *fd) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->get_next_file == NULL) return (-2);
   return (request->get_next_file (request, fle, fd));
}
/*****************************************************************************/
int gftp_parse_url (gftp_request *request, const char *url) {
   char *pos, *endpos, *endhostpos, *str, tempchar;
   const char *stpos;
   int len;

   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (url != NULL, -2);
   
   for (stpos = url; *stpos == ' '; stpos++);
   str = g_malloc (strlen (stpos) + 1);
   strcpy (str, stpos);
   for (pos = str + strlen (str) - 1; *pos == ' '; pos--) *pos = '\0';
   
   pos = str;
   len = strlen (request->url_prefix);
   if (strncmp (pos, request->url_prefix, len) == 0 && strncmp (pos + len, "://", 3) == 0) {
      pos += len + 3;
   }
   
   if ((endhostpos = strchr (pos, '@')) != NULL) {
      /* A user/password was entered */
      if ((endpos = strchr (pos, ':')) == NULL || endhostpos < endpos) {
         /* No password was entered */
         gftp_set_password (request, "");
         endpos = endhostpos;
      }
      
      *endpos = '\0';
      gftp_set_username (request, pos);

      pos = endpos+1;
      if ((endpos = strchr (pos, '@')) != NULL) {
         if (strchr (endpos+1, '@') != NULL) endpos = strchr (endpos+1, '@');
         *endpos = '\0';
         gftp_set_password (request, pos);

         pos = endpos+1;
      }
   }
   else {
      gftp_set_username (request, "anonymous");
      if (request->password) g_free (request->password);
      request->password = NULL;
   }
   
   /* Now get the hostname and an optional port and optional directory */
   endhostpos = pos + strlen (pos);
   if ((endpos = strchr (pos, ':')) != NULL) {
      endhostpos = endpos;
   }
   else if((endpos = strchr(pos, '/')) != NULL) {
      endhostpos = endpos;
   }
   tempchar = *endhostpos;
   *endhostpos = '\0';
   gftp_set_hostname (request, pos);
   *endhostpos = tempchar;

   if ((endpos = strchr (pos, ':')) != NULL) {
      /* A port was entered */
      pos = endpos + 1;
      gftp_set_port (request, strtol (pos, NULL, 10));
   }
   if ((endpos = strchr (pos, '/')) != NULL) {
      gftp_set_directory (request, endpos);
   }
   g_free (str);
   return(0);
}
/*****************************************************************************/
int gftp_set_data_type (gftp_request *request, int data_type) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->set_data_type == NULL) return (-2);
   return (request->set_data_type (request, data_type));
}
/*****************************************************************************/
void gftp_set_hostname (gftp_request *request, const char *hostname) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (hostname != NULL);
   
   if (request->hostname) g_free (request->hostname);
   request->hostname = g_malloc (strlen (hostname) + 1);
   strcpy (request->hostname, hostname);
}
/*****************************************************************************/
void gftp_set_username (gftp_request *request, const char *username) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (username != NULL);
   
   if (request->username) g_free (request->username);
   request->username = g_malloc (strlen (username) + 1);
   strcpy (request->username, username);
}
/*****************************************************************************/
void gftp_set_password (gftp_request *request, const char *password) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (password != NULL);
   
   if (request->password) g_free (request->password);
   request->password = g_malloc (strlen (password) + 1);
   strcpy (request->password, password);
}
/*****************************************************************************/
void gftp_set_account (gftp_request *request, const char *account) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (account != NULL);
   
   if (request->account) g_free (request->account);
   request->account = g_malloc (strlen (account) + 1);
   strcpy (request->account, account);
}
/*****************************************************************************/
int gftp_set_directory (gftp_request *request, const char *directory) {
   g_return_val_if_fail (request != NULL, -2);
   g_return_val_if_fail (directory != NULL, -2);


   if (request->sockfd == NULL) {
      if (directory != request->directory) {
         if (request->directory) g_free (request->directory);
         request->directory = g_malloc (strlen (directory) + 1);
         strcpy (request->directory, directory);
      }
      return (0);
   }
   else if (request->chdir == NULL) {
      return (-2);
   }
   return (request->chdir (request, directory));
}
/*****************************************************************************/
void gftp_set_port (gftp_request *request, unsigned int port) {
   g_return_if_fail (request != NULL);
   
   request->port = port;
}
/*****************************************************************************/
void gftp_set_proxy_hostname (gftp_request *request, const char *hostname) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (hostname != NULL);
   
   if (request->proxy_hostname) g_free (request->proxy_hostname);
   request->proxy_hostname = g_malloc (strlen (hostname) + 1);
   strcpy (request->proxy_hostname, hostname);
}
/*****************************************************************************/
void gftp_set_proxy_username (gftp_request *request, const char *username) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (username != NULL);
   
   if (request->proxy_username) g_free (request->proxy_username);
   request->proxy_username = g_malloc (strlen (username) + 1);
   strcpy (request->proxy_username, username);
}
/*****************************************************************************/
void gftp_set_proxy_password (gftp_request *request, const char *password) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (password != NULL);
   
   if (request->proxy_password) g_free (request->proxy_password);
   request->proxy_password = g_malloc (strlen (password) + 1);
   strcpy (request->proxy_password, password);
}
/*****************************************************************************/
void gftp_set_proxy_account (gftp_request *request, const char *account) {
   g_return_if_fail (request != NULL);
   g_return_if_fail (account != NULL);
   
   if (request->proxy_account) g_free (request->proxy_account);
   request->proxy_account = g_malloc (strlen (account) + 1);
   strcpy (request->proxy_account, account);
}
/*****************************************************************************/
void gftp_set_proxy_port (gftp_request *request, unsigned int port) {
   g_return_if_fail (request != NULL);
   
   request->proxy_port = port;
}
/*****************************************************************************/
int gftp_remove_directory (gftp_request *request, const char *directory) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->rmdir == NULL) return (-2);
   return (request->rmdir (request, directory));
}
/*****************************************************************************/
int gftp_remove_file (gftp_request *request, const char *file) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->rmfile == NULL) return (-2);
   return (request->rmfile (request, file));
}
/*****************************************************************************/
int gftp_make_directory (gftp_request *request, const char *directory) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->mkdir == NULL) return (-2);
   return (request->mkdir (request, directory));
}
/*****************************************************************************/
int gftp_rename_file (gftp_request *request, const char *oldname, const char *newname) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->rename == NULL) return (-2);
   return (request->rename (request, oldname, newname));
}
/*****************************************************************************/
int gftp_chmod (gftp_request *request, const char *file, int mode) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->chmod == NULL) return (-2);
   return (request->chmod (request, file, mode));
}
/*****************************************************************************/
char gftp_site_cmd (gftp_request *request, const char *command) {
   g_return_val_if_fail (request != NULL, -2);
   
   if (request->site == NULL) return (-2);
   return (request->site (request, command));
}
/*****************************************************************************/
void gftp_set_proxy_config (gftp_request *request, const char *proxy_config) {
   int len;

   g_return_if_fail (request != NULL);
   g_return_if_fail (proxy_config != NULL);
   
   if (request->proxy_config) g_free (request->proxy_config);
   len = strlen (proxy_config);

   if (len > 0 && (proxy_config[len - 1] != 'n' || 
   	proxy_config[strlen (proxy_config) - 2] != '%')) {
      
      len += 2;
   }

   request->proxy_config = g_malloc (len + 1);
   strcpy (request->proxy_config, proxy_config);
   if (len != strlen (proxy_config)) {
      request->proxy_config[len - 2] = '%';
      request->proxy_config[len - 1] = 'n';
      request->proxy_config[len] = '\0';
   }
}
/*****************************************************************************/
long gftp_get_file_size (gftp_request *request, const char *filename) {
   g_return_val_if_fail (request != NULL, 0);
   
   if (request->get_file_size == NULL) return (0);
   return (request->get_file_size (request, filename));
}
/*****************************************************************************/
int gftp_need_proxy (gftp_request *request) {
   gftp_proxy_hosts *hostname;
   unsigned char addy[4];
   GList *templist;
   gint32 netaddr;
   char *pos;
   
   request->host = NULL;
   if (request->proxy_hosts == NULL) {
      return (request->proxy_hostname != NULL && *request->proxy_hostname != '\0' &&
      	request->proxy_config != NULL && *request->proxy_config != '\0');
   }

   if (request->logging) {
      request->logging_function (gftp_logging_misc, request->user_data, _("Looking up %s\n"),
      		request->hostname);
   }
   if (!(request->host = gethostbyname (request->hostname))) {
      if (request->logging) {
         request->logging_function (gftp_logging_error, request->user_data, _("Cannot look up hostname %s: %s\n"),
         	request->hostname, g_strerror (errno));
      }
      return (-1);
   }

   templist = request->proxy_hosts;
   while (templist != NULL) {
      hostname = templist->data;
      if (hostname->domain && strlen (request->hostname) > strlen (hostname->domain)) {
         pos = request->hostname + strlen (request->hostname) - strlen (hostname->domain);
         if (strcmp (hostname->domain, pos) == 0) {
            return (0);
         }
      }
      if (hostname->network_address != 0) {
         memcpy (addy, request->host->h_addr_list[0], sizeof (addy));
         netaddr = (((addy[0] & 0xff) << 24) | ((addy[1] & 0xff) << 16) | ((addy[2] & 0xff) << 8) | (addy[3] & 0xff)) & hostname->netmask;
         if (netaddr == hostname->network_address) {
            return (0);
         }
      }
      templist = templist->next;
   }
   return (request->proxy_hostname != NULL && *request->proxy_hostname != '\0' &&
   	request->proxy_config != NULL && *request->proxy_config != '\0');
}
/*****************************************************************************/
