/*
 * ProFTPD: mod_mysql -- Support for connecting to MySQL databases.
 * Time-stamp: <1999-10-04 03:21:21 root>
 * Copyright (c) 1998-1999 Johnie Ingram.
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307, USA.
 */

#define MOD_MYSQL_VERSION "mod_mysql/2.0"

/* -- DO NOT MODIFY THE LINE BELOW UNLESS YOU FEEL LIKE IT --
 * $Libraries: -lm -lmysqlclient $
 */

/* This is mod_mysql, contrib software for proftpd 1.2.0pre3 and above.
   For more information contact Johnie Ingram <johnie@netgod.net>.

   History Log:

   * 1999-09-19: v2.0: Most directives split into mod_sql; invented API.

*/

#include "conf.h"
#include <mysql.h>

/* *INDENT-OFF* */

static MYSQL mod_mysql_server;
static MYSQL *mysqldb = 0;

/* Maximum username field to expect, etc. */
#define ARBITRARY_MAX                   128

static struct
{
  char *sql_host;		/* Data for connecting, set by MySQLInfo. */
  char *sql_user;
  char *sql_pass;
  char *sql_dbname;

  int ok;
  int opens;
} g;

/* *INDENT-ON* */

/* **************************************************************** */

MODRET
sql_cmd_close (cmd_rec * cmd)
{
  log_debug (DEBUG5, "mysql: close [%i] for %s", g.opens, cmd->argv[0]);

  if (!g.ok || g.opens--)
    return DECLINED (cmd);

  if (mysqldb)
    {
      log_debug (DEBUG4, "mysql: disconnecting: %s", mysql_stat (mysqldb));
      mysql_close (mysqldb);
    }
  mysqldb = NULL;
  return DECLINED (cmd);
}

MODRET
sql_cmd_open (cmd_rec * cmd)
{
  if (!g.ok)
    return DECLINED (cmd);

  g.opens++;
  log_debug (DEBUG5, "mysql: open [%i] for %s", g.opens, cmd->argv[0]);
  if (g.opens > 1)
    return HANDLED (cmd);

  if (!(mysqldb = mysql_connect (&mod_mysql_server, g.sql_host,
				 g.sql_user, g.sql_pass)))
    {
      log_pri (LOG_ERR, "mysql: client %s connect FAILED to %s@%s",
	       mysql_get_client_info (), g.sql_user, g.sql_host);
      g.ok = FALSE;
      g.opens = 0;
      return DECLINED (cmd);
    }
  log_debug (DEBUG5, "mysql: connect OK %s -> %s (%s@%s)",
	     mysql_get_client_info (), mysql_get_server_info (mysqldb),
	     g.sql_user, g.sql_host);

  return HANDLED (cmd);
}

MODRET
_do_query (cmd_rec * cmd, const char *query, int update)
{
  int error = 1;

  if (!g.ok)
    return DECLINED (cmd);

  block_signals ();

  /* This forces a quick ping of the remote server, so we know if its there. */
  if (mysqldb)
    mysql_select_db (mysqldb, g.sql_dbname);

  if (!mysqldb
      || ((error = mysql_query (mysqldb, query))
	  && !strcasecmp (mysql_error (mysqldb), "mysql server has gone")))
    {
      /* We need to restart the server link. */
      if (mysqldb)
	log_pri (LOG_ERR, "mysql: server has wandered off (%s/%s)",
		 g.sql_host, g.sql_dbname);
      sql_cmd_open (cmd);
      if (!mysqldb)
	return DECLINED (cmd);
      error = mysql_select_db (mysqldb, g.sql_dbname)
	|| mysql_query (mysqldb, query);
    }

  unblock_signals ();

  if (error)
    {
      log_debug (DEBUG4, "mysql: %s failed: \"%s\": %s",
		 (update) ? "update" : "select",
		 query, mysql_error (mysqldb));
      return DECLINED (cmd);
    }

  log_debug (DEBUG5, "mysql: %s OK: [%s] \"%s\"",
	     (update) ? "update" : "select", g.sql_dbname, query);
  return HANDLED (cmd);
}

MODRET
sql_cmd_update (cmd_rec * cmd)
{
  return _do_query (cmd, cmd->argv[1], TRUE);
}

MODRET
sql_cmd_select (cmd_rec * cmd)
{
  MODRET mr;
  MYSQL_RES *result;
  MYSQL_ROW sql_row;
  int i, j;

  mr = _do_query (cmd, cmd->argv[1], FALSE);
  if (!MODRET_ISHANDLED (mr))
    return DECLINED (mr);

  if ((result = mysql_store_result (mysqldb)))
    {
      int rcount = mysql_num_rows (result);
      int fcount = mysql_num_fields (result);
      int count = rcount * fcount;

      char **data = pcalloc (cmd->tmp_pool, sizeof (char *) * (count + 1));
      for (i = 0, count = 0; i < rcount; i++)
	{
	  sql_row = mysql_fetch_row (result);
	  for (j = 0; j < fcount; j++)
	    data[count++] = pstrdup (cmd->tmp_pool, sql_row[j]);
	}
      mysql_free_result (result);
      data[count] = NULL;
      mr->data = data;
    }
  return mr;
}

static authtable mysql_authtab[] = {
  {0, "dbd_open", sql_cmd_open},
  {0, "dbd_close", sql_cmd_close},
  {0, "dbd_update", sql_cmd_update},
  {0, "dbd_select", sql_cmd_select},

  {0, NULL, NULL}
};

/* **************************************************************** */

MODRET set_sqlinfo (cmd_rec * cmd)
{
  CHECK_ARGS (cmd, 4);
  CHECK_CONF (cmd, CONF_ROOT | CONF_GLOBAL | CONF_VIRTUAL);
  add_config_param_str ("MySQLInfo", 4,
			(void *) cmd->argv[1], (void *) cmd->argv[2],
			(void *) cmd->argv[3], (void *) cmd->argv[4]);
  return HANDLED (cmd);
}

static conftable mysql_conftab[] = {
/* *INDENT-OFF* */

  { "MySQLInfo",              set_sqlinfo,       NULL },

  { 0,	      NULL }

/* *INDENT-ON* */
};

/* **************************************************************** */

static int
mysql_modinit ()
{
  config_rec *c;

  memset (&g, 0, sizeof (g));
  if (!(c = find_config (CURRENT_CONF, CONF_PARAM, "MySQLInfo", FALSE)))
    return 0;

  g.sql_host = pstrdup (session.pool, c->argv[0]);
  g.sql_user = pstrdup (session.pool, c->argv[1]);
  g.sql_pass = pstrdup (session.pool, c->argv[2]);
  g.sql_dbname = pstrdup (session.pool, c->argv[3]);
  g.ok = TRUE;

  log_debug (DEBUG5, "%s: configured: db %s at %s@%s",
	     MOD_MYSQL_VERSION, g.sql_dbname, g.sql_user, g.sql_host);
  return 0;
}

module mysql_module = {
  NULL, NULL,			/* Always NULL */
  0x20,				/* API Version 2.0 */
  "mysql",
  mysql_conftab,		/* SQL configuration handler table */
  NULL,				/* SQL command handler table */
  mysql_authtab,		/* SQL authentication handler table */
  mysql_modinit,		/* Pre-fork "parent-mode" init */
  mysql_modinit			/* Post-fork "child mode" init */
};
