"""

Interface to various cdrtools (currently cdrecord and mkisofs)

"""

from tools import cmdoutput, striplist, which, TRUE, FALSE
from string import split, join, digits, find, strip, letters, atoi, lower
from types import StringType

import os

TRACK_MODE_DATA = "data"
TRACK_MODE_MODE2 = "mode2"
TRACK_MODE_AUDIO = "audio"
TRACK_MODE_XA1 = "xa1"
TRACK_MODE_XA2 = "xa2"

WRITE_MODE_DAO = "dao"

BLANK_MODE_DISC = "disc"
BLANK_MODE_FAST = "fast"
BLANK_MODE_SESSION = "session"

TRACK_STDIN = "-"

class cdrtools:

    def __init__(self):
        self.__cdrtools_devices = []
        self.__cdrtools_cdrecord_command = which("cdrecord")
        self.__cdrtools_mkisofs_command = which("mkisofs")
        self.__cdrtools_scanbus()

    def devices(self):
        """ Returns the devices found by cdrecord. """
        return self.__cdrtools_devices

    def joliet_charsets(self):
        jolietCharsetsTemp = cmdoutput("%s -jcharset help 2>&1" % self.__cdrtools_mkisofs_command, strip = TRUE)
        jolietCharsets = []
        appendCharset = FALSE
        for i in range(len(jolietCharsetsTemp)):
            if (appendCharset == TRUE):
                jolietCharsets.append(jolietCharsetsTemp[i])
            if (jolietCharsetsTemp[i][:14] == "Known charsets"):
                appendCharset = TRUE
        jolietCharsets.sort()
        return jolietCharsets

    def __cdrtools_scanbus(self):
        """ Gets the list of available devices by executing "cdrecord -scanbus". """
        output = cmdoutput("%s -scanbus 2>&1" % self.__cdrtools_cdrecord_command)
        self.__cdrtools_devices = []
        for i in range(len(output)):
            line = output[i]
            if (line[0] == "\t") and (line[1] in digits):
                if (find(line, "CD-ROM") != -1):
                    channel = line[1]
                    id = line[3]
                    lun = line[5]
                    vendor = "%s %s %s" % (strip(line[13:21]), strip(line[24:40]), strip(line[43:47]))
                    self.__cdrtools_devices.append([channel, id, lun, vendor])

class cdrecord:

    valid_track_modes = [ TRACK_MODE_DATA, TRACK_MODE_MODE2, TRACK_MODE_AUDIO, TRACK_MODE_XA1, TRACK_MODE_XA2 ]
    valid_write_modes = [ WRITE_MODE_DAO ]
    valid_blank_modes = [ BLANK_MODE_DISC, BLANK_MODE_FAST, BLANK_MODE_SESSION]

    def __init__(self, bus, target, lun, device):
        if (type(bus) == StringType): bus = atoi(bus)
        if (type(target) == StringType): target = atoi(target)
        if (type(lun) == StringType): lun = atoi(lun)
        self.__cdrecord_bus = bus
        self.__cdrecord_target = target
        self.__cdrecord_lun = lun
        self.__cdrecord_device = device
        self.__cdrecord_verbose_mode = FALSE
        self.__cdrecord_burnfree = FALSE
        self.__cdrecord_overburn = FALSE
        self.__cdrecord_fixate = TRUE
        self.__cdrecord_multisession = FALSE
        self.__cdrecord_dummy_mode = FALSE
        self.__cdrecord_eject = TRUE
        self.__cdrecord_fifosize = 4 * 1024 * 1024                    # 4MB
        self.__cdrecord_speed = 1
        self.__cdrecord_pad_tracks = FALSE
        self.__cdrecord_tracks = []
        self.__cdrecord_track_mode = ""
        self.__cdrecord_write_mode = ""
        self.__cdrecord_swap_audio_tracks = FALSE
        self.__cdrecord_blank_mode = None
        self.__cdrecord_read_version()

    def version(self):
        """ Returns the version string of cdrecord. """
        if (self.__cdrecord_version_micro != None):
            return "%d.%d%s" % (self.__cdrecord_version_major, self.__cdrecord_version_minor, self.__cdrecord_version_micro)
        else:
            return "%d.%d" % (self.__cdrecord_version_major, self.__cdrecord_version_minor)

    def device(self):
        return self.__cdrecord_device

    def driver_opts(self):
        """ Gets the available driver options for a given device. 

        Note: this function doesn't use the cdrecord driveropts yet.
        """
        # output = cmdoutput("cdrecord -checkdrive dev=%s,%s,%s driveropts=help 2>&1" % (bus, target, lun), strip = TRUE)
        driverOpts = []
        if (self.__cdrecord_version_major >= 1) and (self.__cdrecord_version_minor >= 11):
            if (self.__cdrecord_version_micro == None) or (self.__cdrecord_version_micro >= "a02"):
                driverOpts.append("burnfree")
        if (not "burnfree" in driverOpts):
            driverOpts.append("burnproof")
        return driverOpts

    def burnfree(self, value):
        self.__cdrecord_burnfree = value

    def overburn(self, value):
        self.__cdrecord_overburn = value

    def overburn_supported(self):
        if (self.__cdrecord_version_major >= 1) and (self.__cdrecord_version_minor >= 11):
            if (self.__cdrecord_version_micro == None) or (self.__cdrecord_version_micro >= "a01"):
                return TRUE
            else:
                return FALSE
        return FALSE

    def previous_session(self):
        output = cmdoutput("cdrecord -msinfo dev=%d,%d,%d 2>&1" % (self.__cdrecord_bus, self.__cdrecord_target, self.__cdrecord_lun), TRUE)[-1]
        if (find(lower(output), "cannot") != -1) or (not output[0] in digits):
            return None
        else:
            return output

    def verbose_mode(self, value):
        self.__cdrecord_verbose_mode = value

    def dummy_mode(self, value):
        self.__cdrecord_dummy_mode = value

    def fixate(self, value):
        self.__cdrecord_fixate = value

    def multisession(self, value):
        self.__cdrecord_multisession = value

    def eject(self, value):
        self.__cdrecord_eject = value

    def fifosize(self, value):
        if (type(value) == StringType): value = atoi(value)
        self.__cdrecord_fifosize = value

    def speed(self, value):
        if (type(value) == StringType): value = atoi(value)
        self.__cdrecord_speed = value

    def pad_tracks(self, value):
        self.__cdrecord_pad_tracks = value

    def track_mode(self, mode):
        if (cdrecord.valid_track_modes.count(lower(mode)) > 0):
            self.__cdrecord_track_mode = "-%s" % lower(mode)

    def write_mode(self, mode):
        if (cdrecord.valid_write_modes.count(lower(mode)) > 0):
            self.__cdrecord_write_mode = "-%s" % lower(mode)

    def blank_mode(self, mode):
        if (cdrecord.valid_blank_modes.count(lower(mode)) > 0):
            self.__cdrecord_blank_mode = lower(mode)

    def swap_audio_tracks(self, value):
        self.__cdrecord_swap_audio_tracks = value

    def add_track(self, value):
        if (value == TRACK_STDIN):
            self.__cdrecord_tracks.append(value)
        else:
            self.__cdrecord_tracks.append("\"%s\"" % value)

    def command_line(self):
        cmdline = "cdrecord"
        if (self.__cdrecord_verbose_mode): cmdline = "%s -v" % cmdline
        if (self.__cdrecord_burnfree):
            options = self.driver_opts()
            if ("burnfree" in options):
                cmdline = "%s driveropts=burnfree" % cmdline
            elif ("burnproof" in options):
                cmdline = "%s driveropts=burnproof" % cmdline
        if (self.__cdrecord_dummy_mode): cmdline ="%s -dummy" % cmdline
        if (self.__cdrecord_eject): cmdline = "%s -eject" % cmdline

        if (self.__cdrecord_blank_mode == None):
            if (self.__cdrecord_overburn):
                if (self.overburn_supported() == TRUE):
                    self.write_mode(WRITE_MODE_DAO)
                    cmdline = "%s -overburn" % cmdline
            if (self.__cdrecord_fixate == FALSE): cmdline = "%s -nofix" % cmdline
            if (self.__cdrecord_multisession): cmdline = "%s -multi" % cmdline
            if (self.__cdrecord_pad_tracks): cmdline = "%s -pad" % cmdline
            cmdline = "%s %s" % (cmdline, self.__cdrecord_track_mode)
            cmdline = "%s %s" % (cmdline, self.__cdrecord_write_mode)
            if (self.__cdrecord_track_mode == TRACK_MODE_AUDIO):
                if (self.__cdrecord_swap_audio_tracks): cmdline = "%s -swab" % cmdline

        cmdline = "%s -fs=%d" % (cmdline, self.__cdrecord_fifosize)
        cmdline = "%s dev=%d,%d,%d" % (cmdline, self.__cdrecord_bus, self.__cdrecord_target, self.__cdrecord_lun)
        cmdline = "%s speed=%d" % (cmdline, self.__cdrecord_speed)
        
        if (self.__cdrecord_blank_mode == None):
            for i in range(len(self.__cdrecord_tracks)):
                cmdline = "%s %s" % (cmdline, self.__cdrecord_tracks[i])
        else:
            cmdline = "%s -blank %s" % (cmdline, self.__cdrecord_blank_mode)
        return cmdline

    # Private methods of the cdrecord class

    def __cdrecord_read_version(self):
        """ Reads the version string by executing "cdrecord -version". """
        output = cmdoutput("cdrecord -version 2>&1", strip = TRUE)
        versionLine = output[0]
        splitted = split(versionLine, " ")
        version = split(splitted[1], ".")
        self.__cdrecord_version_major = atoi(version[0])
        position = -1
        for i in range(len(version[1])):
            if (version[1][i] in letters):
                position = i
        if (position != -1):
            self.__cdrecord_version_minor = atoi(version[1][:position])
            self.__cdrecord_version_micro = version[1][position:]
        else:
            self.__cdrecord_version_minor = atoi(version[1])
            self.__cdrecord_version_micro = None

class mkisofs:

    def __init__(self):
        self.__mkisofs_verbose_mode = FALSE
        self.__mkisofs_gui_behaviour = FALSE
        self.__mkisofs_disable_deep_relocation = TRUE
        self.__mkisofs_gui_behaviour_supported = FALSE
        self.__mkisofs_full_iso9660_filenames = FALSE
        self.__mkisofs_allow_leading_dots = FALSE
        self.__mkisofs_follow_links = TRUE
        self.__mkisofs_joliet_charset = None
        self.__mkisofs_magic_parameters = None
        self.__mkisofs_device = None
        self.__mkisofs_rational_rock = TRUE
        self.__mkisofs_rock_ridge = TRUE
        self.__mkisofs_omot_trailing_periods = TRUE
        self.__mkisofs_volume_id = None
        self.__mkisofs_output_file = None
        self.__mkisofs_boot_image = None
        self.__mkisofs_boot_catalog = None
        self.__mkisofs_files = []
        self.__mkisofs_check_gui_parameter()

    def verbose_mode(self, value):
        self.__mkisofs_verbose_mode = value

    def gui_behaviour(self, value):
        if ((value) and (self.__mkisofs_gui_behaviour_supported)):
            self.__mkisofs_gui_behaviour = TRUE
        else:
            self.__mkisofs_gui_behaviour = FALSE

    def disable_deep_relocation(self, value):
        self.__mkisofs_disable_deep_relocation = value

    def full_iso9660_filenames(self, value):
        self.__mkisofs_full_iso9660_filenames = value

    def allow_leading_dots(self, value):
        self.__mkisofs_allow_leading_dots = value

    def follow_links(self, value):
        self.__mkisofs_follow_links = value

    def joliet_charset(self, charset):
        self.__mkisofs_joliet_charset = charset

    def multi_session(self, magic_parameters, device):
        if (magic_parameters != None):
            self.__mkisofs_magic_parameters = magic_parameters
            self.__mkisofs_device = device

    def rational_rock(self, value):
        self.__mkisofs_rational_rock = value

    def rock_ridge(self, value):
        self.__mkisofs_rock_ridge = value

    def omit_trailing_periods(self, value):
        self.__mkisofs_omot_trailing_periods = value

    def volume_id(self, id):
        if (id != "") and (id != None):
            self.__mkisofs_volume_id = id

    def output_file(self, filename):
        if (filename != ""):
            self.__mkisofs_output_file = filename

    def boot_image(self, filename):
        if (filename != ""):
            self.__mkisofs_boot_image = filename

    def boot_catalog(self, filename):
        self.__mkisofs_boot_catalog = filename

    def add_file(self, filename):
        if (self.__mkisofs_files.count(filename) == 0):
            self.__mkisofs_files.append(filename)

    def command_line(self):
        """ Returns the complete command line including all set parameters. """
        cmdline = "mkisofs"
        if (self.__mkisofs_verbose_mode): cmdline = "%s -v" % cmdline        
        if (self.__mkisofs_gui_behaviour): cmdline = "%s -gui" % cmdline
        if (self.__mkisofs_disable_deep_relocation): cmdline = "%s -D" % cmdline
        if (self.__mkisofs_full_iso9660_filenames): cmdline = "%s -l" % cmdline
        if (self.__mkisofs_allow_leading_dots): cmdline = "%s -L" % cmdline
        if (self.__mkisofs_follow_links): cmdline = "%s -f" % cmdline
        if (self.__mkisofs_joliet_charset != None): cmdline = "%s -J -jcharset %s" % (cmdline, self.__mkisofs_joliet_charset)
        if (self.__mkisofs_magic_parameters != None): cmdline = "%s -C %s -M %s" % (cmdline, self.__mkisofs_magic_parameters, self.__mkisofs_device)
        if (self.__mkisofs_rational_rock): cmdline = "%s -r" % cmdline
        if (self.__mkisofs_rock_ridge): cmdline = "%s -R" % cmdline
        if (self.__mkisofs_omot_trailing_periods): cmdline = "%s -d" % cmdline
        if (self.__mkisofs_volume_id != None): cmdline = "%s -V \"%s\"" % (cmdline, self.__mkisofs_volume_id)
        if (self.__mkisofs_output_file != None): cmdline = "%s -o \"%s\"" % (cmdline, self.__mkisofs_output_file)
        if ((self.__mkisofs_boot_catalog != None) and (self.__mkisofs_boot_image != None)):
            cmdline = "%s -b \"%s\" -c \"%s\"" % (cmdline, self.__mkisofs_boot_image, self.__mkisofs_boot_catalog)
        if (len(self.__mkisofs_files) > 0):
            cmdline = "%s -graft-points" % cmdline
            for i in range(len(self.__mkisofs_files)):
                filename = self.__mkisofs_files[i]
                if (os.path.isdir(filename)):
                    if (filename[-1] == "/"):
                        filename = filename[:-1]
                    shortfilename = split(filename, "/")[-1]
                    cmdline = "%s \"/%s/\"=\"%s\"" % (cmdline, shortfilename, filename)
                else:
                    cmdline = "%s \"%s\"" % (cmdline, filename)
        return cmdline

    def gui_behaviour_supported(self):
        return self.__mkisofs_gui_behaviour_supported

    def __mkisofs_check_gui_parameter(self):
        output = cmdoutput("mkisofs 2>&1 | grep gui")
        if (len(output) == 0):
            self.__mkisofs_gui_behaviour_supported = FALSE
        else:
            self.__mkisofs_gui_behaviour_supported = TRUE

class readcd:

    def __init__(self, bus, target, lun):
        if (type(bus) == StringType): bus = atoi(bus)
        if (type(target) == StringType): target = atoi(target)
        if (type(lun) == StringType): lun = atoi(lun)
        self.__readcd_bus = bus
        self.__readcd_target = target
        self.__readcd_lun = lun
        self.__readcd_filename = None

    def filename(self, value):
        self.__readcd_filename = value

    def command_line(self):
        cmdline = "readcd"
        cmdline = "%s dev=%d,%d,%d" % (cmdline, self.__readcd_bus, self.__readcd_target, self.__readcd_lun)
        cmdline = "%s -f %s" % (cmdline, self.__readcd_filename)
        return cmdline
