/* $Id: rgpsp_common.cc,v 1.10 2000/09/23 16:32:59 bergo Exp $ */

/*

    GPS/RGPSP - Graphical Process Statistics / Remote GPS Poller
    Copyright (C) 1999 Felipe Bergo and Beat Christen
    gps@seul.org

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/

#include <iostream.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <signal.h>
#include <syslog.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#include <signal.h>
#include <sys/wait.h>
#include "transient.h"

#define CONFIG_FILE "/etc/rgpsp.conf"

#ifdef HAVEFREEBSD
#include <osreldate.h>
#if __FreeBSD_version < 400000
typedef int socklen_t;
#endif
#endif

/* YOUR native file must implement these two! */
void poller_routine();
char *native_description();

/* local ones */
void usage(int argc, char **argv);
void signal_handler(int sig);
void output_error(int level);
int run_servlet();
char * forking_gethostbyname(char *name,int timeout);
void read_config();

/* global vars */
int tcpport     = 24374;
int allow_any   = 1;
int nhosts      = 0;
int amdaemon    = 1;
int silent      = 0;

char *ip_buffer;

int sock=-1, actsock=-1;
char err_msg[512];

int main(int argc,char **argv) {
  int i,result;

  if (argc>200) {
    sprintf(err_msg,"too many arguments (%d), exited",argc);
    output_error(LOG_ERR);
    exit(2);
  }

  ip_buffer=(char *)malloc(32*argc);
  if (ip_buffer==NULL) {
    sprintf(err_msg,"unable to allocate memory, exited",argc);
    output_error(LOG_ERR);
    exit(-1);
  }

  /* parse cmdline */
  for(i=1;i<argc;i++) {
    if (strcmp(argv[i],"-q")==0) {
      silent=1;
      continue;
    }
    if (strcmp(argv[i],"--nodaemon")==0) {
      amdaemon=0;
      continue;
    }
    if ((strcmp(argv[i],"-h")==0)||
	(strcmp(argv[i],"-v")==0)) {
      usage(argc,argv);
      free(ip_buffer);
      return -1;
    }
    if (i<(argc-1)) /* -p */
      if ((strcmp(argv[i],"-p")==0)) {
	tcpport=atoi(argv[i+1]);
	i++;
	continue;
      }
    if (strlen(argv[i])>31) {
      sprintf(err_msg,"parameter too long, skipping: %s",argv[i]);
      output_error(LOG_INFO);
      continue;
    }
    strcpy(&ip_buffer[32*nhosts],argv[i]);
    nhosts++;
    allow_any=0;
  }

  if (amdaemon) { /* daemonize */
    if (fork()>0)
      exit(0);
    chdir("/");
    umask(0);
    setsid();
  }

  read_config();

  if (allow_any) {
    sprintf(err_msg,"warning: any host will be allowed to connect");
    output_error(LOG_NOTICE);
  }

  /* attach signals to catch for proper termination */
  signal(SIGKILL, signal_handler);
  signal(SIGSTOP, signal_handler);
  signal(SIGHUP, signal_handler);
  signal(SIGSEGV, signal_handler);
  signal(SIGPIPE, signal_handler);
  signal(SIGQUIT, signal_handler);
  signal(SIGTERM, signal_handler);
  signal(SIGINT, signal_handler);


  sprintf(err_msg,"ready to answer queries");
  output_error(LOG_INFO);
  result=run_servlet();

  free(ip_buffer);
  ip_buffer=NULL;
  return result;
}

void usage(int argc, char **argv) {
  cerr << "rgpsp (" << native_description() << ") version " << GPS_RELEASE << "\n\n";
  cerr << "Usage: " << argv[0] << " [-p port] [--nodaemon] [host ip] [host ip] [...]\n";
  cerr << "       " << argv[0] << " -h | -v\n\n";
}


void signal_handler(int sig) {
  sprintf(err_msg,"caught signal %d, terminating",sig);
  output_error(LOG_INFO);
  if(actsock!=-1)
    close(actsock);
  if(sock!=-1)
    close(sock);  
  sleep(1);
  if (ip_buffer!=NULL) {
    free(ip_buffer);
    ip_buffer=NULL;
  }
  exit(0);
}

/* connection manager routine */
int run_servlet() {
  int fd;
  socklen_t addrlen;
  struct    sockaddr_in sin;
  struct    sockaddr_in pin;
  int c[4];
  int i,j,status,pid;
  char remote[32];

  sock = socket(AF_INET, SOCK_STREAM, 0);
  
  memset(&sin, 0, sizeof(sin));
  sin.sin_family = AF_INET;
  sin.sin_addr.s_addr = INADDR_ANY;
  sin.sin_port = htons(tcpport);
  
  if (bind(sock,(struct sockaddr *) &sin,sizeof(sin))==-1) {
    sprintf(err_msg,"bind failed on port %d",tcpport);
    output_error(LOG_ERR);
    return(-1);
  }
  
  listen(sock, 5);

  while(1)  { 
    addrlen=(socklen_t)sizeof(pin);
    actsock = accept(sock, (struct sockaddr *)  &pin, &addrlen);
    c[3] = ( pin.sin_addr.s_addr >> 24 ) & 0xff;
    c[2] = ( pin.sin_addr.s_addr >> 16 ) & 0xff;
    c[1] = ( pin.sin_addr.s_addr >> 8 )  & 0xff;
    c[0] = ( pin.sin_addr.s_addr ) & 0xff;
    snprintf(remote, 32, "%d.%d.%d.%d", c[0], c[1], c[2], c[3]);
    
    j=1;
    if (!allow_any)
      for(i=0,j=0;i<nhosts;i++)
	if (strcmp(remote,&ip_buffer[32*i])==0) {
	  j++;
	  break;
	}
    
    if (!j) {
      sprintf(err_msg,"refused illegal connection from %s",remote);
      output_error(LOG_NOTICE);
    } else {
      /* log with parent pid */
      sprintf(err_msg,"accepted connection from %s",remote);
      output_error(LOG_NOTICE);

      /* ok. so we want to fork to the polling routine */
      if ( (pid = fork()) == 0 ) {

	/* and spawn off another child to get rid of the ppid */
	if (fork()>0)
	  exit(0);

	/* streaming in and out */
	close(0); /* stdin */
	dup(actsock);
	close(1); /* stdout */
	dup(actsock);
	close(2); /* stderr */
	dup(actsock);
	close(actsock);
	poller_routine();
	exit(0);
      }
      waitpid(pid,&status,0);
    }
    close(actsock); actsock=-1;
    sleep(1); 
  }

  /* close up both sockets */
  close(sock);  
  /* give client a chance to properly shutdown */
  usleep(400000);
  return 0;
}

void output_error(int level) {
  if (!amdaemon) {
    if (!silent)
      cerr << "** rgpsp ** " << err_msg << "\n";
  } else {
    openlog("rgpsp",LOG_PID,LOG_DAEMON);
    syslog(LOG_DAEMON|level,"%s",err_msg);
    closelog();
  }
}

char * forking_gethostbyname(char *name,int timeout) {
  struct hostent *he;
  int c[4],h;
  int pid;
  time_t zero,now;
  int ipc[2];
  static char result[256];

  zero=time(NULL);

  if (pipe(ipc))
    return NULL;

  pid=fork();

  if (pid==0) {
    he=gethostbyname(name);
    if (he!=NULL) {
      c[3] = ((unsigned char *)(he->h_addr_list[0]))[0];
      c[2] = ((unsigned char *)(he->h_addr_list[0]))[1];
      c[1] = ((unsigned char *)(he->h_addr_list[0]))[2];
      c[0] = ((unsigned char *)(he->h_addr_list[0]))[3];
      h=1;
      write(ipc[1],&h,sizeof(int));
      write(ipc[1],c,4*sizeof(int));
      usleep(20000);
      exit(0);
    }
    h=0;
    write(ipc[1],&h,sizeof(int));
    usleep(20000);
    exit(0);
  } else {
    fcntl(ipc[0],F_SETFL,O_NONBLOCK);
    while(1) {
      if (read(ipc[0],&h,sizeof(int))>=0) {
	fcntl(ipc[0],F_SETFL,0);
	if (h)
	  read(ipc[0],c,sizeof(int)*4);
	waitpid(pid,NULL,0);
	close(ipc[0]);
	close(ipc[1]);
	if (h) {
	  sprintf(result,"%d.%d.%d.%d",c[3],c[2],c[1],c[0]);	
	  return result;
	} else
	  return NULL;
      }
      now=time(NULL);
      if ((now-zero)>=timeout) {
	kill(pid,SIGKILL);
	waitpid(pid,NULL,0);
	sprintf(err_msg,"hostname lookup for %s timed out, excluded",name);
	output_error(LOG_INFO);
	close(ipc[0]);
	close(ipc[1]);
	return NULL;
      }
      usleep(10000);
    }
  }
  return NULL;
}

void read_config() {
  FILE *f;
  char buf[128],*p,*q;
  char *tmpp;
  int cs;
  int onhosts,discard,i;

  onhosts=nhosts;
  discard=0;

  f=fopen(CONFIG_FILE,"r");
  if (!f) return;

  q=(char *)malloc(cs=(5<<10));
  if (q==NULL) {
    sprintf(err_msg,"unable to allocate memory -- config file not read");
    output_error(LOG_WARNING);
    fclose(f);
    return;
  }
  q[0]=0;

  while(fgets(buf,127,f)!=NULL) {
    if (strlen(buf)==0)
      break;
    if (buf[0]=='#')
      continue;
    p=strtok(buf," \n\t");
    if (p==NULL)
      continue;
    strcat(q,p);
    strcat(q,"\t");
    if (strlen(q)>(cs-1024)) {
      cs+=1024;
      q=(char *)realloc(q,cs);
      if (q==NULL) {
	sprintf(err_msg,"unable to allocate memory -- config file not read");
	output_error(LOG_WARNING);
	fclose(f);
	return;
      }
    }
  }
  fclose(f);

  for(p=strtok(q,"\t");p!=NULL;p=strtok(NULL,"\t")) {
    if (strcmp(p,"*")==0) {
      allow_any=1;
      break;
    }
    if (strcmp(p,"!")==0) {
      discard=1;
      continue;
    }    
    if (isdigit(p[0])) {
      ip_buffer=(char *)realloc(ip_buffer,32*(nhosts+3));
      if (ip_buffer==NULL) {
	sprintf(err_msg,"unable to allocate memory");
	output_error(LOG_ERR);
	exit(-1);
      }
      strcpy(&ip_buffer[32*nhosts],p);
      nhosts++;
    } else {
      /* dns lookup in another process -- no locking here! */
      
      tmpp=forking_gethostbyname(p,10);

      if (tmpp==NULL) {
	sprintf(err_msg,"unable to resolve IP address for %s",p);
	output_error(LOG_INFO);
	continue;
      }

      ip_buffer=(char *)realloc(ip_buffer,32*(nhosts+3));

      if (ip_buffer==NULL) {
	sprintf(err_msg,"unable to allocate memory");
	output_error(LOG_ERR);
	exit(-1);
      }

      strcpy(&ip_buffer[32*nhosts],tmpp);
      nhosts++;

    } /* else */

  } /* for */
  free(q);

  if ((discard)&&(onhosts)) {
    for(i=0;i<(nhosts-onhosts);i++)
      strcpy(&ip_buffer[32*i],&ip_buffer[32*(i+nhosts)]);
    nhosts-=onhosts;
    sprintf(err_msg,"exclamation found, discarded %d hosts from command line",
	    onhosts);
    output_error(LOG_INFO);
  }

}
