/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */

/*
 *  Copyright (C) 2003 Takuro Ashie
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "prefs_lang.h"

#include <string.h>
#include <glib/gi18n.h>

#include "kazehakase.h"
#include "kz-dlist.h"
#include "mozilla-prefs.h"
#include "gtk-utils.h"

#define DATA_KEY "KzPrefsLang::info"

static GtkWidget     *prefs_lang_create          (void);
static void           prefs_lang_response        (GtkWidget *widget,
						  gint       response);


static KzPrefsWinPageEntry prefs_entry =
{
	path:            N_("/Language"),
	priority_hint:   0,
	ui_level:        KZ_UI_LEVEL_ALL,
	create:          prefs_lang_create,
	response:        prefs_lang_response,
};


typedef struct _KzPrefsLang
{
	GtkWidget     *main_vbox;
	GtkWidget     *option_menu_lang;
	GtkWidget     *option_menu_autodetect;
	GtkWidget     *lang_dlist;
	gboolean       enabled_list_updated;
} KzPrefsLang;


static void
prefs_lang_destroy (gpointer data)
{
	KzPrefsLang *prefsui = data;

	g_free(prefsui);
}


static void
cb_enabled_list_updated (KzDList *dlist, KzPrefsLang *prefsui)
{
	prefsui->enabled_list_updated = TRUE;
}


static GtkWidget *
prefs_lang_create (void)
{
	KzPrefsLang *prefsui = g_new0(KzPrefsLang, 1);
	GtkWidget *main_vbox, *vbox, *hbox;
	GtkWidget *label, *table, *option_menu, *frame,*dlist;
	gchar *str, **langs;
	guint i;

	prefsui->enabled_list_updated = FALSE;

	main_vbox = gtk_vbox_new (FALSE, 0);
	prefsui->main_vbox = main_vbox;
	g_object_set_data_full (G_OBJECT(main_vbox),
				DATA_KEY, prefsui,
				(GDestroyNotify) prefs_lang_destroy);

	label = kz_prefs_ui_utils_create_title(_("Language"));
	gtk_box_pack_start(GTK_BOX(main_vbox), label,
			   FALSE, FALSE, 0);
	gtk_widget_show(label);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(vbox), 5);
	gtk_box_pack_start(GTK_BOX(main_vbox), vbox,
			   TRUE, TRUE, 0);
	gtk_widget_show(vbox);

	table = gtk_table_new(2, 2, FALSE);
	gtk_box_pack_start(GTK_BOX(vbox), table,
			   FALSE, FALSE, 0);
	gtk_widget_show(table);

	/*
	 *  Default Encoding
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 5);
	gtk_widget_show(frame);

	label = gtk_label_new_with_mnemonic(_("_Default Encoding"));
	gtk_label_set_use_underline(GTK_LABEL(label), TRUE);
	gtk_frame_set_label_widget(GTK_FRAME(frame), label);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	option_menu = gtk_combo_box_new_text();
	prefsui->option_menu_lang = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	GTKUTIL_CREATE_MENU(option_menu, languages, n_languages, name);

	mozilla_prefs_get_string("intl.charset.default", &str);
	GTKUTIL_SET_DEFAULT_HISTORY(option_menu, str,
			    languages, n_languages, code);
	if (str)
		g_free(str);
	/*
	 *  Auto detect
	 */
	frame = gtk_frame_new(NULL);
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 5);
	gtk_widget_show(frame);

	label = gtk_label_new_with_mnemonic(_("A_utodetect Encoding"));
	gtk_frame_set_label_widget(GTK_FRAME(frame), label);
	gtk_widget_show(label);

	hbox = gtk_hbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(hbox), 5);
	gtk_container_add(GTK_CONTAINER(frame), hbox);
	gtk_widget_show(hbox);

	option_menu = gtk_combo_box_new_text();
	prefsui->option_menu_autodetect = option_menu;
	gtk_label_set_mnemonic_widget(GTK_LABEL(label), option_menu);
	gtk_box_pack_start(GTK_BOX(hbox), option_menu, FALSE, FALSE, 0);
	gtk_widget_show(option_menu);

	GTKUTIL_CREATE_MENU(option_menu, 
			    encoding_autodetectors,
			    n_encoding_autodetectors,
			    title);
	
	mozilla_prefs_get_string("intl.charset.detector", &str);
	GTKUTIL_SET_DEFAULT_HISTORY(option_menu, str,
			    encoding_autodetectors, n_encoding_autodetectors,
			    name);
	if (str)
		g_free(str);

	/*
	 *  Accept Language
	 */
	frame = gtk_frame_new(_("Accept Language"));
	gtk_frame_set_label_align(GTK_FRAME(frame), 0.03, 0.5);
	gtk_box_pack_start(GTK_BOX(vbox), frame, TRUE, TRUE, 5);
	gtk_widget_show(frame);

	dlist = kz_dlist_new(_("Available Languages"),
			     _("Enabled Languages"));
	prefsui->lang_dlist = dlist;
	gtk_container_set_border_width(GTK_CONTAINER(dlist), 5);
	gtk_container_add(GTK_CONTAINER(frame), dlist);
	gtk_widget_show(dlist);

	for (i = 0; i < n_accept_languages; i++)
	{
		const gchar *name = accept_languages[i].name;
		const gchar *code = accept_languages[i].code;

		kz_dlist_append_available_item(KZ_DLIST(dlist), name, code);
	}

	mozilla_prefs_get_string("intl.accept_languages", &str);
	langs = g_strsplit(str, ",", -1);
	g_free(str);
	if (langs)
	{
		for (i = 0; langs[i]; i++)
		{
			g_strstrip(langs[i]);
			kz_dlist_column_add_by_id(KZ_DLIST(dlist), langs[i]);
		}
		g_strfreev(langs);
	}
	g_signal_connect(dlist, "enabled-list-updated",
			 G_CALLBACK(cb_enabled_list_updated), prefsui);

	return main_vbox;
}


static void
prefs_lang_apply (KzPrefsLang *prefsui)
{
	guint i, j;

	g_return_if_fail(prefsui);

	i = gtk_combo_box_get_active(GTK_COMBO_BOX(prefsui->option_menu_lang));
	j = gtk_combo_box_get_active(GTK_COMBO_BOX(prefsui->option_menu_autodetect));
	if (i >= 0 && i < n_languages)
	{
		mozilla_prefs_set_string("intl.charset.default",
					 languages[i].code);
	}
	if (j >= 0 && j < n_encoding_autodetectors)
	{
		if (mozilla_prefs_set_string("intl.charset.detector",
					     encoding_autodetectors[j].name));
	}

	if (prefsui->enabled_list_updated)
	{
		KzDList *dlist = KZ_DLIST(prefsui->lang_dlist);
		gint i, num = kz_dlist_get_n_enabled_items(dlist);
		gchar *langs = strdup("");

		for (i = 0; i < num; i++)
		{
			gchar *lang, *tmp;
			lang = kz_dlist_get_enabled_id(dlist, i);
			tmp = langs;
			if (i == 0)
				langs = g_strconcat(langs, lang, NULL);
			else
				langs = g_strconcat(langs, ",", lang, NULL);
			g_free(tmp);
			g_free(lang);
		}
		mozilla_prefs_set_string("intl.accept_languages", langs);
		g_free(langs);
		prefsui->enabled_list_updated = FALSE;
	}
}


static void
prefs_lang_response (GtkWidget *widget, gint response)
{
	KzPrefsLang *prefsui = g_object_get_data(G_OBJECT(widget), DATA_KEY);

	g_return_if_fail(prefsui);

	switch (response) {
	case GTK_RESPONSE_ACCEPT:
	case GTK_RESPONSE_APPLY:
		prefs_lang_apply(prefsui);
		break;
	case GTK_RESPONSE_REJECT:
		break;
	case KZ_RESPONSE_UI_LEVEL_MEDIUM:
		break;
	case KZ_RESPONSE_UI_LEVEL_EXPERT:
		break;
	case KZ_RESPONSE_UI_LEVEL_CUSTOM:
		break;
	default:
		break;
	}
}


KzPrefsWinPageEntry *
prefs_lang_get_entry (gint idx)
{
	if (idx == 0)
		return &prefs_entry;
	else
		return NULL;
}
