/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAudienceRestrictionCondition.cpp - SAML audience condition implementation

   Scott Cantor
   5/27/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;


SAMLAudienceRestrictionCondition::SAMLAudienceRestrictionCondition(const Iterator<const XMLCh*>& audiences)
{
    RTTI(SAMLAudienceRestrictionCondition);
    while (audiences.hasNext())
        m_audiences.push_back(XML::assign(audiences.next()));
}

SAMLAudienceRestrictionCondition::SAMLAudienceRestrictionCondition(DOMElement* e)
{
    RTTI(SAMLAudienceRestrictionCondition);
    fromDOM(e);
}

SAMLAudienceRestrictionCondition::SAMLAudienceRestrictionCondition(istream& in) : SAMLCondition(in)
{
    RTTI(SAMLAudienceRestrictionCondition);
    fromDOM(m_document->getDocumentElement());
}


SAMLAudienceRestrictionCondition::~SAMLAudienceRestrictionCondition()
{
    if (m_bOwnStrings) {
        for (vector<const XMLCh*>::const_iterator i=m_audiences.begin(); i!=m_audiences.end(); i++) {
            XMLCh* temp=const_cast<XMLCh*>(*i);
            XMLString::release(&temp);
        }
    }
}

void SAMLAudienceRestrictionCondition::ownStrings()
{
    if (!m_bOwnStrings) {
        for (vector<const XMLCh*>::iterator i=m_audiences.begin(); i!=m_audiences.end(); i++)
            (*i)=XML::assign(*i);
        m_bOwnStrings=true;
    }
}

void SAMLAudienceRestrictionCondition::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAML_NS,e->getNamespaceURI()))
            throw MalformedException("SAMLAudienceRestrictionCondition::fromDOM() missing saml namespace on root element");

        if (XMLString::compareString(L(AudienceRestrictionCondition),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if (XMLString::compareString(L(Condition),e->getLocalName()) ||
                !type.get() || XMLString::compareString(XML::SAML_NS,type->getNamespaceURI()) ||
                XMLString::compareString(L(AudienceRestrictionConditionType),type->getLocalName()))
                throw MalformedException("SAMLAudienceRestrictionCondition::fromDOM() requires saml:AudienceRestrictionCondition at root");
        }
    }

    // Iterate over audiences.
    DOMElement* a=XML::getFirstChildElement(e,XML::SAML_NS,L(Audience));
    while (a) {
        if (a->hasChildNodes())
            m_audiences.push_back(a->getFirstChild()->getNodeValue());
        a=XML::getNextSiblingElement(a,XML::SAML_NS,L(Audience));
    }
    checkValidity();
}

void SAMLAudienceRestrictionCondition::setAudiences(const Iterator<const XMLCh*>& audiences)
{
    while (m_audiences.size())
        removeAudience(0);
    while (audiences.hasNext())
        addAudience(audiences.next());
}

void SAMLAudienceRestrictionCondition::addAudience(const XMLCh* audience)
{
    if (XML::isEmpty(audience))
        throw SAMLException("audience cannot be null or empty");
    
    ownStrings();
    m_audiences.push_back(XML::assign(audience));
    setDirty();
}

void SAMLAudienceRestrictionCondition::removeAudience(unsigned long index)
{
    if (m_bOwnStrings) {
        XMLCh* ch=const_cast<XMLCh*>(m_audiences[index]);
        XMLString::release(&ch);
    }
    m_audiences.erase(m_audiences.begin()+index);
    ownStrings();
    setDirty();
}

DOMElement* SAMLAudienceRestrictionCondition::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* c = doc->createElementNS(XML::SAML_NS, L(AudienceRestrictionCondition));
    if (xmlns)
        c->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return c;
}

DOMNode* SAMLAudienceRestrictionCondition::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* c=static_cast<DOMElement*>(m_root);
    doc=m_root->getOwnerDocument();
    
    if (m_bDirty) {
        for (vector<const XMLCh*>::const_iterator i=m_audiences.begin(); i!=m_audiences.end(); i++)
            c->appendChild(doc->createElementNS(XML::SAML_NS,L(Audience)))->appendChild(doc->createTextNode(*i));
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(c,XML::SAML_NS);
    }

    return m_root;
}

void SAMLAudienceRestrictionCondition::checkValidity() const
{
    if (m_audiences.size() == 0)
        throw MalformedException("AudienceRestrictionCondition is invalid, requires at least one audience");
}

SAMLObject* SAMLAudienceRestrictionCondition::clone() const
{
    return new SAMLAudienceRestrictionCondition(m_audiences);
}

bool SAMLAudienceRestrictionCondition::eval(const Iterator<const XMLCh*>& audiences) const
{
    if (m_audiences.empty())
        return true;
    else if (audiences.size()==0)
        return false;

    audiences.reset();
    while (audiences.hasNext())
    {
        const XMLCh* current=audiences.next();
        for (vector<const XMLCh*>::const_iterator i=m_audiences.begin(); i!=m_audiences.end(); i++)
        {
            if (!XMLString::compareString(current,*i))
                return true;
        }
    }
    return false;
}

#ifdef HAVE_GOOD_STL
bool SAMLAudienceRestrictionCondition::eval(const Iterator<xstring>& audiences) const
{
    if (m_audiences.empty())
        return true;
    else if (audiences.size()==0)
        return false;

    audiences.reset();
    while (audiences.hasNext())
    {
        const xstring& current=audiences.next();
        for (vector<const XMLCh*>::const_iterator i=m_audiences.begin(); i!=m_audiences.end(); i++)
        {
            if (!XMLString::compareString(current.c_str(),*i))
                return true;
        }
    }
    return false;
}
#endif