/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLAuthenticationStatement.cpp - SAML authentication statement implementation

   Scott Cantor
   5/27/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;

SAMLAuthenticationStatement::SAMLAuthenticationStatement(
    SAMLSubject* subject,
    const XMLCh* authMethod,
    const SAMLDateTime* authInstant,
    const XMLCh* subjectIP,
    const XMLCh* subjectDNS,
    const Iterator<SAMLAuthorityBinding*>& bindings
    ) : SAMLSubjectStatement(subject), m_authMethod(XML::assign(authMethod)),
        m_subjectIP(XML::assign(subjectIP)), m_subjectDNS(XML::assign(subjectDNS))
{
    RTTI(SAMLAuthenticationStatement);
    if (authInstant) {
        m_authInstant=new SAMLDateTime(*authInstant);
        m_authInstant->parseDateTime();
    }
    while (bindings.hasNext())
        m_bindings.push_back(static_cast<SAMLAuthorityBinding*>(bindings.next()->setParent(this)));
}

SAMLAuthenticationStatement::SAMLAuthenticationStatement(DOMElement* e)
    : SAMLSubjectStatement(e), m_subjectIP(NULL), m_subjectDNS(NULL), m_authMethod(NULL), m_authInstant(NULL)
{
    RTTI(SAMLAuthenticationStatement);
    fromDOM(e);
}

SAMLAuthenticationStatement::SAMLAuthenticationStatement(istream& in)
    : SAMLSubjectStatement(in), m_subjectIP(NULL), m_subjectDNS(NULL), m_authMethod(NULL), m_authInstant(NULL)
{
    RTTI(SAMLAuthenticationStatement);
    fromDOM(m_document->getDocumentElement());
}

SAMLAuthenticationStatement::~SAMLAuthenticationStatement()
{
    if (m_bOwnStrings) {
        XMLString::release(&m_authMethod);
        XMLString::release(&m_subjectIP);
        XMLString::release(&m_subjectDNS);
    }
    delete m_authInstant;
    for (vector<SAMLAuthorityBinding*>::const_iterator i=m_bindings.begin(); i!=m_bindings.end(); i++)
        delete (*i);
}

void SAMLAuthenticationStatement::ownStrings()
{
    if (!m_bOwnStrings) {
        m_authMethod=XML::assign(m_authMethod);
        m_subjectIP=XML::assign(m_subjectIP);
        m_subjectDNS=XML::assign(m_subjectDNS);
        m_bOwnStrings=true;
    }
}

void SAMLAuthenticationStatement::fromDOM(DOMElement* e)
{
    if (SAMLConfig::getConfig().strict_dom_checking) {
        if (XMLString::compareString(XML::SAML_NS,e->getNamespaceURI()))
            throw MalformedException("SAMLAuthenticationStatement::fromDOM() missing saml namespace on root element");

        if (XMLString::compareString(L(AuthenticationStatement),e->getLocalName())) {
            auto_ptr<saml::QName> type(saml::QName::getQNameAttribute(e,XML::XSI_NS,L(type)));
            if ((XMLString::compareString(L(Statement),e->getLocalName()) && XMLString::compareString(L(SubjectStatement),e->getLocalName())) ||
                    !type.get() || XMLString::compareString(XML::SAML_NS,type->getNamespaceURI()) ||
                    XMLString::compareString(L(AuthenticationStatementType),type->getLocalName()))
                throw MalformedException("SAMLAuthenticationStatement::fromDOM() requires saml:AuthenticationStatement at root");
        }
    }

    m_authMethod=const_cast<XMLCh*>(e->getAttributeNS(NULL,L(AuthenticationMethod)));
    m_authInstant=new SAMLDateTime(e->getAttributeNS(NULL,L(AuthenticationInstant)));
    m_authInstant->parseDateTime();

    DOMElement* n=XML::getNextSiblingElement(m_subject->toDOM());
    if (n && XML::isElementNamed(n,XML::SAML_NS,L(SubjectLocality))) {
        m_subjectIP=const_cast<XMLCh*>(n->getAttributeNS(NULL,L(IPAddress)));
        m_subjectDNS=const_cast<XMLCh*>(n->getAttributeNS(NULL,L(DNSAddress)));
    }

    n=XML::getFirstChildElement(e,XML::SAML_NS,L(AuthorityBinding));
    while (n) {
        SAMLAuthorityBinding* ab=new SAMLAuthorityBinding(n);
        m_bindings.push_back(static_cast<SAMLAuthorityBinding*>(ab->setParent(this)));
        n=XML::getNextSiblingElement(n,XML::SAML_NS,L(AuthorityBinding));
    }
    
    checkValidity();
}

void SAMLAuthenticationStatement::setAuthMethod(const XMLCh* authMethod)
{
    if (XML::isEmpty(authMethod))
        throw SAMLException("authMethod cannot be null or empty");
        
    if (m_bOwnStrings)
        XMLString::release(&m_authMethod);
    else {
        m_authMethod=NULL;
        ownStrings();
    }
    m_authMethod=XML::assign(authMethod);
    setDirty();
}

void SAMLAuthenticationStatement::setAuthInstant(const SAMLDateTime* authInstant)
{
    delete m_authInstant;
    m_authInstant=NULL;
    if (authInstant) {
        m_authInstant=new SAMLDateTime(*authInstant);
        m_authInstant->parseDateTime();
    }
    ownStrings();
    setDirty();
}

void SAMLAuthenticationStatement::setSubjectIP(const XMLCh* subjectIP)
{
    if (m_bOwnStrings)
        XMLString::release(&m_subjectIP);
    else {
        m_subjectIP=NULL;
        ownStrings();
    }
    m_subjectIP=XML::assign(subjectIP);
    setDirty();
}

void SAMLAuthenticationStatement::setSubjectDNS(const XMLCh* subjectDNS)
{
    if (m_bOwnStrings)
        XMLString::release(&m_subjectDNS);
    else {
        m_subjectDNS=NULL;
        ownStrings();
    }
    m_subjectDNS=XML::assign(subjectDNS);
    setDirty();
}

void SAMLAuthenticationStatement::setBindings(const Iterator<SAMLAuthorityBinding*>& bindings)
{
    while (m_bindings.size())
        removeBinding(0);
    while (bindings.hasNext())
        addBinding(bindings.next());
}

void SAMLAuthenticationStatement::addBinding(SAMLAuthorityBinding* binding)
{
    if (binding) {
        binding->setParent(this);
        m_bindings.push_back(binding);
        ownStrings();
        setDirty();
    }
    else
        throw SAMLException("binding cannot be null");
}

void SAMLAuthenticationStatement::removeBinding(unsigned long index)
{
    SAMLAuthorityBinding* kill=m_bindings[index];
    m_bindings.erase(m_bindings.begin()+index);
    delete kill;
    ownStrings();
    setDirty();
}

DOMElement* SAMLAuthenticationStatement::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* s = doc->createElementNS(XML::SAML_NS, L(AuthenticationStatement));
    if (xmlns)
        s->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAML_NS);
    return s;
}

DOMNode* SAMLAuthenticationStatement::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLSubjectStatement::toDOM(doc,xmlns);
    DOMElement* s=static_cast<DOMElement*>(m_root);

    if (m_bDirty) {
        s->setAttributeNS(NULL,L(AuthenticationMethod),m_authMethod);
        s->setAttributeNS(NULL,L(AuthenticationInstant),m_authInstant->getRawData());
    
        if (!XML::isEmpty(m_subjectIP) || !XML::isEmpty(m_subjectDNS)) {
            DOMElement* loc=s->getOwnerDocument()->createElementNS(XML::SAML_NS,L(SubjectLocality));
            if (!XML::isEmpty(m_subjectIP))
                loc->setAttributeNS(NULL,L(IPAddress),m_subjectIP);
            if (!XML::isEmpty(m_subjectDNS))
                loc->setAttributeNS(NULL,L(DNSAddress),m_subjectDNS);
            s->appendChild(loc);
        }
    
        for (vector<SAMLAuthorityBinding*>::const_iterator i=m_bindings.begin(); i!=m_bindings.end(); i++)
            s->appendChild((*i)->toDOM(s->getOwnerDocument(),false));
    
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(s,XML::SAML_NS);
    }

    return s;
}

void SAMLAuthenticationStatement::checkValidity() const
{
    SAMLSubjectStatement::checkValidity();
    if (XML::isEmpty(m_authMethod) || !m_authInstant)
        throw MalformedException("AuthenticationStatement is invalid, requires AuthenticationMethod and AuthenticationInstant");
}

SAMLObject* SAMLAuthenticationStatement::clone() const
{
    return new SAMLAuthenticationStatement(
        static_cast<SAMLSubject*>(m_subject->clone()),m_authMethod,m_authInstant,m_subjectIP,m_subjectDNS,getBindings().clone()
        );
}
