#include <stdio.h>
#include <string.h>
#include <stddef.h>
#include <sys/types.h>
#include <freeswan.h>

#include "constants.h"
#include "defs.h"
#include "log.h"
#include "libsha2/sha2.h"
#include "alg_info.h"
#include "ike_alg.h"

static void
sha256_hash_final(u_char *hash, sha256_context *ctx)
{
    sha256_final(ctx);
    memcpy(hash, ctx->sha_out, SHA2_256_DIGEST_SIZE);
}

static void
sha384_hash_final(u_char *hash, sha512_context *ctx)
{
    sha512_final(ctx);
    memcpy(hash, ctx->sha_out, SHA2_384_DIGEST_SIZE);
}

static void
sha512_hash_final(u_char *hash, sha512_context *ctx)
{
    sha512_final(ctx);
    memcpy(hash, ctx->sha_out, SHA2_512_DIGEST_SIZE);
}

/* SHA-256 hash test vectors
 * from "The Secure Hash Algorithm Validation System (SHAVS)"
 * July 22, 2004, Lawrence E. Bassham III, NIST
 */

static const u_char sha256_short2_msg[] = {
    0x19
};

static const u_char sha256_short2_msg_digest[] = {
    0x68, 0xaa, 0x2e, 0x2e, 0xe5, 0xdf, 0xf9, 0x6e,
    0x33, 0x55, 0xe6, 0xc7, 0xee, 0x37, 0x3e, 0x3d,
    0x6a, 0x4e, 0x17, 0xf7, 0x5f, 0x95, 0x18, 0xd8,
    0x43, 0x70, 0x9c, 0x0c, 0x9b, 0xc3, 0xe3, 0xd4
};

static const u_char sha256_short4_msg[] = {
    0xe3, 0xd7, 0x25, 0x70, 0xdc, 0xdd, 0x78, 0x7c,
    0xe3, 0x88, 0x7a, 0xb2, 0xcd, 0x68, 0x46, 0x52
};

static const u_char sha256_short4_msg_digest[] = {
    0x17, 0x5e, 0xe6, 0x9b, 0x02, 0xba, 0x9b, 0x58,
    0xe2, 0xb0, 0xa5, 0xfd, 0x13, 0x81, 0x9c, 0xea,
    0x57, 0x3f, 0x39, 0x40, 0xa9, 0x4f, 0x82, 0x51,
    0x28, 0xcf, 0x42, 0x09, 0xbe, 0xab, 0xb4, 0xe8
};

static const u_char sha256_long2_msg[] = {
    0x83, 0x26, 0x75, 0x4e, 0x22, 0x77, 0x37, 0x2f,
    0x4f, 0xc1, 0x2b, 0x20, 0x52, 0x7a, 0xfe, 0xf0,
    0x4d, 0x8a, 0x05, 0x69, 0x71, 0xb1, 0x1a, 0xd5,
    0x71, 0x23, 0xa7, 0xc1, 0x37, 0x76, 0x00, 0x00,
    0xd7, 0xbe, 0xf6, 0xf3, 0xc1, 0xf7, 0xa9, 0x08,
    0x3a, 0xa3, 0x9d, 0x81, 0x0d, 0xb3, 0x10, 0x77,
    0x7d, 0xab, 0x8b, 0x1e, 0x7f, 0x02, 0xb8, 0x4a,
    0x26, 0xc7, 0x73, 0x32, 0x5f, 0x8b, 0x23, 0x74,
    0xde, 0x7a, 0x4b, 0x5a, 0x58, 0xcb, 0x5c, 0x5c,
    0xf3, 0x5b, 0xce, 0xe6, 0xfb, 0x94, 0x6e, 0x5b,
    0xd6, 0x94, 0xfa, 0x59, 0x3a, 0x8b, 0xeb, 0x3f,
    0x9d, 0x65, 0x92, 0xec, 0xed, 0xaa, 0x66, 0xca,
    0x82, 0xa2, 0x9d, 0x0c, 0x51, 0xbc, 0xf9, 0x33,
    0x62, 0x30, 0xe5, 0xd7, 0x84, 0xe4, 0xc0, 0xa4,
    0x3f, 0x8d, 0x79, 0xa3, 0x0a, 0x16, 0x5c, 0xba,
    0xbe, 0x45, 0x2b, 0x77, 0x4b, 0x9c, 0x71, 0x09,
    0xa9, 0x7d, 0x13, 0x8f, 0x12, 0x92, 0x28, 0x96,
    0x6f, 0x6c, 0x0a, 0xdc, 0x10, 0x6a, 0xad, 0x5a,
    0x9f, 0xdd, 0x30, 0x82, 0x57, 0x69, 0xb2, 0xc6,
    0x71, 0xaf, 0x67, 0x59, 0xdf, 0x28, 0xeb, 0x39,
    0x3d, 0x54, 0xd6
};

static const u_char sha256_long2_msg_digest[] = {
    0x97, 0xdb, 0xca, 0x7d, 0xf4, 0x6d, 0x62, 0xc8,
    0xa4, 0x22, 0xc9, 0x41, 0xdd, 0x7e, 0x83, 0x5b,
    0x8a, 0xd3, 0x36, 0x17, 0x63, 0xf7, 0xe9, 0xb2,
    0xd9, 0x5f, 0x4f, 0x0d, 0xa6, 0xe1, 0xcc, 0xbc
};

static const hash_testvector_t sha256_hash_testvectors[] = {
    { sizeof(sha256_short2_msg), sha256_short2_msg, sha256_short2_msg_digest },
    { sizeof(sha256_short4_msg), sha256_short4_msg, sha256_short4_msg_digest },
    { sizeof(sha256_long2_msg),  sha256_long2_msg,  sha256_long2_msg_digest  },
    { 0, NULL, NULL }
};

/* SHA-384 hash test vectors
 * from "The Secure Hash Algorithm Validation System (SHAVS)"
 * July 22, 2004, Lawrence E. Bassham III, NIST
 */

static const u_char sha384_short2_msg[] = {
    0xb9
};

static const u_char sha384_short2_msg_digest[] = {
    0xbc, 0x80, 0x89, 0xa1, 0x90, 0x07, 0xc0, 0xb1,
    0x41, 0x95, 0xf4, 0xec, 0xc7, 0x40, 0x94, 0xfe,
    0xc6, 0x4f, 0x01, 0xf9, 0x09, 0x29, 0x28, 0x2c,
    0x2f, 0xb3, 0x92, 0x88, 0x15, 0x78, 0x20, 0x8a,
    0xd4, 0x66, 0x82, 0x8b, 0x1c, 0x6c, 0x28, 0x3d,
    0x27, 0x22, 0xcf, 0x0a, 0xd1, 0xab, 0x69, 0x38
};

static const u_char sha384_short4_msg[] = {
     0xa4, 0x1c, 0x49, 0x77, 0x79, 0xc0, 0x37, 0x5f,
     0xf1, 0x0a, 0x7f, 0x4e, 0x08, 0x59, 0x17, 0x39
};

static const u_char sha384_short4_msg_digest[] = {
    0xc9, 0xa6, 0x84, 0x43, 0xa0, 0x05, 0x81, 0x22,
    0x56, 0xb8, 0xec, 0x76, 0xb0, 0x05, 0x16, 0xf0,
    0xdb, 0xb7, 0x4f, 0xab, 0x26, 0xd6, 0x65, 0x91,
    0x3f, 0x19, 0x4b, 0x6f, 0xfb, 0x0e, 0x91, 0xea,
    0x99, 0x67, 0x56, 0x6b, 0x58, 0x10, 0x9c, 0xbc,
    0x67, 0x5c, 0xc2, 0x08, 0xe4, 0xc8, 0x23, 0xf7
};

static const u_char sha384_long2_msg[] = {
    0x39, 0x96, 0x69, 0xe2, 0x8f, 0x6b, 0x9c, 0x6d,
    0xbc, 0xbb, 0x69, 0x12, 0xec, 0x10, 0xff, 0xcf,
    0x74, 0x79, 0x03, 0x49, 0xb7, 0xdc, 0x8f, 0xbe,
    0x4a, 0x8e, 0x7b, 0x3b, 0x56, 0x21, 0xdb, 0x0f,
    0x3e, 0x7d, 0xc8, 0x7f, 0x82, 0x32, 0x64, 0xbb,
    0xe4, 0x0d, 0x18, 0x11, 0xc9, 0xea, 0x20, 0x61,
    0xe1, 0xc8, 0x4a, 0xd1, 0x0a, 0x23, 0xfa, 0xc1,
    0x72, 0x7e, 0x72, 0x02, 0xfc, 0x3f, 0x50, 0x42,
    0xe6, 0xbf, 0x58, 0xcb, 0xa8, 0xa2, 0x74, 0x6e,
    0x1f, 0x64, 0xf9, 0xb9, 0xea, 0x35, 0x2c, 0x71,
    0x15, 0x07, 0x05, 0x3c, 0xf4, 0xe5, 0x33, 0x9d,
    0x52, 0x86, 0x5f, 0x25, 0xcc, 0x22, 0xb5, 0xe8,
    0x77, 0x84, 0xa1, 0x2f, 0xc9, 0x61, 0xd6, 0x6c,
    0xb6, 0xe8, 0x95, 0x73, 0x19, 0x9a, 0x2c, 0xe6,
    0x56, 0x5c, 0xbd, 0xf1, 0x3d, 0xca, 0x40, 0x38,
    0x32, 0xcf, 0xcb, 0x0e, 0x8b, 0x72, 0x11, 0xe8,
    0x3a, 0xf3, 0x2a, 0x11, 0xac, 0x17, 0x92, 0x9f,
    0xf1, 0xc0, 0x73, 0xa5, 0x1c, 0xc0, 0x27, 0xaa,
    0xed, 0xef, 0xf8, 0x5a, 0xad, 0x7c, 0x2b, 0x7c,
    0x5a, 0x80, 0x3e, 0x24, 0x04, 0xd9, 0x6d, 0x2a,
    0x77, 0x35, 0x7b, 0xda, 0x1a, 0x6d, 0xae, 0xed,
    0x17, 0x15, 0x1c, 0xb9, 0xbc, 0x51, 0x25, 0xa4,
    0x22, 0xe9, 0x41, 0xde, 0x0c, 0xa0, 0xfc, 0x50,
    0x11, 0xc2, 0x3e, 0xcf, 0xfe, 0xfd, 0xd0, 0x96,
    0x76, 0x71, 0x1c, 0xf3, 0xdb, 0x0a, 0x34, 0x40,
    0x72, 0x0e ,0x16, 0x15, 0xc1, 0xf2, 0x2f, 0xbc,
    0x3c, 0x72, 0x1d, 0xe5, 0x21, 0xe1, 0xb9, 0x9b,
    0xa1, 0xbd, 0x55, 0x77, 0x40, 0x86, 0x42, 0x14,
    0x7e, 0xd0, 0x96
};

static const u_char sha384_long2_msg_digest[] = {
    0x4f, 0x44, 0x0d, 0xb1, 0xe6, 0xed, 0xd2, 0x89,
    0x9f, 0xa3, 0x35, 0xf0, 0x95, 0x15, 0xaa, 0x02,
    0x5e, 0xe1, 0x77, 0xa7, 0x9f, 0x4b, 0x4a, 0xaf,
    0x38, 0xe4, 0x2b, 0x5c, 0x4d, 0xe6, 0x60, 0xf5,
    0xde, 0x8f, 0xb2, 0xa5, 0xb2, 0xfb, 0xd2, 0xa3,
    0xcb, 0xff, 0xd2, 0x0c, 0xff, 0x12, 0x88, 0xc0
};

static const hash_testvector_t sha384_hash_testvectors[] = {
    { sizeof(sha384_short2_msg), sha384_short2_msg, sha384_short2_msg_digest },
    { sizeof(sha384_short4_msg), sha384_short4_msg, sha384_short4_msg_digest },
    { sizeof(sha384_long2_msg),  sha384_long2_msg,  sha384_long2_msg_digest  },
    { 0, NULL, NULL }
};

/* SHA-512 hash test vectors
 * from "The Secure Hash Algorithm Validation System (SHAVS)"
 * July 22, 2004, Lawrence E. Bassham III, NIST
 */

static const u_char sha512_short2_msg[] = {
    0xd0
};

static const u_char sha512_short2_msg_digest[] = {
    0x99, 0x92, 0x20, 0x29, 0x38, 0xe8, 0x82, 0xe7,
    0x3e, 0x20, 0xf6, 0xb6, 0x9e, 0x68, 0xa0, 0xa7,
    0x14, 0x90, 0x90, 0x42, 0x3d, 0x93, 0xc8, 0x1b,
    0xab, 0x3f, 0x21, 0x67, 0x8d, 0x4a, 0xce, 0xee,
    0xe5, 0x0e, 0x4e, 0x8c, 0xaf, 0xad, 0xa4, 0xc8,
    0x5a, 0x54, 0xea, 0x83, 0x06, 0x82, 0x6c, 0x4a,
    0xd6, 0xe7, 0x4c, 0xec, 0xe9, 0x63, 0x1b, 0xfa,
    0x8a, 0x54, 0x9b, 0x4a, 0xb3, 0xfb, 0xba, 0x15
};

static const u_char sha512_short4_msg[] = {
    0x8d, 0x4e, 0x3c, 0x0e, 0x38, 0x89, 0x19, 0x14,
    0x91, 0x81, 0x6e, 0x9d, 0x98, 0xbf, 0xf0, 0xa0
};

static const u_char sha512_short4_msg_digest[] = {
    0xcb, 0x0b, 0x67, 0xa4, 0xb8, 0x71, 0x2c, 0xd7,
    0x3c, 0x9a, 0xab, 0xc0, 0xb1, 0x99, 0xe9, 0x26,
    0x9b, 0x20, 0x84, 0x4a, 0xfb, 0x75, 0xac, 0xbd,
    0xd1, 0xc1, 0x53, 0xc9, 0x82, 0x89, 0x24, 0xc3,
    0xdd, 0xed, 0xaa, 0xfe, 0x66, 0x9c, 0x5f, 0xdd,
    0x0b, 0xc6, 0x6f, 0x63, 0x0f, 0x67, 0x73, 0x98,
    0x82, 0x13, 0xeb, 0x1b, 0x16, 0xf5, 0x17, 0xad,
    0x0d, 0xe4, 0xb2, 0xf0, 0xc9, 0x5c, 0x90, 0xf8
};

static const u_char sha512_long2_msg[] = {
    0xa5, 0x5f, 0x20, 0xc4, 0x11, 0xaa, 0xd1, 0x32,
    0x80, 0x7a, 0x50, 0x2d, 0x65, 0x82, 0x4e, 0x31,
    0xa2, 0x30, 0x54, 0x32, 0xaa, 0x3d, 0x06, 0xd3,
    0xe2, 0x82, 0xa8, 0xd8, 0x4e, 0x0d, 0xe1, 0xde,
    0x69, 0x74, 0xbf, 0x49, 0x54, 0x69, 0xfc, 0x7f,
    0x33, 0x8f, 0x80, 0x54, 0xd5, 0x8c, 0x26, 0xc4,
    0x93, 0x60, 0xc3, 0xe8, 0x7a, 0xf5, 0x65, 0x23,
    0xac, 0xf6, 0xd8, 0x9d, 0x03, 0xe5, 0x6f, 0xf2,
    0xf8, 0x68, 0x00, 0x2b, 0xc3, 0xe4, 0x31, 0xed,
    0xc4, 0x4d, 0xf2, 0xf0, 0x22, 0x3d, 0x4b, 0xb3,
    0xb2, 0x43, 0x58, 0x6e, 0x1a, 0x7d, 0x92, 0x49,
    0x36, 0x69, 0x4f, 0xcb, 0xba, 0xf8, 0x8d, 0x95,
    0x19, 0xe4, 0xeb, 0x50, 0xa6, 0x44, 0xf8, 0xe4,
    0xf9, 0x5e, 0xb0, 0xea, 0x95, 0xbc, 0x44, 0x65,
    0xc8, 0x82, 0x1a, 0xac, 0xd2, 0xfe, 0x15, 0xab,
    0x49, 0x81, 0x16, 0x4b, 0xbb, 0x6d, 0xc3, 0x2f,
    0x96, 0x90, 0x87, 0xa1, 0x45, 0xb0, 0xd9, 0xcc,
    0x9c, 0x67, 0xc2, 0x2b, 0x76, 0x32, 0x99, 0x41,
    0x9c, 0xc4, 0x12, 0x8b, 0xe9, 0xa0, 0x77, 0xb3,
    0xac, 0xe6, 0x34, 0x06, 0x4e, 0x6d, 0x99, 0x28,
    0x35, 0x13, 0xdc, 0x06, 0xe7, 0x51, 0x5d, 0x0d,
    0x73, 0x13, 0x2e, 0x9a, 0x0d, 0xc6, 0xd3, 0xb1,
    0xf8, 0xb2, 0x46, 0xf1, 0xa9, 0x8a, 0x3f, 0xc7,
    0x29, 0x41, 0xb1, 0xe3, 0xbb, 0x20, 0x98, 0xe8,
    0xbf, 0x16, 0xf2, 0x68, 0xd6, 0x4f, 0x0b, 0x0f,
    0x47, 0x07, 0xfe, 0x1e, 0xa1, 0xa1, 0x79, 0x1b,
    0xa2, 0xf3, 0xc0, 0xc7, 0x58, 0xe5, 0xf5, 0x51,
    0x86, 0x3a, 0x96, 0xc9, 0x49, 0xad, 0x47, 0xd7,
    0xfb, 0x40, 0xd2
};

static const u_char sha512_long2_msg_digest[] = {
    0xc6, 0x65, 0xbe, 0xfb, 0x36, 0xda, 0x18, 0x9d,
    0x78, 0x82, 0x2d, 0x10, 0x52, 0x8c, 0xbf, 0x3b,
    0x12, 0xb3, 0xee, 0xf7, 0x26, 0x03, 0x99, 0x09,
    0xc1, 0xa1, 0x6a, 0x27, 0x0d, 0x48, 0x71, 0x93,
    0x77, 0x96, 0x6b, 0x95, 0x7a, 0x87, 0x8e, 0x72,
    0x05, 0x84, 0x77, 0x9a, 0x62, 0x82, 0x5c, 0x18,
    0xda, 0x26, 0x41, 0x5e, 0x49, 0xa7, 0x17, 0x6a,
    0x89, 0x4e, 0x75, 0x10, 0xfd, 0x14, 0x51, 0xf5
};

static const hash_testvector_t sha512_hash_testvectors[] = {
    { sizeof(sha512_short2_msg), sha512_short2_msg, sha512_short2_msg_digest },
    { sizeof(sha512_short4_msg), sha512_short4_msg, sha512_short4_msg_digest },
    { sizeof(sha512_long2_msg),  sha512_long2_msg,  sha512_long2_msg_digest  },
    { 0, NULL, NULL }
};

/* SHA-256, SHA-384, and SHA-512 hmac test vectors
 * from RFC 4231 "Identifiers and Test Vectors for HMAC-SHA-224,
 * HMAC-SHA-256, HMAC-SHA-384, and HMAC-SHA-512"
 * December 2005, M. Nystrom, RSA Security
 */

static const u_char sha2_hmac1_key[] = {
    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
    0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b, 0x0b,
    0x0b, 0x0b, 0x0b, 0x0b
};

static const u_char sha2_hmac1_msg[] = {
    0x48, 0x69, 0x20, 0x54, 0x68, 0x65, 0x72, 0x65
};

static const u_char sha2_hmac1_256[] = {
    0xb0, 0x34, 0x4c, 0x61, 0xd8, 0xdb, 0x38, 0x53,
    0x5c, 0xa8, 0xaf, 0xce, 0xaf, 0x0b, 0xf1, 0x2b,
    0x88, 0x1d, 0xc2, 0x00, 0xc9, 0x83, 0x3d, 0xa7,
    0x26, 0xe9, 0x37, 0x6c, 0x2e, 0x32, 0xcf, 0xf7
};

static const u_char sha2_hmac1_384[] = {
    0xaf, 0xd0, 0x39, 0x44, 0xd8, 0x48, 0x95, 0x62,
    0x6b, 0x08, 0x25, 0xf4, 0xab ,0x46, 0x90, 0x7f,
    0x15, 0xf9, 0xda, 0xdb, 0xe4, 0x10, 0x1e, 0xc6,
    0x82, 0xaa, 0x03, 0x4c, 0x7c, 0xeb, 0xc5, 0x9c,
    0xfa, 0xea, 0x9e, 0xa9, 0x07, 0x6e, 0xde, 0x7f,
    0x4a, 0xf1, 0x52, 0xe8, 0xb2, 0xfa, 0x9c, 0xb6
};

static const u_char sha2_hmac1_512[] = {
    0x87, 0xaa, 0x7c, 0xde, 0xa5, 0xef, 0x61, 0x9d,
    0x4f, 0xf0, 0xb4, 0x24, 0x1a, 0x1d, 0x6c, 0xb0,
    0x23, 0x79, 0xf4, 0xe2, 0xce, 0x4e, 0xc2, 0x78,
    0x7a, 0xd0, 0xb3, 0x05, 0x45, 0xe1, 0x7c, 0xde,
    0xda, 0xa8, 0x33, 0xb7, 0xd6, 0xb8, 0xa7, 0x02,
    0x03, 0x8b, 0x27, 0x4e, 0xae, 0xa3, 0xf4, 0xe4,
    0xbe, 0x9d, 0x91, 0x4e, 0xeb, 0x61, 0xf1, 0x70,
    0x2e, 0x69, 0x6c, 0x20, 0x3a, 0x12, 0x68, 0x54
};

static const u_char sha2_hmac2_key[] = {
    0x4a, 0x65, 0x66, 0x65
};

static const u_char sha2_hmac2_msg[] = {
    0x77, 0x68, 0x61, 0x74, 0x20, 0x64, 0x6f, 0x20,
    0x79, 0x61, 0x20, 0x77, 0x61, 0x6e, 0x74, 0x20,
    0x66, 0x6f, 0x72, 0x20, 0x6e, 0x6f, 0x74, 0x68,
    0x69, 0x6e, 0x67, 0x3f
};

static const u_char sha2_hmac2_256[] = {
    0x5b, 0xdc, 0xc1, 0x46, 0xbf, 0x60, 0x75, 0x4e,
    0x6a, 0x04, 0x24, 0x26, 0x08, 0x95, 0x75, 0xc7,
    0x5a, 0x00, 0x3f, 0x08, 0x9d, 0x27, 0x39, 0x83,
    0x9d, 0xec, 0x58, 0xb9, 0x64, 0xec, 0x38, 0x43
};

static const u_char sha2_hmac2_384[] = {
    0xaf, 0x45, 0xd2, 0xe3, 0x76, 0x48, 0x40, 0x31,
    0x61, 0x7f, 0x78, 0xd2, 0xb5, 0x8a, 0x6b, 0x1b,
    0x9c, 0x7e, 0xf4, 0x64, 0xf5, 0xa0, 0x1b, 0x47,
    0xe4, 0x2e, 0xc3, 0x73, 0x63, 0x22, 0x44, 0x5e,
    0x8e, 0x22, 0x40, 0xca, 0x5e, 0x69, 0xe2, 0xc7,
    0x8b, 0x32, 0x39, 0xec, 0xfa, 0xb2, 0x16, 0x49
};

static const u_char sha2_hmac2_512[] = {
    0x16, 0x4b, 0x7a, 0x7b, 0xfc, 0xf8, 0x19, 0xe2,
    0xe3, 0x95, 0xfb, 0xe7, 0x3b, 0x56, 0xe0, 0xa3,
    0x87, 0xbd, 0x64, 0x22, 0x2e, 0x83, 0x1f, 0xd6,
    0x10, 0x27, 0x0c, 0xd7, 0xea, 0x25, 0x05, 0x54,
    0x97, 0x58, 0xbf, 0x75, 0xc0, 0x5a, 0x99, 0x4a,
    0x6d, 0x03, 0x4f, 0x65, 0xf8, 0xf0, 0xe6, 0xfd,
    0xca, 0xea, 0xb1, 0xa3, 0x4d, 0x4a, 0x6b, 0x4b,
    0x63, 0x6e, 0x07, 0x0a, 0x38, 0xbc, 0xe7, 0x37
};

static const u_char sha2_hmac3_key[] = {
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa
};

static const u_char sha2_hmac3_msg[] = {
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd, 0xdd,
    0xdd, 0xdd
};

static const u_char sha2_hmac3_256[] = {
    0x77, 0x3e, 0xa9, 0x1e, 0x36, 0x80, 0x0e, 0x46,
    0x85, 0x4d, 0xb8, 0xeb, 0xd0, 0x91, 0x81, 0xa7,
    0x29, 0x59, 0x09, 0x8b, 0x3e, 0xf8, 0xc1, 0x22,
    0xd9, 0x63, 0x55, 0x14, 0xce, 0xd5, 0x65, 0xfe
};

static const u_char sha2_hmac3_384[] = {
    0x88, 0x06, 0x26, 0x08, 0xd3, 0xe6, 0xad, 0x8a,
    0x0a, 0xa2, 0xac, 0xe0, 0x14, 0xc8, 0xa8, 0x6f,
    0x0a, 0xa6, 0x35, 0xd9, 0x47, 0xac, 0x9f, 0xeb,
    0xe8, 0x3e, 0xf4, 0xe5, 0x59, 0x66, 0x14, 0x4b,
    0x2a, 0x5a, 0xb3, 0x9d, 0xc1, 0x38, 0x14, 0xb9,
    0x4e, 0x3a, 0xb6, 0xe1, 0x01, 0xa3, 0x4f, 0x27
};

static const u_char sha2_hmac3_512[] = {
    0xfa, 0x73, 0xb0, 0x08, 0x9d, 0x56, 0xa2, 0x84,
    0xef, 0xb0, 0xf0, 0x75, 0x6c, 0x89, 0x0b, 0xe9,
    0xb1, 0xb5, 0xdb, 0xdd, 0x8e, 0xe8, 0x1a, 0x36,
    0x55, 0xf8, 0x3e, 0x33, 0xb2, 0x27, 0x9d, 0x39,
    0xbf, 0x3e, 0x84, 0x82, 0x79, 0xa7, 0x22, 0xc8,
    0x06, 0xb4, 0x85, 0xa4, 0x7e, 0x67, 0xc8, 0x07,
    0xb9, 0x46, 0xa3, 0x37, 0xbe, 0xe8, 0x94, 0x26,
    0x74, 0x27, 0x88, 0x59, 0xe1, 0x32, 0x92, 0xfb
};

static const u_char sha2_hmac4_key[] = {
    0x01, 0x02, 0x03, 0x04, 0x05, 0x06, 0x07, 0x08,
    0x09, 0x0a, 0x0b, 0x0c, 0x0d, 0x0e, 0x0f, 0x10,
    0x11, 0x12, 0x13, 0x14, 0x15, 0x16, 0x17, 0x18,
    0x19
};

static const u_char sha2_hmac4_msg[] = {
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd, 0xcd,
    0xcd, 0xcd 
};

static const u_char sha2_hmac4_256[] = {
    0x82, 0x55, 0x8a, 0x38, 0x9a, 0x44, 0x3c, 0x0e,
    0xa4, 0xcc, 0x81, 0x98, 0x99, 0xf2, 0x08, 0x3a,
    0x85, 0xf0, 0xfa, 0xa3, 0xe5, 0x78, 0xf8, 0x07,
    0x7a, 0x2e, 0x3f, 0xf4, 0x67, 0x29, 0x66, 0x5b
};

static const u_char sha2_hmac4_384[] = {
    0x3e, 0x8a, 0x69, 0xb7, 0x78, 0x3c, 0x25, 0x85,
    0x19, 0x33, 0xab, 0x62, 0x90, 0xaf, 0x6c, 0xa7,
    0x7a, 0x99, 0x81, 0x48, 0x08, 0x50, 0x00, 0x9c,
    0xc5, 0x57, 0x7c, 0x6e, 0x1f, 0x57, 0x3b, 0x4e,
    0x68, 0x01, 0xdd, 0x23, 0xc4, 0xa7, 0xd6, 0x79,
    0xcc, 0xf8, 0xa3, 0x86, 0xc6, 0x74, 0xcf, 0xfb
};

static const u_char sha2_hmac4_512[] = {
    0xb0, 0xba, 0x46, 0x56, 0x37, 0x45, 0x8c, 0x69,
    0x90, 0xe5, 0xa8, 0xc5, 0xf6, 0x1d, 0x4a, 0xf7,
    0xe5, 0x76, 0xd9, 0x7f, 0xf9, 0x4b, 0x87, 0x2d,
    0xe7, 0x6f, 0x80, 0x50, 0x36, 0x1e, 0xe3, 0xdb,
    0xa9, 0x1c, 0xa5, 0xc1, 0x1a, 0xa2, 0x5e, 0xb4,
    0xd6, 0x79, 0x27, 0x5c, 0xc5, 0x78, 0x80, 0x63,
    0xa5, 0xf1, 0x97, 0x41, 0x12, 0x0c, 0x4f, 0x2d,
    0xe2, 0xad, 0xeb, 0xeb, 0x10, 0xa2, 0x98, 0xdd
};

static const u_char sha2_hmac6_key[] = {
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa, 0xaa,
    0xaa, 0xaa, 0xaa
};

static const u_char sha2_hmac6_msg[] = {
    0x54, 0x65, 0x73, 0x74, 0x20, 0x55, 0x73, 0x69,
    0x6e, 0x67, 0x20, 0x4c, 0x61, 0x72, 0x67, 0x65,
    0x72, 0x20, 0x54, 0x68, 0x61, 0x6e, 0x20, 0x42,
    0x6c, 0x6f, 0x63, 0x6b, 0x2d, 0x53, 0x69, 0x7a,
    0x65, 0x20, 0x4b, 0x65, 0x79, 0x20, 0x2d, 0x20,
    0x48, 0x61, 0x73, 0x68, 0x20, 0x4b, 0x65, 0x79,
    0x20, 0x46, 0x69, 0x72, 0x73, 0x74
};

static const u_char sha2_hmac6_256[] = {
    0x60, 0xe4, 0x31, 0x59, 0x1e, 0xe0, 0xb6, 0x7f,
    0x0d, 0x8a, 0x26, 0xaa, 0xcb, 0xf5, 0xb7, 0x7f,
    0x8e, 0x0b, 0xc6, 0x21, 0x37, 0x28, 0xc5, 0x14,
    0x05, 0x46, 0x04, 0x0f, 0x0e, 0xe3, 0x7f, 0x54
};

static const u_char sha2_hmac6_384[] = {
    0x4e, 0xce, 0x08, 0x44, 0x85, 0x81, 0x3e, 0x90,
    0x88, 0xd2, 0xc6, 0x3a, 0x04, 0x1b, 0xc5, 0xb4,
    0x4f, 0x9e, 0xf1, 0x01, 0x2a, 0x2b, 0x58, 0x8f,
    0x3c, 0xd1, 0x1f, 0x05, 0x03, 0x3a, 0xc4, 0xc6,
    0x0c, 0x2e, 0xf6, 0xab, 0x40, 0x30, 0xfe, 0x82,
    0x96, 0x24, 0x8d, 0xf1, 0x63, 0xf4, 0x49, 0x52
};

static const u_char sha2_hmac6_512[] = {
    0x80, 0xb2, 0x42, 0x63, 0xc7, 0xc1, 0xa3, 0xeb,
    0xb7, 0x14, 0x93, 0xc1, 0xdd, 0x7b, 0xe8, 0xb4,
    0x9b, 0x46, 0xd1, 0xf4, 0x1b, 0x4a, 0xee, 0xc1,
    0x12, 0x1b, 0x01, 0x37, 0x83, 0xf8, 0xf3, 0x52,
    0x6b, 0x56, 0xd0, 0x37, 0xe0, 0x5f, 0x25, 0x98,
    0xbd, 0x0f, 0xd2, 0x21, 0x5d, 0x6a, 0x1e, 0x52,
    0x95, 0xe6, 0x4f, 0x73, 0xf6, 0x3f, 0x0a, 0xec,
    0x8b, 0x91, 0x5a, 0x98, 0x5d, 0x78, 0x65, 0x98
};

static const u_char sha2_hmac7_msg[] = {
    0x54, 0x68, 0x69, 0x73, 0x20, 0x69, 0x73, 0x20,
    0x61, 0x20, 0x74, 0x65, 0x73, 0x74, 0x20, 0x75,
    0x73, 0x69, 0x6e, 0x67, 0x20, 0x61, 0x20, 0x6c,
    0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74, 0x68,
    0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63, 0x6b,
    0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x6b, 0x65,
    0x79, 0x20, 0x61, 0x6e, 0x64, 0x20, 0x61, 0x20,
    0x6c, 0x61, 0x72, 0x67, 0x65, 0x72, 0x20, 0x74,
    0x68, 0x61, 0x6e, 0x20, 0x62, 0x6c, 0x6f, 0x63,
    0x6b, 0x2d, 0x73, 0x69, 0x7a, 0x65, 0x20, 0x64,
    0x61, 0x74, 0x61, 0x2e, 0x20, 0x54, 0x68, 0x65,
    0x20, 0x6b, 0x65, 0x79, 0x20, 0x6e, 0x65, 0x65,
    0x64, 0x73, 0x20, 0x74, 0x6f, 0x20, 0x62, 0x65,
    0x20, 0x68, 0x61, 0x73, 0x68, 0x65, 0x64, 0x20,
    0x62, 0x65, 0x66, 0x6f, 0x72, 0x65, 0x20, 0x62,
    0x65, 0x69, 0x6e, 0x67, 0x20, 0x75, 0x73, 0x65,
    0x64, 0x20, 0x62, 0x79, 0x20, 0x74, 0x68, 0x65,
    0x20, 0x48, 0x4d, 0x41, 0x43, 0x20, 0x61, 0x6c,
    0x67, 0x6f, 0x72, 0x69, 0x74, 0x68, 0x6d, 0x2e
};

static const u_char sha2_hmac7_256[] = {
    0x9b, 0x09, 0xff, 0xa7, 0x1b, 0x94, 0x2f, 0xcb,
    0x27, 0x63, 0x5f, 0xbc, 0xd5, 0xb0, 0xe9, 0x44,
    0xbf, 0xdc, 0x63, 0x64, 0x4f, 0x07, 0x13, 0x93,
    0x8a, 0x7f, 0x51, 0x53, 0x5c, 0x3a, 0x35, 0xe2
};

static const u_char sha2_hmac7_384[] = {
    0x66, 0x17, 0x17, 0x8e, 0x94, 0x1f, 0x02, 0x0d,
    0x35, 0x1e, 0x2f, 0x25, 0x4e, 0x8f, 0xd3, 0x2c,
    0x60, 0x24, 0x20, 0xfe, 0xb0, 0xb8, 0xfb, 0x9a,
    0xdc, 0xce, 0xbb, 0x82, 0x46, 0x1e, 0x99, 0xc5,
    0xa6, 0x78, 0xcc, 0x31, 0xe7, 0x99, 0x17, 0x6d,
    0x38, 0x60, 0xe6, 0x11, 0x0c, 0x46, 0x52, 0x3e
};

static const u_char sha2_hmac7_512[] = {
    0xe3, 0x7b, 0x6a, 0x77, 0x5d, 0xc8, 0x7d, 0xba,
    0xa4, 0xdf, 0xa9, 0xf9, 0x6e, 0x5e, 0x3f, 0xfd,
    0xde, 0xbd, 0x71, 0xf8, 0x86, 0x72, 0x89, 0x86,
    0x5d, 0xf5, 0xa3, 0x2d, 0x20, 0xcd, 0xc9, 0x44,
    0xb6, 0x02, 0x2c, 0xac, 0x3c, 0x49, 0x82, 0xb1,
    0x0d, 0x5e, 0xeb, 0x55, 0xc3, 0xe4, 0xde, 0x15,
    0x13, 0x46, 0x76, 0xfb, 0x6d, 0xe0, 0x44, 0x60,
    0x65, 0xc9, 0x74, 0x40, 0xfa, 0x8c, 0x6a, 0x58
};

static const hmac_testvector_t sha256_hmac_testvectors[] = {
    { sizeof(sha2_hmac1_key), sha2_hmac1_key, sizeof(sha2_hmac1_msg), sha2_hmac1_msg, sha2_hmac1_256 },
    { sizeof(sha2_hmac2_key), sha2_hmac2_key, sizeof(sha2_hmac2_msg), sha2_hmac2_msg, sha2_hmac2_256 },
    { sizeof(sha2_hmac3_key), sha2_hmac3_key, sizeof(sha2_hmac3_msg), sha2_hmac3_msg, sha2_hmac3_256 },
    { sizeof(sha2_hmac4_key), sha2_hmac4_key, sizeof(sha2_hmac4_msg), sha2_hmac4_msg, sha2_hmac4_256 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac6_msg), sha2_hmac6_msg, sha2_hmac6_256 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac7_msg), sha2_hmac7_msg, sha2_hmac7_256 },
    { 0, NULL, 0, NULL, NULL }
};

static const hmac_testvector_t sha384_hmac_testvectors[] = {
    { sizeof(sha2_hmac1_key), sha2_hmac1_key, sizeof(sha2_hmac1_msg), sha2_hmac1_msg, sha2_hmac1_384 },
    { sizeof(sha2_hmac2_key), sha2_hmac2_key, sizeof(sha2_hmac2_msg), sha2_hmac2_msg, sha2_hmac2_384 },
    { sizeof(sha2_hmac3_key), sha2_hmac3_key, sizeof(sha2_hmac3_msg), sha2_hmac3_msg, sha2_hmac3_384 },
    { sizeof(sha2_hmac4_key), sha2_hmac4_key, sizeof(sha2_hmac4_msg), sha2_hmac4_msg, sha2_hmac4_384 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac6_msg), sha2_hmac6_msg, sha2_hmac6_384 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac7_msg), sha2_hmac7_msg, sha2_hmac7_384 },
    { 0, NULL, 0, NULL, NULL }
};

static const hmac_testvector_t sha512_hmac_testvectors[] = {
    { sizeof(sha2_hmac1_key), sha2_hmac1_key, sizeof(sha2_hmac1_msg), sha2_hmac1_msg, sha2_hmac1_512 },
    { sizeof(sha2_hmac2_key), sha2_hmac2_key, sizeof(sha2_hmac2_msg), sha2_hmac2_msg, sha2_hmac2_512 },
    { sizeof(sha2_hmac3_key), sha2_hmac3_key, sizeof(sha2_hmac3_msg), sha2_hmac3_msg, sha2_hmac3_512 },
    { sizeof(sha2_hmac4_key), sha2_hmac4_key, sizeof(sha2_hmac4_msg), sha2_hmac4_msg, sha2_hmac4_512 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac6_msg), sha2_hmac6_msg, sha2_hmac6_512 },
    { sizeof(sha2_hmac6_key), sha2_hmac6_key, sizeof(sha2_hmac7_msg), sha2_hmac7_msg, sha2_hmac7_512 },
    { 0, NULL, 0, NULL, NULL }
};

struct hash_desc hash_desc_sha2_256 = {
	algo_type: IKE_ALG_HASH,
	algo_id:   OAKLEY_SHA2_256,
	algo_next: NULL,
	hash_ctx_size:    sizeof(sha256_context),
	hash_block_size:  SHA2_256_BLOCK_SIZE,
	hash_digest_size: SHA2_256_DIGEST_SIZE,
	hash_testvectors: sha256_hash_testvectors,
	hmac_testvectors: sha256_hmac_testvectors,
	hash_init: (void (*)(void *))sha256_init,
	hash_update: (void (*)(void *, const u_char *, size_t ))sha256_write,
	hash_final:(void (*)(u_char *, void *))sha256_hash_final
};

struct hash_desc hash_desc_sha2_384 = {
	algo_type: IKE_ALG_HASH,
	algo_id:   OAKLEY_SHA2_384,
	algo_next: NULL,
	hash_ctx_size:    sizeof(sha512_context),
	hash_block_size:  SHA2_384_BLOCK_SIZE,
	hash_digest_size: SHA2_384_DIGEST_SIZE,
	hash_testvectors: sha384_hash_testvectors,
	hmac_testvectors: sha384_hmac_testvectors,
	hash_init: (void (*)(void *))sha384_init,
	hash_update: (void (*)(void *, const u_char *, size_t ))sha512_write,
	hash_final:(void (*)(u_char *, void *))sha384_hash_final
};

struct hash_desc hash_desc_sha2_512 = {
	algo_type: IKE_ALG_HASH,
	algo_id:   OAKLEY_SHA2_512,
	algo_next: NULL,
	hash_ctx_size:    sizeof(sha512_context),
	hash_block_size:  SHA2_512_BLOCK_SIZE,
	hash_digest_size: SHA2_512_DIGEST_SIZE,
	hash_testvectors: sha512_hash_testvectors,
	hmac_testvectors: sha512_hmac_testvectors,
	hash_init: (void (*)(void *))sha512_init,
	hash_update: (void (*)(void *, const u_char *, size_t ))sha512_write,
	hash_final:(void (*)(u_char *, void *))sha512_hash_final
};

int ike_alg_sha2_init(void);

int
ike_alg_sha2_init(void)
{
    int ret
;
    ret = ike_alg_register_hash(&hash_desc_sha2_256);
    if (ret)
	goto out;
    ret = ike_alg_register_hash(&hash_desc_sha2_384);
    if (ret)
	goto out;
    ret = ike_alg_register_hash(&hash_desc_sha2_512);

out:
	return ret;
}

/*
IKE_ALG_INIT_NAME: ike_alg_sha2_init
*/
