/*
 * lftp and utils
 *
 * Copyright (c) 1996-1997 by Alexander V. Lukyanov (lav@yars.free.net)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <config.h>

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <unistd.h>
#include <assert.h>
#include "MirrorJob.h"
#include "misc.h"
#include "xalloca.h"

void  MirrorJob::PrintStatus(int v)
{
   char *tab="\t";

   if(v!=-1)
      SessionJob::PrintStatus(v);
   else
      tab="";

   if(!Done())
      return;
   printf("%sTotal: %d director%s, %d file%s, %d symlink%s\n",tab,
      dirs,dirs==1?"y":"ies",
      tot_files,tot_files==1?"":"s",
      tot_symlinks,tot_symlinks==1?"":"s");
   if(new_files || new_symlinks)
      printf("%sNew: %d file%s, %d symlink%s\n",tab,
	 new_files,new_files==1?"":"s",
	 new_symlinks,new_symlinks==1?"":"s");
   if(mod_files || mod_symlinks)
      printf("%sModified: %d file%s, %d symlink%s\n",tab,
	 mod_files,mod_files==1?"":"s",
	 mod_symlinks,mod_symlinks==1?"":"s");
   if(del_dirs || del_files || del_symlinks)
      printf("%s%semoved: %d director%s, %d file%s, %d symlink%s\n",tab,
	 flags&DELETE?"R":"To be r",
	 del_dirs,del_dirs==1?"y":"ies",
	 del_files,del_files==1?"":"s",
	 del_symlinks,del_symlinks==1?"":"s");
}

void  MirrorJob::ShowRunStatus(StatusLine *s)
{
   switch(state)
   {
   case(DONE):
      s->Show("");
      break;

   case(WAITING_FOR_SUBGET):
   case(WAITING_FOR_SUBMIRROR):
      if(waiting && !waiting->Done())
	 waiting->ShowRunStatus(s);
      break;

   case(CHANGING_REMOTE_DIR):
      s->Show("CWD  `%s' [%s]",remote_dir,session->CurrentStatus());
      break;

   case(GETTING_SHORT_LIST):
      s->Show("NLST `%s' [%s]",remote_dir,session->CurrentStatus());
      break;

   case(GETTING_LONG_LIST):
      s->Show("LIST `%s' [%s]",remote_dir,session->CurrentStatus());
      break;

   case(GETTING_INFO):
      if(session->IsOpen())
	 s->Show("Getting files info [%s]",session->CurrentStatus());
      break;
   }
}

void  MirrorJob::HandleFile(int how)
{
   ArgV *args;
   int	 res;
   mode_t mode;
   GetJob   *gj;
   struct stat st;

   char *local_name=dir_file(local_dir,file->name);
   local_name=strcpy((char*)alloca(strlen(local_name)+1),local_name);
   char *remote_name=dir_file(remote_dir,file->name);

   if(!(rx_include && regexec(&rxc_include,remote_name,0,0,REG_NOSUB)==0)
    && ((rx_exclude && regexec(&rxc_exclude,remote_name,0,0,REG_NOSUB)==0)
        || (rx_include && !rx_exclude)))
   {
      to_get->next();
      return;
   }

   if(file->defined&file->TYPE)
   {
      switch(file->filetype)
      {
      case(file->NORMAL):
      {
      try_get:
	 bool cont_this=false;
	 if(how!=0)
	 {
	    to_get->next();
	    break;
	 }
	 if(lstat(local_name,&st)!=-1)
	 {
	    if((flags&CONTINUE) && S_ISREG(st.st_mode)
	    && (file->defined&(file->DATE|file->DATE_UNPREC))
	    && file->date + prec < st.st_mtime
	    && (file->defined&file->SIZE) && file->size >= st.st_size)
	       cont_this=true;
	    else
	       remove(local_name);
	    mod_files++;
	 }
	 else
	 {
	    new_files++;
	 }
	 // launch get job
	 args=new ArgV("get");
	 args->Append(file->name);
	 args->Append(local_name);
	 gj=new GetJob(Clone(),args,cont_this);
	 if(file->defined&(file->DATE|file->DATE_UNPREC))
	    gj->SetTime(file->date);
	 waiting=gj;
	 waiting->parent=this;
	 waiting->cmdline=args->Combine();
	 break;
      }
      case(file->DIRECTORY):
      {
	 if(how!=1 || (flags&NO_RECURSION))
	 {
	    to_get->next();
	    break;
	 }
	 if(!strcmp(file->name,".") || !strcmp(file->name,".."))
	 {
	    to_get->next();
	    break;
	 }
	 mode=((file->defined&file->MODE)&&!(flags&NO_PERMS))?file->mode:0755;
	 res=mkdir(local_name,mode|0700);
	 // launch sub-mirror
	 MirrorJob *mj=new MirrorJob(Clone(),local_name,remote_name);
	 waiting=mj;
	 waiting->parent=this;
	 waiting->cmdline=(char*)xmalloc(strlen("\\mirror")+1+
					 strlen(file->name)+1);
	 sprintf(waiting->cmdline,"\\mirror %s",file->name);

	 // inherit flags and other things
	 mj->SetFlags(flags,1);
	 mj->SetPrec(prec);
	 if(rx_include)	mj->SetInclude(rx_include);
	 if(rx_exclude)	mj->SetExclude(rx_exclude);

	 break;
      }
      case(file->SYMLINK):
	 if(how!=0)
	 {
	    to_get->next();
	    break;
	 }
#ifdef HAVE_LSTAT
	 if(file->defined&file->SYMLINK)
	 {
	    struct stat st;
	    if(lstat(local_name,&st)!=-1)
	    {
	       mod_symlinks++;
	       remove(local_name);
	    }
	    else
	    {
	       new_symlinks++;
	    }
	    res=symlink(file->symlink,local_name);
	    if(res==-1)
	       perror(local_name);
	 }
#endif /* LSTAT */
	 to_get->next();
	 break;
      }
   }
   else
   {
      // no info on type -- try to get
      goto try_get;
   }
}

int   MirrorJob::Do()
{
   char	 **glob_res;
   int	 res;
   FileSet  *files;
   const char *rcwd;
   int i;

   switch(state)
   {
   case(DONE):
      return STALL;

   case(CHANGING_REMOTE_DIR):
      res=session->Done();
      if(res==Ftp::IN_PROGRESS)
	 return STALL;
      if(res<0)
      {
	 fprintf(stderr,"mirror: %s\n",session->StrError(res));
	 state=DONE;
	 session->Close();
	 return MOVED;
      }
      session->Close();

      rcwd=session->GetCwd();
      xfree(remote_dir);
      remote_dir=xstrdup(rcwd);

      glob=new RemoteGlob(session,"",Ftp::LONG_LIST);
      state=GETTING_LONG_LIST;
      return MOVED;

   case(GETTING_LONG_LIST):
      if(!glob->Done())
	 return STALL;
      if(glob->Error())
      {
	 fprintf(stderr,"mirror LIST: %s\n",session->StrError(glob->ErrorCode()));
	 state=DONE;
      	 delete glob;
	 glob=0;
// 	 exit_code=1;
	 return MOVED;
      }
      glob_res=glob->GetResult();
      to_get=parse_listing(glob_res);
      delete glob;

      glob=new RemoteGlob(session,"",Ftp::LIST);
      state=GETTING_SHORT_LIST;
      return MOVED;

   case(GETTING_SHORT_LIST):
      if(!glob->Done())
	 return STALL;
      if(glob->Error())
      {
	 fprintf(stderr,"mirror NLST: %s\n",session->StrError(glob->ErrorCode()));
	 state=DONE;
      	 delete glob;
	 glob=0;
// 	 exit_code=1;
	 return MOVED;
      }
      glob_res=glob->GetResult();
      to_get->Merge(glob_res);
      delete glob;
      glob=0;

      state=GETTING_INFO;
      to_get->rewind();
      return MOVED;

   case(GETTING_INFO):
      if(session->IsClosed())
      {
	 get_info=(Ftp::fileinfo*)xmalloc(sizeof(*get_info)*to_get->get_fnum());
	 cnt=0;
	 for(;;)
	 {
	    file=to_get->curr();
	    if(file==0)
	       break;

	    if((file->defined & file->TYPE &&
	       (file->filetype==file->SYMLINK || file->filetype==file->DIRECTORY))
	    || ((file->defined&(file->DATE|file->SIZE))==(file->DATE|file->SIZE)))
	    {
	       to_get->next();
	       continue;
	    }
	    get_info[cnt].get_size=!(file->defined & file->SIZE);
	    if(!get_info[cnt].get_size)
	       get_info[cnt].size=-1;
	    get_info[cnt].get_time=!(file->defined & file->DATE);
	    if(!get_info[cnt].get_time)
	       get_info[cnt].time=(time_t)-1;
	    get_info[cnt].file=file->name;
	    cnt++;
	    to_get->next();
	 }
	 if(cnt)
	 {
	    session->GetInfoArray(get_info,cnt);
      	 }
	 else
	    goto info_done;
      }
      res=session->Done();
      if(res==Ftp::DO_AGAIN)
	 return STALL;
      if(res==Ftp::IN_PROGRESS)
	 return STALL;
      assert(res==Ftp::OK);
      session->Close();

      for(i=0; i<cnt; i++)
      {
	 if(get_info[i].time!=(time_t)-1)
	    to_get->SetDate(get_info[i].file,get_info[i].time);
	 if(get_info[i].size!=-1)
	    to_get->SetSize(get_info[i].file,get_info[i].size);
      }

   info_done:
      xfree(get_info);
      get_info=0;

      to_get->Count(NULL,&tot_files,&tot_symlinks,&tot_files);

      files=local_files(local_dir);

      to_rm=new FileSet(files);
      to_rm->SubtractAny(to_get);

      same=new FileSet(to_get);
      to_get->SubtractSame(files,flags&ONLY_NEWER,prec);
      same->SubtractAny(to_get);

      delete files;

      to_get->rewind();
      waiting=0;
      state=WAITING_FOR_SUBGET;
      return MOVED;

   case(WAITING_FOR_SUBGET):
      if(waiting && waiting->Done())
      {
	 to_get->next();
	 delete waiting;
	 waiting=0;
      }
      if(!waiting)
      {
	 file=to_get->curr();
      	 if(!file)
	 {
	    to_get->rewind();
	    state=WAITING_FOR_SUBMIRROR;
	    return MOVED;
	 }
	 HandleFile(0);
	 return MOVED;
      }
      return STALL;

   case(WAITING_FOR_SUBMIRROR):
      if(waiting && waiting->Done())
      {
	 MirrorJob &mj=*(MirrorJob*)waiting; // we are sure it is a MirrorJob
	 tot_files+=mj.tot_files;
	 new_files+=mj.new_files;
	 mod_files+=mj.mod_files;
	 del_files+=mj.del_files;
	 tot_symlinks+=mj.tot_symlinks;
	 new_symlinks+=mj.new_symlinks;
	 mod_symlinks+=mj.mod_symlinks;
	 del_symlinks+=mj.del_symlinks;
	 dirs+=mj.dirs;
	 del_dirs+=mj.del_dirs;

	 to_get->next();
	 delete waiting;
	 waiting=0;
      }
      if(!waiting)
      {
	 file=to_get->curr();
      	 if(!file)
	 {
	    to_rm->Count(&del_dirs,&del_files,&del_symlinks,&del_files);
	    if(flags&DELETE)
	       to_rm->LocalRemove(local_dir);
	    if(!(flags&NO_PERMS))
	       to_get->LocalChmod(local_dir,flags&ALLOW_SUID?0:S_ISUID|S_ISGID);
	    to_get->LocalUtime(local_dir);
	    if(!(flags&NO_PERMS))
	       same->LocalChmod(local_dir,flags&ALLOW_SUID?0:S_ISUID|S_ISGID);
	    same->LocalUtime(local_dir); // the old mtime can differ up to prec
	    state=DONE;
	    return MOVED;
	 }
	 HandleFile(1);
	 return MOVED;
      }
      return STALL;
   }
   /*NOTREACHED*/
   abort();
}

MirrorJob::MirrorJob(FileAccess *f,const char *new_local_dir,const char *new_remote_dir)
   : SessionJob(f)
{
   local_dir=xstrdup(new_local_dir);
   remote_dir=xstrdup(new_remote_dir);

   to_get=to_rm=same=0;
   file=0;
   glob=0;

   tot_files=new_files=mod_files=del_files=
   tot_symlinks=new_symlinks=mod_symlinks=del_symlinks=0;
   dirs=1; del_dirs=0;

   get_info=0;

   flags=0;

   rx_include=rx_exclude=0;

   prec=0;  // time must be exactly same by default

   if(strcmp(f->GetProto(),"ftp"))
   {
      eprintf("Sorry, mirror currently works with ftp only\n");
      state=DONE;
      return;
   }

   session->Chdir(remote_dir);
   state=CHANGING_REMOTE_DIR;
}

MirrorJob::~MirrorJob()
{
   xfree(get_info);
   xfree(local_dir);
   xfree(remote_dir);
   if(to_get)
      delete to_get;
   if(to_rm)
      delete to_rm;
   if(same)
      delete same;
   // don't delete this->file -- it is a reference
   if(glob)
      delete glob;
   if(rx_include)
   {
      free(rx_include);
      regfree(&rxc_include);
   }
   if(rx_exclude)
   {
      free(rx_exclude);
      regfree(&rxc_exclude);
   }
}

int MirrorJob::SetInclude(const char *s)
{
   if(rx_include)
   {
      free(rx_include);
      rx_include=0;
      regfree(&rxc_include);
   }
   int res=regcomp(&rxc_include,s,REG_NOSUB|REG_EXTENDED);
   if(res!=0)
   {
      char err[1024];
      regerror(res,&rxc_include,err,sizeof(err));
      fprintf(stderr,"mirror: --include - %s\n",err);
      return -1;
   }
   rx_include=xstrdup(s);
   return 0;
}
int MirrorJob::SetExclude(const char *s)
{
   if(rx_exclude)
   {
      free(rx_exclude);
      rx_exclude=0;
      regfree(&rxc_exclude);
   }
   int res=regcomp(&rxc_exclude,s,REG_NOSUB|REG_EXTENDED);
   if(res!=0)
   {
      char err[1024];
      regerror(res,&rxc_exclude,err,sizeof(err));
      fprintf(stderr,"mirror: --exclude - %s\n",err);
      return -1;
   }
   rx_exclude=xstrdup(s);
   return 0;
}
