/*
 * ProFTPD - FTP server daemon
 * Copyright (c) 1997, Public Flood Software
 *  
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Authentication module for ProFTPD
 * $Id: mod_auth.c,v 1.12 1997/11/11 17:17:23 flood Exp $
 */

#include "conf.h"

#ifdef HAVE_CRYPT_H
#include <crypt.h>
#endif

#ifdef USESHADOW
#include <shadow.h>
#endif

#include "privs.h"

/* Return codes form _do_auth() */

#define AUTH_NOPWD	-1		/* Account does not exist */
#define AUTH_BADPWD	-2		/* Password mismatch */
#define AUTH_AGEPWD	-3		/* Password hasn't been changed recently enough */
#define AUTH_DISABLEDPWD -4		/* Account has been disabled */

/* This could be tuned */
#define HASH_TABLE_SIZE	10

#ifdef USESHADOW

/* Shadow password entries are stored as number of days, not seconds and
 * are -1 if unused
 */

#define SP_CVT_DAYS(x)	((x) == (time_t)-1 ? (x) : ((x) * 86400))

#endif /* USESHADOW */

/* From the core module */
extern int core_display_file(const char *,const char *);
extern pid_t mpid;

module auth_module;

typedef struct _idmap {
  struct _idmap *next,*prev;

  int id;			/* uid or gid */
  char *name;			/* user or group name */
} idmap_t;

/* uid/gid -> user/group mapping hash table */

static int auth_tries = 0;

static xaset_t *uid_table[HASH_TABLE_SIZE];
static xaset_t *gid_table[HASH_TABLE_SIZE];

static int _compare_id(idmap_t *m1, idmap_t *m2)
{
  if(m1->id < m2->id)
    return -1;
  if(m1->id > m2->id)
    return 1;
  return 0;
}

static idmap_t *_auth_lookup_id(xaset_t **id_table, int id)
{
  int hash = id % HASH_TABLE_SIZE;
  idmap_t *m;

  if(!id_table[hash])
    id_table[hash] = xaset_create(permanent_pool,
                     (XASET_COMPARE)_compare_id);

  for(m = (idmap_t*)id_table[hash]->xas_list; m; m=m->next)
    if(m->id >= id)
      break;

  if(!m || m->id != id) {
    /* Isn't in the table */
    m = (idmap_t*)pcalloc(id_table[hash]->mempool,sizeof(idmap_t));
    m->id = id;
    xaset_insert_sort(id_table[hash],(xasetmember_t*)m,FALSE);
  } 

  return m;
}

char *auth_map_uid(int uid)
{
  idmap_t *m;

  m = _auth_lookup_id(uid_table,uid);

  if(!m->name) {
    /* Wasn't cached, so perform a lookup */
    struct passwd *pw;

    setpwent();
    if((pw = getpwuid(uid)) == NULL) {
      char buf[10];

      sprintf(buf,"%d",uid);
      m->name = pstrdup(permanent_pool,buf);
    } else
      m->name = pstrdup(permanent_pool,pw->pw_name);
  }

  return m->name;
}

char *auth_map_gid(int gid)
{
  idmap_t *m;

  m = _auth_lookup_id(gid_table,gid);

  if(!m->name) {
    struct group *grp;

    setgrent();
    if((grp = getgrgid(gid)) == NULL) {
      char buf[10];

      sprintf(buf,"%d",gid);
      m->name = pstrdup(permanent_pool,buf);
    } else
      m->name = pstrdup(permanent_pool,grp->gr_name);
  }

  return m->name;
}

/* check_auth is hooked into the main server's auth_hook function,
 * so that we can deny all commands until authentication is complete.
 */

int check_auth(cmd_rec *cmd)
{
  if(get_param_int(cmd->server->conf,"authenticated",FALSE) != 1) {
    send_response("530","Please login with USER and PASS.");
    return FALSE;
  }

  return TRUE;
}

int _auth_shutdown(CALLBACK_FRAME)
{
  log_pri(LOG_ERR,"scheduled main_exit() never ran [from auth:_login_timeout], terminating.");
  end_login(1);
  return 0;				/* Avoid compiler warning */
}

int _login_timeout(CALLBACK_FRAME)
{
  /* Is this the proper behavior when timing out? */
  send_response_async(R_421,"Login Timeout (%d seconds): closing control connection.",
                      TimeoutLogin);
  schedule(main_exit,0,(void*)LOG_NOTICE,"FTP login timed out, disconnected.",
           (void*)0,NULL);
  remove_timer(TIMER_IDLE,ANY_MODULE);
  remove_timer(TIMER_NOXFER,ANY_MODULE);
  add_timer(10,-1,&auth_module,_auth_shutdown);
  return 0;		/* Don't restart the timer */
}

int auth_init_child()
{
  /* Start the login timer */
  if(TimeoutLogin)
    add_timer(TimeoutLogin,TIMER_LOGIN,&auth_module,_login_timeout);
  return 0;
}

int auth_init()
{
  int i;

  for(i = 0; i < HASH_TABLE_SIZE; i++) {
    uid_table[i] = NULL; 
    gid_table[i] = NULL;
  }

  /* By default, enable auth checking */
  set_auth_check(check_auth);
  return 0;
}

#ifdef USESHADOW
static char *_get_pw_info(pool *p, char *u,
                          time_t *lstchg, time_t *min, time_t *max,
                          time_t *warn, time_t *inact, time_t *expire)
{
  struct spwd *sp;
  char *cpw = NULL;

  PRIVS_ROOT
  sp = getspnam(u);
  
  if(sp) {
    cpw = pstrdup(p,sp->sp_pwdp);
    if(lstchg) *lstchg = SP_CVT_DAYS(sp->sp_lstchg);
    if(min) *min = SP_CVT_DAYS(sp->sp_min);
    if(max) *max = SP_CVT_DAYS(sp->sp_max);
    if(warn) *warn = SP_CVT_DAYS(sp->sp_warn);
    if(inact) *inact = SP_CVT_DAYS(sp->sp_inact);
    if(expire) *expire = SP_CVT_DAYS(sp->sp_expire);
  }
#ifdef AUTOSHADOW
  else {
    struct passwd *pw;

    endspent();
    PRIVS_RELINQUISH

    if((pw = getpwnam(u)) != NULL) {
      cpw = pstrdup(p,pw->pw_passwd);
      if(lstchg) *lstchg = (time_t)-1;
      if(min) *min = (time_t)-1;
      if(max) *max = (time_t)-1;
      if(warn) *warn = (time_t)-1;
      if(inact) *inact = (time_t)-1;
      if(expire) *expire = (time_t)-1;
    }
  }
#else
  endspent();
  PRIVS_RELINQUISH
#endif /* AUTOSHADOW */
  return cpw;
}

#else /* USESHADOW */

static char *_get_pw_info(pool *p, char *u,
                          time_t *lstchg, time_t *min, time_t *max,
                          time_t *warn, time_t *inact, time_t *expire)
{
  struct passwd *pw;
  char *cpw = NULL;

  setpwent();

  pw = getpwnam(u);

  if(pw) {
    cpw = pstrdup(p,pw->pw_passwd);

    if(lstchg) *lstchg = (time_t)-1;
    if(min) *min = (time_t)-1;
    if(max) *max = (time_t)-1;
    if(warn) *warn = (time_t)-1;
    if(inact) *inact = (time_t)-1;
    if(expire) *expire = (time_t)-1;
  }

  return cpw;
}

#endif /* USESHADOW */

static int _do_auth(pool *p, xaset_t *conf, char *u, char *pw)
{
  char *cpw = NULL;
  time_t now;
  time_t lstchg = 0,max = 0,inact = 0,disable = 0;
  config_rec *c;

  time(&now);

  if(conf) {
    c = find_config(conf,CONF_PARAM,"UserPassword",FALSE);

    while(c) {
      if(!strcmp(c->argv[0],u)) {
        cpw = (char*)c->argv[1];
        break;
      }

      c = find_config_next(c->next,CONF_PARAM,"UserPassword",FALSE);
    }
  }


  if(cpw) {
    if(!_get_pw_info(p,u,&lstchg,NULL,&max,NULL,&inact,&disable))
      return AUTH_NOPWD;
  } else {
    if(!(cpw = _get_pw_info(p,u,&lstchg,NULL,&max,NULL,&inact,&disable)))
      return AUTH_NOPWD;
  }

  if(strcmp(crypt(pw,cpw),cpw) != 0)
    return AUTH_BADPWD;

  if(lstchg > (time_t)0 && max > (time_t)0 &&
     inact > (time_t)0)
    if(now > lstchg + max + inact)
      return AUTH_AGEPWD;

  if(disable > (time_t)0 && now > disable)
    return AUTH_DISABLEDPWD;

  return 0;
}

/* Handle group based authentication, only checked if pw
 * based fails
 */


static config_rec *_auth_group(char *user, char **group,
                               char **ournamep, char **anonnamep, char *pass)
{
  config_rec *c;
  char *ourname = NULL,*anonname = NULL;
  char **grmem;
  struct group *grp;

  ourname = (char*)get_param_ptr(main_server->conf,"UserName",FALSE);
  if(ournamep && ourname)
    *ournamep = ourname;

  c = find_config(main_server->conf,CONF_PARAM,"GroupPassword",TRUE);

  if(c) do {
    grp = getgrnam(c->argv[0]);

    if(!grp)
      continue;

    for(grmem = grp->gr_mem; *grmem; grmem++)
      if(!strcmp(*grmem,user)) {
        if(!strcmp(crypt(pass,c->argv[1]),c->argv[1]))
          break;
      }

    if(*grmem) {
      if(group)
        *group = c->argv[0];

      if(c->parent)
        c = c->parent;

      if(c->config_type == CONF_ANON)
        anonname = (char*)get_param_ptr(c->subset,"UserName",FALSE);
      if(anonnamep)
        *anonnamep = anonname;
      if(anonnamep && !anonname && ourname)
        *anonnamep = ourname;
      
      break;
    }
  } while((c = find_config_next(c->next,CONF_PARAM,"GroupPassword",TRUE)) != NULL);

  return c;
}


static config_rec *_auth_resolve_user(char **user, char **ournamep,
                                      char **anonnamep)
{
  config_rec *c;
  char *ourname,*anonname = NULL;

  /* Precendence rules:
   *   1. Search for UserAlias directive.
   *   2. Search for Anonymous directive.
   *   3. Normal user login
   */

  ourname = (char*)get_param_ptr(main_server->conf,"UserName",FALSE);

  if(ournamep && ourname)
    *ournamep = ourname; 

  c = find_config(main_server->conf,CONF_PARAM,"UserAlias",TRUE);
  if(c) do {
    if(!strcmp(c->argv[0],*user))
      break;
    } while((c = find_config_next(c->next,CONF_PARAM,"UserAlias",TRUE)) != NULL);

  if(c) {
    *user = c->argv[1];

    /* If the alias is applied inside an <Anonymous> context, we have found
     * our anon block
     */

    if(c->parent && c->parent->config_type == CONF_ANON)
      c = c->parent;
    else
      c = NULL;
  }

  /* Next, search for an anonymous entry */

  if(!c)
    c = find_config(main_server->conf,CONF_ANON,NULL,FALSE);
  else
    find_config_set_top(c);

  if(c) do {
    anonname = (char*)get_param_ptr(c->subset,"UserName",FALSE);
    if(!anonname)
      anonname = ourname;

    if(anonname && !strcmp(anonname,*user)) {
       if(anonnamep)
         *anonnamep = anonname;
       break;
    }
  } while((c = find_config_next(c->next,CONF_ANON,NULL,FALSE)) != NULL);

  return c;
}

static int _auth_check_ftpusers(xaset_t *s, const char *user)
{
  int res = 1;
  FILE *fp;
  char *u,buf[256];

  if(get_param_int(s,"UseFtpUsers",FALSE) != 0) {
    PRIVS_ROOT
    fp = fopen(FTPUSERS_PATH,"r");
    PRIVS_RELINQUISH

    if(!fp)
      return res;

    while(fgets(buf,sizeof(buf)-1,fp)) {
      buf[sizeof(buf)-1] = '\0'; CHOP(buf);

      u = buf; while(isspace(*u) && *u) u++;

      if(!*u || *u == '#')
        continue;

      if(!strcmp(u,user)) {
        res = 0;
        break;
      }
    }

    fclose(fp);
  }

  return res;
}

static int _auth_check_shell(xaset_t *s, const char *shell)
{
  int res = 1;
  FILE *shellf;
  char buf[256];

  if(get_param_int(s,"RequireValidShell",FALSE) != 0 &&
     (shellf = fopen(VALID_SHELL_PATH,"r")) != NULL) {
    res = 0;
    while(fgets(buf,sizeof(buf)-1,shellf)) {
      buf[sizeof(buf)-1] = '\0'; CHOP(buf);

      if(!strcmp(shell,buf)) {
        res = 1;
        break;
      }
    }

    fclose(shellf);
  }

  return res;
}

/* Determine if the user (non-anon) needs a default root dir
 * other than /
 */

static char *_get_default_root(pool *p)
{
  config_rec *c;
  char **ent,*grp,*ret = NULL;
  int found,cnt;

  c = find_config(main_server->conf,CONF_PARAM,"DefaultRoot",FALSE);

  while(c) {
    /* Check the groups acl */
    if(c->argc < 2) {
      ret = c->argv[0];
      break;
    }

    for(ent = (char**)(c->argv+1); *ent; ent++) {
      found = FALSE;
      grp = *ent;

      if(*grp == '!') {
        found = !found;
        grp++;
      }

      for(cnt = session.groups->nelts-1; cnt >= 0; cnt--)
        if(strcmp(*(((char**)session.groups->elts)+cnt),grp) == 0) {
	  found = !found; break;
        }

      if(!found) {
        ent = NULL;
        break;
      }
    }

    if(ent) {				/* group-expression matched */
      ret = c->argv[0];
      break;
    }

    c = find_config_next(c->next,CONF_PARAM,"DefaultRoot",FALSE);
  }

  if(ret) {
    if(!strcmp(ret,"/"))
      ret = NULL;
    else {
      char *realdir;

      if(!strcmp(ret,"~"))
        ret = pstrcat(p,"~",session.user,NULL);

      realdir = dir_realpath(p,ret);

      if(!realdir)
        log_pri(LOG_WARNING,"DefaultRoot %s is invalid or not accessible.",
                ret);
      ret = realdir;
    }
  }

  return ret;
}

/* Next function (the biggie) handles all authentication, setting
 * up chroot() jail, etc.
 */

static int _setup_environment(pool *p, char *user, char *pass)
{
  struct passwd *pw;
  struct group *gr;
  char **gr_mem;
  config_rec *c;
  char *origuser,*ourname,*anonname = NULL,*anongroup = NULL,*ugroup = NULL;
  char ttyname[20];
  char *defroot = NULL,*xferlog = NULL;

  int aclp,i;

  /********************* Authenticate the user here *********************/

  origuser = user;
  c = _auth_resolve_user(&user,&ourname,&anonname);

  /* If c != NULL from this point on, we have an anonymous login */

  aclp = login_check_limits(main_server->conf,FALSE,TRUE,&i);

  if(c) {
    anongroup = (char*)get_param_ptr(c->subset,"GroupName",FALSE);
    if(!anongroup)
      anongroup = (char*)get_param_ptr(main_server->conf,"GroupName",FALSE);

    if(!login_check_limits(c->subset,FALSE,TRUE,&i) || (!aclp && !i) ){
      log_auth(LOG_NOTICE,"ANON %s: LIMIT access denies login from %s [%s]",
                          origuser,
                          session.c->remote_name,
                          inet_ntoa(*session.c->remote_ipaddr));
      return 0;
    }
  }

  if(!c && !aclp) {
    log_auth(LOG_NOTICE,"USER %s: LIMIT access denies login from %s [%s]",
             origuser,session.c->remote_name,
             inet_ntoa(*session.c->remote_ipaddr));
    return 0;
  }

  if(!c || get_param_int(c->subset,"AnonRequirePassword",FALSE) == 1) {
    int authcode;

    if(c)
      authcode = _do_auth(p,c->subset,user,pass);
    else
      authcode = _do_auth(p,main_server->conf,user,pass);

    if(authcode) {
      /* Normal authentication has failed, see if group authentication
       * passes
       */

      if((c = _auth_group(user,&anongroup,&ourname,&anonname,pass)) != NULL) {
        if(c->config_type != CONF_ANON) {
          c = NULL;
          ugroup = anongroup; anongroup = NULL;
        }

        authcode = 0;
      }
    }

    bzero(pass,strlen(pass));

    switch(authcode) {
    case AUTH_NOPWD:
      log_auth(LOG_NOTICE,"USER %s: no such user found from %s [%s]",
               user,session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
      break;
    case AUTH_BADPWD:
      log_auth(LOG_NOTICE,"USER %s: incorrect password from %s [%s]",
               origuser,session.c->remote_name,
               inet_ntoa(*session.c->remote_ipaddr));
      break;
    case AUTH_AGEPWD:
      log_auth(LOG_NOTICE,"USER %s: password expired from %s [%s]",
               user,session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
      break;
    case AUTH_DISABLEDPWD:
      log_auth(LOG_NOTICE,"USER %s: account disabled from %s [%s]",
               user,session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
      break;
    };

    if(authcode != 0)
      return 0;
  }

/* Flood - 7/10/97, not sure what setutent() was used for, but it
 * certainly looks unnecessary now.
 */

 /* setutent(); */

  setgrent();

  if((pw = getpwnam(user)) == NULL) {
    log_pri(LOG_ERR,"failed login, can't find user '%s'",user);
    send_response(R_530,"Configuration Error (can't find user account)");
    return -1;
  }

  /* Root FTP login is not allowed, but don't tell the user that they
   * hit the right password.
   */

  if(pw->pw_uid == 0) {
    if(!c)
      log_auth(LOG_CRIT,"SECURITY VIOLATION: root login attempted from %s [%s]",
               session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
    return 0;
  }

  if(!_auth_check_shell((c ? c->subset : main_server->conf),pw->pw_shell)) {
    log_auth(LOG_NOTICE,"failed login '%s' from %s [%s] (invalid shell)",
               user,session.c->remote_name,
               inet_ntoa(*session.c->remote_ipaddr));
    return 0;
  }

  if(!_auth_check_ftpusers((c ? c->subset : main_server->conf),pw->pw_name)) {
    log_auth(LOG_NOTICE,"failed login '%s' from %s [%s] (user in %s)",
             pw->pw_name,session.c->remote_name,
             inet_ntoa(*session.c->remote_ipaddr),
             FTPUSERS_PATH);
    return 0;
  }

  if(c) {
    struct group *grp;

    session.anon_root = dir_realpath(session.pool,c->name);
    session.anon_user = pstrdup(session.pool,pass);

    if(!session.anon_root) {
      send_response(R_530,"Unable to set anonymous privileges.");
      log_pri(LOG_ERR,"%s dir_realpath(%s), %s",
              session.user,strerror(errno));
      return 0;
    }
  
    strcpy(session.cwd,"/");
    xferlog = get_param_ptr(c->subset,"TransferLog",FALSE);

    if(anongroup) {
      grp = getgrnam(anongroup);
      if(grp)
        pw->pw_gid = grp->gr_gid;
    }
  } else {
    struct group *grp;

    if(ugroup) {
      grp = getgrnam(ugroup);
      if(grp)
        pw->pw_gid = grp->gr_gid;
    }

    strncpy(session.cwd,pw->pw_dir,MAX_PATH_LEN);
  }

  if(c)
    log_auth(LOG_NOTICE,"ANONYMOUS FTP login as '%s' from %s [%s]",
    origuser,session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));
  else
    log_auth(LOG_NOTICE,"FTP login as '%s' from %s [%s]",
    origuser,session.c->remote_name,inet_ntoa(*session.c->remote_ipaddr));

  session.user = pstrdup(permanent_pool,auth_map_uid(pw->pw_uid));
  session.group = pstrdup(permanent_pool,auth_map_gid(pw->pw_gid));

  /* Write the login to wtmp.  This must be done here because we won't
   * have access after we give up root.  This can result in falsified
   * wtmp entries if an error kicks the user out before we get
   * through with the login process.  Oh well.
   */

#if (defined(BSD) && (BSD >= 199103))
  sprintf(ttyname,"ftp%ld",(long)getpid());
#else
  sprintf(ttyname,"ftpd%d",(int)getpid());
#endif

  PRIVS_ROOT

  log_wtmp(ttyname,session.user,session.c->remote_name,
           session.c->remote_ipaddr);

  /* Open the /var/run log for later writing */
  log_open_run(mpid,FALSE);
  /* Open /var/log/ files */
  if(!xferlog && !(xferlog = get_param_ptr(main_server->conf,"TransferLog",FALSE)) )
    xferlog = XFERLOG_PATH;
  log_open_xfer(xferlog);

  initgroups(pw->pw_name,pw->pw_gid);

  PRIVS_RELINQUISH

  session.gids = make_array(session.pool,2,sizeof(int));
  session.groups = make_array(session.pool,2,sizeof(char*));

  if((gr = getgrgid(pw->pw_gid)) != NULL)
    *((char**)push_array(session.groups)) =
                         pstrdup(session.pool,gr->gr_name);

  setgrent();
  while((gr = getgrent()) != NULL && gr->gr_mem)
    for(gr_mem = gr->gr_mem; *gr_mem; gr_mem++) {
      if(!strcmp(*gr_mem,user)) {
        *((int*)push_array(session.gids)) = (int)gr->gr_gid;
	if(pw->pw_gid != gr->gr_gid)
          *((char**)push_array(session.groups)) = 
                         pstrdup(session.pool,gr->gr_name);
        break;
      }
    }

  /* Now check to see if the user has an applicable DefaultRoot */
  
  if(!c && (defroot = _get_default_root(session.pool))) {

    PRIVS_ROOT

    if(chroot(defroot) == -1) {

      PRIVS_RELINQUISH

      send_response(R_530,"Unable to set default root directory.");
      log_pri(LOG_ERR,"%s chroot(\"%s\"): %s",session.user,
              defroot,strerror(errno));
      end_login(1);
    }

    PRIVS_RELINQUISH

    session.anon_root = defroot;

    /* Re-calc the new cwd based on this root dir.  If not applicable
     * place the user in / (of defroot)
     */

    if(strncmp(session.cwd,defroot,strlen(defroot)) == 0) {
      char *newcwd = &session.cwd[strlen(defroot)];

      if(*newcwd == '/')
        newcwd++;
      session.cwd[0] = '/';

      strcpy(&session.cwd[1],newcwd);
    } else
      strcpy(session.cwd,"/");
  }

  PRIVS_ROOT

  if(c && chroot(session.anon_root) == -1) { 
    if(session.uid)
      initgroups(ourname,session.uid);

    PRIVS_RELINQUISH

    send_response(R_530,"Unable to set anonymous privileges.");
    log_pri(LOG_ERR,"%s chroot(): %s",session.user,strerror(errno));
    
    end_login(1);
  }

  if(setregid(pw->pw_gid,pw->pw_gid) == -1 ||
     setreuid(pw->pw_uid,pw->pw_uid) == -1) {

    PRIVS_RELINQUISH

    send_response(R_530,"Unable to set user privileges.");
    log_pri(LOG_ERR,"%s setregid() or setreuid(): %s",
            session.user,strerror(errno));

    end_login(1);
  }

  session.uid = pw->pw_uid;
  /* Overwrite original uid, so PRIVS_ macros no longer 
   * try to do anything 
   */
  session.ouid = pw->pw_uid;
  session.gid = pw->pw_gid;

  /* chdir to the proper directory, do this even if anonymous
   * to make sure we aren't outside our chrooted space.
   */

  if(chdir(session.cwd) == -1) {
    send_response(R_530,"Unable to chdir.");
    log_pri(LOG_ERR,"%s chdir(\"%s\"): %s",session.user,
            session.cwd,strerror(errno));
    end_login(1);
  }

  if(!getcwd(session.cwd,sizeof(session.cwd))) {
    send_response(R_530,"Unable to get current working directory.");
    log_pri(LOG_ERR,"%s getcwd(\"%s\"): %s",session.user,
            session.cwd,strerror(errno));
    end_login(1);
  } else
    dir_setcwd(session.cwd);

  if(c) {
    session.proc_prefix =
    pstrcat(permanent_pool,session.c->remote_name,
            ": anonymous/",pass,NULL);

    session.anon_config = c;
    session.flags = SF_ANON;
  } else {
    session.proc_prefix = pstrdup(permanent_pool,session.c->remote_name);
            
    session.flags = 0;
  }

  /* Default transfer mode is ASCII */
  session.flags |= SF_ASCII;
 
  /* Authentication complete, user logged in, now kill the login
   * timer.
   */

  main_set_idle();

  remove_timer(TIMER_LOGIN,&auth_module);
  return 1;
}
  
static char *cmd_user(cmd_rec *cmd)
{
  int nopass = 0,cur = 0;
  logrun_t *l;
  config_rec *c;
  char *user;

  if(cmd->argc != 2)
    return "500 'USER': command requires a parameter.";

  user = cmd->argv[1];

  remove_config(cmd->server->conf,C_USER,FALSE);
  add_config_param_set(&cmd->server->conf,C_USER,1,
                       pstrdup(cmd->server->pool,user));

  /* Determine how many users are currently connected */

  PRIVS_ROOT
  while((l = log_read_run(NULL)) != NULL)
    if(!strcmp(l->user,user))
      cur++;
  PRIVS_RELINQUISH

  remove_config(cmd->server->conf,"CURRENT-CLIENTS",FALSE);
  add_config_param_set(&cmd->server->conf,"CURRENT-CLIENTS",1,
                       (void*)cur);

  /* Try to determine what MaxClients applies to the user
   * (if any) and count through the runtime file to see
   * if this would exceed the max.
   */

  c = _auth_resolve_user(&user,NULL,NULL);

  if(c && get_param_int(c->subset,"AnonRequirePassword",FALSE) != 1)
      nopass++;

  c = find_config((c ? c->subset : cmd->server->conf),
                  CONF_PARAM,"MaxClients",FALSE);

  if(c) {
    int max = (int)c->argv[0];
    char *maxstr = "Sorry, maximum number of allowed clients (%m) already connected.";
    char maxn[10];

    sprintf(maxn,"%d",max);
    if(c->argc > 1)
      maxstr = c->argv[1];
    
    if(cur >= max) {
      send_response(R_530,"%s",
                    sreplace(cmd->tmp_pool,maxstr,"%m",maxn,NULL));

      log_auth(LOG_NOTICE,"connection refused (max clients %d)",
               max);
      end_login(0);
    }

  }

  /* Make sure the passwd file is open in case we chroot right away,
   * so we can still access it.
   */

  setpwent();
  setgrent();
  if(nopass)
    send_response(R_331,"Anonymous login ok, send your complete e-mail address as password.");
  else
    send_response(R_331,"Password required for %s.",cmd->argv[1]);
  return NULL;
}

static char *cmd_pass(cmd_rec *cmd)
{
  char *display = NULL;
  char *user,*grantmsg;
  int res = 0;

  if(cmd->argc < 2)
    return "500 'PASS': command requires a parameter.";

  user = (char*)get_param_ptr(cmd->server->conf,C_USER,FALSE);

  if(!user)
    return R_503 " Login with USER first.";

  
  if((res = _setup_environment(cmd->tmp_pool,user,cmd->arg)) == 1) {
    add_config_param_set(&cmd->server->conf,"authenticated",1,(void*)1);
    set_auth_check(NULL);

    if(session.flags & SF_ANON)
      display = (char*)get_param_ptr(session.anon_config->subset,
                                     "DisplayLogin",FALSE);
    if(!display)
      display = (char*)get_param_ptr(cmd->server->conf,
                                     "DisplayLogin",FALSE);

    if(display)
      core_display_file(R_230,display);

    if((grantmsg = 
        (char*)get_param_ptr((session.anon_config ? session.anon_config->subset :
                              cmd->server->conf),"AccessGrantMsg",FALSE)) != NULL) {
      grantmsg = sreplace(cmd->tmp_pool,grantmsg,"%u",user,NULL);

      send_response(R_230,"%s",grantmsg,NULL);
    } else {
      if(session.flags & SF_ANON)
        send_response(R_230,"Anonymous access granted, restrictions apply.");
      else
        send_response(R_230,"User %s logged in.",user);
    }

    return NULL;
  }

  remove_config(cmd->server->conf,C_USER,FALSE);

  if(res == 0) {
    int max;

    max = get_param_int(main_server->conf,"MaxLoginAttempts",FALSE);
    if(max == -1)
      max = 3;

    if(++auth_tries > max) {
      send_response(R_530,"Login incorrect");
      log_auth(LOG_NOTICE,"Maximum login attemps exceeded from %s [%s]",
               session.c->remote_name,
               inet_ntoa(*session.c->remote_ipaddr));
      end_login(0);
    }

    return R_530 " Login incorrect.";
  }

  return NULL;
}

cmdtable auth_commands[] = {
  { C_USER,	G_NONE,		cmd_user,	FALSE,	FALSE,	NULL },
  { C_PASS,	G_NONE,		cmd_pass,	FALSE,  FALSE,	NULL },
  { NULL,	NULL,		NULL,		FALSE,	FALSE,  NULL }
};

/* Module interface */

module auth_module = {
  NULL,NULL,				/* Always NULL */
  0x10,					/* API Version 1.0 */
  "auth",
  NULL,					/* No config in mod_auth */
  auth_commands,
  auth_init,auth_init_child
};

