/* code to manage the field star menu and operation.
 */

#include <stdio.h>
#include <math.h>

#include <Xm/Form.h>
#include <Xm/ToggleB.h>
#include <Xm/PushB.h>
#include <Xm/TextF.h>
#include <Xm/Label.h>
#include <Xm/Separator.h>

#ifdef __STDC__
#include <stdlib.h>
typedef const void * qsort_arg; 
#else
typedef void * qsort_arg;
#endif

#include "P_.h"
#include "astro.h"
#include "circum.h"

#define	DUPSEP		degrad(2/3600.)	/* dup if < this close, rads */
#define	DUPMAG		(3.0)		/* dup if mags differ < this much */
#define	GSCFOVLIM	degrad(15.0)	/* max fov we'll fet */

extern Widget toplevel_w;
#define XtD XtDisplay(toplevel_w)
extern Colormap xe_cm;

extern Now *mm_get_now P_((void));
extern char *syserrstr P_((void));
extern double delra P_((double dra));
extern int GSCFetch P_((double ra0, double dec0, double fov, double fmag,
    ObjF **opp, int nopp, char msg[]));
extern int GSCSetup P_((char *cdp, char *chp, char sep, char msg[]));
extern int existsh P_((char *name));
extern int gscesofetch P_((char *host, Now *np, double ra, double dec,
    double fov, double mag, ObjF **opp, int nop, char *msg));
extern int gscnetfetch P_((char *host, Now *np, double ra, double dec,
    double fov, double mag, ObjF **opp, int nop, char *msg));
extern int obj_cir P_((Now *np, Obj *op));
extern int ppmchkfile P_((char *file, char *msg));
extern int ppmfetch P_((char *file, Now *np, double ra, double dec,
    double fov, double mag, int saohd, ObjF **opp, char *msg));
extern Obj *db_scan P_((DBScan *sp));  
extern void db_scaninit P_((DBScan *sp, int mask, ObjF *op, int nop));
extern void db_update P_((Obj *op));
extern void hlp_dialog P_((char *tag, char *deflt[], int ndeflt));
extern void prompt_map_cb P_((Widget w, XtPointer client, XtPointer call));
extern void set_something P_((Widget w, char *resource, XtArgVal value));
extern void set_xmstring P_((Widget w, char *resource, char *txt));
extern void watch_cursor P_((int want));
extern void wtip P_((Widget w, char *tip));
extern void xe_msg P_((char *msg, int app_modal));

static int objfd_cmp P_((qsort_arg v1, qsort_arg v2));
static int fs_nodbdups P_((ObjF *fop, int nfop, double ra, double dec,
    double fov));
static int scanchk P_((Obj *mop, Obj *op, double cdec));
static void fs_create_fsform P_((void));
static void fs_save P_((void));
static void fs_setup P_((void));
static void apply_cb P_((Widget w, XtPointer client, XtPointer call));
static void ok_cb P_((Widget w, XtPointer client, XtPointer call));
static void tb_cb P_((Widget w, XtPointer client, XtPointer call));
static void fsprocess P_((void));
static void cancel_cb P_((Widget w, XtPointer client, XtPointer call));
static void help_cb P_((Widget w, XtPointer client, XtPointer call));

static Widget fsform_w;		/* the main form dialog */
static Widget cdtf_w, cdtb_w;	/* GSC CD-ROM text field and toggle button */
static Widget nettf_w, nettb_w;	/* GSC network text field and toggle button */
static Widget chtf_w, chtb_w;	/* GSC Cache text field and toggle button */
static Widget etf_w, etb_w;	/* ESO text field and toggle button */
static Widget ppmtf_w, ppmtb_w;	/* PPM/SAO text field and toggle button */
static Widget saohdtb_w;	/* PPM TB on when want SAO/HD names */
static Widget nodupt_w;		/* no dups toggle button */

/* code to pass to the TB callbacks.
 * used to enforce that net can't be used with CDROM or cache.
 */
enum TB {CDROMTB, CACHETB, NETTB, ESOTB};

static int cd_on;		/* whether GSC CDROM is currently on */
static char *cd_dn;		/* GSC CDROM dir */
static int ch_on;		/* whether GSC Cache is currently on */
static char *ch_dn;		/* GSC Cache dir */
static int net_on;		/* whether GSC network connection is on */
static char *net_host;		/* GSC network host name */
static int e_on;		/* whether ESO network connection is on */
static char *e_host;		/* ESO network host name */
static int ppm_on;		/* whether ppm is currently on */
static char *ppm_fn;		/* ppm filename */
static int wantsaohd;		/* whether we want to use SAO/HD names */
static int nodup_on;		/* whether dup checking is on */

/* call to set up without actually bringing up the menus.
 */
void
fs_create()
{
	if (!fsform_w) {
	    fs_create_fsform();
	    fs_save();
	}
}

void
fs_manage()
{
	if (!fsform_w) {
	    fs_create_fsform();
	    fs_save();
	}

	if (XtIsManaged(fsform_w))
	    XtUnmanageChild (fsform_w);
	else {
	    fs_setup();
	    XtManageChild (fsform_w);
	}
}

/* call to fetch a set of ObjF from the field star sources.
 * don't complain just if none are turned on at the moment.
 * we may eliminate any that seem to be dups of existing FIXED db objects.
 * we turn off any option that appears to not be working.
 * each o_flag in each resulting star is marked with FLDSTAR.
 * return -1 if something looks busted else count.
 * N.B. we only malloc *opp if we return > 0.
 */
int
fs_fetch (np, ra, dec, fov, mag, opp)
Now *np;	/* now */
double ra, dec;	/* at np->n_epoch */
double fov;	/* field of view, rads */
double mag;	/* limiting mag */
ObjF **opp;	/* we set *opp to a malloced list of ObjF */
{
	ObjF *op = NULL;
	int nop = 0;
	int s = 0;
	char msg[256];
	double gscfov;
	int i;

	if (!cd_on && !ch_on && !net_on && !e_on && !ppm_on)
	    return (-1);

	watch_cursor (1);

	/* the fetch tools all want and return J2000 values */
	if (epoch != J2000)
	    precess (epoch == EOD ? mjd : epoch, J2000, &ra, &dec);

	/* get the PPM stars, if desired */
	if (ppm_on) {
	    nop = ppmfetch (ppm_fn, np, ra, dec, fov, mag, wantsaohd, &op, msg);
	    if (nop < 0)
		ppm_on = 0;
	}

	/* now the GSC, if desired -- enforce max fov */
	gscfov = fov > GSCFOVLIM ? GSCFOVLIM : fov;
	msg[0] = '\0';
	if (net_on) {
	    nop = gscnetfetch (net_host, np, ra, dec, gscfov, mag, &op,nop,msg);
	    if (nop < 0)
		net_on = 0;
	} else if (e_on) {
	    nop = gscesofetch (e_host, np, ra, dec, gscfov, mag, &op, nop, msg);
	    if (nop < 0)
		e_on = 0;
	} else if (cd_on || ch_on) {
	    nop = GSCFetch (ra, dec, gscfov, mag, &op, nop, msg);
	    if (nop < 0)
		cd_on = 0;
	}
	if (msg[0])
	    xe_msg (msg, 1);

	/* set the s_fields to np */
	for (i = 0; i < nop; i++)
	    obj_cir (np, (Obj *)&op[i]);

	/* squeeze out stars which duplicate each other or the main database.
	 */
	if (nodup_on && nop > 0) {
	    int newnop = fs_nodbdups (op, nop, ra, dec, fov);
	    if (newnop < nop) {
		/* shrink down to just newnop entries now */
		nop = newnop;
		if (nop > 0)
		    op = (ObjF *) realloc ((void *)op, nop*sizeof(ObjF));
		else {
		    free ((void *)op);	/* *all* are dups! */
		    op = NULL;
		    nop = 0;
		}
	    }
	} else {
	    /* still set the FLDSTAR flag on all entries */
	    for (i = 0; i < nop; i++)
		((Obj *)(&op[i]))->o_flags |= FLDSTAR;
	}

	/* pass back the result, if there's something left */
	if (nop > 0) {
	    *opp = op;
	    s = nop;
	} else if (nop < 0) {
	    fs_setup();
	    s = -1;
	}

	watch_cursor (0);

	return (s);
}

/* called to put up or remove the watch cursor.  */
void
fs_cursor (c)
Cursor c;
{
	Window win;

	if (fsform_w && (win = XtWindow(fsform_w)) != 0) {
	    if (c)
		XDefineCursor (XtD, win, c);
	    else
		XUndefineCursor (XtD, win);
	}
}

/* qsort-style comparison of two ObjF's by dec */
static int
objfd_cmp (v1, v2)
qsort_arg v1, v2;
{
	Obj *op1 = (Obj *)v1;
	Obj *op2 = (Obj *)v2;
	double d = op1->s_dec - op2->s_dec;

	if (d < 0.0)
	    return (-1);
	if (d > 0.0)
	    return (1);
	return (0);
}

/* squeeze out all entries in fop[] which are located within DUPSEP and have a
 *   magnitude within DUPMAG of any FIXED object currently in the database. We
 *   also squeeze out any GSC stars which meet those same criteria for any PPM
 *   star.
 * when finished, all remaining entries will be contiguous at the front of the
 *   fop array and each will have FLDSTAR set in o_flag.
 * return the number of objects which remain.
 * N.B. we assume FLDSTAR bit is initially off in each o_flag.
 * N.B. we assume all fop[] have already been obj_cir()'d to mm_get_now().
 * N.B. we assume all entries in fop[] are within fov of ra/dec.
 */
static int
fs_nodbdups (fop, nfop, ra, dec, fov)
ObjF *fop;
int nfop;
double ra, dec, fov;
{
	double rov = fov/2;
	Obj *op;
	DBScan dbs;
	int l, u, m;
	Obj *mop;
	double diff;
	double cdec;

	/* sort fop by increasing dec */
	qsort ((void *)fop, nfop, sizeof(ObjF), objfd_cmp);

	/* mark all GSC stars with FLDSTAR flag which are close to any PPM.
	 * don't compare GSC and PPM stars with themselves.
	 */
	for (m = 0; m < nfop; m++) {
	    mop = (Obj *)&fop[m];

	    /* only check for GSC stars close to PPM stars, not v.v. */
	    if (mop->o_name[0] == 'G')
		continue;

	    /* scan each way from mop and mark close GSC stars */
	    cdec = cos(mop->s_dec);
	    for (u = m+1; u < nfop; u++) {
		op = (Obj *)&fop[u];
		if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
		    break;
		if (op->o_name[0] == 'G'
			&& cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
			&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
		    op->o_flags |= FLDSTAR;
	    }
	    for (l = m-1; l >= 0; l--) {
		op = (Obj *)&fop[l];
		if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
		    break;
		if (op->o_name[0] == 'G'
			&& cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
			&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
		    op->o_flags |= FLDSTAR;
	    }
	}

	/* scan all FIXED objects and mark all field stars with FLDSTAR
	 * which appears to be the same any one.
	 */
	for (db_scaninit(&dbs, FIXEDM, NULL, 0); (op = db_scan(&dbs)) != 0; ) {
	    /* update the s_ra/dec fields */
	    db_update (op);

	    /* skip ops outside the given field */
	    cdec = cos(op->s_dec);
	    if (fabs(op->s_dec - dec) > rov || cdec*delra(op->s_ra - ra) > rov)
		continue;

	    /* binary search to find the fop closest to op */
	    l = 0;
	    u = nfop - 1;
	    while (l <= u) {
		m = (l+u)/2;
		mop = (Obj *)(&fop[m]);
		diff = mop->s_dec - op->s_dec;
		if (diff < 0.0)
		    l = m+1;
		else
		    u = m-1;
	    }

	    /* scan each way from m and mark all that are dups with FLDSTAR.
	     * N.B. here, FLDSTAR marks *dups* (not the entries to keep)
	     * N.B. remember, u and l have crossed each other by now.
	     */
	    for (; l < nfop; l++)
		if (scanchk ((Obj *)(&fop[l]), op, cdec) < 0)
		    break;
	    for (; u >= 0; --u)
		if (scanchk ((Obj *)(&fop[u]), op, cdec) < 0)
		    break;
	}

	/* squeeze all entries marked with FLDSTAR to the front of fop[]
	 * and mark those that remain with FLDSTAR (yes, the bit now finally
	 * means what it says).
	 */
	for (u = l = 0; u < nfop; u++) {
	    mop = (Obj *)(&fop[u]);
	    if (!(mop->o_flags & FLDSTAR)) {
		mop->o_flags |= FLDSTAR; /* now the mark means a *keeper* */
		if (u > l)
		    memcpy ((void *)(&fop[l]), (void *)mop, sizeof(ObjF));
		l++;
	    }
	}

	return (l);
}

/* if mop is further than DUPSEP from op in dec, return -1.
 * then if mop is within DUPSEP in ra and within DUPMAG in mag too set FLDSTAR
 *   in mop->o_flags.
 * return 0.
 */
static int
scanchk (mop, op, cdec)
Obj *mop, *op;
double cdec;
{
	if (fabs(mop->s_dec - op->s_dec) >= DUPSEP)
	    return (-1);
	if (cdec*delra(mop->s_ra - op->s_ra) < DUPSEP
		&& fabs(get_mag(mop) - get_mag(op)) < DUPMAG)
	    mop->o_flags |= FLDSTAR;
	return (0);
}

static void
fs_create_fsform()
{
	Widget f, w;
	Arg args[20];
	int n;

	/* create form */
	n = 0;
	XtSetArg (args[n], XmNautoUnmanage, False); n++;
	XtSetArg (args[n], XmNmarginHeight, 10); n++;
	XtSetArg (args[n], XmNmarginWidth, 10); n++;
	XtSetArg (args[n], XmNverticalSpacing, 10); n++;
	XtSetArg (args[n], XmNcolormap, xe_cm); n++;
	fsform_w = XmCreateFormDialog (toplevel_w, "FieldStars", args, n);
	set_something (fsform_w, XmNcolormap, (XtArgVal)xe_cm);
	XtAddCallback (fsform_w, XmNmapCallback, prompt_map_cb, NULL);
        XtAddCallback (fsform_w, XmNhelpCallback, help_cb, NULL);

	/* set some stuff in the parent DialogShell.
	 * setting XmNdialogTitle in the Form didn't work..
	 */
	n = 0;
	XtSetArg (args[n], XmNtitle, "xephem Field Stars Setup"); n++;
	XtSetValues (XtParent(fsform_w), args, n);

	/* make the GSC CD-ROM toggle/text */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 45); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	cdtb_w = XmCreateToggleButton (fsform_w, "GSCCD", args, n);
	XtAddCallback (cdtb_w,XmNvalueChangedCallback,tb_cb,(XtPointer)CDROMTB);
	set_xmstring (cdtb_w, XmNlabelString, "GSC CDROM Directory:");
	wtip (cdtb_w, "Whether to use the CDROM for GSC stars");
	XtManageChild (cdtb_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 48); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	cdtf_w = XmCreateTextField (fsform_w, "GSCCDDirectory", args, n);
	wtip (cdtf_w, "Pathname to the root of the mounted CDROM");
	XtManageChild (cdtf_w);

	/* make the GSC Cache toggle/text */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, cdtf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 45); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	chtb_w = XmCreateToggleButton (fsform_w, "GSCCache", args, n);
	XtAddCallback (chtb_w,XmNvalueChangedCallback,tb_cb,(XtPointer)CACHETB);
	set_xmstring (chtb_w, XmNlabelString,  "GSC Cache Directory:");
	wtip (chtb_w, "Whether to use a local disk for GSC stars");
	XtManageChild (chtb_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, cdtf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 48); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	chtf_w = XmCreateTextField (fsform_w, "GSCCacheDirectory", args, n);
	wtip (chtf_w, "Pathname to the GSC subdirectory");
	XtManageChild (chtf_w);

	/* make the GSC network toggle/text */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, chtf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 45); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	nettb_w = XmCreateToggleButton (fsform_w, "GSCNet", args, n);
	XtAddCallback (nettb_w, XmNvalueChangedCallback,tb_cb,(XtPointer)NETTB);
	set_xmstring (nettb_w, XmNlabelString, "GSC Network Host Name:");
	wtip (nettb_w, "Whether to use the Internet to fetch GSC stars");
	XtManageChild (nettb_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, chtf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 48); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	nettf_w = XmCreateTextField (fsform_w, "GSCNetHost", args, n);
	wtip (nettf_w, "Host name from which to fetch GSC stars");
	XtManageChild (nettf_w);

	/* make the ESO network toggle/text */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, nettf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 45); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	etb_w = XmCreateToggleButton (fsform_w, "ESONet", args, n);
	XtAddCallback (etb_w, XmNvalueChangedCallback,tb_cb,(XtPointer)ESOTB);
	set_xmstring (etb_w, XmNlabelString, "ESO Network Host Name:");
	wtip (etb_w, "Whether to use the Internet to ESO to fetch GSC stars");
	XtManageChild (etb_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, nettf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 48); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	etf_w = XmCreateTextField (fsform_w, "ESONetHost", args, n);
	wtip (etf_w, "Host name at ESO from which to fetch GSC stars");
	XtManageChild (etf_w);

	/* make the PPM toggle/text */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, etf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 45); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	ppmtb_w = XmCreateToggleButton (fsform_w, "PPM", args, n);
	set_xmstring (ppmtb_w, XmNlabelString, "PPM Filename: ");
	wtip (ppmtb_w, "Whether to access a PPM catalog");
	XtManageChild (ppmtb_w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, etf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 48); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	ppmtf_w = XmCreateTextField (fsform_w, "PPMFilename", args, n);
	wtip (ppmtf_w, "Pathname of ppm.xe file");
	XtManageChild (ppmtf_w);

	/* make the PPM/SAO name toggle somewhat indented */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, ppmtf_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 10); n++;
	XtSetArg (args[n], XmNalignment, XmALIGNMENT_BEGINNING); n++;
	saohdtb_w = XmCreateToggleButton (fsform_w, "SAOHD", args, n);
	set_xmstring(saohdtb_w,XmNlabelString,"Use SAO or HD Name if Assigned");
	wtip (saohdtb_w,"Use SAO or HD designation instead of PPM if possible");
	XtManageChild (saohdtb_w);

	/* make the No dups toggle under a sep */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, saohdtb_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (fsform_w, "Sep", args, n);
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNleftPosition, 30); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	XtSetArg (args[n], XmNrightPosition, 70); n++;
	nodupt_w = XmCreateToggleButton (fsform_w, "NoDups", args, n);
	set_xmstring (nodupt_w, XmNlabelString, "No Duplicates");
	wtip (nodupt_w, "Skip objects which appear to already be in memory");
	XtManageChild (nodupt_w);

	/* make the controls across the bottom under a separator */

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, nodupt_w); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	w = XmCreateSeparator (fsform_w, "Sep", args, n);
	XtManageChild (w);

	n = 0;
	XtSetArg (args[n], XmNtopAttachment, XmATTACH_WIDGET); n++;
	XtSetArg (args[n], XmNtopWidget, w); n++;
	XtSetArg (args[n], XmNbottomAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNleftAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNrightAttachment, XmATTACH_FORM); n++;
	XtSetArg (args[n], XmNfractionBase, 13); n++;
	f = XmCreateForm (fsform_w, "CF", args, n);
	XtManageChild (f);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 1); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 3); n++;
	    w = XmCreatePushButton (f, "Ok", args, n);
	    wtip (w, "Test settings and close");
	    XtAddCallback (w, XmNactivateCallback, ok_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 4); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 6); n++;
	    w = XmCreatePushButton (f, "Apply", args, n);
	    wtip (w, "Test settings and remain up");
	    XtAddCallback (w, XmNactivateCallback, apply_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 7); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 9); n++;
	    w = XmCreatePushButton (f, "Close", args, n);
	    wtip (w, "Close this menu without doing anything");
	    XtAddCallback (w, XmNactivateCallback, cancel_cb, NULL);
	    XtManageChild (w);

	    n = 0;
	    XtSetArg (args[n], XmNleftAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNleftPosition, 10); n++;
	    XtSetArg (args[n], XmNrightAttachment, XmATTACH_POSITION); n++;
	    XtSetArg (args[n], XmNrightPosition, 12); n++;
	    w = XmCreatePushButton (f, "Help", args, n);
	    wtip (w, "More detailed descriptions");
	    XtAddCallback (w, XmNactivateCallback, help_cb, NULL);
	    XtManageChild (w);
}

/* set up the dialog according to our static state */
static void
fs_setup ()
{
	XmToggleButtonSetState (ppmtb_w, ppm_on, False);
	XmToggleButtonSetState (saohdtb_w, wantsaohd, False);
	if (ppm_fn)
	    XmTextFieldSetString (ppmtf_w, ppm_fn);

	XmToggleButtonSetState (cdtb_w, cd_on, False);
	if (cd_dn)
	    XmTextFieldSetString (cdtf_w, cd_dn);

	XmToggleButtonSetState (nettb_w, net_on, False);
	if (net_host)
	    XmTextFieldSetString (nettf_w, net_host);

	XmToggleButtonSetState (etb_w, e_on, False);
	if (e_host)
	    XmTextFieldSetString (etf_w, e_host);

	XmToggleButtonSetState (chtb_w, ch_on, False);
	if (ch_dn)
	    XmTextFieldSetString (chtf_w, ch_dn);

	XmToggleButtonSetState (nodupt_w, nodup_on, False);
}

/* save the dialog as our static state.
 * we test the new settings and might change them if they seem to be
 *   unreasonable.
 */
static void
fs_save ()
{
	char msg[1024];

	watch_cursor (1);

	/* PPM */
	ppm_on = XmToggleButtonGetState (ppmtb_w);
	wantsaohd = XmToggleButtonGetState (saohdtb_w);
	if (ppm_fn)
	    XtFree (ppm_fn);
	ppm_fn = XmTextFieldGetString (ppmtf_w);
	if (ppm_on && ppmchkfile (ppm_fn, msg) < 0) {
	    xe_msg (msg, 1);
	    ppm_on = 0;
	    fs_setup();
	}

	/* CDROM and Cache */
	cd_on = XmToggleButtonGetState (cdtb_w);
	if (cd_dn)
	    XtFree (cd_dn);
	cd_dn = XmTextFieldGetString (cdtf_w);
	ch_on = XmToggleButtonGetState (chtb_w);
	if (ch_dn)
	    XtFree (ch_dn);
	ch_dn = XmTextFieldGetString (chtf_w);
	if ((cd_on || ch_on) && GSCSetup (cd_on?cd_dn:NULL, ch_on?ch_dn:NULL,
								'/', msg) < 0) {
	    xe_msg (msg, 1);
	    cd_on = 0;
	    ch_on = 0;
	    fs_setup ();
	}

	/* xephem network connection */
	net_on = XmToggleButtonGetState (nettb_w);
	if (net_host)
	    XtFree (net_host);
	net_host = XmTextFieldGetString (nettf_w);
	if (net_on) {
	    /* test by actually fetching a small patch */
	    Now *np = mm_get_now();
	    double fov = degrad(1);
	    double mag = 13.0;
	    int nop;
	    ObjF *op;

	    op = NULL;
	    nop = gscnetfetch (net_host, np, 0.0, 0.0, fov, mag, &op, 0, msg);
	    if (nop < 0) {
		xe_msg (msg, 1);
		net_on = 0;
		fs_setup ();
	    } else if (nop > 0)
		free ((void *)op);
	}

	/* ESO connection */
	e_on = XmToggleButtonGetState (etb_w);
	if (e_host)
	    XtFree (e_host);
	e_host = XmTextFieldGetString (etf_w);
	if (e_on) {
	    /* test by actually fetching a small patch */
	    Now *np = mm_get_now();
	    double fov = degrad(1);
	    double mag = 13.0;
	    int nop;
	    ObjF *op;

	    op = NULL;
	    nop = gscesofetch (e_host, np, 0.0, 0.0, fov, mag, &op, 0, msg);
	    if (nop < 0) {
		xe_msg (msg, 1);
		e_on = 0;
		fs_setup ();
	    } else if (nop > 0)
		free ((void *)op);
	}

	watch_cursor (0);

	nodup_on = XmToggleButtonGetState (nodupt_w);
}

/* called from Apply */
/* ARGSUSED */
static void
apply_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	fsprocess();
}

/* called from Ok */
/* ARGSUSED */
static void
ok_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	fsprocess();

	/* outta here */
	XtUnmanageChild (fsform_w);
}

/* handle the affirmative actions of ok or apply */
static void
fsprocess()
{
	/* save dialog state */
	fs_save();
}

/* called from Ok */
/* ARGSUSED */
static void
cancel_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	/* outta here */
	XtUnmanageChild (fsform_w);
}

/* called from any of the CDROM, network or Cache toggle buttons.
 * client is one of the TB enums to tell us which.
 */
/* ARGSUSED */
static void
tb_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
	if (XmToggleButtonGetState(w))
	    switch ((int)client) {
	    case CDROMTB:	/* FALLTHRU */
	    case CACHETB:
		/* turn off net and ESO if turning on CDROM or Cache */
		XmToggleButtonSetState (nettb_w, False, False);
		XmToggleButtonSetState (etb_w, False, False);
		break;
	    case NETTB:
		/* turn off CDROM and Cache and ESO if turning on network */
		XmToggleButtonSetState (cdtb_w, False, False);
		XmToggleButtonSetState (chtb_w, False, False);
		XmToggleButtonSetState (etb_w, False, False);
		break;
	    case ESOTB:
		/* turn off CDROM and Cache and net if turning on ESO */
		XmToggleButtonSetState (cdtb_w, False, False);
		XmToggleButtonSetState (chtb_w, False, False);
		XmToggleButtonSetState (nettb_w, False, False);
		break;
	    default:
		printf ("FS: bad client: %d\n", (int)client);
		exit(1);
	    }
}

/* called from Ok */
/* ARGSUSED */
static void
help_cb (w, client, call)
Widget w;
XtPointer client;
XtPointer call;
{
        static char *msg[] = {
"Set up which field star sources to use and where they are located."
};

	hlp_dialog ("FieldStars", msg, sizeof(msg)/sizeof(msg[0]));

}
