/*
 **************************************************************************
 *
 * Boot-ROM-Code to load an operating system across a TCP/IP network.
 *
 * Module:  menu.c
 * Purpose: Display a menu of bootimage choices
 * Entries: domenu
 *
 **************************************************************************
 *
 * Copyright (C) 1995,1996 Gero Kuhlmann <gero@gkminix.han.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include "../../headers/general.h"
#include "../../headers/memory.h"
#include "../public/net.h"
#include "../public/arpa.h"
#include "../public/romlib.h"
#include "./bootpriv.h"
#include "./menu.h"


/*
 **************************************************************************
 * 
 * Internal representation of each bootimage description
 */
struct image {
  int		tagnum;			/* image tag number		*/
  unsigned char	label[MAX_LABEL_LEN];	/* label for image		*/
  t_ipaddr	server;			/* server IP address		*/
  t_ipaddr	gateway;		/* gateway IP address		*/
  unsigned char filename[MAX_FNAM_LEN];	/* bootimage file name		*/
};



/*
 **************************************************************************
 * 
 * Global variables:
 */
static struct image def_image;		/* default image from BOOTP	*/
static int def_timeout;			/* default keyboard timeout	*/
static int def_choice;			/* default menu choice		*/
static int imagenum;			/* number of menu items		*/
static struct image *imagelist;		/* list of menu items		*/



/*
 **************************************************************************
 * 
 * Get a numeric response from the user
 */
static int getselect(void)
{
  int num = -1;
  int pos = 0;
  int i;
  char ch;

  /* Clear keyboard buffer */
  while (chkkey() >= 0) ;

  /* Main loop */
  while (TRUE) {
	if ((ch = getkey(def_timeout * 18)) < 0 || ch == CHR_CR)
		return(num < 0 ? def_choice : num);
	else if (ch == CHR_ESC)
		return(-1);
	else if (ch == CHR_BS) {
		if (pos > 0) {
			printf("\b \b");
			num = num / 10;
			pos--;
		}
		if (pos == 0)
			num = -1;
		continue;
	} else if (ch < '0' || ch > '9')
		continue;
	i = ch - '0';
	if (num >= 0)
		i += num * 10;
	if (i >= imagenum)
		continue;
	num = i;
	printf("%c", ch);
	pos++;
  }
}



/*
 **************************************************************************
 * 
 * Decode image file description tag
 */
static int decode_image(unsigned char *cp, int tagnum)
{
  unsigned char *bp;
  int strlen = *cp++;
  int i;

  /* Copy tag number */
  imagelist[imagenum].tagnum = tagnum;

  /* Copy label */
  bp = imagelist[imagenum].label;
  for (i = 0; *cp && *cp != ':' && i < MAX_LABEL_LEN - 1 && strlen > 0; i++) {
	*bp++ = *cp++;
	strlen--;
  }
  *bp = '\0';
  /* Skip until next colon */
  for ( ; *cp && *cp != ':' && strlen > 0; cp++, strlen--) ;
  if (!*cp || strlen == 0)
	return(FALSE);
  cp++;
  strlen--;

  /* Copy server IP address */
  bp = cp;
  for ( ; *cp && *cp != ':' && strlen > 0; cp++, strlen--) ;
  if (!*cp || strlen == 0)
	return(FALSE);
  if (*bp == ':')
	imagelist[imagenum].server = IP_ANY;
  else if ((imagelist[imagenum].server = resolve((char *)bp)) == IP_ANY)
	return(FALSE);
  cp++;
  strlen--;

  /* Copy gateway IP address */
  bp = cp;
  for ( ; *cp && *cp != ':' && strlen > 0; cp++, strlen--) ;
  if (!*cp || strlen == 0)
	return(FALSE);
  if (*bp == ':')
	imagelist[imagenum].gateway = IP_ANY;
  else if ((imagelist[imagenum].gateway = resolve((char *)bp)) == IP_ANY)
	return(FALSE);
  cp++;
  strlen--;

  /* Copy filename */
  bp = imagelist[imagenum].filename;
  for (i = 0; *cp && i < MAX_FNAM_LEN - 1 && strlen > 0; i++) {
	*bp++ = *cp++;
	strlen--;
  }
  *bp = '\0';
  if (*cp && strlen != 0)
	return(FALSE);

  imagenum++;
  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Decode default parameter tag
 */
static int decode_params(unsigned char *cp)
{
  int *resp;
  int strlen = *cp++;
  int i, offs;

  /* Decode the parameter tag string */
  while (*cp && strlen > 0) {
	/* Check for valid strings */
	if (!memcmp(cp, "timeout=",  8)) {
		offs = 8;
		resp = &def_timeout;
	} else if (!memcmp(cp, "default=", 8)) {
		offs = 8;
		resp = &def_choice;
	} else
		return(FALSE);
	/* Decode any number following the equal sign */
	cp += offs;
	strlen -= offs;
	i = 0;
	while (strlen > 0 && *cp >= '0' && *cp <= '9') {
		i = i * 10 + (*cp++ - '0');
		strlen--;
	}
	if ((*cp && *cp != ':') || strlen < 0)
		return(FALSE);
	*resp = i;
	/* Skip colon */
	cp++;
	strlen--;
  }

  /* Adjust default choice */
  if (def_choice >= MENU_IMG_FIRST) {
	for (i = 0; i < imagenum; i++)
		if (imagelist[i].tagnum == def_choice) {
			def_choice = i;
			break;
		}
  }
  if (def_choice >= imagenum)
	return(FALSE);

  return(TRUE);
}



/*
 **************************************************************************
 * 
 * Display a menu of bootimage choices
 */
int domenu(struct bootp *bp)
{
  int i;
  unsigned char *cp;

  /*
   * Check the menu magic number in the BOOTP parameter area, including
   * the menu definition version number.
   */
  if ((cp = get_vend(bp, MENU_ID)) == NULL || *cp != 6 ||
      *((unsigned long *)(cp + 1)) != MENU_MAGIC)
	return(MENU_INVALID); 

#if MENU_VER_MINOR > 0
  if (*(cp + 5) != MENU_VER_MAJOR || *(cp + 6) > MENU_VER_MINOR) {
#else
  if (*(cp + 5) != MENU_VER_MAJOR) {
#endif
	printf("MENU: Invalid version number\n");
	return(MENU_INVALID);
  }

  /*
   * Initialize the name resolver to allow for symbolic names in BOOTP
   * tags.
   */
  res_config(bp);

 /*
  * Decode all parameters in the BOOTP block
  */
  if (imagelist == NULL) {
	/*
	 * Initialize the default image description from the BOOTP block. This
	 * is necessary to be able to restore the BOOTP block with repetitive
	 * calls to this function.
	 */
	memcpy(def_image.filename, bp->bp_sname, BOOTP_SNAME_SIZE);
	def_image.server = ntohl(bp->bp_siaddr);
	def_image.gateway = IP_ANY;
	if((cp = get_vend(bp, VEND_ROUTER)) != NULL)
		def_image.gateway = ntohl(*((t_ipaddr *)(cp + 1)));

	/* Get enough memory for internal tables */
	if ((imagelist = malloc(sizeof(struct image) * MENU_IMG_NUM)) == NULL) {
		printf("MENU: Not enough memory for image list\n");
		return(MENU_INVALID);
	}

	/* Decode the image file descriptions */
	for (i = MENU_IMG_FIRST; i <= MENU_IMG_LAST; i++)
		if ((cp = get_vend(bp, i)) != NULL && !decode_image(cp, i)) {
			printf("MENU: Invalid image description %d\n", i);
			goto errend;
		}

	/* Decode the default parameter tag */
	if ((cp = get_vend(bp, MENU_PARAMS)) != NULL && !decode_params(cp)) {
		printf("MENU: Invalid TAG %d\n", MENU_PARAMS);
errend:
		free(imagelist);
		imagelist = NULL;
		return(MENU_INVALID);
	}
  }

  /* Print the message strings */
  printf("\n\n");
  for (i = MENU_DISP_FIRST; i <= MENU_DISP_LAST; i++)
	if ((cp = get_vend(bp, i)) != NULL) {
		int len = *cp++;
		printf("%ls\n", cp, len);
	}

  /* Print the menu and wait for a user keypress */
  if (imagenum > 0) {
	printf("\n\n");
	for (i = 0; i < imagenum; i++)
		printf("[%s%d]\t%s\n", i < 10 ? " ":"", i, imagelist[i].label);
	printf("\nSelect a choice: ");
	i = getselect();
	printf("\n\n");
	if (i < 0 || !imagelist[i].filename[0])
		return(MENU_ABORT);
	memcpy(bp->bp_file, imagelist[i].filename, BOOTP_FILE_SIZE);
	/* Set IP of tftp server */
	if (imagelist[i].server != IP_ANY) {
		bp->bp_siaddr = htonl(imagelist[i].server);
		memset(bp->bp_sname, 0, sizeof(bp->bp_sname));
	} else {
		bp->bp_siaddr = htonl(def_image.server);
		memcpy(bp->bp_sname, def_image.filename, BOOTP_SNAME_SIZE);
	}
	/* Set gateway IP to reach the server */
	if (imagelist[i].gateway != IP_ANY)
		set_gateway(imagelist[i].gateway);
	else
		set_gateway(def_image.gateway);
	printf("Selected %s\n\n", imagelist[i].label);
  }

  return(MENU_OK);
}

