/*   Conversion of LaTeX files to Lyx

    Author: John Collins, collins@phys.psu.edu.
    22 Jan 96

    (C) John Collins & Penn State University.
*/

#include "texparse.h"
#include <string.h>



//==================================


void HelpMsg (sos & f = lite_cout){
   f <<
   "Usage:\n\n"
   "    textolyx texfile [lyxfile]\n\n"
   "parses texfile, and writes result on lyxfile.  If lyxfile is not "
   "specified, the output is written to stdout.\n\n"
   "    textolyx -\n\n"
   "takes its input from stdin, i.e., it works as a filter. E.g.:\n\n"
   "    textolyx - <texfile >lyxfile\n\n"
   "Error messages are always written to stderr.\n";
}


int FindLyxDoc (ListHead &Logic,
                ListHead &PrePreamble, ListHead &Preamble,
                ListHead &Body, ListHead &Tail) {
/* From the parsed LaTeX source in Logic, make the components of a Lyx
   document: Prepreamble, Preamble, Body, Tail.
   Return zero if have success, nonzero otherwise.

   PrePreamble is the material before the \documentclass line.
   Preamble is the material between the \documentclass line and the
      document's body.
   Body is the main material (between \begin{document} and \end{document}.
   Tail is the material after the \end{document}.

   */

   BaseTeXObject *Current = &Logic,
                 *DocClassLine = NULL,
                 *DocEnv = NULL;

   // Find the \documentclass line:
   while (Current && (Current->IsA() != IDDocClass) )
      Current = Current->next;
   if (!Current) {
      lite_cerr << "No \\documentclass line near head of file.\n";
      return 31;
   } else {
      lite_cerr << "Found \\documentclass line.\n";
      DocClassLine = Current;
   }


   // Find the Body of the document:
   while (Current && (Current->IsA() != IDDocumentBody)) {
      Current = Current->next;
   }
   if (Current == NULL) {
     lite_cerr << "Couldn't find document body after \\documentclass line.\n";
     return 32;
   }
   lite_cerr << "Found document body.\n";
   DocEnv = Current;


   /* Bundle up the stuff before the \documentclass line (prepreamble),
      between \documentclass line and body (preamble),
      the body,
      and the tail (after the \end{document}).
   */
   if (Logic.next != DocClassLine) {
     MoveTo (PrePreamble, Logic.next, DocClassLine->prev);
   }
   if (DocClassLine->next != DocEnv) {
     MoveTo (Preamble, DocClassLine, DocEnv->prev);
   }
   Current = DocEnv -> next;
   MoveTo (Body, DocEnv, DocEnv);
   // Skip Blank
   while (Current && (Current->IsA() == IDPar))
      Current = Current->next;
   if (Current) {
     MoveTo (Tail, Current);
   }
   return 0;
}


int WriteLyxFile (sos &dest, ListHead &PrePreamble, ListHead &Preamble,
                              ListHead &Body, ListHead &Tail){
/* Write a Lyx document to the file dest, which is assumed to be open.
   Leave dest open.
   Return zero if have success, nonzero otherwise.
   Note that Lyx ignores newlines, so newlines may be inserted to make
      prettyprint the file (for easier debugging, etc).

   PrePreamble is the material before the \documentclass line.
   Preamble is the material between the \documentclass line and the
      document's body.
   Body is the main material (between \begin{document} and \end{document}.
   Tail is the material after the \end{document}.

*/

   LyxState ls;
   dest << "# Lyx file created by Latextolyx\n"
        << "\\lyxformat 2.10\n";
   dest << "\\begin_preamble\n";
   if (PrePreamble.next) {
      dest << "%#{Lyx}Prepreamble Material before the documentclass line\n";
      PrePreamble.WriteLyxAll (ls,dest);
      dest << "%#{Lyx}EndPrepreamble\n";
   }
   Preamble.WriteLyxAll (ls,dest);
   dest << "\n\\end_preamble\n\n";

   dest << "\\layout Standard\n";
   // Ensure paragraph has started.  ?? This is to compensate for
   // a bug in the WriteLyx methods that needs to be corrected.
   ls.InPar = 1;
   Body.WriteLyxAll (ls,dest);

   if (Tail.next) {
      lite_cerr << "There's material after the \\end{document}.\n";
      dest << "\n\n \\layout Section\n"
        << "======= MATERIAL FROM AFTER THE END OF LATEX DOCUMENT ======\n\n";
      Tail.WriteLyxAll (ls,dest);
   }

   if (dest.bad()) {
      lite_cerr << "Problems with writing a Lyx file.\n";
      return 1;
   } else {
      return 0;
   }
}


int MakeLyxFile (char * sourcename, char * destname){
/* Parse TeX file and make Lyx file.
   If sourcename==NULL, read from stdin.
   If destname==NULL, send the output to stdout.
   Return 0 on success, non-zero otherwise.
*/
   ListHead Logic;
   Buffer LineList;
   TeXState state;
   ListHead PrePreamble, Preamble, Body, Tail;
   int retcode = 0;  // for error codes

   if (sourcename) {
      FILE * source =fopen(sourcename, "r");
      if (source) {
         lite_cerr  << "Reading...\n";
         LineList.Read(source);
         fclose (source);
      } else {
         lite_cerr << "File \"" << sourcename << "\" does not exist.\n";
         return 11;
      }
   } else {
      LineList.Read(stdin);
   }
   if (LineList.ErrorCode != 0) {
      lite_cerr
         << "Error in reading the source file after "
         << (int)LineList.NumLastRead << " lines.\n";
      return 12;
   }
   lite_cerr <<
   "===========\n\nNumber of lines = " << (int)LineList.NumLines() << '\n';


   // Now split into elementary TeX objects.
   lite_cerr << "Parsing...\n";
   retcode = Logic.Tokenize (LineList);
   if (retcode) {
      lite_cerr << "Error code " << retcode << " from tokenizing.\n"
                   "I will not continue.\n";
      return 21;
   }
   // and parse the TeX
   // To see what I have: Logic.WriteTeXAll (lite_cerr);
   retcode = ParseAll(&Logic, state);
   if (retcode == 100) {
      lite_cerr << "I cannot continue with those errors.\n";
      return 22;
   } else if (retcode != 0){
      lite_cerr << "I will ignore the errors and try and make a useful file.\n";
   }



   retcode = FindLyxDoc(Logic, PrePreamble, Preamble, Body, Tail);
   if (retcode) {
      lite_cerr <<
         "Since I failed to find in the TeX file the document structure I\n"
         "need, I will not write the Lyx file.\n";
      return retcode;
   }


   if (destname) {
      fsos dest (destname);
      if (!dest.IsOpen()) {
         lite_cerr << "File \"" << destname << "\" cannot be written.\n";
         return 3;
      }
      WriteLyxFile (dest, PrePreamble, Preamble, Body, Tail);
      dest.close();
   } else {
      WriteLyxFile (lite_cout, PrePreamble, Preamble, Body, Tail);
   }
   return 0;
}


//=======================================================================
//=======================================================================

int main(int argc, char** argv) {
   char *sourcename = NULL, *destname = NULL;
   // NULL values will be used to flag stdin/stdout

   lite_cerr << "TeXtoLyX, version 0.1\n";
   if (argc <= 1){
      HelpMsg();
      return 1;
   } else if (strcmp(argv[1], "-") == 0){
      // Work as filter
   } else {
      sourcename = argv[1];
      if (argc >=3) destname = argv[2];
   }
   return MakeLyxFile (sourcename, destname);
}

