/* `gf256t.c' -- Routines to construct the Discrete Logarithm and
   AntiLogarithm tables for GF(2^8).

   Copyright (C) 1996 Free Software Foundation, Inc.

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

   Written by Fung-Chai Lim (fclim@acm.org).

   This file is part of GNU ecc.  */

#ifdef HAVE_CONFIG_H
#if defined (CONFIG_BROKETS)
/* We use <config.h> instead of "config.h" so that a compilation
   using -I. -I$srcdir will use ./config.h rather than $srcdir/config.h
   (which it would do because it found this file in $srcdir).  */
#include <config.h>
#else
#include "config.h"
#endif
#endif

#include <stdio.h>

static void make_log_table _PROTO_ ((void));
static void make_antilog_table _PROTO_ ((void));
static void print_table _PROTO_ ((const int *const table, FILE *file));

static int log_table [256];
static int antilog_table [256];

     void
make_tables ()
{
  make_antilog_table ();
  make_log_table ();
}

     int
substitute (string, file)
     const char *const string;
     FILE *file;
{
  if (strcmp (string, "ANTILOG") == 0)
    print_table (antilog_table, file);
  else if (strcmp (string, "LOG") == 0)
    print_table (log_table, file);
  else
    return 1;
  return 0;
}

/* The representation of Galois Field GF(2^8) used in this library is
   constructed from GF(2)[X] modulo P(x) where
     P(x) = x^8 + x^4 + x^3 + x^2 + 1
   which is a primitive polynomial.

   The polynomials chosen are
     f1(x) = 1
     f2(x) = x f1(x) modulo P(x)
     f3(x) = x f2(x) modulo P(x)
     ...
     fn(x) = x f[n-1](x) modulo P(x)
   where the modulo is done in GF(2)[X].

   The Discrete Antilogarithm table is the set of values of these polynomials
   evaluated at x = 2 which is a cyclic element of GF(2^8).  The Log table
   is a simple inverse.

   See, for instance, Richard Blahut's "Theory and Practice of Error Control
   Codes".  */

/* Returns 1 if POLY has degree equal to 8.  */
#if notdef
	/* This one is more correct.  But what the hey.  */
#define deg_equals_8_p(poly)	(((poly) < (1 << 9)) && ((poly) & (1 << 8)))
#else
#define deg_equals_8_p(poly)	((poly) & (1 << 8))
#endif

/* The primitive polynomial (x^8 + x^4 + x^3 + x^2 + 1).  */
#define prim_poly		((1 << 8) | (1 << 4) | (1 << 3) | (1 << 2) | 1)

/* This macro does, more accurately, a subtraction rather than a modulo.  */
#define modulo_prim(poly)	((poly) ^ prim_poly)

     static void
make_antilog_table ()
{
  int i;
  int poly;

  poly = 1;		/* f1(x) */
  for (i = 0; i < 256; i++)
    {
      antilog_table [i] = poly;
      poly <<= 1;	/* multiply by x = 2.  */
      /* The code to do modulo P(x) simplifies to just a subtraction if we
	 take the modulo everytime the degree of POLY reaches 8.  */
      if (deg_equals_8_p (poly))
	poly = modulo_prim (poly);
    }
}

     static void
make_log_table ()
{
  int i;

  log_table [0] = 255;	/* Infinity */
  for (i = 0; i < 255; i++)
    log_table [antilog_table [i]] = i;
}

     static void
print_table (table, file)
     const int *const table;
     FILE *file;
{
  int i;

  for (i = 0; i < 256; i++)
    {
      if ((i % 8) == 0)
	fprintf (file, "\n  ");
      fprintf (file, "%4d", table[i]);
      if (i != 255)
	fputc (',', file);
    }
  fputc ('\n', file);
}
