/* AX.25 Utilities: Attach an interface.
 * Bruce Perens, November 1994
 *
 * Copyright 1994 Bruce Perens.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 */
#include <stdlib.h>
#include <unistd.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <string.h>

#define	NAME_SIZE	64

static const char	Prototype[] = "/dev/pty";
#define	PROTOTYPE_BASE	5 /* index to "pty" in prototype. */

static int
open_pseudo_tty(const char * name, int * slave)
{
	char		slaveName[NAME_SIZE];
	int		master = open(name, O_RDWR, 0);
	struct termios	t;

	if ( master < 0 )
		return -1;

	strcpy(slaveName, name);
	
	slaveName[PROTOTYPE_BASE] = 't';

	/* Close master again to jettison any pernicious listeners on slave
	 * side. I'd like to be able to lock opens on the slave side while
	 * this is going on. I can't see any way to do that if I'm not root.
	 */
	chown(slaveName, 0, 0);
	chmod(slaveName, 0600);
	close(master);

	if ( (master = open(name, O_RDWR, 0)) < 0 )
		return -1;

	if ( (*slave = open(slaveName, O_RDWR, 0)) < 0 ) {
		close(master);
		return -1;
	}

	if ( tcgetattr(*slave, &t) == 0 ) {
		/*
		 * Attempt to provide a consistent environment upon open.
		 * Of course if you are running a script you can override
		 * this by running stty.
		 */
		t.c_cc[VINTR]	= 'c' & 0x1f;
		t.c_cc[VQUIT]	= '\\' & 0x1f;
		t.c_cc[VERASE]	= 'h' & 0x1f;
		t.c_cc[VKILL]	= 'u' & 0x1f;
		t.c_cc[VEOF]	= 'd' & 0x1f;
		t.c_cc[VEOL]	= '\n';
		t.c_cc[VSTOP]	= 's' & 0x1f;
		t.c_cc[VSTART]	= 'q' & 0x1f;
		t.c_cc[VSUSP]	= 'z' & 0x1f;
		t.c_cc[VLNEXT]	= 'v' & 0x1f;
		t.c_cc[VWERASE]	= 'w' & 0x1f;
		t.c_cc[VREPRINT]= 'r' & 0x1f;
		t.c_cc[VDISCARD]= 'o' & 0x1f;
		t.c_iflag = BRKINT|ICRNL;
		t.c_oflag = OPOST;
		t.c_cflag = B9600|CS8|CREAD|HUPCL;
		t.c_lflag = ISIG|ICANON|ECHO|ECHOE;
		t.c_line = 0;
		tcsetattr(*slave, TCSANOW, &t);
	}

	return master;
}

int
get_pseudo_tty(int * slave)
{
	char			name[NAME_SIZE];
	char * const		ones = &name[sizeof(Prototype)];
	char * const		tens = &name[sizeof(Prototype) - 1];

	strcpy(name, Prototype);
	name[sizeof(Prototype) + 1] = '\0';

	for ( *tens = 'p'; *tens <= 's'; ++*tens ) {
		int	n;
		for ( n = 0; n < 16; n++ ) {
			static const char Hexits[16] = "0123456789abcdef";
			int	master;

			*ones = Hexits[n];
			master = open_pseudo_tty(name, slave);

			if ( master >= 0 )
				return master;
		}
	}
	return -1;
}
