/* This software is Copyright 1995 by Karl-Johan Johnsson
 *
 * Permission is hereby granted to copy, reproduce, redistribute or otherwise
 * use this software as long as: there is no monetary profit gained
 * specifically from the use or reproduction of this software, it is not
 * sold, rented, traded or otherwise marketed, and this copyright notice is
 * included prominently in any copy made. 
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. ANY USE OF THIS
 * SOFTWARE IS AT THE USER'S OWN RISK.
 */
#include "global.h"
#include <sys/stat.h>
#include "child.h"
#include "codes.h"
#include "newsrc.h"
#include "resource.h"
#include "file.h"
#include "server.h"
#include "util.h"
#include "widgets.h"
#include "xutil.h"

#define NO_GROUPS	4096
#define NEW_GROUPS	 256

#define MAX_GROUP_LEN	1024

static int comp_group_nodes(const void *a, const void *b)
{
    return strcmp( (*((GROUP **)a))->name, (*((GROUP **)b))->name);
}

int get_newsgroups(void)
{
    char	*buffer;
    long	status;

    buffer = server_comm(main_server, "LIST\r\n", False);
    if (!buffer)
	return -1;
    sscanf(buffer, "%ld", &status);
    if (status != NNTP_OK_GROUPS)
	return status;

    global.no_groups = 0;
    for (buffer = server_read(main_server) ;
	 buffer && !IS_DOT(buffer) ;
	 buffer = server_read(main_server)) {
	char *c;
	long first, last;
	char mod;

	if (global.no_groups > global.max_groups - 2) {
	    long	i = global.max_groups;

	    global.max_groups += NO_GROUPS;
	    global.groups =
		(GROUP **)XtRealloc((char *)global.groups,
				    global.max_groups *
				    sizeof(GROUP *));

	    while (i < global.max_groups)
		global.groups[i++] = NULL;

	}

	c = strchr(buffer, ' ');
	if (c) {
	    GROUP	*group;
	    int		k;

	    *(c++) = '\0';
	    k = sscanf(c, "%ld%ld%*c%c", &last, &first, &mod);
	    if (k < 2)
		continue;
	    if (k == 2)
		mod = 'y';
	    if (mod == '=' || mod == 'x')
		continue;
	    group = (GROUP *)XtMalloc(sizeof(GROUP));
	    global.groups[global.no_groups++] = group;
	    group->description = NULL;
	    group->read_arts = NULL;
	    group->subscribed = False;
	    group->found_in_newsrc = False;
	    group->ahead_flag = False;
	    group->moderated = (mod == 'm');
	    group->name = XtNewString(buffer);
	    group->first_art = first;
	    group->last_art = last;
	}
    }

    if (!buffer) {
	long	i;

	for (i = 0 ; i < global.no_groups ; i++)
	    XtFree(global.groups[i]->name);
	XtFree((char *)global.groups);

	global.groups = NULL;
	global.no_groups = 0;
	global.max_groups = 0;

	return -1;
    }

    qsort(global.groups, global.no_groups,
	  sizeof(GROUP *), comp_group_nodes);

    return NNTP_OK_GROUPS;
}

/* the grouplist must be sorted here */
GROUP *find_group(char *name)
{
    long	first = 0;
    long	last = global.no_groups - 1;

    while (first <= last &&
	   global.groups[first]->subscribed) {
	if (strcmp(name, global.groups[first]->name) == 0)
	    return global.groups[first];
	first++;
    }

    while (first <= last) {
	long	mid = (first + last) / 2;
	int	comp = strcmp(name, global.groups[mid]->name);

	if (comp == 0)
	    return global.groups[mid];

	if (comp < 0)
	    last = mid - 1;
	else
	    first = mid + 1;
    }

    return NULL;
}

int check_for_new_groups(void)
{
    GROUP	**list = NULL;
    long	l_max = 0;
    struct tm	*tm;
    char	command[64];
    char	*buffer;
    long	status, n = 0;

    if (!res_check_for_new_groups() || global.last_time == 0)
	return True;

    global.new_groups = NULL;
    global.no_new_groups = 0;

    tm = gmtime(&global.last_time);
    sprintf(command, "NEWGROUPS %02d%02d%02d %02d%02d%02d GMT\r\n",
	    tm->tm_year, tm->tm_mon+1, tm->tm_mday,
	    tm->tm_hour, tm->tm_min, tm->tm_sec);
    buffer = server_comm(main_server, command, False);
    if (!buffer)
	return -1;
    status = atoi(buffer);
    if (status != NNTP_OK_NEWGROUPS) {
	fprintf(stderr,
		"check_for_new_groups failed: Message from server is: %s\n",
		buffer);
	return status;
    }

    l_max = NEW_GROUPS;
    list = (GROUP **)XtMalloc(l_max * sizeof(GROUP *));

    for (buffer = server_read(main_server) ;
	 buffer && !IS_DOT(buffer) ;
	 buffer = server_read(main_server)) {
	char	mod;
	char	*c;

	if (n >= l_max) {
	    l_max += NEW_GROUPS;
	    list = (GROUP **)XtRealloc((char *)list, l_max * sizeof(GROUP *));
	}

	c = strchr(buffer, ' ');
	if (c) {
	    *c++ = '\0';
	    if (sscanf(c, "%*d%*d%*c%c", &mod) == 1 &&
		(mod == '=' || mod == 'x'))
		continue;
	}
	list[n] = find_group(buffer);
	if (!list[n])
	    list[n] = create_group(buffer);
	n++;
    }

    sort_groups();

    if (!buffer) {
	XtFree((char *)list);
	return False;
    }

    if (n <= 0)
	XtFree((char *)list);
    else {
	global.new_groups = list;
	global.no_new_groups = n;
    }

    return NNTP_OK_NEWGROUPS;
}

/* this is a weak point */
static void put_subscribed_groups_first(long n)
{
    GROUP	**subs;
    long	i, j;

    if (n == 0)
	return;

    subs = (GROUP **)XtMalloc(n * sizeof(GROUP *));
    for (i = j = global.no_groups - 1 ; i >= 0 ; i--) {
	if (global.groups[i]->subscribed)
	    subs[global.groups[i]->disp] = global.groups[i];
	else
	    global.groups[j--] = global.groups[i];
    }
    for (i = 0 ; i < n ; i++)
	global.groups[i] = subs[i];

    XtFree((char *)subs);
}

void sort_groups(void)
{
    long i, n;

    for (n = 0 ; n < global.no_groups ; n++)
	if (!global.groups[n]->subscribed) break;

    for (i = n ; i < global.no_groups ; i++)
	if (global.groups[i]->subscribed) {
	    GROUP *temp = global.groups[n];

	    global.groups[n] = global.groups[i];
	    global.groups[i] = temp;
	    n++;
	}

    if (n < global.no_groups)
	qsort(&global.groups[n], global.no_groups - n,
	      sizeof(GROUP *), comp_group_nodes);
}

int update_newsrc(void)
{
    char	*newsrc_file = res_newsrc_file();
    int		fill         = res_fill_newsrc_file();
    char	*path;
    FILE	*fp;
    long	i;
    int		ok;

    if (!newsrc_file) {
	fprintf(stderr, "knews: newsrcFile is NULL!!\n");
	return False;
    }

    path = expand_path(newsrc_file);
    if (!path)
	return False;

    sighup_may_exit = False;

    (void)unlink(path);
    fp = fopen_mkdir(path, "w", True);
    XtFree(path);
    if (!fp) {
	sighup_may_exit = True;
	if (caught_sighup)
	    sighup_handler(0);
	return False;
    }

    for (i = 0 ; i < global.no_groups ; i++)
	if (fill || global.groups[i]->read_arts ||
	    global.groups[i]->subscribed) {
	    ART_LIST_NODE	*loop = global.groups[i]->read_arts;

	    fprintf(fp, "%s%c", global.groups[i]->name,
		    global.groups[i]->subscribed ? ':' : '!');
	    if (loop)
		fprintf(fp, " ");
	    while (loop) {
		fprintf(fp, "%ld", loop->first);
		if (loop->first != loop->last)
		    fprintf(fp, "-%ld", loop->last);
		loop = loop->next;
		if (loop)
		    fprintf(fp, ",");
	    }
	    fprintf(fp, "\n");
	}

    ok = (fclose(fp) != EOF);

    sighup_may_exit = True;
    if (caught_sighup)
	sighup_handler(0);

    return ok;
}

void calc_no_unread(GROUP *group)
{
    ART_LIST_NODE *loop;

    if (group->first_art == 0 && group->last_art == 0) {
	group->no_unread = 0;
	return;
    }

    group->no_unread = group->last_art - group->first_art + 1;
    for (loop = group->read_arts ; loop ; loop = loop->next) {
	if (loop->last > group->last_art) {
	    if (loop->first <= group->last_art)
		group->no_unread -= (group->last_art - loop->first + 1);
	    return;
	} else if (loop->first < group->first_art) {
	    if (loop->last >= group->first_art)
		group->no_unread -= (loop->last - group->first_art +1);
	} else {
	    group->no_unread -= (loop->last - loop->first + 1);
	}
    }
}

int get_descriptions(void)
{
    char	*file_name = res_descriptions_file();
    FILE	*fp = NULL;
    SERVER	*server;
    char	*buffer;
    long	n;

    if (res_retrieve_descriptions()) {
	set_message("Retrieving group descriptions...", False);
	server = main_server;
	buffer = server_comm(main_server, "LIST NEWSGROUPS\r\n", False);
	if (!buffer)
	    return -1;
	sscanf(buffer, "%ld", &n);
	if (n != NNTP_OK_GROUPS)
	    return n;

	if (file_name)
	    fp = fopen_expand(file_name, "w", True);
	server_set_bs(server, fp);
    } else {
	int	fd;

	if (!file_name)
	    return 0;
	fd = open_expand(file_name, O_RDONLY, 0, True);
	if (fd < 0)
	    return 0;

	set_message("Reading group descriptions from file...", False);
	server = server_create(fd);
    }

    for (buffer = server_read(server) ;
	 buffer && !IS_DOT(buffer) ;
	 buffer = server_read(server)) {
	GROUP	*group;
	char	*c;

	c = buffer;
	while (*c != '\0' && *c != '\t' && *c != ' ') c++;
	if (*c == '\0')
	    continue;

	*c++ = '\0';
	while (*c == ' ' || *c == '\t') c++;
	group = find_group(buffer);
	if (group) {
	    if (group->description) {
		long	len = strlen(group->description);

		/* if duplicate descriptions, take the longest */
		if (len >= strlen(c))
		    continue;
		XtFree(group->description);
	    }

	    group->description = XtNewString(c);
	}
    }

    if (fp) {
	server_set_bs(server, NULL);
	if (fclose(fp) == EOF || !buffer)
	    unlink_expand(file_name);
    }

    if (server != main_server)
	server_free(server);
    else if (!buffer)
	return -1;

    return NNTP_OK_GROUPS;
}

GROUP *create_group(char *name)
{
    GROUP	*group;

    global.groups =
	(GROUP **)XtRealloc((char *)global.groups,
			    (global.no_groups + 1) *
			    sizeof(GROUP *));
    group = global.groups[global.no_groups++] =
	(GROUP *)XtMalloc(sizeof(GROUP));
    group->name = XtNewString(name);
    group->description = NULL;
    group->no_unread = 0;
    group->first_art = 0;
    group->last_art = 0;
    group->read_arts = NULL;
    group->disp = -1;
    group->subscribed = False;
    group->moderated = False;
    group->found_in_newsrc = False;
    group->ahead_flag = False;

    return group;
}

void check_read_arts_sanity(GROUP *temp)
{
    ART_LIST_NODE	*loop;

    for (loop = temp->read_arts ; loop ; loop = loop->next)
	if (loop->first > loop->last ||
	    (loop->next && loop->last > loop->next->first)) {
	    fprintf(stderr,
		    "knews: Bad list of read articles in .newsrc file, "
		    "group %s.  Marking all articles unread.\n",
		    temp->name);
	    free_read_arts_list(temp);
	    temp->read_arts = NULL;
	    return;
	}
}

static ART_LIST_NODE *parse_read_arts_list(FILE *fp)
{
    ART_LIST_NODE	*result;
    int			c;
    long		first = 0, last = 0;

    do {
	c = getc(fp);
    } while (c != EOF && c != '\n' && IS_SPACE(c));
    if (c == EOF || c == '\n')
	return NULL;

    if (c >= '0' && c <= '9') {
	while (c >= '0' && c <= '9') {
	    first *= 10;
	    first += c - '0';
	    c = getc(fp);
	}

	if (c == '-' || c == ',' || c == '\n' || c == EOF) {
	    if (c != '-')
		last = first;
	    else {
		c = getc(fp);
		if (c >= '0' || c <= '9') {
		    while (c >= '0' && c <= '9') {
			last *= 10;
			last += c - '0';
			c = getc(fp);
		    }
		}
	    }

	    if (c == ',' || c == EOF || c == '\n') {
		result = (ART_LIST_NODE *)XtMalloc(sizeof *result);
		result->first = first;
		result->last = last;

		if (c == ',')
		    result->next = parse_read_arts_list(fp);
		else
		    result->next = NULL;

		return result;
	    }
	}
    }

    fprintf(stderr, "knews: Parse error in newsrc.\n");
    while (c != EOF && c != '\n')
	c = getc(fp), putchar(c);
    return NULL;
}

static void create_new_newsrc(char *path)
{
    FILE	*fp;

    fp = fopen_mkdir(path, "w", True);
    popup_filenotice(fp ? "Created newsrc file" :
		     "Failed to create newsrc file", path, fp == NULL);
    if (!fp)
	return;

    fprintf(fp, "news.answers:\nnews.newusers.questions:\n");
    fclose(fp);
}

void parse_newsrc(int create)
{
    char	*new = res_newsrc_file();
    char 	*old = res_old_newsrc_file();
    char	buffer[1024];
    char	*path;
    FILE	*fp;
    long	pos;

    if (!new) {
	fprintf(stderr, "knews: newsrcFile is NULL!\n");
	return;
    }

    path = expand_path(new);
    if (!path)
	return;

    fp = fopen(path, "r");
    if (!fp) {
	perror(path);
	create_new_newsrc(path);
	fp = fopen(path, "r");
	if (!fp) {
	    XtFree(path);
	    return;
	}
    }

    global.last_time = 0;
    if (old) {
	char		*opath;
	struct stat	sbuf;

	opath = expand_path(old);
	if (opath) {
	    if (stat(opath, &sbuf) == 0) {
		unlink(opath);
		if (S_ISREG(sbuf.st_mode))
		    global.last_time = sbuf.st_mtime;
	    }
	    if (link(path, opath) < 0)
		perror("knews: link");
	    XtFree(opath);
	}
    }
    XtFree(path);

    pos = 0;
    for (;;) {
	int	n = 0, c;

	while (n < sizeof buffer - 1 &&
	       (c = getc(fp)) != EOF &&
	       c != ':' && c != '!' && c != '\n')
	    buffer[n++] = c;

	buffer[n] = '\0';

	if (c == ':' || c == '!') {
	    GROUP	*temp;

	    temp = find_group(buffer);
	    if (!temp && create)
		temp = create_group(buffer);
	    if (!temp)
		fprintf(stderr, "Bogus group in newsrc file: %s\n", buffer);
	    else if (temp->found_in_newsrc)
		fprintf(stderr, "Group %s appeared twice in newsrc.\n",
			buffer);
	    else {
		temp->subscribed = (c == ':');
		temp->found_in_newsrc = True;
		if (temp->subscribed)
		    temp->disp = pos++;
		temp->read_arts = parse_read_arts_list(fp);
		check_read_arts_sanity(temp);
		continue;
	    }
	} else if (c == EOF) {
	    if (n > 0)
		fprintf(stderr, "Premature end of file in newsrc.\n");
	    break;
	} else if (c == '\n') {
	    fprintf(stderr, "Garbage line in newsrc: %s\n", buffer);
	} else {
	    fprintf(stderr, "Group name too long in newsrc file: %s\n",
		    buffer);
	}

	while (c != EOF && c != '\n') /* skip to end-of-line */
	    c = getc(fp);
    }

    fclose(fp);
    put_subscribed_groups_first(pos);
    sort_groups();
}

int get_newsgroups_from_newsrc(void)
{
    char	*tmp;

    parse_newsrc(True);
    tmp = rescan();
    if (!tmp)
	return -1;

    return atoi(tmp);
}

char *rescan(void)
{
    char	message[128];
    char	*buffer, *p;
    long	n;

    for (n = 0 ; n < global.no_groups ; n++)
	if (!global.groups[n]->subscribed)
	    break;
    if (n == 0)
	return CODE_TO_STR(NNTP_OK_GROUPS);

    strcpy(message, "Rescan in progress...  ");
    p = message + strlen(message);

    if (res_read_active_file()) {
	long	i, j, k, m;
	buffer = server_comm(main_server, "LIST\r\n", True);
	if (!buffer || atoi(buffer) != NNTP_OK_GROUPS)
	    return buffer;

	buffer = server_read(main_server);
	j = m = global.no_groups / 16;
	i = 0;
	while (buffer && !IS_DOT(buffer)) {
	    char	*c = strchr(buffer, ' ');

	    if (c) {
		*c++ = '\0';
		for (k = 0 ; k < n ; k++) {
		    if (strcmp(buffer, global.groups[k]->name) == 0) {
			sscanf(c, "%ld%ld",
			       &global.groups[k]->last_art,
			       &global.groups[k]->first_art);
			break;
		    }
		}
	    }

	    if (j-- <= 0) {
		sprintf(p, "%3ld%%", i / global.no_groups);
		set_message(message, False);
		j = m;
	    }
	
	    buffer = server_read(main_server);
	    i += 100;
	}

	if (!buffer)
	    return NULL;
    } else {
	int	list = res_try_list_active();
	char	command[1024];
	long	i, j, first, last;
	char	m;

	for (i = 0, j = 0 ; j < global.no_groups ; j++) {
	    if (!global.groups[j]->subscribed)
		break;
	    if (strlen(global.groups[j]->name) > 480) {
		fprintf(stderr, "Group name too long: %s\n",
			global.groups[j]->name);
		continue;
	    }

	    if (list) {
		sprintf(command, "LIST ACTIVE %s\r\n", global.groups[j]->name);
		server_write(main_server, command);
		buffer = server_read(main_server);
		if (!buffer)
		    return NULL;
		if (atol(buffer) == NNTP_OK_GROUPS) {
		    buffer = server_read(main_server);
		    while (buffer && !IS_DOT(buffer)) {
			char	*c = strchr(buffer, ' ');

			if (c) {
			    *c++ = '\0';
			    m = 'y';
			    if (case_strcmp(global.groups[j]->name,
					    buffer) == 0 &&
				sscanf(c, "%ld%ld%*c%c",
				       &last, &first, &m) >= 2 &&
				m != '-' && m != 'x') {
				global.groups[j]->first_art = first;
				global.groups[j]->last_art = last;
				global.groups[j]->moderated = (m != 'm');
			    }
			}
			buffer = server_read(main_server);
		    }

		    if (!buffer)
			return NULL;
		} else {
		    fprintf(stderr,
			    "knews: 'LIST ACTIVE wildmat' failed, message "
			    "from server is: \n"
			    "       %s\n"
			    "       Falling back to GROUP\n", buffer);
		    list = False;
		}
	    }

	    if (!list) {
		sprintf(command, "GROUP %s\r\n", global.groups[j]->name);
		server_write(main_server, command);
		buffer = server_read(main_server);
		if (!buffer)
		    return NULL;
		if (atoi(buffer) != NNTP_OK_GROUP)
		    fprintf(stderr, "knews: Bogus group: %s\n",
			    global.groups[j]->name);
		else
		    sscanf(buffer, "%*d%*d%ld%ld",
			   &global.groups[j]->first_art,
			   &global.groups[j]->last_art);
	    }

	    i += 100;
	    sprintf(p, "%ld%%", i / n);
	    set_message(message, False);
	}
    }

    return CODE_TO_STR(NNTP_OK_GROUPS);
}
