/*  Copyright (c) 1995 John E. Davis (davis@space.mit.edu)
 *  All rights reserved.
 */
/* Read startup .slrnrc file */

/* This file contains the following types of lines:
 *    server SERVER-NAME NEWSRC-FILE
 *    setkey KEYMAP FUNCTION KEYSEQUENCE
 *    unsetkey KEYMAP KEYSEQUENCE
 *
 * Lines are considered commented out if they start with a '%' character.
 */

#include <config.h>

#include <stdio.h>
#ifdef HAVE_STDLIB_H
# include <stdlib.h>
#endif

#include <string.h>
#include <slang.h>


#include "slrn.h"
#include "group.h"
#include "misc.h"
#include "art.h"
#include "post.h"
#include "startup.h"
#include "score.h"
#include "uudecode.h"
#ifdef HAS_MIME_SUPPORT
#include "mime.h"
#endif


#if SLANG_VERSION < 9924
#define SLcmd_execute_string sl_cmd_execute_string
#define SLcmd_Cmd_Table_Type sl_cmd_cmd_table_type
#define SLcmd_Cmd_Type sl_cmd_cmd_type

typedef struct sl_cmd_cmd_table_type
{
   struct sl_cmd_cmd_type *table;
   int argc;
   char *string_args[SLCMD_MAX_ARGS];
   int int_args[SLCMD_MAX_ARGS];
   FLOAT float_args[SLCMD_MAX_ARGS];
   int arg_type[SLCMD_MAX_ARGS];
}
sl_cmd_cmd_table_type;

typedef struct sl_cmd_cmd_type
{
   int (*cmdfun)(int, struct sl_cmd_cmd_table_type *);
   char cmd[32];
   char arg_type[SLCMD_MAX_ARGS];
}
sl_cmd_cmd_type;

static sl_cmd_execute_string (char *, SLcmd_Cmd_Table_Type *);
extern int SLatoi(unsigned char *);
#endif

static SLcmd_Cmd_Table_Type Slrn_Cmd_Table;


static int unsetkey_fun (int, SLcmd_Cmd_Table_Type *);
static int setkey_fun (int, SLcmd_Cmd_Table_Type *);
static int server_fun (int, SLcmd_Cmd_Table_Type *);
static int color_fun (int, SLcmd_Cmd_Table_Type *);
static int user_data_fun (int, SLcmd_Cmd_Table_Type *);
static int ignore_quote_fun (int, SLcmd_Cmd_Table_Type *);
static int set_quote_fun (int, SLcmd_Cmd_Table_Type *);
static int autobaud_fun (int, SLcmd_Cmd_Table_Type *);
static int set_variable_fun (int, SLcmd_Cmd_Table_Type *);
static int nnrp_fun (int, SLcmd_Cmd_Table_Type *);

static SLcmd_Cmd_Type Slrn_Startup_File_Cmds[] =
{
     {unsetkey_fun, "unsetkey", "SS"},
     {setkey_fun, "setkey", "SSS"},
     {server_fun, "server", "SS"},
     {color_fun, "color", "SSS"},
     {set_variable_fun, "set", "SG"},
     {user_data_fun, "replyto", "S"},
     {user_data_fun, "organization", "S"},
     {user_data_fun, "signature", "S"},
     {user_data_fun, "hostname", "S"},
     {user_data_fun, "realname", "S"},
     {user_data_fun, "followup", "S"},
     {user_data_fun, "username", "S"},
     {user_data_fun, "scorefile", "S"},
     {user_data_fun, "cc_followup_string", "S"},
#ifndef NO_DECODE
     {user_data_fun, "decode_directory", "S"},
#endif
     {user_data_fun, "editor_command", "S"},
     {nnrp_fun, "nnrpaccess", "SSS" },
     {ignore_quote_fun, "ignore_quotes", "S"},
     {set_quote_fun, "quote_string", "S"},
     {autobaud_fun, "autobaud", ""},
     {NULL, "", ""}
};

static int This_Line_Num;		       /* current line number in startup file */
static char *This_File;
static char *This_Line;		       /* line being parsed */

static void exit_malloc_error (void)
{
   slrn_exit_error ("%s: Line %d\n%sMemory Allocation Failure",
		    This_File, This_Line_Num, This_Line);
}

static void exit_unknown_object (void)
{
   slrn_exit_error ("%s: Unsupported option on line %d\n%s",
		    This_File, This_Line_Num, This_Line);
}


static int setkey_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *map = table->string_args[1];
   char *fun = table->string_args[2];
   char *key = table->string_args[3];
   
   SLKeyMap_List_Type *kmap = NULL;
   
   if (!strcmp (map, "group")) kmap = Slrn_Group_Keymap;
   else if (!strcmp (map, "article")) kmap = Slrn_Article_Keymap;
   else slrn_exit_error ("%s: line %d:\n%sNo such keymap: %s", This_File, This_Line_Num, This_Line, map);
   
   if (SLang_define_key (key, fun, kmap) != 0)
     {
	slrn_exit_error ("%s: line %d:\n%serror defining key.", This_File, This_Line_Num, This_Line);
     }
   return 0;
}

int Slrn_Autobaud = 0;
static int autobaud_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   Slrn_Autobaud = 1;
   return 0;
}


static int unsetkey_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *map = table->string_args[1];
   char *key = table->string_args[2];
   SLKeyMap_List_Type *kmap = NULL;
   
   if (!strcmp (map, "group")) kmap = Slrn_Group_Keymap;
   else if (!strcmp (map, "article")) kmap = Slrn_Article_Keymap;
   else slrn_exit_error ("%s: line %d:\n%sNo such keymap: %s",
			 This_File, This_Line_Num, This_Line, map);
   
   SLang_undefine_key (key, kmap);
   return 0;
}

static SLRegexp_Type Quote_Regexp;
SLRegexp_Type *Slrn_Ignore_Quote_Regexp;

static void compile_quote_regexp (char *str)
{
   static unsigned char compiled_pattern_buf[256];
   
   Quote_Regexp.pat = (unsigned char *) str;
   Quote_Regexp.buf = compiled_pattern_buf;
   Quote_Regexp.case_sensitive = 1;
   Quote_Regexp.buf_len = sizeof (compiled_pattern_buf);
   
   if (SLang_regexp_compile (&Quote_Regexp))
     {
	slrn_exit_error ("%s: line %d:\n%sInvalid regular expression.",
			 This_File, This_Line_Num, This_Line);
     }
   Slrn_Ignore_Quote_Regexp = &Quote_Regexp;
}

static int ignore_quote_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   compile_quote_regexp (table->string_args[1]);
   return 0;
}

static int set_quote_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   static char quote[32];
   
   strncpy (quote, table->string_args[1], 31);
   quote[31] = 0;
   Slrn_Quote_String = quote;
   return 0;
}

typedef struct
{
   char *what;
   int *valuep;
}
Set_Int_Type;

static Set_Int_Type Int_Things_To_Set [] =
{
     {"use_xgtitle", &Slrn_Use_Xgtitle},
     {"show_article", &Slrn_Startup_With_Article},
     {"author_display", &Slrn_Show_Author},
     {"show_descriptions", &Slrn_Group_Display_Descriptions},
     {"no_backups", &Slrn_No_Backups},
     {"beep", &SLtt_Ignore_Beep},
     {"unsubscribe_new_groups", &Slrn_Unsubscribe_New_Groups},
     {"show_thread_subject", &Slrn_Show_Thread_Subject},
     {"mouse", &Slrn_Use_Mouse},
     {"query_next_group", &Slrn_Query_Next_Group},
     {"query_next_article", &Slrn_Query_Next_Article},
     {"confirm_actions", &Slrn_User_Wants_Confirmation},
     {"cc_followup", &Slrn_Auto_CC_To_Poster},
     {"use_tmpdir", &Slrn_Use_Tmpdir},
     {"sorting_method", &Slrn_Sorting_Mode},
     {"uncollapse_threads", &Slrn_Threads_Visible},
     {"read_active", &Slrn_List_Active_File},
     {"use_metamail", &Slrn_Use_Meta_Mail},
#ifndef __os2__
     {"use_blink", &SLtt_Blink_Mode},
#endif
     {"wrap_flags", &Slrn_Wrap_Mode},
     {"query_read_group_cutoff", &Slrn_Query_Group_Cutoff},
     {"prompt_next_group", &Slrn_Prompt_Next_Group},
     {"use_header_numbers", &Slrn_Use_Header_Numbers},
#ifdef HAS_MIME_SUPPORT
     {"use_mime", &Slrn_Use_Mime},
#else
     {"use_mime", NULL},
#endif
     {NULL, NULL}
};

typedef struct
{
   char *what;
   char **svaluep;
}
Set_String_Type;

static Set_String_Type String_Things_To_Set [] =
{
     {"save_posts", &Slrn_Save_Posts_File},
     {"save_directory", &Slrn_Save_Directory},
#ifndef NO_DECODE
     {"decode_directory", &Slrn_Decode_Directory},
#else
     {"decode_directory", NULL},
#endif
     
#ifdef HAS_MIME_SUPPORT
     {"mime_charset", &Slrn_Mime_Display_Charset},
#else
     {"mime_charset", NULL},
#endif
#ifndef VMS
     {"sendmail_command", &Slrn_SendMail_Command},
#endif
     {NULL, NULL}
};

static int set_variable_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *what = table->string_args[1];
   int ivalue = table->int_args[2];
   Set_Int_Type *ip = Int_Things_To_Set;
   char *svalue = table->string_args[2];
   Set_String_Type *sp = String_Things_To_Set;
   int type = table->arg_type[2];
   
   if (type == STRING_TYPE)
     {
	while (sp->what != NULL)
	  {
	     if (!strcmp (sp->what, what))
	       {
		  char *ss;
		  
		  if (sp->svaluep == NULL) return 0;
		  
		  ss = *sp->svaluep;
		  
		  if (ss != NULL) SLFREE (ss);
		  if (NULL == (ss = SLmake_string(svalue)))
		    exit_malloc_error ();
		  
		  *sp->svaluep = ss;
		  return 0;
	       }
	     sp++;
	  }
     }
   else if (type == INT_TYPE) while (ip->what != NULL)
     {
	if (!strcmp (ip->what, what))
	  {
	     if (ip->valuep == NULL) return 0;
	     *ip->valuep = ivalue;
	     return 0;
	  }
	ip++;
     }
   exit_unknown_object ();
   return -1;
}


typedef struct Server_List_Type
{
   struct Server_List_Type *next;
   char *file;
   char *host;
}
Server_List_Type;

Server_List_Type *Server_List;

static int server_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *file = NULL, *host;
   char *the_file = table->string_args[2], *the_host = table->string_args[1];
   int n;
   Server_List_Type *s;
   
   n = strlen (the_file);
   
   if ((NULL == (s = (Server_List_Type *) SLMALLOC (sizeof (Server_List_Type))))
       || (file = SLMALLOC (n + 2 + strlen (the_host))) == NULL)
     {
	exit_malloc_error ();
     }
   
   strcpy (file, the_file);
   host = file + (n + 1);
   strcpy (host, the_host);
   
   s->next = Server_List;  Server_List = s;
   s->host = host;
   s->file = file;
   return 0;
}

typedef struct
{
   char *name;
   int value;
}
Color_Handle_Type;

Color_Handle_Type Color_Handles[] =
{
     {"normal", 0},
     {"status", STATUS_COLOR},
     {"menu", MENU_COLOR},
     {"menu_press", MENU_PRESS_COLOR},
     {"headers", HEADER_COLOR},
     {"group", GROUP_COLOR},
     {"subject", SUBJECT_COLOR},
     {"author", AUTHOR_COLOR},
     {"error", ERROR_COLOR},
     {"cursor", CURSOR_COLOR},
     {"article", ARTICLE_COLOR},
     {"tree", TREE_COLOR},
     {"quotes", QUOTE_COLOR},
     {"signature", SIGNATURE_COLOR},
     {"thread_number", THREAD_NUM_COLOR},
     {"header_number", HEADER_NUMBER_COLOR},
     {"high_score", HIGH_SCORE_COLOR},
     {NULL, -1}
};


static int color_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *what = table->string_args[1];
   char *fg = table->string_args[2];
   char *bg = table->string_args[3];
   Color_Handle_Type *ct = Color_Handles;
   
   while (ct->name != NULL)
     {
	if (!strcmp (ct->name, what))
	  {
	     SLtt_set_color (ct->value, what, fg, bg);
	     return 0;
	  }
	ct++;
     }
   exit_unknown_object ();
   return 0;
}

/*----------------------------------------------------------------------*\
* static int user_data_fun ();
 *
 * convenient mechanism to set Slrn_User_Info fields without adding
 * extra environment variables
 *
 * recognized fields
 *
 *   replyto		- alternative name for replies
 *   organization	- use double quotes if there are spaces!
 *   signature		- an alternate to ~/.signature (for news posting)
 *   hostname		- full name of the current host
\*----------------------------------------------------------------------*/

typedef struct
{
   char *name;
   char **addr;
   unsigned int size;
}
User_Info_Variable_Type;

User_Info_Variable_Type User_Info_Variables[] =
{
     {"realname", &Slrn_User_Info.realname, 0},
     {"username", &Slrn_User_Info.username, 0},
     {"hostname", (char **)Slrn_User_Info.host, MAX_HOST_NAME_LEN},
     {"replyto", &Slrn_User_Info.replyto, 0},
     {"organization", &Slrn_User_Info.org, 0},
     {"followup", &Slrn_User_Info.followup_string, 0},
     {"signature", &Slrn_User_Info.signature, 0},
     {"scorefile", &Slrn_Score_File, 0},
     {"cc_followup_string", &Slrn_Courtesy_CC_Message, 0},
#ifndef NO_DECODE
     {"decode_directory", &Slrn_Decode_Directory, 0},
#endif
     {"editor_command", &Slrn_Editor, 0},
     {NULL, NULL, 0}
};

static int user_data_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *what = table->string_args[0];
   char *field = table->string_args[1];
   User_Info_Variable_Type *u = User_Info_Variables;
   char **ptr, *contents;
   unsigned int n;
   
   while (u->name != NULL)
     {
	if (!strcmp (u->name, what))
	  {
	     n = strlen (field);
	     
	     if (u->size)
	       {
		  contents = (char *) u->addr;
		  strncpy (contents, field, u->size);
		  contents [u->size - 1] = 0;
	       }
	     else
	       {
		  ptr = u->addr;
		  if ((contents = SLMALLOC (n + 1)) == NULL)
		    exit_malloc_error ();
		  strcpy (contents, field);
		  
		  if (*ptr != NULL) SLFREE (*ptr);
		  *ptr = contents;
	       }
	     return 0;
	  }
	u++;
     }
   exit_unknown_object ();
   return -1;
}


/*----------------------------------------------------------------------*\
* static int nnrp_fun ();
 *
 * convenient mechanism to set nnrp Slrn_User_Info fields without adding
 * extra environment variables
 *
 * recognized fields
 *
 *   nnrpaccess         - used to log in to a server using authinfo
 *                        it has the following format.
 *                         "host  username  password"
\*----------------------------------------------------------------------*/

static int nnrp_fun (int argc, SLcmd_Cmd_Table_Type *table)
{
   char *server = table->string_args[1];
   char *name = table->string_args[2];
   char *pass = table->string_args[3];
   int n;
   
   /* Skip this one it is not the one that we are looking for */
   if (strcmp(server, Slrn_Server_Name))
     return 0;
   
   if (Slrn_User_Info.nnrpname != NULL) SLFREE (Slrn_User_Info.nnrpname);
   if (Slrn_User_Info.nnrppass != NULL) SLFREE (Slrn_User_Info.nnrppass);
   
   n = strlen(name);
   if (NULL == (Slrn_User_Info.nnrpname = (char *) SLMALLOC(n + 1)))
     exit_malloc_error ();
   strcpy(Slrn_User_Info.nnrpname, name);
   
   n = strlen(pass);
   if (NULL == (Slrn_User_Info.nnrppass = (char *) SLMALLOC(n + 1)))
     exit_malloc_error ();
   strcpy(Slrn_User_Info.nnrppass, pass);
   
   return 0;
}

static void slrn_init_keymaps (void)
{
   slrn_init_group_keymap ();
   slrn_init_article_keymap ();
}

void slrn_initialize (void)
{
   char *white = "white", *blue = "blue", *red = "red", *yellow = "yellow";
   char *black = "black";
   slrn_init_keymaps ();
   SLang_init_case_tables ();
   compile_quote_regexp ("^ ?[:>]");
   
   /* default colors -- suitable for a color xterm */
   
   SLtt_set_color (0, NULL, black, white);
   SLtt_set_color (STATUS_COLOR, NULL, yellow, blue);
   SLtt_set_color (MENU_COLOR, NULL, yellow, blue);
   SLtt_set_color (HEADER_COLOR, NULL, "brightcyan", white);
   SLtt_set_color (GROUP_COLOR, NULL, blue, white);
   SLtt_set_color (SUBJECT_COLOR, NULL, black, white);
   SLtt_set_color (AUTHOR_COLOR, NULL, "magenta", white);
   SLtt_set_color (ERROR_COLOR, NULL, red, white);
   SLtt_set_color (CURSOR_COLOR, NULL, "brightgreen", white);
   SLtt_set_color (ARTICLE_COLOR, NULL, blue, white);
   SLtt_set_color (TREE_COLOR, NULL, red, white);
   SLtt_set_color (QUOTE_COLOR, NULL, red, white);
   SLtt_set_color (SIGNATURE_COLOR, NULL, red, white);
   SLtt_set_color (THREAD_NUM_COLOR, NULL, blue, white);
   SLtt_set_color (HIGH_SCORE_COLOR, NULL, red, white);
   SLtt_set_color (HEADER_NUMBER_COLOR, NULL, "green", white);
   SLtt_set_color (MENU_PRESS_COLOR, NULL, blue, yellow);
   
#ifndef __os2__
   /* We are not using the blink characters */
   SLtt_Blink_Mode = 0;
#endif
}

void slrn_read_startup_file (char *name)
{
   FILE *fp;
   char file[256];
   char line[256];
   SLPreprocess_Type pt;
   
   if (-1 == SLprep_open_prep (&pt))
     {
	slrn_exit_error ("Error initializing S-Lang preprocessor.");
     }
   
   
   fp = slrn_open_home_file (name, "r", file, 0);
   if (fp == NULL) return;
   
   This_File = file;
   This_Line = line;
   
   Slrn_Cmd_Table.table = Slrn_Startup_File_Cmds;
   
   This_Line_Num = 0;
   while (NULL != fgets (line, sizeof(line) - 1, fp))
     {
	This_Line_Num++;
	if (SLprep_line_ok (line, &pt))
	  (void) SLcmd_execute_string (line, &Slrn_Cmd_Table);
	
	if (SLang_Error) exit_unknown_object ();
     }
   slrn_fclose (fp);
   
   SLprep_close_prep (&pt);
}



char *slrn_map_file_to_host (char *host)
{
   Server_List_Type *s = Server_List;
   
   while (s != NULL)
     {
	if (!strcmp (host, s->host)) return s->file;
	s = s->next;
     }
   return NULL;
}


#if SLANG_VERSION < 9924
static SLcmd_Cmd_Type *sl_cmd_find_command (char *s, SLcmd_Cmd_Type *cmd)
{
   char *cmdstr;
   register char chs = *s++, ch;
   while (1)
     {
	cmdstr = cmd->cmd;
	ch = *cmdstr++;
	if (ch == 0) break;
	if ((ch == chs) && !strcmp (s, cmdstr)) return cmd;
	cmd++;
     }
   return NULL;
}

static int sl_cmd_execute_string (char *str, SLcmd_Cmd_Table_Type *table)
{
   char *s, *b = NULL, *arg_type, *last_str;
   SLcmd_Cmd_Type *cmd;
   char buf[256];
   int token_present;
   int i, ret = -1, len;
   int argc;
   
   if ((0 == SLang_extract_token (&str, buf, 0))
       || (*buf == '%'))
     return 0;
   if (SLang_Error) return -1;
   
   if (((len = strlen (buf)) >= 32)
       || (NULL == (cmd = sl_cmd_find_command (buf, table->table))))
     {
	SLang_doerror ("Invalid command.");
	return -1;
     }
   
   
   if (NULL == (s = (char *) SLMALLOC(1 + len)))
     {
	SLang_Error = SL_MALLOC_ERROR;
	return -1;
     }
   strcpy (s, buf);
   argc = 0;
   table->string_args[argc++] = s;
   
   arg_type = cmd->arg_type;
   
   while (*arg_type)
     {
	int guess_type = 0;
	
	last_str = str;
	token_present = SLang_extract_token (&str, buf, 0);
	if (SLang_Error) goto error;
	if (token_present)
	  {
	     b = buf;
	     len = strlen (b);
	     if ((*b == '"') && (len > 1))
	       {
		  b++;
		  len -= 2;
		  b[len] = 0;
		  guess_type = STRING_TYPE;
	       }
	     else guess_type = SLang_guess_type (buf);
	  }
	
	switch (*arg_type++)
	  {
	     /* variable argument number */
	   case 'v':
	     if (token_present == 0) break;
	   case 'V':
	     if (token_present == 0)
	       {
		  SLang_doerror ("Expecting argument.");
		  goto error;
	       }
	     
	     last_str = slrn_skip_whitespace (last_str);
	     len = strlen (last_str);
	     str = last_str + len;
	     
	     s = SLmake_nstring (last_str, len);
	     if (s == NULL) goto error;
	     table->arg_type[argc] = STRING_TYPE;
	     table->string_args[argc++] = s;
	     break;
	     
	   case 's':
	     if (token_present == 0) break;
	   case 'S':
	     if (token_present == 0)
	       {
		  SLang_doerror ("Expecting string argument.");
		  goto error;
	       }
	     
	     s = SLmake_nstring (b, len);
	     
	     if (s == NULL) goto error;
	     table->arg_type[argc] = STRING_TYPE;
	     table->string_args[argc++] = s;
	     break;
	     
	     /* integer argument */
	   case 'i':
	     if (token_present == 0) break;
	   case 'I':
	     if ((token_present == 0) || (INT_TYPE != guess_type))
	       {
		  SLang_doerror ("Expecting integer argument.");
		  SLang_Error = TYPE_MISMATCH;
		  goto error;
	       }
	     
	     table->arg_type[argc] = INT_TYPE;
	     table->int_args[argc++] = SLatoi((unsigned char *) buf);
	     break;
	     
	     /* floating point arg */
#ifdef FLOAT_TYPE
	   case 'f':
	     if (token_present == 0) break;
	   case 'F':
	     if ((token_present == 0) || (STRING_TYPE == guess_type))
	       {
		  SLang_doerror ("Expecting floating point argument.");
		  SLang_Error = TYPE_MISMATCH;
		  goto error;
	       }
	     table->arg_type[argc] = FLOAT_TYPE;
	     table->float_args[argc++] = atof(buf);
	     break;
#endif
	     /* Generic type */
	   case 'g':
	     if (token_present == 0) break;
	   case 'G':
	     if (token_present == 0)
	       {
		  SLang_doerror ("Expecting an argument.");
		  SLang_Error = TYPE_MISMATCH;
		  goto error;
	       }
	     
	     switch (guess_type)
	       {
		case INT_TYPE:
		  table->arg_type[argc] = INT_TYPE;
		  table->int_args[argc++] = SLatoi((unsigned char *) buf);
		  break;
		  
		case STRING_TYPE:
		  s = SLmake_nstring (b, len);
		  if (s == NULL) goto error;
		  
		  table->arg_type[argc] = STRING_TYPE;
		  table->string_args[argc++] = s;
		  break;
#ifdef FLOAT_TYPE
		case FLOAT_TYPE:
		  table->arg_type[argc] = FLOAT_TYPE;
		  table->float_args[argc++] = atof(buf);
#endif
	       }
	     break;
	  }
     }
   
   /*                 call function */
   ret = (*cmd->cmdfun)(argc, table);
   
   error:
   for (i = 0; i < argc; i++)
     {
	if (NULL != table->string_args[i])
	  {
	     SLFREE (table->string_args[i]);
	     table->string_args[i] = NULL;
	  }
     }
   return ret;
   
}
#endif
