/*--------------------------------*-C-*---------------------------------*
 * This module is all new by Robert Nation
 * (nation@rocket.sanders.lockheed.com).
 *
 * Additional modifications by mj olesen <olesen@me.queensu.ca>
 * No additional restrictions are applied.
 *
 * As usual, the author accepts no responsibility for anything, nor does
 * he guarantee anything whatsoever.
 *
 * Design of this module was heavily influenced by the original xvt
 * design of this module. See info relating to the original xvt elsewhere
 * in this package.
 *----------------------------------------------------------------------*/
#include "rxvt.h"

#include <unistd.h>		/*#define HAVE_UNISTD_H*/
#include <ctype.h>
#include <sys/types.h>
#include <sys/ioctl.h>		/*#define HAVE_SYS_IOCTL_H*/
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xatom.h>

#include "screen.h"
#include "command.h"
#include "xsetup.h"
#include "debug.h"
#include "graphics.h"
#include "defaults.h"

/* chunk size for retrieving the selection property */
#define PROP_SIZE 4096

/* Memory copy methods -- rgg 20/11/95 */
#define MEMCOPY         memcpy          /* no overlap -- rgg 04/11/95 */
/* #define MEMCOPY      memmove */      /* allows overlap -- rgg 04/11/95 */

/* convert pixel dimensions to row/column values */
#define Pixel2Col(x)	(((x) - MARGIN) / RxvtWin.fwidth)
#define Pixel2Row(y)	(((y) - MARGIN) / RxvtWin.fheight)

/*----------------------------------------------------------------------*
 * extern variables referenced
 */

/*----------------------------------------------------------------------*
 * extern variables declared here
 */
int fore_color = FG_COLOR;
int back_color = BG_COLOR;

WindowInfo RxvtWin;

/* set default rstyle colors */
#define INITIAL_RSTYLE		(0)
#define CURRENT_RSTYLE		(rstyle)

/* #define OLD_COLOR_MODEL */		/* use the old color model */

#if !defined (COLOR) && !defined (OLD_COLOR_MODEL)
# define OLD_COLOR_MODEL
#endif

#define NEW_WRAPTYPE
/* Todo: I'd like a way to `remember' which lines have been autowrapped
 * so that mouse selection will not include a newline
 */

#define NEW_SELECTION

/*----------------------------------------------------------------------*
 * local variables
 */
#ifdef COLOR
typedef	unsigned int rend_t;
#else
typedef	unsigned char rend_t;
#endif

typedef struct {
   text_t *text;		/* all the text, including scrollback */
   rend_t *rend;		/* rendition, using the `RS_' flags */
   int row, col;		/* cursor position */
   int tscroll, bscroll;	/* top/bottom of scroll region */
   int relative;		/* relative origin mode flag */
   int autowrap;		/* auto-wrap flag */
   int wrapnext;		/* need to wrap for next char? */
   int insert;			/* insert mode (vs. overstrike) */
   int charset;			/* character set number (0..3) */
} Screen_t;

/* how the screen accounting works [I think]
 *
 * `rend' contains the rendition information (font, bold, etc)
 *
 * `text' contains all the text including the scrollback buffer, each line
 * is a fixed length == (RxvtWin.cols+1) the final character of each is
 * '\n' for wrapped lines and `\0' for non-wrapped lines
 *
 * The layout is such that text rows [0 .. RxvtWin.scrollback - 1] correspond
 * to the scrollback region amd text rows
 * [RxvtWin.scrollback .. RxvtWin.scrollback + RxvtWin.rows - 1]
 * correspond to the screen region [0 .. RxvtWin.rows-1]
 *
 * `row', `tscroll', `bscroll' are bounded by (0, RxvtWin.rows)
 *
 * `col' is bounded by (0, RxvtWin.cols)
 *
 * `RxvtWin.scrollback'
 *	is the maximum number of lines to save in the scrollback buffer.
 *	This will be a fixed number for any particular rxvt instance.
 *
 * `RxvtWin.offset'
 *	the offset back into the scrollback buffer for our current view
 *
 * `RxvtWin.scrolled'
 *	how many lines have been scrolled (saved)
 *		0 <= scrolled <= RxvtWin.scrollback
 *
 * The selection region is defined for [0 .. RxvtWin.rows-1], which
 * corresponds to the regular screen and for [-1 .. -RxvtWin.scrolled]
 * which corresponds to the scrolled region.
 *
 * Actually, the current account mechanism for selection is really
 * screwed-up.  Although it is apparently permissible to select from the
 * scrollback buffer, all of the row values for the selection mechanism
 * are in the [0 .. RxvtWin.rows-1] range which gives some very strange
 * results and is the likely cause of all the selection problems.
 */

static Screen_t screen;
#ifdef NO_SECONDARY_SCREEN
# define NSCREENS	0
#else
# define NSCREENS	1
static Screen_t sec_screen;	/* secondary screens */
#endif
static int current_screen = PRIMARY;

static rend_t rstyle = INITIAL_RSTYLE;
static int focus = 0;
/* this is a pointer to an array of size cols, with a 1 for any
 * location where there is a tab stop */
static char *tabs = NULL;
static text_t *linebuf = NULL;

/* Data for save-screen */
static struct {
   int row, col, charset;
   rend_t rstyle;
   char charset_char;
} save;

/* This tells what's actually on the screen */
static text_t *displayed_text = NULL;
static rend_t *displayed_rend = NULL;

static char charsets [4] = {'B','B','B','B'};

static struct {
   text_t *text;		/* selected text */
   int len;			/* length of selected text */
   int op;			/* current operation */
#  define SELECTION_CLEAR	0
#  define SELECTION_BEGIN	1
#  define SELECTION_INIT	2
#  define SELECTION_ONGOING	3
#  define SELECTION_COMPLETE	4
   int screen;			/* which screen is being used */
   struct {
      int row, col;
   } beg, end, anchor;
   /* int firstr, lastr; */		/* firstr <= row < lastr */
} selection = {NULL, 0, SELECTION_CLEAR, PRIMARY, {0, 0}, {0, 0}, {0, 0} };

/*----------------------------------------------------------------------*
 * extern functions referenced
 */

/*----------------------------------------------------------------------*
 * local functions
 */
static int scroll_text (int row1, int row2, int count);
static INLINE void selection_check (void);

/*----------------------------------------------------------------------*/
/*
 * simple check for integrity of the screen data structures
 */
#ifdef DEBUG_SCREEN
static void
check_text (char *str)
{
   int r, x;
   static char *prev = "?";

   fprintf (stderr, "%s\n", str);
   for (r = 0, x = RxvtWin.cols;
	r < (RxvtWin.rows + RxvtWin.scrollback);
	r++, x += (RxvtWin.cols+1))
     {
     	if (screen.text [x] != '\0'
#ifdef NEW_WRAPTYPE
	    && screen.text [x] != '\n'
#endif
	    )
	  {
	     fprintf (stderr,"%s: %s Violation on row %d\n", str, prev, i);
	     exit (EXIT_FAILURE);
	  }
     }
   MEM_CHECK ("check_text", str);
   prev = str;
}
#else
#define check_text(a)
#endif

/*----------------------------------------------------------------------*
 * Reset the screen - called whenever the screen needs to be repaired
 * due to re-sizing or initialization
 *----------------------------------------------------------------------*/
void
scr_reset (void)
{
   static int prev_rows = -1, prev_cols = -1;
   Screen_t *scr;
   int i;

#ifdef DEBUG_SCREEN
   fprintf (stderr, "reset\n");
#endif
   if (prev_cols == RxvtWin.cols && prev_rows == RxvtWin.rows) return;
#ifdef DEBUG_SCREEN
   fprintf (stderr, "reset continue\n");
#endif
   /* In case rows/columns are invalid */
   if (RxvtWin.cols <= 0) RxvtWin.cols = 80;
   if (RxvtWin.rows <= 0) RxvtWin.rows = 24;

   if (prev_rows > 0)
     {
	RxvtWin.scrolled -= (RxvtWin.rows - prev_rows);
	if (RxvtWin.scrolled < 0)
	  RxvtWin.scrolled = 0;
	else if (RxvtWin.scrolled > RxvtWin.scrollback)
	  RxvtWin.scrolled = RxvtWin.scrollback;
     }

   for (i = 0; i <= NSCREENS; i++)
     {
	text_t *text;
	rend_t *rend;
	int l, r, sl;

	/* allocate space for scrollback (primary screen) + screen */
	if (i == 0)
	  {
	     sl = RxvtWin.scrollback;
	     scr = &screen;
	  }
#if NSCREENS
	else
	  {
	     sl = 0;
	     scr = &sec_screen;
	  }
#endif
	text = MALLOC (((RxvtWin.rows+sl)*(RxvtWin.cols+1)) * sizeof(text_t),
		       "text");
	rend = MALLOC (((RxvtWin.rows+sl)*(RxvtWin.cols+1)) * sizeof(rend_t),
		       "rend");

 	/* copy from old buffer to new buffer, as appropriate. */
	l = prev_rows;
	if (l > 0)
	  l += sl;
	l--;
	for (r = (RxvtWin.rows+sl-1); r >= 0; r--, l--)
	  {
	     int c, roffset;

	     roffset = r * (RxvtWin.cols+1);
	     text [roffset + RxvtWin.cols] = 0;
#ifdef OLD_COLOR_MODEL
	     rend [roffset + RxvtWin.cols] = INITIAL_RSTYLE;
#else
	     rend [roffset + RxvtWin.cols] = CURRENT_RSTYLE;
#endif

	     for (c = 0; c < RxvtWin.cols; c++)
	       {
		  if (l >= 0 && c < prev_cols)
		    {
		       int prev = l * (prev_cols+1);
		       text [roffset + c] = scr->text [prev + c];
		       rend [roffset + c] = scr->rend [prev + c];
		    }
		  else
		    {
		       /* In case there's nothing to copy */
		       text [roffset + c] = ' ';
#ifdef OLD_COLOR_MODEL
		       rend [roffset + c] = INITIAL_RSTYLE;
#else
		       rend [roffset + c] = CURRENT_RSTYLE;
#endif
		    }
	       }
	  }
	FREE (scr->text, "Screen.text","scr_reset");
	FREE (scr->rend, "Screen.rend","scr_reset");
	scr->text = text;
	scr->rend = rend;

	/* Make sure the cursor is on the screen */
	if (prev_rows > 0)
	  scr->row += (RxvtWin.rows - prev_rows);

	if (scr->row < 0)
	  scr->row = 0;
	else if (scr->row >= RxvtWin.rows)
	  scr->row = (RxvtWin.rows-1);

	if (scr->col < 0)
	  scr->col = 0;
	else if (scr->col >= RxvtWin.cols)
	  scr->col = (RxvtWin.cols-1);

	/* reset scroll regions */
	scr->tscroll = 0;
	scr->bscroll = (RxvtWin.rows-1);
     }
   prev_cols = RxvtWin.cols;
   prev_rows = RxvtWin.rows;

   FREE (displayed_text,"displayed_text","scr_reset");
   FREE (displayed_rend,"displayed_rend","scr_reset");
   displayed_text = MALLOC ((RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(text_t),
			    "displayed_text");
   displayed_rend = MALLOC ((RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(rend_t),
			    "displayed_rend");

   memset (displayed_text, ' ',
	   (RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(text_t));
   memset (displayed_rend, INITIAL_RSTYLE,
	   (RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(rend_t));

   /* Make sure the cursor is on the screen */
   if (save.row >= RxvtWin.rows) save.row = (RxvtWin.rows-1);
   if (save.col >= RxvtWin.cols) save.col = (RxvtWin.cols-1);

   FREE (tabs, "tabs", "scr_reset");
   tabs = MALLOC (RxvtWin.cols * sizeof(char),"tabs");

   for (i = 0; i < RxvtWin.cols; i++)
     tabs [i] = ((i%8) == 0);

   FREE (linebuf, "linebuf", "scr_reset");
   linebuf = MALLOC ((RxvtWin.cols+1) * sizeof(text_t), "linebuf");
   tty_resize (RxvtWin.cols, RxvtWin.rows);
}

/*----------------------------------------------------------------------*
 * Perform any initialisation on the screen data structures
 * Called just once at startup
 *----------------------------------------------------------------------*/
void
scr_init (void)
{
   Screen_t *scr;
   int i;
#ifdef DEBUG_SCREEN
   fprintf (stderr, "scr_init");
#endif

   for (i = 0, scr = &screen; i <= NSCREENS; i++)
     {
	scr->text = NULL;
	scr->rend = NULL;
	scr->row = scr->col = 0;
	scr->relative = 0;
	scr->autowrap = 1;
	scr->wrapnext = 0;
	scr->insert = 0;
	scr->charset = 0;
#if NSCREENS
	scr = &sec_screen;
#endif
     }
   fore_color = FG_COLOR;
   back_color = BG_COLOR;
   save.rstyle = rstyle = INITIAL_RSTYLE;
   save.row = save.col = 0;
   save.charset = 0;
   save.charset_char = 'B';

   RxvtWin.scrolled = 0;
   RxvtWin.offset = 0;
   scr_reset ();
   Gr_reset ();
}

/*----------------------------------------------------------------------*
 * Restore power-on configuration
 * Clears screen, restores default fonts, etc
 *----------------------------------------------------------------------*/
void
scr_power_on (void)
{
   Screen_t *scr;
   int i, r;

   fore_color = FG_COLOR;
   back_color = BG_COLOR;
   save.rstyle = rstyle = INITIAL_RSTYLE;
#if NSCREENS
   scr_change_screen (SECONDARY);
   scr_erase_screen (DEL_ENTIRE);
#endif
   scr_change_screen (PRIMARY);
   scr_erase_screen (DEL_ENTIRE);
   XClearWindow (Xdisplay, RxvtWin.vt);

   for (i = 0, scr = &screen; i <= NSCREENS; i++)
     {
	scr->tscroll = 0;
	scr->bscroll = (RxvtWin.rows-1);
	scr->row = scr->col = 0;
	scr->relative = 0;
	scr->autowrap = 1;
	scr->insert = 0;
	scr->charset = 0;
#if NSCREENS
	scr = &sec_screen;
#endif
     }
   for (i = 0; i < sizeof(charsets)/sizeof(charsets[0]); i++)
     charsets [i] = 'B';
   RxvtWin.offset = 0;

   memset (displayed_text, ' ',
	   (RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(text_t));
   memset (displayed_rend, INITIAL_RSTYLE,
	   (RxvtWin.rows * (RxvtWin.cols+1)) * sizeof(rend_t));

   for (r = 0, i = RxvtWin.cols; r < RxvtWin.rows; r++, i += (RxvtWin.cols+1))
     displayed_text [i] = '\0';
   save.row = save.col = 0;
   save.charset = 0;
   save.charset_char = 'B';
   scr_reset ();
   scr_refresh (SLOW_REFRESH);
   Gr_reset ();
}

/*----------------------------------------------------------------------*
 * Handle a backspace
 *----------------------------------------------------------------------*/
void
scr_backspace (void)
{
   check_text("backspace");

   if (selection.op) selection_check ();
   if (screen.col == 0 && screen.row > 0)
     {
	screen.row--;
	screen.col = (RxvtWin.cols-1);
     }
   else if (screen.wrapnext)
     {
#ifdef NEW_WRAPTYPE
	/* fixme here? */
#endif
	screen.wrapnext = 0;
     }
   else
     scr_move (-1, 0, RELATIVE);
}

/*----------------------------------------------------------------------*
 * Ring the bell
 *----------------------------------------------------------------------*/
void
scr_bell (void)
{
#ifdef MAPALERT
   if (map_alert)
     XMapWindow (Xdisplay, RxvtWin.parent);
#endif
#ifdef VISUAL_BELL
   if (visual_bell)
     {
	scr_rvideo_mode (1);
	scr_rvideo_mode (0);
     }
   else
#endif
   XBell (Xdisplay, 0);
}

/*----------------------------------------------------------------------*
 * Change between the main and alternate screen
 *----------------------------------------------------------------------*/
int
scr_change_screen (int scrn)
{
#if NSCREENS
   int x, sb;

   check_text("change screen");

   RxvtWin.offset = 0;
   if (current_screen == scrn)
     return current_screen;

   /* swap screens */
   sb = (RxvtWin.scrollback) * (RxvtWin.cols+1);
   for (x = 0; x < (RxvtWin.rows) * (RxvtWin.cols+1); x++)
     {
	text_t t;
	rend_t r;

	t = screen.text [x+sb];
	screen.text [x+sb] = sec_screen.text [x];
	sec_screen.text [x] = t;

	r = screen.rend [x+sb];
	screen.rend [x+sb] = sec_screen.rend [x];
	sec_screen.rend [x] = r;
     }

   x = screen.row;
   screen.row = sec_screen.row;
   sec_screen.row = x;

   x = screen.col;
   screen.col = sec_screen.col;
   sec_screen.col = x;

   x = screen.relative;
   screen.relative = sec_screen.relative;
   sec_screen.relative = x;

   x = screen.autowrap;
   screen.autowrap = sec_screen.autowrap;
   sec_screen.autowrap = x;

   x = screen.wrapnext;
   screen.wrapnext = sec_screen.wrapnext;
   sec_screen.wrapnext = x;

   x = screen.insert;
   screen.insert = sec_screen.insert;
   sec_screen.insert = x;

   x = screen.charset;
   screen.charset = sec_screen.charset;
   sec_screen.charset = x;
   if (graphics_up)
     {
	Gr_scroll (0);
	Gr_ChangeScreen ();
     }
   x = current_screen;
   current_screen = scrn;
   return x;
#else
   RxvtWin.offset = 0;
   return current_screen;
#endif
}

/*----------------------------------------------------------------------*
 * Sets the rstyle parameter to reflect the selected font
 *----------------------------------------------------------------------*/
static INLINE void
set_font_style (void)
{
   check_text("font style");

   rstyle &= ~RS_FONTMASK;
   switch (charsets [screen.charset]) {
    case '0': rstyle |= RS_GRFONT;	break;
    case 'A': rstyle |= RS_GBFONT;	break;
   }
}

/*----------------------------------------------------------------------*
 * Change the rendition style
 *----------------------------------------------------------------------*/
void
scr_rendition (int mode, int style)
{
   check_text("change rendition");

   if (mode)
     {
	rstyle &= ~style;
	switch (style) {
	 case ~RS_NONE:
	   fore_color = FG_COLOR;
	   back_color = BG_COLOR;
	   break;
#ifndef USE_FAKE_BOLD
	 case RS_BOLD:
	   if (fore_color >= BOLD_COLOR0)
	     scr_fore_color (fore_color - BOLD_COLOR0);
	   break;
	 case RS_BLINK:
	   if (back_color >= BOLD_COLOR0)
	     scr_back_color (back_color - BOLD_COLOR0);
	   break;
#endif
	}
	rstyle = SET_FGBG (rstyle, fore_color, back_color);
     }
   else
     {
	rstyle |= style;
#ifndef USE_FAKE_BOLD
	switch (style) {
	 case RS_BOLD:
	   if (fore_color >= ANSI_COLOR0 && fore_color <= ANSI_COLOR7)
	     scr_fore_color (fore_color - ANSI_COLOR0);
	   break;
	 case RS_BLINK:
	   if (back_color >= ANSI_COLOR0 && back_color <= ANSI_COLOR7)
	     scr_back_color (back_color - ANSI_COLOR0);
	   break;
	}
#endif
     }
   set_font_style ();
}

/*----------------------------------------------------------------------*
 * Indicate a change of keyboard focus
 *----------------------------------------------------------------------*/
void
scr_focus (int in_focus)
{
   check_text("focus");
   focus = in_focus;
}

void
scr_add_lines (text_t *str, int nl_count, int n)
{
   int i, roffset;		/* row offset */

   check_text("add lines");
   if (selection.op) selection_check ();

   if (nl_count > 0)
     {
	nl_count += (screen.row - screen.bscroll);

	if ((nl_count > 0) &&
	    (screen.tscroll == 0) && (screen.bscroll == (RxvtWin.rows-1)))
	  {
	     scroll_text (screen.tscroll, screen.bscroll, nl_count);
	     screen.row -= nl_count;
	     if (screen.row < -RxvtWin.scrollback)
	       screen.row = -RxvtWin.scrollback;
	  }
     }

   if (screen.col >= RxvtWin.cols)
     screen.col = (RxvtWin.cols-1);

   roffset = (screen.row + RxvtWin.scrollback) * (RxvtWin.cols+1);
   for (i = 0; i < n; i++)
     {
	/*
	 * Adds a single character at the current cursor location
	 * new lines in the string
	 */
	RxvtWin.offset = 0;
	if (graphics_up)
	  Gr_scroll (0);
	switch (str [i]) {
	 case 127:	break;
	 case '\t':	scr_tab (1);	break;
	 case '\n':
#ifdef NEW_WRAPTYPE
	   screen.text [roffset + (RxvtWin.cols)] = 0;
#endif
	   if (screen.row == screen.bscroll)
	     scroll_text (screen.tscroll, screen.bscroll, 1);
	   else if (screen.row < (RxvtWin.rows-1))
	     screen.row++;
	   roffset = (screen.row + RxvtWin.scrollback) * (RxvtWin.cols+1);
	   screen.wrapnext = 0;
	   break;

	 case '\r':
#ifdef NEW_WRAPTYPE
	   screen.text [roffset + (RxvtWin.cols)] = 0;
#endif
	   screen.col = 0;
	   screen.wrapnext = 0;
	   break;

	 default:
	   if (screen.wrapnext)
	     {
		if (screen.row == screen.bscroll)
		  scroll_text (screen.tscroll, screen.bscroll, 1);
		else if (screen.row < (RxvtWin.rows-1))
		  screen.row++;
		roffset = ((screen.row + RxvtWin.scrollback) *
			   (RxvtWin.cols+1));
		screen.col = 0;
		screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
		screen.text [roffset + (RxvtWin.cols)] = 0;
#endif
	     }
	   if (screen.insert)
	     scr_insdel_chars (1, INSERT);
	   screen.text [roffset + (screen.col)] = str [i];
	   screen.rend [roffset + (screen.col)] = CURRENT_RSTYLE;
	   screen.col++;
	   if (screen.col == RxvtWin.cols)
	     {
		screen.col--;
		screen.wrapnext = screen.autowrap;
#ifdef NEW_WRAPTYPE
		/* for next time */
		screen.text [roffset + (RxvtWin.cols)] =
		  (screen.autowrap) ? '\n' : 0;
#endif
	     }
	}
     }
}

/*----------------------------------------------------------------------*
 * Scroll the text on the screen
 * Scroll COUNT lines from ROW1 to ROW2 inclusive (ROW1 <= ROW2)
 * scrolling is up for a +ve COUNT and down for a -ve COUNT
 *----------------------------------------------------------------------*/
static int
scroll_text (int row1, int row2, int count)
{
   int r;
   text_t *t_dst, *t_src;
   rend_t *r_dst, *r_src;

#ifdef NEW_SELECTION
   if (selection.op)		/* move selected region too */
#endif
     {
#ifdef NEW_SELECTION
	selection.beg.row -= count;
#endif
	selection.end.row -= count;
	selection.anchor.row -= count;
	/* could check ranges here and make sure selection is okay
	 * don't scroll into scrollback depending on the region etc,
	 * but leave for now
	 */
     }

   if (count > 0)		/* scroll up */
     {
	int n, x;
	/* if the line scrolls off the top of the screen,
	 * shift the entire scrollback buffer too */
	if ((row1 == 0) && (current_screen == PRIMARY))
	  {
	     row1 = -RxvtWin.scrollback;
	     RxvtWin.scrolled += count;
	     if (RxvtWin.scrolled > RxvtWin.scrollback)
	       RxvtWin.scrolled = RxvtWin.scrollback;
	  }

	x = ((row1 + RxvtWin.scrollback) * (RxvtWin.cols+1));
	t_dst = &screen.text [x];
	r_dst = &screen.rend [x];

	n = (row2 - row1 + 1);
	if (count > n)
	  {
	     count = n;
	     n = 0;
	  }
	else
	  {
	     n -= count;
	  }

	x += count * (RxvtWin.cols+1);
	t_src = &screen.text [x];
	r_src = &screen.rend [x];

	/* Forward overlapping memcpy's -- probably OK */
	if (n > 0)
	  {
	     n *= (RxvtWin.cols+1);
	     MEMCOPY (t_dst, t_src, n * sizeof(text_t));
	     t_dst += n;
	     MEMCOPY (r_dst, r_src, n * sizeof(rend_t));
	     r_dst += n;
	  }

	/* copy blank lines in at the bottom */
	memset (t_dst, ' ',
		(count * (RxvtWin.cols+1)) * sizeof(text_t));
#ifdef OLD_COLOR_MODEL
	memset (r_dst, INITIAL_RSTYLE,
		(count * (RxvtWin.cols+1)) * sizeof(rend_t));
#else
	  {
	     rend_t *rdst = r_dst;
	     int total = (count * (RxvtWin.cols+1));
	     while (total--) *rdst++ = CURRENT_RSTYLE;
	  }
#endif
	t_dst--;		/* terminate each line */
	for (r = 0; r < (count+1); r++)
	  {
	     *t_dst = 0;
	     t_dst += (RxvtWin.cols+1);
	  }
     }
   else if (count < 0)		/* scroll down */
     {
	int x;

	/* do one line at a time to avoid backward overlapping memcpy's */
	x = (row2 + RxvtWin.scrollback) * (RxvtWin.cols+1);
	t_dst = &screen.text [x];
	r_dst = &screen.rend [x];

	x += (count) * (RxvtWin.cols+1);
	t_src = &screen.text [x];
	r_src = &screen.rend [x];
	for (r = row2; r >= (row1 - count); r--)
	  {
	     MEMCOPY (t_dst, t_src, (RxvtWin.cols+1) * sizeof(text_t));
	     t_dst -= (RxvtWin.cols+1);
	     t_src -= (RxvtWin.cols+1);

	     MEMCOPY (r_dst, r_src, (RxvtWin.cols+1) * sizeof(rend_t));
	     r_dst -= (RxvtWin.cols+1);
	     r_src -= (RxvtWin.cols+1);
	  }
	/* copy blank lines in at the top */
	for (; r >= row1; r--)
	  {
	     memset (t_dst, ' ', RxvtWin.cols * sizeof(text_t));
	     t_dst [RxvtWin.cols] = 0;
#ifdef OLD_COLOR_MODEL
	     memset (r_dst, INITIAL_RSTYLE, RxvtWin.cols * sizeof(rend_t));
#else
	       {
		  rend_t *rdst = r_dst;
		  int total = RxvtWin.cols;
		  while (total--) *rdst++ = CURRENT_RSTYLE;
	       }
#endif
	     t_dst -= (RxvtWin.cols+1);
	     r_dst -= (RxvtWin.cols+1);
	  }
     }
   if (graphics_up)
     Gr_scroll (count);
   return count;
}

/*----------------------------------------------------------------------*
 * Move the cursor to a new tab position
 *
 * COUNT is +ve, move forward.  COUNT is -ve, move backward
 *----------------------------------------------------------------------*/
void
scr_tab (int count)
{
   int x = screen.col;

   if (count > 0) 		/* tab forward */
     {
	int i;
	for (i = x+1; i < RxvtWin.cols; i++)
	  {
	     if (tabs [i])
	       {
		  x = i;
		  count--;
		  if (!count) break;
	       }
	  }
     }
   else if (count < 0)		/* tab backward */
     {
	int i;
	count = -count;
	for (i = x-1; i >= 0; i--)
	  {
	     if (tabs [i])
	       {
		  x = i;
		  count--;
		  if (!count) break;
	       }
	  }
     }
   else
     return;

   if (x != screen.col)
     scr_move (x, 0, ROW_RELATIVE);
}

/*----------------------------------------------------------------------*
 * Move the cursor to a new position.
 * The relative argument is a pair of flags that specify relative
 * rather than absolute motion.
 *----------------------------------------------------------------------*/
void
scr_move (int x, int y, int relative)
{
   RxvtWin.offset = 0;
   if (graphics_up)
     Gr_scroll (0);
   screen.col = (relative & COL_RELATIVE) ? screen.col + x : x;

   if (screen.col < 0)
     screen.col = 0;
   else if (screen.col >= RxvtWin.cols)
     screen.col = (RxvtWin.cols-1);

   if (relative & ROW_RELATIVE)
     {
	if (y > 0)
	  {
	     if ((screen.row <= screen.bscroll) &&
		 (screen.row + y > screen.bscroll))
	       screen.row = screen.bscroll;
	     else
	       screen.row += y;
	  }
	else if (y < 0)
	  {
	     if ((screen.row >= screen.tscroll) &&
		 (screen.row + y < screen.tscroll))
	       screen.row = screen.tscroll;
	     else
	       screen.row += y;
	  }
     }
   else
     {
	if (screen.relative)		/* relative origin mode */
	  {
	     screen.row = y + screen.tscroll;
	     if (screen.row > screen.bscroll)
	       screen.row = screen.bscroll;
	  }
	else
	  screen.row = y;
     }
   if (screen.row < 0) screen.row = 0;
   else if (screen.row >= RxvtWin.rows)
     screen.row = (RxvtWin.rows-1);
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* fixme here? */
#endif
}

/*----------------------------------------------------------------------*
 * Move the cursor down one line and scroll if necessary.
 *----------------------------------------------------------------------*/
void
scr_index (int dirn)
{
   RxvtWin.offset = 0;
   if (graphics_up)
     Gr_scroll (0);

   check_text ("index");

   if ((screen.row == screen.bscroll && dirn == UP) ||
       (screen.row == screen.tscroll && dirn == DOWN))
     scroll_text (screen.tscroll, screen.bscroll, dirn);
   else
     screen.row += dirn;
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* fixme here? */
#endif
   if (selection.op) selection_check ();
}

/*----------------------------------------------------------------------*
 * Save the cursor position and rendition style.
 *----------------------------------------------------------------------*/
void
scr_save_cursor (void)
{
   save.row = screen.row;
   save.col = screen.col;
   save.rstyle = rstyle;
   save.charset = screen.charset;
   save.charset_char = charsets [screen.charset];
}

/*----------------------------------------------------------------------*
 * Restore the cursor position and rendition style.
 *----------------------------------------------------------------------*/
void
scr_restore_cursor (void)
{
   screen.row = save.row;
   screen.col = save.col;
   rstyle = SET_FGBG (save.rstyle, fore_color, back_color);
   screen.charset = save.charset;
   charsets [screen.charset] = save.charset_char;
   set_font_style ();
}

/*----------------------------------------------------------------------*
 * erase part or the whole of a line
 *----------------------------------------------------------------------*/
void
scr_erase_line (int mode)
{
   int x, count;
   check_text("erase line");

   RxvtWin.offset = 0;
   if (graphics_up)
     Gr_scroll (0);

   x = (screen.row + RxvtWin.scrollback) * (RxvtWin.cols+1);
   switch (mode) {
    case DEL_END:		/* 0: erase to end */
      check_text ("erase line end");
      x += (screen.col);
      count = (RxvtWin.cols - screen.col);
      break;

    case DEL_START:		/* 1: erase to beginning */
      check_text ("erase line start");
      count = (screen.col + 1);
      break;

    case DEL_ENTIRE:		/* 2: erase entire */
      check_text("erase line entire");
      count = (RxvtWin.cols);
      break;

    default:
      check_text("erase line none");
      return;
   }

   memset (&screen.text [x], ' ', count * sizeof(text_t));
#ifdef OLD_COLOR_MODEL
   memset (&screen.rend [x], INITIAL_RSTYLE, count * sizeof(rend_t));
#else
     {
	rend_t *rdst = &screen.rend [x];
	while (count--) *rdst++ = CURRENT_RSTYLE;
     }
#endif
   x = (screen.row + RxvtWin.scrollback) * (RxvtWin.cols+1) + (RxvtWin.cols);
   screen.text [x] = 0;

   check_text("erase line done");
   if (selection.op) selection_check ();
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
   /* fixme here? */
#endif
}

/*----------------------------------------------------------------------*
 * erase part or the whole of the screen
 *----------------------------------------------------------------------*/
void
scr_erase_screen (int mode)
{
   int r, startr, startc, endr, endc;

   check_text("erase screen");

   switch (mode) {
    case DEL_END:		/* 0: erase to end */
      startr = screen.row; endr = (RxvtWin.rows-1);
      startc = screen.col; endc = (RxvtWin.cols-1);
      break;
    case DEL_START:		/* 1: erase to beginning */
      startr = 0; endr = screen.row;
      startc = 0; endc = screen.col;
      break;
    case DEL_ENTIRE:		/* 2: erase entire */
      startr = 0; endr = (RxvtWin.rows-1);
      startc = 0; endc = (RxvtWin.cols-1);
      Gr_ClearScreen ();
      break;
    default:
      return;
      break;
   }
   for (r = startr; r <= endr; r++)
     {
	int x, count;
	x = ((r + RxvtWin.scrollback - RxvtWin.offset) * (RxvtWin.cols+1));

	count = (r == endr) ? (endc + 1) : (RxvtWin.cols);

	if (r == startr)
	  {
	     x += startc;
	     count -= startc;
	  }

	memset (&screen.text [x], ' ', count * sizeof(text_t));
#ifdef OLD_COLOR_MODEL
	memset (&screen.rend [x], INITIAL_RSTYLE, count * sizeof(rend_t));
#else
	  {
	     rend_t *rdst = &screen.rend [x];
	     while (count--) *rdst++ = CURRENT_RSTYLE;
	  }
#endif
     }
}

/*----------------------------------------------------------------------*
 * Fill the screen with E's
 *----------------------------------------------------------------------*/
void
scr_E (void)
{
   int r, x;

   check_text ("E");

   x = ((RxvtWin.scrollback - RxvtWin.offset) * (RxvtWin.cols+1));
   for (r = 0; r < (RxvtWin.rows); r++)
     {
	memset (&screen.text [x], 'E',
		RxvtWin.cols * sizeof(text_t));
	memset (&screen.rend [x], INITIAL_RSTYLE,
		RxvtWin.cols * sizeof(rend_t));

	x += (RxvtWin.cols + 1);
	screen.text [x-1] = 0;
     }
}

/*----------------------------------------------------------------------*
 * Insert or delete count lines and scroll
 * insdel == +1
 *	delete lines, scroll up the bottom of the screen to fill the gap
 * insdel == -1
 *	insert lines, scroll down the lower lines
 * other values of insdel are undefined
 *----------------------------------------------------------------------*/
void
scr_insdel_lines (int count, int insdel)
{
   check_text("insdel lines");

   if (screen.row > screen.bscroll)
     return;

   if (count > screen.bscroll - screen.row + 1)
     {
	if (insdel == DELETE)
	  return;
	else if (insdel == INSERT)
	  count = screen.bscroll - screen.row + 1;
     }

   RxvtWin.offset = 0;
   if (graphics_up)
     Gr_scroll (0);
   scroll_text (screen.row, screen.bscroll, insdel*count);
   screen.wrapnext = 0;
#ifdef NEW_WRAPTYPE
	/* fixme here? */
#endif
}

/*----------------------------------------------------------------------*
 * insert or Delete count characters from the current position
 * insdel == +2, erase  chars
 * insdel == +1, delete chars
 * insdel == -1, insert chars
 *----------------------------------------------------------------------*/
void
scr_insdel_chars (int count, int insdel)
{
   int roffset;			/* row offset */
   text_t *text, *textend;
   rend_t *rend, *rendend;

   check_text("insdel chars");

   if (insdel == ERASE)
     {
	if (count > screen.col)
	  count = screen.col;
	if (count <= 0)
	  return;
	screen.col -= count;	/* move backwards */
	insdel = DELETE;	/* delete chars */
     }
   else if (count > (RxvtWin.cols - screen.col))
     {
	count = (RxvtWin.cols - screen.col);
     }
   if (count <= 0)
     return;

   RxvtWin.offset = 0;
   if (graphics_up)
     Gr_scroll (0);
   if (selection.op) selection_check ();

   roffset = (screen.row + RxvtWin.scrollback) * (RxvtWin.cols+1);

   text = &screen.text [roffset + (screen.col)];
   rend = &screen.rend [roffset + (screen.col)];
   if (insdel == DELETE)
     {
#ifdef NEW_WRAPTYPE
	screen.text [roffset + (RxvtWin.cols)] = 0;	/* not autowrapped */
#endif
	/* overlapping copy */
	while (*text && text [count])
	  {
	     *text = text [count];
	     *rend = rend [count];
	     text++;
	     rend++;
	  }

	/* fill in the end of the line */
	while (*text)
	  {
	     *text = ' ';
#ifdef OLD_COLOR_MODEL
	     *rend = INITIAL_RSTYLE;
#else
	     *rend = CURRENT_RSTYLE;
#endif
	     text++;
	     rend++;
	  }
     }
   else
     {
	/* INSERT count characters */
	textend = &screen.text [roffset + (RxvtWin.cols-1)];
	rendend = &screen.rend [roffset + (RxvtWin.cols-1)];

	textend [1] = 0;	/* not autowrapped */
	while (textend - count >= text)
	  {
	     *textend = *(textend-count);
	     *rendend = *(rendend-count);
	     textend--;
	     rendend--;
	  }

	/* fill in the gap */
	while (textend >= text)
	  {
	     *textend = ' ';
#ifdef OLD_COLOR_MODEL
	     *rendend = INITIAL_RSTYLE;
#else
	     *rendend = CURRENT_RSTYLE;
#endif
	     textend--;
	     rendend--;
	  }
     }
   screen.wrapnext = 0;
}

/*----------------------------------------------------------------------*
 * set the scroll region
 *----------------------------------------------------------------------*/
void
scr_scroll_region (int top, int bot)
{
   if (top < 0) top = 0;
   if (bot >= RxvtWin.rows) bot = (RxvtWin.rows-1);
   if (top > bot) return;

   screen.tscroll = top;
   screen.bscroll = bot;
   scr_move (0, 0, 0);
}

/*----------------------------------------------------------------------*
 * Set or unset automatic wrapping
 *----------------------------------------------------------------------*/
void
scr_autowrap (int mode)
{
   screen.autowrap = (mode != 0);
}

/*----------------------------------------------------------------------*
 * Set or unset margin origin mode
 *
 * In absolute origin mode, line numbers are counted relative to top margin
 * of screen, the cursor can be moved outside the scrolling region. In
 * relative mode line numbers are relative to top margin of scrolling
 * region and the cursor cannot be moved outside
 *----------------------------------------------------------------------*/
void
scr_relative_origin (int mode)
{
   screen.relative = (mode != 0);
   scr_move (0, 0, 0);
}

/*----------------------------------------------------------------------*
 * Set or unset automatic insert mode
 *----------------------------------------------------------------------*/
void
scr_insert_mode (int mode)
{
   screen.insert = (mode != 0);
}

/*----------------------------------------------------------------------*
 * Move the display so that line represented by scrollbar value y is at
 * the top of the screen
 *----------------------------------------------------------------------*/
#ifndef SCROLLBAR_NONE
void
scr_move_to (int y)
{
   check_text("move to");

   RxvtWin.offset = (((sbar.h-1) - y) * ((RxvtWin.rows-1) + RxvtWin.scrolled)
		     / (sbar.h-1)) - (RxvtWin.rows-1);

   if (RxvtWin.offset < 0)
     RxvtWin.offset = 0;
   else if (RxvtWin.offset > RxvtWin.scrolled)
     RxvtWin.offset = RxvtWin.scrolled;
   if (graphics_up)
     Gr_scroll (0);
}
#endif

/*----------------------------------------------------------------------*
 * page the screen up/down NLINES
 *----------------------------------------------------------------------*/
void
scr_page (int direction, int nlines)
{
   check_text("page");

   if (nlines <= 0)
     nlines = 1;
   else if (nlines > RxvtWin.rows)
     nlines = RxvtWin.rows;
   RxvtWin.offset += nlines * direction;

   if (RxvtWin.offset < 0)
     RxvtWin.offset = 0;
   else if (RxvtWin.offset > RxvtWin.scrolled)
     RxvtWin.offset = RxvtWin.scrolled;
   if (graphics_up)
     Gr_scroll (0);
}

/*
 * If (row,col) is within a selected region of text, remove the selection
 */
static INLINE void
selection_check (void)
{
   int c1, c2, r1, r2;
   check_text("check selection");

   if (current_screen != selection.screen)
     return;

#ifdef NEW_SELECTION
   if ((selection.anchor.row < -RxvtWin.scrolled) ||
       (selection.anchor.row >= RxvtWin.rows) ||
       (selection.beg.row < -RxvtWin.scrolled) ||
       (selection.beg.row >= RxvtWin.rows) ||
       (selection.end.row < -RxvtWin.scrolled) ||
       (selection.end.row >= RxvtWin.rows))
     {
	scr_selection_clear ();
	return;
     }

   r1 = (screen.row - RxvtWin.offset);
   c1 = ((r1 - selection.anchor.row) * (r1 - selection.end.row));
#else
   c1 = ((screen.row - selection.anchor.row) *
	 (screen.row - selection.end.row));
#endif
   /* selection.anchor.row > screen.row - RxvtWin.offset
    * or
    * selection.end.row > screen.row - RxvtWin.offset
    */
   if (c1 < 0)
     scr_selection_clear ();
   /* selection.anchor.row == screen.row || selection.end.row == screen.row */
   else if (c1 == 0)
     {
	/* We're on the same row as the start or end of selection */
	if ((selection.anchor.row < selection.end.row) ||
	    ((selection.anchor.row == selection.end.row) &&
	     (selection.anchor.col < selection.end.col)))
	  {
	     r1 = selection.anchor.row;
	     c1 = selection.anchor.col;
	     r2 = selection.end.row;
	     c2 = selection.end.col;
	  }
	else
	  {
	     r1 = selection.end.row;
	     c1 = selection.end.col;
	     r2 = selection.anchor.row;
	     c2 = selection.anchor.col;
	  }
	if ((screen.row == r1) && (screen.row == r2))
	  {
	     if ((screen.col >= c1) && (screen.col <= c2))
	       scr_selection_clear ();
	  }
	else if (((screen.row == r1) && (screen.col >= c1)) ||
		 ((screen.row == r2) && (screen.col <= c2)))
	  scr_selection_clear ();
     }
}

#if 0
void
scr_selection_range (int firstr, int lastr)
{
   if (firstr >= lastr ||
       firstr < 0 || firstr >= RxvtWin.rows ||
       lastr <= 0 || lastr > RxvtWin.rows)
    return;
   selection.firstr = firstr;
   selection.lastr = lastr;
}
#endif

/*----------------------------------------------------------------------*
 * make the selection currently delimited by the selection end markers
 *----------------------------------------------------------------------*/
void
scr_selection_make (Time time)
{
   text_t *str;
   int r, c1, c2, startr, startc, endr, endc;
   int roffset;			/* row offset */

   switch (selection.op) {
    case SELECTION_ONGOING:
      break;

    case SELECTION_INIT:
      scr_selection_clear ();
      selection.end.row = selection.anchor.row = selection.beg.row;
      selection.end.col = selection.anchor.col = selection.beg.col;
      /*drop*/
    case SELECTION_BEGIN:
      selection.op = SELECTION_COMPLETE;
    default:
      return;
      break;
   }
   selection.op = SELECTION_COMPLETE;

   FREE (selection.text, "selection.text", "selection make");
   selection.text = NULL;
   selection.len = 0;

   selection.screen = current_screen;
   /* Set start/end row/col to point to the selection endpoints */
   if (selection.end.row < selection.anchor.row ||
       (selection.end.row == selection.anchor.row &&
	selection.end.col <= selection.anchor.col))
     {
	startr = selection.end.row; endr = selection.anchor.row;
	startc = selection.end.col; endc = selection.anchor.col;
     }
   else
     {
	startr = selection.anchor.row; endr = selection.end.row;
	startc = selection.anchor.col; endc = selection.end.col;
     }

   if (
#ifdef NEW_SELECTION
       (startr < -RxvtWin.scrolled || endr >= RxvtWin.rows)
#else
       (startr < (RxvtWin.offset - RxvtWin.scrollback) ||
	endr >= (RxvtWin.offset + RxvtWin.rows))
#endif
       )
     {
	scr_selection_clear ();
	return;
     }

   str = MALLOC(((endr - startr + 1)*(RxvtWin.cols+1) + 1) * sizeof(text_t),
		"sel_text");
   selection.text = str;
   *str = '\0';

   /* save all points between start and end with selection flag */
#ifdef NEW_SELECTION
   roffset = ((startr + RxvtWin.scrollback) * (RxvtWin.cols+1));
#else
   roffset = ((startr + RxvtWin.scrollback - RxvtWin.offset) *
	      (RxvtWin.cols+1));
#endif
   for (r = startr; r <= endr; r++)
     {
	c1 = (r == startr) ? startc : 0;
	c2 = (r == endr) ? endc : (RxvtWin.cols-1);

	while (c1 <= c2)
	  {
	     *str++ = screen.text [roffset + c1];
	     c1++;
	  }
	if (c2 == (RxvtWin.cols-1))
	  {
#ifdef NEW_WRAPTYPE
	     /* not autowrap */
	     if (screen.text [roffset + (RxvtWin.cols)] != '\n')
#endif
	       {
		  str--;
		  while ((str >= selection.text) && isspace (*str)) str--;
		  str++;
		  *str = '\n';
		  str++;
	       }
	  }
	roffset += (RxvtWin.cols+1);
     }
   *str = '\0';

   selection.len = strlen (selection.text);
   if (selection.len <= 0)
     return;
   XSetSelectionOwner (Xdisplay, XA_PRIMARY, RxvtWin.vt, time);
   if (XGetSelectionOwner (Xdisplay, XA_PRIMARY) != RxvtWin.vt)
     print_error ("Could not get primary selection");

   /* Place in CUT_BUFFER0 for backup */
   XChangeProperty (Xdisplay, DefaultRootWindow (Xdisplay), XA_CUT_BUFFER0,
		    XA_STRING, 8, PropModeReplace,
		    selection.text, selection.len);
}

/*----------------------------------------------------------------------*
 * respond to a request for our current selection
 *----------------------------------------------------------------------*/
void
scr_selection_send (XSelectionRequestEvent *rq)
{
   XEvent event;

   event.xselection.type	= SelectionNotify;
   event.xselection.display	= rq->display;
   event.xselection.requestor	= rq->requestor;
   event.xselection.selection	= rq->selection;
   event.xselection.target	= rq->target;
   event.xselection.property	= None;
   event.xselection.time	= rq->time;

   if (rq->target == XA_STRING)
     {
	XChangeProperty (Xdisplay, rq->requestor, rq->property,
			 XA_STRING, 8,
			 PropModeReplace,
			 selection.text, selection.len);
	event.xselection.property = rq->property;
     }
   XSendEvent (Xdisplay, rq->requestor, False, 0, &event);
}

/*----------------------------------------------------------------------*
 * Request the current primary selection
 *----------------------------------------------------------------------*/
void
scr_selection_request (Time time, int x, int y)
{
   Atom prop;

   /* First check that the release is within the window */
   if (x < 0 || x >= RxvtWin.pwidth || y < 0 || y >= RxvtWin.pheight)
     return;
   if (selection.text != NULL)
     {
	/* The selection is internal */
#ifdef REPLACE_SELECTION_NEWLINE
	text_t *p = selection.text;
	text_t *pmax = p + selection.len;
	while (p < pmax)
	  {
	     if (*p == '\n') *p = '\r';
	     p++;
	  }
#endif
	tty_write (selection.text, selection.len);
	return;
     }

   if (XGetSelectionOwner (Xdisplay, XA_PRIMARY) == None)
     {
	/* No primary selection so use the cut buffer */
	scr_paste_primary (DefaultRootWindow (Xdisplay),
			   XA_CUT_BUFFER0, False);
	return;
     }
   prop = XInternAtom (Xdisplay, "VT_SELECTION", False);
   XConvertSelection (Xdisplay, XA_PRIMARY, XA_STRING,
		      prop, RxvtWin.vt, time);
}

/*----------------------------------------------------------------------*
 * Respond to a notification that a primary selection has been sent
 *----------------------------------------------------------------------*/
void
scr_paste_primary (Window win, int prop, int Delete)
{
   Atom actual_type;
   int i;
   long nitems, bytes_after, nread;
   unsigned char * data;

   if (prop == None)
     return;

   nread = 0;
   do
     {
	if ((XGetWindowProperty (Xdisplay, win, prop,
				 nread/4, PROP_SIZE, Delete,
				 AnyPropertyType, &actual_type, &i,
				 &nitems, &bytes_after,
				 &data) != Success) ||
	    (actual_type != XA_STRING))
	  {
	     XFree (data);
	     return;
	  }

	nread += nitems;
#ifdef REPLACE_SELECTION_NEWLINE
	/* make a \n to \r mapping for cut and paste only */
	for (i = 0; i < nitems; i++)
	  if (data [i] == '\n') data [i] = '\r';
#endif
	tty_write (data, nitems);
	XFree (data);
     } while (bytes_after > 0);
}

/*----------------------------------------------------------------------*
 * Clear the current selection
 *----------------------------------------------------------------------*/
void
scr_selection_clear (void)
{
   int x, nrow;

   selection.op = SELECTION_CLEAR;
   selection.end.row = selection.anchor.row = 0;
   selection.end.col = selection.anchor.col = 0;
   nrow = RxvtWin.rows;
   if (current_screen == PRIMARY)
     nrow += RxvtWin.scrollback;

   for (x = 0; x < nrow * (RxvtWin.cols+1); x++)
     screen.rend [x] &= ~RS_SELECTED;
}

void
scr_selection_delete (void)
{
   FREE (selection.text, "sel_text", "clear_sel");
   selection.text = NULL;
   selection.len = 0;

   selection.op = SELECTION_CLEAR;
   scr_selection_clear ();
}

/*----------------------------------------------------------------------*
 * mark selected points (used by scr_selection_extend)
 *----------------------------------------------------------------------*/
static void
scr_set_clr_sel (int set, int startr, int startc, int endr, int endc)
{
   int r, c1, c2, roffset;

#ifdef NEW_SELECTION
   /* startr <= endr */
   if ((startr < -RxvtWin.scrolled) || (endr >= RxvtWin.rows))
     {
	scr_selection_clear ();
	return;
     }
#endif

   for (r = startr; r <= endr; r++)
     {
#ifdef NEW_SELECTION
	roffset = ((r + RxvtWin.scrollback) * (RxvtWin.cols+1));
#else
	roffset = ((r + RxvtWin.scrollback - RxvtWin.offset) *
		   (RxvtWin.cols+1));
#endif
	c1 = (r == startr) ? startc : 0;
	c2 = (r == endr) ? endc : (RxvtWin.cols-1);

	while (c1 <= c2)
	  {
	     if (set)
	       screen.rend [roffset + c1] |= RS_SELECTED;
	     else
	       screen.rend [roffset + c1] &= ~RS_SELECTED;
	     c1++;
	  }
     }
}

/*----------------------------------------------------------------------*
 * start a selection using the specified unit
 *----------------------------------------------------------------------*/
void
scr_selection_start (int x, int y)
{
#ifdef NEW_SELECTION
   if (selection.op)
     {
	/* startr <= endr */
	if ((selection.end.row < -RxvtWin.scrolled) ||
	    (selection.anchor.row < - RxvtWin.scrolled))
	  {
	     scr_selection_clear ();
	  }
	else
	  /* determine direction of new selection */
	  if (selection.end.row < selection.anchor.row ||
	      (selection.end.row == selection.anchor.row &&
	       selection.end.col <= selection.anchor.col))
	  scr_set_clr_sel (0,	/* up */
			   selection.end.row, selection.end.col,
			   selection.anchor.row, selection.anchor.col);
	else
	  scr_set_clr_sel (0,	/* down */
			   selection.anchor.row, selection.anchor.col,
			   selection.end.row, selection.end.col);
     }
#endif
   selection.op = SELECTION_INIT;

   selection.beg.col = Pixel2Col (x);
   selection.beg.row = Pixel2Row (y);

#ifdef NEW_SELECTION
   selection.beg.row -= RxvtWin.offset;
#endif
}

/*----------------------------------------------------------------------*
 * extend the selection
 *----------------------------------------------------------------------*/
void
scr_selection_extend (int hilite, int x, int y)
{
   int old_row, old_col, old_dirn, dirn;

   switch (selection.op) {
    case SELECTION_INIT:
      scr_selection_clear ();
      selection.end.col = selection.anchor.col = selection.beg.col;
      selection.end.row = selection.anchor.row = selection.beg.row;
      /*drop*/
    case SELECTION_BEGIN:
      selection.op = SELECTION_BEGIN;
      break;

    case SELECTION_COMPLETE:
    case SELECTION_ONGOING:
      selection.op = SELECTION_ONGOING;
      break;

    case SELECTION_CLEAR:
      scr_selection_start (x, y);
    default:
      return;
      break;
   }

   /* Remember old selection for virtual removal */
   old_row = selection.end.row;
   old_col = selection.end.col;

#ifdef NEW_SELECTION
   if ((old_row < -RxvtWin.scrolled) ||
       (selection.anchor.row < -RxvtWin.scrolled))
     {
	scr_selection_clear ();
	return;
     }
#endif

   /* Figure out where new selection is */
   selection.end.col = Pixel2Col (x);
   selection.end.row = Pixel2Row (y);

   if (selection.end.col < 0)
     selection.end.col = 0;
   else if (selection.end.col >= RxvtWin.cols)
     selection.end.col = (RxvtWin.cols-1);

#ifdef NEW_SELECTION
   selection.end.row -= RxvtWin.offset;
   if (selection.end.row < -RxvtWin.scrolled)
     {
	scr_selection_clear ();
	return;
     }
#else
   if (selection.end.row < 0)
     selection.end.row = 0;
#endif
   else if (selection.end.row >= RxvtWin.rows)
     selection.end.row = (RxvtWin.rows-1);

   if ((selection.op == SELECTION_BEGIN) &&
       ((selection.end.col != selection.anchor.col) ||
	(selection.end.row != selection.anchor.row)))
     selection.op = SELECTION_ONGOING;

   /* If new selection is same as old selection just return
    * or if no highlighting was requested
    */
   if (!hilite ||
       (selection.end.row == old_row && selection.end.col == old_col))
     return;

   /* virtual removal -- delete old highlighting and replace with new */

   /* determine direction of old selection */
   old_dirn = (old_row < selection.anchor.row ||
	       (old_row == selection.anchor.row &&
		old_col <= selection.anchor.col)) ? UP : DOWN;

   /* determine direction of new selection */
   dirn = (selection.end.row < selection.anchor.row ||
	   (selection.end.row == selection.anchor.row &&
	    selection.end.col <= selection.anchor.col)) ? UP : DOWN;

   /* If old and new direction are different, clear old, set new */
   if (dirn != old_dirn)
     {
	if (old_dirn == UP)
	  {
	     scr_set_clr_sel (0,
			      old_row, old_col,
			      selection.anchor.row, selection.anchor.col);
	     scr_set_clr_sel (1,
			      selection.anchor.row, selection.anchor.col,
			      selection.end.row, selection.end.col);
	  }
	else
	  {
	     scr_set_clr_sel (0,
			      selection.anchor.row, selection.anchor.col,
			      old_row, old_col);
	     scr_set_clr_sel (1,
			      selection.end.row, selection.end.col,
			      selection.anchor.row, selection.anchor.col);
	  }
     }
   else
     {
	if (old_dirn == UP)
	  {
	     if (old_row < selection.end.row ||
		 (old_row == selection.end.row &&
		  old_col < selection.end.col))
	       {
		  scr_set_clr_sel (0,
				   old_row, old_col,
				   selection.end.row, selection.end.col);
		  scr_set_clr_sel (1,
				   selection.end.row, selection.end.col,
				   selection.end.row, selection.end.col);
	       }
	     else
	       {
		  scr_set_clr_sel (1,
				   selection.end.row, selection.end.col,
				   old_row, old_col);
	       }
	  }
	else
	  {
	     if (selection.end.row < old_row ||
		 (selection.end.row == old_row &&
		  selection.end.col < old_col))
	       {
		  scr_set_clr_sel (0,
				   selection.end.row, selection.end.col,
				   old_row, old_col);
		  scr_set_clr_sel (1,
				   selection.end.row, selection.end.col,
				   selection.end.row, selection.end.col);
	       }
	     else
	       {
		  scr_set_clr_sel (1,
				   old_row, old_col,
				   selection.end.row, selection.end.col);
	       }
	  }
     }
   /* scr_refresh_region (old_row, selection.end.row, SLOW_REFRESH);*/
}

#ifdef MULTIPLE_CLICKS
/*----------------------------------------------------------------------*
 * Double click selection, by Edward. Der-Hua Liu, Taiwan
 * Added cut char class support:	A. Haritsis <ah@doc.ic.ac.uk>
 *----------------------------------------------------------------------*/
void
scr_selection_multiclick (int clicks, int x, int y)
{
   int c, r, startx, endx;

   switch (clicks) {
    case 1:			/* single click */
      scr_selection_start (x, y);
      return;
      break;

    case 2:			/* double click */
      c = Pixel2Col (x);
      if (c < 0) c = 0;
      else if (c >= RxvtWin.cols) c = (RxvtWin.cols-1);
      startx = endx = c;

      r = Pixel2Row (y);
      if (r < 0) r = 0;
      else if (r >= RxvtWin.rows) r = (RxvtWin.rows-1);
      x = (r + RxvtWin.scrollback - RxvtWin.offset)*(RxvtWin.cols+1);

      while (startx > 0 && !strchr (rs_cutchars, screen.text [x+startx-1]))
	startx--;
      while (endx < (RxvtWin.cols-1) &&
	     !strchr (rs_cutchars, screen.text [x+endx+1]))
	endx++;
      break;

    case 3:			/* triple click */
      startx = 0;
      endx = (RxvtWin.cols-1);
      break;

    default:
      return;
      break;
   }
   scr_selection_start (startx * RxvtWin.fwidth + MARGIN, y);
   scr_selection_extend (1, endx  * RxvtWin.fwidth + MARGIN, y);
}
#endif

/*----------------------------------------------------------------------*
 * Report the current cursor position
 *----------------------------------------------------------------------*/
void
scr_report_position (void)
{
   tty_printf ("\033[%d;%dR", screen.row + 1, screen.col + 1);
}

/*----------------------------------------------------------------------*
 * Set a font
 *----------------------------------------------------------------------*/
void
scr_set_charset (int set, text_t a)
{
   charsets [set] = a;
   set_font_style ();
}

/*----------------------------------------------------------------------*
 * choose a font
 *----------------------------------------------------------------------*/
void
scr_choose_charset (int set)
{
   screen.charset = set;
   set_font_style ();
}

/*----------------------------------------------------------------------*
 * for the box starting at x,y with size width, height
 * touch the displayed values
 *----------------------------------------------------------------------*/
void
scr_touch (int x, int y, int width, int height)
{
   int r1, r2, c1, c2;

   check_text ("scr_touch");

   c1 = Pixel2Col (x);
   r1 = Pixel2Row (y);
   c2 = c1 + 1 + width  / RxvtWin.fwidth;
   r2 = r1 + 1 + height / RxvtWin.fheight;

   if (r1 < 0) r1 = 0; else if (r1 >= RxvtWin.rows) r1 = (RxvtWin.rows-1);
   if (r2 < 0) r2 = 0; else if (r2 >= RxvtWin.rows) r2 = (RxvtWin.rows-1);
   if (c1 < 0) c1 = 0; else if (c1 >= RxvtWin.cols) c1 = (RxvtWin.cols-1);
   if (c2 < 0) c2 = 0; else if (c2 >= RxvtWin.cols) c2 = (RxvtWin.cols-1);

   c2 = (c2 - c1 + 1);
   for (r1 = r1; r1 <= r2; r1++)
     {
	int x = r1 * (RxvtWin.cols+1) + c1;
	memset (&displayed_text [x], 0, c2 * sizeof(text_t));
	memset (&displayed_rend [x], INITIAL_RSTYLE, c2 * sizeof(rend_t));
     }
}

/*----------------------------------------------------------------------*
 * refresh the region defined by rows STARTR and ENDR, inclusively.
 *
 * Actually draws to the X window
 * For X related speed-ups, this is a good place to fiddle.
 * The arrays displayed_text and displayed_rend contain what I
 * believe is currently shown on the screen. The arrays in screen contain
 * what should be displayed. This routine can decide how to refresh the
 * screen. Calls in command.c decide when to refresh.
 *----------------------------------------------------------------------*/
void
scr_refresh_region (int startr, int endr, int type)
{
   static int d_xcursor = 0;
   int r, xrow, xrow2, y1, outline, xcursor;
   unsigned long newgcm = 0;
   XGCValues newgcv;
   GC thisGC;

   /* fix up startr/endr order */
   if (startr > endr)
     {
	r = startr;
	startr = endr;
	endr = r;
     }

   if (type == NO_REFRESH	/* Window not visible, don't update */
       || startr < 0 || endr >= RxvtWin.rows)
     return;

#ifdef DEBUG_SCREEN
   check_text ("refresh region");
   fprintf (stderr, "refresh region\n");
#endif

   if (d_xcursor < (RxvtWin.rows * (RxvtWin.cols+1)))
     displayed_rend [d_xcursor] = 0xFF;

   xcursor = ((screen.row + RxvtWin.scrollback)*(RxvtWin.cols+1)
	      + screen.col);
   screen.rend [xcursor] |= RS_CURSOR;
   d_xcursor = (screen.row + RxvtWin.offset);
   if (d_xcursor >= RxvtWin.rows)
     {
	d_xcursor = 0;
	screen.rend [xcursor] &= ~RS_CURSOR;
     }
   else
     {
	d_xcursor *= (RxvtWin.cols+1);
     }
   d_xcursor += screen.col;

#ifdef USE_XCOPYAREA
   if (type == FAST_REFRESH && !graphics_up)
     {
	/* scroll using bitblt wherever possible
	 * a dirty approximation will ignore the rendition field here
	 * and fix it up later */
	for (r = startr; r <= endr; r++)
	  displayed_text [r * (RxvtWin.cols+1) + RxvtWin.cols] = 0;
	for (r = startr; r <= endr; r++)
	  {
	     int k, xrow, xrow2;

	     xrow = ((r + RxvtWin.scrollback - RxvtWin.offset) *
		     (RxvtWin.cols+1));
	     xrow2 = r * (RxvtWin.cols+1);
	     if (!strcmp (&displayed_text[xrow2], &screen.text[xrow]))
	       continue;
	     /* look for a similar line */
	     for (k = startr; k <= endr; k++)
	       {
		  xrow2 = k * (RxvtWin.cols+1);
		  if (!strcmp (&displayed_text[xrow2], &screen.text[xrow]))
		    break;
	       }
	     /* found it */
	     if (k <= endr)
	       {
		  int count, j;

		  j = r;
		  xrow  += (RxvtWin.cols+1);
		  xrow2 += (RxvtWin.cols+1);
		  r++;
		  for (count = 1;
		       ((r <= endr) &&
			(!strcmp (&displayed_text[xrow2],
				  &(screen.text[xrow]))));
		       count++)
		    {
		       xrow  += (RxvtWin.cols+1);
		       xrow2 += (RxvtWin.cols+1);
		       r++;
		    }
		  r--;
		  XCopyArea (Xdisplay, RxvtWin.vt, RxvtWin.vt, Xgc,
			     MARGIN, k * RxvtWin.fheight + MARGIN,
			     RxvtWin.pwidth,
			     count * RxvtWin.fheight,
			     MARGIN, j * RxvtWin.fheight + MARGIN);

		  /* Forward overlapping memcpy's are probably OK,
		   * but backwards doesn't work on SunOS 4.1.3 */
		  k *= (RxvtWin.cols+1);
		  j *= (RxvtWin.cols+1);
		  if (k > j)
		    {
		       while (count-- > 0)
			 {
			    MEMCOPY (&displayed_text [j],
				     &displayed_text [k],
				     (RxvtWin.cols+1) * sizeof(text_t));
			    MEMCOPY (&displayed_rend [j],
				     &displayed_rend [k],
				     (RxvtWin.cols+1) * sizeof(rend_t));
			    k += (RxvtWin.cols+1);
			    j += (RxvtWin.cols+1);
			 }
		    }
		  else
		    {
		       k += (count-1) * (RxvtWin.cols+1);
		       j += (count-1) * (RxvtWin.cols+1);
		       while (count-- > 0)
			 {
			    MEMCOPY (&displayed_text [j],
				     &displayed_text [k],
				     (RxvtWin.cols+1) * sizeof(text_t));
			    MEMCOPY (&displayed_rend [j],
				     &displayed_rend [k],
				     (RxvtWin.cols+1) * sizeof(rend_t));
			    k -= (RxvtWin.cols+1);
			    j -= (RxvtWin.cols+1);
			 }
		    }
	       }
	  }
     }
#endif

   xrow  = (startr + RxvtWin.scrollback - RxvtWin.offset) * (RxvtWin.cols+1);
   xrow2 = startr * (RxvtWin.cols+1);
   y1    = MARGIN + Xfont->ascent + startr * (RxvtWin.fheight);

   /* For a first cut, do it one character at a time */
   for (r = startr; r <= endr; r++)
     {
	int c;
	for (c = 0; c < RxvtWin.cols; c++)
	  {
	     int x, x1, count;

	     x = xrow + c;
	     x1 = xrow2 + c;
	     if ((displayed_text [x1] != screen.text [x]) ||
		 (displayed_rend [x1] != screen.rend [x]))
	       {
		  int fore, back, rend;

		  displayed_text [x1] = screen.text [x];
		  displayed_rend [x1] = screen.rend [x];
		  rend = screen.rend [x];
		  linebuf [0] = screen.text [x];
		  x1 = c * RxvtWin.fwidth + MARGIN;

		  x++; c++;
		  for (count = 1;
		       (rend == screen.rend [x] && c < RxvtWin.cols &&
			(displayed_text [xrow2+c] != screen.text [x] ||
			 displayed_rend [xrow2+c] != screen.rend [x] ||
			 (c+1 < RxvtWin.cols &&
			  displayed_text [xrow2+c+1] != screen.text [x+1])));
		       count++)
		    {
		       displayed_text [xrow2+c]= screen.text [x];
		       displayed_rend [xrow2+c]= screen.rend [x];
		       linebuf [count] = screen.text [x];
		       x++; c++;
		    }
		  c--;

		  outline = False;
		  thisGC = Xgc;

		  fore = GET_FG_COLOR (rend);
		  back = GET_BG_COLOR (rend);
		  rend = GET_ATTR (rend);
		  if (rend != 0)
		    {
		       if ((rend & RS_RVID) && (rend & RS_SELECTED))
			 rend &= ~(RS_RVID | RS_SELECTED);

		       if (!focus && (rend & RS_CURSOR))
			 {
			    outline = True;
			    rend &= ~RS_CURSOR;
			 }
		       if ((rend & RS_CURSOR) && (rend&(RS_RVID|RS_SELECTED)))
			 rend &= ~(RS_CURSOR | RS_RVID | RS_SELECTED);

		       if (rend & (RS_CURSOR | RS_RVID | RS_SELECTED))
			 thisGC = Xrvgc;

		       switch (rend & RS_FONTMASK) {
			case RS_GRFONT:
			  for (x = 0; x < count; x++)
			    if (linebuf [x] >= 0x5F && linebuf [x] < 0x7F)
			    linebuf [x] = ((linebuf [x] == 0x5F) ?
					   0x7F : linebuf [x] - 0x5F);
			  break;
			case RS_GBFONT:
			  for (x = 0; x < count; x++)
			    if (linebuf [x] == '#') linebuf [x] = '\036';
			  break;
		       }
		    }
#ifdef COLOR
		  if (rend & (RS_CURSOR | RS_RVID | RS_SELECTED))
		    {
		       if (fore != 0)
			 {
			    newgcv.background = pixel_colors [fore];
			    newgcm |= GCBackground;
			 }
		       if (back != 0)
			 {
			    newgcv.foreground = pixel_colors [back];
			    newgcm |= GCForeground;
			 }
		    }
		  else
		    {
		       if (fore != 0)
			 {
			    newgcv.foreground = pixel_colors [fore];
			    newgcm = GCForeground;
			 }
		       if (back != 0)
			 {
			    newgcv.background = pixel_colors [back];
			    newgcm |= GCBackground;
			 }
		    }

		  if (newgcm != 0)
		    {
		       XChangeGC (Xdisplay, thisGC, newgcm, &newgcv);
		    }
#endif
		  XDrawImageString (Xdisplay, RxvtWin.vt, thisGC,
				    x1, y1,
				    linebuf, count);

		  if (rend != 0)
		    {
		       if (
#ifdef USE_FAKE_BOLD
			   (rend & RS_BOLD) || (rend & RS_BLINK)

#else
			   ((rend & RS_BOLD)  && fore == FG_COLOR) ||
			   ((rend & RS_BLINK) && back == BG_COLOR)
#endif
			   )
			 XDrawString (Xdisplay, RxvtWin.vt, thisGC,
				      x1+1, y1,
				      linebuf, count);

		       /* On the smallest font,
			* underline overwrites the next row */
		       if ((rend & RS_ULINE) && (Xfont->descent > 1))
			 XDrawLine (Xdisplay, RxvtWin.vt, thisGC, x1, y1+1,
				    x1 + count * RxvtWin.fwidth - 1, y1+1);
		    }
#ifdef COLOR
		  if (newgcm != 0)
		    {
		       if (thisGC == Xrvgc)
			 {
			    newgcv.foreground = pixel_colors [BG_COLOR];
			    newgcv.background = pixel_colors [FG_COLOR];
			 }
		       else
			 {
			    newgcv.foreground = pixel_colors [FG_COLOR];
			    newgcv.background = pixel_colors [BG_COLOR];
			 }
		       XChangeGC (Xdisplay, thisGC, newgcm, &newgcv);
		       newgcm = 0;
		    }
#endif
		  if (outline)
		    XDrawRectangle (Xdisplay, RxvtWin.vt, thisGC, x1,
				    y1-Xfont->ascent,
				    RxvtWin.fwidth-1,
				    RxvtWin.fheight-1);
	       }
	  }
	xrow  += (RxvtWin.cols+1);
	xrow2 += (RxvtWin.cols+1);
	y1    += (RxvtWin.fheight);
     }
   screen.rend [xcursor] &= ~RS_CURSOR;
}

void
scr_refresh (int type)
{
   scr_refresh_region (0, (RxvtWin.rows-1), type);
}

/*----------------------------------------------------------------------*
 * value ==  1, sets a new tab stop at the current location
 * value ==  0, clears any tab stops at the current location
 * value == -1, clears all tabs
 *----------------------------------------------------------------------*/
void
scr_set_tab (int value)
{
   check_text("set_tab");

   if (value < 0)
     {
	int i;
	for (i = 0; i < RxvtWin.cols; i++)
	  tabs [i] = 0;
     }
   else
     {
	if (screen.col < RxvtWin.cols)
	  tabs [screen.col] = value;
     }
}

/*
 * change graphics context
 */
static void
swap_background (int bg)
{
   unsigned long color;
   GC t;

   if (bg == FG_COLOR || bg == BG_COLOR)
     {
	color = pixel_colors [bg];
     }
   else
     {
	color = pixel_colors [FG_COLOR];
	pixel_colors [FG_COLOR] = pixel_colors [BG_COLOR];
	pixel_colors [BG_COLOR] = color;
     }

   t = Xrvgc; Xrvgc = Xgc; Xgc = t;
   XSetWindowBackground (Xdisplay, RxvtWin.vt, color);

   XClearWindow (Xdisplay, RxvtWin.vt);
   scr_touch (0, 0, RxvtWin.pwidth, RxvtWin.pheight);
}

/*
 * secure the keyboard
 */
#ifdef SECURE_KBD
void
scr_secure (void)
{
   static int secured = 0;

   check_text("secure");
   if (secured)
     {
	secured = 0;
	XUngrabKeyboard (Xdisplay, CurrentTime);
     }
   else if (XGrabKeyboard (Xdisplay, RxvtWin.parent, True,
			   GrabModeAsync, GrabModeAsync,
			   CurrentTime) == GrabSuccess)
     secured = 1;
   else
     return;
   swap_background (-1);
}
#endif	/* SECURE_KBD */

/*
 * toggle reverse video settings
 */
void
scr_rvideo_mode (int use_rvid)
{
   static int rvid = 0;
   check_text("rev_vid");
   if (rvid == (use_rvid = (use_rvid != 0))) return;
   rvid = use_rvid;
   swap_background (rvid ? FG_COLOR : BG_COLOR);
}

/*
 * Set the text foreground color
 */
void
scr_fore_color (int color)
{
#ifdef COLOR
   if (Xdepth >= 3)
     {
	if (color == (FG_COLOR - ANSI_COLOR0)) color = FG_COLOR;
#ifndef USE_FAKE_BOLD
	else if (rstyle & RS_BOLD) color += BOLD_COLOR0;
	else
#endif
	  color += ANSI_COLOR0;
     }
   else
#endif
     color = (color == 7) ? BG_COLOR : FG_COLOR; /* normal or reverse */

   fore_color = color;
   rstyle = SET_FG_COLOR (rstyle, fore_color);
}

/*
 * Set the text foreground color
 */
void
scr_back_color (int color)
{
#ifdef COLOR
   if (Xdepth >= 3)
     {
	if (color == (BG_COLOR - ANSI_COLOR0)) color = BG_COLOR;
#ifndef USE_FAKE_BOLD
	else if (rstyle & RS_BLINK) color += BOLD_COLOR0;
	else
#endif
	  color += ANSI_COLOR0;
     }
   else
#endif
     color = (color == 0) ? FG_COLOR : BG_COLOR; /* normal or reverse */

   back_color = color;
   rstyle = SET_BG_COLOR (rstyle, back_color);
}

#ifdef PRINT_PIPE
void
scr_printscreen (int fullhist)
{
   text_t *text;
   int i, lim, ll;
   FILE *fd;

   fd = popen (rs_print_pipe,"w");
   if (fd == NULL)
     {
	print_error ("can't open printer pipe");
	return;
     }

   lim = RxvtWin.rows;
   if (fullhist)
     lim += RxvtWin.offset;

   text = &screen.text [((RxvtWin.scrollback - RxvtWin.offset) *
			 (RxvtWin.cols+1))];
   for (i = 0; i < lim; i++)
     {
	for (ll = (RxvtWin.cols-1); text[ll] == ' '; ll--)
	  continue;
	ll++;
	fprintf (fd, "%.*s\n", ll, text);
	text += (RxvtWin.cols+1);
     }

   fflush (fd);
   /* Don't ask why, but pclose() didn't work on SunOS 4.1.3! */
#if 0
   pclose (fd);
#else
   fclose (fd);
#endif
}
#endif

/* hidden debugging dump */
int
scr_dump_state (void)
{
   printf ("\nWindow %dx%d [chars] %dx%d [pixels] (font = %dx%d), scroll = %d/%d\n",
	   RxvtWin.cols, RxvtWin.rows, RxvtWin.pwidth, RxvtWin.pheight,
	   RxvtWin.fwidth, RxvtWin.fheight,
	   screen.tscroll, screen.bscroll);

   printf ("%d lines scrollback, %d lines scrolled, offset = %d\n",
	   RxvtWin.scrollback, RxvtWin.scrolled, RxvtWin.offset);

   printf ("selection = screen %d, op = ", selection.screen);

   switch (selection.op) {
    case SELECTION_CLEAR:	printf ("CLEAR");	break;
    case SELECTION_BEGIN:	printf ("BEGIN");	break;
    case SELECTION_INIT:	printf ("INIT");	break;
    case SELECTION_ONGOING:	printf ("ONGOING");	break;
    case SELECTION_COMPLETE:	printf ("COMPLETE");	break;
    default:			printf ("Unknown");	break;
   }
   printf ("\n\trow/col\n");
   printf ("beg\t%d %d\n", selection.beg.row, selection.beg.col);
   printf ("end\t%d %d\n", selection.end.row, selection.end.col);
   printf ("anchor\t%d %d\n", selection.anchor.row, selection.anchor.col);
   printf ("cursor\t%d %d\n", screen.row, screen.col);

   printf ("selection text = %d chars\n", selection.len);
   if (selection.text && selection.len) {
      int i;
      printf ("<text>\n\"");
      for (i = 0; i < selection.len; i++) {
	 switch (selection.text[i]) {
	  case '\n': printf ("\\n"); break;
	  case '\r': printf ("\\r"); break;
	 }
	 printf ("%c", selection.text[i]);
      }
      printf ("\"\n</text>");
   }

   return 1;
}

#define KeyState(x) (((x)&(ShiftMask|ControlMask))+(((x) & Mod1Mask)? 2 : 0))
/* adds together the bits:
 * shift key -> 1
 * meta key  -> 2
 * control key -> 4
 */

#ifndef NO_MOUSE_REPORT
void
mouse_report (XButtonEvent *event)
{
   int but, col, row;

   but = (040 + ((event->button ? (event->button - Button1) : 3)
		 + (KeyState(event->state) << 2)));
   col = 041 + Pixel2Col (event->x);
   row = 041 + Pixel2Row (event->y);

   tty_printf ("\033[M%c%c%c", but, col, row);
}

#if 0	/* not yet! */
void
mouse_tracking (int report, int x, int y, int firstrow, int lastrow)
{
   static int startx, starty, top, bot;

   if (report)
     {
	/* If either coordinate is past the end of the line:
	 * "ESC [ T CxCyCxCyCxCy"
	 * The parameters are startx, starty, endx, endy,
	 * mousex, and mousey */

	if ((selection.beg.row < selection.end.row) ||
	    ((selection.beg.row == selection.end.row) &&
	     (selection.beg.col < selection.end.col)))
	  {
	     if (selection.beg.row >= top && selection.end.row <= bot)
	       tty_printf ("\033[t"); /* start/end are valid locations */
	     else
	       tty_printf ("\033[T%c%c%c%c",
			   selection.beg.col + 1, selection.beg.row + 1,
			   selection.end.col + 1, selection.end.row + 1);
	  }
	else
	  {
	     if (selection.end.row >= top && selection.beg.row <= bot)
	       tty_printf ("\033[t");	/* start/end are valid locations */
	     else
	       tty_printf ("\033[T%c%c%c%c",
			   selection.end.col + 1, selection.end.row + 1,
			   selection.beg.col + 1, selection.beg.row + 1)
	  }
	tty_printf ("%c%c", Pixel2Col (x) + 1, Pixel2Row (y) + 1);
     }
   else
     {
	/* convert column/row to pixels */
	x = (x - 1) * RxvtWin.fwidth  + MARGIN;
	y = (y - 1) * RxvtWin.fheight + MARGIN;

	scr_selection_start (x ,y);
	top = firstrow;
	bot = lastrow;
     }
}
#endif
#endif
/*----------------------- end-of-file (C source) -----------------------*/
