/* List PRs that were closed between certain dates.
   Copyright (C) 1996, 1997 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to the Free
Software Foundation, 59 Temple Place - Suite 330, Boston, MA 02111, USA.  */

/* Initial version.  To use this to find the PRs fixed in a particular
   release, run it as
      getclose migration freeze lastfreeze
   where the options are
      migration   -- date of the migration for the current release
      freeze      -- date of the freeze for the current release
      lastfreeze  -- date of the freeze for the *previous* release
   The results will be in the form
      FIXED:foo:1234:the synopsis
      MAYBE:bar:1236:the synopsis
   Those marked as FIXED were closed before the migration into the new release.
   Those marked as MAYBE were closed after migration, but before freeze; thus,
   the fix may or may not have made it into the current release.  These need
   to be researched to find out if the changes, if any, were included. */

#include "config.h"
#include "gnats.h"
#include "query.h"

void usage (), version();

/* The name this program was run with.  */
char *program_name;

/* If 1, print the name of each PR that we look at.  */
int verbose;

/* When we migrated, froze the migrated code, and finally, when we
   froze the last release.  */
time_t migration, freeze, lastfreeze;

struct option long_options[] =
{
  {"verbose", 0, NULL, 'v'},
  {"version", 0, NULL, 'V'},
  {NULL, 0, NULL, 0}
};

/* If 1, we're running the daemon.  */
int is_daemon = 0;

time_t
got_closed (p)
     char *p;
{
  char *final1, *final2;
  char *c;

  /* Skip ahead to the last time it was closed.  */
  final1 = p;
  do
    {
      final2 = final1;
      final1 = strstr (final2, "-closed\n");
      if (final1)
	final1 += 8;
    } while (final1);
  if (! final2)
    return (time_t)0;

  /* Grab the date, and return the time it represents.  */
  final1 = strstr (final2, "When: ");
  if (! final1)
    return (time_t)0;
  c = strchr (final1 + 6, '\n');
  if (c)
    *c = '\0';

  return get_date (final1 + 6, NULL);
}

void
do_stat (path, p)
     char *path, *p;
{
  if (verbose)
    fprintf (stderr, "Reading %s\n", p);

  if (get_pr (path, p, 0))
    {
      time_t closed = got_closed (pr[AUDIT_TRAIL].value);
      
      /* We don't care about ones that had no closed status in its
	 audit trail.  */
      if (!closed)
	return;

      /* fixed in this release */
      if ((closed > lastfreeze) && (closed < migration))
	printf ("FIXED:%s:%s:%s\n", pr[CATEGORY].value,
		pr[NUMBER].value, pr[SYNOPSIS].value);
      /* maybe, maybe not */
      else if ((closed >= migration) && (closed <= freeze))
	printf ("MAYBE:%s:%s:%s\n", pr[CATEGORY].value,
		pr[NUMBER].value, pr[SYNOPSIS].value);
      /* fixed too late for release */
      else if (closed > freeze)
	  ;
      /* else fixed for a previous release - ignore */
    }
}

void
do_prlist ()
{
  Index *i;
  char *path = (char *) xmalloc (PATH_MAX);
  int len = strlen (gnats_root);

  for (i = index_chain; i ; i = i->next)
    if (strcmp (i->state, "closed") == 0
	&& regcmp ("no", i->confidential) == 0)
      {
	sprintf (path, "%s/%s/%s", gnats_root, i->category, i->number);
	do_stat (path, path + len + 1);
      }

  free (path);
}

int failed_dates = 0;
time_t
procdate (d)
     char *d;
{
  time_t t;

  t = get_date (d, NULL);
  if (!t || t == -1)
    {
      fprintf (stderr, "%s: invalid date `%s'\n",
	       program_name, d);
      failed_dates = 1;
    }

  return t;
}

int
main (argc, argv)
     int argc;
     char **argv;
{
  int optc;

  program_name = basename (argv[0]);
  verbose = 0;

  while ((optc = getopt_long (argc, argv, "hvV",
			      long_options, (int *) 0)) != EOF)
    {
      switch (optc)
	{
	case 'V':
	  version ();
	  exit (0);
	  break;

	case 'v':
	  verbose = 1;
	  break;

	case 'h':
	  usage (0);
	  break;

	default:
	  usage (1);
	}
    }

  if (optind == argc || optind > 3)
    usage (1);

  migration = procdate (argv[optind++]);
  freeze = procdate (argv[optind++]);
  lastfreeze = procdate (argv[optind++]);

  if (failed_dates)
    exit (1);

  if (verbose)
    {
      fprintf (stderr, "Migration date: %d\n", migration);
      fprintf (stderr, "Freeze date:    %d\n", freeze);
      fprintf (stderr, "Last freeze:    %d\n", lastfreeze);
    }

  configure ();
  init_gnats ();

  index_chain = get_index ();
  if (index_chain == NULL)
    {
      fprintf (stderr, "%s: couldn't read the index\n", program_name);
      exit (1);
    }

  /* We need this in order to read in the audit trail.  */
  query_format = FORMAT_FULL;

  do_prlist ();

  exit (0);
}

void
usage (x)
  int x;
{
  fprintf (stderr, "Usage: %s [-v|--verbose] [-V|--version] migration freeze lastfreeze\n", program_name);
  fprintf (stderr, "      -v | --verbose     Print the name of each PR as we check it.\n");
  fprintf (stderr, "      -V | --version     Print version and exit.\n");
  fprintf (stderr, "      To use this to find the PRs fixed in a particular release, run it as\n");
  fprintf (stderr, "            getclose migration freeze lastfreeze\n");
  fprintf (stderr, "         where the options are\n");
  fprintf (stderr, "            migration   -- date of the migration for the current release\n");
  fprintf (stderr, "            freeze      -- date of the freeze for the current release\n");
  fprintf (stderr, "            lastfreeze  -- date of the freeze for the *previous* release\n");
  fprintf (stderr, "         The results will be in the form\n");
  fprintf (stderr, "            FIXED:foo:1234:the synopsis\n");
  fprintf (stderr, "            MAYBE:bar:1236:the synopsis\n");
  fprintf (stderr, "         Those marked as FIXED were closed before the migration into the new release.\n");
  fprintf (stderr, "         Those marked as MAYBE were closed after migration, but before freeze; thus,\n");
  fprintf (stderr, "         the fix may or may not have made it into the current release.  These need\n");
  fprintf (stderr, "         to be researched to find out if the changes, if any, were included.\n");

  exit (x);
}

void
version ()
{
  printf ("getclose [-v|--verbose] %s\n", version_string);
}
