/***********************************************************************/
/* QUERY.C - Functions related to QUERY,STATUS and EXTRACT             */
/***********************************************************************/
/*
 * THE - The Hessling Editor. A text editor similar to VM/CMS xedit.
 * Copyright (C) 1991-1997 Mark Hessling
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to:
 *
 *    The Free Software Foundation, Inc.
 *    675 Mass Ave,
 *    Cambridge, MA 02139 USA.
 *
 *
 * If you make modifications to this software that you feel increases
 * it usefulness for the rest of the community, please email the
 * changes, enhancements, bug fixes as well as any and all ideas to me.
 * This software is going to be maintained and enhanced as deemed
 * necessary by the community.
 *
 * Mark Hessling                 Email:             M.Hessling@qut.edu.au
 * PO Box 203                    Phone:                    +617 3802 0800
 * Bellara                       http://www.gu.edu.au/gext/the/markh.html
 * QLD 4507                      **** Maintainer PDCurses & REXX/SQL ****
 * Australia                     ************* Author of THE ************
 */

/*
$Id: query.c 2.1 1995/06/24 16:30:52 MH Rel MH $
*/

#include <the.h>
#include <proto.h>

#include <query.h>

#ifdef HAVE_PROTO
static short extract_pending(short,CHARTYPE *);
static short extract_point(short,CHARTYPE *);
static short extract_prefix(short,CHARTYPE *);
static short extract_reserved(short,CHARTYPE *,CHARTYPE);
static short extract_ring(short,CHARTYPE *,CHARTYPE);
static short extract_colour(short,CHARTYPE *,CHARTYPE,CHARTYPE *);
static void get_etmode(CHARTYPE *,CHARTYPE *);
#else
static short extract_pending();
static short extract_point();
static short extract_prefix();
static short extract_reserved();
static short extract_ring();
static short extract_colour();
static void get_etmode();
#endif

/*man-start*********************************************************************


========================================================================
QUERY, EXTRACT and STATUS
========================================================================

     The following lists the valid variables that can be queried and
     also shows what values are returned. For both <QUERY> and <STATUS>,
     the values are concatenated together and displayed as a single
     line. For <EXTRACT> the REXX variables that are set are defined.
     The capitalised portion of the variable is the minimum abbreviation
     recognised.

     The bracketed text at the end of the description indicates from
     which commands a valid response will be supplied.
     (Q-Query, E-Extract, M-Modify and S-Status).

ALT
     The number of alterations to the current file since the last <SAVE>
     or automatic save via <SET AUTOSAVE>.
     (QES)

     alt.0           - 2
     alt.1           - Number of alterations since last SAVE or autosave
     alt.2           - Number of alterations since last SAVE

ARBchar
     The status of <SET ARBCHAR> and the ARBCHAR characters.
     (QEMS)

     arbchar.0       - 3
     arbchar.1       - ON|OFF
     arbchar.2       - multiple match character
     arbchar.3       - single match character

AUtosave
     The status of <SET AUTOSAVE> and/or the frequency setting.
     (QEMS)

     autosave.0      - 1
     autosave.1      - OFF|n

BACKup
     Indicates if a .bak file is kept after editing.
     (QEMS)

     backup.0        - 1
     backup.1        - ON|OFF|TEMP|KEEP

BEEP
     Indicates if the bell is sounded on display of error messages.
     Set by <SET BEEP>
     (QEMS)

     beep.0          - 1
     beep.1          - ON|OFF

BLOCK
     Returns information about the marked <block>, if any.
     (QE)

     block.0         - 6 if a marked block exists, or 1 for NONE
     block.1         - type of marked block (LINE|BOX|NONE|WORD|COLUMN)
     block.2         - line number of start of block
     block.3         - column number of start of block
     block.4         - line number of end of block
     block.5         - column number of end of block
     block.6         - file name containing marked block

CASE
     The settings related to the CASE of characters entered, searched
     for, changed and sorted. Set by <SET CASE>
     (QEMS)

     case.0          - 4
     case.1          - MIXED|UPPER|LOWER
     case.2          - RESPECT|IGNORE
     case.3          - RESPECT|IGNORE
     case.4          - RESPECT|IGNORE

CLEARScreen
     Indicates if the screen is cleared on exit from THE. Set by
     <SET CLEARSCREEN>.
     (QEMS)

     clearscreen.0   - 1
     clearscreen.1   - ON|OFF

CLOCK
     Indicates if the time is displayed on the status line. Set by
     <SET CLOCK>.
     (QEMS)

     clock.0         - 1
     clock.1         - ON|OFF

CMDArrows
     Returns the settings for arrow key behaviour on <command line>.
     Set by <SET CMDARROWS>.
     (QEMS)

     cmdarrows.0     - 1
     cmdarrows.1     - RETRIEVE|TAB

CMDline
     The settings for the <command line>. Set by <SET CMDLINE>.
     (QEMS)

     If CMDLINE OFF

     cmdline.0       - 1
     cmdline.1       - OFF

     If CMDLINE ON, BOTTOM or TOP

     cmdline.0       - 3
     cmdline.1       - ON|TOP|BOTTOM
     cmdline.2 *     - line number within window of command line
     cmdline.3 *     - contents of command line

     * these values are only returned with <EXTRACT>

COLOR [*|area]
     Displays the current color settings for the file.
     Set by <SET COLOR>.

     With the ['*'] option, (or no option), returns color settings
     for all areas of the screen.
     (QE)

     color.0         - 23
     color.1         - FILEAREA   followed by its color
     color.2         - CURLINE    followed by its color
     color.3         - BLOCK      followed by its color
     color.4         - CBLOCK     followed by its color
     color.5         - CMDLINE    followed by its color
     color.6         - IDLINE     followed by its color
     color.7         - MSGLINE    followed by its color
     color.8         - ARROW      followed by its color
     color.9         - PREFIX     followed by its color
     color.10        - PENDING    followed by its color
     color.11        - SCALE      followed by its color
     color.12        - TOFEOF     followed by its color
     color.13        - CTOFEOF    followed by its color
     color.14        - TABLINE    followed by its color
     color.15        - SHADOW     followed by its color
     color.16        - STATAREA   followed by its color
     color.17        - DIVIDER    followed by its color
     color.18        - RESERVED   followed by its color
     color.19        - NONDISP    followed by its color
     color.20        - HIGHLIGHT  followed by its color
     color.21        - CHIGHLIGHT followed by its color
     color.22        - SLK        followed by its color
     color.23        - GAP        followed by its color

     With the ['area'] option, returns color settings
     for the specified  area of the screen.
     (E)

     color.0         - 1
     color.1         - ['area'] followed by its color

COLUMN
     Displays the current value of the focus column.
     (QE)

     column.0        - 1
     column.1        - Column number of focus column

COMPAT
     The settings for the compatibility mode. Set by <SET COMPAT>.
     (QEMS)

     compat.0        - 3
     compat.1        - THE|XEDIT|KEDIT (compatibility LOOK)
     compat.2        - THE|XEDIT|KEDIT (compatibility FEEL)
     compat.3        - THE|XEDIT|KEDIT (compatibility KEYS)

CURline
     The value and position of the <current line> within the current file.
     Set by <SET CURLINE>. Also provides the contents of the <focus line>
     and an indication of the status of the <focus line> for the current
     session.
     (QEMS)

     With <EXTRACT> command:

     curline.0       - 6
     curline.1       - curline setting
     curline.2       - line number within window of current line
     curline.3       - contents of focus line
     curline.4       - ON|OFF (ON if line has changed or been added this session)
     curline.5       - OLD|OLD CHANGED|NEW CHANGED
     curline.6       - selection level of focus line (same as select.1)

     The setting of curline.5 is as follows:
       OLD:            The line existed in its current form in the
                       file before THE began.
       OLD CHANGED:    The line was in the file before THE started, 
                       but has been changed during the current editing
                       session.
       NEW CHANGED:    The line has been added to the file during
                       the current editing session.

     With <QUERY>, <MODIFY> and <STATUS> commands:

     curline.0       - 1
     curline.1       - curline setting

CURSor
     The position of the cursor within the displayed screen and file
     at the time the <EXTRACT> /CURSOR/ is issued and the position of
     the cursor at the time the <macro> was issued.
     If the cursor is not in the <filearea>, then line and column
     values will be set to -1.
     (QE)

     cursor.0        - 8
     cursor.1        - line number of cursor in screen (current)
     cursor.2        - column number of cursor in screen (current)
     cursor.3        - line number of cursor in file (current)
     cursor.4        - column number of cursor in file (current)
     cursor.5        - line number of cursor in screen (original)
     cursor.6        - column number of cursor in screen (original)
     cursor.7        - line number of cursor in file (original)
     cursor.8        - column number of cursor in file (original)

CURSORSTay
     The setting for the behaviour of the cursor after a file scroll.
     Set by <SET CURSORSTAY>.
     (QEMS)

     cursorstay.0    - 1
     cursorstay.1    - ON|OFF

DEFSORT
     Specifies the order in which files are sorted in the DIR.DIR file.
     Set by <SET DEFSORT>.
     (QEMS)

     defsort.0       - 2
     defsort.1       - DIRECTORY|NAME|SIZE|TIME|DATE|OFF
     defsort.2       - ASCENDING|DESCENDING

DIRFILEID
     The value of the path and filename of the focus line in a DIR.DIR
     file.
     (E)

     dirfileid.0     - 2
     dirfileid.1     - full path of directory
     dirfileid.2     - file name at focus line

DISPlay
     Indicates the range of selection levels current. Set by
     <SET DISPLAY>.
     (QEMS)

     display.0       - 2
     display.1       - display low value
     display.2       - display high value

EOF
     Indicates if the <current line> is on the <Bottom-of-File line>.
     (QES)

     eof.0           - 1
     eof.1           - ON|OFF

EOLout
     Returns the value of the end-of-line character(s).
     (QEMS)

     eolout.0        - 1
     eolout.1        - LF|CRLF|CR|NONE

ETMODE
     Indicates if extended display mode is set. Set by <SET ETMODE>.
     (QEMS)

     etmode.0        - 2
     etmode.1        - ON|OFF
     etmode.2        - character ranges if not all ON or OFF

FExt
     The extension of the current file. The characters following
     the trailing '.' character. Same as FType.
     (QEM)

     fext.0          - 1
     fext.1          - File extension.

FILEName
     The full filename of the current file, including any file
     extension.
     (QEM)

     filename.0      - 1
     filename.1      - Full file name.

FMode
     The file mode of the current file. Undex Un*x, theis will
     always return an empty string. Other platforms returns the
     first character of the file's path. ie the disk drive.
     (QEM)

     fmode.0         - 1
     fmode.1         - File mode.

FName
     The fname portion of the current file. See <SET FILENAME> for a
     full description of the components of a file name.
     (QEM)

     fname.0         - 1
     fname.1         - File name.

FPath
     The path name of the current file. This includes a trailing
     directory separator.
     (QEM)

     fpath.0         - 1
     fpath.1         - File path.

FType
     The extension of the current file. The characters following
     the trailing '.' character.
     (QEM)

     ftype.0         - 1
     ftype.1         - File extension.

FULLFName
     Indicates if the fully qualified filename is displayed on the
     <idline>.
     (QEMS)

     fullfname.0     - 1
     fullfname.1     - ON|OFF

GETENV variable
     The value of the supplied environment 'variable' or ***invalid***
     if the 'variable' does not exist. On platforms other than Unix
     the supplied variable name is uppercased before obtaining the
     environment variable value.
     (E)

     getenv.0        - 1
     getenv.1        - value of variable

HEX
     Indicates if hexadecimal values in commands are interpreted
     as hexadecimal values or not. Set by <SET HEX>.
     (QEMS)

     hex.0           - 1
     hex.1           - ON|OFF

HEXDISPlay
     Indicates if the current character is displayed on the <status line>.
     Set by <SET HEXDISPLAY>.
     (QEMS)

     hexdisplay.0    - 1
     hexdisplay.1    - ON|OFF

HEXShow
     Returns details of HEXSHOW placement. Set by <SET HEXSHOW>.
     (QEMS)

     hexshow.0       - 2
     hexshow.1       - ON|OFF
     hexshow.2       - line displaying first hexshow line

HIGHlight
     Returns details of HIGHLIGHT settings. Set by <SET HIGHLIGHT>.
     (QEMS)

     highlight.0     - 1 or 3 (if highlight.1 = SELECT)
     highlight.1     - OFF|ALTERED|TAGGED|SELECT
     highlight.2     - minimum (or only) selection level for SELECT
     highlight.3     - maximum selection level for SELECT

IDline
     Indicates if the <idline> is displayed for a file. Set by
     <SET IDLINE>.
     (QEMS)

     idline.0        - 1
     idline.1        - ON|OFF

IMPMACro
     Indicates if implied <macro> processing is on or off. Set by
     <SET IMPMACRO>.
     (QEMS)

     impmacro.0      - 1
     impmacro.1      - ON|OFF

IMPOS
     Indicates if implied operating system command processing is on 
     or off. Set by <SET IMPOS> or <SET IMPCMSCP>.
     (QEMS)

     impos.0         - 1
     impos.1         - ON|OFF

INPUTMode
     Indicates the inputmode for the current view.
     Set by <SET INPUTMODE>.
     (QEMS)

     inputmode.0     - 1
     inputmode.1     - OFF|FULL|LINE

INSERTmode
     Indicates if currently in insert mode or overstrike mode. Set by
     <SET INSERTMODE>.
     (QEMS)

     insertmode.1    - 1
     insertmode.1    - ON|OFF

LASTmsg
     Return the text of the last error message generated.
     (E)

     lastmsg.0       - 1
     lastmsg.1       - text of last message.

LASTRC
     Return code from last command issued from <command line>.
     (QES)

     lastrc.0        - 1
     lastrc.1        - Last return code.

LENgth
     Length of the current line.
     (QES)

     length.0        - 1
     length.1        - Length of current line.

LIne
     Line number of <focus line> in current file.
     (QES)

     line.0          - 1
     line.1          - Line number

LINEFLAG
     Returns information about the flags set on the <focus line>
     (QEMS)

     lineflag.0      - 3
     lineflag.1      - NEW|NONEW
     lineflag.2      - CHANGE|NOCHANGE
     lineflag.3      - TAG|NOTAG

LINENd
     Indicates if multiple commands allowed on command line
     and the delimiter. Set by <SET LINEND>.
     (QEMS)

     linend.0        - 2
     linend.1        - ON|OFF
     linend.2        - delimiter

LScreen
     Displays the size of the current screen and window. Also
     shows the upper left corner of the window.
     (QE)

     lscreen.0       - 6
     lscreen.1       - height of current window
     lscreen.2       - width of current window
     lscreen.3       - screen line of upper left corner of window
     lscreen.4       - screen column of upper left corner of window
     lscreen.5       - height of screen
     lscreen.6       - width of screen

MACRO
     Indicates if macros are executed before commands. Set by
     <SET MACRO>.
     (QEMS)

     macro.0         - 1
     macro.1         - ON|OFF

MACROExt
     The current setting for a macro's file extension. Set by
     <SET MACROEXT>.
     (QEMS)

     macroext.0      - 1
     macroext.1      - Default file extension

MACROPath
     The path that THE looks for by default for macro files. Set by
     <SET MACROPATH>.
     (QEMS)

     macropath.0     - 1
     macropath.1     - Path for macro files.

MARgins
     The settings for left and right margins and paragraph indent.
     Set by <SET MARGINS>.
     (QEMS)

     margins.0       - 3
     margins.1       - left column
     margins.2       - right column
     margins.3       - indent value (column or offset from left margin)

MONITOR
     Indicates if the combination of monitor and the curses package
     supports colour. If the curses package supports colour, then
     monitor.1 is set to COLOR and monitor.2 can be COLOR or MONO
     depending on whether the monitor supports colour. If monitor.1
     is MONO then monitor.2 will also be set to MONO.
     (QES)

     monitor.0       - 2
     monitor.1       - COLOR|MONO
     monitor.2       - COLOR|MONO

MSGLine
     Returns details of where the <message line> is displayed. Set by
     <SET MSGLINE>.
     (QEMS)

     msgline.0       - 4
     msgline.1       - ON
     msgline.2       - line position of <message line>
     msgline.3       - number of message lines available
     msgline.4       - OVERLAY (returned for compatibility reasons)

MSGMode
     Indicates if messages are suppressed. Set by <SET MSGMODE>.
     (QEMS)

     msgmode.0       - 1
     msgmode.1       - ON|OFF

NBFile
     Returns with the number of files currently in the <ring>.
     (QES)

     nbfile.0        - 1
     nbfile.1        - Number of files in ring

NEWLines
     Indicates if NEWLINES variable is set to LEFT or ALIGNED.
     Set by <SET NEWLINES>.
     (QEMS)

     newlines.0      - 1
     newlines.1      - ALIGNED|LEFT

NONDisp
     Returns the character that is displayed for extended characters
     that are not displayed. Set by <SET NONDISP>.
     (QEMS)

     nondisp.0       - 1
     nondisp.1       - char

MOUSE
     Indicates if mouse support has been enabled.
     Set by <SET MOUSE>.
     (QEMS)

     mouse.0         - 1
     mouse.1         - ON|OFF

NUMber
     Indicates if line numbers are displayed in the prefix area.
     Set by <SET NUMBER>.
     (QEMS)

     number.0        - 1
     number.1        - ON|OFF

PENDing [BLOCK] [OLDNAME] name|*
     Returns information about pending prefix commands.
     (E)

     pending.0       - 4
     pending.1       - line number in file
     pending.2       - newname - actual name entered in prefix area
     pending.3       - oldname - original name of macro after synonym resolution
     pending.4       - BLOCK or null

Point [*]
     Returns the name and line number of the focus line, or names and
     line numbers of all lines in a file if '*' is specified.
     (QE)

     With no arguments:

     point.0         - 0 or 1       (0 if focus line not named)
     point.1         - line number and name of line (if line is named)

     With ['*'] argument:
     (E)

     point.0         - number of named lines in the file
     point.1         - line number and name for first named line
     point.n         - line number and name for nth named line

POSition
     Indicates if LINE/COL is displayed on <idline>. Set by
     <SET POSITION>.
     (QMS)

     position.0      - 1
     position.1      - ON|OFF

     The current/focus line/column is also returned via <EXTRACT>.
     (E)

     position.0      - 3
     position.1      - ON|OFF
     position.2      - current or focus line
     position.3      - current or focus column

PREfix [Synonym [*]]
     Indicates if prefix is displayed for the view and if displayed
     where is is displayed. See <SET PREFIX>.
     (QEMS)

     prefix.0        - 1 or 2     (1 if prefix.1 is OFF, 2 otherwise)
     prefix.1        - ON|OFF|NULLS
     prefix.2        - LEFT|RIGHT        (if prefix.1 is ON or NULLS)
     prefix.3        - width of prefix area
     prefix.4        - width of prefix gap

     With ['Synonym'] option, the name of the macrofile (oldname)
     is returned that is associated with the synonym. If name 
     is not a synonym then name is returned.
     (E)

     prefix.0        - 1
     prefix.1        - oldname

     With ['Synonym' ['*']] option, all prefix synonyms are returned.
     (E)

     prefix.0        - number of prefix synonyms
     prefix.1        - newname oldname
     prefix.n        - newname oldname

PRINTER
     Returns the value of the printer port or spooler. Set by
     <SET PRINTER>
     (QEMS)

     printer.0       - 1
     printer.1       - port or spooler name

REPROFile
     Indicates if the <profile> is to be re-executed for each
     subsequent file to be edited. Set by <SET REPROFILE>.
     (QEMS)

     reprofile.0     - 1
     reprofile.1     - ON|OFF

RESERved [*]
     Return with list of the screen rows that have been reserved.
     Set by <SET RESERVED>.
     (QES)

     reserved.0      - 0 if no reserved lines; 1 otherwise
     reserved.1      - list of screen row numbers reserved

     With ['*'] option, the line number, followed by the colour
     specification and <reserved line> contents are returned; one
     variable for each reserved line.
     (E)

     reserved.0      - the number of reserved lines
     reserved.1      - first reserved line
     reserved.2      - second reserved line
     reserved.n      - nth reserved line

REXXOUTput
     Indicates if <REXX> output is captured to a file or not and the
     line number limit of lines to be displayed. Set by <SET REXXOUTPUT>.
     (QEMS)

     rexxoutput.0    - 2
     rexxoutput.1    - FILE|DISPLAY
     rexxoutput.2    - line number limit

RING
     Returns details of each file being edited.
     (QE)

     With COMPAT set to XEDIT(feel), the values set are:

     ring.0          - number of variables returned (ring.1 + 1)
     ring.1          - number of files in the <ring>
     ring.2          - IDLINE of first file in the ring
     ring.3          - IDLINE of second file in the ring
     ring.n          - IDLINE of nth file in the ring

     With COMPAT set to THE or KEDIT(feel), the values set are:

     ring.0          - number of files in the <ring>
     ring.1          - IDLINE of first file in the ring
     ring.2          - IDLINE of second file in the ring
     ring.n          - IDLINE of nth file in the ring

SCALe
     Returns details of <scale line>. Set by <SET SCALE>.
     (QEMS)

     scale.0         - 2
     scale.1         - ON|OFF
     scale.2         - line displaying scale line

SCOPE
     Returns information about whether <shadow lines> will be affected by
     commands or not. Set by <SET SCOPE>.
     (QEMS)

     scope.0         - 1
     scope.1         - ALL|DISPLAY

SCReen
     Returns the number and orientation of THE screens. Set by 
     <SET SCREEN>.
     (QEMS)

     screen.0        - 2
     screen.1        - Number of screens displayed
     screen.2        - HORIZONTAL|VERTICAL

SELect
     Returns the selection level of the <focus line> and the
     maximum selection level for the file. Set by <SET SELECT> and
     <ALL> commands.
     (QES)

     select.0        - 2
     select.1        - selection level of focus line
     select.2        - maximum selection level for file

SHADow
     Returns the status of the display of shadow lines. Set by
     <SET SHADOW>.
     (QEMS)

     shadow.0        - 1
     shadow.1        - ON|OFF

SHOWkey key
     Returns the commands and parameters assigned to the 'key'
     passed as an argument.  This keyvalue is returned as
     READV.3 from a call to <READV> KEY.
     (E)

     showkey.0       - the number of commands assigned
     showkey.1       - first command/parameter assigned
     showkey.n       - last command/parameter assigned

     If no 'key' supplied as a parameter:

     showkey.0       - 1
     showkey.1       - INVALID KEY

SIze
     Returns the number of lines in the current file.
     (QES)

     size.0          - 1
     size.1          - Lines in current file.

STATUSLine
     Indicates if the <status line> is displayed and if so, where.
     Set by <SET STATUSLINE>.
     (QEMS)

     statusline.0    - 1
     statusline.1    - TOP|BOTTOM|OFF

STAY
     Indicates if the focus line stays where it is after a successful
     THE command or an unsuccessful <LOCATE> command.  Set by <SET STAY>.
     (QEMS)

     stay.0          - 1
     stay.1          - ON|OFF

TABKey
     Returns settings about behaviour of TAB key.
     tabkey.1 indicates behaviour while not in insert mode
     tabkey.2 indicates behaviour while in insert mode
     Set by <SET TABKEY>.
     (QEMS)

     tabkey.0        - 2
     tabkey.1        - TAB|CHARACTER
     tabkey.2        - TAB|CHARACTER

TABLine
     Returns details of if and where the <tab line> is displayed.
     Set by <SET TABLINE>.
     (QEMS)

     tabline.0       - 2
     tabline.1       - ON|OFF
     tabline.2       - line displaying tab line

TABS
     Returns settings about tab columns. Set by <SET TABS>.
     (QEMS)

     tabs.0          - 1
     tabs.1          - actual tab columns or "INCR n"

TABSIn
     Indicates if TABSIN processing is on or off and the size of the
     tabs setting. Set by <SET TABSIN>.
     (QEMS)

     tabsin.0        - 2
     tabsin.1        - ON|OFF
     tabsin.2        - size of tabs

TABSOut
     Indicates if TABSOUT processing is on or off and the size of the
     tabs setting. Set by <SET TABSOUT>.
     (QEMS)

     tabsout.0       - 2
     tabsout.1       - ON|OFF
     tabsout.2       - size of tabs

TERMinal
     Identifies the terminal type currently being used.
     (QES)

     terminal.0      - 1
     terminal.1      - DOS|OS2|X11|WIN32|$TERM value under Unix

TOF
     Indicates if the <current line> is on the <Top-of-File line>.
     (QES)

     tof.0           - 1
     tof.1           - ON|OFF

TYPEAhead
     Indicates if THE will wait until all keyboard input has been
     processed before updating the screen display. Set by
     <SET TYPEAHEAD>.
     (QEMS)

     typeahead.0     - 1
     typeahead.1     - ON|OFF

Verify
     Returns verify column settings. Set by <SET VERIFY>.
     (QEMS)

     verify.0        - 1
     verify.1        - Column pair of verify start and end columns.

VERShift
     Returns the value of the <vershift> internal variable.
     (QES)

     vershift.0      - 1
     vershift.1      - VERSHIFT value

VERSION
     Returns information about name of application (THE) and version
     information.
     (QES)

     version.0       - 4
     version.1       - THE
     version.2       - version string eg. 1.5
     version.3       - platform version (DOS,OS2,UNIX,X11,WIN32)
     version.4       - version status information eg. release date, beta

Width
     Returns maximum line width setting. Set by -w command line switch
     on starting THE.
     (QEMS)

     width.0         - 1
     width.1         - Maximum line width value.

WORD
     Specifies how THE defines a word.
     (QEMS)

     word.0          - 1
     word.1          - ALPHANUM|NONBLANK

WORDWrap
     Indicates if WORDWRAP is on or off. Set by <SET WORDWRAP>.
     (QEMS)

     wordwrap.0      - 1
     wordwrap.1      - ON|OFF

WRap
     Indicates if WRAP is on or off. Set by <SET WRAP>.
     (QEMS)

     wrap.0          - 1
     wrap.1          - ON|OFF

XTERMinal
     Returns the current value of the X11 terminal program.
     Only applicable in X version. Set by <SET XTERMINAL>.
     (QEMS)

     xterminal.0     - 1
     xterminal.1     - X11 terminal program

Zone
     Returns zone column settings. Set by <SET ZONE>.
     (QEMS)

     zone.0          - 2
     zone.1          - Zone start column
     zone.2          - Zone end column



========================================================================
IMPLIED EXTRACT
========================================================================

     The above <REXX> variables set by the <EXTRACT> command may also
     be obtained by a REXX macro as an implied EXTRACT. Each variable
     above that may be set by an explicit EXTRACT command may also be
     used as an external function to REXX to obtain the same information.
     eg. The REXX commands:

         'EXTRACT /SIZE/CURLINE/'
         Say size.1 curline.1

     may be substituted with:

         Say size.1() curline.1()


========================================================================
BOOLEAN FUNCTIONS
========================================================================

     THE also provides other information to the REXX interpreter via
     boolean functions. These functions return either '1' or '0'
     depending on the information queried.

after()
     Returns '1' if the cursor is currently after the last non-blank
     character on the line, or if the line is blank.

blank()
     Returns '1' if the line the <cursor field> is completely blank.

batch()
     Returns '1' if THE is being run in batch mode. ie THE was
     started with the -b switch.

block()
     Returns '1' if the marked <block> is within the current view.

before()
     Returns '1' if the cursor is currently before the first non-blank
     character on the line, or if the line is blank.

bottomedge()
     Returns '1' if the cursor is on the bottom edge of the <filearea>
     or <prefix area>.

command()
     Returns '1' if the <command line> is on. This is different to
     the definition in KEDIT; "Returns '1' if the cursor is on the
     command line." To get the equivalent KEDIT functionality
     use incommand();

current()
     Returns '1' if the cursor is on the <current line>.

dir()
     Returns '1' if the current file is the special DIR.DIR file.

end()
     Returns '1' if the cursor is on the last non-blank character on
     the line.

eof()
     Returns '1' if the cursor is on the <Bottom-of-File> line
     and the cursor is not on the <command line>.

first()
     Returns '1' if the cursor is in column 1 of the current window.

focuseof()
     Returns '1' if the focus line is the <Bottom-of-File> line
     whether the cursor is on it or not.

focustof()
     Returns '1' if the <focus line> is the <Top-of-File> line
     whether the cursor is on it or not.

inblock()
     Returns '1' if the cursor is in the marked <block>.

incommand()
     Returns '1' if the cursor is on the <command line>.

initial()
     Returns '1' if the function is called from the <profile>.

inprefix()
     Returns '1' if the cursor is located in the <prefix area>.

leftedge()
     Returns '1' if the cursor is on the left edge of the <filearea>.

modifiable()
     Returns '1' if the cursor is located in an area that can be changed.
     ie. not on <Top-of-File line> or <Bottom-of-File line> nor on a
     <shadow line>.

rightedge()
     Returns '1' if the cursor is on the right edge of the <filearea>.

spacechar()
     Returns '1' if the cursor is on a space character.

tof()
     Returns '1' if the cursor is on the <Top-of-File line>
     and the cursor is not on the <command line>.

topedge()
     Returns '1' if the cursor is on the top edge of the <filearea>.

verone()
     Returns '1' if the column 1 of the file is being displayed in
     column 1.

========================================================================
OTHER FUNCTIONS
========================================================================

     The following functions provide features to simplify THE macros
     written in REXX.

valid_target(target[,anything])
     The first argument is the <target> to be validated. If a second,
     optional argument is supplied, the target to be validated can
     consist of a target followed by any optional characters. This can
     be useful if the arguments to a macro consist of a target followed
     by another argument. If a valid target is supplied, the remainder
     of the string passed to valid_target() is returned following the
     first line affected and the number of lines to the target.

     Returns 'ERROR' if the supplied target is invalid.
     Returns 'NOTFOUND' if the supplied target is valid, but not found.

     If a valid target, returns the first line affected by the target
     followed by the number of lines to the target, and optionally the
     remainder of the argument. eg. 

     if the focus line is 12 and valid_target() is called as

         result = valid_target(":7")      ===> result = "12 -5"

     if the focus line is 12 and valid_target is called as

         result = valid_target(":7 /fred/",junk), ===> result = "12 -5 /fred/"

run_os(command[,stdin_stem[,stdout_stem[,stderr_stem]]])
     This function allows the macro writer to call an operating system
     command and have the standard streams; 'stdin', 'stdout' and 'stderr'
     redirected to or from REXX arrays.

     The first argument is the operating system command to execute.
     The command can include any command line switches appropriate 
     to the command.

     All other arguments comprise a stem name (including a trailing '.')
     which refers to the REXX arrays where 'stdin', 'stdout' and 'stderr'
     are to be redirected.

     As with all REXX arrays, the value of the 0th element (stem.0)
     contains the number of elements in the array.

     The only restriction with the names of the stem variables is
     that the stem name for the 'stdin' stem cannot be the same as the
     stem for 'stdout' or 'stderr'.

     The stem name for 'stdout' and 'stderr' can be the same; the contents
     of the resulting output stems will consist of 'stdout' and 'stderr' in
     the order that the command generates this output.

     Return values:

          0    - successful
          1005 - invalid argument or syntax
          1012 - problems with system redirection of streams
          1094 - out of memory
          1099 - error interfacing to REXX interpreter

     all other numbers, return code from operating system command

     eg. to spell check the words "The Hessling Editr" with ispell

         in.0 = 3
         in.1 = "The"
         in.2 = "Hessling"
         in.3 = "Editr"
         rc = run_os("ispell -a","in.","out.")
         
     sets:

          out.0 --> 3
          out.1 --> "*"
          out.2 --> "#"
          out.3 --> "& edits edit editor"

**man-end**********************************************************************/

struct query_item
 {
  CHARTYPE query;                            /* valid query response ? */
  CHARTYPE *name;                                      /* name of item */
  short min_len;                     /* minimum length of abbreviation */
  short item_number;                         /* unique number for item */
  short number_values;                    /* number of values returned */
 };
typedef struct query_item QUERY_ITEM;

/***********************************************************************/
/* Keep the following items in alphabetic order of name.               */
/***********************************************************************/
#ifdef MSWIN
 static QUERY_ITEM __far query_item[] =
#else
 static QUERY_ITEM query_item[] =
#endif
  {
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"alt",3,ITEM_ALT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"arbchar",3,ITEM_ARBCHAR,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"autosave",2,ITEM_AUTOSAVE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"backup",4,ITEM_BACKUP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"beep",4,ITEM_BEEP,1},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"block",5,ITEM_BLOCK,0},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"case",4,ITEM_CASE,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"clearscreen",6,ITEM_CLEARSCREEN,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"clock",5,ITEM_CLOCK,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"cmdarrows",4,ITEM_CMDARROWS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"cmdline",3,ITEM_CMDLINE,2},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"color",5,ITEM_COLOR,1},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"column",3,ITEM_COLUMN,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"compat",6,ITEM_COMPAT,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"curline",3,ITEM_CURLINE,6},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"cursor",4,ITEM_CURSOR,8},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"cursorstay",8,ITEM_CURSORSTAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"defsort",7,ITEM_DEFSORT,2},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"dirfileid",9,ITEM_DIRFILEID,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"display",4,ITEM_DISPLAY,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"eof",3,ITEM_EOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"eolout",3,ITEM_EOLOUT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"etmode",6,ITEM_ETMODE,2},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"fext",2,ITEM_FEXT,1},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"filename",5,ITEM_FILENAME,1},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"fmode",2,ITEM_FMODE,1},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"fname",2,ITEM_FNAME,1},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"fpath",2,ITEM_FPATH,1},
   {QUERY_QUERY|             QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"ftype",2,ITEM_FTYPE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"fullfname",6,ITEM_FULLFNAME,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"getenv",6,ITEM_GETENV,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hex",3,ITEM_HEX,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hexdisplay",7,ITEM_HEXDISPLAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"hexshow",4,ITEM_HEXSHOW,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"highlight",4,ITEM_HIGHLIGHT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"idline",2,ITEM_IDLINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"impmacro",6,ITEM_IMPMACRO,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"impos",5,ITEM_IMPOS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"inputmode",6,ITEM_INPUTMODE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"insertmode",6,ITEM_INSERTMODE,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"lastmsg",4,ITEM_LASTMSG,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"lastrc",6,ITEM_LASTRC,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"length",3,ITEM_LENGTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"line",2,ITEM_LINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"lineflag",8,ITEM_LINEFLAG,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"linend",5,ITEM_LINEND,2},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"lscreen",2,ITEM_LSCREEN,6},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"macro",5,ITEM_MACRO,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"macroext",6,ITEM_MACROEXT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"macropath",6,ITEM_MACROPATH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"margins",3,ITEM_MARGINS,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"monitor",7,ITEM_MONITOR,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"mouse",5,ITEM_MOUSE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"msgline",4,ITEM_MSGLINE,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"msgmode",4,ITEM_MSGMODE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"nbfile",3,ITEM_NBFILE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"newlines",4,ITEM_NEWLINES,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"nondisp",4,ITEM_NONDISP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"number",3,ITEM_NUMBER,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"pending",4,ITEM_PENDING,4},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"point",1,ITEM_POINT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"position",3,ITEM_POSITION,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"prefix",3,ITEM_PREFIX,0},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"printer",7,ITEM_PRINTER,1},
   {QUERY_READV                                        ,(CHARTYPE *)"readv",5,ITEM_READV,3},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"reprofile",6,ITEM_REPROFILE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"reserved",5,ITEM_RESERVED,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"rexxoutput",7,ITEM_REXXOUTPUT,2},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"ring",4,ITEM_RING,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"scale",4,ITEM_SCALE,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"scope",5,ITEM_SCOPE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"screen",3,ITEM_SCREEN,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"select",3,ITEM_SELECT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"shadow",4,ITEM_SHADOW,1},
   {                         QUERY_EXTRACT             ,(CHARTYPE *)"showkey",4,ITEM_SHOWKEY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"size",2,ITEM_SIZE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"statusline",7,ITEM_STATUSLINE,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"stay",4,ITEM_STAY,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabkey",4,ITEM_TABKEY,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabline",4,ITEM_TABLINE,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabs",4,ITEM_TABS,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabsin",5,ITEM_TABSIN,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"tabsout",5,ITEM_TABSOUT,2},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"terminal",4,ITEM_TERMINAL,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"tof",3,ITEM_TOF,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"typeahead",5,ITEM_TYPEAHEAD,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"verify",1,ITEM_VERIFY,1},
   {QUERY_QUERY             |QUERY_EXTRACT             ,(CHARTYPE *)"vershift",4,ITEM_VERSHIFT,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT             ,(CHARTYPE *)"version",7,ITEM_VERSION,4},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"width",1,ITEM_WIDTH,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"word",4,ITEM_WORD,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"wordwrap",5,ITEM_WORDWRAP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"wrap",2,ITEM_WRAP,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"xterminal",5,ITEM_XTERMINAL,1},
   {QUERY_QUERY|QUERY_STATUS|QUERY_EXTRACT|QUERY_MODIFY,(CHARTYPE *)"zone",1,ITEM_ZONE,2},
   {0                                                   ,NULL,0,0,0},
  };
CHARTYPE *block_name[] = {
                          (CHARTYPE *)"",
                          (CHARTYPE *)"LINE",
                          (CHARTYPE *)"BOX",
                          (CHARTYPE *)"STREAM",
                          (CHARTYPE *)"COLUMN",
                          (CHARTYPE *)"WORD"
                         };

VALUE item_values[MAX_VARIABLES_RETURNED];

/***********************************************************************/
#ifdef HAVE_PROTO
short find_item(CHARTYPE *item_name,CHARTYPE query_type)
#else
short find_item(item_name,query_type)
CHARTYPE *item_name;
CHARTYPE query_type;
#endif
/***********************************************************************/
{
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short itemno = (-1);
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :find_item");
#endif
/*---------------------------------------------------------------------*/
/* Using the supplied abbreviation for an item, find the entry in the  */
/* query_item[] array. It should return with the index pointer (for    */
/* QUERY_EXTRACT) or an index to the first item in the function_item[] */
/* rexx.c for other query types.                                       */
/*---------------------------------------------------------------------*/
 for (i=0;query_item[i].name != NULL;i++)
    {
     if (equal(query_item[i].name,item_name,query_item[i].min_len)
     && (query_item[i].query & query_type))
       {
        if (query_type == QUERY_EXTRACT)
           itemno = i;
        else
           itemno = query_item[i].item_number;
        break;
       }
    }
#ifdef TRACE
 trace_return();
#endif
 return(itemno);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short show_status(void)
#else
short show_status()
#endif
/***********************************************************************/
{
#define STATUS_COLS 5
/*-------------------------- external data ----------------------------*/
 extern short terminal_lines;
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0,k=0;
 short lineno=0,colno=0;
 short number_variables=0;
 short item_width=0,column=0,column_width=0,col[STATUS_COLS];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :show_status");
#endif
/*---------------------------------------------------------------------*/
/* For each item that is displayable, display it...                    */
/*---------------------------------------------------------------------*/
 column = 0;
 column_width = COLS / STATUS_COLS;
 col[0] = 0;
 for (i=1;i<STATUS_COLS;i++)
    col[i] = col[i-1]+column_width+1;
 wclear(stdscr);
 for (i=0;query_item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are queryable...                       */
/*---------------------------------------------------------------------*/
     if (query_item[i].query & QUERY_STATUS)
       {
        number_variables = get_item_values(query_item[i].item_number,(CHARTYPE *)"",QUERY_STATUS,0L,NULL,0L);
        item_width = 0;
/*---------------------------------------------------------------------*/
/* Obtain the total length of the setting values...                    */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
           item_width += item_values[j].len+1;
        item_width--;     /* reduce by 1 for last value's blank at end */
/*---------------------------------------------------------------------*/
/* If the length of the variables is > the screen width, go to next    */
/* line.                                                               */
/*---------------------------------------------------------------------*/
        if (item_width+col[column] > COLS)
          {
           column = colno = 0;
           lineno++;
          }
        else
           colno = col[column];
/*---------------------------------------------------------------------*/
/* Display the variables. For the first value, display in BOLD.        */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           if (j == 0)
              attrset(A_BOLD);
           mvaddstr(lineno,colno,(DEFCHAR *)item_values[j].value);
           attrset(A_NORMAL);
           colno += item_values[j].len+1;
          }
        colno--;
/*---------------------------------------------------------------------*/
/* Now have to determine where to display the next set of variables... */
/* If the just-displayed values go past the last column, or we are     */
/* already in the last column, go to the next line...                  */
/*---------------------------------------------------------------------*/
        if (colno >= col[STATUS_COLS-1] || column == STATUS_COLS-1)
          {
           lineno++;
           column = 0;
          }
        else
          {
/*---------------------------------------------------------------------*/
/* ...else if the the just-displayed values go past the 2nd column...  */
/* If the just-displayed values go past the last column, go on to the  */
/* next line...                                                        */
/*---------------------------------------------------------------------*/
           for (k=0;k<STATUS_COLS;k++)
              if (colno < col[k])
                {
                 column = k;
                 break;
                }
          }
       }
    }
 mvaddstr(terminal_lines-2,0,HIT_ANY_KEY);
 refresh();

#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short save_status(CHARTYPE *filename)
#else
short save_status(filename)
CHARTYPE *filename;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern CHARTYPE sp_path[MAX_FILE_NAME+1] ;
 extern CHARTYPE sp_fname[MAX_FILE_NAME+1] ;
 extern CHARTYPE *the_version;
 extern CHARTYPE *the_release;
/*--------------------------- local data ------------------------------*/
 register short i=0,j=0;
 short number_variables=0,rc=RC_OK;
 FILE *fp=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c   :save_status");
#endif
/*---------------------------------------------------------------------*/
/* Get the fully qualified filename from the supplied filename.        */
/*---------------------------------------------------------------------*/
 strrmdup(strtrans(filename,OSLASH,ISLASH),ISLASH);
 if ((rc = splitpath(filename)) != RC_OK)
   {
    display_error(10,filename,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
/*---------------------------------------------------------------------*/
/* splitpath() has set up sp_path  and sp_fname. Append the filename.  */
/*---------------------------------------------------------------------*/
 strcat((DEFCHAR *)sp_path,(DEFCHAR *)sp_fname);
/*---------------------------------------------------------------------*/
/* If the file already exists, display an error.                       */
/*---------------------------------------------------------------------*/
  if (file_exists(sp_path))
   {
    display_error(8,filename,FALSE);
#ifdef TRACE
    trace_return();
#endif
    return(rc);
   }
 fp = fopen((DEFCHAR *)sp_path,"w");
/*---------------------------------------------------------------------*/
/* For each item that is modifiable, get its value...                  */
/*---------------------------------------------------------------------*/
 fprintf(fp,"/* This file generated by STATUS command of THE %s %s*/\n",the_version,the_release);
 for (i=0;query_item[i].name != NULL;i++)
    {
/*---------------------------------------------------------------------*/
/* Get only those settings that are modifiable...                      */
/*---------------------------------------------------------------------*/
     if (query_item[i].query & QUERY_MODIFY)
       {
        number_variables = get_item_values(query_item[i].item_number,(CHARTYPE *)"",QUERY_MODIFY,0L,NULL,0L);
        fputs("'set",fp);
/*---------------------------------------------------------------------*/
/* Write the variables to the file...                                  */
/*---------------------------------------------------------------------*/
        for (j=0;j<number_variables+1;j++)
          {
           fputc(' ',fp);
           fputs((DEFCHAR *)item_values[j].value,fp);
          }
        fprintf(fp,"'\n");
       }
    }
 fflush(fp);
 fclose(fp);
#ifdef TRACE
 trace_return();
#endif
 return(RC_OK);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short set_extract_variables(short itemno)
#else
short set_extract_variables(itemno)
short itemno;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VALUE item_values[MAX_VARIABLES_RETURNED];
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short rc=RC_OK,number_values=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   set_extract_variables");
#endif
 number_values = atoi((DEFCHAR *)item_values[0].value);
 for (i=0;i<number_values+1;i++)
  {
   if ((rc = set_rexx_variable(query_item[itemno].name,item_values[i].value,item_values[i].len,i)) != RC_OK)
      break;
  } 
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
short get_item_values(short itemno,CHARTYPE *itemargs,CHARTYPE query_type,LINETYPE argc,CHARTYPE *arg,LINETYPE arglen)
#else
short get_item_values(itemno,itemargs,query_type,argc,arg,arglen)
short itemno;
CHARTYPE *itemargs;
CHARTYPE query_type;
LINETYPE argc;
CHARTYPE *arg;
LINETYPE arglen;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*
 * DO NOT USE temp_cmd as temporary working space, use trec
 */
 extern CHARTYPE term_name[20];
 extern CHARTYPE dir_path[MAX_FILE_NAME+1];
 extern CHARTYPE xterm_program[MAX_FILE_NAME+1];
 extern CHARTYPE *trec;
 extern unsigned short file_start;
 extern CHARTYPE *the_version;
 extern CHARTYPE *the_release;
 extern CHARTYPE the_macro_path[MAX_FILE_NAME+1];
 extern CHARTYPE macro_suffix[12];
 extern CHARTYPE CLEARSCREENx;
 extern CHARTYPE CMDARROWSTABCMDx;
 extern bool REPROFILEx;
 extern bool CAPREXXOUTx;
 extern int DEFSORTx;
 extern int DIRORDERx;
 extern LINETYPE CAPREXXMAXx;
 extern CHARTYPE TABI_ONx;
 extern CHARTYPE TABI_Nx;
 extern bool BEEPx;
 extern bool CLOCKx;
 extern bool HEXDISPLAYx;
 extern CHARTYPE number_of_files;
 extern bool INSERTMODEx;
 extern CHARTYPE tabkey_insert;
 extern CHARTYPE tabkey_overwrite;
 extern short lastrc;
 extern CHARTYPE *last_message;
 extern bool in_profile;
 extern bool batch_only;
 extern bool rexx_support;
 extern bool scroll_cursor_stay;
 extern unsigned short rec_len;
 extern unsigned short cmd_rec_len;
 extern unsigned short pre_rec_len;
 extern bool colour_support;
 extern bool NONDISPx;
 extern bool MOUSEx;
 extern ROWTYPE STATUSLINEx;
 extern CHARTYPE *cmd_rec;
 extern CHARTYPE *rec;
 extern short compatible_look;
 extern short compatible_feel;
 extern short compatible_keys;
 extern LINETYPE original_screen_line;
 extern LINETYPE original_screen_column;
 extern LINETYPE original_file_line;
 extern LINETYPE original_file_column;
 extern short terminal_lines;
 extern bool TYPEAHEADx;
 extern bool horizontal;
 extern CHARTYPE display_screens;

#if defined(UNIX) || defined(OS2) || defined(EMX)
 extern CHARTYPE *spooler_name;
#endif
 extern VIEW_DETAILS *vd_mark;
/*--------------------------- local data ------------------------------*/
 static CHARTYPE num0[3];
 static CHARTYPE num1[10];
 static CHARTYPE num2[10];
 static CHARTYPE num3[40];
 static CHARTYPE num4[40];
 static CHARTYPE num5[10];
 static CHARTYPE num6[10];
 static CHARTYPE num7[10];
 static CHARTYPE num8[10];
 static CHARTYPE rsrvd[MAX_FILE_NAME+1];
 static LINE *curr;
 register int i=0;
 short y=0,x=0;
 short number_variables = 1;
 LINETYPE true_line=0L;
 bool bool_flag=FALSE;
 CHARTYPE cursor_char=0;
 CHARTYPE *tmpbuf=NULL;
 LINETYPE current_screen_line=(-1L);
 LINETYPE current_screen_column=(-1L);
 LINETYPE current_file_line=(-1L);
 LINETYPE current_file_column=(-1L);
 int key=0;
 CHARTYPE *keyname=NULL;
 LENGTHTYPE col=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   get_item_values");
#endif
/*---------------------------------------------------------------------*/
/* Only use itemno to reference query_item[] array if it is NOT a      */
/* boolean function...                                                 */
/*---------------------------------------------------------------------*/
 if (itemno < ITEM_START_BOOLEANS)
    number_variables = query_item[itemno].number_values;
/*---------------------------------------------------------------------*/
/* Now that we have a valid item, set up its values...                 */
/*---------------------------------------------------------------------*/
 switch(itemno)
   {
    case ITEM_AFTER_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_FILEAREA:
                 if ((x+CURRENT_VIEW->verify_col-1) >= rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x >= cmd_rec_len)
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:
                 if (x >= pre_rec_len)
                    bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BATCH_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_ALT:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_FILE->autosave_alt);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_FILE->save_alt);
         item_values[1].value = num1;
         item_values[2].value = num2;
         item_values[1].len = strlen((DEFCHAR *)num1);
         item_values[2].len = strlen((DEFCHAR *)num2);
         break;
    case ITEM_ARBCHAR:
         if (CURRENT_VIEW->arbchar_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->arbchar_multiple;
         num1[1] ='\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         num2[0] = CURRENT_VIEW->arbchar_single;
         num2[1] ='\0';
         item_values[3].value = num2;
         item_values[3].len = 1;
         break;
    case ITEM_AUTOSAVE:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_FILE->autosave);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         if (CURRENT_FILE->autosave == 0)
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_BACKUP:
         switch(CURRENT_FILE->backup)
           {
            case BACKUP_OFF:
                 item_values[1].value = (CHARTYPE *)"OFF";
                 break;
            case BACKUP_TEMP:
                 item_values[1].value = (CHARTYPE *)"TEMP";
                 break;
            case BACKUP_KEEP:
                 item_values[1].value = (CHARTYPE *)"KEEP";
                 break;
           }
         item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
         break;
    case ITEM_BEFORE_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         bool_flag = FALSE;
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_FILEAREA:
                 if ((x+CURRENT_VIEW->verify_col-1) < memne(rec,' ',rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_COMMAND:
                 if (x < memne(cmd_rec,' ',cmd_rec_len))
                    bool_flag = TRUE;
                 break;
            case WINDOW_PREFIX:/* cursor can't go before 1st non-blank */
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BEEP:
         if (BEEPx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_BLANK_FUNCTION:
         item_values[1].value = (CHARTYPE *)"0"; /* FALSE by default */
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_FILEAREA:
                 if (rec_len == 0)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
            case WINDOW_PREFIX:
                 if (pre_rec_len == 0)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
            case WINDOW_COMMAND:
                 if (cmd_rec_len == 0)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
           }
         item_values[1].len = 1;
         break;
    case ITEM_BLOCK:
         if (MARK_VIEW == NULL)
           {
            item_values[1].value = (CHARTYPE *)"NONE";
            item_values[1].len = 4;
            number_variables = 1;
           }
          else
           {
            number_variables = 6;
            item_values[1].value = block_name[MARK_VIEW->mark_type];
            item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
            sprintf((DEFCHAR *)num1,"%ld",MARK_VIEW->mark_start_line);
            item_values[2].value = num1;
            item_values[2].len = strlen((DEFCHAR *)num1);
            sprintf((DEFCHAR *)num2,"%d",MARK_VIEW->mark_start_col);
            item_values[3].value = num2;
            item_values[3].len = strlen((DEFCHAR *)num2);
            sprintf((DEFCHAR *)num3,"%ld",MARK_VIEW->mark_end_line);
            item_values[4].value = num3;
            item_values[4].len = strlen((DEFCHAR *)num3);
            sprintf((DEFCHAR *)num4,"%d",MARK_VIEW->mark_end_col);
            item_values[5].value = num4;
            item_values[5].len = strlen((DEFCHAR *)num4);
            strcpy((DEFCHAR *)trec,(DEFCHAR *)MARK_FILE->fpath);
            strcat((DEFCHAR *)trec,(DEFCHAR *)MARK_FILE->fname);
            item_values[6].value = (CHARTYPE *)trec;
            item_values[6].len = strlen((DEFCHAR *)trec);
           }
         break;
    case ITEM_BLOCK_FUNCTION:
         if (CURRENT_VIEW == MARK_VIEW)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_BOTTOMEDGE_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_FILEAREA
         &&  y == CURRENT_SCREEN.rows[WINDOW_FILEAREA]-1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_CASE:
         switch(CURRENT_VIEW->case_enter)
           {
            case CASE_MIXED:
                 item_values[1].value = (CHARTYPE *)"MIXED";
                 break;
            case CASE_UPPER:
                 item_values[1].value = (CHARTYPE *)"UPPER";
                 break;
            case CASE_LOWER:
                 item_values[1].value = (CHARTYPE *)"LOWER";
                 break;
            default:
                 break;
           }
         item_values[1].len = 5;
         switch(CURRENT_VIEW->case_locate)
           {
            case CASE_IGNORE:
                 item_values[2].value = (CHARTYPE *)"IGNORE";
                 item_values[2].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[2].value = (CHARTYPE *)"RESPECT";
                 item_values[2].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_change)
           {
            case CASE_IGNORE:
                 item_values[3].value = (CHARTYPE *)"IGNORE";
                 item_values[3].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[3].value = (CHARTYPE *)"RESPECT";
                 item_values[3].len = 7;
                 break;
            default:
                 break;
           }
         switch(CURRENT_VIEW->case_sort)
           {
            case CASE_IGNORE:
                 item_values[4].value = (CHARTYPE *)"IGNORE";
                 item_values[4].len = 6;
                 break;
            case CASE_RESPECT:
                 item_values[4].value = (CHARTYPE *)"RESPECT";
                 item_values[4].len = 7;
                 break;
            default:
                 break;
           }
         break;
    case ITEM_CLEARSCREEN:
         if (CLEARSCREENx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_CLOCK:
         if (CLOCKx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_COMMAND_FUNCTION:
         if (CURRENT_WINDOW_COMMAND != NULL)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_CMDARROWS:
         if (CMDARROWSTABCMDx)
           {
            item_values[1].value = (CHARTYPE *)"TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"RETRIEVE";
            item_values[1].len = 8;
           }
         break;
    case ITEM_CMDLINE:
         switch(CURRENT_VIEW->cmd_line)
           {
            case 'B':
                 item_values[1].value = (CHARTYPE *)"BOTTOM";
                 item_values[1].len = 6;
                 break;
            case 'T':
                 item_values[1].value = (CHARTYPE *)"TOP";
                 item_values[1].len = 3;
                 break;
            case 'O':
                 item_values[1].value = (CHARTYPE *)"OFF";
                 item_values[1].len = 3;
                 number_variables = 1;
                 break;
           }
         if (CURRENT_VIEW->cmd_line == 'O')
            break;
         if (query_type == QUERY_EXTRACT
         ||  query_type == QUERY_FUNCTION)
           {
            sprintf((DEFCHAR *)num1,"%d",CURRENT_SCREEN.start_row[WINDOW_COMMAND]+1);
            item_values[2].value = num1;
            item_values[2].len = strlen((DEFCHAR *)num1);
            item_values[3].value = cmd_rec;
            item_values[3].len = cmd_rec_len;
            number_variables = 3;
           }
         else
            number_variables = 1;
         break;
    case ITEM_COLOR:
         number_variables = extract_colour(itemno,rsrvd,query_type,itemargs);
         break;
    case ITEM_COLUMN:
         if (batch_only
         ||  CURRENT_VIEW->current_window != WINDOW_FILEAREA)
            sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->current_column);
         else
           {
            getyx(CURRENT_WINDOW,y,x);
            sprintf((DEFCHAR *)num1,"%d",x+CURRENT_VIEW->verify_col);
           }
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;                       
    case ITEM_COMPAT:
         switch(compatible_look)
           {
            case COMPAT_THE:
                 item_values[1].value = (CHARTYPE *)"THE";
                 item_values[1].len = 3;
                 break;
            case COMPAT_XEDIT:
                 item_values[1].value = (CHARTYPE *)"XEDIT";
                 item_values[1].len = 5;
                 break;
            case COMPAT_KEDIT:
                 item_values[1].value = (CHARTYPE *)"KEDIT";
                 item_values[1].len = 5;
                 break;
           }
         switch(compatible_feel)
           {
            case COMPAT_THE:
                 item_values[2].value = (CHARTYPE *)"THE";
                 item_values[2].len = 3;
                 break;
            case COMPAT_XEDIT:
                 item_values[2].value = (CHARTYPE *)"XEDIT";
                 item_values[2].len = 5;
                 break;
            case COMPAT_KEDIT:
                 item_values[2].value = (CHARTYPE *)"KEDIT";
                 item_values[2].len = 5;
                 break;
           }
         switch(compatible_keys)
           {
            case COMPAT_THE:
                 item_values[3].value = (CHARTYPE *)"THE";
                 item_values[3].len = 3;
                 break;
            case COMPAT_XEDIT:
                 item_values[3].value = (CHARTYPE *)"XEDIT";
                 item_values[3].len = 5;
                 break;
            case COMPAT_KEDIT:
                 item_values[3].value = (CHARTYPE *)"KEDIT";
                 item_values[3].len = 5;
                 break;
           }
         break;
    case ITEM_CURLINE:
         if (CURRENT_VIEW->current_base == POSITION_MIDDLE)
           {
            if (CURRENT_VIEW->current_off == 0)
               strcpy((DEFCHAR *)rsrvd,"M");
            else
               sprintf((DEFCHAR *)rsrvd,"M%+d",CURRENT_VIEW->current_off);
           }
         else
            sprintf((DEFCHAR *)rsrvd,"%d",CURRENT_VIEW->current_off);
         item_values[1].value = rsrvd;
         item_values[1].len = strlen((DEFCHAR *)rsrvd);
         if (query_type == QUERY_EXTRACT
         ||  query_type == QUERY_FUNCTION)
           {
            sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->current_row+1);
            item_values[2].value = num1;
            item_values[2].len = strlen((DEFCHAR *)num1);
            curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,
                           (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE),
                           CURRENT_FILE->number_lines);
            item_values[3].value = (CHARTYPE *)curr->line;
            item_values[3].len = curr->length;
            item_values[4].value = (curr->new_flag||curr->changed_flag)?(CHARTYPE *)"ON":(CHARTYPE *)"OFF";
            item_values[4].len = strlen((DEFCHAR *)item_values[4].value);
            if (curr->new_flag)
               item_values[5].value = (CHARTYPE *)"NEW CHANGED";
            else
               if (curr->changed_flag)
                  item_values[5].value = (CHARTYPE *)"OLD CHANGED";
               else
                  item_values[5].value = (CHARTYPE *)"OLD";
            item_values[5].len = strlen((DEFCHAR *)item_values[5].value);
            sprintf((DEFCHAR *)num2,"%d",curr->select);
            item_values[6].value = num2;
            item_values[6].len = strlen((DEFCHAR *)num2);
           }
         else
            number_variables = 1;
         break;
    case ITEM_CURSOR:
         get_cursor_position(&current_screen_line,&current_screen_column,
                             &current_file_line,&current_file_column);
         sprintf((DEFCHAR *)num1,"%ld",current_screen_line);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%ld",current_screen_column);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         sprintf((DEFCHAR *)num3,"%ld",current_file_line);
         item_values[3].value = num3;
         item_values[3].len = strlen((DEFCHAR *)num3);
         sprintf((DEFCHAR *)num4,"%ld",current_file_column);
         item_values[4].value = num4;
         item_values[4].len = strlen((DEFCHAR *)num4);
         sprintf((DEFCHAR *)num5,"%ld",original_screen_line);
         item_values[5].value = num5;
         item_values[5].len = strlen((DEFCHAR *)num5);
         sprintf((DEFCHAR *)num6,"%ld",original_screen_column);
         item_values[6].value = num6;
         item_values[6].len = strlen((DEFCHAR *)num6);
         sprintf((DEFCHAR *)num7,"%ld",original_file_line);
         item_values[7].value = num7;
         item_values[7].len = strlen((DEFCHAR *)num7);
         sprintf((DEFCHAR *)num8,"%ld",original_file_column);
         item_values[8].value = num8;
         item_values[8].len = strlen((DEFCHAR *)num8);
         break;
    case ITEM_CURSORSTAY:
         if (scroll_cursor_stay)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_CURRENT_FUNCTION:
         if (CURRENT_VIEW->current_window != WINDOW_COMMAND
         &&  CURRENT_VIEW->focus_line == CURRENT_VIEW->current_line)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_DIR_FUNCTION:
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_DEFSORT:
         switch(DEFSORTx)
           {
            case DIRSORT_DIR:
                 item_values[1].value = (CHARTYPE *)"DIRECTORY";
                 break;
            case DIRSORT_NAME:
                 item_values[1].value = (CHARTYPE *)"NAME";
                 break;
            case DIRSORT_SIZE:
                 item_values[1].value = (CHARTYPE *)"SIZE";
                 break;
            case DIRSORT_TIME:
                 item_values[1].value = (CHARTYPE *)"TIME";
                 break;
            case DIRSORT_DATE:
                 item_values[1].value = (CHARTYPE *)"DATE";
                 break;
            case DIRSORT_NONE:
                 item_values[1].value = (CHARTYPE *)"OFF";
                 break;
           }
         item_values[1].len = strlen((DEFCHAR*)item_values[1].value);
         if (DIRORDERx == DIRSORT_ASC)
           {
            item_values[2].value = (CHARTYPE *)"ASCENDING";
            item_values[2].len = 9;
           }
         else
           {
            item_values[2].value = (CHARTYPE *)"DESCENDING";
            item_values[2].len = 10;
           }
         break;
    case ITEM_DIRFILEID:
         true_line = (-1L);
         if (CURRENT_FILE->pseudo_file == PSEUDO_DIR)
           {
            if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
               if (CURRENT_TOF || CURRENT_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->current_line;
            else
               if (FOCUS_TOF || FOCUS_BOF)
                  true_line = (-1L);
               else
                  true_line = CURRENT_VIEW->focus_line;
           }
         if (true_line == (-1L))
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
            item_values[2].value = (CHARTYPE *)"";
            item_values[2].len = 0;
           }
         else
           {
            curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
            item_values[1].value = (CHARTYPE *)dir_path;
            item_values[1].len = strlen((DEFCHAR *)dir_path);
            if (curr->length < file_start)
              {
               item_values[2].value = (CHARTYPE *)"";
               item_values[2].len = 0;
              }
            else
              {
               item_values[2].value = (CHARTYPE *)curr->line+file_start;
               item_values[2].len = strlen((DEFCHAR *)curr->line+file_start);
              }
           }
         break;
    case ITEM_END_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         item_values[1].value = (CHARTYPE *)"0"; /* set FALSE by default */
         getyx(CURRENT_WINDOW,y,x);
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_FILEAREA:
                 if (x+CURRENT_VIEW->verify_col == rec_len)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
            case WINDOW_PREFIX:
                 if (pre_rec_len > 0
                 &&  pre_rec_len-1 == x)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
            case WINDOW_COMMAND:
                 if (cmd_rec_len > 0
                 &&  cmd_rec_len-1 == x)
                    item_values[1].value = (CHARTYPE *)"1";
                 break;
           }
         item_values[1].len = 1;
         break;
    case ITEM_DISPLAY:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->display_low);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_VIEW->display_high);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         break;
    case ITEM_EOF:
         if (CURRENT_BOF)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_EOF_FUNCTION:
         if (FOCUS_BOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_EOLOUT:
         switch(CURRENT_FILE->eolout)
           {
            case EOLOUT_LF:
                 item_values[1].value = (CHARTYPE *)"LF";
                 item_values[1].len = 2;
                 break;
            case EOLOUT_CR:
                 item_values[1].value = (CHARTYPE *)"CR";
                 item_values[1].len = 2;
                 break;
            case EOLOUT_CRLF:
                 item_values[1].value = (CHARTYPE *)"CRLF";
                 item_values[1].len = 4;
                 break;
            case EOLOUT_NONE:
                 item_values[1].value = (CHARTYPE *)"NONE";
                 item_values[1].len = 4;
                 break;
           }
         break;
    case ITEM_ETMODE:
         get_etmode(num1,rsrvd);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_FIRST_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         if (x == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FOCUSEOF_FUNCTION:
         if (FOCUS_BOF)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FOCUSTOF_FUNCTION:
         if (FOCUS_TOF)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_FILENAME:
         item_values[1].value = (CHARTYPE *)CURRENT_FILE->fname;
         item_values[1].len = strlen((DEFCHAR *)CURRENT_FILE->fname);
         break;
    case ITEM_FMODE:
#if defined(UNIX)
         item_values[1].value = (CHARTYPE *)"";
         item_values[1].len = 0;
#else
         num0[0] = (CHARTYPE)*(CURRENT_FILE->fpath);
         num0[1] = '\0';
         item_values[1].value = num0;
         item_values[1].len = 1;
#endif
         break;
    case ITEM_FNAME:
         strcpy((DEFCHAR*)rsrvd,(DEFCHAR*)CURRENT_FILE->fname);
         x = strzreveq(rsrvd,'.');
         if (x != (-1))
            rsrvd[x] = '\0';
         item_values[1].value = rsrvd;
         item_values[1].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_FPATH:
         item_values[1].value = (CHARTYPE *)CURRENT_FILE->fpath;
         item_values[1].len = strlen((DEFCHAR *)CURRENT_FILE->fpath);
         break;
    case ITEM_FTYPE:
    case ITEM_FEXT:
         x = strzreveq(CURRENT_FILE->fname,'.');
         if (x == (-1))
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)CURRENT_FILE->fname+x+1;
            item_values[1].len = strlen((DEFCHAR *)CURRENT_FILE->fname+x+1);
           }
         break;
    case ITEM_FULLFNAME:
         if (CURRENT_FILE->display_actual_filename)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_GETENV:
#if defined(UNIX)
         if (query_type == QUERY_FUNCTION)
            tmpbuf = (CHARTYPE *)getenv((DEFCHAR *)arg);
         else
            tmpbuf = (CHARTYPE *)getenv((DEFCHAR *)itemargs);
#else
         if (query_type == QUERY_FUNCTION)
            tmpbuf = (CHARTYPE *)getenv((DEFCHAR *)make_upper(arg));
         else
            tmpbuf = (CHARTYPE *)getenv((DEFCHAR *)make_upper(itemargs));
#endif
         if (tmpbuf == NULL)
            item_values[1].value = (CHARTYPE *)"***invalid***";
         else
            item_values[1].value = tmpbuf;
         item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
         break;
    case ITEM_HEX:
         if (CURRENT_VIEW->hex)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_HEXDISPLAY:
         if (HEXDISPLAYx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_HEXSHOW:
         if (CURRENT_VIEW->hexshow_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->hexshow_base == POSITION_MIDDLE)
            sprintf((DEFCHAR *)rsrvd,"M%+d",CURRENT_VIEW->hexshow_off);
         else
            sprintf((DEFCHAR *)rsrvd,"%d",CURRENT_VIEW->hexshow_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_HIGHLIGHT:
         switch (CURRENT_VIEW->highlight)
           {
            case HIGHLIGHT_NONE:
                 item_values[1].value = (CHARTYPE *)"OFF";
                 break;
            case HIGHLIGHT_ALT:
                 item_values[1].value = (CHARTYPE *)"ALTERED";
                 break;
            case HIGHLIGHT_TAG:
                 item_values[1].value = (CHARTYPE *)"TAGGED";
                 break;                                    
            case HIGHLIGHT_SELECT:
                 item_values[1].value = (CHARTYPE *)"SELECT";
                 sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->highlight_low);
                 item_values[2].value = num1;
                 item_values[2].len = strlen((DEFCHAR *)num1);
                 sprintf((DEFCHAR *)num2,"%d",CURRENT_VIEW->highlight_high);
                 item_values[3].value = num2;
                 item_values[3].len = strlen((DEFCHAR *)num2);
                 number_variables = 3;
                 break;                                    
            default:
                 break;                                    
           }
         item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
         break;
    case ITEM_IDLINE:
         if (CURRENT_VIEW->id_line)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPMACRO:
         if (CURRENT_VIEW->imp_macro)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_IMPOS:
         if (CURRENT_VIEW->imp_os)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_INBLOCK_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         bool_flag = FALSE;
         if (CURRENT_VIEW == MARK_VIEW
         &&  CURRENT_VIEW->current_window == WINDOW_FILEAREA)
           {
            getyx(CURRENT_WINDOW_FILEAREA,y,x);
            switch(MARK_VIEW->mark_type)
              {
               case M_LINE:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line))
                      bool_flag = TRUE;
                    break;
               case M_BOX:
               case M_WORD:
               case M_COLUMN:
                    if ((CURRENT_VIEW->focus_line >= MARK_VIEW->mark_start_line)
                    &&  (CURRENT_VIEW->focus_line <= MARK_VIEW->mark_end_line)
                    &&  (x + CURRENT_VIEW->verify_start >= MARK_VIEW->mark_start_col)
                    &&  (x + CURRENT_VIEW->verify_start <= MARK_VIEW->mark_end_col))
                      bool_flag = TRUE;
                    break;
               case M_STREAM:
                    break;
               default:
                    break;
              }
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INCOMMAND_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INITIAL_FUNCTION:
         if (in_profile)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INPREFIX_FUNCTION:
         if (CURRENT_VIEW->current_window == WINDOW_PREFIX)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_INPUTMODE:
         if (CURRENT_VIEW->inputmode == INPUTMODE_OFF)
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->inputmode == INPUTMODE_FULL)
           {
            item_values[1].value = (CHARTYPE *)"FULL";
            item_values[1].len = 4;
           }
         if (CURRENT_VIEW->inputmode == INPUTMODE_LINE)
           {
            item_values[1].value = (CHARTYPE *)"LINE";
            item_values[1].len = 4;
           }
         break;
    case ITEM_INSERTMODE:
         if (INSERTMODEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_LASTMSG:
         item_values[1].value = (CHARTYPE *)last_message;
         item_values[1].len = strlen((DEFCHAR *)last_message);
         break;
    case ITEM_LASTRC:
         sprintf((DEFCHAR *)num1,"%d",lastrc);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_LEFTEDGE_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_FILEAREA
         &&  x == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_LENGTH:
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,CURRENT_VIEW->current_line,CURRENT_FILE->number_lines);
         sprintf((DEFCHAR *)num1,"%d",curr->length);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_LINE:
         sprintf((DEFCHAR *)num1,"%ld",
                 (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE));
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_LINEFLAG:
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,
                         (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE),
                         CURRENT_FILE->number_lines);
         if (curr->new_flag)
            item_values[1].value = (CHARTYPE *)"NEW";
         else
            item_values[1].value = (CHARTYPE *)"NONEW";
         item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
         if (curr->changed_flag)
            item_values[2].value = (CHARTYPE *)"CHANGE";
         else
            item_values[2].value = (CHARTYPE *)"NOCHANGE";
         item_values[2].len = strlen((DEFCHAR *)item_values[2].value);
         if (curr->tag_flag)
            item_values[3].value = (CHARTYPE *)"TAG";
         else
            item_values[3].value = (CHARTYPE *)"NOTAG";
         item_values[3].len = strlen((DEFCHAR *)item_values[3].value);
         break;
    case ITEM_LINEND:
         if (CURRENT_VIEW->linend_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         num1[0] = CURRENT_VIEW->linend_value;
         num1[1] = '\0';
         item_values[2].value = num1;
         item_values[2].len = 1;
         break;
    case ITEM_LSCREEN:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_SCREEN.screen_rows);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_SCREEN.screen_cols);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         sprintf((DEFCHAR *)num3,"%d",CURRENT_SCREEN.screen_start_row+1);
         item_values[3].value = num3;
         item_values[3].len = strlen((DEFCHAR *)num3);
         sprintf((DEFCHAR *)num4,"%d",CURRENT_SCREEN.screen_start_col+1);
         item_values[4].value = num4;
         item_values[4].len = strlen((DEFCHAR *)num4);
         sprintf((DEFCHAR *)num5,"%d",terminal_lines);
         item_values[5].value = num5;
         item_values[5].len = strlen((DEFCHAR *)num5);
         sprintf((DEFCHAR *)num6,"%d",COLS);
         item_values[6].value = num6;
         item_values[6].len = strlen((DEFCHAR *)num6);
         break;
    case ITEM_MACRO:
         if (CURRENT_VIEW->macro)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_MACROEXT:
         if (strlen((DEFCHAR *)macro_suffix) == 0)
            item_values[1].value = (CHARTYPE *)macro_suffix;
         else
            item_values[1].value = (CHARTYPE *)macro_suffix+1;
         item_values[1].len = strlen((DEFCHAR *)macro_suffix);
         break;
    case ITEM_MACROPATH:
         item_values[1].value = (CHARTYPE *)the_macro_path;
         item_values[1].len = strlen((DEFCHAR *)the_macro_path);
         break;
    case ITEM_MARGINS:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->margin_left);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_VIEW->margin_right);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         if (CURRENT_VIEW->margin_indent_offset_status)
            sprintf((DEFCHAR *)num3,"%+d",CURRENT_VIEW->margin_indent);
         else
            sprintf((DEFCHAR *)num3,"%d",CURRENT_VIEW->margin_indent);
         item_values[3].value = num3;
         item_values[3].len = strlen((DEFCHAR *)num3);
         break;
    case ITEM_MODIFIABLE_FUNCTION:
         switch(CURRENT_VIEW->current_window)
           {
            case WINDOW_FILEAREA:
                 if (batch_only)
                   {
                    bool_flag = FALSE;
                    break;
                   }
                 getyx(CURRENT_WINDOW,y,x);
                 if (FOCUS_TOF 
                 ||  FOCUS_BOF
                 ||  CURRENT_SCREEN.sl[y].line_type == LINE_SHADOW)
                    bool_flag = FALSE;
                 else
                    bool_flag = TRUE;
                 break;
            default:
                 bool_flag = TRUE;
                 break;
           }
         if (bool_flag)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_MONITOR:
#ifdef A_COLOR
         if (colour_support)
           {
            item_values[1].value = (CHARTYPE *)"COLOR";
            item_values[1].len = 5;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"MONO";
            item_values[1].len = 4;
           }
          item_values[2].value = (CHARTYPE *)"COLOR";
          item_values[2].len = 5;
          break;
#else
          item_values[1].value = (CHARTYPE *)"MONO";
          item_values[1].len = 4;
          item_values[2].value = (CHARTYPE *)"MONO";
          item_values[2].len = 4;
         break;
#endif
    case ITEM_MOUSE:
         if (MOUSEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_MSGLINE:
         item_values[1].value = (CHARTYPE *)"ON";
         item_values[1].len = 2;
         if (CURRENT_VIEW->msgline_base == POSITION_MIDDLE)
            sprintf((DEFCHAR *)rsrvd,"M%+d",CURRENT_VIEW->msgline_off);
         else
            sprintf((DEFCHAR *)rsrvd,"%d",CURRENT_VIEW->msgline_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen((DEFCHAR *)rsrvd);
         sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->msgline_rows);
         item_values[3].value = num1;
         item_values[3].len = strlen((DEFCHAR *)num1);
         item_values[4].value = (CHARTYPE *)"OVERLAY";
         item_values[4].len = 7;
         break;
    case ITEM_MSGMODE:
         if (CURRENT_VIEW->msgmode_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_NBFILE:
         sprintf((DEFCHAR *)num1,"%d",number_of_files);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_NEWLINES:
         if (CURRENT_VIEW->newline_aligned)
           {
            item_values[1].value = (CHARTYPE *)"ALIGNED";
            item_values[1].len = 7;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"LEFT";
            item_values[1].len = 4;
           }
         break;
    case ITEM_NONDISP:
         num1[0] = NONDISPx;
         num1[1] ='\0';
         item_values[1].value = num1;
         item_values[1].len = 1;
         break;
    case ITEM_NUMBER:
         if (CURRENT_VIEW->number)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_PENDING:
         number_variables = extract_pending(itemno,itemargs);
         break;
    case ITEM_POINT:
         switch(query_type)
           {
            case QUERY_QUERY:
            case QUERY_MODIFY:
                 true_line = (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE);
                 curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
                 if (curr->name == NULL)  /* line not named */
                   {
                    item_values[1].value = (CHARTYPE *)"";
                    item_values[1].len = 0;
                   }
                 else
                   {
                    item_values[1].value = curr->name;
                    item_values[1].len = strlen((DEFCHAR *)curr->name);
                   }
                 break;
            default:
                 number_variables = extract_point(itemno,itemargs);
                 break;
           }
         break;
    case ITEM_POSITION:
         if (CURRENT_VIEW->position_status)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (query_type == QUERY_EXTRACT
         ||  query_type == QUERY_FUNCTION)
           {
            get_current_position(current_screen,&true_line,&col);
            sprintf((DEFCHAR *)num1,"%ld",true_line);
            item_values[2].value = num1;
            item_values[2].len = strlen((DEFCHAR *)num1);
            sprintf((DEFCHAR *)num2,"%d",col);
            item_values[3].value = num2;
            item_values[3].len = strlen((DEFCHAR *)num2);
           }
         else
            number_variables = 1;
         break;
    case ITEM_PREFIX:
         if (rexx_support)
           {
            if (strcmp((DEFCHAR *)itemargs,"") != 0)
              {
               number_variables = extract_prefix(itemno,itemargs);
               break;
              }
           }
         else
           {
            if (strcmp((DEFCHAR *)itemargs,"") != 0)
              {
               display_error(2,(CHARTYPE *)itemargs,FALSE);
               number_variables = EXTRACT_ARG_ERROR;
               break;
              }
           }
         if (CURRENT_VIEW->prefix == PREFIX_OFF)
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
            item_values[2].value = (CHARTYPE *)"";  /* this set to empty deliberately */
            item_values[2].len = 0;
            number_variables = 1;
            break;
           }
         if ((CURRENT_VIEW->prefix&PREFIX_STATUS_MASK) == PREFIX_ON)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"NULLS";
            item_values[1].len = 5;
           }
         if ((CURRENT_VIEW->prefix&PREFIX_LOCATION_MASK) == PREFIX_LEFT)
           {
            item_values[2].value = (CHARTYPE *)"LEFT";
            item_values[2].len = 4;
           }
         else
           {
            item_values[2].value = (CHARTYPE *)"RIGHT";
            item_values[2].len = 5;
           }
         sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->prefix_width);
         item_values[3].value = num1;
         item_values[3].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_VIEW->prefix_gap);
         item_values[4].value = num2;
         item_values[4].len = strlen((DEFCHAR *)num2);
         number_variables = 4;
         break;
    case ITEM_PRINTER:
#if defined(UNIX) || defined(OS2) || defined(EMX)
         item_values[1].value = (CHARTYPE *)spooler_name;
         item_values[1].len = strlen((DEFCHAR *)spooler_name);
#else
         item_values[1].value = (CHARTYPE *)"LPT1";
         item_values[1].len = 4;
#endif
         break;
    case ITEM_REPROFILE:
         if (REPROFILEx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
         else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_READV:
         while(1)
           {
#ifdef CAN_RESIZE
         if (is_termresized())
           {
            (void)THE_Resize(0,0);
            (void)THERefresh((CHARTYPE *)"");
           }
#endif
            key = my_getch(stdscr);
#ifdef CAN_RESIZE
            if (is_termresized())
               continue;
#endif     
            break;
           }
         keyname = get_key_name(key);
         if (keyname == NULL)
           {
            item_values[1].value = (CHARTYPE *)"";
            item_values[1].len = 0;
           }
         else
           {
            item_values[1].value = keyname;
            item_values[1].len = strlen((DEFCHAR *)keyname);
           }
         if (key < 256)
           {
            sprintf((DEFCHAR *)num1,"%d",key);
            item_values[2].value = num1;
            item_values[2].len = strlen((DEFCHAR *)num1);
           }
         else
           {
            item_values[2].value = (CHARTYPE *)"";
            item_values[2].len = 0;
           }
         sprintf((DEFCHAR *)num2,"%d",key);
         item_values[3].value = num2;
         item_values[3].len = strlen((DEFCHAR *)num2);
         break;
    case ITEM_RESERVED:
         number_variables = extract_reserved(itemno,itemargs,query_type);
         break;
    case ITEM_REXXOUTPUT:
         if (CAPREXXOUTx)
           {
            item_values[1].value = (CHARTYPE *)"FILE";
            item_values[1].len = 4;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"DISPLAY";
            item_values[1].len = 7;
           }
         sprintf((DEFCHAR *)num1,"%ld",CAPREXXMAXx);
         item_values[2].value = num1;
         item_values[2].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_RIGHTEDGE_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_FILEAREA
         &&  x == getmaxx(CURRENT_WINDOW)-1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_RING:
         if (query_type == QUERY_FUNCTION)
            number_variables = number_of_files;
         else
            number_variables = extract_ring(itemno,rsrvd,query_type);
         break;
    case ITEM_SCALE:
         if (CURRENT_VIEW->scale_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->scale_base == POSITION_MIDDLE)
            sprintf((DEFCHAR *)rsrvd,"M%+d",CURRENT_VIEW->scale_off);
         else
            sprintf((DEFCHAR *)rsrvd,"%d",CURRENT_VIEW->scale_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_SCOPE:
         if (CURRENT_VIEW->scope_all)
           {
            item_values[1].value = (CHARTYPE *)"ALL";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"DISPLAY";
            item_values[1].len = 7;
           }
         break;
    case ITEM_SCREEN:
         sprintf((DEFCHAR *)num1,"%d",display_screens);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         if (horizontal)
           {
            item_values[2].value = (CHARTYPE *)"HORIZONTAL";
            item_values[2].len = 10;
           }
          else
           {
            item_values[2].value = (CHARTYPE *)"VERTICAL";
            item_values[2].len = 8;
           }
         break;
    case ITEM_SELECT:
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,
                         (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE),
                         CURRENT_FILE->number_lines);
         sprintf((DEFCHAR *)num1,"%d",curr->select);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         x = 0;
         curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,1L,CURRENT_FILE->number_lines);
         while(curr->next != NULL)
           {
            if (curr->select > x)
               x = curr->select;
            curr = curr->next;
           }
         sprintf((DEFCHAR *)num2,"%d",x);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         break;
    case ITEM_SHADOW:
         if (CURRENT_VIEW->shadow)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_SHOWKEY:
         if (strcmp((DEFCHAR *)itemargs,"") == 0)
           {
            item_values[1].value = (CHARTYPE *)"INVALID KEY";
            item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
            break;
           }
         key = atoi((DEFCHAR *)itemargs);
         function_key(key,OPTION_EXTRACT);
         number_variables = EXTRACT_VARIABLES_SET;
         break;
    case ITEM_SIZE:
         sprintf((DEFCHAR *)num1,"%ld",CURRENT_FILE->number_lines);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_SPACECHAR_FUNCTION:
         cursor_char = (CHARTYPE)(winch(CURRENT_WINDOW) & A_CHARTEXT);
         if (cursor_char == ' ')
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_STATUSLINE:
         switch(STATUSLINEx)
           {
            case 'B':
                 item_values[1].value = (CHARTYPE *)"BOTTOM";
                 item_values[1].len = 6;
                 break;
            case 'T':
                 item_values[1].value = (CHARTYPE *)"TOP";
                 item_values[1].len = 3;
                 break;
            case 'O':
                 item_values[1].value = (CHARTYPE *)"OFF";
                 item_values[1].len = 3;
                 break;
            case 'G':
                 item_values[1].value = (CHARTYPE *)"GUI";
                 item_values[1].len = 3;
                 break;
           }
         break;
    case ITEM_STAY:
         if (CURRENT_VIEW->stay)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_TABKEY:
         if (tabkey_overwrite == 'T')
           {
            item_values[1].value = (CHARTYPE *)"TAB";
            item_values[1].len = 3;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"CHARACTER";
            item_values[1].len = 9;
           }
         if (tabkey_insert == 'T')
           {
            item_values[2].value = (CHARTYPE *)"TAB";
            item_values[2].len = 3;
           }
          else
           {
            item_values[2].value = (CHARTYPE *)"CHARACTER";
            item_values[2].len = 9;
           }
         break;
    case ITEM_TABLINE:
         if (CURRENT_VIEW->tab_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         if (CURRENT_VIEW->tab_base == POSITION_MIDDLE)
            sprintf((DEFCHAR *)rsrvd,"M%+d",CURRENT_VIEW->tab_off);
         else
            sprintf((DEFCHAR *)rsrvd,"%d",CURRENT_VIEW->tab_off);
         item_values[2].value = rsrvd;
         item_values[2].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_TABS:
         strcpy((DEFCHAR *)rsrvd,"");
         if (CURRENT_VIEW->tabsinc)
           {
            sprintf((DEFCHAR *)rsrvd,"INCR %d",CURRENT_VIEW->tabsinc);
           }
         else
           {
            for (i=0;i<CURRENT_VIEW->numtabs;i++)
               {
                sprintf((DEFCHAR *)num1,"%d ",CURRENT_VIEW->tabs[i]);
                strcat((DEFCHAR *)rsrvd,(DEFCHAR *)num1);
               }
            if (query_type == QUERY_QUERY
            ||  query_type == QUERY_STATUS)
               rsrvd[COLS-7] = '\0';
           }
         item_values[1].value = rsrvd;
         item_values[1].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_TABSIN:
         if (TABI_ONx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         sprintf((DEFCHAR *)num1,"%d",TABI_Nx);
         item_values[2].value = num1;
         item_values[2].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_TABSOUT:
         if (CURRENT_FILE->tabsout_on)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         sprintf((DEFCHAR *)num1,"%d",CURRENT_FILE->tabsout_num);
         item_values[2].value = num1;
         item_values[2].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_TERMINAL:
         item_values[1].value = term_name;
         item_values[1].len = strlen((DEFCHAR *)term_name);
         break;
    case ITEM_TOF:
         if (CURRENT_TOF)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_TOF_FUNCTION:
         if (FOCUS_TOF && CURRENT_VIEW->current_window != WINDOW_COMMAND)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_TOPEDGE_FUNCTION:
         if (batch_only)
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
            break;
           }
         getyx(CURRENT_WINDOW,y,x);
         if (CURRENT_VIEW->current_window == WINDOW_FILEAREA
         &&  y == 0)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_TYPEAHEAD:
         if (TYPEAHEADx)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_VERIFY:
         sprintf((DEFCHAR *)num3,"%d %d",CURRENT_VIEW->verify_start,CURRENT_VIEW->verify_end);
         item_values[1].value = num3;
         item_values[1].len = strlen((DEFCHAR *)num3);
         break;
    case ITEM_VERSHIFT:
         sprintf((DEFCHAR *)rsrvd,"%ld",(LINETYPE)CURRENT_VIEW->verify_col - (LINETYPE)CURRENT_VIEW->verify_start);
         item_values[1].value = rsrvd;
         item_values[1].len = strlen((DEFCHAR *)rsrvd);
         break;
    case ITEM_VERONE_FUNCTION:
         if (CURRENT_VIEW->verify_col == 1)
           {
            item_values[1].value = (CHARTYPE *)"1";
            item_values[1].len = 1;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"0";
            item_values[1].len = 1;
           }
         break;
    case ITEM_VERSION:
         item_values[1].value = (CHARTYPE *)"THE";
         item_values[1].len = 3;
         item_values[2].value = (CHARTYPE *)the_version;
         item_values[2].len = strlen((DEFCHAR *)the_version);
         item_values[3].value = (CHARTYPE *)"???";
#if defined(DOS)
# if defined(EMX)
         if (_osmode == DOS_MODE)
            item_values[3].value = (CHARTYPE *)"DOS";
         else
            item_values[3].value = (CHARTYPE *)"OS2";
#else
         item_values[3].value = (CHARTYPE *)"DOS";
# endif
#endif
#if defined(OS2)
# if defined(EMX)
         if (_osmode == DOS_MODE)
            item_values[3].value = (CHARTYPE *)"DOS";
         else
            item_values[3].value = (CHARTYPE *)"OS2";
#else
         item_values[3].value = (CHARTYPE *)"OS2";
# endif
#endif
#if defined(UNIX)
         item_values[3].value = (CHARTYPE *)"UNIX";
#endif
#if defined(XCURSES)
         item_values[3].value = (CHARTYPE *)"X11";
#endif
#if defined(MSWIN)
         item_values[3].value = (CHARTYPE *)"MS-WINDOWS";
#endif
#if defined(WIN32)
         item_values[3].value = (CHARTYPE *)"WIN32";
#endif
         item_values[3].len = strlen((DEFCHAR *)item_values[3].value);
         item_values[4].value = (CHARTYPE *)the_release;
         item_values[4].len = strlen((DEFCHAR *)item_values[4].value);
         break;
    case ITEM_WIDTH:
         sprintf((DEFCHAR *)num1,"%d",max_line_length);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         break;
    case ITEM_WORD:
         if (CURRENT_VIEW->word == 'A')
           {
            item_values[1].value = (CHARTYPE *)"ALPHANUM";
            item_values[1].len = 8;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"NONBLANK";
            item_values[1].len = 8;
           }
         break;
    case ITEM_WORDWRAP:
         if (CURRENT_VIEW->wordwrap)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_WRAP:
         if (CURRENT_VIEW->wrap)
           {
            item_values[1].value = (CHARTYPE *)"ON";
            item_values[1].len = 2;
           }
          else
           {
            item_values[1].value = (CHARTYPE *)"OFF";
            item_values[1].len = 3;
           }
         break;
    case ITEM_XTERMINAL:
         item_values[1].value = xterm_program;
         item_values[1].len = strlen((DEFCHAR *)xterm_program);
         break;
    case ITEM_ZONE:
         sprintf((DEFCHAR *)num1,"%d",CURRENT_VIEW->zone_start);
         item_values[1].value = num1;
         item_values[1].len = strlen((DEFCHAR *)num1);
         sprintf((DEFCHAR *)num2,"%d",CURRENT_VIEW->zone_end);
         item_values[2].value = num2;
         item_values[2].len = strlen((DEFCHAR *)num2);
         break;
    default:
         break;
   }
/*---------------------------------------------------------------------*/
/* If an "error" condition exists, do not set any values.              */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    switch(query_type)
      {
       case QUERY_EXTRACT:
       case QUERY_FUNCTION:
       case QUERY_READV:
            sprintf((DEFCHAR *)num0,"%d",number_variables);
            item_values[0].value = num0;
            item_values[0].len = strlen((DEFCHAR *)num0);
            break;
       case QUERY_STATUS:
       case QUERY_QUERY:
       case QUERY_MODIFY:
            item_values[0].value = query_item[itemno].name;
            item_values[0].len = strlen((DEFCHAR *)query_item[itemno].name);
            break;
       default:
            break;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_point(short itemno,CHARTYPE *params)
#else
static short extract_point(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern short compatible_feel;
/*--------------------------- local data ------------------------------*/
 register short i=0;
 short number_variables = query_item[itemno].number_values;
 static CHARTYPE num4[15];
 LINE *curr=NULL;
 LINETYPE true_line=0L;
 short rc=RC_OK;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_point");
#endif
 if (strcmp((DEFCHAR *)params,"") == 0) /* get name for focus line only */
   {
    true_line = (compatible_feel==COMPAT_XEDIT)?CURRENT_VIEW->current_line:get_true_line(TRUE);
    curr = lll_find(CURRENT_FILE->first_line,CURRENT_FILE->last_line,true_line,CURRENT_FILE->number_lines);
    if (curr->name == NULL)  /* line not named */
       number_variables = 0;
    else
      {
       sprintf((DEFCHAR *)num4,"%ld %s",true_line,curr->name);
       item_values[1].value = num4;
       item_values[1].len = strlen((DEFCHAR *)num4);
       number_variables = 1;
      }
   }
 else
   {
    if (strcmp((DEFCHAR *)params,"*") != 0)     /* argument must be "*" */
      {
       display_error(1,(CHARTYPE *)params,FALSE);
       number_variables = EXTRACT_ARG_ERROR;
      }
    else
      {
       curr = CURRENT_FILE->first_line;
       for(true_line=0,i=0;curr != NULL;true_line++)
         {
          if (curr->name != NULL)  /* line is named */
            {
             sprintf((DEFCHAR *)num4,"%ld %s",true_line,curr->name);
             rc = set_rexx_variable(query_item[itemno].name,num4,strlen((DEFCHAR *)num4),++i);
             if (rc == RC_SYSTEM_ERROR)
               {
                display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
                trace_return();
#endif
                return(EXTRACT_ARG_ERROR);
               }
            }
          curr = curr->next;
         }
       sprintf((DEFCHAR *)num4,"%d",i);
       rc = set_rexx_variable(query_item[itemno].name,num4,strlen((DEFCHAR *)num4),0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_pending(short itemno,CHARTYPE *params)
#else
static short extract_pending(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
#define PEN_PARAMS  4
 CHARTYPE *word[PEN_PARAMS+1];
 CHARTYPE strip[PEN_PARAMS];
 unsigned short num_params=0;
 short number_variables = query_item[itemno].number_values;
 bool find_block=FALSE;
 bool find_oldname=FALSE;
 bool valid_args = FALSE;
 CHARTYPE *name=NULL;
 static CHARTYPE num1[7];
 PPC *curr_ppc=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_pending");
#endif
 strip[0]=STRIP_BOTH;
 strip[1]=STRIP_BOTH;
 strip[2]=STRIP_BOTH;
 strip[4]=STRIP_BOTH;
 num_params = param_split(params,word,PEN_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* No parameters, error.                                               */
/*---------------------------------------------------------------------*/
    case 0:
           display_error(3,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 1  parameter, only name.                                            */
/*---------------------------------------------------------------------*/
    case 1:
           if (equal((CHARTYPE *)"block",word[0],5)
           ||  equal((CHARTYPE *)"oldname",word[0],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           name = word[0];
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, BLOCK and name or OLDNAME and name.                  */
/*---------------------------------------------------------------------*/
    case 2:
           if (equal((CHARTYPE *)"block",word[0],5)
           &&  equal((CHARTYPE *)"oldname",word[1],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((CHARTYPE *)"block",word[1],5)
           &&  equal((CHARTYPE *)"oldname",word[0],7))
             {
              display_error(3,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           if (equal((CHARTYPE *)"block",word[0],5))
             {
              name = word[1];
              find_block = TRUE;
              break;
             }
           if (equal((CHARTYPE *)"oldname",word[0],7))
             {
              name = word[1];
              find_oldname = TRUE;
              break;
             }
           display_error(1,word[0],FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 3  parameters, BLOCK and OLDNAME and name.                          */
/*---------------------------------------------------------------------*/
    case 3:
           if (equal((CHARTYPE *)"block",word[0],5)
           &&  equal((CHARTYPE *)"oldname",word[1],7))
              valid_args = TRUE;
           if (equal((CHARTYPE *)"block",word[1],5)
           &&  equal((CHARTYPE *)"oldname",word[0],7))
              valid_args = TRUE;
           if (!valid_args)
             {
              display_error(1,word[0],FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           find_block = find_oldname = TRUE;
           name = word[2];
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
/*---------------------------------------------------------------------*/
/* No pending prefix commands, return 0.                               */
/*---------------------------------------------------------------------*/
    if (CURRENT_FILE->first_ppc == NULL)
       number_variables = 0;
    else
/*---------------------------------------------------------------------*/
/* Have to search each pending prefix command...                       */
/*---------------------------------------------------------------------*/
      {
/*---------------------------------------------------------------------*/
/* If we are to look for OLDNAME, find a synonym for it if one exists..*/
/*---------------------------------------------------------------------*/
       if (find_oldname)
          name = find_prefix_oldname(name);
/*---------------------------------------------------------------------*/
/* For each pending prefix command...                                  */
/*---------------------------------------------------------------------*/
       curr_ppc = CURRENT_FILE->first_ppc;
       while(1)
         {
          if (curr_ppc == NULL)
             break;
/*---------------------------------------------------------------------*/
/* If we want to match on any name...                                  */
/*---------------------------------------------------------------------*/
          if (strcmp((DEFCHAR *)name,"*") == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we matching on any BLOCK command...                             */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (curr_ppc->ppc_block_command)
/*---------------------------------------------------------------------*/
/* We have found the first BLOCK command with any name.                */
/*---------------------------------------------------------------------*/
                   break;
                else
                  {
/*---------------------------------------------------------------------*/
/* Go back and look for another...                                     */
/*---------------------------------------------------------------------*/
                   curr_ppc = curr_ppc->next;
                   continue;
                  }
               }
             else
/*---------------------------------------------------------------------*/
/* We have found the first command with any name.                      */
/*---------------------------------------------------------------------*/
                break;
            }
/*---------------------------------------------------------------------*/
/* We want to find a specific command...                               */
/*---------------------------------------------------------------------*/
          if (memcmpi(curr_ppc->ppc_command,name,strlen((DEFCHAR *)name)) == 0)
            {
/*---------------------------------------------------------------------*/
/* Are we looking for a specific BLOCK command...                      */
/*---------------------------------------------------------------------*/
             if (find_block)
               {
                if (curr_ppc->ppc_block_command)
/*---------------------------------------------------------------------*/
/* We have found the first specific BLOCK command.                     */
/*---------------------------------------------------------------------*/
                   break;
               }
             else
/*---------------------------------------------------------------------*/
/* We have found the first specific command.                           */
/*---------------------------------------------------------------------*/
                break;
            }
          curr_ppc = curr_ppc->next;
         }
/*---------------------------------------------------------------------*/
/* Did we find a matching pending prefix command ?                     */
/*---------------------------------------------------------------------*/
       if (curr_ppc == NULL)
          number_variables = 0;
       else
         {
/*---------------------------------------------------------------------*/
/* Yes we did. Set all of the REXX variables to the correct values...  */
/*---------------------------------------------------------------------*/
          sprintf((DEFCHAR *)num1,"%ld",curr_ppc->ppc_line_number);
          item_values[1].value = num1;
          item_values[1].len = strlen((DEFCHAR *)num1);
          item_values[2].value = curr_ppc->ppc_command;
          item_values[2].len = strlen((DEFCHAR *)item_values[2].value);
          item_values[3].value = find_prefix_synonym(curr_ppc->ppc_command);
          item_values[3].len = strlen((DEFCHAR *)item_values[2].value);
          if (curr_ppc->ppc_block_command)
             item_values[4].value = (CHARTYPE *)"BLOCK";
          else
             item_values[4].value = (CHARTYPE *)"";
          item_values[4].len = strlen((DEFCHAR *)item_values[4].value);
         }
      }
   }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_prefix(short itemno,CHARTYPE *params)
#else
static short extract_prefix(itemno,params)
short itemno;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern LINE *first_prefix_synonym;
/*--------------------------- local data ------------------------------*/
 register short i=0;
#define PRE_PARAMS  3
 CHARTYPE *word[PRE_PARAMS+1];
 CHARTYPE strip[PRE_PARAMS];
 unsigned short num_params=0;
 short number_variables = query_item[itemno].number_values;
 static CHARTYPE num4[15];
 LINE *curr=NULL;
 short rc=RC_OK;
 CHARTYPE *tmpbuf=NULL;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_prefix");
#endif
 strip[0]=STRIP_BOTH;
 strip[1]=STRIP_BOTH;
 strip[2]=STRIP_BOTH;
 num_params = param_split(params,word,PRE_PARAMS,WORD_DELIMS,TEMP_PARAM,strip);
 switch(num_params)
   {
/*---------------------------------------------------------------------*/
/* None or 1 parameters, error.                                        */
/*---------------------------------------------------------------------*/
    case 0:
    case 1:
           display_error(3,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
/*---------------------------------------------------------------------*/
/* 2  parameters, Synonym and name.                                    */
/*---------------------------------------------------------------------*/
    case 2:
           if (!equal((CHARTYPE *)"synonym",word[0],1))
             {
              display_error(13,(CHARTYPE *)"",FALSE);
              number_variables = EXTRACT_ARG_ERROR;
              break;
             }
           break;
/*---------------------------------------------------------------------*/
/* Too many parameters, error.                                         */
/*---------------------------------------------------------------------*/
    default:
           display_error(2,(CHARTYPE *)"",FALSE);
           number_variables = EXTRACT_ARG_ERROR;
           break;
   }
/*---------------------------------------------------------------------*/
/* If the validation of parameters is successful...                    */
/*---------------------------------------------------------------------*/
 if (number_variables >= 0)
   {
    if (strcmp((DEFCHAR *)word[1],"*") == 0)
/*---------------------------------------------------------------------*/
/* Get oldname for all synonyms...                                     */
/*---------------------------------------------------------------------*/
      {
       curr = first_prefix_synonym;
       i = 0;
       while(curr != NULL)
         {
          tmpbuf = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(strlen((DEFCHAR *)curr->name)+strlen((DEFCHAR *)curr->line)+1));
          if (tmpbuf == (CHARTYPE *)NULL)
            {
             display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          strcpy((DEFCHAR *)tmpbuf,(DEFCHAR *)curr->name);
          strcat((DEFCHAR *)tmpbuf," ");
          strcat((DEFCHAR *)tmpbuf,(DEFCHAR *)curr->line);
          rc = set_rexx_variable(query_item[itemno].name,tmpbuf,strlen((DEFCHAR *)tmpbuf),++i);
          (*the_free)(tmpbuf);
          if (rc == RC_SYSTEM_ERROR)
            {
             display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
          curr = curr->next;
         }
       sprintf((DEFCHAR *)num4,"%d",i);
       rc = set_rexx_variable(query_item[itemno].name,num4,strlen((DEFCHAR *)num4),0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
          number_variables = EXTRACT_ARG_ERROR;
         }
       else
          number_variables = EXTRACT_VARIABLES_SET;
      }
    else
/*---------------------------------------------------------------------*/
/* Get oldname for named synonym...                                    */
/*---------------------------------------------------------------------*/
      {
       item_values[1].value = find_prefix_synonym(word[1]);
       item_values[1].len = strlen((DEFCHAR *)item_values[1].value);
       number_variables = 1;
      }
 }
#ifdef TRACE
 trace_return();
#endif
 return(number_variables);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_reserved(short itemno,CHARTYPE *params,CHARTYPE query_type)
#else
static short extract_reserved(itemno,params,query_type)
short itemno;
CHARTYPE *params;
CHARTYPE query_type;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
/*--------------------------- local data ------------------------------*/
 bool line_numbers_only=TRUE;
 static RESERVED *curr_rsrvd;
 CHARTYPE *tmpbuf=NULL,*attr_string=NULL;
 short number_variables=0;
 short rc=RC_OK;
 static CHARTYPE rsrvd[100];
 short y=0,x=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_reserved");
#endif
 if (query_type == QUERY_EXTRACT
 &&  strcmp((DEFCHAR *)params,"*") == 0)
    line_numbers_only = FALSE;

 curr_rsrvd = CURRENT_FILE->first_reserved;
 strcpy((DEFCHAR *)rsrvd,"");
 while(curr_rsrvd != NULL)
   {
    if (line_numbers_only)
      {
       y = strlen((DEFCHAR *)curr_rsrvd->spec) + 1;
       if ((x + y) > 80)
          break;
       strcat((DEFCHAR *)rsrvd,(DEFCHAR *)curr_rsrvd->spec);
       strcat((DEFCHAR *)rsrvd," ");
       x += y;
      }
    else
      {
       attr_string = get_colour_strings(curr_rsrvd->attr);
       if (attr_string == (CHARTYPE *)NULL)
         {
#ifdef TRACE
          trace_return();
#endif
          return(EXTRACT_ARG_ERROR);
         }
       tmpbuf = (CHARTYPE *)(*the_malloc)(sizeof(CHARTYPE)*(strlen((DEFCHAR *)attr_string)+strlen((DEFCHAR *)curr_rsrvd->line)+strlen((DEFCHAR *)curr_rsrvd->spec)+3));
       if (tmpbuf == (CHARTYPE *)NULL)
         {
          display_error(30,(CHARTYPE *)"",FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(EXTRACT_ARG_ERROR);
         }
       strcpy((DEFCHAR *)tmpbuf,(DEFCHAR *)curr_rsrvd->spec);
       strcat((DEFCHAR *)tmpbuf," ");
       strcat((DEFCHAR *)tmpbuf,(DEFCHAR *)attr_string);
       (*the_free)(attr_string);
       strcat((DEFCHAR *)tmpbuf,(DEFCHAR *)curr_rsrvd->line);
       rc = set_rexx_variable(query_item[itemno].name,tmpbuf,strlen((DEFCHAR *)tmpbuf),++number_variables);
       (*the_free)(tmpbuf);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
          trace_return();
#endif
          return(EXTRACT_ARG_ERROR);
         }
      }
    curr_rsrvd = curr_rsrvd->next;
   }
 if (line_numbers_only)
   {
    if (x == 0)
       number_variables = 0;
    else
      {
       number_variables = 1;
       item_values[1].value = rsrvd;
       item_values[1].len = strlen((DEFCHAR *)rsrvd);
      }
    rc = number_variables;
   }
 else
   {
    sprintf((DEFCHAR *)rsrvd,"%d",number_variables);
    rc = set_rexx_variable(query_item[itemno].name,rsrvd,strlen((DEFCHAR *)rsrvd),0);
    if (rc == RC_SYSTEM_ERROR)
      {
       display_error(54,(CHARTYPE *)"",FALSE);
       rc = EXTRACT_ARG_ERROR;
      }
    else
       rc = EXTRACT_VARIABLES_SET;
   }
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_ring(short itemno,CHARTYPE *buffer,CHARTYPE query_type)
#else
static short extract_ring(itemno,buffer,query_type)
short itemno;
CHARTYPE *buffer;
CHARTYPE query_type;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern VIEW_DETAILS *vd_first;
 extern CHARTYPE display_screens;
 extern CHARTYPE number_of_files;
 extern short compatible_look;
 extern short compatible_feel;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 VIEW_DETAILS *curr;
 FILE_DETAILS *first_view_file=NULL;
 bool process_view=FALSE;
 LINETYPE lineno=0L;
 LENGTHTYPE col=0;
 register int i=0,j=0;
 int offset = 0;
 short number_files=0;
 bool view_being_displayed=FALSE;
 CHARTYPE num1[5];
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_ring");
#endif
 if (compatible_feel == COMPAT_XEDIT)
    offset = 1;
 curr = vd_current;
 for (j=0;j<number_of_files;)
   {
    process_view = TRUE;
    if (curr->file_for_view->file_views > 1)
      {
       if (first_view_file == curr->file_for_view)
          process_view = FALSE;
       else
          first_view_file = curr->file_for_view;
      }
    if (process_view)
      {
       j++;
       view_being_displayed=FALSE;
       for (i=0;i<display_screens;i++)
         {
          if (SCREEN_VIEW(i) == curr)
            {
             view_being_displayed = TRUE;
             get_current_position(i,&lineno,&col);
            }
         }
       if (!view_being_displayed)
         {
          lineno = (curr->current_window==WINDOW_COMMAND)?curr->current_line:curr->focus_line;
          col = curr->current_column;
         }
       if (compatible_look == COMPAT_XEDIT)
          sprintf((DEFCHAR *)buffer,"%s%s Size=%ld Line=%ld Col=%d Alt=%d,%d",
                  curr->file_for_view->fpath,
                  curr->file_for_view->fname,
                  curr->file_for_view->number_lines,
                  lineno,col,
                  curr->file_for_view->autosave_alt,
                  curr->file_for_view->save_alt);
       else
          sprintf((DEFCHAR *)buffer,"%s%s Line=%ld Col=%d Size=%ld Alt=%d,%d",
                  curr->file_for_view->fpath,
                  curr->file_for_view->fname,
                  lineno,col,
                  curr->file_for_view->number_lines,
                  curr->file_for_view->autosave_alt,
                  curr->file_for_view->save_alt);
       if (query_type == QUERY_QUERY)
         {
          display_error(0,buffer,TRUE);
         }
       else
         {
          rc = set_rexx_variable(query_item[itemno].name,buffer,strlen((DEFCHAR *)buffer),++number_files+offset);
          if (rc == RC_SYSTEM_ERROR)
            {
             display_error(54,(CHARTYPE *)"",FALSE);
#ifdef TRACE
             trace_return();
#endif
             return(EXTRACT_ARG_ERROR);
            }
         }
      }
    curr = curr->next;
    if (curr == NULL)
       curr = vd_first;
   }

 if (query_type != QUERY_QUERY)
   {
    sprintf((DEFCHAR *)num1,"%d",number_files);
    rc = set_rexx_variable(query_item[itemno].name,num1,strlen((DEFCHAR *)num1),offset);
    if (rc == RC_SYSTEM_ERROR)
      {
       display_error(54,(CHARTYPE *)"",FALSE);
       rc = EXTRACT_ARG_ERROR;
      }
    else
       rc = EXTRACT_VARIABLES_SET;
    if (offset == 1)
      {
       sprintf((DEFCHAR *)num1,"%d",number_files+offset);
       rc = set_rexx_variable(query_item[itemno].name,num1,strlen((DEFCHAR *)num1),0);
       if (rc == RC_SYSTEM_ERROR)
         {
          display_error(54,(CHARTYPE *)"",FALSE);
          rc = EXTRACT_ARG_ERROR;
         }
       else
          rc = EXTRACT_VARIABLES_SET;
      }
   }
 else
    rc = EXTRACT_VARIABLES_SET;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
/***********************************************************************/
#ifdef HAVE_PROTO
static void get_etmode(CHARTYPE *onoff,CHARTYPE *list)
#else
static void get_etmode(onoff,list)
CHARTYPE *onoff,*list;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern bool   etmode_flag[256];
/*--------------------------- local data ------------------------------*/
 bool on_flag=FALSE;
 bool off_flag=FALSE;
 bool last_state=FALSE;
 char tmp[15];
 register int i=0;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   get_etmode");
#endif
 strcpy((DEFCHAR *)list,"");
 last_state = TRUE;
 for (i=0;i<256;i++)
   {
    if (etmode_flag[i] == TRUE)
       off_flag = TRUE;
    else
       on_flag = TRUE;
    if (last_state != etmode_flag[i])
      {
       if (last_state == FALSE)
         {
          sprintf(tmp,"%d",i-1);
          strcat((DEFCHAR *)list,tmp);
         }
       else
         {
          sprintf(tmp," %d-",i);
          strcat((DEFCHAR *)list,tmp);
         }
       last_state = etmode_flag[i];
      }
   }
 if (*(list+strlen((DEFCHAR *)list)-1) == '-')
   {
    sprintf(tmp,"%d",255);
    strcat((DEFCHAR *)list,tmp);
   }
 if (on_flag)
   {
    strcpy((DEFCHAR *)onoff,"ON");
    if (!off_flag)
       strcpy((DEFCHAR *)list,"");
   }
 else
   {
    strcpy((DEFCHAR *)onoff,"OFF");
    strcpy((DEFCHAR *)list,"");
   }
#ifdef TRACE
 trace_return();
#endif
 return;
}
/***********************************************************************/
#ifdef HAVE_PROTO
static short extract_colour(short itemno,CHARTYPE *buffer,CHARTYPE query_type,CHARTYPE *params)
#else
static short extract_colour(itemno,buffer,query_type,params)
short itemno;
CHARTYPE *buffer;
CHARTYPE query_type;
CHARTYPE *params;
#endif
/***********************************************************************/
{
/*-------------------------- external data ----------------------------*/
 extern AREAS valid_areas[ATTR_MAX];
 extern CHARTYPE *trec;
 extern short terminal_lines;
/*--------------------------- local data ------------------------------*/
 short rc=RC_OK;
 register int i=0,j=0;
 int start=0,end=0,number_variables=0,off=0;
 bool found=FALSE;
 CHARTYPE *attr_string=NULL;
 CHARTYPE num1[15];
 CHARTYPE save_msgline_base = CURRENT_VIEW->msgline_base;
 short save_msgline_off = CURRENT_VIEW->msgline_off;
 ROWTYPE save_msgline_rows = CURRENT_VIEW->msgline_rows;
 bool save_msgmode_status = CURRENT_VIEW->msgmode_status;
/*--------------------------- processing ------------------------------*/
#ifdef TRACE
 trace_function("query.c:   extract_colour");
#endif

 if (blank_field(params)
 ||  strcmp((DEFCHAR*)params,"*") == 0)
   {
    start = 0;
    end = ATTR_MAX;
   }
 else
   {
    for (i=0;i<ATTR_MAX;i++)
      {
       if (equal(valid_areas[i].area,params,valid_areas[i].area_min_len))
         {
          found = TRUE;
          break;
         }
      }
    if (!found)
      {
       display_error(1,params,FALSE);
#ifdef TRACE
       trace_return();
#endif
       return(EXTRACT_ARG_ERROR);
      }
    start = i;
    end = i+1;
   }
 if (query_type == QUERY_QUERY)
   {
    CURRENT_VIEW->msgline_base   = POSITION_TOP;
    CURRENT_VIEW->msgline_off    = 1;
    CURRENT_VIEW->msgline_rows   = min(terminal_lines-1,end-start);
    CURRENT_VIEW->msgmode_status = TRUE;
   }

 for (i=start;i<end;i++)
   {
    attr_string = get_colour_strings(CURRENT_FILE->attr+i);
    if (attr_string == (CHARTYPE *)NULL)
      {
#ifdef TRACE
       trace_return();
#endif
       return(EXTRACT_ARG_ERROR);
      }
    sprintf((DEFCHAR *)buffer,"%s%s %s",
            (query_type == QUERY_QUERY) ? "color " : "",
            valid_areas[i].area,
            attr_string);
    (*the_free)(attr_string);

    if (query_type == QUERY_QUERY)
       display_error(0,buffer,TRUE);
    else
      {
       number_variables++;
       item_values[number_variables].len = strlen((DEFCHAR *)buffer);
       memcpy((DEFCHAR*)trec+off,(DEFCHAR*)buffer,(item_values[number_variables].len)+1);
       item_values[number_variables].value = trec+off;
       off += (item_values[number_variables].len)+1;
      }
   }
 if (query_type == QUERY_QUERY)
   {
    CURRENT_VIEW->msgline_base   = save_msgline_base;
    CURRENT_VIEW->msgline_off    = save_msgline_off;
    CURRENT_VIEW->msgline_rows   = save_msgline_rows;
    CURRENT_VIEW->msgmode_status = save_msgmode_status;
    rc = EXTRACT_VARIABLES_SET;
   }
 else
    rc = number_variables;
#ifdef TRACE
 trace_return();
#endif
 return(rc);
}
