/* sane - Scanner Access Now Easy.
   Copyright (C) 1997 David Mosberger-Tang
   This file is part of the SANE package.

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of the
   License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

   As a special exception, the authors of SANE give permission for
   additional uses of the libraries contained in this release of SANE.

   The exception is that, if you link a SANE library with other files
   to produce an executable, this does not by itself cause the
   resulting executable to be covered by the GNU General Public
   License.  Your use of that executable is in no way restricted on
   account of linking the SANE library code into it.

   This exception does not, however, invalidate any other reasons why
   the executable file might be covered by the GNU General Public
   License.

   If you submit changes to SANE to the maintainers to be included in
   a subsequent release, you agree by submitting the changes that
   those changes may be distributed with this exception intact.

   If you write modifications of your own for SANE, it is your choice
   whether to permit this exception to apply to your modifications.
   If you do not wish that, delete this exception notice.

   This file implements a SANE backend for the Connectix QuickCam.  At
   present, only the color camera is supported though the driver
   should be able to easily accommodate black and white cameras.

   Portions of this code are derived from hpscanpbm-0.3a.  It's
   copyright notice is reproduced here:

   hpscanpbm
   User-level SCSI HP Scanjet driver
   Written by David Etherton, etherton@netcom.com
   HPUX Port by John Fehr, fehr@ninja.aes.mb.doe.ca
   Little hacks by Steve Enns  ennss@nhrisv.nhrc.sk.doe.ca
   Copyright 1995, David Etherton
   Released under the terms of the GPL.
   *NO WARRANTY*   */

#include <sane/config.h>

#include <limits.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>

#include <sane/sane.h>
#include <sane/saneopts.h>
#include <sane/sanei_scsi.h>
#include <hp.h>

#define BACKEND_NAME	hp
#include <sane/sanei_backend.h>

#ifndef PATH_MAX
# define PATH_MAX	1024
#endif

#include <sane/sanei_config.h>
#define HP_CONFIG_FILE "hp.conf"

#define MM_PER_INCH		25.4

#define PREFER_DEVICEPIXEL

#ifdef PREFER_DEVICEPIXEL
# define PIXELS_PER_MM		(300.0 / MM_PER_INCH)
#else
# define DECIPOINTS_PER_MM	(720.0 / MM_PER_INCH)
#endif

static int num_devices;
static HP_Device *first_dev;
static HP_Scanner *first_handle;

static const SANE_String_Const mode_list[] =
  {
    "Lineart", "Halftone", "Gray", "Color",
    0
  };

#define INQ_LEN	255
static const u_int8_t inquiry[] =
  {
    0x12, 0x00, 0x00, 0x00, INQ_LEN, 0x00
  };

static const u_int8_t test_unit_ready[] =
  {
    0x00, 0x00, 0x00, 0x00, 0x00, 0x00
  };

static size_t
max_string_size (const SANE_String_Const strings[])
{
  size_t size, max_size = 0;
  int i;

  for (i = 0; strings[i]; ++i)
    {
      size = strlen (strings[i]) + 1;
      if (size > max_size)
	max_size = size;
    }
  return max_size;
}

static SANE_Status
read_data (int fd, u_char * dest, size_t *len)
{
  static u_char read_6[6];

  memset (read_6, 0, sizeof (read_6));
  read_6[0] = 0x08;
  read_6[2] = *len >> 16;
  read_6[3] = *len >> 8;
  read_6[4] = *len;
  return sanei_scsi_cmd (fd, read_6, sizeof (read_6), dest, len);
}

static SANE_Status
write_data (int fd, void *data, size_t len)
{
  char write_6[4096];

  memset (write_6, 0, 6);
  write_6[0] = 0xa;
  write_6[2] = len >> 16;
  write_6[3] = len >> 8;
  write_6[4] = len;
  memcpy (write_6 + 6, data, len);

  return sanei_scsi_cmd (fd, write_6, 6 + len, 0, 0);
}

static SANE_Status
write_string (int fd, const char *format, ...)
{
  va_list args;
  char buf[1024];

  va_start (args, format);
  vsprintf (buf, format, args);
  return write_data (fd, buf, strlen (buf));
}

static SANE_Status
inquire (int fd, int request_code, char request_letter,
	 char reply_letter, int *valp)
{
  u_char buf[128], letter;
  SANE_Status status;
  int code;
  size_t len;

  *valp = 0;

  sprintf ((char *) buf, "\033*s%d%c", request_code, request_letter);
  write_string (fd, (char *) buf);
  len = sizeof (buf);
  status = read_data (fd, buf, &len);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "inquire: read_data failed (%s)\n", sane_strstatus (status));
      return status;
    }

  if (sscanf ((char *) buf, "\033*s%d%c%dV", &code, &letter, valp) != 3)
    {
      DBG(1, "inquire: returned string is malformed (%s)\n", buf);
      return SANE_STATUS_IO_ERROR;
    }
  if (code != request_code || letter != reply_letter)
    {
      DBG(1, "inquire: inquired %d/%c but received %d/%c\n",
	  request_code, reply_letter, code, letter);
      return SANE_STATUS_IO_ERROR;
    }
  return SANE_STATUS_GOOD;
}

static SANE_Status
attach (const char *devname, HP_Device **devp)
{
  char result[INQ_LEN], *str, *t;
  int i, fd, hp_scanner;
  SANE_Status status;
  HP_Device *dev;
  size_t size;
  struct
    {
      int code;
      char request_letter;
      char reply_letter;
      int default_value;
      int value;
    }
  inq_params[] =
    {
#ifdef PREFER_DEVICEPIXEL
      {10489, 'L', 'k',     0, 0},	/* min_x */
      {10489, 'H', 'g',  2549, 0},	/* max_x */
      {10490, 'L', 'k',     0, 0},	/* min_y */
      {10490, 'H', 'g',  3507, 0},	/* max_y */
      {10323, 'L', 'k',    12, 0},	/* min_dpi */
      {10323, 'H', 'g',  1600, 0},	/* max_dpi */
#else
      {10329, 'L', 'k',     0, 0},	/* min_x */
      {10329, 'H', 'g',  6118, 0},	/* max_x */
      {10330, 'L', 'k',     0, 0},	/* min_y */
      {10330, 'H', 'g', 10078, 0},	/* max_y */
      {10323, 'L', 'k',    38, 0},	/* min_dpi */
      {10323, 'H', 'g',   600, 0},	/* max_dpi */
#endif
      {10317, 'L', 'k',     0, 0},	/* min_brightness */
      {10317, 'H', 'g',     0, 0},	/* max_brightness */
      {10316, 'L', 'k',     0, 0},	/* min_contrast */
      {10316, 'H', 'g',     0, 0},	/* max_contrast */
    };

  for (dev = first_dev; dev; dev = dev->next)
    if (strcmp (dev->sane.name, devname) == 0)
      {
	if (devp)
	  *devp = dev;
	return SANE_STATUS_GOOD;
      }

  DBG(3, "attach: opening %s\n", devname);
  status = sanei_scsi_open (devname, &fd, 0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "attach: open failed (%s)\n", sane_strstatus (status));
      return SANE_STATUS_INVAL;
    }

  DBG(3, "attach: sending INQUIRY\n");
  size = sizeof (result);
  status = sanei_scsi_cmd (fd, inquiry, sizeof (inquiry), result, &size);
  if (status != SANE_STATUS_GOOD || size < 16)
    {
      DBG(1, "attach: inquiry failed (%s)\n", sane_strstatus (status));
      sanei_scsi_close (fd);
      return status;
    }

  DBG(3, "attach: sending TEST_UNIT_READY\n");
  status = sanei_scsi_cmd (fd, test_unit_ready, sizeof (test_unit_ready),
			   0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "attach: test unit ready failed (%s)\n",
	  sane_strstatus (status));
      sanei_scsi_close (fd);
      return status;
    }

  for (i = 0; i < NELEMS (inq_params); ++i)
    if (inquire (fd, inq_params[i].code, inq_params[i].request_letter,
		 inq_params[i].reply_letter, &inq_params[i].value)
	!= SANE_STATUS_GOOD)
    {
      inq_params[i].value = inq_params[i].default_value;
      DBG(1, "using default value of %d\n", inq_params[i].value);
    }

  sanei_scsi_close (fd);

  /* are we really dealing with a scanner by HP? */
  hp_scanner = ((result[0] == 0x03 || result[0] == 0x06)
		&& strncmp (result + 8, "HP", 2) == 0);
  if (!hp_scanner)
    {
      DBG(1, "attach: device doesn't look like a HP scanner\n");
      return SANE_STATUS_INVAL;
    }

  dev = malloc (sizeof (*dev));
  if (!dev)
    return SANE_STATUS_NO_MEM;

  memset (dev, 0, sizeof (*dev));
#ifdef PREFER_DEVICEPIXEL
  /* Converts pixels to mm, for minimum change.  */
  dev->x_range.min = SANE_FIX (inq_params[0].value / PIXELS_PER_MM);
  dev->x_range.max = SANE_FIX (inq_params[1].value / PIXELS_PER_MM);
  dev->x_range.quant = 0;

  dev->y_range.min = SANE_FIX (inq_params[2].value / PIXELS_PER_MM);
  dev->y_range.max = SANE_FIX (inq_params[3].value / PIXELS_PER_MM);
  dev->y_range.quant = 0;
#else
  dev->x_range.min = SANE_FIX (inq_params[0].value / DECIPOINTS_PER_MM);
  dev->x_range.max = SANE_FIX (inq_params[1].value / DECIPOINTS_PER_MM);
  dev->x_range.quant = 0;

  dev->y_range.min = SANE_FIX (inq_params[2].value / DECIPOINTS_PER_MM);
  dev->y_range.max = SANE_FIX (inq_params[3].value / DECIPOINTS_PER_MM);
  dev->y_range.quant = 0;
#endif

  dev->dpi_range.min = inq_params[4].value;
  dev->dpi_range.max = inq_params[5].value;
  dev->dpi_range.quant = 1;

  dev->brightness_range.min   = inq_params[6].value;
  dev->brightness_range.max   = inq_params[7].value;
  dev->brightness_range.quant = 1;

  dev->contrast_range.min   = inq_params[8].value;
  dev->contrast_range.max   = inq_params[9].value;
  dev->contrast_range.quant = 1;

  str = malloc (17);
  memcpy (str, result + 16, 16);
  str[16] = ' ';
  t = strchr (str, ' ');
  *t = '\0';

  dev->sane.name   = strdup (devname);
  dev->sane.vendor = "HP";
  dev->sane.model  = str;
  dev->sane.type   = "flatbed scanner";

  DBG(1, "attach: found HP ScanJet model %s, x=%g-%gmm, y=%g-%gmm, "
      "resolution=%d-%ddpi\n", dev->sane.model,
      SANE_UNFIX (dev->x_range.min), SANE_UNFIX (dev->x_range.max),
      SANE_UNFIX (dev->y_range.min), SANE_UNFIX (dev->y_range.max),
      dev->dpi_range.min, dev->dpi_range.max);

  ++num_devices;
  dev->next = first_dev;
  first_dev = dev;

  if (devp)
    *devp = dev;
  return SANE_STATUS_GOOD;
}

static SANE_Status
init_options (HP_Scanner *s)
{
  int i;

  memset (s->opt, 0, sizeof (s->opt));
  memset (s->val, 0, sizeof (s->val));

  for (i = 0; i < NUM_OPTIONS; ++i)
    {
      s->opt[i].size = sizeof (SANE_Word);
      s->opt[i].cap = SANE_CAP_SOFT_SELECT | SANE_CAP_SOFT_DETECT;
    }

  s->opt[OPT_NUM_OPTS].title = SANE_TITLE_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].desc = SANE_DESC_NUM_OPTIONS;
  s->opt[OPT_NUM_OPTS].cap = SANE_CAP_SOFT_DETECT;
  s->val[OPT_NUM_OPTS].w = NUM_OPTIONS;

  /* "Mode" group: */

  s->opt[OPT_MODE_GROUP].title = "Scan Mode";
  s->opt[OPT_MODE_GROUP].desc = "";
  s->opt[OPT_MODE_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_MODE_GROUP].cap = 0;
  s->opt[OPT_MODE_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* scan mode */
  s->opt[OPT_MODE].name = SANE_NAME_SCAN_MODE;
  s->opt[OPT_MODE].title = SANE_TITLE_SCAN_MODE;
  s->opt[OPT_MODE].desc = SANE_DESC_SCAN_MODE;
  s->opt[OPT_MODE].type = SANE_TYPE_STRING;
  s->opt[OPT_MODE].size = max_string_size (mode_list);
  s->opt[OPT_MODE].constraint_type = SANE_CONSTRAINT_STRING_LIST;
  s->opt[OPT_MODE].constraint.string_list = mode_list;
  s->val[OPT_MODE].s = strdup (mode_list[2]);

  /* resolution */
  s->opt[OPT_RESOLUTION].name = SANE_NAME_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].title = SANE_TITLE_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].desc = SANE_DESC_SCAN_RESOLUTION;
  s->opt[OPT_RESOLUTION].type = SANE_TYPE_INT;
  s->opt[OPT_RESOLUTION].unit = SANE_UNIT_DPI;
  s->opt[OPT_RESOLUTION].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_RESOLUTION].constraint.range = &s->hw->dpi_range;
  s->val[OPT_RESOLUTION].w = 18;

  /* "Geometry" group: */

  s->opt[OPT_GEOMETRY_GROUP].title = "Geometry";
  s->opt[OPT_GEOMETRY_GROUP].desc = "";
  s->opt[OPT_GEOMETRY_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_GEOMETRY_GROUP].cap = SANE_CAP_ADVANCED;
  s->opt[OPT_GEOMETRY_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* top-left x */
  s->opt[OPT_TL_X].name = SANE_NAME_SCAN_TL_X;
  s->opt[OPT_TL_X].title = SANE_TITLE_SCAN_TL_X;
  s->opt[OPT_TL_X].desc = SANE_DESC_SCAN_TL_X;
  s->opt[OPT_TL_X].type = SANE_TYPE_FIXED;
  s->opt[OPT_TL_X].unit = SANE_UNIT_MM;
  s->opt[OPT_TL_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_X].constraint.range = &s->hw->x_range;
  s->val[OPT_TL_X].w = 0;

  /* top-left y */
  s->opt[OPT_TL_Y].name = SANE_NAME_SCAN_TL_Y;
  s->opt[OPT_TL_Y].title = SANE_TITLE_SCAN_TL_Y;
  s->opt[OPT_TL_Y].desc = SANE_DESC_SCAN_TL_Y;
  s->opt[OPT_TL_Y].type = SANE_TYPE_FIXED;
  s->opt[OPT_TL_Y].unit = SANE_UNIT_MM;
  s->opt[OPT_TL_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_TL_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_TL_Y].w = 0;

  /* bottom-right x */
  s->opt[OPT_BR_X].name = SANE_NAME_SCAN_BR_X;
  s->opt[OPT_BR_X].title = SANE_TITLE_SCAN_BR_X;
  s->opt[OPT_BR_X].desc = SANE_DESC_SCAN_BR_X;
  s->opt[OPT_BR_X].type = SANE_TYPE_FIXED;
  s->opt[OPT_BR_X].unit = SANE_UNIT_MM;
  s->opt[OPT_BR_X].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_X].constraint.range = &s->hw->x_range;
  s->val[OPT_BR_X].w = s->hw->x_range.max;

  /* bottom-right y */
  s->opt[OPT_BR_Y].name = SANE_NAME_SCAN_BR_Y;
  s->opt[OPT_BR_Y].title = SANE_TITLE_SCAN_BR_Y;
  s->opt[OPT_BR_Y].desc = SANE_DESC_SCAN_BR_Y;
  s->opt[OPT_BR_Y].type = SANE_TYPE_FIXED;
  s->opt[OPT_BR_Y].unit = SANE_UNIT_MM;
  s->opt[OPT_BR_Y].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BR_Y].constraint.range = &s->hw->y_range;
  s->val[OPT_BR_Y].w = s->hw->y_range.max;

  /* "Enhancement" group: */

  s->opt[OPT_ENHANCEMENT_GROUP].title = "Enhancement";
  s->opt[OPT_ENHANCEMENT_GROUP].desc = "";
  s->opt[OPT_ENHANCEMENT_GROUP].type = SANE_TYPE_GROUP;
  s->opt[OPT_ENHANCEMENT_GROUP].cap = 0;
  s->opt[OPT_ENHANCEMENT_GROUP].constraint_type = SANE_CONSTRAINT_NONE;

  /* brightness */
  s->opt[OPT_BRIGHTNESS].name = SANE_NAME_BRIGHTNESS;
  s->opt[OPT_BRIGHTNESS].title = SANE_TITLE_BRIGHTNESS;
  s->opt[OPT_BRIGHTNESS].desc = SANE_DESC_BRIGHTNESS;
  s->opt[OPT_BRIGHTNESS].type = SANE_TYPE_INT;
  s->opt[OPT_BRIGHTNESS].unit = SANE_UNIT_NONE;
  s->opt[OPT_BRIGHTNESS].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_BRIGHTNESS].constraint.range = &s->hw->brightness_range;
  s->val[OPT_BRIGHTNESS].w = 0;

  /* contrast */
  s->opt[OPT_CONTRAST].name = SANE_NAME_CONTRAST;
  s->opt[OPT_CONTRAST].title = SANE_TITLE_CONTRAST;
  s->opt[OPT_CONTRAST].desc = SANE_DESC_CONTRAST;
  s->opt[OPT_CONTRAST].type = SANE_TYPE_INT;
  s->opt[OPT_CONTRAST].unit = SANE_UNIT_NONE;
  s->opt[OPT_CONTRAST].constraint_type = SANE_CONSTRAINT_RANGE;
  s->opt[OPT_CONTRAST].constraint.range = &s->hw->contrast_range;
  s->val[OPT_CONTRAST].w = 0;
  if (s->hw->contrast_range.min == s->hw->contrast_range.max)
    /* plain ScanJet doesn't support contrast */
    s->opt[OPT_CONTRAST].cap |= SANE_CAP_INACTIVE;
  return SANE_STATUS_GOOD;
}

static SANE_Status
do_cancel (HP_Scanner *s)
{
  s->scanning = SANE_FALSE;

  if (s->fd >= 0)
    {
      sanei_scsi_close (s->fd);
      s->fd = -1;
    }
  return SANE_STATUS_CANCELLED;
}

SANE_Status
sane_init (SANE_Int *version_code, SANE_Auth_Callback authorize)
{
  char dev_name[PATH_MAX];
  size_t len;
  FILE *fp;

  DBG_INIT();

  if (version_code)
    *version_code = SANE_VERSION_CODE (V_MAJOR, V_MINOR, 0);

  fp = sanei_config_open (HP_CONFIG_FILE);
  if (!fp)
    {
      /* default to /dev/scanner instead of insisting on config file */
      attach ("/dev/scanner", 0);
      return SANE_STATUS_GOOD;
    }

  while (fgets (dev_name, sizeof (dev_name), fp))
    {
      if (dev_name[0] == '#')		/* ignore line comments */
	continue;
      len = strlen (dev_name);
      if (dev_name[len - 1] == '\n')
	dev_name[--len] = '\0';

      if (!len)
	continue;			/* ignore empty lines */

      attach (dev_name, 0);
    }
  fclose (fp);
  return SANE_STATUS_GOOD;
}

void
sane_exit (void)
{
  HP_Device *dev, *next;

  for (dev = first_dev; dev; dev = next)
    {
      next = dev->next;
      free ((void *) dev->sane.name);
      free ((void *) dev->sane.model);
      free (dev);
    }
}

SANE_Status
sane_get_devices (const SANE_Device ***device_list, SANE_Bool local_only)
{
  static const SANE_Device **devlist = 0;
  HP_Device *dev;
  int i;

  if (devlist)
    free (devlist);

  devlist = malloc ((num_devices + 1) * sizeof (devlist[0]));
  if (!devlist)
    return SANE_STATUS_NO_MEM;

  i = 0;
  for (dev = first_dev; i < num_devices; dev = dev->next)
    devlist[i++] = &dev->sane;
  devlist[i++] = 0;

  *device_list = devlist;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_open (SANE_String_Const devicename, SANE_Handle *handle)
{
  SANE_Status status;
  HP_Device *dev;
  HP_Scanner *s;

  if (devicename[0])
    {
      for (dev = first_dev; dev; dev = dev->next)
	if (strcmp (dev->sane.name, devicename) == 0)
	  break;

      if (!dev)
	{
	  status = attach (devicename, &dev);
	  if (status != SANE_STATUS_GOOD)
	    return status;
	}
    }
  else
    /* empty devicname -> use first device */
    dev = first_dev;

  if (!dev)
    return SANE_STATUS_INVAL;

  s = malloc (sizeof (*s));
  if (!s)
    return SANE_STATUS_NO_MEM;
  memset (s, 0, sizeof (*s));
  s->fd = -1;
  s->hw = dev;

  init_options (s);

  /* insert newly opened handle into list of open handles: */
  s->next = first_handle;
  first_handle = s;

  *handle = s;
  return SANE_STATUS_GOOD;
}

void
sane_close (SANE_Handle handle)
{
  HP_Scanner *prev, *s;

  /* remove handle from list of open handles: */
  prev = 0;
  for (s = first_handle; s; s = s->next)
    {
      if (s == handle)
	break;
      prev = s;
    }
  if (!s)
    {
      DBG(1, "close: invalid handle %p\n", handle);
      return;		/* oops, not a handle we know about */
    }

  if (s->scanning)
    do_cancel (handle);

  if (prev)
    prev->next = s->next;
  else
    first_handle = s->next;

  free (handle);
}

const SANE_Option_Descriptor *
sane_get_option_descriptor (SANE_Handle handle, SANE_Int option)
{
  HP_Scanner *s = handle;

  if ((unsigned) option >= NUM_OPTIONS)
    return 0;
  return s->opt + option;
}

SANE_Status
sane_control_option (SANE_Handle handle, SANE_Int option,
		     SANE_Action action, void *val, SANE_Int *info)
{
  HP_Scanner *s = handle;
  SANE_Status status;
  SANE_Word cap;

  if (info)
    *info = 0;

  if (s->scanning)
    return SANE_STATUS_DEVICE_BUSY;

  if (option >= NUM_OPTIONS)
    return SANE_STATUS_INVAL;

  cap = s->opt[option].cap;

  if (!SANE_OPTION_IS_ACTIVE (cap))
    return SANE_STATUS_INVAL;

  if (action == SANE_ACTION_GET_VALUE)
    {
      switch (option)
	{
	  /* word options: */
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	case OPT_BR_X:
	case OPT_BR_Y:
	case OPT_NUM_OPTS:
	case OPT_BRIGHTNESS:
	case OPT_CONTRAST:
	  *(SANE_Word *) val = s->val[option].w;
	  return SANE_STATUS_GOOD;

	  /* string options: */
	case OPT_MODE:
	  strcpy (val, s->val[option].s);
	  return SANE_STATUS_GOOD;
	}
    }
  else if (action == SANE_ACTION_SET_VALUE)
    {
      if (!SANE_OPTION_IS_SETTABLE (cap))
	return SANE_STATUS_INVAL;

      status = sanei_constrain_value (s->opt + option, val, info);
      if (status != SANE_STATUS_GOOD)
	return status;

      switch (option)
	{
	  /* (mostly) side-effect-free word options: */
	case OPT_RESOLUTION:
	case OPT_TL_X:
	case OPT_TL_Y:
	case OPT_BR_X:
	case OPT_BR_Y:
	  if (info && s->val[option].w != *(SANE_Word *) val)
	    *info |= SANE_INFO_RELOAD_PARAMS;
	  /* fall through */
	case OPT_NUM_OPTS:
	case OPT_BRIGHTNESS:
	case OPT_CONTRAST:
	  s->val[option].w = *(SANE_Word *) val;
	  return SANE_STATUS_GOOD;

	case OPT_MODE:
	  if (s->val[option].s)
	    free (s->val[option].s);
	  s->val[option].s = strdup (val);
	  return SANE_STATUS_GOOD;
	}
    }
  return SANE_STATUS_INVAL;
}

SANE_Status
sane_get_parameters (SANE_Handle handle, SANE_Parameters *params)
{
  HP_Scanner *s = handle;

  if (!s->scanning)
    {
      double width, height, dpi;
      const char *mode;

      memset (&s->params, 0, sizeof (s->params));

      width  = SANE_UNFIX (s->val[OPT_BR_X].w - s->val[OPT_TL_X].w);
      height = SANE_UNFIX (s->val[OPT_BR_Y].w - s->val[OPT_TL_Y].w);
      dpi    = s->val[OPT_RESOLUTION].w;

      /* make best-effort guess at what parameters will look like once
         scanning starts.  */
      if (dpi > 0.0 && width > 0.0 && height > 0.0)
	{
	  double dots_per_mm = dpi / MM_PER_INCH;

	  s->params.pixels_per_line = width * dots_per_mm;
	  s->params.lines = height * dots_per_mm;
	}

      mode = s->val[OPT_MODE].s;
      if (strcmp (mode, "Lineart") == 0 || strcmp (mode, "Halftone") == 0)
	{
	  s->params.format = SANE_FRAME_GRAY;
	  s->params.bytes_per_line = (s->params.pixels_per_line + 7) / 8;
	  s->params.depth = 1;
	}
      else if (strcmp (mode, "Gray") == 0)
	{
	  s->params.format = SANE_FRAME_GRAY;
	  s->params.bytes_per_line = s->params.pixels_per_line;
	  s->params.depth = 8;
	}
      else
	{
	  /* XXX assume single pass--are there any one-pass HPs? */
	  s->params.format = SANE_FRAME_RGB;
	  s->params.bytes_per_line = 3 * s->params.pixels_per_line;
	  s->params.depth = 8;
	}
      s->params.last_frame = SANE_TRUE;
    }
  if (params)
    *params = s->params;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_start (SANE_Handle handle)
{
  int mode, x0, x1, y0, y1;
  HP_Scanner *s = handle;
  SANE_Status status;

  /* First make sure we have a current parameter set.  Some of the
     parameters will be overwritten below, but that's OK.  */
  status = sane_get_parameters (s, 0);
  if (status != SANE_STATUS_GOOD)
    return status;

  status = sanei_scsi_open (s->hw->sane.name, &s->fd, 0, 0);
  if (status != SANE_STATUS_GOOD)
    {
      DBG(1, "open of %s failed: %s\n",
	  s->hw->sane.name, sane_strstatus (status));
      return status;
    }

  /* reset scanner; returns all parameters to defaults */
  write_string (s->fd, "\033E");

  /* x and y dpi: */
  write_string (s->fd, "\033*a%dR", s->val[OPT_RESOLUTION].w);
  write_string (s->fd, "\033*a%dS", s->val[OPT_RESOLUTION].w);

#ifdef PREFER_DEVICEPIXEL
  /* set scan extents, in 1/300'ths of an inch */
  x0 = SANE_UNFIX (s->val[OPT_TL_X].w * PIXELS_PER_MM);
  y0 = SANE_UNFIX (s->val[OPT_TL_Y].w * PIXELS_PER_MM);
  x1 = SANE_UNFIX (s->val[OPT_BR_X].w * PIXELS_PER_MM);
  y1 = SANE_UNFIX (s->val[OPT_BR_Y].w * PIXELS_PER_MM);
  write_string (s->fd, "\033*f%dX", x0);
  write_string (s->fd, "\033*f%dY", y0);
  write_string (s->fd, "\033*f%dP", x1 - x0 + 1);
  write_string (s->fd, "\033*f%dQ", y1 - y0 + 1);
#else
  /* set scan extents, in 1/720'ths of an inch */
  x0 = SANE_UNFIX (s->val[OPT_TL_X].w * DECIPOINTS_PER_MM);
  y0 = SANE_UNFIX (s->val[OPT_TL_Y].w * DECIPOINTS_PER_MM);
  x1 = SANE_UNFIX (s->val[OPT_BR_X].w * DECIPOINTS_PER_MM);
  y1 = SANE_UNFIX (s->val[OPT_BR_Y].w * DECIPOINTS_PER_MM);
  write_string (s->fd, "\033*a%dX", x0);
  write_string (s->fd, "\033*a%dY", y0);
  write_string (s->fd, "\033*a%dP", x1 - x0 + 1);
  write_string (s->fd, "\033*a%dQ", y1 - y0 + 1);
#endif

  /* (the original scanjet only allows -1, 0, or 1) */
  write_string (s->fd, "\033*a%dL", s->val[OPT_BRIGHTNESS].w);
  write_string (s->fd, "\033*a%dK", s->val[OPT_CONTRAST].w);

  /* grayscale and color modes seems to need to have their outputs
     inverted; thresholded and dithered modes do not (the default) */
  if (strcmp (s->val[OPT_MODE].s, "Gray") == 0
      || strcmp (s->val[OPT_MODE].s, "Color") == 0)
    write_string (s->fd, "\033*a1I");

  mode = 4;
  /* data format is thresholded by default; pixel format is 8
     pixels/byte */
  if (strcmp (s->val[OPT_MODE].s, "Halftone") == 0)
    /* data format: dithered */
    write_string (s->fd, "\033*a3T");
  else if (strcmp (s->val[OPT_MODE].s, "Gray") == 0)
    {
      /* data format is greyscale, pixel format is one pixel/byte */
      write_string (s->fd, "\033*a4T");
      write_string (s->fd, "\033*a8G");
      mode = 5;
    }
  else if (strcmp (s->val[OPT_MODE].s, "Color") == 0)
    {
      /* data format is 24bit color, pixel format is one pixel/three
	 bytes */
      write_string (s->fd, "\033*a5T");
      write_string (s->fd, "\033*a24G");
      mode = 6;
    }

  /* inquire resulting size of image after setting it up */
  if (((status = inquire (s->fd, 1024, 'E', 'd', &s->params.pixels_per_line))
       != SANE_STATUS_GOOD)
      || ((status = inquire (s->fd, 1025, 'E', 'd', &s->params.bytes_per_line))
	  != SANE_STATUS_GOOD)
      || ((status = inquire (s->fd, 1026, 'E', 'd', &s->params.lines))
	  != SANE_STATUS_GOOD))
    {
      DBG(1, "inquiry of scan parameters failed: %s", sane_strstatus (status));
      return status;
    }

  s->bytes_to_read = s->params.bytes_per_line * s->params.lines;

  DBG(1, "%d pixels per line, %d bytes, %d lines high, dpi=%d\n",
      s->params.pixels_per_line, s->params.bytes_per_line, s->params.lines,
      s->val[OPT_RESOLUTION].w);

  write_string (s->fd, "\033*f0S");		/* start scan */

  s->scanning = SANE_TRUE;
  return SANE_STATUS_GOOD;
}

SANE_Status
sane_read (SANE_Handle handle, SANE_Byte *buf, SANE_Int max_len, SANE_Int *len)
{
  HP_Scanner *s = handle;
  SANE_Status status;
  size_t nread;

  *len = 0;

  if (s->bytes_to_read == 0)
    {
      do_cancel (s);
      return SANE_STATUS_EOF;
    }

  if (!s->scanning)
    return do_cancel (s);

  nread = max_len;
  if (nread > s->bytes_to_read)
    nread = s->bytes_to_read;

  status = read_data (s->fd, buf, &nread);
  if (status != SANE_STATUS_GOOD)
    {
      do_cancel (s);
      return SANE_STATUS_IO_ERROR;
    }
  *len = nread;
  s->bytes_to_read -= nread;

  return SANE_STATUS_GOOD;
}

void
sane_cancel (SANE_Handle handle)
{
  HP_Scanner *s = handle;

  s->scanning = SANE_FALSE;
}

SANE_Status
sane_set_io_mode (SANE_Handle handle, SANE_Bool non_blocking)
{
  return SANE_STATUS_UNSUPPORTED;
}

SANE_Status
sane_get_select_fd (SANE_Handle handle, SANE_Int *fd)
{
  return SANE_STATUS_UNSUPPORTED;
}
