/*
 * bltGrPs.c --
 *
 *      This module implements the "postscript" operation for BLT graph widget.
 *
 * Copyright 1991-1997 Bell Labs Innovations for Lucent Technologies.
 *
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted, provided
 * that the above copyright notice appear in all copies and that both that the
 * copyright notice and warranty disclaimer appear in supporting documentation,
 * and that the names of Lucent Technologies any of their entities not be used
 * in advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.
 *
 * Lucent Technologies disclaims all warranties with regard to this software,
 * including all implied warranties of merchantability and fitness.  In no event
 * shall Lucent Technologies be liable for any special, indirect or
 * consequential damages or any damages whatsoever resulting from loss of use,
 * data or profits, whether in an action of contract, negligence or other
 * tortuous action, arising out of or in connection with the use or performance
 * of this software.
 */

/*
 * -----------------------------------------------------------------
 *
 * PostScript routines to print a graph
 *
 * -----------------------------------------------------------------
 */
#include "bltGraph.h"
#include <X11/Xutil.h>
#include <X11/Xatom.h>
#if defined(__STDC__)
#include <stdarg.h>
#else
#include <varargs.h>
#endif

#define PS_MAXPATH	1500	/* Maximum number of components in a PostScript
				 * (level 1) path. */

#define SCRATCH_LENGTH  (BUFSIZ*2)

static int ColorModeParse _ANSI_ARGS_((ClientData clientData,
	Tcl_Interp *interp, Tk_Window tkwin, char *value, char *widgRec,
	int offset));
static char *ColorModePrint _ANSI_ARGS_((ClientData clientData,
	Tk_Window tkwin, char *widgRec, int offset,
	Tcl_FreeProc **freeProcPtr));

static Tk_CustomOption colorModeOption =
{
    ColorModeParse, ColorModePrint, (ClientData)0,
};

extern Tk_CustomOption bltLengthOption;
extern Tk_CustomOption bltPadOption;

/*
 * PageExtents --
 *
 * 	Convenience structure to pass information to various routines
 *	indicating how the PostScript page is arranged.
 *
 */
typedef struct {
    int width, height;		/* Dimensions of the page.  */
    int bbWidth, bbHeight;	/* Dimensions of the bounding box.
				 * This includes the graph and padding. */
    int adjustX, adjustY;	/* Distance from page edge to upper left
				 * corner of the bounding box */
    float maxScale;		/* Scale to maximize plot aspect ratio */
} PageExtents;

#define DEF_PS_CENTER		"1"
#define DEF_PS_COLOR_MAP	(char *)NULL
#define DEF_PS_COLOR_MODE	"color"
#define DEF_PS_DECORATIONS	"1"
#define DEF_PS_FONT_MAP		(char *)NULL
#define DEF_PS_LANDSCAPE	"0"
#define DEF_PS_MAXPECT		"0"
#define DEF_PS_PADX		"1.0i"
#define DEF_PS_PADY		"1.0i"
#define DEF_PS_PAPERHEIGHT	"11.0i"
#define DEF_PS_PAPERWIDTH	"8.5i"

static Tk_ConfigSpec configSpecs[] =
{
    {TK_CONFIG_BOOLEAN, "-center", "psCenter", "PsCenter",
	DEF_PS_CENTER, Tk_Offset(PostScript, center),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-colormap", "psColorMap", "PsColorMap",
	DEF_PS_COLOR_MAP, Tk_Offset(PostScript, colorVarName), 0},
    {TK_CONFIG_CUSTOM, "-colormode", "psColorMode", "PsColorMode",
	DEF_PS_COLOR_MODE, Tk_Offset(PostScript, colorMode),
	TK_CONFIG_DONT_SET_DEFAULT, &colorModeOption},
    {TK_CONFIG_BOOLEAN, "-decorations", "psDecorations", "PsDecorations",
	DEF_PS_DECORATIONS, Tk_Offset(PostScript, decorations),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_STRING, "-fontmap", "psFontMap", "PsFontMap",
	DEF_PS_FONT_MAP, Tk_Offset(PostScript, fontVarName), 0},
    {TK_CONFIG_CUSTOM, "-height", "psHeight", "PsHeight",
	(char *)NULL, Tk_Offset(PostScript, reqHeight),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_BOOLEAN, "-landscape", "psLandscape", "PsLandscape",
	DEF_PS_LANDSCAPE, Tk_Offset(PostScript, landscape),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_BOOLEAN, "-maxpect", "psMaxpect", "PsMaxpect",
	DEF_PS_MAXPECT, Tk_Offset(PostScript, maxpect),
	TK_CONFIG_DONT_SET_DEFAULT},
    {TK_CONFIG_CUSTOM, "-padx", "psPadX", "PsPadX",
	DEF_PS_PADX, Tk_Offset(PostScript, padX), 0, &bltPadOption},
    {TK_CONFIG_CUSTOM, "-pady", "psPadY", "PsPadY",
	DEF_PS_PADY, Tk_Offset(PostScript, padY), 0, &bltPadOption},
    {TK_CONFIG_CUSTOM, "-paperheight", "psPaperHeight", "PsPaperHeight",
	DEF_PS_PAPERHEIGHT, Tk_Offset(PostScript, reqPaperHeight),
	0, &bltLengthOption},
    {TK_CONFIG_CUSTOM, "-paperwidth", "psPaperWidth", "PsPaperWidth",
	DEF_PS_PAPERWIDTH, Tk_Offset(PostScript, reqPaperWidth),
	0, &bltLengthOption},
    {TK_CONFIG_CUSTOM, "-width", "psWidth", "PsWidth",
	(char *)NULL, Tk_Offset(PostScript, reqWidth),
	TK_CONFIG_DONT_SET_DEFAULT, &bltLengthOption},
    {TK_CONFIG_END, NULL, NULL, NULL, NULL, 0, 0}
};

extern void Blt_PrintMarkers _ANSI_ARGS_((Graph *graphPtr, int under));
extern void Blt_PrintMarkers _ANSI_ARGS_((Graph *graphPtr, int under));
extern void Blt_PrintElements _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_PrintActiveElements _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_PrintLegend _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_PrintGrid _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_PrintAxes _ANSI_ARGS_((Graph *graphPtr));
extern void Blt_PrintAxisLimits _ANSI_ARGS_((Graph *graphPtr));

/*
 *----------------------------------------------------------------------
 *
 * ColorModeParse --
 *
 *	Convert the string representation of a PostScript color mode
 *	into the enumerated type representing the color level:
 *
 *	    PS_MODE_COLOR 	- Full color
 *	    PS_MODE_GRAYSCALE  	- Color converted to grayscale
 *	    PS_MODE_MONOCHROME 	- Only black and white
 *
 * Results:
 *	The return value is a standard Tcl result.  The color level is
 *	written into the page layout information structure.
 *
 * Side Effects:
 *	Future invocations of the "postscript" option will use this
 *	variable to determine how color information will be displayed
 *	in the PostScript output it produces.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
ColorModeParse(clientData, interp, tkwin, value, widgRec, offset)
    ClientData clientData;	/* not used */
    Tcl_Interp *interp;		/* Interpreter to send results back to */
    Tk_Window tkwin;		/* not used */
    char *value;		/* New legend position string */
    char *widgRec;		/* Graph widget record */
    int offset;			/* Offset of colorMode field in record */
{
    ColorMode *modePtr = (ColorMode *)(widgRec + offset);
    unsigned int length;
    char c;

    c = value[0];
    length = strlen(value);
    if ((c == 'c') && (strncmp(value, "color", length) == 0)) {
	*modePtr = PS_MODE_COLOR;
    } else if ((c == 'g') && (strncmp(value, "grayscale", length) == 0)) {
	*modePtr = PS_MODE_GRAYSCALE;
    } else if ((c == 'g') && (strncmp(value, "greyscale", length) == 0)) {
	*modePtr = PS_MODE_GRAYSCALE;
    } else if ((c == 'm') && (strncmp(value, "monochrome", length) == 0)) {
	*modePtr = PS_MODE_MONOCHROME;
    } else {
	Tcl_AppendResult(interp, "bad color mode \"", value, "\": should be \
\"color\", \"grayscale\", or \"monochrome\"", (char *)NULL);
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfColorMode --
 *
 *	Convert the PostScript mode value into the string representing
 *	a valid color mode.
 *
 * Results:
 *	The static string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfColorMode(colorMode)
    ColorMode colorMode;
{
    switch (colorMode) {
    case PS_MODE_COLOR:
	return "color";
    case PS_MODE_GRAYSCALE:
	return "grayscale";
    case PS_MODE_MONOCHROME:
	return "monochrome";
    default:
	return "unknown color mode";
    }
}

/*
 *----------------------------------------------------------------------
 *
 * ColorModePrint --
 *
 *	Convert the current color mode into the string representing a
 *	valid color mode.
 *
 * Results:
 *	The string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static char *
ColorModePrint(clientData, tkwin, widgRec, offset, freeProcPtr)
    ClientData clientData;	/* not used */
    Tk_Window tkwin;		/* not used */
    char *widgRec;		/* PostScript structure record */
    int offset;			/* field of colorMode in record */
    Tcl_FreeProc **freeProcPtr;	/* not used */
{
    ColorMode mode = *(ColorMode *)(widgRec + offset);

    return (NameOfColorMode(mode));
}


void
Blt_PostScriptAppend TCL_VARARGS_DEF(Graph *, arg1)
{
    va_list argList;
    Graph *graphPtr;
    char *fmt;
    PostScript *psPtr; 

    graphPtr = TCL_VARARGS_START(Graph *, arg1, argList);
    fmt = va_arg(argList, char *);
    psPtr = (PostScript *)graphPtr->postscript;
    vsprintf(psPtr->scratchPtr, fmt, argList);
    va_end(argList);
    Tcl_DStringAppend(psPtr->dsPtr, psPtr->scratchPtr, -1);
}

/*
 *----------------------------------------------------------------------
 *
 * XColorToPostScript --
 *
 *	Convert the a XColor (from its RGB values) to a PostScript
 *	command.  If a Tcl color map variable exists, it will be
 *	consulted for a PostScript translation based upon the color
 *	name.
 *
 * Results:
 *	The string representing the color mode is returned.
 *
 *----------------------------------------------------------------------
 */
static void
XColorToPostScript(graphPtr, colorPtr, attr)
    Graph *graphPtr;
    XColor *colorPtr;		/* Color value to be converted */
    MonoAttribute attr;		/* If non-zero, this represents a foreground
				 * color, otherwise a background color */
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    float red, green, blue;

    /* If the color name exists in Tcl array variable, use that translation */
    if (psPtr->colorVarName != NULL) {
	char *colorDesc;

	colorDesc = Tcl_GetVar2(graphPtr->interp, psPtr->colorVarName,
	    Tk_NameOfColor(colorPtr), 0);
	if (colorDesc != NULL) {
	    Tcl_DStringAppend(psPtr->dsPtr, colorDesc, -1);
	    Tcl_DStringAppend(psPtr->dsPtr, " ", -1);
	    return;
	}
    }
    /* Otherwise convert the X color pointer to its PostScript RGB values */
    red = (float)colorPtr->red / 65535.0;
    green = (float)colorPtr->green / 65535.0;
    blue = (float)colorPtr->blue / 65535.0;
    sprintf(psPtr->scratchPtr, "%g %g %g %s  ", red, green, blue,
	(attr == PS_MONO_FOREGROUND) ? "SetFgColor" : "SetBgColor");
    Tcl_DStringAppend(psPtr->dsPtr, psPtr->scratchPtr, -1);
}

/*
 *----------------------------------------------------------------------
 *
 * ReverseBits --
 *
 *	Convert a byte from a X image into PostScript image order.
 *	This requires not only the nybbles to be reversed but also
 *	their bit values.
 *
 * Results:
 *	The converted byte is returned.
 *
 *----------------------------------------------------------------------
 */
INLINE static unsigned char
ReverseBits(byte)
    register unsigned char byte;
{
    byte = ((byte >> 1) & 0x55) | ((byte << 1) & 0xaa);
    byte = ((byte >> 2) & 0x33) | ((byte << 2) & 0xcc);
    byte = ((byte >> 4) & 0x0f) | ((byte << 4) & 0xf0);
    return (byte);
}

/*
 * -------------------------------------------------------------------------
 *
 * Blt_BitmapToPostScript --
 *
 *      Output a PostScript image string of the given bitmap image.
 *      It is assumed the image is one bit deep and a zero value
 *      indicates an off-pixel.  To convert to PostScript, the bits
 *      need to be reversed from the X11 image order.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      The PostScript image string is appended to interp->result.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_BitmapToPostScript(graphPtr, bitmap, width, height)
    Graph *graphPtr;
    Pixmap bitmap;
    int width, height;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    register unsigned int byte = 0;
    register int x, y, bitPos;
    unsigned long pixel;
    XImage *imagePtr;
    int byteCount = 0;

    imagePtr = XGetImage(graphPtr->display, bitmap, 0, 0, width, height, 1,
	ZPixmap);
    Tcl_DStringAppend(psPtr->dsPtr, "\n\t<", -1);
    bitPos = 0;			/* Suppress compiler warning */
    for (y = 0; y < height; y++) {
	byte = 0;
	for (x = 0; x < width; x++) {
	    pixel = XGetPixel(imagePtr, x, y);
	    bitPos = x % 8;
	    byte |= (unsigned char)(pixel << bitPos);
	    if (bitPos == 7) {
		Blt_PostScriptAppend(graphPtr, "%02x", ReverseBits(byte));
		byteCount++;
		byte = 0;
	    }
	    if (byteCount >= 30) {
		Tcl_DStringAppend(psPtr->dsPtr, "\n\t ", -1);
		byteCount = 0;
	    }
	}
	if (bitPos != 7) {
	    Blt_PostScriptAppend(graphPtr, "%02x", ReverseBits(byte));
	    byteCount++;
	}
    }
    Tcl_DStringAppend(psPtr->dsPtr, ">\n", -1);
    XDestroyImage(imagePtr);
}

/*
 *----------------------------------------------------------------------
 *
 * NameOfAtom --
 *
 *	Wrapper routine for Tk_GetAtomName.  Returns NULL instead of
 *	"?bad atom?" if the atom cannot be found.
 *
 * Results:
 *	The name of the atom is returned if found. Otherwise NULL.
 *
 *----------------------------------------------------------------------
 */
static char *
NameOfAtom(tkwin, atom)
    Tk_Window tkwin;
    Atom atom;
{
    char *result;

    result = Tk_GetAtomName(tkwin, atom);
    if ((result[0] == '?') && (strcmp(result, "?bad atom?") == 0)) {
	return NULL;
    }
    return (result);
}

/*
 * -----------------------------------------------------------------
 *
 * XFontStructToPostScript --
 *
 *      Map X11 font to a PostScript font. Currently, only fonts whose
 *      FOUNDRY property are "Adobe" are converted. Simply gets the
 *      XA_FULL_NAME and XA_FAMILY properties and pieces together a
 *      PostScript fontname.
 *
 * Results:
 *      Returns the mapped PostScript font name if one is possible.
 *	Otherwise returns NULL.
 *
 * -----------------------------------------------------------------
 */
static char *
XFontStructToPostScript(tkwin, font, fontPtr)
    Tk_Window tkwin;		/* Window to query for atoms */
    Tk_Font font;
    XFontStruct *fontPtr;	/* Font structure to map to name */
{
    Atom atom;
    char *fullName, *family, *foundry;
    register char *src, *dest;
    char *start;
    static char string[200];	/* What size? */

    if (XGetFontProperty(fontPtr, XA_FULL_NAME, &atom) == False) {
	return NULL;
    }
    fullName = NameOfAtom(tkwin, atom);
    if (fullName == NULL) {
	return NULL;
    }
    family = foundry = NULL;
    if (XGetFontProperty(fontPtr, Tk_InternAtom(tkwin, "FOUNDRY"), &atom)) {
	foundry = NameOfAtom(tkwin, atom);
    }
    if (XGetFontProperty(fontPtr, XA_FAMILY_NAME, &atom)) {
	family = NameOfAtom(tkwin, atom);
    }
    /*
     * Try to map the font only if the foundry is Adobe
     */
    if ((foundry == NULL) || (family == NULL)) {
	return NULL;
    }
    if ((strcmp(foundry, "Adobe") != 0) &&
	(strcmp(foundry, "Linotype") != 0) &&
	(strcmp(foundry, "Bitstream") != 0)) {
#ifndef notdef
	fprintf(stderr, "huh? Full name (%s) for non-PS (%s) font:\n\t%s \n",
	    fullName, foundry, Tk_NameOfFont(font));
#endif
	return NULL;
    }
    src = fullName + strlen(family);

    /*
     * Special case: Fix mapping of "New Century Schoolbook"
     */
    if ((*family == 'N') && (strcmp(family, "New Century Schoolbook") == 0)) {
	family = "NewCenturySchlbk";
    }
    /*
     * PostScript font name is in the form <family>-<type face>
     */
    sprintf(string, "%s-", family);
    dest = start = string + strlen(string);

    /*
     * Append the type face (part of the full name trailing the family name)
     * to the the PostScript font name, removing any spaces or dashes
     *
     * ex. " Bold Italic" ==> "BoldItalic"
     */
    while (*src != '\0') {
	if ((*src != ' ') && (*src != '-')) {
	    *dest++ = *src;
	}
	src++;
    }
    if (dest == start) {
	--dest;			/* Remove '-' to leave just the family name */
    }
    *dest = '\0';		/* Make a valid string */
    return string;
}

/*
 * -------------------------------------------------------------------
 * Routines to convert X drawing functions to PostScript commands.
 * -------------------------------------------------------------------
 */
void
Blt_ClearBackgroundToPostScript(graphPtr)
    Graph *graphPtr;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    Tcl_DStringAppend(psPtr->dsPtr, " 1.0 1.0 1.0 SetBgColor\n", -1);
}

void
Blt_BackgroundToPostScript(graphPtr, colorPtr)
    Graph *graphPtr;
    XColor *colorPtr;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    XColorToPostScript(graphPtr, colorPtr, PS_MONO_BACKGROUND);
    Tcl_DStringAppend(psPtr->dsPtr, "\n", -1);
}

void
Blt_ForegroundToPostScript(graphPtr, colorPtr)
    Graph *graphPtr;
    XColor *colorPtr;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    XColorToPostScript(graphPtr, colorPtr, PS_MONO_FOREGROUND);
    Tcl_DStringAppend(psPtr->dsPtr, "\n", -1);
}

void
Blt_LineWidthToPostScript(graphPtr, lineWidth)
    Graph *graphPtr;
    int lineWidth;
{
    if (lineWidth < 1) {
	lineWidth = 1;
    }
    Blt_PostScriptAppend(graphPtr, "%d setlinewidth\n", lineWidth);
}

void
Blt_LineDashesToPostScript(graphPtr, dashesPtr)
    Graph *graphPtr;
    Dashes *dashesPtr;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    Tcl_DStringAppend(psPtr->dsPtr, "[ ", -1);
    if (dashesPtr != NULL) {
	register int i;

	for (i = 0; i < dashesPtr->numValues; i++) {
	    Blt_PostScriptAppend(graphPtr, " %d", dashesPtr->valueArr[i]);
	}
    }
    Tcl_DStringAppend(psPtr->dsPtr, "] 0 setdash\n", -1);
}

void
Blt_SetLineAttributes(graphPtr, colorPtr, lineWidth, dashesPtr)
    Graph *graphPtr;
    XColor *colorPtr;
    int lineWidth;
    Dashes *dashesPtr;
{
    Blt_ForegroundToPostScript(graphPtr, colorPtr);
    Blt_LineWidthToPostScript(graphPtr, lineWidth);
    Blt_LineDashesToPostScript(graphPtr, dashesPtr);
}

void
Blt_RectangleToPostScript(graphPtr, x, y, width, height)
    Graph *graphPtr;
    int x, y;
    int width, height;
{
    Blt_PostScriptAppend(graphPtr, "%d %d %d %d Box Fill\n\n", x, y, width, height);
}

void
Blt_LinesToPostScript(graphPtr, pointPtr, numPoints)
    Graph *graphPtr;
    register XPoint *pointPtr;
    int numPoints;
{
    register int i;

    Blt_PostScriptAppend(graphPtr, "newpath %d %d moveto\n", pointPtr->x, 
	 pointPtr->y);
    pointPtr++;
    for (i = 1; i < numPoints; i++, pointPtr++) {
	Blt_PostScriptAppend(graphPtr, "%d %d lineto\n", pointPtr->x, pointPtr->y);
    }
}

void
Blt_PolygonToPostScript(graphPtr, pointArr, numPoints)
    Graph *graphPtr;
    XPoint *pointArr;
    int numPoints;
{
    Blt_LinesToPostScript(graphPtr, pointArr, numPoints);
    Blt_PostScriptAppend(graphPtr, "%d %d lineto closepath Fill\n", pointArr[0].x, 
	pointArr[0].y);
}

void
Blt_SegmentsToPostScript(graphPtr, segPtr, numSegments)
    Graph *graphPtr;
    register XSegment *segPtr;
    int numSegments;
{
    register int i;

    for (i = 0; i < numSegments; i++, segPtr++) {
	Blt_PostScriptAppend(graphPtr, "%d %d %d %d LS\n", segPtr->x1, segPtr->y1, 
		segPtr->x2, segPtr->y2);
    }
}

#ifdef notdef
void
Blt_RectanglesToPostScript(graphPtr, rectArr, numRects)
    Graph *graphPtr;
    XRectangle rectArr[];
    int numRects;
{
    register int i;

    for (i = 0; i < numRects; i++) {
	Blt_RectangleToPostScript(graphPtr, rectArr[i].x, rectArr[i].y,
	    (int)rectArr[i].width, (int)rectArr[i].height);
    }
}

#endif

/*
 * The Border structure used internally by the Tk_3D* routines.
 * The following is a copy of it from tk3d.c.
 */

typedef struct {
    Screen *screen;		/* Screen on which the border will be used. */
    Visual *visual;		/* Visual for all windows and pixmaps using
				 * the border. */
    int depth;			/* Number of bits per pixel of drawables where
				 * the border will be used. */
    Colormap colormap;		/* Colormap out of which pixels are
				 * allocated. */
    int refCount;		/* Number of different users of
				 * this border.  */
    XColor *bgColorPtr;		/* Background color (intensity
				 * between lightColorPtr and
				 * darkColorPtr). */
    XColor *darkColorPtr;	/* Color for darker areas (must free when
				 * deleting structure). NULL means shadows
				 * haven't been allocated yet.*/
    XColor *lightColorPtr;	/* Color used for lighter areas of border
				 * (must free this when deleting structure).
				 * NULL means shadows haven't been allocated
				 * yet. */
    Pixmap shadow;		/* Stipple pattern to use for drawing
				 * shadows areas.  Used for displays with
				 * <= 64 colors or where colormap has filled
				 * up. */
    GC bgGC;			/* Used (if necessary) to draw areas in
				 * the background color. */
    GC darkGC;			/* Used to draw darker parts of the
				 * border. None means the shadow colors
				 * haven't been allocated yet.*/
    GC lightGC;			/* Used to draw lighter parts of
				 * the border. None means the shadow colors
				 * haven't been allocated yet. */
    Tcl_HashEntry *hashPtr;	/* Entry in borderTable (needed in
				 * order to delete structure). */
} Border;

void
Blt_Print3DRectangle(graphPtr, border, x, y, width, height, borderWidth,
    relief)
    Graph *graphPtr;		/* File pointer to write PS output. */
    Tk_3DBorder border;		/* Token for border to draw. */
    int x, y;			/* Coordinates of rectangle */
    int width, height;		/* Region to be drawn. */
    int borderWidth;		/* Desired width for border, in pixels. */
    int relief;			/* Should be either TK_RELIEF_RAISED or
                                 * TK_RELIEF_SUNKEN;  indicates position of
                                 * interior of window relative to exterior. */
{
    Border *borderPtr = (Border *)border;
    XColor lightColor, darkColor;
    XColor *lightColorPtr, *darkColorPtr;
    XColor *topColor, *bottomColor;
    XPoint points[7];
    int twiceWidth = (borderWidth * 2);

    if ((width < twiceWidth) || (height < twiceWidth)) {
	return;
    }
    if ((borderPtr->lightColorPtr == NULL) || (borderPtr->darkColorPtr == NULL)) {
	Screen *screenPtr;

	lightColor = *borderPtr->bgColorPtr;
	screenPtr = Tk_Screen(graphPtr->tkwin);
	if (lightColor.pixel == WhitePixelOfScreen(screenPtr)) {
	    darkColor.red = darkColor.blue = darkColor.green = 0x00;
	} else {
	    darkColor.red = darkColor.blue = darkColor.green = 0xFF;
	}
	lightColorPtr = &lightColor;
	darkColorPtr = &darkColor;
    } else {
	lightColorPtr = borderPtr->lightColorPtr;
	darkColorPtr = borderPtr->darkColorPtr;
    }

    /*
     * Handle grooves and ridges with recursive calls.
     */

    if ((relief == TK_RELIEF_GROOVE) || (relief == TK_RELIEF_RIDGE)) {
	int halfWidth, insideOffset;

	halfWidth = borderWidth / 2;
	insideOffset = borderWidth - halfWidth;
	Blt_Print3DRectangle(graphPtr, border, x, y, width, height, halfWidth,
	    (relief == TK_RELIEF_GROOVE) ? TK_RELIEF_SUNKEN :
	    TK_RELIEF_RAISED);
	Blt_Print3DRectangle(graphPtr, border, x + insideOffset,
	    y + insideOffset, width - insideOffset * 2,
	    height - insideOffset * 2, halfWidth,
	    (relief == TK_RELIEF_GROOVE) ? TK_RELIEF_RAISED :
	    TK_RELIEF_SUNKEN);
	return;
    }
    if (relief == TK_RELIEF_RAISED) {
	topColor = lightColorPtr;
	bottomColor = darkColorPtr;
    } else if (relief == TK_RELIEF_SUNKEN) {
	topColor = darkColorPtr;
	bottomColor = lightColorPtr;
    } else {
	topColor = bottomColor = borderPtr->bgColorPtr;
    }
    Blt_BackgroundToPostScript(graphPtr, bottomColor);
    Blt_RectangleToPostScript(graphPtr, x, y + height - borderWidth,
	width, borderWidth);
    Blt_RectangleToPostScript(graphPtr, x + width - borderWidth, y,
	borderWidth, height);
    points[0].x = points[1].x = points[6].x = x;
    points[0].y = points[6].y = y + height;
    points[1].y = points[2].y = y;
    points[2].x = x + width;
    points[3].x = x + width - borderWidth;
    points[3].y = points[4].y = y + borderWidth;
    points[4].x = points[5].x = x + borderWidth;
    points[5].y = y + height - borderWidth;
    if (relief != TK_RELIEF_FLAT) {
	Blt_BackgroundToPostScript(graphPtr, topColor);
    }
    Blt_PolygonToPostScript(graphPtr, points, 7);
}

void
Blt_3DRectangleToPostScript(graphPtr, border, x, y, width, height,
    borderWidth, relief)
    Graph *graphPtr;
    Tk_3DBorder border;		/* Token for border to draw. */
    int x, y;			/* Coordinates of top-left of border area */
    int width, height;		/* Dimension of border to be drawn. */
    int borderWidth;		/* Desired width for border, in pixels. */
    int relief;			/* Should be either TK_RELIEF_RAISED or
                                 * TK_RELIEF_SUNKEN;  indicates position of
                                 * interior of window relative to exterior. */
{
    Border *borderPtr = (Border *)border;

    /*
     * I'm assuming that the rectangle is to be drawn as a background.
     * Setting the pen color as foreground or background only affects
     * the plot when the colormode option is "monochrome".
     */
    Blt_BackgroundToPostScript(graphPtr, borderPtr->bgColorPtr);
    Blt_RectangleToPostScript(graphPtr, x, y, width, height);
    Blt_Print3DRectangle(graphPtr, border, x, y, width, height, borderWidth,
	relief);
}

void
Blt_StippleToPostScript(graphPtr, bitmap, width, height, fillOrStroke)
    Graph *graphPtr;
    Pixmap bitmap;
    int width, height;
    int fillOrStroke;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    Blt_PostScriptAppend(graphPtr, "%d %d\n", width, height);
    Blt_BitmapToPostScript(graphPtr, bitmap, width, height);
    Tcl_DStringAppend(psPtr->dsPtr, (fillOrStroke) ? "true" : "false", -1);
    Tcl_DStringAppend(psPtr->dsPtr, " StippleFill\n", -1);
}

/*
 * -------------------------------------------------------------------------
 *
 * Blt_PhotoToPostScript --
 *
 *      Output a PostScript image string of the given bitmap image.
 *      It is assumed the image is one bit deep and a zero value
 *      indicates an off-pixel.  To convert to PostScript, the bits
 *      need to be reversed from the X11 image order.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      The PostScript image string is appended to interp->result.
 *
 * -------------------------------------------------------------------------
 */
void
Blt_PhotoToPostScript(graphPtr, photoToken, x, y)
    Graph *graphPtr;
    Tk_PhotoHandle photoToken;
    int x, y;			/* Origin of photo image */
{
    int redOffset, greenOffset, blueOffset;
    unsigned char red, green, blue;
    char *sep;
    int numBytes;
    register int count;
    register int i;
    Tk_PhotoImageBlock block;

    Tk_PhotoGetImage(photoToken, &block);
    Blt_PostScriptAppend(graphPtr, "\n/tmpStr %d string def\n", block.width * 3);
    Blt_PostScriptAppend(graphPtr, "gsave\n  %d %d translate\n  %d %d scale\n",
		 x, y + block.height, block.width, -block.height);
    Blt_PostScriptAppend(graphPtr, "  %d %d 8\n", block.width, block.height);
    Blt_PostScriptAppend(graphPtr, "  [%d 0 0 %d 0 %d] \
{ currentfile tmpStr readhexstring pop }\n  false 3 colorimage\n\n  ", 
	    block.width, -block.height, block.height);

    redOffset = block.offset[0];
    blueOffset = block.offset[1];
    greenOffset = block.offset[2];
    numBytes = block.pitch * block.height;

    count = 0;
    for (i = 0; i < numBytes; i += block.pixelSize) {
	red = block.pixelPtr[i + redOffset];
	green = block.pixelPtr[i + blueOffset];
	blue = block.pixelPtr[i + greenOffset];
	sep = "";
	count += 6;
	if (count >= 60) {
	    sep = "\n  ";
	    count = 0;
	}
	Blt_PostScriptAppend(graphPtr, "%02x%02x%02x%s", red, green, blue, sep);
    }
    Blt_PostScriptAppend(graphPtr, "\ngrestore\n\n");
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_WindowToPostScript --
 *
 *      Takes a snapshot of an X drawable (pixmap or window) and
 *	returns the PostScript output.
 *
 * Results:
 *      None.
 *
 *----------------------------------------------------------------------
 */
void
Blt_WindowToPostScript(graphPtr, tkwin, x, y, width, height)
    Graph *graphPtr;
    Tk_Window tkwin;
    int x, y;
    int width, height;
{
    XColor *colorPtr, *colorArr;
    unsigned long *dataArr;
    Tcl_HashTable colorTable;
    Tcl_HashEntry *hPtr;
    int numPixels;
    register int count;
    register int i;
    char *sep;
    char *fmt1 = "\n\
/tmpStr %d string def\n\
gsave\n\
  %d %d translate\n\
  %d %d scale\n";
    char *fmt2 = "  %d %d 8\n\
  [%d 0 0 %d 0 %d] { currentfile tmpStr readhexstring pop }\n\
  false 3 colorimage\n\n  ";

    numPixels = Blt_GetSnapshot(graphPtr->tkwin, Tk_WindowId(tkwin), width, height,
	&colorTable, &colorArr, &dataArr);
    if (numPixels < 0) {
	/* Can't grab window image so paint the window area grey */
	Blt_PostScriptAppend(graphPtr, 
"%% Can't grab window \"%s\": drawing region gray\n0.5 0.5 0.5 SetBgColor\n",
	    Tk_PathName(tkwin));
	Blt_RectangleToPostScript(graphPtr, x, y, width, height);
	return;
    }
    /*
     * Now, we can go back through the array of pixels, replacing each pixel
     * of the image with its RGB value.
     */
    Blt_PostScriptAppend(graphPtr, fmt1, width * 3, x, y + height, width, -height);
    Blt_PostScriptAppend(graphPtr, fmt2, width, height, width, -height, height);
    count = 0;
    for (i = 0; i < numPixels; i++) {
	hPtr = Tcl_FindHashEntry(&colorTable, (char *)dataArr[i]);
	colorPtr = (XColor *)Tcl_GetHashValue(hPtr);
	sep = "";
	count += 6;
	if (count >= 60) {
	    sep = "\n  ";
	    count = 0;
	}
	Blt_PostScriptAppend(graphPtr, "%02x%02x%02x%s",
	    ((unsigned int)colorPtr->red >> 8),
	    ((unsigned int)colorPtr->green >> 8),
	    ((unsigned int)colorPtr->blue >> 8), sep);
    }
    Blt_PostScriptAppend(graphPtr, "\ngrestore\n\n");
    free((char *)colorArr);
    free((char *)dataArr);
    Tcl_DeleteHashTable(&colorTable);
}

/*
 * -----------------------------------------------------------------
 *
 * FontToPostScript --
 *
 *      Map the Tk font to a PostScript font and point size.
 *
 *	If a Tcl array variable was specified, each element should be
 *	indexed by the X11 font name and contain a list of 1-2
 *	elements; the PostScript font name and the desired point size.
 *	The point size may be omitted and the X font point size will
 *	be used.
 *
 *	Otherwise, if the foundry is "Adobe", we try to do a plausible
 *	mapping looking at the full name of the font and building a
 *	string in the form of "Family-TypeFace".
 *
 * Returns:
 *      None.
 *
 * Side Effects:
 *      PostScript commands are output to change the type and the
 *      point size of the current font.
 *
 * -----------------------------------------------------------------
 */
static void
FontToPostScript(graphPtr, font)
    Graph *graphPtr;
    Tk_Font font;		/* Tk font to query about */
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    XFontStruct *fontPtr = (XFontStruct *)font;
    char *fontName;
    float pointSize;
    Tcl_DString fontNameStr;

    fontName = "Helvetica-Bold"; /* Default font */
    pointSize = 12.0;		/* Default point size */

#if (TK_MAJOR_VERSION >= 8)
    /* Can you believe what I have to go through to get an XFontStruct? */
    fontPtr = XLoadQueryFont(Tk_Display(graphPtr->tkwin), Tk_NameOfFont(font));
#endif
    if (fontPtr != NULL) {
	unsigned long fontProp;

	if (XGetFontProperty(fontPtr, XA_POINT_SIZE, &fontProp) != False) {
	    pointSize = (float)fontProp / 10.0;
	}
    }
    if (psPtr->fontVarName != NULL) {
	char *fontInfo;

	fontInfo = Tcl_GetVar2(graphPtr->interp, psPtr->fontVarName,
	    Tk_NameOfFont(font), 0);
	if (fontInfo != NULL) {
	    int numProps;
	    char **propArr = NULL;

	    if (Tcl_SplitList(graphPtr->interp, fontInfo, &numProps,
		    &propArr) == TCL_OK) {
		fontName = propArr[0];
		if (numProps == 2) {
		    int newSize;

		    Tcl_GetInt(graphPtr->interp, propArr[1], &newSize);
		    pointSize = (float)newSize;
		}
	    }
	    Blt_PostScriptAppend(graphPtr, "%g /%s SetFont\n",pointSize, fontName);
	    if (propArr != (char **)NULL) {
		free((char *)propArr);
	    }
	    return;
	}
    }
    Tcl_DStringInit(&fontNameStr);

    /* 
     * First try looking at the XFontStruct for clues about the font.
     * Not ready for WIN32. We'll have to work harder to determine the
     * font characteristics.
     */

#if (TK_MAJOR_VERSION >= 8)
    if (fontPtr != NULL) {
	fontName = XFontStructToPostScript(graphPtr->tkwin, font, fontPtr);
	XFreeFont(Tk_Display(graphPtr->tkwin), fontPtr);
    }
#ifdef notdef
    if (fontName == NULL) {
	/* 
	 * Tk_PostScriptFontName only really works with PostScript fonts 
	 * it knows about, and doesn't tell us when it bails out.
	 */
	pointSize = (float)Tk_PostscriptFontName(font, &fontNameStr);
	fontName = Tcl_DStringValue(&fontNameStr);
    }
#endif

#else
    fontName = XFontStructToPostScript(graphPtr->tkwin, font, fontPtr);
#endif /* TK_MAJOR_VERSION >= 8 */

    if ((fontName == NULL) || (fontName[0] == '\0')) {
	fontName = "Helvetica-Bold";	/* Defaulting to a known PS font */
    }
    Blt_PostScriptAppend(graphPtr, "%g /%s SetFont\n", pointSize, fontName);
    Tcl_DStringFree(&fontNameStr);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_PrintText --
 *
 *      Output PostScript commands to print a text string. The string
 *      may be rotated at any arbitrary angle, and placed according
 *      the anchor type given. The anchor indicates how to interpret
 *      the window coordinates as an anchor for the text bounding box.
 *      If a background color is specified (i.e. bgColorPtr != NULL),
 *      output a filled rectangle the size of the bounding box in the
 *      given background color.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      Text string is drawn using the given font and GC on the graph
 *      window at the given coordinates, anchor, and rotation
 *
 * -----------------------------------------------------------------
 */
void
Blt_PrintText(graphPtr, text, attrPtr, x, y)
    Graph *graphPtr;
    char *text;			/* Text to convert to PostScript */
    TextAttributes *attrPtr;	/* Text attribute information */
    int x, y;			/* Window coordinates to draw text */
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    Point2D center;		/* Upper left corner of the text region */
    double theta;
    int width, height;
    int bbWidth, bbHeight;
    register char *src, *dest;
    int count;
    TextExtents *extentsPtr;
    TextSegment *textPtr;
    register int i;

    if ((text == NULL) || (*text == '\0')) {	/* Empty string, do nothing */
	return;
    }
    theta = FMOD(attrPtr->theta, (double)360.0);
    extentsPtr = Blt_GetCompoundTextExtents(attrPtr->font, text, attrPtr->leader,
	attrPtr->justify);
    width = extentsPtr->width + PADDING(attrPtr->padX);
    height = extentsPtr->height + PADDING(attrPtr->padY);
    Blt_GetBoundingBox(width, height, theta, &bbWidth, &bbHeight,
	(XPoint *)NULL);
    /*
     * Find the center of the bounding box
     */
    center = Blt_TranslateBoxCoords(x, y, bbWidth, bbHeight, attrPtr->anchor);
    center.x += (bbWidth * 0.5);
    center.y += (bbHeight * 0.5);

    /* Initialize text (sets translation and rotation) */
    Blt_PostScriptAppend(graphPtr, "%d %d %g %g %g BeginText\n", width, height, 
	attrPtr->theta, center.x, center.y);

    /* If a background is needed, draw a rectangle first before the text */
    if (attrPtr->bgColorPtr != (XColor *)NULL) {
	Blt_BackgroundToPostScript(graphPtr, attrPtr->bgColorPtr);
	Blt_RectangleToPostScript(graphPtr, 0, 0, width, height);
    }
    FontToPostScript(graphPtr, attrPtr->font);
    Blt_ForegroundToPostScript(graphPtr, attrPtr->fgColorPtr);
    y = attrPtr->padTop;
    x = attrPtr->padLeft;
    textPtr = extentsPtr->textSegArr;
    for (i = 0; i < extentsPtr->numSegments; i++, textPtr++) {
	if (textPtr->numChars < 1) {
	    continue;
	}
	Tcl_DStringAppend(psPtr->dsPtr, "(", -1);
	count = 0;
	dest = psPtr->scratchPtr;
	src = textPtr->text;
	for (i = 0; i < textPtr->numChars; i++) {
	    if (count > BUFSIZ) {
		*dest = '\0', dest = psPtr->scratchPtr;
		Tcl_DStringAppend(psPtr->dsPtr, dest, -1);
		count = 0;
	    }
	    /*
	     * On the off chance that special PostScript characters are
	     * embedded in the text string, escape the following
	     * characters: "\", "(", and ")"
	     */
	    if ((*src == '\\') || (*src == '(') || (*src == ')')) {
		*dest++ = '\\';
	    }
	    *dest++ = *src++;
	    count++;
	}
	*dest = '\0';
	Tcl_DStringAppend(psPtr->dsPtr, psPtr->scratchPtr, -1);
	Blt_PostScriptAppend(graphPtr, ") %d %d %d DrawAdjText\n", textPtr->width, 
		x + textPtr->x, y + textPtr->y);
    }
    /* End text mode */
    free((char *)extentsPtr);
    Tcl_DStringAppend(psPtr->dsPtr, "EndText\n", -1);
}

/*
 * -----------------------------------------------------------------
 *
 * Blt_PrintLine --
 *
 *      Outputs PostScript commands to print a multi-segmented line.
 *      It assumes a procedure DashesProc was previously defined.
 *
 * Results:
 *      None.
 *
 * Side Effects:
 *      Line is printed.
 *
 * -----------------------------------------------------------------
 */
void
Blt_PrintLine(graphPtr, pointPtr, numPoints)
    Graph *graphPtr;
    register XPoint *pointPtr;
    int numPoints;
{
    register int i;

    if (numPoints <= 0) {
	return;
    }
    Blt_PostScriptAppend(graphPtr, " newpath %d %d moveto\n", pointPtr->x, 
	pointPtr->y);
    pointPtr++;
    for (i = 1; i < (numPoints - 1); i++, pointPtr++) {
	if (i % PS_MAXPATH) {
	    Blt_PostScriptAppend(graphPtr, " %d %d lineto\n", pointPtr->x, 
		pointPtr->y);
	} else {
	    Blt_PostScriptAppend(graphPtr,
		" %d %d lineto\nDashesProc stroke\n newpath %d %d moveto\n",
		pointPtr->x, pointPtr->y, pointPtr->x, pointPtr->y);
	}
    }
    Blt_PostScriptAppend(graphPtr, " %d %d lineto\n DashesProc stroke\n", 
	pointPtr->x, pointPtr->y);
}

void
Blt_PrintBitmap(graphPtr, bitmap, scaleX, scaleY)
    Graph *graphPtr;
    Pixmap bitmap;		/* Bitmap to be converted to PostScript */
    double scaleX, scaleY;
{
    int width, height;
    float scaledWidth, scaledHeight;

    Tk_SizeOfBitmap(graphPtr->display, bitmap, &width, &height);
    scaledWidth = (float)(width *scaleX);
    scaledHeight = (float)(height *scaleY);
    Blt_PostScriptAppend(graphPtr,
	 "  gsave\n    %g %g translate\n    %g %g scale\n",
	scaledWidth * -0.5, scaledHeight * 0.5, scaledWidth, -scaledHeight);
    Blt_PostScriptAppend(graphPtr, "    %d %d true [%d 0 0 %d 0 %d] {",
	width, height, width, -height, height);
    Blt_BitmapToPostScript(graphPtr, bitmap, width, height);
    Blt_PostScriptAppend(graphPtr, "    } imagemask\n  grestore\n");
}

void
Blt_DestroyPostScript(graphPtr)
    Graph *graphPtr;
{
    Tk_FreeOptions(configSpecs, (char *)graphPtr->postscript, graphPtr->display, 0);
    free((char *)graphPtr->postscript);
}

/*
 *----------------------------------------------------------------------
 *
 * CgetOp --
 *
 *----------------------------------------------------------------------
 */
/*ARGSUSED*/
static int
CgetOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;
    Tcl_Interp *interp;
    int argc;
    char *argv[];
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    if (Tk_ConfigureValue(interp, graphPtr->tkwin, configSpecs, (char *)psPtr,
	    argv[4], 0) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * ----------------------------------------------------------------------
 *
 * ConfigureOp --
 *
 *      This procedure is invoked to print the graph in a file.
 *
 * Results:
 *      A standard TCL result.
 *
 * Side effects:
 *      A new PostScript file is created.
 *
 * ----------------------------------------------------------------------
 */
static int
ConfigureOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;
    Tcl_Interp *interp;
    int argc;			/* Number of options in argv vector */
    char **argv;		/* Option vector */
{
    int flags = TK_CONFIG_ARGV_ONLY;
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    if (argc == 3) {
	return (Tk_ConfigureInfo(interp, graphPtr->tkwin, configSpecs,
		(char *)psPtr, (char *)NULL, flags));
    } else if (argc == 4) {
	return (Tk_ConfigureInfo(interp, graphPtr->tkwin, configSpecs,
		(char *)psPtr, argv[3], flags));
    }
    if (Tk_ConfigureWidget(interp, graphPtr->tkwin, configSpecs, argc - 3, argv + 3,
	    (char *)psPtr, flags) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 * --------------------------------------------------------------------------
 *
 * GetPageExtents --
 *
 * 	Calculate the bounding box required for the plot and its
 * 	padding.  First get the size of the plot (by default, it's the
 * 	size of graph's X window).  If the plot is bigger than the
 * 	designated paper size, or if the "-maxpect" option is turned
 * 	on, make the bounding box same size as the page.  The bounding
 * 	box will still maintain its padding, therefore the plot area
 * 	will grow or shrink.
 *
 * 	Since the values set are reliant upon the width and height of the
 *	graph, this must be called each time PostScript is generated.
 *
 * Results: None.
 *
 * Side Effects:
 *	graph->width and graph->height are set to the postscript plot
 *	Extents.
 *
 * --------------------------------------------------------------------------
 */
static void
GetPageExtents(graphPtr, pagePtr)
    Graph *graphPtr;
    PageExtents *pagePtr;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;

    if (psPtr->reqWidth > 0) {
	graphPtr->width = psPtr->reqWidth;
    }
    if (psPtr->reqHeight > 0) {
	graphPtr->height = psPtr->reqHeight;
    }
    /*
     * Calculate dimension of bounding box (plot + padding).  The
     * bounding box is always aligned with the page, so correct for
     * orientation.
     */
    if (psPtr->landscape) {
	pagePtr->bbWidth = graphPtr->height;
	pagePtr->bbHeight = graphPtr->width;
    } else {
	pagePtr->bbWidth = graphPtr->width;
	pagePtr->bbHeight = graphPtr->height;
    }
    pagePtr->bbWidth += PADDING(psPtr->padX);
    pagePtr->bbHeight += PADDING(psPtr->padY);

    /*
     * Calculate the size of the page.  If no requested size was made
     * (i.e. the request is zero), default the paper size to the size
     * of the bounding box.
     */
    if (psPtr->reqPaperWidth > 0) {
	pagePtr->width = psPtr->reqPaperWidth;
    } else {
	pagePtr->width = pagePtr->bbWidth;
    }
    if (psPtr->reqPaperHeight > 0) {
	pagePtr->height = psPtr->reqPaperHeight;
    } else {
	pagePtr->height = pagePtr->bbHeight;
    }
    /*
     * Reset the size of the bounding box if it's bigger than the page.
     */
    if (pagePtr->bbHeight > pagePtr->height) {
	if (psPtr->landscape) {
	    graphPtr->width = pagePtr->bbHeight - PADDING(psPtr->padY);
	} else {
	    graphPtr->height = pagePtr->bbHeight - PADDING(psPtr->padY);
	}
	pagePtr->bbHeight = pagePtr->height;
    }
    if (pagePtr->bbWidth > pagePtr->width) {
	if (psPtr->landscape) {
	    graphPtr->height = pagePtr->bbWidth - PADDING(psPtr->padX);
	} else {
	    graphPtr->width = pagePtr->bbWidth - PADDING(psPtr->padX);
	}
	pagePtr->bbWidth = pagePtr->width;
    }
    pagePtr->maxScale = 1.0;
    if (psPtr->maxpect) {
	float xScale, yScale, scale;
	int hPad, vPad;
	int plotWidth, plotHeight;

	hPad = PADDING(psPtr->padX);
	vPad = PADDING(psPtr->padY);
	plotWidth = (pagePtr->bbWidth - hPad);
	plotHeight = (pagePtr->bbHeight - vPad);
	xScale = (float)(pagePtr->width - hPad) / plotWidth;
	yScale = (float)(pagePtr->height - vPad) / plotHeight;
	scale = MIN(xScale, yScale);
	pagePtr->bbWidth = (int)(plotWidth * scale + 0.5) + hPad;
	pagePtr->bbHeight = (int)(plotHeight * scale + 0.5) + vPad;
	pagePtr->maxScale = scale;
    }
    pagePtr->adjustX = pagePtr->adjustY = 0;
    if (psPtr->center) {
	if (pagePtr->width > pagePtr->bbWidth) {
	    pagePtr->adjustX = (pagePtr->width - pagePtr->bbWidth) / 2;
	}
	if (pagePtr->height > pagePtr->bbHeight) {
	    pagePtr->adjustY = (pagePtr->height - pagePtr->bbHeight) / 2;
	}
    }
}

/*
 * --------------------------------------------------------------------------
 *
 * PrintPreamble
 *
 *    	The PostScript preamble calculates the needed translation and scaling
 *    	to make X11 coordinates compatible with PostScript.
 *
 * ---------------------------------------------------------------------
 */

#ifdef TIME_WITH_SYS_TIME
#include <sys/time.h>
#include <time.h>
#else
#ifdef HAVE_SYS_TIME_H
#include <sys/time.h>
#else
#include <time.h>
#endif /* HAVE_SYS_TIME_H */
#endif /* TIME_WITH_SYS_TIME */

static int
PrintPreamble(graphPtr, pagePtr, fileName)
    Graph *graphPtr;
    PageExtents *pagePtr;
    char *fileName;
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    long date;
    char *version;
    float xMin, yMax, xMax, yMin;
    float dpiX, dpiY;
    float yScale, xScale;	/* Scales to convert pixels to pica */
    Screen *screenPtr;
    Tcl_DString filePath;
    char *libDir, *pathToProlog;
    FILE *f;

    if (fileName == NULL) {
	fileName = Tk_PathName(graphPtr->tkwin);
    }
    Blt_PostScriptAppend(graphPtr, "%%!PS-Adobe-3.0 EPSF-3.0\n\
%%%%Pages: 1\n%%%%Title: (%s)\n\
%%%%DocumentNeededResources: font Helvetica Courier\n", fileName);
    /*
     * Compute the scale factors to convert PostScript to X11 coordinates.
     * Round the pixels per inch (dpi) to an integral value before computing
     * the scale.
     */
#define MM_INCH (float)25.4
#define PICA_INCH (float)72.0
    screenPtr = Tk_Screen(graphPtr->tkwin);
    dpiX = (WidthOfScreen(screenPtr) * MM_INCH) / WidthMMOfScreen(screenPtr);
    dpiY = (HeightOfScreen(screenPtr) * MM_INCH) / HeightMMOfScreen(screenPtr);
    xScale = PICA_INCH / ROUND(dpiX);
    yScale = PICA_INCH / ROUND(dpiY);

    /*
     * Calculate the bounding box for the plot.  The bounding box
     * contains the graph and any designated padding (-padx or -pady
     * options).  Compute the lower left and upper right coordinates
     * of the bounding box.
     */
    xMin = (float)(pagePtr->adjustX) * xScale;
    yMax = (float)(pagePtr->adjustY) * yScale;
    xMax = (float)(pagePtr->adjustX + pagePtr->bbWidth) * xScale;
    yMin = (float)(pagePtr->adjustY + pagePtr->bbHeight) * yScale;
    Blt_PostScriptAppend(graphPtr, "%%%%BoundingBox: %5.0f %5.0f %5.0f %5.0f\n",
	xMin, yMax, xMax, yMin);

    version = Tcl_GetVar(graphPtr->interp, "blt_version", TCL_GLOBAL_ONLY);
    if (version == NULL) {
	version = "???";
    }
    date = time((time_t *) NULL);
    Blt_PostScriptAppend(graphPtr, 
	"%%%%Creator: BLT %s (version %s)\n%%%%CreationDate: %s%%%%EndComments\n", 
	    Tk_Class(graphPtr->tkwin), version, ctime(&date));

    /*
     * Read a standard prolog file from file and append it to the
     * generated PostScript output stored in interp->result.
     */
    libDir = Tcl_GetVar(graphPtr->interp, "blt_library", TCL_GLOBAL_ONLY);
    if (libDir == NULL) {
	Tcl_AppendResult(graphPtr->interp, "couldn't find BLT script library:",
	    "global variable \"blt_library\" doesn't exist", (char *)NULL);
	return TCL_ERROR;
    }

    Tcl_DStringInit(&filePath);
    Tcl_DStringAppend(&filePath, libDir, -1);
    Tcl_DStringAppend(&filePath, "/bltGraph.pro", -1);
    pathToProlog = Tcl_DStringValue(&filePath);
    f = fopen(pathToProlog, "r");
    if (f == NULL) {
	Tcl_AppendResult(graphPtr->interp, "couldn't open prologue file \"",
	    pathToProlog, "\": ", Tcl_PosixError(graphPtr->interp), (char *)NULL);
	return TCL_ERROR;
    }
    Blt_PostScriptAppend(graphPtr, "\n%% including file \"%s\"\n\n", pathToProlog);
    while (fgets(psPtr->scratchPtr, SCRATCH_LENGTH, f) != NULL) {
	Tcl_DStringAppend(psPtr->dsPtr, psPtr->scratchPtr, -1);
    }
    if (ferror(f)) {
	Tcl_AppendResult(graphPtr->interp, "error reading prologue file \"",
	 pathToProlog, "\": ", Tcl_PosixError(graphPtr->interp), (char *)NULL);
	fclose(f);
	Tcl_DStringFree(&filePath);
	return TCL_ERROR;
    }
    fclose(f);
    Tcl_DStringFree(&filePath);

    /*
     * Set the conversion from PostScript to X11 coordinates.  Scale
     * pica to pixels and flip the y-axis (the origin is the upperleft
     * corner).
     */
    Tcl_DStringAppend(psPtr->dsPtr,
	"% Transform coordinate system to use X11 coordinates\n\n\
% Flip the y-axis by changing the origin and reversing the scale,\n\
% making the origin to the upper left corner\n", -1);
    Blt_PostScriptAppend(graphPtr, "%f -%f scale\n0 %d translate\n\n", 
	xScale, yScale, -pagePtr->height);
    Tcl_DStringAppend(psPtr->dsPtr, "% User defined page layout\n\n", -1);
    
    Blt_PostScriptAppend(graphPtr, "%% Set color level\n/CL %d def\n\n", 
	psPtr->colorMode);
    Blt_PostScriptAppend(graphPtr, "%% Set origin\n%d %d translate\n\n",
	pagePtr->adjustX + psPtr->padLeft, pagePtr->adjustY + psPtr->padTop);
    if (psPtr->landscape) {
	Blt_PostScriptAppend(graphPtr, 
		 "%% Landscape orientation\n0 %g translate\n-90 rotate\n",
		 ((float)graphPtr->width * pagePtr->maxScale));
    }
    if (psPtr->maxpect) {
	Blt_PostScriptAppend(graphPtr, "\n%% Set max aspect ratio\n%g %g scale\n", 
		pagePtr->maxScale, pagePtr->maxScale);
    }
    Tcl_DStringAppend(psPtr->dsPtr, "\n%%EndSetup\n\n", -1);
    return TCL_OK;
}


static void
PrintExterior(graphPtr, psPtr)
    Graph *graphPtr;
    PostScript *psPtr;
{
    register int i;
    XRectangle rectArr[4];
    TextAttributes textAttr;

    rectArr[0].x = rectArr[0].y = rectArr[3].x = rectArr[1].x = 0;
    rectArr[0].width = rectArr[3].width = graphPtr->width;
    rectArr[0].height = graphPtr->topMargin;
    rectArr[3].y = (int)graphPtr->yMax;
    rectArr[3].height = graphPtr->bottomMargin;
    rectArr[2].y = rectArr[1].y = graphPtr->yMin;
    rectArr[1].width = graphPtr->leftMargin;
    rectArr[2].height = rectArr[1].height =
	(int)(graphPtr->yMax - graphPtr->yMin);
    rectArr[2].x = graphPtr->xMax;
    rectArr[2].width = graphPtr->rightMargin;

    /* Clear the surrounding margins and clip the plotting surface */
    if (psPtr->decorations) {
	Blt_BackgroundToPostScript(graphPtr,
	    Tk_3DBorderColor(graphPtr->border));
    } else {
	Blt_ClearBackgroundToPostScript(graphPtr);
    }
    for (i = 0; i < 4; i++) {
	Blt_RectangleToPostScript(graphPtr, rectArr[i].x, rectArr[i].y,
	    (int)rectArr[i].width, (int)rectArr[i].height);
    }
    /* Interior 3D border */
    if ((psPtr->decorations) && (graphPtr->plotBW > 0)) {
	int x, y, width, height;

	x = graphPtr->xMin - graphPtr->plotBW;
	y = graphPtr->yMin - graphPtr->plotBW;
	width = (graphPtr->xMax - graphPtr->xMin) + (2 * graphPtr->plotBW);
	height = (graphPtr->yMax - graphPtr->yMin) + (2 * graphPtr->plotBW);
	Blt_Print3DRectangle(graphPtr, graphPtr->border, x, y, width, height,
	    graphPtr->plotBW, graphPtr->plotRelief);
    }
    if (GetLegendSite(graphPtr) < LEGEND_SITE_PLOT) {
	/*
	 * Print the legend if we're using a site which lies in one
	 * of the margins (left, right, top, or bottom) of the graph.
	 */
	Blt_PrintLegend(graphPtr);
    }
    Blt_SetTextAttributes(&textAttr, graphPtr->marginFg, graphPtr->font, 0.0,
	TK_ANCHOR_CENTER, graphPtr->justify);

    if (graphPtr->title != NULL) {
	Blt_PrintText(graphPtr, graphPtr->title, &textAttr, graphPtr->titleX,
	    graphPtr->titleY);
    }
    Blt_PrintAxes(graphPtr);
}

/*
 *----------------------------------------------------------------------
 *
 * OutputOp --
 *
 *      This procedure is invoked to print the graph in a file.
 *
 * Results:
 *      Standard TCL result.  TCL_OK if plot was successfully printed,
 *	TCL_ERROR otherwise.
 *
 * Side effects:
 *      A new PostScript file is created.
 *
 *----------------------------------------------------------------------
 */
static int
OutputOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    Tcl_Interp *interp;
    int argc;			/* Number of options in argv vector */
    char **argv;		/* Option vector */
{
    PostScript *psPtr = (PostScript *)graphPtr->postscript;
    Legend *legendPtr = graphPtr->legendPtr;
    int x, y, width, height;
    int result = TCL_ERROR;
    PageExtents pageExtents;
    LegendSite site;
    Tcl_DString postScriptDStr;
    char scratchSpace[SCRATCH_LENGTH + 1];
    FILE *f = NULL;
    char *fileName;		/* Name of file to write PostScript output
                                 * If NULL, output is returned via
                                 * interp->result. */

    if (graphPtr->flags & RESET_AXES) {
	Blt_ResetAxes(graphPtr);
    }
    fileName = NULL;
    psPtr->scratchPtr = scratchSpace;
    psPtr->dsPtr = &postScriptDStr;
    if (argc > 3) {
	if (argv[3][0] != '-') {
	    fileName = argv[3];	/* First argument is the file name. */
	    argv++, argc--;
	}
	if (Tk_ConfigureWidget(interp, graphPtr->tkwin, configSpecs, argc - 3,
		argv + 3, (char *)psPtr, TK_CONFIG_ARGV_ONLY) != TCL_OK) {
	    return TCL_ERROR;
	}
	if (fileName != NULL) {
	    f = fopen(fileName, "w");
	    if (f == NULL) {
		Tcl_AppendResult(interp, "can't create \"", fileName, "\": ",
		    Tcl_PosixError(interp), (char *)NULL);
		return TCL_ERROR;
	    }
	}
    }
    GetPageExtents(graphPtr, &pageExtents);
    Blt_LayoutGraph(graphPtr);
    graphPtr->flags |= COORDS_WORLD;
    Blt_TransformGraph(graphPtr);

    Tcl_DStringInit(psPtr->dsPtr);

    result = PrintPreamble(graphPtr, &pageExtents, fileName);
    if (result != TCL_OK) {
	goto error;
    }
    /*
     * Determine rectangle of the plotting area for the graph window
     */
    x = graphPtr->xMin - graphPtr->plotBW;
    y = graphPtr->yMin - graphPtr->plotBW;
    width = (graphPtr->xMax - graphPtr->xMin) + (2 * graphPtr->plotBW);
    height = (graphPtr->yMax - graphPtr->yMin) + (2 * graphPtr->plotBW);

    FontToPostScript(graphPtr, graphPtr->font);
    if (psPtr->decorations) {
	Blt_BackgroundToPostScript(graphPtr, graphPtr->plotBg);
    } else {
	Blt_ClearBackgroundToPostScript(graphPtr);
    }
    Blt_RectangleToPostScript(graphPtr, x, y, width, height);
    Tcl_DStringAppend(psPtr->dsPtr, "gsave clip\n\n", -1);
    /*
     * Draw the grid, elements, and markers in the interior of the graph
     * (plotting surface).
     */
    site = GetLegendSite(graphPtr);

    if ((site >= LEGEND_SITE_PLOT) && (!legendPtr->raised)) {
	/* Print legend underneath elements and markers */
	Blt_PrintLegend(graphPtr);
    }
    if (!graphPtr->gridPtr->hidden) {
	Blt_PrintGrid(graphPtr);
    }
    Blt_PrintMarkers(graphPtr, TRUE);
    Blt_PrintAxisLimits(graphPtr);
    Blt_PrintElements(graphPtr);
    if ((site >= LEGEND_SITE_PLOT) && (legendPtr->raised)) {
	/* Print legend above elements (but not markers) */
	Blt_PrintLegend(graphPtr);
    }
    Blt_PrintMarkers(graphPtr, FALSE);
    Blt_PrintActiveElements(graphPtr);
    Tcl_DStringAppend(psPtr->dsPtr, "\n% Unset clipping\ngrestore\n\n", -1);
    PrintExterior(graphPtr, psPtr);
    Tcl_DStringAppend(psPtr->dsPtr, "showpage\n%Trailer\ngrestore\nend\n%EOF\n",
	 -1);
    /*
     * If a file name was given, write the results to that file
     */
    if (f != NULL) {
	fputs(Tcl_DStringValue(psPtr->dsPtr), f);
	if (ferror(f)) {
	    Tcl_AppendResult(interp, "error writing file \"", fileName, "\": ",
		Tcl_PosixError(interp), (char *)NULL);
	    goto error;
	}
    } else {
	Tcl_DStringResult(interp, psPtr->dsPtr);
    }
    result = TCL_OK;
  error:
    if (f != NULL) {
	fclose(f);
    }
    /* Reset height and width of graph window */
    graphPtr->width = Tk_Width(graphPtr->tkwin);
    graphPtr->height = Tk_Height(graphPtr->tkwin);
    graphPtr->flags = COORDS_WORLD;
    /*
     * Redraw the graph in order to re-calculate the layout as soon as
     * possible. This is in the case the crosshairs are active.
     */
    Blt_RedrawGraph(graphPtr);
    Tcl_DStringFree(psPtr->dsPtr);
    return result;
}

/*
 *----------------------------------------------------------------------
 *
 * Blt_CreatePostScript --
 *
 *      Creates a postscript structure.
 *
 * Results:
 *      Always TCL_OK.
 *
 * Side effects:
 *      A new PostScript structure is created.
 *
 *----------------------------------------------------------------------
 */
int
Blt_CreatePostScript(graphPtr)
    Graph *graphPtr;
{
    PostScript *psPtr;

    psPtr = (PostScript *)calloc(1, sizeof(PostScript));
    assert(psPtr);
    psPtr->colorMode = PS_MODE_COLOR;
    psPtr->center = TRUE;
    psPtr->decorations = TRUE;
    graphPtr->postscript = psPtr;

    if (Blt_ConfigureWidgetComponent(graphPtr->interp, graphPtr->tkwin,
	    "Postscript", "postscript", configSpecs, 0, (char **)NULL,
	    (char *)psPtr, 0) != TCL_OK) {
	return TCL_ERROR;
    }
    return TCL_OK;
}

/*
 *--------------------------------------------------------------
 *
 * Blt_PostScriptOp --
 *
 *	This procedure is invoked to process the Tcl command
 *	that corresponds to a widget managed by this module.
 *	See the user documentation for details on what it does.
 *
 * Results:
 *	A standard Tcl result.
 *
 * Side effects:
 *	See the user documentation.
 *
 *--------------------------------------------------------------
 */
static Blt_OpSpec psOps[] =
{
    {"cget", 2, (Blt_Operation)CgetOp, 3, 3, "option",},
    {"configure", 2, (Blt_Operation)ConfigureOp, 2, 0, "?option value?...",},
    {"output", 1, (Blt_Operation)OutputOp, 2, 0,
	"?fileName? ?option value?...",},
};

static int numPsOps = sizeof(psOps) / sizeof(Blt_OpSpec);

int
Blt_PostScriptOp(graphPtr, interp, argc, argv)
    Graph *graphPtr;		/* Graph widget record */
    Tcl_Interp *interp;
    int argc;			/* # arguments */
    char **argv;		/* Argument list */
{
    Blt_Operation proc;
    int result;

    proc = Blt_LookupOperation(interp, numPsOps, psOps, BLT_OPER_ARG2,
	argc, argv);
    if (proc == NULL) {
	return TCL_ERROR;
    }
    result = (*proc) (graphPtr, interp, argc, argv);
    return (result);
} 
