/***

serv.c  - TCP/UDP port statistics module
Written by Gerard Paul Java
Copyright (c) Gerard Paul Java 1997, 1998

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

***/

#include <curses.h>
#include <panel.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/socket.h>
#include <netinet/ip.h>
#include <netinet/udp.h>
#include <linux/if_packet.h>
#include "if_ether.h"
#include "tcphdr.h"
#include "dirs.h"
#include "ipcsum.h"
#include "deskman.h"
#include "packet.h"
#include "stdwinset.h"
#include "input.h"
#include "attrs.h"
#include "serv.h"
#include "error.h"
#include "log.h"
#include "timer.h"

#define SCROLLUP 0
#define SCROLLDOWN 1

extern void open_socket(int *fd);
extern void adjustpacket(char *tpacket, unsigned short family, char **packet,
			 int *readlen);
extern void writeutslog(struct portlistent *list, unsigned long nsecs, FILE * logfile);

void initportlist(struct portlist *list)
{
    list->head = list->tail = NULL;
    list->firstvisible = list->lastvisible = NULL;
    list->count = 0;

    list->borderwin = newwin(14, 80, 6, 0);
    list->borderpanel = new_panel(list->borderwin);
    wattrset(list->borderwin, BOXATTR);
    box(list->borderwin, ACS_VLINE, ACS_HLINE);

    wmove(list->borderwin, 0, 1);
    wprintw(list->borderwin, " Proto/Port ");
    wmove(list->borderwin, 0, 16);
    wprintw(list->borderwin, " Pkts ");
    wmove(list->borderwin, 0, 26);
    wprintw(list->borderwin, " Bytes ");
    wmove(list->borderwin, 0, 34);
    wprintw(list->borderwin, " PktsTo ");
    wmove(list->borderwin, 0, 44);
    wprintw(list->borderwin, " BytesTo ");
    wmove(list->borderwin, 0, 52);
    wprintw(list->borderwin, " PktsFrom ");
    wmove(list->borderwin, 0, 62);
    wprintw(list->borderwin, " BytesFrom ");

    list->win = newwin(12, 78, 7, 1);
    list->panel = new_panel(list->win);

    stdwinset(list->win);
    wtimeout(list->win, -1);
    wattrset(list->win, STDATTR);
    colorwin(list->win);
    update_panels();
    doupdate();
}

struct portlistent *addtoportlist(struct portlist *list, unsigned int protocol,
				  unsigned int port, int *nomem)
{
    struct portlistent *ptemp;

    ptemp = malloc(sizeof(struct portlistent));

    if (ptemp == NULL) {
	printnomem();
	*nomem = 1;
	return NULL;
    }
    if (list->head == NULL) {
	ptemp->prev_entry = NULL;
	list->head = ptemp;
	list->firstvisible = ptemp;
    }
    if (list->tail != NULL) {
	list->tail->next_entry = ptemp;
	ptemp->prev_entry = list->tail;
    }
    list->tail = ptemp;
    ptemp->next_entry = NULL;

    ptemp->protocol = protocol;
    ptemp->port = port;
    ptemp->count = ptemp->bcount = 0;
    ptemp->icount = ptemp->ibcount = 0;
    ptemp->ocount = ptemp->obcount = 0;

    list->count++;
    ptemp->idx = list->count;

    if (list->count <= 12)
	list->lastvisible = ptemp;

    wmove(list->borderwin, 13, 1);
    wprintw(list->borderwin, " %u entries ", list->count);

    return ptemp;
}

int portinlist(struct porttab *table, unsigned int port)
{
    struct porttab *ptmp = table;

    while (ptmp != NULL) {
	if (ptmp->port == port)
	    return 1;

	ptmp = ptmp->next_entry;
    }

    return 0;
}

int goodport(unsigned int port, struct porttab *table)
{
    return ((port < 1024) || (portinlist(table, port)));
}

struct portlistent *inportlist(struct portlist *list,
			       unsigned int protocol, unsigned int port)
{
    struct portlistent *ptmp = list->head;

    while (ptmp != NULL) {
	if ((ptmp->port == port) && (ptmp->protocol == protocol))
	    return ptmp;

	ptmp = ptmp->next_entry;
    }

    return NULL;
}

void printportent(struct portlist *list, struct portlistent *entry,
		  unsigned int idx)
{
    unsigned int target_row;

    if ((entry->idx < idx) || (entry->idx > idx + 11))
	return;

    target_row = entry->idx - idx;

    wattrset(list->win, STDATTR);
    wmove(list->win, target_row, 1);
    if (entry->protocol == IPPROTO_TCP)
	wprintw(list->win, "TCP");
    else if (entry->protocol == IPPROTO_UDP)
	wprintw(list->win, "UDP");

    wprintw(list->win, "/%u: ", entry->port);
    wattrset(list->win, HIGHATTR);
    wmove(list->win, target_row, 12);
    wprintw(list->win, "%8lu %10lu %8lu %10lu %8lu %10lu", entry->count,
	    entry->bcount, entry->icount, entry->ibcount, entry->ocount,
	    entry->obcount);
}

void destroyportlist(struct portlist *list)
{
    struct portlistent *ptmp = list->head;
    struct portlistent *ctmp = NULL;

    if (list->head != NULL)
	ctmp = list->head->next_entry;

    while (ptmp != NULL) {
	free(ptmp);
	ptmp = ctmp;

	if (ctmp != NULL)
	    ctmp = ctmp->next_entry;
    }
}

void printport(struct portlist *list, unsigned int protocol, unsigned int port,
	     int br, unsigned int idx, int fromto, struct porttab *ports)
{
    struct portlistent *listent;
    int nomem = 0;

    if (goodport(port, ports)) {
	listent = inportlist(list, protocol, port);

	if ((listent == NULL) && (!nomem))
	    listent = addtoportlist(list, protocol, port, &nomem);

	if (listent == NULL)
	    return;

	listent->count++;
	listent->bcount += br;

	if (fromto == 0) {	/* from */
	    listent->obcount += br;
	    listent->ocount++;
	} else {		/* to */
	    listent->ibcount += br;
	    listent->icount++;
	}

	printportent(list, listent, idx);
    }
}

void scrollservwin(struct portlist *table, int direction, int *idx)
{
    wattrset(table->win, STDATTR);
    if (direction == SCROLLUP) {
	if (table->lastvisible != table->tail) {
	    wscrl(table->win, 1);
	    table->lastvisible = table->lastvisible->next_entry;
	    table->firstvisible = table->firstvisible->next_entry;
	    (*idx)++;
	    wmove(table->win, 11, 0);
	    scrollok(table->win, 0);
	    wprintw(table->win, "%78c", 32);
	    scrollok(table->win, 1);
	    printportent(table, table->lastvisible, *idx);
	}
    } else {
	if (table->firstvisible != table->head) {
	    wscrl(table->win, -1);
	    table->lastvisible = table->lastvisible->prev_entry;
	    table->firstvisible = table->firstvisible->prev_entry;
	    (*idx)--;
	    wmove(table->win, 0, 0);
	    wprintw(table->win, "%78c", 32);
	    printportent(table, table->firstvisible, *idx);
	}
    }
}

void pageservwin(struct portlist *table, int direction, int *idx)
{
    int i = 1;

    if (direction == SCROLLUP) {
	while ((i <= 9) && (table->lastvisible != table->tail)) {
	    i++;
	    scrollservwin(table, direction, idx);
	}
    } else {
	while ((i <= 9) && (table->firstvisible != table->head)) {
	    i++;
	    scrollservwin(table, direction, idx);
	}
    }
}

void servmon(char *ifname, struct porttab *ports, int logging, time_t logspan)
{
    int fd;
    char buf[8192];
    char *ipacket;
    char *tpacket;

    unsigned int iphlen;
    struct sockaddr_pkt fromaddr;
    int br;
    int exitloop = 0;

    unsigned int idx = 1;

    unsigned int sport = 0;
    unsigned int dport = 0;

    time_t starttime, startlog, timeint, now;

    int csum;
    int ck;

    int ch;

    struct portlist list;

    FILE *logfile = NULL;

    initportlist(&list);

    move(24, 1);
    printkeyhelp("Up/Down/PgUp/PgDn", "-scroll window  ", stdscr);
    stdexitkeyhelp();

    update_panels();
    doupdate();

    if (logging) {
	logfile = opentlog();

	if (logfile == NULL)
	    logging = 0;
    }
    writelog(logging, logfile, "******** TCP/UDP service monitor started");

    open_socket(&fd);

    starttime = startlog = timeint = time((time_t *) NULL);

    do {
	getpacket(fd, buf, &fromaddr, &ch, &br, 5, list.win);

	if (ch != ERR)
	    switch (ch) {
	    case KEY_UP:
		scrollservwin(&list, SCROLLDOWN, &idx);
		break;
	    case KEY_DOWN:
		scrollservwin(&list, SCROLLUP, &idx);
		break;
	    case KEY_PPAGE:
		pageservwin(&list, SCROLLDOWN, &idx);
		break;
	    case KEY_NPAGE:
		pageservwin(&list, SCROLLUP, &idx);
		break;
	    case 'q':
	    case 'Q':
	    case 'x':
	    case 'X':
	    case 24:
		exitloop = 1;
	    }
	if (br > 0) {
	    if ((fromaddr.spkt_protocol == ntohs(ETH_P_IP)) &&
		(strcmp(ifname, fromaddr.spkt_device) == 0)) {
		adjustpacket(buf, fromaddr.spkt_family, &ipacket,
			     &br);

		if (ipacket == NULL)
		    continue;

		iphlen = ((struct iphdr *) ipacket)->ihl * 4;
		ck = ((struct iphdr *) ipacket)->check;
		((struct iphdr *) ipacket)->check = 0;
		csum = in_cksum((u_short *) ipacket, iphlen);

		if (ck != csum)
		    continue;

		tpacket = ipacket + iphlen;

		if (((struct iphdr *) ipacket)->protocol == IPPROTO_TCP) {
		    sport = ntohs(((struct tcphdr *) tpacket)->source);
		    dport = ntohs(((struct tcphdr *) tpacket)->dest);
		} else if (((struct iphdr *) ipacket)->protocol == IPPROTO_UDP) {
		    sport = ntohs(((struct udphdr *) tpacket)->source);
		    dport = ntohs(((struct udphdr *) tpacket)->dest);
		}
		if ((((struct iphdr *) ipacket)->protocol == IPPROTO_TCP) ||
		    (((struct iphdr *) ipacket)->protocol == IPPROTO_UDP)) {
		    printport(&list, ((struct iphdr *) ipacket)->protocol,
		       sport, ntohs(((struct iphdr *) ipacket)->tot_len),
			      idx, 0, ports);
		    printport(&list, ((struct iphdr *) ipacket)->protocol,
		       dport, ntohs(((struct iphdr *) ipacket)->tot_len),
			      idx, 1, ports);
		}
	    }
	}
		
	now = time((time_t *) NULL);
	
	if (now - timeint >= 5) {
	    printelapsedtime(starttime, now, 13, 20, list.borderwin);
	    timeint = now;
	}
	    
	if (((now - startlog) >= logspan) && (logging)) {
	    writeutslog(list.head, now - starttime, logfile);
	    startlog = now;
	}
	
	update_panels();
	doupdate();
    } while (!exitloop);

    if (logging) {
	writeutslog(list.head, time((time_t *) NULL) - starttime, logfile);
	writelog(logging, logfile, "******** TCP/UDP service monitor stopped");
	fclose(logfile);
    }
    del_panel(list.panel);
    delwin(list.win);
    del_panel(list.borderpanel);
    delwin(list.borderwin);
    update_panels();
    doupdate();
    destroyportlist(&list);
}

void portdlg(unsigned int *port, int *aborted, int mode)
{
    WINDOW *bw;
    PANEL *bp;
    WINDOW *win;
    PANEL *panel;

    struct FIELDLIST list;

    bw = newwin(12, 47, 5, 5);
    bp = new_panel(bw);

    win = newwin(10, 45, 6, 6);
    panel = new_panel(win);

    wattrset(bw, BOXATTR);
    box(bw, ACS_VLINE, ACS_HLINE);

    wattrset(win, STDATTR);
    colorwin(win);
    stdwinset(win);
    wtimeout(win, -1);

    if (mode == 0) {
	wmove(win, 1, 1);
	wprintw(win, "Port numbers below 1024 are reserved");
	wmove(win, 2, 1);
	wprintw(win, "for TCP/IP services, and are normally");
	wmove(win, 3, 1);
	wprintw(win, "the only ones monitored by the TCP/UDP");
	wmove(win, 4, 1);
	wprintw(win, "statistics module.  If you wish to monitor");
	wmove(win, 5, 1);
	wprintw(win, "a higher-numbered port, enter it here.");
    } else {
	wmove(win, 1, 1);
	wprintw(win, "Enter the port number you wish to remove");
	wmove(win, 2, 1);
	wprintw(win, "from the list you created.  You cannot");
	wmove(win, 3, 1);
	wprintw(win, "remove ports numbered less then 1024");
	wmove(win, 4, 1);
	wprintw(win, "though.");
    }

    wmove(win, 9, 1);
    stdkeyhelp(win);

    initfields(&list, 1, 6, 13, 8);
    addfield(&list, 5, 0, 0, "");

    fillfields(&list, aborted);

    if (!(*aborted))
	*port = atoi(list.list->buf);

    del_panel(bp);
    delwin(bw);
    del_panel(panel);
    delwin(win);
    destroyfields(&list);

}

void saveportlist(struct porttab *table)
{
    struct porttab *ptmp = table;
    int fd;
    int bw;
    int resp;

    fd = open(PORTFILE, O_WRONLY | O_TRUNC | O_CREAT, S_IRUSR | S_IWUSR);

    if (fd < 0) {
	errbox("Unable to open port list file", ANYKEY_MSG, &resp);
	return;
    }
    while (ptmp != NULL) {
	bw = write(fd, &(ptmp->port), sizeof(unsigned int));
	ptmp = ptmp->next_entry;
    }

    close(fd);
}

void addmoreports(struct porttab **table)
{
    unsigned int port;
    int aborted;
    int resp;
    struct porttab *ptmp;

    portdlg(&port, &aborted, 0);

    if (!aborted) {
	if (portinlist(*table, port))
	    errbox("Duplicate port entered", ANYKEY_MSG, &resp);
	else {
	    ptmp = malloc(sizeof(struct porttab));

	    ptmp->port = port;
	    ptmp->prev_entry = NULL;
	    ptmp->next_entry = *table;

	    if (*table != NULL)
		(*table)->prev_entry = ptmp;

	    *table = ptmp;
	    saveportlist(*table);
	}
    }
    update_panels();
    doupdate();
}

void loadaddports(struct porttab **table)
{
    int fd;
    struct porttab *ptemp;
    struct porttab *tail = NULL;
    int br;
    int resp;

    *table = NULL;

    fd = open(PORTFILE, O_RDONLY);
    if (fd < 0)
	return;

    do {
	ptemp = malloc(sizeof(struct porttab));

	br = read(fd, &(ptemp->port), sizeof(unsigned int));

	if (br < 0) {
	    errbox("Error reading port list", ANYKEY_MSG, &resp);
	    close(fd);
	    return;
	}
	if (br > 0) {
	    if (*table == NULL) {
		*table = ptemp;
		ptemp->prev_entry = NULL;
	    }
	    if (tail != NULL) {
		tail->next_entry = ptemp;
		ptemp->prev_entry = tail;
	    }
	    tail = ptemp;
	    ptemp->next_entry = NULL;
	} else
	    free(ptemp);

    } while (br > 0);

    close(fd);
}

void delport(struct porttab **table, unsigned int port)
{
    struct porttab *ptmp = *table;

    while ((ptmp->port != port) && (ptmp != NULL))
	ptmp = ptmp->next_entry;

    if (ptmp != NULL) {
	if (ptmp == *table) {
	    *table = (*table)->next_entry;
	    if (*table != NULL)
		(*table)->prev_entry = NULL;
	} else {
	    ptmp->prev_entry->next_entry = ptmp->next_entry;

	    if (ptmp->next_entry != NULL)
		ptmp->next_entry->prev_entry = ptmp->prev_entry;
	}

	free(ptmp);
    }
}

void removeaport(struct porttab **table)
{
    unsigned int port;
    unsigned int aborted;
    int resp;

    portdlg(&port, &aborted, 1);

    if (!aborted) {
	if (portinlist(*table, port)) {
	    delport(table, port);
	    saveportlist(*table);
	} else
	    errbox("Unknown port value", ANYKEY_MSG, &resp);
    }
}


void destroyporttab(struct porttab *table)
{
    struct porttab *ptemp = table;
    struct porttab *ctemp;

    if (ptemp != NULL)
	ctemp = ptemp->next_entry;

    while (ptemp != NULL) {
	free(ptemp);
	ptemp = ctemp;

	if (ctemp != NULL)
	    ctemp = ctemp->next_entry;
    }
}
