/* midi2abc - converts MIDI file to abc format files
 * 
 * version 1.6.4 (27th September 1996)
 *
 * written by James Allwright 5th May 1996
 * Department of Electronics and Computer Science,
 * University of Southampton, UK
 * Comments to jra@ecs.soton.ac.uk
 * 
 * based on public domain 'midifilelib' package.
 *
 * This code may be freely distributed.
 */

#include <stdio.h>
#ifndef PCCFIX
#include <ctype.h>
#endif
#include "midifile.h"

/* The following can be increased if they are not large enough */
/* to cope with the MIDI files to be converted */
#define MAXNOTES 2000
#define NTEXT 100
#define MAXTEXT 3000
static FILE *F;
int division;    /* from the file header */
long tempo = 500000; /* the default tempo is 120 beats/minute */
int unitlen;
long laston = 0;
int length[MAXNOTES], aclen[MAXNOTES], npitch[MAXNOTES];
int vel[30];
char feature[MAXNOTES];
char text[MAXTEXT];
int text_place, text_count;
int textitem[NTEXT], texttuneplace[NTEXT];
int bar[MAXNOTES];
int lastpitch, lastlen, lastchan;
int notes = 0;
int num[MAXNOTES], denom[MAXNOTES], rest[MAXNOTES];
int trans[256], back[256];
char atog[256];
int symbol[256];
int key[12], middle;
int sharps;
int channel;
int track, trackno;
int format;
int xunit;
int extractm, extractl, extracta;
int asig, bsig;
int Qval;
int trackstart[30];

advance()
{
  textitem[text_count] = text_place;
  texttuneplace[text_count] = notes;
  text_count = text_count + 1;
  text_place = text_place + strlen(&text[text_place]) + 1;
  if (text_count > NTEXT) {
    printf("Aborting : too many text items\n");
    exit(0);
  };
  if (text_place > MAXTEXT) {
    printf("Aborting : too much text\n");
    exit(0);
  };
};

filegetc()
{
    return(getc(F));
}

/* for crack */
extern int arg_index;

int quantize(xunit)
int xunit;
{
  int j;
  int spare;
  int toterror;

  spare = 0;
  toterror = 0;
  for (j=0; j<notes; j++) {
    num[j] = (2*(length[j] + spare + (xunit/4)))/xunit;
    denom[j] = 2;
    spare = spare + length[j] - (num[j]*xunit/denom[j]);
    if (spare > 0) {
      toterror = toterror + spare;
    } else {
      toterror = toterror - spare;
    };
    /* gradually forget old errors so that if xunit is slightly off,
       errors don't accumulate over several bars */
    spare = (spare * 96)/100;
  };
  return(toterror);
};

printQ()
{
  float Tnote, freq;

  Tnote = mf_ticks2sec((long)xunit*unitlen/4, division, tempo);
  freq = 60.0/Tnote;
  printf("Q:1/4=%d\n", (int) (freq+0.5));
};

long countlen(start)
int start;
{
  long total;
  int j;

  total = 0;
  for (j=start; j<notes; j++) {
    total = total + length[j];
  };
  return (total);
};

long counttrack(anacrusis)
int anacrusis;
{
  long total;
  int j, trackno;

  /* find first non-empty track */
  trackno = 0;
  while ((trackstart[trackno+1] - trackstart[trackno]) == 0) {
    trackno = trackno + 1;
  };
  total = 0;
  /* count up actual units */
  for (j=trackstart[trackno]+anacrusis; j<trackstart[trackno+1]; j++) {
    total = total + length[j];
  };
  return (total);
};

void guesslengths()
{
  int i;
  int trial[100];
  float avlen, factor, tryx;
  int min;

  min = countlen(0);
  avlen = ((float)(min))/((float)notes);
  tryx = avlen * 0.75;
  factor = tryx/100;
  for (i=0; i<100; i++) {
    trial[i] = quantize((int) tryx);
    if (trial[i] < min) {
      min = trial[i];
      xunit = (int) tryx;
    };
    tryx = tryx + factor;
  };
};

int readnum(num) 
char *num;
{
  int t;
  char *p;
  
  t = 0;
  p = num;
  while (((int)*p >= '0') && ((int)*p <= '9')) {
    t = t * 10 + (int) *p - '0';
    p = p + 1;
  };
  return t;
};

int readnump(p) 
char **p;
{
  int t;
  
  t = 0;
  while (((int)**p >= '0') && ((int)**p <= '9')) {
    t = t * 10 + (int) **p - '0';
    *p = *p + 1;
  };
  return t;
};

void readsig(a, b, sig)
int *a, *b;
char *sig;
{
  char *p;
  int t;

  p = sig;
  if ((int)*p == 'C') {
    *a = 4;
    *b = 4;
    return;
  };
  *a = readnump(&p);
  if ((int)*p != '/') {
    printf("Expecting / in time signature found %c!\n", *p);
    exit(0);
  };
  p = p + 1;
  *b = readnump(&p);
  if ((*a == 0) || (*b == 0)) {
    printf("%d/%d is not a valid time signature!\n", *a, *b);
    exit(0);
  };
  t = *b;
  while (t > 1) {
    if (t%2 != 0) {
      printf("Bad key signature, divisor must be a power of 2!\n");
      exit(0);
    } else {
      t = t/2;
    };
  };
};

void findkey()
{
  int j;
  int max, min, n[12], key_score[12];
  int minkey, minblacks;
  char sharp[13], flat[13], shsymbol[13], flsymbol[13];
  int t, issharp;
  static int keysharps[12] = {0, -5, 2, -3, 4, -1, 6, 1, -4, 3, -2, 5};
   
  /* analyse pitches */
  /* find key */
  for (j=0; j<12; j++) {
    n[j] = 0;
  };
  min = npitch[0];
  max = npitch[0];
  for (j=0; j<notes; j++) {
    if (npitch[j] > max) {
      max = npitch[j];
    } else {
      if (npitch[j] < min) {
        min = npitch[j];
      };
    };
    n[npitch[j] % 12] = n[npitch[j] % 12] + 1;
  };
  /* count black notes for each key */
  /* assume pitch = 0 is C */
  minkey = 0;
  minblacks = notes;
  for (j=0; j<12; j++) {
    key[j] = 0;
    key_score[j] = n[(j+1)%12] + n[(j+3)%12] + n[(j+6)%12] +
                   n[(j+8)%12] + n[(j+10)%12];
    /* printf("Score for key %d is %d\n", j, key_score[j]); */
    if (key_score[j] < minblacks) {
      minkey = j;
      minblacks = key_score[j];
    };
  };
  /* Do last note analysis */
  /* switch to minor mode if it gives same number of accidentals */
  if ((minkey != ((lastpitch+3)%12)) && 
      (key_score[minkey] == key_score[(lastpitch+3)%12])) {
         minkey = (lastpitch+3)%12;
  };
  /* switch to major mode if it gives same number of accidentals */
  if ((minkey != (lastpitch%12)) && 
      (key_score[minkey] == key_score[lastpitch%12])) {
         minkey = lastpitch%12;
  };
  if (minkey != (lastpitch%12)) {
    printf("%% Last note suggests ");
    switch((lastpitch+12-minkey)%12) {
    case(2):
      printf("Dorian ");
      break;
    case(4):
      printf("Phrygian ");
      break;
    case(5):
      printf("Lydian ");
      break;
    case(7):
      printf("Mixolydian ");
      break;
    case(9):
      printf("minor ");
      break;
    case(11):
      printf("Locrian ");
      break;
    default:
      printf("unknown ");
      break;
    };
    printf("mode tune\n");
  };
  sharps = keysharps[minkey];
  strcpy(sharp,    "ccddeffggaab");
  strcpy(shsymbol, "=^=^==^=^=^=");
  if (sharps == 6) {
    sharp[6] = 'e';
    shsymbol[6] = '^';
  };
  strcpy(flat, "cddeefggaabb");
  strcpy(flsymbol, "=_=_==_=_=_=");
  /* Print out key */

  if (sharps >= 0) {
    if (sharps == 6) {
      printf("K:F#");
    } else {
      printf("K:%c", sharp[minkey] + 'A' - 'a');
    };
    issharp = 1;
  } else {
    if (sharps == -1) {
      printf("K:%c", flat[minkey] + 'A' - 'a');
    } else {
      printf("K:%cb", flat[minkey] + 'A' - 'a');
    };
    issharp = 0;
  };
  if (sharps >= 0) {
    printf(" %% %d sharps\n", sharps);
  } else {
    printf(" %% %d flats\n", -sharps);
  };
  /* do conversion to abc pitches */
  middle = (min + (max - min)/2 + 6)/12 * 12;
  /* printf("Middle = %d\n", middle); */
  key[(minkey+1)%12] = 1;
  key[(minkey+3)%12] = 1;
  key[(minkey+6)%12] = 1;
  key[(minkey+8)%12] = 1;
  key[(minkey+10)%12] = 1;
  for (j=0; j<256; j++) {
    t = j%12;
    if (issharp) {
      atog[j] = sharp[t];
      symbol[j] = shsymbol[t];
    } else {
      atog[j] = flat[t];
      symbol[j] = flsymbol[t];
    };
    trans[j] = 7*(j/12)+((int) atog[j] - 'a');
    if (j < middle) {
      atog[j] = (char) (int) atog[j] + 'A' - 'a';
    };
    if (key[t] == 0) {
      back[trans[j]] = j;
    };
  };
};

printpitch(j)
int j;
{
  int p, po;

  p = npitch[j];
  po = p % 12;
  /* printf("\npitch %d ", p); */
  if ((back[trans[p]] != p) || (key[po] == 1)) {
    printf("%c%c", symbol[po], atog[p]);
    back[trans[p]] = p;
  } else {
    printf("%c", atog[p]);
  };
  while (p >= middle + 12) {
    printf("'");
    p = p - 12;
  };
  while (p < middle - 12) {
    printf(",");
    p = p + 12;
  };
};

printfract(a, b)
int a, b;
{
  int c, d;

  c = a;
  d = b;
  /* print out length */
  if (((c % 2) == 0) && ((d % 2) == 0)) {
    c = c/2;
    d = d/2;
  };
  if (c != 1) {
    printf("%d", c);
  };
  if (d != 1) {
    printf("/%d", d);
  };
};

printlen(j)
int j;
{
  printfract(num[j], denom[j]);
};

setrest(j)
int j;
{
  int oldnum;

  oldnum = num[j];
  num[j] = (int) ((float) (num[j] * aclen[j])/length[j] + 0.5);
  if (num[j] == 0) num[j] = 1;
  rest[j] = oldnum - num[j];
};

straightnote(j)
int j;
{
  setrest(j);
  if ((num[j] == 1) && (rest[j] >= 1)) {
    printf(".");
    num[j] = num[j] + rest[j];
    rest[j] = 0;
  };
  printpitch(j);
  printlen(j);
  if (rest[j] > 0) {
    printf(" z");
    printfract(rest[j], denom[j]);
    if (feature[j+1] != ' ') {
      printf(" ");
    };
  };
};

setfeatures(bars, barsize, anacrusis)
int barsize, anacrusis;
{
  int j;
  int analen, anacount, trackno;
  int ncount, barcount, sincebar, naturalbreak;
  int cum_error, units;

/* calculate length of anacrusis */
  analen = 0;
  for (j=0; j<=anacrusis; j++) analen = analen + num[j];
/* work out fiddle factor to get right number of bars*/
  if (bars != 0) {
    /* find first non-empty track */
    trackno = 0;
    while ((trackstart[trackno+1] - trackstart[trackno]) == 0) {
      trackno = trackno + 1;
    };
    units = 0;
    /* count up actual units */
    for (j=trackstart[trackno]; j<trackstart[trackno+1]; j++) {
      units = units + num[j];
    };
    cum_error = (bars*barsize*2) - units;
  } else {
    cum_error = 0;
  };
/* add bar lines */
  for (j=0; j<notes; j++) {
    bar[j] = 0;
  };
  trackno = 0;
  barcount = 1;
/*  for (j=anacrusis; j<notes-1; j++) { */
  for (j=0; j<notes-1; j++) {
    if (j >= trackstart[trackno + 1]) {
      /* skip over empty tracks */
      while (j >= trackstart[trackno + 1]) trackno = trackno + 1;
      /* skip over anacrusis */
      anacount = num[j];
      while ((anacount < analen) && (j<notes-1)) {
        j = j + 1;
        anacount = anacount + num[j];
      };
      bar[j] = 1;
      ncount = 0;
    };
    ncount = ncount + num[j];
    if (ncount >= barsize*2) {
      if (ncount == barsize*2) {
        bar[j+1] = 1;
        ncount = 0;
      } else {
        int overflow;

        overflow = ncount - barsize*2;
        if ((ncount > num[j]) && ((2*overflow > num[j]) || 
            ((2*overflow == num[j]) && (cum_error > 0)))) {
          bar[j] = 1;
          ncount = num[j];
          if (num[j] > overflow) {
            printf("%% Short bar : missing %d/2 units in bar %d\n", 
                num[j]-overflow, barcount);
          } else {
            printf("%% Extended bar : extra %d/2 units in bar %d\n",
                -num[j]+overflow, barcount);
          };
          cum_error = cum_error - (num[j]-overflow);
        } else {
          bar[j+1] = 1;
          printf("%% Extended bar : extra %d/2 units in bar %d\n", 
                overflow, barcount);
          cum_error = cum_error + overflow;
          ncount = 0;
        };
      };
      barcount = barcount + 1;
    };
  };
  if (ncount+num[notes-1] != 2*barsize) {
    printf("%% Last bar contains %d/2 units not %d\n", 
           ncount+num[notes-1], barsize);
  };
  printf("%% Number of bars = %d\n", barcount);
/* add spaces */
  if ((barsize ==6) || (barsize == 9)) {
    naturalbreak = 6;
  } else {
    naturalbreak = 8;
  };
  sincebar = num[0];
  feature[0] = 'x';
  trackno = 0;
  for (j=1; j<notes; j++) {
    if (j >= trackstart[trackno + 1]) {
      /* skip over empty tracks */
      while (j >= trackstart[trackno + 1]) trackno = trackno + 1;
      if (j != 1) sincebar = 0;
    };
    if (bar[j] == 1) {
      feature[j] = 'x';
      sincebar = 0;
    } else {
      if (sincebar % naturalbreak == 0) {
        feature[j] = ' ';
      } else {
        feature[j] = 'x';
      };
    };
    sincebar = sincebar + num[j];
  };
  /* put in chords */
  for (j=0; j<notes; j++) {
    if (num[j] == 0) {
       feature[j] = '+';
    };
  };
/* look for triplets and broken rhythm */
  if ((barsize%2 == 0) && (barsize != 6)) {
    int start, len;

    start = 0;
    j = 0;
    len = 0;
    sincebar = 0;
    while ((j<notes) && (start<notes)) {
      while ((len < 4) && (j<notes)) {
        len = len + num[j];
        if (bar[j] == 1) {
          len = num[j];
          start = j;
        };
        j = j + 1;
      };
      while (len > 4) {
        len = len - num[start];
        sincebar = sincebar + num[start];
        start = start + 1;
        if (bar[start] == 1) sincebar = 0;
      };
      if ((sincebar % 4 == 0) && (len == 4)) {
        if (j-start == 2) {
          /* check for broken rhythm */
          int total, dt, split;

          total = length[start] + length[start+1];
          dt = total/12;
          split = ((length[start]+dt)*6)/total;
          if (split == 2) feature[start] = '<';
          if (split == 4) feature[start] = '>';
        };
        if (j-start == 3) {
          /* check for triplet */
          int total, dt, b1, b2;

          total = length[start] + length[start+1] + length[start+2];
          dt = total/12;
          b1 = ((length[start]+dt)*6)/total;
          b2 = ((length[start]+length[start+1]+dt)*6)/total;
          if ((b1 == 2) && (b2 == 4)) feature[start] = 't';
        };
      };
      len = len - num[start];
      sincebar = sincebar + num[start];
      start = start + 1;
      if (bar[start] == 1) sincebar = 0;
    };
  };
};

int getarg(option, argc, argv)
char *option;
char *argv[];
int argc;
{
  int j, place;

  place = -1;
  for (j=0; j<argc; j++) {
    if (strcmp(option, argv[j]) == 0) {
      place = j + 1;
    };
  };
  return (place);
};

findana(barsize, anacrusis)
int barsize;
int *anacrusis;
{
  int i, len;
  int min, max, nmax;

  min = 128;
  max = 0;
  nmax = 0;
  i = 0;
  len = 0;
  while ((len+num[i] <= barsize*2) && (i < 30)) {
    if ((vel[i] > max) || ((vel[i] == max) && (nmax == 0))) {
      max = vel[i];
      nmax = i;
    };
    if (vel[i] < min) min = vel[i];
    i = i + 1;
    len = len + num[i];
  };
  if (min == max) {
    printf("No difference in note velocities to deduce anacrusis from\n");
    nmax = 0;
  };
  *anacrusis = nmax;
};

main(argc,argv)
char *argv[];
int argc;
{
  FILE *efopen();
  int j, text_ctr;
  int barcount;
  int barsize, anacrusis, bars;
  int arg;
  int freshline;
  int voicecount;

  arg = getarg("-a", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    anacrusis = readnum(argv[arg]);
  } else {
    anacrusis = 0;
  };
  arg = getarg("-m", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    readsig(&asig, &bsig, argv[arg]);
  } else {
    asig = 4;
    bsig = 4;
  };
  arg = getarg("-Q", argc, argv);
  if (arg != -1) {
    Qval = readnum(argv[arg]);
  } else {
    Qval = 0;
  };
  extractm = (getarg("-xm", argc, argv) != -1);
  extractl = (getarg("-xl", argc, argv) != -1);
  extracta = (getarg("-xa", argc, argv) != -1);
  if ((asig*4)/bsig >= 3) {
    unitlen =8;
  } else {
    unitlen = 16;
  };
  arg = getarg("-b", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    bars = readnum(argv[arg]);
  } else {
    bars = 0;
  };
  arg = getarg("-t", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    track = readnum(argv[arg]);
  } else {
    track = 0;
  };
  arg = getarg("-c", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    channel = readnum(argv[arg]) - 1;
  } else {
    channel = -1;
  };
  arg = getarg("-f", argc, argv);
  if ((arg != -1) && (arg < argc)) {
    F = efopen(argv[arg],"rb");
    printf("%% input file %s\n", argv[arg]);
  } else {
    printf("midi2abc version 1.6.4\n  usage :\n");
    printf("midi2abc <options>\n");
    printf("         -a <notes in anacrusis>\n");
    printf("         -xa  extract anacrusis from file ");
    printf("(find first strong note)\n");
    printf("         -m <time signature>\n");
    printf("         -xm  extract time signature from file\n");
    printf("         -xl  extract absolute note lengths from file\n");
    printf("         -b <bars wanted in output>\n");
    printf("         -Q <tempo in quarter-notes per minute>\n");
    printf("         -t <track>\n");
    printf("         -c <channel>\n");
    printf("         -f <input file>\n");
    printf("Only the -f option is compulsory.");
    printf(" Use only one of -xl, -b and -Q.\n");
    printf("If none of these is present, the");
    printf(" program attempts to guess a \n");
    printf("suitable note length.\n");
    exit(0);
  };
  xunit = 0;
  text_place = 0;
  text_count = 0;
  trackno = 0;
  initfuncs();
  Mf_getc = filegetc;
  midifile();
  fclose(F);
  if (Qval != 0) {
    xunit = mf_sec2ticks((60.0*4.0)/((float)(Qval*unitlen)), division, tempo);
  };
  printf("X: 1\n");
  printf("T: \n");
  printf("M: %d/%d\n", asig, bsig);
  printf("L: 1/%d\n", unitlen);
  barsize = asig * unitlen/bsig;
  if (bars > 0) {
    xunit = (int) (counttrack(anacrusis)/((long)(bars*barsize)));
  };
  if (notes > 0) {
    if (xunit == 0) {
      guesslengths();
    };
    quantize(xunit);
    if (extracta) findana(barsize, &anacrusis);
    printQ();
    findkey();
    setfeatures(bars, barsize, anacrusis);
  };
/* print out tune */
  barcount = 0;
  j = 0;
  text_ctr = 0;
  text_place = 0;
  freshline = 1;
  voicecount = 1;
  trackno = 0;
  while (j < notes) {
    while ((text_ctr < text_count) && (texttuneplace[text_ctr] <= j)) {
      if (!freshline) printf("\\\n");
      printf("%s\n", &text[textitem[text_ctr]]);
      freshline = 1;
      text_ctr = text_ctr + 1;
    };
    if ((j == trackstart[trackno]) && (format == 1)) {
      /* skip over empty tracks */
      while (trackstart[trackno] == trackstart[trackno+1]) {
        trackno = trackno + 1;
      };
      printf("V:%d\n", voicecount);
      freshline = 1;
      voicecount = voicecount + 1;
      trackno = trackno + 1;
    };
    freshline = 0;
    if (bar[j] == 1) {
      printf("|");
      if (j > anacrusis) barcount = barcount + 1;
      if (barcount == 4) {
        printf("\n");
        barcount = 0;
      };
    };
    switch ((int) feature[j])  {
    case '>':
      printf(" ");
      printpitch(j);
      printf(">");
      printpitch(j+1);
      j = j + 2;
      break;
    case '<':
      printf(" ");
      printpitch(j);
      printf("<");
      printpitch(j+1);
      j = j + 2;
      break;
    case 't':
      printf(" (3");
      printpitch(j);
      printpitch(j+1);
      printpitch(j+2);
      j = j + 3;
      break;
    case '+':
      printf(" +");
      {
      int i;

        i = j;
        while((num[i] == 0) && (i < notes)) {
          i = i + 1;
        };
        while (j < i) {
          printpitch(j);
          printlen(i);
          j = j + 1;
        };
        printpitch(j);
        printlen(j);
      };
      printf("+");
      break;
    case 'x':
      straightnote(j);
      j = j + 1;
      break;
    case ' ':
      printf(" ");
      straightnote(j);
      j = j + 1;
      break;
    default:
      printf("\nunexpected feature %c j = %d!\n", feature[j], j);
      j = j + 1;
    };
  };
  printf("|\n");
  while ((text_ctr < text_count) && (texttuneplace[text_ctr] <= j)) {
    printf("%s\n", &text[textitem[text_ctr]]);
    freshline = 1;
    text_ctr = text_ctr + 1;
  };
  exit(0);
}

FILE *
efopen(name,mode)
char *name;
char *mode;
{
    FILE *f;
#ifndef PCCFIX
    extern int errno;
    extern char *sys_errlist[];
    extern int sys_nerr;
    char *errmess;
#endif

    if ( (f=fopen(name,mode)) == NULL ) {
        (void) fprintf(stderr,"*** ERROR *** Cannot open '%s'!\n",name);
#ifndef PCCFIX
        if ( errno <= sys_nerr )
            errmess = sys_errlist[errno];
        else
            errmess = "Unknown error!";
        (void) fprintf(stderr,"************* Reason: %s\n",errmess);
#endif
        exit(1);
    }
    return(f);
}

error(s)
char *s;
{
    fprintf(stderr,"Error: %s\n",s);
}

txt_header(xformat,ntrks,ldivision)
{
    division = ldivision; 
    format = xformat;
    if (format != 0) {
      printf("%% format %d file %d tracks\n", format, ntrks);
    };
/*
    printf("Header format=%d ntrks=%d division=%d\n",format,ntrks,division);
*/
}

txt_trackstart()
{
  laston = Mf_currtime;
  trackstart[trackno] = notes;
  trackno = trackno + 1;
  if (format != 0) {
    sprintf(&text[text_place], "%% Track %d start\0", trackno);
    advance();
  };
}

txt_trackend()
{
  trackstart[trackno] = notes;
  lastlen = Mf_currtime - laston;
  if (trackstart[trackno-1] != notes) {
    npitch[notes-1] = lastpitch;
    length[notes-1] = lastlen;
    aclen[notes-1] = lastlen;
  };
}

int trackfilter()
{
   return ((track == 0) || (track == trackno));
};

int chantrackfilter(chan)
int chan;
{
  return (((channel == -1) || (channel == chan)) &&
          ((track == 0) || (track == trackno)));
};

txt_noteon(chan,pitch,vol)
{
  if (chantrackfilter(chan)) {
    if (vol != 0) {
      /* record velocity of first 30 notes */
      if (notes < 30) vel[notes] = vol;
      aclen[notes] == 0;
      if (trackstart[trackno-1] != notes) {
        length[notes-1] = Mf_currtime - laston;
        if (aclen[notes-1] == 0) {
          aclen[notes-1] = length[notes-1];
        };
        npitch[notes-1] = lastpitch;
      };
      notes = notes + 1;
      if (notes >= MAXNOTES) {
        printf("Aborting : Too many notes for midi2abc\n");
        exit(0);
      };
      laston = Mf_currtime;
      lastpitch = pitch;
      lastchan = chan;
    } else {
      /* zero vol means note off */
      if ((pitch == lastpitch) && (chan == lastchan)) {
        aclen[notes-1] = Mf_currtime - laston;
      };
      lastlen = Mf_currtime - laston;
    };
  };
}

txt_noteoff(chan,pitch,vol)
{
  if (chantrackfilter(chan)) {
    if ((pitch == lastpitch) && (chan == lastchan)) {
      aclen[notes-1] = Mf_currtime - laston;
    };
    lastlen = Mf_currtime - laston;
  };
}

txt_pressure(chan,pitch,press)
{
  if (chantrackfilter(chan)) {
    sprintf(&text[text_place], "%% Pressure, chan=%d pitch=%d press=%d\0",
            chan+1,pitch,press);
    advance();
  };
}

txt_parameter(chan,control,value)
{
  if (chantrackfilter(chan)) {
    sprintf(&text[text_place], "%% Parameter, chan=%d c1=%d c2=%d\0",
            chan+1,control,value);
    advance();
  };
}

txt_pitchbend(chan,msb,lsb)
{
  if (chantrackfilter(chan)) {
    sprintf(&text[text_place], "%% Pitchbend, chan=%d msb=%d lsb=%d\0",
           chan+1,msb,lsb);
    advance();
  };
}

txt_program(chan,program)
{
  if (chantrackfilter(chan)) {
    sprintf(&text[text_place], "%%%%MIDI program %d %d\0",
           chan+1, program);
    advance();
  };
}

txt_chanpressure(chan,press)
{
  if (chantrackfilter(chan)) {
    sprintf(&text[text_place], "%% Channel pressure, chan=%d pressure=%d\0",
            chan+1,press);
    advance();
  };
}

txt_sysex(leng,mess)
char *mess;
{
  if (trackfilter()) {
    prtime();
    sprintf(&text[text_place], "%% Sysex, leng=%d\0",leng);
    advance();
  };
}

txt_metamisc(type,leng,mess)
char *mess;
{
  if (trackfilter()) {
    prtime();
    sprintf(&text[text_place], 
         "%% Meta event, unrecognized, type=0x%02x leng=%d\0",type,leng);
    advance();
  };
}

txt_metaspecial(type,leng,mess)
char *mess;
{
  if (trackfilter()) {
    prtime();
    sprintf(&text[text_place], 
     "%% Meta event, sequencer-specific, type=0x%02x leng=%d\0",type,leng);
    advance();
  };
}

txt_metatext(type,leng,mess)
char *mess;
{ 
  static char *ttype[] = {
    NULL,
    "Text Event",        /* type=0x01 */
    "Copyright Notice",    /* type=0x02 */
    "Sequence/Track Name",
    "Instrument Name",    /* ...     */
    "Lyric",
    "Marker",
    "Cue Point",        /* type=0x07 */
    "Unrecognized"
  };
  int unrecognized = (sizeof(ttype)/sizeof(char *)) - 1;
  register int n, c;
  register char *p = mess;
  char *buff;

  if (trackfilter()) {
    if ( type < 1 || type > unrecognized )
        type = unrecognized;
    prtime();
/*
    printf("Meta Text, type=0x%02x (%s)  leng=%d\n",type,ttype[type],leng);
*/
    buff = &text[text_place];
    sprintf(buff, "%% %s : \0", ttype[type]);
    buff = buff + strlen(buff);
    for ( n=0; n<leng; n++ ) {
        c = *p++;
        sprintf(buff, 
             (isprint(c)||isspace(c)) ? "%c\0" : "\\0x%02x\0" , c);
        buff = buff + 1;
    }
    advance();
  };
}

txt_metaseq(num)
{  
  if (trackfilter()) {
    prtime();
    sprintf(&text[text_place], "Meta event, sequence number = %d\0",num);
    advance();
  };
}

txt_metaeot()
{
    prtime();
/*
    printf("Meta event, end of track\n");
*/
}

txt_keysig(sf,mi)
char sf, mi;
{
  if (trackfilter()) {
    prtime();
    sprintf(&text[text_place], 
           "%% MIDI Key signature, sharp/flats=%d  minor=%d\0",
            (int) sf, (int) mi);
    advance();
  };
}

txt_tempo(ltempo)
long ltempo;
{
    tempo = ltempo;
    prtime();
/*
    printf(" Tempo, microseconds-per-MIDI-quarter-note=%d\n",tempo);
*/
}

txt_timesig(nn,dd,cc,bb)
{
  if (trackfilter()) {
    int denom = 1;
    while ( dd-- > 0 )
      denom *= 2;
    sprintf(&text[text_place], 
          "%% Time signature=%d/%d  MIDI-clocks/click=%d  32nd-notes/24-MIDI-clocks=%d\0", 
    nn,denom,cc,bb);
    advance();
    if (extractm) {
      asig = nn;
      bsig = denom;
      if ((asig*4)/bsig >= 3) {
        unitlen =8;
      } else {
        unitlen = 16;
      };
    };
    if (extractl) {
      xunit = (division*bb*4)/(8*unitlen);
    };
  };
}

txt_smpte(hr,mn,se,fr,ff)
{
    prtime();
/*
    printf("SMPTE, hour=%d minute=%d second=%d frame=%d fract-frame=%d\n",
        hr,mn,se,fr,ff);
*/
}

txt_arbitrary(leng,mess)
char *mess;
{
    prtime();
/*
    printf("Arbitrary bytes, leng=%d\n",leng);
*/
}

prtime()
{
/*
  if(SECONDS)
    printf("Time=%f   ",mf_ticks2sec(Mf_currtime,division,tempo));
  else
    printf("Time=%ld  ",Mf_currtime);
*/
}

initfuncs()
{
    Mf_error = error;
    Mf_header =  txt_header;
    Mf_trackstart =  txt_trackstart;
    Mf_trackend =  txt_trackend;
    Mf_noteon =  txt_noteon;
    Mf_noteoff =  txt_noteoff;
    Mf_pressure =  txt_pressure;
    Mf_parameter =  txt_parameter;
    Mf_pitchbend =  txt_pitchbend;
    Mf_program =  txt_program;
    Mf_chanpressure =  txt_chanpressure;
    Mf_sysex =  txt_sysex;
    Mf_metamisc =  txt_metamisc;
    Mf_seqnum =  txt_metaseq;
    Mf_eot =  txt_metaeot;
    Mf_timesig =  txt_timesig;
    Mf_smpte =  txt_smpte;
    Mf_tempo =  txt_tempo;
    Mf_keysig =  txt_keysig;
    Mf_seqspecific =  txt_metaspecial;
    Mf_text =  txt_metatext;
    Mf_arbitrary =  txt_arbitrary;
}
