/* tomidi.c  version 1.5.1
 * This file is part of the code for abc2midi a converter from abc format 
 * to MIDI.  
 * 
 * 19th August 1996
 * James Allwright
 * Department of Electronics and Computer Science,
 * University of Southampton, UK
 *
 * Macintosh Port 30th July 1996 
 * Wil Macaulay (wil@syndesis.com)
 
 */

#include "abc.h"
#include "midifile.h"
#include <stdio.h>
#ifdef __MWERKS__
#define __MACINTOSH__ 1
#endif /* __MWERKS__ */

char* index();
#ifdef __MACINTOSH__
#include <string.h>
int setOutFileCreator(char *fileName,unsigned long theType,
                      unsigned long theCreator);
#define index(a,b) (int)strchr((a),(b))
#endif /* __MACINTOSH__ */ 
#define DIV 480
#define MAXPARTS 100
#define MAXLINE 100
#define MAXDEFERRED 100

FILE *fp;
extern int lineno;
int check;
char basemap[7], workmap[7];
int  basemul[7], workmul[7];
char* deferred[MAXDEFERRED];
int ndeferred;
#ifdef PCCFIX
#define maxnotes 5000
int pitch[maxnotes], num[maxnotes], denom[maxnotes], feature[maxnotes];
#else
int maxnotes;
int *pitch, *num, *denom, *feature;
#endif
int notes;
int xmatch;
int dotune, pastheader;
long tempo;
char text_buffer[3000];
int text_end;
int sf, mi;
int tuplecount, tfact_num, tfact_denom, tnote_num, tnote_denom;
int specialtuple;
int inchord, chordcount;
int inslur;
char part[MAXPARTS];
int parts, partno;
int part_start[26];
int time_num, time_denom, tempo_num, tempo_denom;
int err_num, err_denom;
int relative_tempo, Qtempo;
int default_length;
int division = DIV;
int div_factor;
long delta_time;
long tracklen, tracklen1;
char outname[40], outbase[40];
int bar_num, bar_denom, barno, barsize, beat;
int b_num, b_denom;
int loudnote, mednote, softnote, channel, program;
int channels[16];
int transpose;
int hornpipe;
int gracenotes;
int gfact_num, gfact_denom;
int middle_c;
int gchords, g_started;
int basepitch, seventh, force_fun;
int karaoke, wcount, ktied, kspace, koverflow;
char* words[500];
struct notetype {
  int base; 
  int chan;
  int vel;
};
enum chordmode {maj, min, aug, dim} mode;
struct notetype gchord, fun;
int g_num, g_denom;
int g_next;
char gchord_seq[40];
int gchord_len[40];
int g_ptr;
/* queue for notes waiting to end */
/* allows us to do general polyphony */
#define QSIZE 40
struct Qitem {
  int delay;
  int pitch;
  int chan;
  int next;
};
struct Qitem Q[QSIZE];
int Qhead, freehead, freetail;
#ifdef NOFTELL
char buffer[8000];
int buffcount, buffering;
#endif
int maxvoice;

int getarg(option, argc, argv)
/* look for argument 'option' in command line */
char *option;
char *argv[];
int argc;
{
  int j, place;

  place = -1;
  for (j=0; j<argc; j++) {
    if (strcmp(option, argv[j]) == 0) {
      place = j + 1;
    };
  };
  return (place);
};

int* checkmalloc(bytes)
/* malloc with error checking */
int bytes;
{
  int *p;
  char* malloc();

  p = (int*) malloc(bytes);
  if (p == NULL) {
    printf("Out of memory error - cannot handle that many notes.\n");
    exit(0);
  };
  return (p);
};

char* addstring(s)
/* create space for string and store it in memory */
char* s;
{
  char* p;

  p = (char*) checkmalloc(strlen(s)+1);
  strcpy(p, s);
  return(p);
};

event_init(argc, argv, filename)
/* this routine is called first by parseabc.c */
int argc;
char* argv[];
char **filename;
{
  int j;
  int narg;

  /* look for code checking option */
  if (getarg("-c", argc, argv) != -1) {
    check = 1;
  } else {
    check = 0;
  };
#ifndef PCCFIX
  narg = getarg("-n", argc, argv);
  if (narg != -1) {
    maxnotes = readnumf(argv[narg]);
  } else {
    maxnotes = 10000;
  };
  /* allocate space for notes */
  pitch = checkmalloc(maxnotes*sizeof(int));
  num = checkmalloc(maxnotes*sizeof(int));
  denom = checkmalloc(maxnotes*sizeof(int));
  feature = checkmalloc(maxnotes*sizeof(int));
#endif
  if ((getarg("-h", argc, argv) != -1) || (argc < 2)) {
    printf("abc2midi version 1.5.1\n");
    printf("Usage : abc2midi <abc file> [reference number] [-c]");
#ifndef PCCFIX
    printf(" [-n X]");
#endif
    printf("\n");
    printf("        [reference number] selects a tune\n");
    printf("        -c selects checking only\n");
#ifndef PCCFIX
    printf("        -n X reserves X units of note storage (default 10000)\n");
#endif
    exit(0);
  } else {
    xmatch = 0;
    if ((argc >= 3) && (isdigit(*argv[2]))) {
      xmatch = readnumf(argv[2]);
    };
    *filename = argv[1];
    strcpy(outbase, argv[1]);
    for (j = 0; j<strlen(outbase); j++) {
      if (outbase[j] == '.') outbase[j] = '\0';
    };
  };
  dotune = 0;
  parseroff();
};

event_text(s)
/* text found in abc file */
char *s;
{
  char msg[200];

  sprintf(msg, "Ignoring text: %s\0", s);
  event_warning(msg);
};

event_reserved(p)
/* reserved character H-Z found in abc file */
char p;
{
  char msg[200];

  sprintf(msg, "Ignoring reserved character %c\0", p);
  event_warning(msg);
};

event_tex(s)
/* TeX command found - ignore it */
char *s;
{
};

event_linebreak()
{
};

event_musiclinebreak()
{
  if (dotune) {
    addfeature(NEWLINE, 0, 0, 0);
  };
};

event_eof()
/* end of abc file encountered */
{
  if (dotune) {
    dotune = 0;
    parseroff();
    finishfile();
  };
  printf("End of File reached\n");
#ifndef PCCFIX
  free(pitch);
  free(num);
  free(denom);
  free(feature);
#endif
};

event_error(s)
/* generic error handler */
char *s;
{
  printf("Error in line %d : %s\n", lineno, s);
};

event_warning(s)
/* generic warning handler - for flagging possible errors */
char *s;
{
  printf("Warning in line %d : %s\n", lineno, s);
};

event_comment(s)
/* comment found in abc */
char *s;
{
  if (dotune) {
    if (pastheader) {
      addfeature(TEXT, text_end, 0, 0);
      addtext(s);
    } else {
      addfeature(TEXT, text_end, 0, 0);
      addtext(s);
    };
  };
};

event_specific(package, s)
/* package-specific command found i.e. %%NAME */
/* only %%MIDI commands are actually handled */
char *package, *s;
{
  char msg[200], command[40];
  char *p;

  if (strcmp(package, "MIDI") == 0) {
    int ch;
    int done;

    p = s;
    done = 0;
    skipspace(&p);
    readstr(command, &p);
    if (strcmp(command, "channel") == 0) {
      skipspace(&p);
      ch = readnump(&p) - 1;
      addfeature(CHANNEL, ch, 0, 0);
      done = 1;
    };
    if (strcmp(command, "transpose") == 0) {
      int neg, val;

      skipspace(&p);
      neg = 0;
      if (*p == '+') p = p + 1;
      if (*p == '-') {
        p = p + 1;
        neg = 1;
      };
      skipspace(&p);
      val = readnump(&p);
      if (neg) val = - val;
      if (pastheader) {
        addfeature(TRANSPOSE, val, 0, 0);
      } else {
        transpose = val;
      };
      done = 1;
    };
    if (strcmp(command, "C") == 0) {
      int val;

      skipspace(&p);
      val = readnump(&p);
      middle_c = val;
      done = 1;
    };

    if (strcmp(command, "grace") == 0) {
      int a, b;
      char msg[40];

      skipspace(&p);
      a = readnump(&p);
      if (*p != '/') {
        event_error("Need / in MIDI grace command");
      } else {
        p = p + 1;
      };
      b = readnump(&p);
      if ((a < 1) || (b < 1) || (a >= b)) {
        sprintf(msg, "%d/%d is not a suitable fraction", a, b);
        event_error(msg);
      } else {
        if (pastheader) {
          addfeature(SETGRACE, a, 0, b);
        } else {
          gfact_num = a;
          gfact_denom = b;
        };
      };
      done = 1;
    };
    if (strcmp(command, "gchordon") == 0) {
      addfeature(GCHORDON, 0, 0, 0);
      done = 1;
    };
    if (strcmp(command, "gchordoff") == 0) {
      addfeature(GCHORDOFF, 0, 0, 0);
      done = 1;
    };
    if (done == 0) {
      /* add as a command to be interpreted later */
      if (ndeferred >= MAXDEFERRED) {
        event_error("Too many deferredally interpreted commands");
      } else {
        deferred[ndeferred] = addstring(s);
        addfeature(DYNAMIC, ndeferred, 0, 0);
        ndeferred = ndeferred + 1;
      };
    };
  } else {
    strcpy(msg, "%");
    strcpy(msg+strlen(msg), package);
    strcpy(msg+strlen(msg),s);
    event_comment(msg);
  };
};

set_gchords(s)
/* set up a string which indicates how to generate accompaniment from */
/* guitar chords (i.e. "A", "G" in abc). */
/* called from dodeferred(), startfile() and setbeat() */
char* s;
{
  int seq_len;
  char* p;
  int j;

  p = s;
  j = 0;
  seq_len = 0;
  while (((*p == 'z') || (*p == 'c') || (*p == 'f')) && (j<39)) {
    gchord_seq[j] = *p;
    p = p + 1;
    if ((*p >= '0') && (*p <= '9')) {
      gchord_len[j] = readnump(&p);
    } else {
      gchord_len[j] = 1;
    };
    seq_len = seq_len + gchord_len[j];
    j = j + 1;
  };
  if (seq_len == 0) {
    event_error("Bad gchord");
    gchord_seq[0] = 'z';
    gchord_len[0] = 1;
    seq_len = 1;
  };
  gchord_seq[j] = '\0';
  if (j == 39) {
    event_error("Sequence string too long");
  };
  /* work out unit delay in 1/4 notes*/
  g_num = time_num * 4;
  g_denom = time_denom * seq_len;
  reduce(&g_num, &g_denom);
};

dodeferred(s)
/* handle package-specific command which has been held over to be */
/* interpreted as MIDI is being generated */
char* s;
{
  int j;
  char* p;
  char command[40];
  int done;

  p = s;
  skipspace(&p);
  readstr(command, &p);
  skipspace(&p);
  done = 0;
  if (strcmp(command, "program") == 0) {
    int chan, prog;

    skipspace(&p);
    prog = readnump(&p);
    chan = channel;
    skipspace(&p);
    if ((*p >= '0') && (*p <= '9')) {
      chan = prog - 1;
      prog = readnump(&p);
    };
    write_program(prog, chan);
    done = 1;
  };
  if (strcmp(command, "gchord") == 0) {
    set_gchords(p);
    done = 1;
  };
  if (strcmp(command, "chordprog") == 0) {
    int prog;

    prog = readnump(&p);
    write_program(prog, gchord.chan);
    done = 1;
  };
  if (strcmp(command, "bassprog") == 0) {
    int prog;

    prog = readnump(&p);
    write_program(prog, fun.chan);
    done = 1;
  };
  if (strcmp(command, "chordvol") == 0) {
    gchord.vel = readnump(&p);
    done = 1;
  };
  if (strcmp(command, "bassvol") == 0) {
    fun.vel = readnump(&p);
    done = 1;
  };
  if (strcmp(command, "beat") == 0) {
    skipspace(&p);
    loudnote = readnump(&p);
    skipspace(&p);
    mednote = readnump(&p);
    skipspace(&p);
    softnote = readnump(&p);
    skipspace(&p);
    beat = readnump(&p);
    if (beat == 0) {
      beat = barsize;
    };
    done = 1;
  };
  if (strcmp(command, "control") == 0) {
    int chan, n, datum;
    char data[20];
    char sel[40];

    skipspace(&p);
    chan = channel;
    if (isalpha(*p)) {
      readstr(sel, &p);
      skipspace(&p);
      if (strcmp(sel, "bass") == 0) {
        chan = fun.chan;
      };
      if (strcmp(sel, "chord") == 0) {
        chan = gchord.chan;
      };
    };
    n = 0;
    while ((n<20) && (*p >= '0') && (*p <= '9')) {
      datum = readnump(&p);
      if (datum > 127) {
        event_error("data must be in the range 0 - 127");
        datum = 0;
      };
      data[n] = (char) datum;
      n = n + 1;
      skipspace(&p);
    };
    write_control(chan, data, n);
    done = 1;
  };
  if (done == 0) {
    event_error("Command not recognized");
  }; 
}; 

event_field(k, f) 
/* Handles R: T: and any other field not handled elsewhere */
char k; 
char *f;
{
  if (dotune) {
    switch (k) {
    case 'T':
      addfeature(TITLE, text_end, 0, 0);
      addtext(f);
      break;
    case 'R':
      {
        char* p;

        p = f;
        skipspace(&p);
        if ((strncmp(p, "Hornpipe", 8) == 0) || 
            (strncmp(p, "hornpipe", 8) == 0)) {
          hornpipe = 1;
        };
      };
      break;
    default:
      addfeature(TEXT, text_end, 0, 0);
      addch(k);
      addch(':');
      addtext(f);
    };
  };
};

event_words(p)
/* handles a w: field in the abc */
char* p;
{
  int l;
  char* s;

  karaoke = 1;
  /* strip off leading spaces */
  s = p;
  while (*s == ' ') {
    s = s + 1;
  };
  words[wcount] = addstring(s);
  wcount = wcount + 1;
  /* strip off any trailing spaces */
  l = strlen(s) - 1;
  while (*(words[wcount-1]+l) == ' ') {
    *(words[wcount-1]+l) = '\0';
    l = l - 1;
  };
};

char_out(list, out, ch)
/* routine for building up part list */
char* list;
char** out;
char ch;
{
  if (*out - list >= MAXPARTS) {
    event_error("Expanded part is too large");
  } else {
    **out = ch;
    *out = *out + 1;
    parts = parts + 1;
  };
};

read_spec(spec, part)
/* converts a P: field to a list of part labels */
/* e.g. P:A(AB)3(CD)2 becomes P:AABABABCDCD */
char spec[];
char part[];
{
  char* in;
  char* out;
  int i, j;
  int stackptr;
  char* stack[10];

  stackptr = 0;
  in = spec;
  out = part;
  while (((*in >= 'A') && (*in <= 'Z')) || (*in == '(') ||
         (*in == ')') || ((*in >= '0') && (*in <= '9'))) {
    if ((*in >= 'A') && (*in <= 'Z')) {
      char_out(part, &out, *in);
      in = in + 1;
    };
    if (*in == '(') {
      if (stackptr < 10) {
        stack[stackptr] = out;
        stackptr = stackptr + 1;
      } else {
        event_error("nesting too deep in part specification");
      };
      in = in + 1;
    };
    if (*in == ')') {
      in = in + 1;
      if (stackptr > 0) {
        int repeats;
        char* start;
        char* stop;

        if ((*in >= '0') && (*in <= '9')) {
          repeats = readnump(&in);
        } else {
          repeats = 1;
        };
        stackptr = stackptr - 1;
        start = stack[stackptr];
        stop = out;
        for (i=1; i<repeats; i++) {
          for (j=0; j<((int) (stop-start)); j++) {
            char_out(part, &out, *(start+j));
          };
        };
      } else {
        event_error("Too many )'s in part specification");
      };
    };
    if ((*in >= '0') && (*in <= '9')) {
      int repeats;
      char ch;

      repeats = readnump(&in);
      if (out > spec) {
        ch = *(out-1);
        for (i = 1; i<repeats; i++) {
          char_out(part, &out, ch);
        };
      } else {
        event_error("No part to repeat in part specification");
      };
    };
  };
  if (stackptr != 0) {
    event_error("Too many ('s in part specification");
  };
};
  
event_part(s)
/* handles a P: field in the abc */
char* s;
{
  char* p;

  if (dotune) {
    p = s;
    skipspace(&p);
    if (pastheader) {
      if (((int)*p < 'A') || ((int)*p > 'Z')) {
        event_error("Part must be one of A-Z");
        return;
      };
      part_start[(int)*p - (int)'A'] = notes;
      addfeature(PART, (int)*p, 0, 0);
    } else {
      parts = 0;
      read_spec(p, part);
    };
  };
};

event_voice(n, s)
/* handles a V: field in the abc */
int n;
char *s;
{
  addfeature(VOICE, n, 0, 0);
  if (n > maxvoice) {
    maxvoice = n;
  };
};

event_length(n)
/* handles an L: field in the abc */
int n;
{
  default_length = n;
};

event_blankline()
/* blank line found in abc signifies the end of a tune */
{
  if (dotune) {
    finishfile();
    parseroff();
    dotune = 0;
  };
};

event_refno(n)
/* handles an X: field (which indicates the start of a tune) */
int n;
{
  if (dotune) {
    finishfile();
    parseroff();
    dotune = 0;
  };
  printf("Reference X: %d\n", n);
  if ((n == xmatch) || (xmatch == 0)) {
    parseron();
    dotune = 1;
    pastheader = 0;
    strcpy(outname, outbase);
    numstr(n, outname+strlen(outname));
    strcpy(outname+strlen(outname), ".mid");
    startfile();
  };
};

event_tempo(n, a, b, rel)
/* handles a Q: field e.g. Q: a/b = n  or  Q: Ca/b = n */
int n;
int a, b, rel;
{
  int t_num, t_denom;
  int new_div;
  long new_tempo;

  if (dotune) {
    if (pastheader) {
      tempo_num = a;
      tempo_denom = b;
      relative_tempo = rel;
      tempounits(&t_num, &t_denom);
      new_tempo = (long) 60*1000000*t_denom/(n*4*t_num);
      new_div = (int) ((float)DIV*(float)new_tempo/(float)tempo + 0.5);
      addfeature(TEMPO, new_div, 0, 0);
    } else {
      Qtempo = n;
      tempo_num = a;
      tempo_denom = b;
      relative_tempo = rel;
    };
  };
};

event_timesig(n, m)
/* handles an M: field  M:n/m */
int n, m;
{
  if (dotune) {
    if (pastheader) {
      addfeature(TIME, n, 0, m);
    } else {
      time_num = n;
      time_denom = m;
    };
  };
};

event_key(sharps, s, minor, modmap, modmul)
/* handles a K: field */
int sharps; /* sharps is number of sharps in key signature */
int minor; /* a boolean 0 or 1 */
char *s; /* original string following K: */
char modmap[7]; /* array of accidentals to be applied */
int  modmul[7]; /* array giving multiplicity of each accent (1 or 2) */
{
  if (dotune) {
    setmap(sharps, basemap, basemul);
    altermap(basemap, basemul, modmap, modmul);
    copymap();
    if (pastheader) {
      addfeature(KEY, sharps, 0, minor);
    } else {
      addfeature(DOUBLE_BAR, 0, 0, 0);
      sf = sharps;
      mi = minor;
    };
    pastheader = 1;
    headerprocess();
  };
};

event_graceon()
/* a { in the abc */
{
  if (gracenotes) {
    event_error("Nested grace notes not allowed");
  };
  gracenotes = 1;
  addfeature(GRACEON, 0, 0, 0);
};

event_graceoff()
/* a } in the abc */
{
  if (!gracenotes) {
    event_error("} without matching {");
  };
  gracenotes = 0;
  addfeature(GRACEOFF, 0, 0, 0);
};

event_rep1()
/* |1 in the abc */
{
  if ((notes == 0) || (feature[notes-1] != SINGLE_BAR)) {
    event_error("[1 must follow a single bar");
  } else {
    feature[notes-1] = BAR1;
  };
};

event_rep2()
/* :|2 in the abc */
{
  if ((notes == 0) || (feature[notes-1] != REP_BAR)) {
    event_error("[2 must follow a :| ");
  } else {
    feature[notes-1] = REP_BAR2;
  };
};

event_slur(t)
/* called when s in encountered in the abc */
int t;
{
  if (t) {
    addfeature(SLUR_ON, 0, 0, 0);
    inslur = 1;
  } else {
    addfeature(SLUR_OFF, 0, 0, 0);
    slurfix();
    inslur = 0;
  };
};

event_sluron(t)
/* called when ( is encountered in the abc */
int t;
{
  if (t == 1) {
    addfeature(SLUR_ON, 0, 0, 0);
    inslur = 1;
  };
};

event_sluroff(t)
/* called when ) is encountered */
int t;
{
  if (t == 0) {
    addfeature(SLUR_OFF, 0, 0, 0);
    slurfix();
    inslur = 0;
  };
};

slurfix()
/* post processing - slur might mean tie */
{
  /* after a SLUR_OFF, the last SLUR_TIE must be removed */
  int p;
  int doingchord;

  doingchord = inchord;
  p = notes - 1;
  while ((p>0) && (feature[p] != SLUR_TIE) && (feature[p] != SLUR_ON)) {
    if (feature[p] == CHORDOFF) {
      doingchord = 1;
    };
    if (feature[p] == CHORDON) {
      doingchord = 0;
    };
    p = p - 1;
  };
  if (!doingchord) {
    if (feature[p] == SLUR_TIE) {
      feature[p] = NONOTE;
    } else {
      event_error("Malformed slur");
    };
  } else {
    /* remove SLUR_TIE from every note in chord */
    while ((p>0) && (feature[p] != CHORDON)) {
      if (feature[p] == SLUR_TIE) {
        feature[p] = NONOTE;
      };
      p = p - 1;
    };
  };
};

event_tie()
/* a tie - has been encountered in the abc */
{
  if (!inslur) {
    addfeature(TIE, 0, 0, 0);
  };
};

event_rest(n,m)
/* rest of n/m in the abc */
int n, m;
{
  int num, denom;

  num = n;
  denom = m;
  if (inchord) chordcount = chordcount + 1;
  if (tuplecount > 0) {
    num = num * tfact_num;
    denom = denom * tfact_denom;
    if (tnote_num == 0) {
      tnote_num = num;
      tnote_denom = denom;
    } else {
      if (tnote_num * denom != num * tnote_denom) {
        if (!specialtuple) {
          event_warning("Different length notes in tuple");
        };
      };
    };
    if ((!gracenotes) &&
        ((!inchord) || ((inchord) && (chordcount == 1)))) {
      tuplecount = tuplecount - 1;
    };
  };
  addfeature(REST, 0, num*4, denom*default_length);
};

event_bar(type)
/* handles bar lines of various types in the abc */
int type;
{
  int newtype;

  if ((type == THIN_THICK) || (type == THICK_THIN)) {
    newtype = DOUBLE_BAR;
  } else {
    newtype = type;
  };
  addfeature(newtype, 0, 0, 0);
  copymap();
};

event_space()
/* space character in the abc is ignored by abc2midi */
{
  /* ignore */
  /* printf("Space event\n"); */
};

event_lineend(ch, n)
/* called when \ or ! or * or ** is encountered at the end of a line */
char ch;
int n;
{
  /* ignore */
};

event_broken(type, mult)
/* handles > >> >>> < << <<< in the abc */
int type, mult;
{
  addfeature(type, mult, 0, 0);
};

event_tuple(n, q, r)
/* handles triplets (3 and general tuplets (n:q:r in the abc */
int n, q, r;
{
  if (tuplecount > 0) {
    event_error("nested tuples");
  } else {
    if (r == 0) {
      specialtuple = 0;
      tuplecount = n;
    } else {
      specialtuple = 1;
      tuplecount = r;
    };
    if (q != 0) {
      tfact_num = q;
      tfact_denom = n;
    } else {
      if ((n < 2) || (n > 9)) {
        event_error("Only tuples (2 - (9 allowed");
        tfact_num = 1;
        tfact_denom = 1;
        tuplecount = 0;
      } else {
        /* deduce tfact_num using standard abc rules */
        if ((n == 2) || (n == 4) || (n == 8)) tfact_num = 3;
        if ((n == 3) || (n == 6)) tfact_num = 2;
        if ((n == 5) || (n == 7) || (n == 9)) {
          if ((time_num % 3) == 0) {
            tfact_num = 3;
          } else {
            tfact_num = 2;
          };
        };
        tfact_denom = n;
      };
    };
    tnote_num = 0;
    tnote_denom = 0;
  };
};

event_chord()
/* a + has been encountered in the abc */
{
  if (inchord) {
    event_chordoff();
  } else {
    event_chordon();
  };
};

event_chordon()
/* handles a chord start [ in the abc */
{
  if (inchord) {
    event_error("Attempt to nest chords");
  } else {
    addfeature(CHORDON, 0, 0, 0);
  };
  inchord = 1;
  chordcount = 0;
};

event_chordoff()
/* handles a chord close ] in the abc */
{
  if (!inchord) {
    event_error("Chord already finished");
  } else {
    addfeature(CHORDOFF, 0, 0, 0);
  };
  inchord = 0;
  chordcount = 0;
};

splitstring(s, sep, handler)
/* breaks up string into fields with sep as the field separator */
/* and calls handler() for each sub-string */
char* s;
char sep;
void (*handler)();
{
  char* out;
  char* p;
  int fieldcoming;

  p = s;
  fieldcoming = 1;
  while (fieldcoming) {
    out = p;
    while ((*p != '\0') && (*p != sep)) p = p + 1;
    if (*p == sep) {
      *p = '\0';
      p = p + 1;
    } else {
      fieldcoming = 0;
    };
    (*handler)(out);
  };
};

void event_handle_instruction(), event_handle_gchord();

event_instruction(s)
/* handles a ! ... ! event in the abc */
char* s;
{
  splitstring(s, ';', event_handle_instruction);
};

event_gchord(s)
/* handles guitar chords " ... " */
char* s;
{
  splitstring(s, ';', event_handle_gchord);
};

void event_handle_gchord(s)
/* handler for the guitar chords */
char* s;
{
  int basepitch;
  char accidental, note;
  char* p;
  int seventh, force_fun;
  enum chordmode mode;

  if ((*s >= '0') && (*s <= '5')) {
    event_finger(s);
    return;
  };
  p = s;
  if ((*p >= 'A') && (*p <= 'G')) {
    note = *p - (int) 'A' + (int) 'a';
    force_fun = 0;
    p = p + 1;
  } else {
    if ((*p >= 'a') && (*p <= 'g')) {
      note = *p;
      force_fun = 1;
      p = p + 1;
    } else {
      event_error("Guitar chord does not start with A-G or a-g");
      return;
    };
  };
  accidental = '=';
  if (*p == '#') {
    accidental = '^';
    p = p + 1;
  };
  if (*p == 'b') {
    accidental = '_';
    p = p + 1;
  };
  basepitch = pitchof(note, accidental, 1, 0) - middle_c;
  mode = maj;
  seventh = 0;
  if (*p == '7') {
    seventh = 1;
    p = p + 1;
  };
  if (*p == 'm') {
    mode = min;
    p = p + 1;
  };
  if ((*p == 'a') && (*(p+1) == 'u') && (*(p+2) == 'g')) {
    mode = aug;
    p = p + 3;
  };
  if ((*p == 'd') && (*(p+1) == 'i') && (*(p+2) == 'm')) {
    mode = dim;
    p = p + 3;
  };
  if (*p == '7') {
    seventh = 1;
    p = p + 1;
  };
  addfeature(GCHORD, basepitch, 0, 
        (force_fun*8) + (((int) mode)*2) + seventh);
};

void event_handle_instruction(s)
/* handler for ! ! instructions - none implemented yet ! */
char* s;
{
  char buff[MAXLINE];

  sprintf(buff, "instruction !%s! ignored", s);
  event_warning(buff);
};

event_finger(p)
/* a 1, 2, 3, 4 or 5 has been found in a guitar chord field */
char *p;
{
  /* does nothing */
};

event_note(decorators, accidental, mult, note, octave, n, m)
/* handles a note in the abc */
int decorators[DECSIZE];
int mult;
char accidental, note;
int octave, n, m;
{
  int pitch;
  int num, denom;

  num = n;
  denom = m;
  if (inchord) chordcount = chordcount + 1;
  if (tuplecount > 0) {
    num = num * tfact_num;
    denom = denom * tfact_denom;
    if (tnote_num == 0) {
      tnote_num = num;
      tnote_denom = denom;
    } else {
      if (tnote_num * denom != num * tnote_denom) {
        if (!specialtuple) {
          event_warning("Different length notes in tuple");
        };
      };
    };
    if ((!gracenotes) &&
        ((!inchord) || ((inchord) && (chordcount == 1)))) {
      tuplecount = tuplecount - 1;
    };
  };
  pitch = pitchof(note, accidental, mult, octave);
  if (decorators[FERMATA]) {
    num = num*2;
  };
  if ((decorators[ROLL]) || (decorators[ORNAMENT])) {
    doroll(note, octave, num, denom, pitch);
  } else {
    if (decorators[STACCATO]) {
      addfeature(NOTE, pitch, num*4, denom*2*default_length);
      addfeature(REST, pitch, num*4, denom*2*default_length);
    } else {
      addfeature(NOTE, pitch, num*4, denom*default_length);
      if (inslur) {
        addfeature(SLUR_TIE, 0, 0, 0);
      };
    };
  };
};

doroll(note, octave, n, m, pitch)
/* applies a roll to a note */
char note;
int octave, n, m;
int pitch;
{
  char up, down;
  int t;
  int upoct, downoct, pitchup, pitchdown;
  char *anoctave = "cdefgab";

  upoct = octave;
  downoct = octave;
  t = (int) ((long) index(anoctave, note)  - (long) anoctave);
  up = *(anoctave + ((t+1) % 7));
  down = *(anoctave + ((t+6) % 7));
  if (up == 'c') upoct = upoct + 1;
  if (down == 'b') downoct = downoct - 1;
  pitchup = pitchof(up, basemap[(int)up - 'a'], 1, upoct);
  pitchdown = pitchof(down, basemap[(int)down - 'a'], 1, downoct);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
  addfeature(NOTE, pitchup, n*4, m*default_length*5);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
  addfeature(NOTE, pitchdown, n*4, m*default_length*5);
  addfeature(NOTE, pitch, n*4, m*default_length*5);
};

int pitchof(note, accidental, mult, octave)
/* finds MIDI pitch value for note */
char note, accidental;
int mult, octave;
{
  int p;
  char acc;
  int mul, noteno;
  static int scale[7] = {0, 2, 4, 5, 7, 9, 11};
  char *anoctave = "cdefgab";

  p = (int) ((long) index(anoctave, note) - (long) anoctave);
  p = scale[p];
  acc = accidental;
  mul = mult;
  noteno = (int)note - 'a';
  if (acc == ' ') {
    acc = workmap[noteno];
    mul = workmul[noteno];
  } else {
    workmap[noteno] = acc;
    workmul[noteno] = mul;
  };
  if (acc == '^') p = p + mul;
  if (acc == '_') p = p - mul;
  return p + 12*octave + middle_c;
};

setmap(sf, map, mult)
/* work out accidentals to be applied to each note */
int sf; /* number of sharps in key signature -7 to +7 */
char map[7];
int mult[7];
{
  int j;

  for (j=0; j<7; j++) {
    map[j] = '=';
    mult[j] = 1;
  };
  if (sf >= 1) map['f'-'a'] = '^';
  if (sf >= 2) map['c'-'a'] = '^';
  if (sf >= 3) map['g'-'a'] = '^';
  if (sf >= 4) map['d'-'a'] = '^';
  if (sf >= 5) map['a'-'a'] = '^';
  if (sf >= 6) map['e'-'a'] = '^';
  if (sf >= 7) map['b'-'a'] = '^';
  if (sf <= -1) map['b'-'a'] = '_';
  if (sf <= -2) map['e'-'a'] = '_';
  if (sf <= -3) map['a'-'a'] = '_';
  if (sf <= -4) map['d'-'a'] = '_';
  if (sf <= -5) map['g'-'a'] = '_';
  if (sf <= -6) map['c'-'a'] = '_';
  if (sf <= -7) map['f'-'a'] = '_';
};

altermap(basemap, basemul, modmap, modmul)
/* apply modifiers to a set of accidentals */
char basemap[7], modmap[7];
int basemul[7], modmul[7];
{
  int i;

  for (i=0; i<7; i++) {
    if (modmap[i] != ' ') {
      basemap[i] = modmap[i];
      basemul[i] = modmul[i];
    };
  };
};

copymap()
{
  int j;

  for (j=0; j<7; j++) {
    workmap[j] = basemap[j];
    workmul[j] = basemul[j];
  };
};

addfeature(f, p, n, d) 
/* place feature in internal table */
int f, p, n, d;
{
  feature[notes] = f;
  pitch[notes] = p;
  num[notes] = n;
  denom[notes] = d;
  if ((f == NOTE) || (f == REST)) {
    reduce(&num[notes], &denom[notes]);
  };
  notes = notes + 1;
  if (notes > maxnotes) {
    event_error("Maximum number of notes exceeded: use -n option to set limit");
    exit(0);
    notes = 0;
  };
};

numstr(n, out)
/* convert number to string */
int n;
char *out;
{
  char s[10];
  char* p;
  int i, t;

  t = n;
  i = 9;
  while ((t > 0) || (i == 9)){
    s[i] = (char) (t % 10 + (int)'0');
    t = t/10;
    i = i - 1;
  };
  i = i + 1;
  p = out;
  while (i < 10) {
    *p = s[i];
    p = p + 1;
    i = i + 1;
  };
  *p = '\0';
};
  
addtext(s)
/* put some text in internal text buffer */
char *s;
{
  strcpy(&text_buffer[text_end], s);
  text_end= text_end+ strlen(&text_buffer[text_end]) + 1;
  if (text_end > 3000) {
    event_error("String buffer overflow");
  };
};

addch(ch)
/* put character in internal text buffer */
char ch;
{
  text_buffer[text_end] = ch;
  text_end = text_end + 1;
  text_buffer[text_end] = '\0';
};

int nullputc(c)
/* dummy putc for abc checking option */
char c;
{
  int t;

  t = ((int) c) & 0xFF;
  return (t);
};

/* workaround for problems with PCC compiler */
/* data may be written to an internal buffer */

#ifdef NOFTELL
int myputc(c)
char c;
{
  int t;

  if (buffering) {
    buffer[buffcount] = c;
    buffcount = buffcount + 1;
    t = ((int) c) & 0xFF;
  } else {
    t = putc(c, fp);
  };
  return (t);
};
#else
myputc(c)
char c;
{
  return (putc(c,fp));
};
#endif

#ifdef NOFTELL
flushbuffer()
{
  int t;

  t = 0;
  while (t < buffcount) {
    putc(buffer[t], fp);
    t = t + 1;
  };
  buffcount = 0;
  buffering = 0;
};

setbuffering(t)
int t;
{
  buffering = t;
};
#endif

preprocess()
/* massage data before generating MIDI from it */
/* adjust lengths for a>b, a>>B etc. */
/* set up chord length in CHORDOFF field */
/* connect up tied notes */
{
  int j;
  int inchord;
  int chord_num, chord_denom;

  j = 0;
  inchord = 0;
  while (j<notes) {
    switch (feature[j]) {
    case CHORDON:
      if (inchord) {
        event_error("Nested chords");
      };
      inchord = 1;
      chord_num = -1;
      j = j + 1;
      break;
    case CHORDOFF:
      if ((!inchord) || (chord_num == -1)) {
        event_error("Unexpected end of chord");
      } else {
        num[j] = chord_num;
        denom[j] = chord_denom;
      };
      inchord = 0;
      j = j + 1;
      break;
    case NOTE:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case REST:
      if ((inchord) && (chord_num == -1)) {
        chord_num = num[j];
        chord_denom = denom[j];
      };
      j = j + 1;
      break;
    case GT:
      if ((feature[j-1] != NOTE) || (feature[j+1] != NOTE) ||
          (num[j-1] != num[j+1]) || (denom[j-1] != denom[j+1])) {
        event_error("Broken rhythm only allowed between same length notes");
      } else {
        switch (pitch[j]) {
        case 1:
          lenmul(j-1, 4, 3);
          lenmul(j+1, 2, 3);
          break;
        case 2:
          lenmul(j-1, 7, 4);
          lenmul(j+1, 1, 4);
          break;
        case 3:
          lenmul(j-1, 15, 8);
          lenmul(j+1, 1, 8);
          break;
        default:
           event_error("Broken rhythm type not supported");
           break;
        };
      };
      j = j + 1;
      break;
    case LT:
      if ((feature[j-1] != NOTE) || (feature[j+1] != NOTE) ||
          (num[j-1] != num[j+1]) || (denom[j-1] != denom[j+1])) {
        event_error("Broken rhythm only allowed between same length notes");
      } else {
        switch (pitch[j]) {
        case 1:
          lenmul(j-1, 2, 3);
          lenmul(j+1, 4, 3);
          break;
        case 2:
          lenmul(j-1, 1, 4);
          lenmul(j+1, 7, 4);
          break;
        case 3:
          lenmul(j-1, 1, 8);
          lenmul(j+1, 15, 8);
          break;
        default:
           event_error("Broken rhythm type not supported");
           break;
        };
      };
      j = j + 1;
      break;
    case TIE:
      dotie(j, inchord, chord_num, chord_denom);
      j = j + 1;
      break;
    case SLUR_TIE:
      dotie(j, inchord, chord_num, chord_denom);
      j = j + 1;
      break;
    default:
      j = j + 1;
      break;
    };
  };
};

dotie(j, xinchord, xchord_num, xchord_denom)
/* called by preprocess() to handle ties */
int j, xinchord;
int xchord_num, xchord_denom;
{
  int last, next;
  int done;
  int inchord;
  int chord_num, chord_denom;

  chord_num = xchord_num;
  chord_denom = xchord_denom;
  /* find note to be tied */
  last = j;
  while ((last > 0) && (feature[last] != NOTE) && (feature[last] != REST)) {
    last = last - 1;
  };
  if (feature[last] != NOTE) {
    event_error("Cannot find note before tie");
  } else {
    inchord = xinchord;
    feature[last] = TIED_NOTE;
    num[j] = num[last];
    denom[j] = denom[last];
    next = j;
    done = 0;
    while (!done) {
      /* find next applicable note */
      if (inchord) {
        while((next < notes) && (feature[next] != CHORDOFF)) {
          next = next + 1;
        };
        if (num[last]*chord_denom != chord_num*denom[last]) {
          event_error("Tied note in chord must have same length as chord");
        };
        inchord = 0;
      };
      while ((next < notes) && (feature[next] != NOTE) &&
             (feature[next] != REST) && (feature[next] != CHORDON)) {
        next = next + 1;
      };
      if ((next < notes) && (feature[next] == CHORDON)) {
        inchord = 1;
        chord_num = 0;
        /* pick out the appropriate note */
        while ((feature[next] != CHORDOFF) &&
               (!((feature[next] == NOTE) && (pitch[last] == pitch[next])))) {
          if ((chord_num == 0) && 
              ((feature[next] == NOTE) || (feature[next] == REST))) {
            chord_num = num[next];
            chord_denom = denom[next];
          };
          next = next + 1;
        };
      };
      if ((next >= notes) || (feature[next] != NOTE) ||
          (pitch[last] != pitch[next])) {
        if (feature[j] != SLUR_TIE) {
          event_error("Cannot find same pitch notes to tie");
        };
        done = 1;
      } else {
        /* add length[next] to length[j] */
        num[j] = (num[next]*denom[j]) + (num[j]*denom[next]);
        denom[j] = denom[j]*denom[next];
        reduce(&num[j], &denom[j]);
        feature[next] = REST;
        next = next + 1;
        /* look for another tie */
        while ((next < notes) && (feature[next] != TIE) &&
               (feature[next] != SLUR_TIE) && (feature[next] != REST) && 
               (feature[next] != NOTE) && (feature[next] != CHORDON)) {
          next = next + 1;
        };
        if ((next >= notes) || 
            ((feature[next] != TIE) && (feature[next] != SLUR_TIE))) {
          done = 1;
        } else {
          feature[next] = NONOTE;
        };
      };
    };
  };
};

dograce()
/* assign lengths to grace notes before generating MIDI */
{
  int start, end, p, next;
  int grace_num, grace_denom;
  int next_num, next_denom;
  int fact_num, fact_denom;
  int j;
  int nextinchord;

  j = 0;
  while (j < notes) {
    /* first find start of grace notes */
    while ((j < notes) && (feature[j] != GRACEON)) {
      if (feature[j] == GRACEOFF) {
        event_error("} with no matching {");
      };
      if (feature[j] == SETGRACE) {
        gfact_num = pitch[j];
        gfact_denom = denom[j];
      };
      j = j + 1;
    };
    start = j + 1;
    /* now find end of grace notes */
    while ((j < notes) && (feature[j] != GRACEOFF)) {
      if ((feature[j] == GRACEON) && (j != start - 1)) {
        event_error("nested { not allowed");
      };
      j = j + 1;
    };
    end = j - 1;
    /* now find following note */
    nextinchord = 0;
    while ((j < notes) && (feature[j] != NOTE) && 
           (feature[j] != TIED_NOTE) && (feature[j] != REST)) {
      if (feature[j] == GRACEON) {
        event_error("Intervening note needed between grace notes");
      };
      if (feature[j] == CHORDON) {
        nextinchord = 1;
      };
      j = j + 1;
    };
    next = j;
    if ((start < end) && (j >= notes)) {
      event_error("No note found to follow grace notes");
    };
    if (j < notes) {
      /* count up grace units */
      grace_num = 0;
      grace_denom = 1;
      p = start;
      while (p <= end) {
        if ((feature[p] == NOTE) || (feature[p] == REST)) {
          grace_num = grace_num * denom[p] + grace_denom * num[p];
          grace_denom = grace_denom * denom[p];
          reduce(&grace_num, &grace_denom);
        };
        p = p + 1;
      };
      /* adjust notes */
      /* the CHORDOFF element holds the actual duration of a chord */
      p = next;
      while ((p == next) || ((nextinchord) && (p < notes))) {
        next_num = num[p];
        next_denom = denom[p];
        num[p] = num[p] * (gfact_denom - gfact_num);
        denom[p] = next_denom * gfact_denom;
        reduce(&num[p], &denom[p]);
        if (feature[p] == CHORDOFF) {
          nextinchord = 0;
        };
        p = p + 1;
        while ((feature[p] != NOTE) && (feature[p] != CHORDOFF) &&
               (feature[p] != REST) && (p < notes)) {
          p = p + 1;
        };
      };
      fact_num = next_num * grace_denom * gfact_num;
      fact_denom = next_denom * grace_num * gfact_denom;
      reduce(&fact_num, &fact_denom);
      if (feature[next] == TIED_NOTE) {
        while ((feature[next] != TIE) && (feature[next] != SLUR_TIE) &&
               (next < notes)) {
          next = next + 1;
        };
        if ((feature[next] == TIE) || (feature[next] == SLUR_TIE)) {
          /* adjust notes */
          next_num = num[next];
          next_denom = denom[next];
          num[next] = num[next] * (gfact_denom - gfact_num);
          denom[next] = next_denom * gfact_denom;
          reduce(&num[next], &denom[next]);
        };
      };
      /* adjust length of grace notes */
      p = start;
      while (p <= end) {
        if ((feature[p] == NOTE) || (feature[p] == REST)) {
          num[p] = num[p] * fact_num;
          denom[p] = denom[p] * fact_denom;
          reduce(&num[p], &denom[p]);
        };
        p = p + 1;
      };
    };
  };
};

lenmul(n, a, b)
/* multiply note length by a/b */
int n, a, b;
{
  num[n] = num[n] * a;
  denom[n] = denom[n] * b;
  reduce(&num[n], &denom[n]);
};

softcheckbar(pass)
/* allows repeats to be in mid-bar */
int pass;
{
  if ((bar_num-barsize*(bar_denom) >= 0) || (barno <= 0)) {
    checkbar(pass);
  };
};

checkbar(pass) 
/* check to see we have the right number of notes in the bar */
int pass;
{
  char msg[80];

  /* allow zero length bars for typesetting purposes */
  if ((bar_num-barsize*(bar_denom) != 0) && 
      (bar_num != 0) && ((pass == 2) || (barno != 0))) {
    sprintf(msg, "Bar %d has \0", barno);
    numstr(bar_num, msg+strlen(msg));
    if (bar_denom != 1) {
      strcpy(msg+strlen(msg), "/");
      numstr(bar_denom, msg+strlen(msg));
    };
    strcpy(msg+strlen(msg), " units instead of ");
    numstr(barsize, msg+strlen(msg)); 
    if (pass == 2) {
      strcpy(msg+strlen(msg), " in repeat");
    };
    event_warning(msg);
  };
  if (bar_num > 0) {
    barno = barno + 1;
  };
  bar_num = 0;
  bar_denom = 1;
  /* zero place in gchord sequence */
  g_ptr = 0;
  addtoQ(0, g_denom, -1, g_ptr, 0);
  g_next = 0;
};

addunits(a, b)
/* add a/b to the count of units in the bar */
int a, b;
{
  bar_num = bar_num*(b*b_denom) + (a*b_num)*bar_denom;
  bar_denom = bar_denom * (b*b_denom);
  reduce(&bar_num, &bar_denom);
};

reduce(a, b)
/* elimate common factors in fraction a/b */
int *a, *b;
{
  int t, n, m;

  /* find HCF using Euclid's algorithm */
  if (*a > *b) {
    n = *a;
    m = *b;
  } else {
    n = *b;
    m = *a;
  };
  while (m != 0) {
    t = n % m;
    n = m;
    m = t;
  };
  *a = *a/n;
  *b = *b/n;
};

save_state(vec, a, b, c, d, e)
/* save status when we go into a repeat */
int vec[5];
int a, b, c, d, e;
{
  vec[0] = a;
  vec[1] = b;
  vec[2] = c;
  vec[3] = d;
  vec[4] = e;
};

restore_state(vec, a, b, c, d, e)
/* restore status when we loop back to do second repeat */
int vec[5];
int *a, *b, *c, *d, *e;
{
  *a = vec[0];
  *b = vec[1];
  *c = vec[2];
  *d = vec[3];
  *e = vec[4];
};

zerobar()
/* start a new count of beats in the bar */
{
  bar_num = 0;
  bar_denom = 1;
  barno = barno + 1;
};

placerep(j)
/* patch up missing repeat */
int j;
{
  event_warning("Assuming repeat");
  switch(feature[j]) {
  case DOUBLE_BAR:
    feature[j] = BAR_REP;
    break;
  case SINGLE_BAR:
    feature[j] = BAR_REP;
    break;
  case REP_BAR:
    feature[j] = DOUBLE_REP;
    break;
  case BAR_REP:
    event_error("Too many end repeats");
    break;
  case DOUBLE_REP:
    event_error("Too many end repeats");
    break;
  default:
    event_error("Internal error - please report");
    break;
  };
};

fixreps()
/* find and correct missing repeats in music */
{
  int j;
  int rep_point;
  int expect_repeat;
  int use_next;

  expect_repeat = 0;
  use_next = 0;
  j = 0;
  while (j < notes) {
    switch(feature[j]) {
    case SINGLE_BAR:
      if (use_next) {
        rep_point = j;
        use_next = 0;
      };
      break;
    case DOUBLE_BAR:
      rep_point = j;
      use_next = 0;
      break;
    case BAR_REP:
      expect_repeat = 1;
      use_next = 0;
      break;
    case REP_BAR:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 0;
      rep_point = j;
      use_next = 0;
      break;
    case REP_BAR2:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 0;
      use_next = 1;
      break;
    case DOUBLE_REP:
      if (!expect_repeat) {
        placerep(rep_point);
      };
      expect_repeat = 1;
      break;
    default:
      break;
    };
    j = j + 1;
  };
};

dohornpipe()
/* apply hornpipe rythm if there was a R:hornpipe field */
{
  int j;
  int lastj, lastbar, lastnum;
  int inchord, chordcount;

  zerobar();
  set_meter(time_num, time_denom);
  inchord = 0;
  barno = 0;
  lastj = 0;
  lastbar = -1;
  lastnum = 0;
  j = 0;
  while (j < notes) {
    switch(feature[j]) {
    case NOTE:
      if (!inchord) {
        int this_num, this_denom;

        this_num = num[j];
        this_denom = denom[j];
        /* only look at 1/8 notes */
        if ((num[j] == 1) && (denom[j] == 2)) {
          if (bar_denom == 1) {
            lastj = j;
            lastbar = barno;
            lastnum = bar_num*2;
          };
          if ((bar_denom == 2) && (barno == lastbar) && 
              (bar_num - lastnum == 1)){
            lenmul(lastj, 4, 3);
            lenmul(j, 2, 3);
          };
        };
        addunits(this_num, this_denom);
      } else {
        chordcount = chordcount + 1;
        if (chordcount == 1) addunits(num[j], denom[j]);
      };
      break;
    case REST:
      addunits(num[j], denom[j]);
      break;
    case TIED_NOTE:
      addunits(num[j], denom[j]);
      break;
    case CHORDON:
      inchord = 1;
      chordcount = 0;
      break;
    case CHORDOFF:
      inchord = 0;
      break;
    case SINGLE_BAR:
      zerobar();
      break;
    case DOUBLE_BAR:
      zerobar();
      break;
    case BAR_REP:
      zerobar();
      break;
    case REP_BAR:
      zerobar();
      break;
    case BAR1:
      zerobar();
      break;
    case REP_BAR2:
      zerobar();
      break;
    case DOUBLE_REP:
      zerobar();
      break;
    case TIME:
      set_meter(pitch[j], denom[j]);
      setbeat();
      break;
    };
    j = j + 1;
  };
};

int findchannel()
/* work out next available channel */
{
  int j;

  j = 0;
  while ((j<16) && (channels[j] != 0)) {
    j = j + 1;
  };
  if (j == 16) {
    event_error("Not enough channels");
    j = 0;
  };
  return (j);
};

findpart(j, partno)
/* find out where part starts */
int *j;
int *partno;
{
  char newpart;
  int  n;

  *partno = *partno + 1;
  while ((*partno <= parts) && 
         (part_start[(int)part[*partno] - (int) 'A'] == -1)) {
    event_error("Part not defined");
    *partno = *partno + 1;
  };
  if (*partno > parts) {
    *j = notes;
  } else {
    newpart = part[*partno];
    n = (int) newpart - (int) 'A';
    *j = part_start[n];
  };
};

checkline(j, w)
int *j;
int *w;
/* make sure number of syllables in w: matches number of notes */
{
  char msg[100];

  if (koverflow > 0) {
    sprintf(msg, "%d more syllables than notes for word line %d", 
                 koverflow, *w);
    event_error(msg);
  } else {
    while ((*j < notes) && (feature[*j] != NEWLINE)) {
      koverflow = koverflow - advancenote(j);
    };
    if (koverflow < 0) {
      sprintf(msg, "%d more notes than syllables for word line %d", 
                 -koverflow, *w);
      event_error(msg);
    };
  };     
  *j = *j + 1;
  koverflow = 0;
};

char getword(syllable, place, w, j)
/* picks up next syllable out of w: field */
char syllable[200];
int* place;
int* w;
int* j;
{
  char c;
  int i;

  i = 0;
  if (*w >= wcount) {
    syllable[i] = '\0';
    return ('\0');
  };
  if (*place == 0) {
    if ((*w % 2) == 0) {
      syllable[i] = '/';
    } else {
      syllable[i] = '\\';
    };
    i = i + 1;
  };
  if (kspace) {
    syllable[i] = ' ';
    i = i + 1;
  };
  c = *(words[*w]+(*place));
  while ((c != ' ') && (c != '-') && (c != '\0') && (c != '|') && (c != '*') &&
         (c != '_') && (c != '\\')) {
    syllable[i] = c;
    if (syllable[i] == '~') syllable[i] = ' ';
    if ((syllable[i] == '\\') && (*(words[*w]+(*place+1)) == '-')) {
      syllable[i] = '-';
      *place = *place + 1;
    };
    i = i + 1;
    *place = *place + 1;
    c = *(words[*w]+(*place));
  };
  switch (c)  {
  case ' ':
    /* skip over any other spaces */
    while (*(words[*w]+(*place)) == ' ') {
      *place = *place + 1;
    };
    kspace = 1;
    break;
  case '-':
    *place = *place + 1;
    kspace = 0;
    break;
  case '_':
    *place = *place + 1;
    kspace = 0;
    break;
  case '\0':
    *w = *w + 1;
    kspace = 0;
    *place = 0;
    break;
  case '\\':
    *w = *w + 1;
    kspace = 1;
    *place =0;
    break;
  case '|':
    *place = *place + 1;
    break;
  case '*':
    kspace = 0;
    *place = *place + 1;
    break;
  };
  syllable[i] = '\0';
  return (c);
};

int advancenote(j)
/* part of w: field handling - steps on one note */
int* j;
{
  int done;

  if (feature[*j] == NEWLINE) {
    koverflow = koverflow + 1;
    return(0);
  };
  done = 0;
  if (ktied) {
    ktied = 0;
    return(0);
  };
  while ((*j < notes) && (done == 0) && (feature[*j] != NEWLINE)) {
    switch(feature[*j]) {
    case NOTE:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
        done = 1;
      };
      break;
    case REST:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case TIED_NOTE:
      {
        int i;

        /* find tie */
        i = *j;
        while ((i < notes) && (feature[i] != TIE) &&
               (feature[i] != SLUR_TIE)) i = i + 1;
        if ((feature[i] != TIE) && (feature[i] != SLUR_TIE)) {
          event_error("Note tie error");
          i = *j;
        };
        if (!inchord) {
          delay(num[i], denom[i], 0);
          done = 1;
          ktied = 1;
        };
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[*j], denom[*j], 0);
      done = 1;
      break;
    case VOICE:
      if (pitch[*j] != 1) {
        int foundvoice;

        foundvoice = 0;
        while ((*j < notes) && (!foundvoice)) {
          if (((feature[*j] == VOICE) && (pitch[*j] == 1)) ||
              (feature[*j] == PART)) {
            foundvoice = 1;
          } else {
            *j = *j + 1;
          };
        };
      };
      break;
    case TEMPO:
      if (*j > 0) {
        div_factor = pitch[*j];
      };
      break;
    case TITLE:
      break;
    default:
      break;
    };
    *j = *j + 1;
  };
  if (done == 0) {
    koverflow = koverflow + 1;
  };
  return(done);
};

advancebar(j)
/* part of w: field handling - goes to start of next bar */
int* j;
{
  int done;

  if (feature[*j] == NEWLINE) {
    koverflow = koverflow + 1;
    return(0);
  };
  ktied = 0;
  done = 0;
  while ((*j < notes) && (done == 0) && (feature[*j] != NEWLINE)) {
    switch(feature[*j]) {
    case NOTE:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case REST:
      if (!inchord) {
        delay(num[*j], denom[*j], 0);
      };
      break;
    case TIED_NOTE:
      {
        int i;

        /* find tie */
        i = *j;
        while ((i < notes) && (feature[i] != TIE) &&
               (feature[i] != SLUR_TIE)) i = i + 1;
        if ((feature[i] != TIE) && (feature[i] != SLUR_TIE)) {
          event_error("Note tie error");
          i = *j;
        };
        if (!inchord) {
          delay(num[i], denom[i], 0);
        };
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[*j], denom[*j], 0);
      break;
    case VOICE:
      if (pitch[*j] != 1) {
        int foundvoice;

        foundvoice = 0;
        while ((*j < notes) && (!foundvoice)) {
          if (((feature[*j] == VOICE) && (pitch[*j] == 1)) ||
              (feature[*j] == PART)) {
            foundvoice = 1;
          } else {
            *j = *j + 1;
          };
        };
      };
      break;
    case SINGLE_BAR:
      done = 1;
      break;
    case DOUBLE_BAR:
      done = 1;
      break;
    case BAR_REP:
      done = 1;
      break;
    case REP_BAR:
      done = 1;
      break;
    case BAR1:
      done = 1;
      break;
    case REP_BAR2:
      done = 1;
      break;
    case DOUBLE_REP:
      done = 1;
      break;
    case TEMPO:
      if (*j > 0) {
        div_factor = pitch[*j];
      };
      break;
    default:
      break;
    };
    *j = *j + 1;
  };
  if (done == 0) {
    koverflow = koverflow + 1;
  };
};

karaoketrack()
/* write out karaoke track based on w: fields */
{
  int w;
  int j;
  char syllable[40];
  int place;
  char term;

  text_data("@LENGL");
  j = 0;
  while ((j < notes) && (feature[j] != TITLE)) j = j+1;
  if (feature[j] == TITLE) {
    char atitle[200];

    strcpy(atitle, "@T");
    strcpy(atitle+2, &text_buffer[pitch[j]]);
    text_data(atitle);
  };
  w = 0;
  place = 0;
  j = 0;
  inchord = 0;
  kspace = 0;
  ktied = 0;
  koverflow = 0;
  while (w < wcount) {
    term = getword(syllable, &place, &w, &j);
    if (strlen(syllable) > 0) text_data(syllable);
    switch (term) {
      case ' ':
        advancenote(&j);
        break;
      case '-':
        advancenote(&j);
        break;
      case '_':
        advancenote(&j);
        break;
      case '\0':
        advancenote(&j);
        checkline(&j, &w);
        break;
      case '\\':
        advancenote(&j);
        break;
      case '|':
        advancebar(&j);
        break;
      case '*':
        advancenote(&j);
        advancenote(&j);
        break;
    };
  };
};

long mywritetrack(xtrack)
/* this routine writes a MIDI track  */
int xtrack;
{
  int track;
  int j, pass;
  int expect_repeat;
  int slurring;
  int state[5];
  int foundvoice;
  void mf_write_tempo();

  track = xtrack;
  if (karaoke) {
    if (track == 2) {
      delta_time = 0;
      karaoketrack();
      return(0L);
    } else {
      if (track > 2) track = track - 1;
    };
  };
  if (track == 0) {
    if (karaoke) {
      text_data("@KMIDI KARAOKE FILE");
    };
    mf_write_tempo(tempo);
    /* write key */
    write_keysig(sf, mi);
    /* write timesig */
    write_meter(time_num, time_denom);
    if (maxvoice > 0) {
       return(0);
    };
  } else {
    printf("track %d\n", track);
  };
  tracklen = 0;
  partno = -1;
  g_started = 0;
  g_ptr = 0;
  addtoQ(0, g_denom, -1, g_ptr, 0);
  g_next = 0;
  tuplecount = 0;
  inchord = 0;
  channel = findchannel();
  channels[channel] = 1;
  fun.chan = findchannel();
  channels[fun.chan] = 1;
  gchord.chan = findchannel();
  channels[fun.chan] = 0;
  channels[channel] = 0;
  if (program != -1) {
    write_program(program, channel);
  };
  /* write notes */
  j = 0;
  if ((maxvoice == 0) || (track == 1)) {
    foundvoice = 1;
  } else {
    foundvoice = 0;
    while ((j < notes) && (!foundvoice)) {
      if ((feature[j] == VOICE) && (pitch[j] == track)){
        foundvoice = 1;
      };
      j = j + 1;
    };
  };
  barno = 0;
  bar_num = 0;
  bar_denom = 1;
  delta_time = 0;
  err_num = 0;
  err_denom = 1;
  pass = 1;
  save_state(state, j, barno, div_factor, transpose, channel);
  slurring = 0;
  expect_repeat = 0;
  while (j < notes) {
    switch(feature[j]) {
    case NOTE:
      noteon(j);
      /* set up note off */
      addtoQ(num[j], denom[j], pitch[j] + transpose, channel, -1);
      if (!inchord) {
        delay(num[j], denom[j], 0);
        addunits(num[j], denom[j]);
      };
      break;
    case REST:
      if (!inchord) {
        delay(num[j], denom[j], 0);
        addunits(num[j], denom[j]);
      };
      break;
    case TIED_NOTE:
      {
        int i;

        /* find tie */
        i = j;
        while ((i < notes) && (feature[i] != TIE) &&
               (feature[i] != SLUR_TIE)) i = i + 1;
        if ((feature[i] != TIE) && (feature[i] != SLUR_TIE)) {
          event_error("Note tie error");
          i = j;
        };
        noteon(j);
        /* set up note off */
        addtoQ(num[i], denom[i], pitch[j] + transpose, channel, -1);
        if (!inchord) {
          delay(num[j], denom[j], 0);
          addunits(num[j], denom[j]);
        };
      };
      break;
    case CHORDON:
      inchord = 1;
      break;
    case CHORDOFF:
      inchord = 0;
      delay(num[j], denom[j], 0);
      addunits(num[j], denom[j]);
      break;
    case PART:
      if (parts == -1) {
        char msg[1];

        msg[0] = (char) pitch[j];
        mf_write_meta_event((long) 0, marker, msg, 1);
      } else {
        findpart(&j, &partno);
      };
      break;
    case VOICE:
      if (pitch[j] == track) {
        foundvoice = 1;
      } else {
        foundvoice = 0;
        while ((j < notes) && (!foundvoice)) {
          if (((feature[j] == VOICE) && (pitch[j] == track)) ||
              ((feature[j] == PART) && (track == 1))) {
            foundvoice = 1;
          } else {
            j = j + 1;
          };
        };
      };
      break;
    case TEXT:
      mf_write_meta_event((long) 0, text_event, &text_buffer[pitch[j]],
                          strlen(&text_buffer[pitch[j]]));
      break;
    case TITLE:
      mf_write_meta_event((long) 0, sequence_name, &text_buffer[pitch[j]],
                          strlen(&text_buffer[pitch[j]]));
      break;
    case SINGLE_BAR:
      checkbar(pass);
      break;
    case DOUBLE_BAR:
      checkbar(pass);
      break;
    case BAR_REP:
      softcheckbar(pass);
      if (expect_repeat) {
        event_error("Expected end repeat not found at |:");
      };
      save_state(state, j, barno, div_factor, transpose, channel);
      expect_repeat = 1;
      pass = 1;
      break;
    case REP_BAR:
      softcheckbar(pass);
      if (pass == 1) {
        if (!expect_repeat) {
          event_error("Found unexpected :|");
        } else {
          pass = 2;
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
        };
      } else {
        pass = 1;
        expect_repeat = 0;
      };
      break;
    case BAR1:
      checkbar(pass);
      if (pass == 2) {
        while ((j<notes) && (feature[j] != REP_BAR2)) {
          j = j + 1;
        };
        barno = barno + 1;
        if (feature[j] != REP_BAR2) {
          event_error("Not found :|2");
        };
        pass = 1;
        expect_repeat = 0;
      } else {
        if (!expect_repeat) {
          event_error("|1 or | [1 found outside repeat");
        };
      };
      break;
    case REP_BAR2:
      checkbar(pass);
      if (pass == 1) {
        if (!expect_repeat) {
          event_error(":|2 found outside repeat context");
        } else {
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
          pass = 2;
        };
      } else {
        event_error("Found unexpected :|2");
      };
      break;
    case DOUBLE_REP:
      softcheckbar(pass);
      if (pass == 2) {
        expect_repeat = 1;
        save_state(state, j, barno, div_factor, transpose, channel);
        pass = 1;
      } else {
        if (!expect_repeat) {
          event_error("Found unexpected ::");
          expect_repeat = 1;
          save_state(state, j, barno, div_factor, transpose, channel);
          pass = 1;
        } else {
          restore_state(state, &j, &barno, &div_factor, &transpose, &channel);
          slurring = 0;
          pass = 2;
        };
      };
      break;
    case GCHORD:
      basepitch = pitch[j];
      seventh = denom[j] & 1;
      mode = (enum chordmode) ((denom[j] & 6) >> 1);
      force_fun = (denom[j] & 8) >> 3;
      g_started = 1;
      break;
    case GCHORDON:
      gchords = 1;
      break;
    case GCHORDOFF:
      gchords = 0;
      break;
    case DYNAMIC:
      dodeferred(deferred[pitch[j]]);
      break;
    case KEY:
      write_keysig(pitch[j], denom[j]);
      break;
    case TIME:
      write_meter(pitch[j], denom[j]);
      break;
    case TEMPO:
      if (j > 0) {
        div_factor = pitch[j];
      };
      break;
    case CHANNEL:
      channel = pitch[j];
      break;
    case PROGRAM:
      write_program(pitch[j], denom[j]);
      break;
    case TRANSPOSE:
      transpose = pitch[j];
      break;
    case SLUR_ON:
      if (slurring) {
        event_error("Unexpected start of slur found");
      };
      slurring = 1;
      break;
    case SLUR_OFF:
      if (!slurring) {
        event_error("Unexpected end of slur found");
      };
      slurring = 0;
      break;
    default:
      break;
    };
    j = j + 1;
  };
  if (expect_repeat) {
    event_error("Missing :| at end of tune");
  };
  clearQ();
  tracklen = tracklen + delta_time;
  if (track == 1) {
    tracklen1 = tracklen;
  } else {
    if ((track != 0) && (tracklen != tracklen1)) {
      char msg[100];

      sprintf(msg, "Track %d is %d units long not %d", 
              track, tracklen, tracklen1);
      event_warning(msg);
    };
  };
  return (delta_time);
};

set_meter(n, m)
int n, m;
{
  /* set up barsize */
  barsize = n;
  if (barsize % 3 == 0) {
    beat = 3;
  } else {
    if (barsize % 2 == 0) {
      beat = 2;
    } else {
      beat = barsize;
    };
  };
  /* correction factor to make sure we count in the right units */
  if (m > 4) {
    b_num = m/4;
    b_denom = 1;
  } else {
   b_num = 1;
   b_denom = 4/m;
  };
};

write_meter(n, m)
int n, m;
{
  int t, c, dd;
  char data[4];

  set_meter(n, m);

  dd = 0;
  t = m;
  while (t > 1) {
    dd = dd + 1;
    t = t/2;
  };
  data[0] = (char)n;
  data[1] = (char)dd;
  if (n%2 == 0) {
    data[2] = (char)(24*2*n/m);
  } else {
    data[2] = (char)(24*n/m);
  };
  data[3] = 8;
  mf_write_meta_event((long) 0, time_signature, data, 4);
};

write_keysig(sf, mi)
int sf, mi;
{
  char data[2];

  data[0] = (char) (0xff & sf);
  data[1] = (char) mi;
  mf_write_meta_event((long) 0, key_signature, data, 2);
};

noteon(n)
/* compute note data and call noteon_data to write MIDI note event */
int n;
{
  int vel;

  /* set velocity */
  if (bar_num == 0) {
    vel = loudnote;
  } else {
    if ((bar_denom == 1) && ((bar_num % beat) == 0)) {
      vel = mednote;
    } else {
      vel = softnote;
    };
  };
  noteon_data(pitch[n] + transpose, channel, vel);
};

text_data(s)
/* write text event */
char* s;
{
  mf_write_meta_event(delta_time, text_event, s, strlen(s));
  tracklen = tracklen + delta_time;
  delta_time = (long) 0;
};

noteon_data(pitch, channel, vel)
int pitch, channel, vel;
{
  midi_noteon(delta_time, pitch, channel, vel);
  tracklen = tracklen + delta_time;
  delta_time = (long) 0;
};

write_program(p, channel)
int p, channel;
{
  char data[1];

  data[0] = p;
  mf_write_midi_event(delta_time, program_chng, channel, data, 1);
  tracklen = tracklen + delta_time;
  delta_time = 0;
};

write_control(channel, data, n)
/* write MIDI control event */
int channel, n;
char data[];
{
  mf_write_midi_event(delta_time, control_change, channel, data, n);
};

delay(a, b, c)
/* wait for time a/b */
int a, b, c;
{
  int dt;

  dt = (div_factor*a)/b + c;
  err_num = err_num * b + ((div_factor*a)%b)*err_denom;
  err_denom = err_denom * b;
  reduce(&err_num, &err_denom);
  dt = dt + (err_num/err_denom);
  err_num = err_num%err_denom;
  timestep(dt, 0);
};

midi_noteon(delta_time, pitch, chan, vel)
long delta_time;
int pitch, chan, vel;
{
  char data[2];

  data[0] = (char) pitch;
  data[1] = (char) vel;
  mf_write_midi_event(delta_time, note_on, chan, data, 2);
  channels[chan] = 1;
};

midi_noteoff(delta_time, pitch, chan)
long delta_time;
int pitch, chan;
{
  char data[2];

  data[0] = (char) pitch;
  data[1] = (char) 0;
  mf_write_midi_event(delta_time, note_off, chan, data, 2);
};

save_note(num, denom, pitch, chan, vel)
/* queue up note */
int num, denom;
int pitch, chan, vel;
{
  noteon_data(pitch, chan, vel);
  addtoQ(num, denom, pitch, chan, -1);
};

dogchords(i)
/* generate accompaniment notes */
int i;
{
  if ((i == g_ptr) && (g_ptr < strlen(gchord_seq))) {
    int len;
    char action;

    action = gchord_seq[g_ptr];
    len = gchord_len[g_ptr];
    if (force_fun && (action == 'c')) {
      action = 'f';
    };
    switch (action) {
    case 'z':
      break;
    case 'c':
      /* do chord */
      if (g_started && gchords) {
        save_note(g_num*len, g_denom, basepitch+gchord.base, gchord.chan, 
                  gchord.vel);
        if ((mode == min) || (mode == dim)) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+3, 
                    gchord.chan, gchord.vel);
        } else {
          save_note(g_num*len, g_denom, basepitch+gchord.base+4, 
                    gchord.chan, gchord.vel);
        };
        if ((mode == maj) || (mode == min)) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+7, gchord.chan,
                  gchord.vel);
        };
        if (mode == dim) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+6, gchord.chan,
                  gchord.vel);
        };
        if (mode == aug) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+8, gchord.chan,
                  gchord.vel);
        };
        if (seventh) {
          save_note(g_num*len, g_denom, basepitch+gchord.base+11, 
                    gchord.chan, gchord.vel);
        };
      };
      break;
    case 'f':
      if (g_started && gchords) {
        /* do fundamental */
        save_note(g_num*len, g_denom, basepitch+fun.base, fun.chan, fun.vel);
      };
    };
    g_ptr = g_ptr + 1;
    addtoQ(g_num*len, g_denom, -1, g_ptr, 0);
  };
};

/* routines to handle note queue */

addtoQ(num, denom, pitch, chan, d) 
int num, denom, pitch, chan, d;
{
  int i, done;
  int wait;
  int *ptr;

  wait = ((div_factor*num)/denom) + d;
  /* find free space */
  if (freehead == -1) {
    event_error("Note off queue overflow");
  } else {
    i = freehead;
    freehead = Q[freehead].next;
  };
  Q[i].pitch = pitch;
  Q[i].chan = chan;
  /* find place in queue */
  ptr = &Qhead;
  done = 0;
  while (!done) {
    if (*ptr == -1) {
      *ptr = i;
      Q[i].next = -1;
      Q[i].delay = wait;
      done = 1;
    } else {
      if (Q[*ptr].delay > wait) {
        Q[*ptr].delay = Q[*ptr].delay - wait;
        Q[i].next = *ptr;
        Q[i].delay = wait;
        *ptr = i;
        done = 1;
      } else {
        wait = wait - Q[*ptr].delay;
        ptr = &Q[*ptr].next;
      };
    };
  };
};

removefromQ(i)
int i;
{
  if (i == -1) {
    event_error("Internal error - nothing to remove from queue");
  } else {
    if (Q[Qhead].delay != 0) {
      event_error("Internal error - queue head has non-zero time");
    };
    Qhead = Q[i].next;
    Q[i].next = freehead;
    freehead = i;
  };
};

clearQ()
{
  int time;

  /* remove gchord requests */
  time = 0;
  while ((Qhead != -1) && (Q[Qhead].pitch == -1)) {
    time = time + Q[Qhead].delay;
    Qhead = Q[Qhead].next;
  };
  if (Qhead != -1) {
    timestep(time, 1);
  };
  /* do any remaining note offs, but don't do chord request */
  while (Qhead != -1) {
    event_error("Sustained notes beyond end of track");
    timestep(Q[Qhead].delay+1, 1);
  };
};

printQ()
{
  int t;

  t = Qhead;
  printf("Q:");
  while (t != -1) {
    printf("p(%d)-%d->", Q[t].pitch, Q[t].delay);
    t = Q[t].next;
  };
  printf("\n");
};

timestep(t, atend)
int t;
int atend;
{
  int time;
  int headtime;

  time = t;
  /* process any notes waiting to finish */
  while ((Qhead != -1) && (Q[Qhead].delay < time)) {
    headtime = Q[Qhead].delay;
    delta_time = delta_time + (long) headtime;
    time = time - headtime;
    advanceQ(headtime);
    if (Q[Qhead].pitch == -1) {
      if (!atend) {
        dogchords(Q[Qhead].chan);
      };
    } else {
      midi_noteoff(delta_time, Q[Qhead].pitch, Q[Qhead].chan);
      tracklen = tracklen + delta_time;
      delta_time = (long) 0;
    };
    removefromQ(Qhead);
  };
  if (Qhead != -1) {
    advanceQ(time);
  };
  delta_time = delta_time + (long) time;
};

advanceQ(t)
int t;
{
  if (Qhead == -1) {
    event_error("Internal error - empty queue");
  } else {
    Q[Qhead].delay = Q[Qhead].delay - t;
  };
};

startfile()
/* called at the beginning of an abc tune by event_refno */
/* This sets up all the default values */
{
  int j;

  printf("scanning tune\n");
  /* set up defaults */
  sf = 0;
  mi = 0;
  setmap(0, basemap, basemul);
  copymap();
  time_num = 4;
  time_denom = 4;
  default_length = -1;
  event_tempo(120, 1, 4, 0);
  notes = 0;
  ndeferred = 0;
  text_end= 0;
  loudnote = 105;
  mednote = 95;
  softnote = 80;
  program = -1;
  gfact_num = 1;
  gfact_denom = 3;
  transpose = 0;
  hornpipe = 0;
  karaoke = 0;
  wcount = 0;
  parts = -1;
  middle_c = 60;
  for (j=0; j<26; j++) {
    part_start[j] = -1;
  };
  for (j=0; j<16;j++) {
    channels[j] = 0;
  };
  gchords = 1;
  set_gchords("z");
  fun.base = 36;
  fun.vel = 80;
  gchord.base = 48;
  gchord.vel = 75;
#ifdef NOFTELL
  buffcount = 0;
  buffering = 0;
#endif
};

tempounits(t_num, t_denom)
/* interprets Q: once default length is known */
int *t_num, *t_denom;
{
  /* calculate unit for tempo */
  if (tempo_num == 0) {
    *t_num = 1;
    *t_denom = default_length;
  } else {
    if (relative_tempo) {
      *t_num = tempo_num;
      *t_denom = tempo_denom*default_length;
    } else {
      *t_num = tempo_num;
      *t_denom = tempo_denom;
    };
  };
};

setbeat()
/* default accompaniment patterns for various time signatures */
{
  int i;

  /* set up chord/fundamental sequence if not already set */
  if ((time_num == 2) && (time_denom == 2)) {
    set_gchords("fzczfzcz");
  };
  if (((time_num == 2) || (time_num == 4)) && (time_denom == 4)) {
    set_gchords("fzczfzcz");
  };
  if ((time_num == 3) && (time_denom == 4)) {
    set_gchords("fzczcz");
  };
  if ((time_num == 6) && (time_denom == 8)) {
    set_gchords("fzcfzc");
  };
  if ((time_num == 9) && (time_denom == 8)) {
    set_gchords("fzcfzcfzc");
  };
};

headerprocess()
/* called after the K: field has been reached, signifying the end of */
/* the header and the start of the tune */
{
  int t_num, t_denom;
  int i;

  gracenotes = 0; /* not in a grace notes section */
  inchord = 0; /* not in a chord */
  inslur = 0; /* not in a slur */
  /* calculate time for a default length note */
  if (default_length == -1) {
    if (((float) time_num)/time_denom < 0.75) {
      default_length = 16;
    } else {
      default_length = 8;
    };
  };

  tempounits(&t_num, &t_denom);
  /* make tempo in terms of 1/4 notes */
  tempo = (long) 60*1000000*t_denom/(Qtempo*4*t_num);
  div_factor = division;
  setbeat();
  maxvoice = 0;

  /* initialize queue of notes waiting to finish */
  Qhead = -1;
  freehead = 0;
  for (i=0; i<QSIZE-1; i++) {
    Q[i].next = i + 1;
  };
  Q[QSIZE-1].next = -1;
  freetail = QSIZE-1;
};

finishfile()
/* end of tune has been reached - write out MIDI file */
{
  if (!pastheader) {
    event_error("No K: field found at start of tune");
  } else {
    int i; 

    if (parts > -1) {
      addfeature(PART, 'Z', 0, 0);
    };
    preprocess();
    if ((parts == -1) && (maxvoice <= 1)) {
      fixreps();
    };
    dograce();
    if (hornpipe) {
      dohornpipe();
    };
    if (check) {
      Mf_putc = nullputc;
      for (i=0; i<= maxvoice; i++) {
        mywritetrack(i);
      };
    } else {
      if ((fp = fopen(outname, "w")) == 0L)
        exit(1);
      printf("writing MIDI file %s\n", outname);
      Mf_putc = myputc;
      Mf_writetrack = mywritetrack;
      if ((maxvoice == 0) && (!karaoke)) {
        mfwrite(0, 1, division, fp);
      } else {
        if (maxvoice == 0) maxvoice = 1;
        mfwrite(1, maxvoice+karaoke+1, division, fp);
      };
      fclose(fp);
#ifdef __MACINTOSH__
      (void) setOutFileCreator(outname,'Midi','ttxt');
#endif // __MACINTOSH__

    };
    for (i=0; i<ndeferred; i++) {
      free(deferred[i]);
    };
    for (i=0; i<wcount; i++) {
      free(words[i]);
    };
  };
};

