// soundheader.h

/******************************************************************************
 *
 *  MiXViews - an X window system based sound & data editor/processor
 *
 *  Copyright (c) 1993, 1994 Regents of the University of California
 *
 *  Author:     Douglas Scott
 *  Date:       December 13, 1994
 *
 *  Permission to use, copy and modify this software and its documentation
 *  for research and/or educational purposes and without fee is hereby granted,
 *  provided that the above copyright notice appear in all copies and that
 *  both that copyright notice and this permission notice appear in
 *  supporting documentation. The author reserves the right to distribute this
 *  software and its documentation.  The University of California and the author
 *  make no representations about the suitability of this software for any 
 *  purpose, and in no event shall University of California be liable for any
 *  damage, loss of data, or profits resulting from its use.
 *  It is provided "as is" without express or implied warranty.
 *
 ******************************************************************************/


// Specialized Header subclasses for reading, parsing, and verifying headers
// on various types of on-disk sound files.

#ifndef SOUNDHEADER_H
#ifdef __GNUG__
#pragma interface
#endif
#define SOUNDHEADER_H

#include "diskstruct.h"
#include "header.h"
#include "formatrequester.h"

#define _SND ((int)0x2e736e64)
#define DNS_ ((int)0x646e732e)
#define SF_MAGIC ((int)0x0001a364)	/* used by old BSD/IRCAM style headers */
#define SF_SWAPMAGIC ((int)0x64a30100)	/* same value, byte-swapped */

#define IRCAM_SUNMAGIC ((int)0x64a30200)	/* special values for new IRCAM */
#define IRCAM_MIPSMAGIC ((int)0x64a30300)
#define IRCAM_NEXTMAGIC ((int)0x64a30400)

#define SIZEOF_BSD_HEADER 1024
#define MAXCOMM 512

class SoundHeader : public Header {
	typedef Header Super;
public:		// class methods
	enum Type { None=0x1, Snd=0x2, Hybrid=0x4, Ircam=0x8, Aifc=0x10, Wave=0x20 };
	enum Chans { MaxChannels = 4 };
	static Type defaultHeaderType() { return default_HeaderType; }
	static Type setDefaultHeaderType(Type type) {
		Type oldType = default_HeaderType;
		default_HeaderType = type;
		return oldType;
	}
		// meta-constructors
	static SoundHeader* create(DataFile*, DataType, int, int,
	                           double, boolean reading);
	static SoundHeader* create(Type, DataType, int, int, double);
public:		// object methods
	virtual int sampleRate() { return samprate; }
	virtual double peakAmp() { return peakamp; }
	virtual Type headerType() { return header_type; }
protected:
	redefined int secondsToBytes(double);
	redefined int checkHeader();
	virtual boolean isValid(DataType)=0;
	SoundHeader(DataType, int rate, int chans, double peak, int magic);
	class SoundConfigRequester
			: public Header::ConfigRequester, public FormatRequester {
	public:
		SoundConfigRequester(Header *);
	protected:
    	redefined void configureRequest(Request *);
	};
	friend void SoundConfigRequester::configureRequest(Request *);
	redefined const char* magicError();
public:
	redefined Requester* configRequester() {
		return new SoundConfigRequester(this);
	}
protected:	// object members
	Type header_type;
	int samprate;
	double peakamp;
private:	// class members
	static Type default_HeaderType;
};

// this is a generic header for either NeXT or Sparc files

class SndOrIrcamSoundHeader : public SoundHeader {
protected:
	SndOrIrcamSoundHeader(DataType, int rate, int ch, double pk, int mgc);
	redefined boolean isMagic();
	redefined boolean magicIsSwapped();
	redefined int readInfo(DataFile *);
	redefined int readComment(DataFile *);
	redefined int writeComment(DataFile *);

	int readSndInfo(DataFile *);
	int readIrcamInfo(DataFile *);
	int extractPeak(const char *);
	int extractComment(const char *);
	int loadPeak(const char *);
	int loadComment(const char *);
	redefined boolean isValid(DataType t) {
		return (t == MuLawData || t == ShortData || t == FloatData);
	}
protected:
	enum SF_Code { SF_END = 0, SF_MAXAMP, SF_COMMENT,
	               SF_PVDATA, SF_AUDIOENCOD, SF_CODMAX = SF_AUDIOENCOD
	};
	struct SFCode {
		short	code;
		short	bsize;
	};
	struct SFMaxamp {
		float	value[MaxChannels];
		int	samploc[MaxChannels];
		int	timetag;
	};
	struct SFComment {
		char 	comment[MAXCOMM];
	};
	struct SFPvData {			// this code written by pvanal
		short   frameSize;
		short   frameIncr;
	};
	struct SFAudioEncod {		// ditto
		short   encoding;
		short   grouping;
	};
	class SndStruct : public DiskStruct {
		typedef DiskStruct Super;
	public:
		SndStruct(int dl=0, int ds=0, int fmt=0, int sr=0, int ch=0) :
			magic(_SND), data_loc(dl), data_size(ds), format(fmt),
			samp_rate(sr), nchans(ch) {}
		redefined long readSize();
	protected:
		redefined ValueSetterBase** valueSetters();
		enum { NumElements = 6 };
	private:
		friend class SndOrIrcamSoundHeader;
	private:
		int magic;
		int data_loc;
		int data_size;
		int format;
		int samp_rate;
		int nchans;
	};
	class IrcamStruct : public VariableBlockDiskStruct {
		typedef VariableBlockDiskStruct Super;
	public:
		IrcamStruct(float sr=0, int ch=0, int pm=0, int trim=0);
		virtual ~IrcamStruct();
		redefined long readSize() { return SIZEOF_BSD_HEADER - lengthTrim; }
		boolean isValid() { return sf_magic == SF_MAGIC; }
	protected:
		redefined ValueSetterBase** valueSetters();
		redefined void createVariableBlock();
		enum { NumElements = 4 };
	private:
		redefined char* variableBlockAddress() { return data; }
		redefined long variableBlockLength();
		friend class SndOrIrcamSoundHeader;
		friend class IrcamHeader;
		friend class HybridHeader;
	private:
		int sf_magic;
		float sf_srate;
		int sf_chans;
		int sf_packmode;
		char* data;		// first 2 bytes will contain first sf_code
		int lengthTrim;
	};
private:
	const char *getSFCode(const char *, int);
	int putSFCode(const char *, const char *, const SFCode *);
};

class IrcamHeader : public SndOrIrcamSoundHeader {
	typedef SndOrIrcamSoundHeader Super;
public:
	IrcamHeader(DataType, int rate, int chans=1, double peak=0);
	IrcamHeader();
protected:
	redefined int diskHeaderSize() { return SIZEOF_BSD_HEADER; }
	redefined int diskHeaderCommentSize() { return diskHeaderSize() - 17; }
	redefined boolean isMagic();
	redefined boolean isValid(DataType t) {
		return (t == ShortData || t == FloatData);
	}
	redefined int writeInfo(DataFile *);
};

class SndHeader : public SndOrIrcamSoundHeader {
	typedef SndOrIrcamSoundHeader Super;
public:
	SndHeader(DataType, int rate, int chans=1);
	SndHeader();
protected:
	redefined int diskHeaderSize() { return 28; }
	redefined int diskHeaderCommentSize() { return 4; }
	redefined int diskHeaderInfoSize() { return 24; }
	redefined boolean isValid(DataType);
	redefined int writeInfo(DataFile *);
	redefined int writeComment(DataFile *);
	redefined int commentLength();
};

class HybridHeader : public SndHeader {
	typedef SndHeader Super;
public:
	HybridHeader(DataType, int rate, int chans=1, double peak=0);
	HybridHeader();
protected:
	redefined int diskHeaderSize() { return SIZEOF_BSD_HEADER; }
	redefined int diskHeaderCommentSize() { return diskHeaderSize() - 28 - 17; }
	redefined boolean isValid(DataType);
	redefined int writeInfo(DataFile *);
	redefined int writeComment(DataFile *);
};

#endif
