/* -*-Mode: C;-*-
 * XDELTA - RCS replacement and delta generator
 * Copyright (C) 1997  Josh MacDonald
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * $Id: grow.c 1.3.1.3 Sun, 12 Oct 1997 20:29:34 -0700 jmacd $
 */

#include "xdelta.h"

#include <ctype.h>

inline gint
ilog2 (gint arg)
{
  gint log = 0;

  while (arg >>= 1) log += 1;

  return log;
}

static gint
grow_1_int (const guint8 *from_seg,
	    const guint8 *to_seg,
	    gint          from_index,
	    gint          to_index,
	    gint          lower_from_bound,
	    gint          upper_from_bound,
	    gint          lower_to_bound,
	    gint          upper_to_bound,
	    gint          segment_len,
	    gint          from_segment_index,
	    Match        *match)
{
  gint max_start_offset;
  gint max_finish_offset;
  gint start_offset = 0;
  gint finish_offset = segment_len; /* if the cksum is not false, this
				     * much should match.  we test it
				     * first. */

#ifdef DEBUG_XD
  {
    Checksum ck1;
    Checksum ck2;

    init_checksum_1 (from_seg + from_index, segment_len, &ck1);
    init_checksum_1 (to_seg + to_index, segment_len, &ck2);

    if (ck1.low != ck2.low || ck1.high != ck1.high)
      {
	g_print ("error in grow! mismatched checksums\n");
	exit (1);
      }

  }
#endif

  if (memcmp (from_seg + from_index,
	      to_seg + to_index,
	      segment_len) != 0)
    return FALSE;

  max_start_offset  = MIN (from_index /* - 0 */, to_index - lower_to_bound);
  max_finish_offset = MIN (upper_to_bound - to_index, upper_from_bound - from_index);

  while (start_offset < max_start_offset &&
	 from_seg[from_index - start_offset - 1] ==
	 to_seg  [to_index   - start_offset - 1])
    start_offset += 1;

  while (finish_offset < max_finish_offset &&
	 from_seg[from_index + finish_offset] ==
	 to_seg  [to_index   + finish_offset])
    finish_offset += 1;

  match->from_real_offset = from_index - start_offset;
  match->from_segment_index = from_segment_index;
  match->to_low   = to_index   - start_offset;
  match->to_high  = to_index   + finish_offset;
  match->length   = match->to_high - match->to_low;
  match->low_neg_leftover = 0;
  match->high_pos_leftover = 0;

#ifdef DEBUG_XD
  if (strncmp (from_seg + match->from_real_offset, to_seg + match->to_low, match->length) != 0)
    {
      g_print ("grow fails!!!\n");
      exit (2);
    }
#endif

  return TRUE;
}

gint
grow_1 (MatchQuery *query,          /* the query */
	CKList     *ckl,            /* the cksum that matched */
	gint        segment_len,    /* length checksummed, ipow2(width) */
	gint        lower_to_limit, /* where to stop matching below */
	gint        upper_to_limit, /* where to stop matching above */
	gint        to_index,       /* where in the to segment cksum was found */
	Match      *match)          /* return match params */
{
  FromSegment* from = ckl->array->from;
  gint from_index = ((ckl->cksum - from->ckarray->cksums) * segment_len);

  return grow_1_int (from->real_seg,
		     query->real_to_seg,
		     from_index,
		     to_index,
		     0,
		     from->real_len,
		     lower_to_limit,
		     upper_to_limit,
		     segment_len,
		     from->segment_index,
		     match);
}

static gint
grow_2_int (const guint8 *from_seg,         /* the real from segment */
	    const guint8 *to_seg,           /* the real to segment */
	    gint          from_hindex,      /* all length and index arguments */
	    gint          to_hindex,        /* are in HASH indices/units. */
	    gint          lower_from_bound_hindex,
	    gint          upper_from_bound_hindex,
	    gint          lower_to_bound_hindex,
	    gint          upper_to_bound_hindex,
	    gint          segment_len,
	    gint          from_segment_index,
	    gint         *from_hindex_to_rindex_map,
	    gint         *to_hindex_to_rindex_map,
	    Match        *match)
{
  gint from_rindex         = from_hindex_to_rindex_map[from_hindex];
  gint to_rindex           = to_hindex_to_rindex_map[to_hindex];

  gint start_hoffset       = 0;
  gint finish_hoffset      = segment_len;

  gint start_roffset       = 0;
  gint finish_to_roffset   = to_hindex_to_rindex_map[to_hindex + segment_len] - to_rindex;
  gint finish_from_roffset = from_hindex_to_rindex_map[from_hindex + segment_len] - from_rindex;
  gint finish_roffset      = finish_to_roffset;

  gint max_start_hoffset;
  gint max_finish_hoffset;

  gint max_start_roffset;
  gint max_finish_roffset;

  if (finish_to_roffset != finish_from_roffset)
    /* segments differ in real length. */
    return FALSE;

  if (memcmp (from_seg + from_rindex,
	      to_seg + to_rindex,
	      finish_roffset) != 0)
    /* false alarm. */
    return FALSE;

  max_start_hoffset  = MIN (from_hindex /* - 0 */, to_hindex - lower_to_bound_hindex);
  max_finish_hoffset = MIN (upper_to_bound_hindex - to_hindex,
			    upper_from_bound_hindex - from_hindex);

  max_start_roffset  = MIN (from_rindex - from_hindex_to_rindex_map[from_hindex - max_start_hoffset],
			    to_rindex   - to_hindex_to_rindex_map[to_hindex - max_start_hoffset]);

  max_finish_roffset = MIN (from_hindex_to_rindex_map[from_hindex + max_finish_hoffset] - from_rindex,
			    to_hindex_to_rindex_map[to_hindex + max_finish_hoffset]     - to_rindex);

  while (start_roffset < max_start_roffset &&
	 from_seg[from_rindex - start_roffset - 1] ==
	 to_seg  [to_rindex   - start_roffset - 1])
    {
      if (to_hindex_to_rindex_map[to_hindex - start_hoffset] == to_rindex - start_roffset)
	start_hoffset += 1;

      start_roffset += 1;
    }

  while (finish_roffset < max_finish_roffset &&
	 from_seg[from_rindex + finish_roffset] ==
	 to_seg  [to_rindex   + finish_roffset])
    {
      if (to_hindex_to_rindex_map[to_hindex + finish_hoffset] == to_rindex + finish_roffset)
	finish_hoffset += 1;

      finish_roffset += 1;
    }

#ifdef DEBUG_XD
  if (memcmp (from_seg + (from_rindex - start_roffset),
	      to_seg +   (to_rindex - start_roffset),
	      finish_roffset + start_roffset) != 0)
    {
      g_print ("grow fails!!!\n");
      abort ();
    }
#endif

  match->from_real_offset = from_rindex - start_roffset;
  match->from_segment_index = from_segment_index;
  match->to_low   = to_hindex   - start_hoffset;
  match->to_high  = to_hindex   + finish_hoffset;
  match->length   = match->to_high - match->to_low;
  match->low_neg_leftover = (to_rindex - start_roffset) - to_hindex_to_rindex_map[match->to_low];
  match->high_pos_leftover = to_hindex_to_rindex_map[match->to_high] - (to_rindex + finish_roffset);

#ifdef DEBUG_XD
  g_print ("found match from=(%d-%d) to=(%d-%d)\n",
	   from_rindex - start_roffset,
	   from_rindex + finish_roffset,
	   to_rindex - start_roffset,
	   to_rindex + finish_roffset);
  assert (match->low_neg_leftover >= 0 && match->high_pos_leftover >= 0);

  assert (to_rindex   - start_roffset ==
	  to_hindex_to_rindex_map [match->to_low] + match->low_neg_leftover);
  assert (to_rindex + finish_roffset ==
	  to_hindex_to_rindex_map [match->to_high] - match->high_pos_leftover);

#endif

  return TRUE;
}

gint
grow_2 (MatchQuery *query,          /* the query */
	CKList     *ckl,            /* the cksum that matched */
	gint        segment_len,    /* length checksummed, ipow2(width) */
	gint        lower_to_limit, /* where to stop matching below */
	gint        upper_to_limit, /* where to stop matching above */
	gint        to_index,       /* where in the to segment cksum was found */
	Match      *match)          /* return match params */
{
  FromSegment* from = ckl->array->from;
  gint from_index = ((ckl->cksum - from->ckarray->cksums) * segment_len);

  return grow_2_int (from->real_seg,
		     query->real_to_seg,
		     from_index,
		     to_index,
		     0,
		     from->real_len,
		     lower_to_limit,
		     upper_to_limit,
		     segment_len,
		     from->segment_index,
		     from->multibyte_map,
		     query->multibyte_to_map,
		     match);
}
