/* appicon.c- icon for applications (not mini-window)
 *
 *  WindowMaker window manager
 * 
 *  Copyright (c) 1997 Alfredo K. Kojima
 * 
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "wconfig.h"

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <stdlib.h>
#include <string.h>

#include "generic/wwmlib.h"
#include "WindowMaker.h"
#include "wcore.h"
#include "window.h"
#include "icon.h"
#include "appicon.h"
#include "actions.h"
#include "stacking.h"
#include "dock.h"
#include "funcs.h"
#include "defaults.h"
#include "workspace.h"

#ifdef SUPERFLUOUS
#include "superfluous.h"
#endif

/* 
 * icon_file for the dock is got from the preferences file by
 * using the classname/instancename
 */

/**** Global variables ****/
extern Cursor wCursor[WCUR_LAST];
extern WPreferences wPreferences;

#define MOD_MASK wPreferences.modifier_mask 

void appIconMouseDown(WObjDescriptor *desc, XEvent *event);
static void iconDblClick(WObjDescriptor *desc, XEvent *event);
static void iconExpose(WObjDescriptor *desc, XEvent *event);


/*
 *---------------------------------------------------------------------- 
 * wAppIconCreateFromIcon--
 * 	Create an icon for use in the Dock, by "converting" a normal
 * icon to a dock icon. Tries to guess how the app is launched and
 * if it can't, fail.
 * 
 * Returns:
 * 	A WAppIcon structure or NULL, if the icon can't be docked.
 * 
 *---------------------------------------------------------------------- 
 */
WAppIcon*
wAppIconCreateFromIcon(WIcon *icon)
{
    WAppIcon *dicon;
    char **argv;
    int argc;

    dicon = wmalloc(sizeof(WAppIcon));
    memset(dicon, 0, sizeof(WAppIcon));
    dicon->yindex = -1;
    dicon->xindex = -1;
    
    dicon->icon = icon;

    /* check WM_COMMAND */
    if (!wApplicationOf(icon->owner->main_window)) {
	dicon->main_window = icon->owner->client_win;
    } else {
	dicon->main_window = icon->owner->main_window;
    }
    if (!XGetCommand(dpy, dicon->main_window, &argv, &argc) || !argv) {
	free(dicon);
	return NULL;
    }
    dicon->command = FlattenStringList(argv, argc);
    XFreeStringList(argv);
    dicon->wm_class = strdup(icon->owner->wm_class);
    dicon->wm_instance = strdup(icon->owner->wm_instance);
    
    return dicon;
}


WAppIcon*
wAppIconCreateForCommand(WScreen *scr, char *command, char *wm_class,
			 char *wm_instance)
{
    WAppIcon *dicon;

    dicon = wmalloc(sizeof(WAppIcon));
    memset(dicon, 0, sizeof(WAppIcon));
    dicon->yindex = -1;
    dicon->xindex = -1;

    if (command)
	dicon->command = strdup(command);

    if (wm_class)
      dicon->wm_class = strdup(wm_class);
    if (wm_instance)
      dicon->wm_instance = strdup(wm_instance);

    dicon->icon =
	wIconCreateWithImage(scr, wGetImageForWindowName(scr, wm_class,
							 wm_instance));

    /* will be overriden by dock */
    dicon->icon->core->descriptor.handle_mousedown = appIconMouseDown;
    dicon->icon->core->descriptor.handle_expose = iconExpose;
    dicon->icon->core->descriptor.parent_type = WCLASS_APPICON;
    dicon->icon->core->descriptor.parent = dicon;
    AddToStackList(dicon->icon->core);

    return dicon;
}



WAppIcon*
wAppIconCreate(WWindow *leader_win)
{
    WAppIcon *aicon;

    aicon = wmalloc(sizeof(WAppIcon));
    memset(aicon, 0, sizeof(WAppIcon));
    aicon->yindex = -1;
    aicon->xindex = -1;

    aicon->icon = wIconCreate(leader_win, 
			      wGetImageForWindowName(leader_win->screen_ptr,
						     leader_win->wm_class,
						     leader_win->wm_instance));
    /* will be overriden if docked */
    aicon->icon->core->descriptor.handle_mousedown = appIconMouseDown;
    aicon->icon->core->descriptor.handle_expose = iconExpose;
    aicon->icon->core->descriptor.parent_type = WCLASS_APPICON;
    aicon->icon->core->descriptor.parent = aicon;
    AddToStackList(aicon->icon->core);
    aicon->icon->show_title = 0;
    wIconUpdate(aicon->icon);

    return aicon;
}


void
wAppIconDestroy(WAppIcon *aicon)
{
    RemoveFromStackList(aicon->icon->core);
    wIconDestroy(aicon->icon);
    if (aicon->command)
	free(aicon->command);
#ifdef OFFIX_DND
    if (aicon->dnd_command)
	free(aicon->dnd_command);
#endif
    if (aicon->wm_instance)
      XFree(aicon->wm_instance);
    if (aicon->wm_class)
      XFree(aicon->wm_class);

    free(aicon);
}

#ifdef NEWAPPICON
static void
drawCorner(WIcon *icon, WWindow *wwin, int active)
{
    WScreen *scr = wwin->screen_ptr;
    XPoint points[3];
    GC gc;

    points[0].x = 2;
    points[0].y = 2;
    points[1].x = 12;
    points[1].y = 2;
    points[2].x = 2;
    points[2].y = 12;
    if (active) {
	gc=scr->focused_texture->any.gc;
    } else {
	gc=scr->unfocused_texture->any.gc;
    }
    XFillPolygon(dpy, icon->core->window, gc, points, 3,
		 Convex, CoordModeOrigin);
}
#endif /* NEWAPPICON */


#ifdef DOCK
#ifdef WS_INDICATOR
static void
updateDockNumbers(WScreen *scr)
{
   int length;
   char *ws_numbers;
   GC numbers_gc;
   XGCValues my_gc_values;
   unsigned long my_v_mask = (GCForeground);
   WAppIcon *dicon = scr->dock->icon_array[0];

   my_gc_values.foreground = scr->white_pixel;
   numbers_gc = XCreateGC(dpy, dicon->icon->core->window,
			  my_v_mask, &my_gc_values);
   
   ws_numbers = malloc(20);
   sprintf(ws_numbers, "%i [ %i ]", scr->current_workspace+1, 
	   ((scr->current_workspace/10)+1));
   length = strlen(ws_numbers);

   XClearArea(dpy, dicon->icon->core->window, 2, 2, 50, 
	      scr->icon_title_font->y+1, False);

#ifndef I18N_MB
   XSetFont(dpy, numbers_gc, scr->icon_title_font->font->fid);
   XSetForeground(dpy, numbers_gc, scr->black_pixel);
   XDrawString(dpy, dicon->icon->core->window,
               numbers_gc, 4, scr->icon_title_font->y+3, ws_numbers, length);
   XSetForeground(dpy, numbers_gc, scr->white_pixel);
   XDrawString(dpy, dicon->icon->core->window,
               numbers_gc, 3, scr->icon_title_font->y+2, ws_numbers, length);
#else
   XSetForeground(dpy, numbers_gc, scr->black_pixel);
   XmbDrawString(dpy, dicon->icon->core->window,
                 scr->icon_title_font->font, numbers_gc, 4,
                 scr->icon_title_font->y+3, ws_numbers, length);
   XSetForeground(dpy, numbers_gc, scr->white_pixel);
   XmbDrawString(dpy, dicon->icon->core->window,
                 scr->icon_title_font->font, numbers_gc, 3,
                 scr->icon_title_font->y+2, ws_numbers, length);
#endif /* I18N_MB */

   XFreeGC(dpy, numbers_gc);
   free(ws_numbers);
}
#endif /* WS_INDICATOR */
#endif /* DOCK */

void
wAppIconPaint(WAppIcon *aicon)
{
    WScreen *scr = aicon->icon->core->screen_ptr;

    wIconPaint(aicon->icon);

#ifdef DOCK
# ifdef WS_INDICATOR
    if (aicon->docked && scr->dock && aicon->yindex==0)
	updateDockNumbers(scr);
# endif
    if (scr->dock_dots && aicon->docked && !aicon->running
	&& aicon->command!=NULL) {
	XSetClipMask(dpy, scr->copy_gc, scr->dock_dots->mask);
	XSetClipOrigin(dpy, scr->copy_gc, 0, 0);
	XCopyArea(dpy, scr->dock_dots->image, aicon->icon->core->window, 
		  scr->copy_gc, 0, 0, scr->dock_dots->width,
		  scr->dock_dots->height, 0, 0);
    }
#endif /* DOCK */

#ifdef NEWAPPICON
    if (!wPreferences.strict_ns && aicon->icon->owner!=NULL) {
	int active=0;
	if (aicon->main_window==None) {
	    active = (aicon->icon->owner->flags.focused ? 1 : 0);
	} else {
	    WApplication *wapp;

	    wapp = wApplicationOf(aicon->main_window);
	    if (!wapp) {
		active = aicon->icon->owner->flags.focused;
		wWarning("error in wAppIconPaint(). Please report it");
	    } else {
		active = wapp->main_window_desc->flags.focused;
		if (wapp->main_window_desc->flags.hidden
		    || !wapp->main_window_desc->flags.mapped)
		  active = -1;
	    }
	}
	if (active>=0)
	  drawCorner(aicon->icon, aicon->icon->owner, active);
    }
#endif /* NEWAPPICON */
    XSetClipMask(dpy, scr->copy_gc, None);
    if (aicon->launching) {
	XFillRectangle(dpy, aicon->icon->core->window, scr->stipple_gc, 
		       0, 0, wPreferences.icon_size, wPreferences.icon_size);
    }
}



#define canBeDocked(wwin)  ((wwin) && ((wwin)->wm_class||(wwin)->wm_instance))


/******************************************************************/

static void 
iconExpose(WObjDescriptor *desc, XEvent *event)
{
    wAppIconPaint(desc->parent);
}

static void
iconDblClick(WObjDescriptor *desc, XEvent *event)
{
    WAppIcon *aicon = desc->parent;
    WApplication *wapp;

    assert(aicon->icon->owner!=NULL);
	
    wapp = wApplicationOf(aicon->icon->owner->main_window);
    /* aproveita que ta aqui pra atualizar */
    wapp->main_window = aicon->icon->owner->main_window;
#ifdef DEBUG0
    if (!wapp) {
	wWarning("could not find application descriptor for app icon!!");
	return;
    }
#endif
    wUnhideApplication(wapp, event->xbutton.button==Button2);
    if (event->xbutton.state & MOD_MASK) {
	wHideOtherApplications(aicon->icon->owner);
    }
}


void 
appIconMouseDown(WObjDescriptor *desc, XEvent *event)
{
    WAppIcon *aicon = desc->parent;
    WIcon *icon = aicon->icon;
    XEvent ev;
    int x=aicon->x_pos, y=aicon->y_pos;
    int dx=event->xbutton.x, dy=event->xbutton.y;
    int grabbed=0;
    int done=0;
#ifdef DOCK
    WScreen *scr = icon->core->screen_ptr;
    WWorkspace *workspace = scr->workspaces[scr->current_workspace];
    int shad_x = 0, shad_y = 0, docking=0, dockable, collapsed = 0;
    int ix, iy;
#ifdef SUPERFLUOUS
    Pixmap ghost = None;
#endif
#endif

    if ((desc->click_timestamp>0) 
	&& (event->xbutton.time - desc->click_timestamp <= DEF_DBLCLICK_TIME)) {
	desc->click_timestamp = -1;
	
	iconDblClick(desc, event);
	return;
    }
    desc->click_timestamp = event->xbutton.time;
    
#ifdef DEBUG
    puts("Moving icon");
#endif
    if (event->xbutton.state & MOD_MASK)
	wLowerFrame(icon->core);
    else
	wRaiseFrame(icon->core);

    
    /* TODO: remove this send event and make event be passed directly
     * to the client *//*
    if (icon->owner && icon->icon_win)
      XSendEvent(dpy, icon->icon_win, False, ButtonPressMask, event);
			*/
    if (XGrabPointer(dpy, icon->core->window, True, ButtonMotionMask
		     |ButtonReleaseMask, GrabModeAsync, GrabModeAsync, None,
		     None, CurrentTime) !=GrabSuccess) {
	wWarning("pointer grab failed for appicon move");
    }
#ifdef DOCK
    if (wPreferences.flags.nodock && wPreferences.flags.nofiend)
      dockable = 0;
    else
      dockable = canBeDocked(icon->owner);
#endif

    while (!done) {
	wMaskEvent(PointerMotionMask|ButtonReleaseMask|ButtonMotionMask
		   |ExposureMask, &ev);
	switch (ev.type) {
	 case Expose:
	    DispatchEvent(&ev);
	    break;

	 case MotionNotify:
	    if (!grabbed) {
		if (abs(dx-ev.xmotion.x)>=MOVE_THRESHOLD
		    || abs(dy-ev.xmotion.y)>=MOVE_THRESHOLD) {
		    XChangeActivePointerGrab(dpy, ButtonMotionMask
					     |ButtonReleaseMask, 
					     wCursor[WCUR_MOVE], CurrentTime);
		    grabbed=1;
		} else {
		    break;
		}
	    }
	    x = ev.xmotion.x_root - dx;
	    y = ev.xmotion.y_root - dy;
	    XMoveWindow(dpy, icon->core->window, x, y);
#ifdef DOCK
	    if (dockable) {
                if (scr->dock && wDockSnapIcon(scr->dock, aicon, x, y,
                                               &ix, &iy, False)) {
                    shad_x = scr->dock->x_pos + ix*wPreferences.icon_size;
		    shad_y = scr->dock->y_pos + iy*wPreferences.icon_size;

                    if (scr->last_dock != scr->dock && collapsed) {
                        scr->last_dock->collapsed = 1;
                        wDockHideIcons(scr->last_dock);
                        collapsed = 0;
                    }
                    if (!collapsed && (collapsed = scr->dock->collapsed)) {
                        scr->dock->collapsed = 0;
                        wDockShowIcons(scr->dock);
                    }

                    scr->last_dock = scr->dock;

		    XMoveWindow(dpy, scr->dock_shadow, shad_x, shad_y);
		    if (!docking) {
			Window wins[2];

			wins[0] = icon->core->window;
			wins[1] = scr->dock_shadow;
			XRestackWindows(dpy, wins, 2);
#ifdef SUPERFLUOUS
			if (icon->pixmap!=None)
			    ghost = MakeGhostIcon(scr, icon->pixmap);
			else
			    ghost = MakeGhostIcon(scr, icon->core->window);
			XSetWindowBackgroundPixmap(dpy, scr->dock_shadow,
						   ghost);
			XClearWindow(dpy, scr->dock_shadow);
#endif
                        XMapWindow(dpy, scr->dock_shadow);
		    }
		    docking = 1;
                } else if (workspace->fiend &&
                           wDockSnapIcon(workspace->fiend, aicon, x, y,
                                         &ix, &iy, False)) {
                    shad_x = workspace->fiend->x_pos + ix*wPreferences.icon_size;
		    shad_y = workspace->fiend->y_pos + iy*wPreferences.icon_size;
		    
                    if (scr->last_dock != workspace->fiend && collapsed) {
                        scr->last_dock->collapsed = 1;
                        wDockHideIcons(scr->last_dock);
                        collapsed = 0;
                    }
                    if (!collapsed && (collapsed = workspace->fiend->collapsed)) {
                        workspace->fiend->collapsed = 0;
                        wDockShowIcons(workspace->fiend);
                    }

                    scr->last_dock = workspace->fiend;

		    XMoveWindow(dpy, scr->dock_shadow, shad_x, shad_y);
		    if (!docking) {
			Window wins[2];

			wins[0] = icon->core->window;
			wins[1] = scr->dock_shadow;
			XRestackWindows(dpy, wins, 2);
#ifdef SUPERFLUOUS
			if (icon->pixmap!=None)
			    ghost = MakeGhostIcon(scr, icon->pixmap);
			else
			    ghost = MakeGhostIcon(scr, icon->core->window);
			XSetWindowBackgroundPixmap(dpy, scr->dock_shadow,
						   ghost);
			XClearWindow(dpy, scr->dock_shadow);
#endif
			XMapWindow(dpy, scr->dock_shadow);
		    }
                    docking = 1;
		} else if (docking) {
		    XUnmapWindow(dpy, scr->dock_shadow);
                    docking = 0;
		}
            }
#endif /* DOCK */
	    break;

	 case ButtonRelease:
	    XUngrabPointer(dpy, CurrentTime);
#ifdef DOCK
            if (docking) {
		/* icon was docked */
		SlideWindow(icon->core->window, x, y, shad_x, shad_y);
		XUnmapWindow(dpy, scr->dock_shadow);
                wDockAttachIcon(scr->last_dock, aicon, ix, iy);
            } else {
#endif
		XMoveWindow(dpy, icon->core->window, x, y);
		aicon->x_pos = x;
		aicon->y_pos = y;
#ifdef DOCK
            }
            if (collapsed) {
                scr->last_dock->collapsed = 1;
                wDockHideIcons(scr->last_dock);
                collapsed = 0;
            }
#ifdef SUPERFLUOUS
	    if (ghost!=None)
	      XFreePixmap(dpy, ghost);
	    XSetWindowBackground(dpy, scr->dock_shadow, scr->white_pixel);
#endif
#endif  /* DOCK */
            done = 1;
	    break;
	}
    }
#ifdef DEBUG
    puts("End icon move");
#endif
    
}
