/* --------------------------------------------------------------------*/
/*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
/*                                                                     */
/*                                     ,--^,                           */
/*                               _ ___/ /|/                            */
/*                           ,;'( )__, ) '                             */
/*                          ;;  //   L__.                              */
/*                          '   \   /  '                               */
/*                               ^   ^                                 */
/*                                                                     */
/*                                                                     */
/*    This program is distributed in the hope that it will be useful.  */
/*    Use and copying of this software and preparation of derivative   */
/*    works based upon this software are permitted, so long as the     */
/*    following conditions are met:                                    */
/*           o credit to the authors is acknowledged following         */
/*             current academic behaviour                              */
/*           o no fees or compensation are charged for use, copies,    */
/*             or access to this software                              */
/*           o this copyright notice is included intact.               */
/*      This software is made available AS IS, and no warranty is made */
/*      about the software or its performance.                         */
/*                                                                     */
/*      Bug descriptions, use reports, comments or suggestions are     */
/*      welcome. Send them to                                          */
/*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
/*-------------------------------------------------------------------- */
/*=====================================================================*/
/*    serrano/prgm/project/bigloo/runtime/Clib/cports.c                */
/*    -------------------------------------------------------------    */
/*    Author      :  Manuel Serrano                                    */
/*    Creation    :  Thu Jul 23 15:34:53 1992                          */
/*    Last change :  Fri Mar 20 09:12:54 1998 (serrano)                */
/*    -------------------------------------------------------------    */
/*    Input ports handling                                             */
/*=====================================================================*/
#include <stdio.h>
#include <errno.h>
#include <string.h>
#if( !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <termio.h>
#endif
#if( !defined( sony_news ) && \
     !(defined( NeXT ) && (defined( mc68000 ) || defined( i386 ))) )
#   include <unistd.h>
#endif
#include <sys/file.h>
/*---------------------------------------------------------------------*/
/*    The file `gc_private.h' must not be included into this file. To  */
/*    prevent this inclusion, we define the GC_PRIVATE_H macro. We     */
/*    have to protect this file, because `gc_private.h' use an         */
/*    incorrect prototype for `sbrk' that is incompatible we some      */
/*    operating system (such as Linux). Linux uses a prototype in      */
/*    the file `unisys.h' which is not compatible with the one used    */
/*    in file `gc_private.h'.                                          */
/*---------------------------------------------------------------------*/
#if( defined( i386 ) )
#   define GC_PRIVATE_H
#endif
#include <bigloo1.9c.h>
#if( defined( sony_news ) || (defined( NeXT ) && defined( mc68000 )) )
#   include <ctype.h>
#endif
 
/*---------------------------------------------------------------------*/
/*    isascii                                                          */
/*---------------------------------------------------------------------*/
#if( !defined( isascii ) )
#   define isascii( c ) (!((c) & ~0177))
#endif

/*---------------------------------------------------------------------*/
/*    Global variables                                                 */
/*---------------------------------------------------------------------*/
obj_t current_output_port, current_input_port, current_error_port;
long default_io_bufsiz;

/*---------------------------------------------------------------------*/
/*    Is Bigloo case sensitive ?                                       */
/*---------------------------------------------------------------------*/
extern obj_t bigloo_case_sensitive;

/*---------------------------------------------------------------------*/
/*    External definitions.                                            */
/*---------------------------------------------------------------------*/
extern obj_t  string_to_bstring();
extern obj_t  string_to_symbol();
extern obj_t  string_to_keyword();
extern obj_t  make_real();
extern void   c_error();

/*---------------------------------------------------------------------*/
/*    Prototypes                                                       */
/*---------------------------------------------------------------------*/
static bool_t pipe_name_p( char * );
static char  *pipe_name( char * );

/*---------------------------------------------------------------------*/
/*     make_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
make_output_port( char *name, FILE *file, obj_t kindof )
{
   obj_t new_output_port;

   new_output_port = GC_MALLOC( OUTPUT_PORT_SIZE );
   
   new_output_port->output_port_t.header = MAKE_HEADER( OUTPUT_PORT_TYPE, 0 );
   new_output_port->output_port_t.file   = file;
   new_output_port->output_port_t.name   = name;
   new_output_port->output_port_t.kindof = kindof;
   
   return BREF( new_output_port );
}

/*---------------------------------------------------------------------*/
/*    open_output_file ...                                             */
/*---------------------------------------------------------------------*/
obj_t
open_output_file( obj_t name )
{
   FILE *file;

#if( HAVE_PIPE )
   if( pipe_name_p( BSTRING_TO_STRING( name ) ) )
   {
      if( !(file = popen( pipe_name( BSTRING_TO_STRING( name ) ), "w" )) )
	 return BFALSE;
      
      return make_output_port( BSTRING_TO_STRING( name ), file, KINDOF_PIPE );
   }
   else
#endif
   {
      if( !(file = fopen( BSTRING_TO_STRING( name ), "w" )) )
	 return BFALSE;
      
      return make_output_port( BSTRING_TO_STRING( name ), file, KINDOF_FILE );
   }
}

/*---------------------------------------------------------------------*/
/*    append_output_file ...                                           */
/*---------------------------------------------------------------------*/
obj_t
append_output_file( obj_t name )
{
   FILE *file;
   
   if( !(file = fopen( BSTRING_TO_STRING( name ), "a+" )) )
      return BFALSE;

   return make_output_port( BSTRING_TO_STRING( name ), file, KINDOF_FILE );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    open_output_string ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_output_string()
{
   char *buffer;
   obj_t port;

   port = GC_MALLOC( OUTPUT_STRING_PORT_SIZE );

   /* We allocate a buffer filled of zero */
   buffer = (char *)(GC_MALLOC( OUTPUT_STRING_PORT_BUFFER_SIZE + 1 ));

   port->output_string_port_t.header = MAKE_HEADER( OUTPUT_STRING_PORT_TYPE,0);
   port->output_string_port_t.buffer = buffer;
   port->output_string_port_t.size   = OUTPUT_STRING_PORT_BUFFER_SIZE;
   port->output_string_port_t.offset = 0;
   
   return BREF( port );
}
 
/*---------------------------------------------------------------------*/
/*    close_output_port ...                                            */
/*---------------------------------------------------------------------*/
obj_t
close_output_port( obj_t port )
{
   if( OUTPUT_STRING_PORTP( port ) )
   {
      obj_t res;
	   
      res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );
      OUTPUT_PORT( port ).file = 0L;

      return res;
   }
   else
   {
      /* We do not close console ports (e.g. stdout, stderr) */
      switch( (long)(OUTPUT_PORT( port ).kindof) )
      {
	 case KINDOF_FILE:
	    fclose( OUTPUT_PORT( port ).file );
	    break;
	    
#if( HAVE_PIPE )
	 case KINDOF_PIPE:
	    pclose( OUTPUT_PORT( port ).file );
	    break;
#endif
      }
      
      return port;
   }
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    make_input_port ...                                              */
/*---------------------------------------------------------------------*/
obj_t
make_input_port( obj_t name, FILE *file, obj_t kindof, obj_t bbufsiz )
{
   obj_t new_input_port;
   long  buflen;
   long  bufsiz = CINT( bbufsiz );

   /* We allocate the Bigloo input port */
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.file     = file;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = bufsiz + 1;
   new_input_port->input_port_t.backward = 0L;
   new_input_port->input_port_t.forward  = 0L;
   new_input_port->input_port_t.remember = 0L;
   new_input_port->input_port_t.name     = name;
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;
   new_input_port->input_port_t.kindof   = kindof;

   /* We fill the buffer up */
   buflen = fread( BUFFER( BREF( new_input_port ) ),
                   1,
                   bufsiz,
                   file );

   /* We check for errors */
   if( ferror( file ) )
      C_FAILURE( "open_input_file",
		 "Error while reading on file",
		 BINT( ferror( file ) ) );

   /* We check for end of file */
   if( feof( file ) )
      new_input_port->input_port_t.eof = 1;

   /* We write buffer sentinel */
   BUFFER( BREF( new_input_port ) )[ buflen ] = '\0';

   return BREF( new_input_port );
}
   
/*---------------------------------------------------------------------*/
/*    open_input_file ...                                              */
/*    -------------------------------------------------------------    */
/*    !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!! WARNING !!!  */
/*    We fill up its associated buffer when opening an input port.     */
/*    -------------------------------------------------------------    */
/*    This function open two kind of files. Regular file and Unix      */
/*    like pipes when the file name is something like "| ...".         */
/*---------------------------------------------------------------------*/
obj_t
open_input_file( obj_t name, obj_t bbufsiz )
{
   FILE *file;
   
#if( HAVE_PIPE )
   if( pipe_name_p( BSTRING_TO_STRING( name ) ) )
   {
      if( !(file = popen( pipe_name( BSTRING_TO_STRING( name ) ), "r" )) )
	 return BFALSE;

      return make_input_port( name, file, KINDOF_PIPE, bbufsiz );
   }
   else
#endif
   {
      if( !(file = fopen( BSTRING_TO_STRING( name ), "r" )) )
	 return BFALSE;
      
      return make_input_port( name, file, KINDOF_FILE, bbufsiz );
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_console ...                                           */
/*---------------------------------------------------------------------*/
obj_t
open_input_console()
{
   obj_t new_input_port;
   long  bufsiz = default_io_bufsiz;
   
   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.kindof   = KINDOF_CONSOLE;
   new_input_port->input_port_t.file     = stdin;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = default_io_bufsiz + 1;
   new_input_port->input_port_t.backward = default_io_bufsiz;
   new_input_port->input_port_t.forward  = default_io_bufsiz;
   new_input_port->input_port_t.remember = default_io_bufsiz;
   new_input_port->input_port_t.name     = string_to_bstring( "[stdin]" );
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;

   /* We simply fill the buffer with zero */
   memset( (char *)BUFFER( BREF( new_input_port ) ), 0, bufsiz );

   /* We flush the current input port */
   fflush( stdin );
   
   return BREF( new_input_port );
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    file_to_input_port ...                                           */
/*---------------------------------------------------------------------*/
obj_t
file_to_input_port( FILE *file )
{
   if( file == stdin )
      return open_input_console();
   else
   {
      obj_t new_input_port;
      long  bufsiz = default_io_bufsiz;
   
      new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

      new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0);
      new_input_port->input_port_t.kindof   = KINDOF_FILE;
      new_input_port->input_port_t.file     = file;
      new_input_port->input_port_t.filepos  = 0L;
      new_input_port->input_port_t.eof      = 0;
      new_input_port->input_port_t.bufsiz   = default_io_bufsiz + 1;
      new_input_port->input_port_t.backward = default_io_bufsiz;
      new_input_port->input_port_t.forward  = default_io_bufsiz;
      new_input_port->input_port_t.remember = default_io_bufsiz;
      new_input_port->input_port_t.name     = string_to_bstring( "[file]" );
      new_input_port->input_port_t.annexe   = 0L;
      new_input_port->input_port_t.anxsiz   = 0L;
   
      /* We simply fill the buffer with zero */
      memset( (char *)BUFFER( BREF( new_input_port ) ), 0, bufsiz );

      /* We flush the current input port */
      fflush( file );
      
      return BREF( new_input_port );
   }
}

/*---------------------------------------------------------------------*/
/*    open_input_string ...                                            */
/*---------------------------------------------------------------------*/
obj_t
open_input_string( obj_t string )
{
   obj_t new_input_port;
   long  bufsiz = STRING_LENGTH( string );

   new_input_port = GC_MALLOC( INPUT_PORT_SIZE + bufsiz + 1 );

   new_input_port->input_port_t.header   = MAKE_HEADER( INPUT_PORT_TYPE, 0 );
   new_input_port->input_port_t.kindof   = KINDOF_STRING;
   new_input_port->input_port_t.file     = 0L;
   new_input_port->input_port_t.filepos  = 0L;
   new_input_port->input_port_t.eof      = 0;
   new_input_port->input_port_t.bufsiz   = bufsiz + 1;
   new_input_port->input_port_t.backward = 0L;
   new_input_port->input_port_t.forward  = 0L;
   new_input_port->input_port_t.remember = 0L;
   new_input_port->input_port_t.name     = string;
   new_input_port->input_port_t.annexe   = 0L;
   new_input_port->input_port_t.anxsiz   = 0L;
	
   /* We simply fill the buffer with the string */
   strcpy( (char *)BUFFER( BREF( new_input_port ) ),
	   BSTRING_TO_STRING( string ) );
	
   /* We _have_ seen the end of file */
   new_input_port->input_port_t.eof = 1;

   return BREF( new_input_port );   
}

/*---------------------------------------------------------------------*/
/*    input_port_debug ...                                             */
/*---------------------------------------------------------------------*/
obj_t
input_port_debug( s, f )
obj_t s , f;
{
   FILE *file = OUTPUT_PORT( f ).file;
   
   fprintf( file, "====== input_port_debug ===============================\n" );
   fprintf( file, "%s:\n\n", INPUT_PORT( s ).name );
   fprintf( file, "   backward   : %d '%c'\n",
            INPUT_PORT( s ).backward,
            BUFFER( s )[ INPUT_PORT( s ).backward ]);
   fprintf( file, "   forward    : %d '%c'\n",
            INPUT_PORT( s ).forward,
            BUFFER( s )[ INPUT_PORT( s ).forward ]);
   fprintf( file, "   remember   : %d '%c'\n",
            INPUT_PORT( s ).remember,
            BUFFER( s )[ INPUT_PORT( s ).remember ]);
   fprintf( file, "   mark       : %d '%c'\n",
            INPUT_PORT( s ).mark,
            BUFFER( s )[ INPUT_PORT( s ).mark ]);
   fprintf( file, "   eof        : %d\n", INPUT_PORT( s ).eof );
   fprintf( file, "   buffer     : [%s]\n", BUFFER( s ) );
   fprintf( file, "===================================================\n" );

   return s;
}

/*---------------------------------------------------------------------*/
/*    close_input_port ...                                             */
/*---------------------------------------------------------------------*/
obj_t
close_input_port( obj_t port )
{
   if( (INPUT_PORTP( port )) )
   {
      
      /* We do not close the console port */
      switch( (long)(INPUT_PORT( port ).kindof) )
      {
	 case KINDOF_FILE:
	    INPUT_PORT( port ).eof = 1;
	    INPUT_PORT_RESET_ANNEXE( port );
	    memset( (char *)BUFFER( BREF( port ) ),
		    0,
		    INPUT_PORT( port ).bufsiz );
	    
	    fclose( INPUT_PORT( port ).file );
	    break;
	    
#if( HAVE_PIPE ) 
	 case KINDOF_PIPE:
	    INPUT_PORT( port ).eof = 1;
	    INPUT_PORT_RESET_ANNEXE( port );
	    BUFFER( port )[ 0 ] = '\0';
	    memset( (char *)BUFFER( BREF( port ) ),
		    0,
		    INPUT_PORT( port ).bufsiz );
	    
	    pclose( INPUT_PORT( port ).file );
	    break;
#endif
      }
   } 

   return port;
}

/*---------------------------------------------------------------------*/
/*    void                                                             */
/*    buffer_to_annexe ...                                             */
/*    -------------------------------------------------------------    */
/*    On deplace le buffer dans l'annexe. Deux cas peuvent se          */
/*    presenter:                                                       */
/*       - l'annexe n'existe pas, on la creer de la taille du          */
/*         buffer                                                      */
/*       - l'annexe existe, on la fait grandir.                        */
/*---------------------------------------------------------------------*/
void
buffer_to_annexe( obj_t port )
{
   if( !INPUT_PORT( port ).annexe )
   /* on est dans le premier cas, l'annexe n'existe pas */
   {
      unsigned char *annexe;
      long  size = INPUT_PORT( port ).bufsiz - 1;
      
      annexe = (unsigned char *)GC_MALLOC_ATOMIC( size );
      
      memcpy( annexe, BUFFER( port ), size );
      
      INPUT_PORT( port ).anxsiz = size;
      INPUT_PORT( port ).annexe = annexe;
   }
   else
   /* on est dans le deuxieme cas, l'annexe existe deja */
   {
      unsigned char *annexe;
      long  size = INPUT_PORT( port ).bufsiz + INPUT_PORT( port ).anxsiz - 1 ;
      
      annexe = (unsigned char *)GC_MALLOC_ATOMIC( size );
      
      memcpy( annexe,
	      INPUT_PORT( port ).annexe,
	      INPUT_PORT( port ).anxsiz );
      memcpy( annexe + INPUT_PORT( port ).anxsiz,
	      BUFFER( port ),
	      INPUT_PORT( port ).bufsiz );
      
      INPUT_PORT( port ).anxsiz = size;
      INPUT_PORT( port ).annexe = annexe;
   }
}

/*---------------------------------------------------------------------*/
/*    input_port_fill_buffer ...                                       */
/*    -------------------------------------------------------------    */
/*    Si on a lu un end-of-file, on ne peut rien lire de plus.         */
/*    Idem si le buffer est deja plein : runner == 0                   */
/*    -------------------------------------------------------------    */
/*    S'il est encore possible de lire des choses on commence par      */
/*    reajuster le buffer. C'est a dire le recaler a gauche.           */
/*    -------------------------------------------------------------    */
/*    Quand on tompe sur un end-of-file, on n'a pas besoin de          */
/*    mettre un '\0' dans le buffer car C s'en charge.                 */
/*    -------------------------------------------------------------    */
/*    Voici l'etat du buffer avant de lire quoi que ce soit:           */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       |                   | |                | |            |0|     */
/*       +-------------------+-+----------------+-+------------+-+     */
/*       0               backward              forward      bufsiz     */
/*                                                                     */
/*    Apres le rewind l'etat devient:                                  */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |           |0|?|?|?|?|?|?|?|?|?|?|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*                                                                     */
/*    Et pour finir on a:                                              */
/*       +-+-----------------+-+---------------------------------+     */
/*       | |                 | |                               |0|     */
/*       +-+-----------------+-+---------------------------------+     */
/*     0 = backward        forward                          bufsiz     */
/*---------------------------------------------------------------------*/
bool_t
input_port_fill_buffer( obj_t port )
{
   long backward = INPUT_PORT( port ).backward;
   /* Debugging tools */
#ifdef DEBUG_PORT 
   puts( "Avant le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif

   /* the buffer is full, we have to use an annexe */
   if( !INPUT_PORT( port ).eof && (!backward || INPUT_PORT( port ).annexe) )
   {
      if( (INPUT_PORT( port ).kindof != KINDOF_CONSOLE) )
      {
#ifdef DEBUG_PORT
	 puts( "Le buffer est plein" );
#endif 		
	 /* We copy the buffer into the annexe */
	 buffer_to_annexe( port );

	 /* We setup buffer pointers */
	 backward = INPUT_PORT( port ).bufsiz - 1;
	 INPUT_PORT( port ).backward = backward;
      }
      else
      {
	 if( INPUT_PORT( port ).forward == INPUT_PORT( port ).bufsiz )
	 {
	    the_failure( string_to_bstring( "input_port_fill_buffer" ),
		         string_to_bstring( "token too large " ),
		         string_to_bstring( "on console" ) );
	    exit( -1 );
	 }
      }
   }

   /* There is two ways of readings. Reading on a regular file */
   /* and reading on a console. We select the proper one here. */
   if( (INPUT_PORT( port ).kindof == KINDOF_FILE) ||
       (INPUT_PORT( port ).kindof == KINDOF_PIPE) )
   {
      /* We have already found eof. We stop here */
      if( INPUT_PORT( port ).eof )
         return 0;
      else
      {  
         unsigned char *buffer  = BUFFER( port );
         long           bufsiz  = INPUT_PORT( port ).bufsiz;
         long           buflen;

	 /* We shift left the buffer */
	 strcpy( (char *)buffer, (char *)(buffer + backward) );

	 /* We set the current buffer length */
         buflen = bufsiz - backward - 1;

	 /* We fill the buffer up */
         buflen += fread( buffer + buflen, 1, bufsiz - buflen - 1,
                          INPUT_PORT( port ).file );

	 /* We check for errors */
	 if( ferror( INPUT_PORT( port ).file ) )
	    C_FAILURE( "input_port_fill_buffer",
		       "Error while reading on file",
		       BINT( ferror( INPUT_PORT( port ).file ) ) );
	 
	 /* We check for end of file */
	 if( feof( INPUT_PORT( port ).file ) )
            INPUT_PORT( port ).eof = 1;

	 /* We set the buffer sentinel */
         buffer[ buflen ] = '\0';
      }
   }
   else
      if( INPUT_PORT( port ).kindof == KINDOF_CONSOLE )
      {
         unsigned char *buffer  = BUFFER( port );
         long           bufsiz  = INPUT_PORT( port ).bufsiz;
	 long           forward = INPUT_PORT( port ).forward;
         long           buflen;

         if( INPUT_PORT( port ).eof )
	    return 0;
         else
         {
	    /* We shift left the buffer */
            strcpy( (char *)buffer, (char *)(buffer + backward) );
            
            buflen = forward - backward - 1;
            
            fgets( (char *)(buffer + buflen),
                   bufsiz - buflen - 1,
                   INPUT_PORT( port ).file );

            if( feof( INPUT_PORT( port ).file ) )
	    {
	       INPUT_PORT( port ).eof = 1;

               if( buffer[ buflen ] == '\0' )
	       {
		  /* we reset the buffer for next reading */
		  memset( &buffer[ buflen ], 0, forward - buflen - 1 );
		  
		  INPUT_PORT( port ).forward  = INPUT_PORT( port ).forward -
		     INPUT_PORT( port ).backward - 1;
		  INPUT_PORT( port ).remember = INPUT_PORT( port ).remember -
		     INPUT_PORT( port ).backward;

		  INPUT_PORT( port ).backward = 0L;

		  return 0;
	       }
	    }
         }
      }
      else
         return 0;

   /* We are done not. We just have to setup the port pointers */
   INPUT_PORT( port ).forward  = INPUT_PORT( port ).forward -
                                 INPUT_PORT( port ).backward - 1;
   INPUT_PORT( port ).remember = INPUT_PORT( port ).remember -
                                 INPUT_PORT( port ).backward;
      
   INPUT_PORT( port ).backward = 0L;
      
   /* More debug */
#ifdef DEBUG_PORT
   puts( "Apres le fill_buffer: " );
   input_port_debug( port, current_output_port );
#endif      

   return 1;
}

/*---------------------------------------------------------------------*/
/*    obj                                                              */
/*    reset_console ...                                                */
/*    -------------------------------------------------------------    */
/*    On purge les ports d'entree pour que le ^C marche bien sous      */
/*    l'interprete. Le seul entroit ou cette fonction est utilisee     */
/*    la fonction `repl' (voir Eval/eval.scm).                         */
/*---------------------------------------------------------------------*/
obj_t
reset_console( obj_t port )
{
   long size = INPUT_PORT( port ).bufsiz - 1;

   if( INPUT_PORT( port ).kindof == KINDOF_CONSOLE )
   {
      INPUT_PORT( port ).backward = size;
      INPUT_PORT( port ).forward  = size;
      INPUT_PORT( port ).remember = size;
      memset( (char *)BUFFER( port ) , 0, size );
      fflush( stdin );
   }

   return BUNSPEC;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_string ...                                        */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' et fini en `backward'.   */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_string( obj_t s )
{
   char  bck;
   obj_t res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

/* printf( "get_small_string:  mark: %d\n", INPUT_PORT( s ).mark );    */
/* printf( "                 annexe: %s\n", INPUT_PORT( s ).annexe );  */
/* printf( "                 buffer: %s\n", BUFFER( s ) );  */
	
   if( INPUT_PORT( s ).annexe )
   {
      char *aux;
      long  size   = INPUT_PORT_GET_LENGTH( s );
      long  anxsiz = INPUT_PORT( s ).anxsiz;

      aux = malloc( size );

      memcpy( aux, INPUT_PORT( s ).annexe, anxsiz );
      memcpy( &(aux[ anxsiz ]), BUFFER( s ), INPUT_PORT( s ).backward );
				       
      res = string_to_bstring( aux );

      free( aux );
      
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
   else
      res = string_to_bstring( &BUFFER( s )[ INPUT_PORT( s ).mark ] );
	
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_small_string ...                                  */
/*    -------------------------------------------------------------    */
/*    Cette fonction n'existe que pour le lecteur `Bigloo'. Quand      */
/*    le lecteur lit une chaine "...", il veut pouvoir acces tres      */
/*    facilement au coprs de la chaine, sans les guillemets. Cette     */
/*    fonction se charge de ce travail.                                */
/*    -------------------------------------------------------------    */
/*    Cette fonction construit une chaine `Bigloo' qui est extraite    */
/*    du buffer. La chaine commence en `mark' + 1 et fini en           */
/*    `backward' - 1.                                                  */
/*    Pour pouvoir faire un strcpy, on efface momentanement le char    */
/*    suivant `backward', on est donc obliger de le backuper.          */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_small_string( obj_t s )
{
   char  bck;
   obj_t res;
   
   bck = BUFFER( s )[ INPUT_PORT( s ).backward - 1 ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = '\0';

/* printf( "get_small_string:  mark: %d\n", ( INPUT_PORT( s ).mark ) );*/
/* printf( "                 annexe: %s\n", INPUT_PORT( s ).annexe );  */
/* printf( "                 buffer: %s\n", BUFFER( s ) );  */
	
   if( INPUT_PORT( s ).annexe )
   {
      char *aux;

      aux = malloc( INPUT_PORT_GET_LENGTH( s ) );
      
      memcpy( aux,
	      &INPUT_PORT( s ).annexe[ 1 ],
	      INPUT_PORT( s ).anxsiz - 1 );
      memcpy( &(aux[ INPUT_PORT( s ).anxsiz - 1 ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward - 1 );
				       
      res = string_to_bstring( aux );

      free( aux );
      
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
   else
      res = string_to_bstring(&BUFFER( s )[ INPUT_PORT( s ).mark + 1 ]);
	
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_symbol ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_symbol( obj_t s )
{
   char           bck;
   obj_t          res;
   unsigned char *buf;
	
   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      buf = malloc( INPUT_PORT_GET_LENGTH( s ) );
      
      memcpy( buf, INPUT_PORT( s ).annexe, INPUT_PORT( s ).anxsiz );
      memcpy( &(buf[ INPUT_PORT( s ).anxsiz ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward );
   }
   else
      buf = &BUFFER( s )[ INPUT_PORT( s ).mark ];
		
   if( (BUNSPEC == bigloo_case_sensitive) || !CBOOL( bigloo_case_sensitive ) )
   {
      unsigned char *walk;

      for( walk = buf; *walk; walk++ )
         if( isascii( *walk ) )
            *walk = toupper( *walk );
   }
   
   res = string_to_symbol( buf );
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   if( INPUT_PORT( s ).annexe )
   {
      free( buf );
		
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
		
   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_keyword ...                                       */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_keyword( obj_t s )
{
   char           bck;
   obj_t          res;
   unsigned char *buf;
	
   bck = BUFFER( s )[ INPUT_PORT( s ).backward - 1 ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      buf = malloc( INPUT_PORT_GET_LENGTH( s ) );
      
      memcpy( buf, INPUT_PORT( s ).annexe, INPUT_PORT( s ).anxsiz );
      memcpy( &(buf[ INPUT_PORT( s ).anxsiz ]),
	      BUFFER( s ),
	      INPUT_PORT( s ).backward );
   }
   else
      buf = &BUFFER( s )[ INPUT_PORT( s ).mark ];
		
   if( (BUNSPEC == bigloo_case_sensitive) || bigloo_case_sensitive )
   {
      unsigned char *walk;

      for( walk = buf; *walk; walk++ )
         if( isascii( *walk ) )
            *walk = toupper( *walk );
   }
   
   res = string_to_keyword( buf );
   BUFFER( s )[ INPUT_PORT( s ).backward - 1 ] = bck;

   if( INPUT_PORT( s ).annexe )
   {
      free( buf );
		
      INPUT_PORT( s ).anxsiz = 0L;
      INPUT_PORT( s ).annexe = 0L;
   }
		
   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_get_fixnum ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_fixnum( obj_t s )
{
   char bck;
   long res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      FAILURE( string_to_bstring( "get-flonum" ),
	       string_to_bstring( "Token too large" ),
	       BUNSPEC );
   }
   else
      res = atol( (const char *)(&BUFFER( s )[ INPUT_PORT( s ).mark ]) );
	
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return BINT( res );
}

/*---------------------------------------------------------------------*/
/*    input_port_get_flonum ...                                        */
/*---------------------------------------------------------------------*/
obj_t
input_port_get_flonum( obj_t s )
{
   char  bck;
   obj_t res;

   bck = BUFFER( s )[ INPUT_PORT( s ).backward ];
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = '\0';

   if( INPUT_PORT( s ).annexe )
   {
      FAILURE( string_to_bstring( "get-flonum" ),
	       string_to_bstring( "Token too large" ),
	       BUNSPEC );
   }
   else
      res = make_real( strtod( (const char *)(&BUFFER( s )[ INPUT_PORT( s ).mark ]), 0 ) );
   
   BUFFER( s )[ INPUT_PORT( s ).backward ] = bck;

   return res;
}

/*---------------------------------------------------------------------*/
/*    intput_port_read_string ...                                      */
/*    -------------------------------------------------------------    */
/*    Cette fonction retourne une chaine fraichement allouee           */
/*    -------------------------------------------------------------    */
/*    Cette fonction fonctionne aussi bien pour les `consoles' que pour*/
/*    les fichiers.                                                    */
/*---------------------------------------------------------------------*/
obj_t
intput_port_read_string( obj_t port, long number )
{
   obj_t          res;
   long           len;
   char           aux;
   unsigned char *buffer = BUFFER( port );

   /* On verifie que la taille de ce qu'on veut lire n'est pas plus grande */
   /* que la taille des buffers du port.                                   */
   len = INPUT_PORT( port ).bufsiz > number ?
             number : INPUT_PORT( port ).bufsiz;

   /* On regarde si on n'a pas deja lu eof sur le port */
   if( INPUT_PORT( port ).eof )
   {
      long buf_len;
      long offset;

      /* Puisqu'on a lu eof, il faut regarder quelle est la taille de la */
      /* chaine contenue dans le buffer.                                 */
      buf_len = (long)strlen( (const char *)(&BUFFER( port )[ INPUT_PORT( port ).backward ]) );

      len = buf_len > len ? len : buf_len;

      /* On peut maintenant construire la nouvelle chaine. */
      offset = INPUT_PORT( port ).backward + len;
      
      aux = buffer[ offset ]; 
      buffer[ offset ] = '\0';
      res = string_to_bstring( &BUFFER( port )[ INPUT_PORT( port ).backward ] );
      buffer[ offset ] = aux;
      
      /* on ajuste les curseurs */
      INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len;
   }
   else
      /* on regarde si on a assez de chose dans le buffer */
      if( (INPUT_PORT( port ).bufsiz - INPUT_PORT( port ).backward) < number )
      {
         long offset;

         /* on calcule la chaine resultat */
         offset = INPUT_PORT( port ).backward + number;
         
         aux = buffer[ offset ];
         buffer[ offset ] = '\0';
         res = string_to_bstring( &BUFFER( port )[ INPUT_PORT( port ).backward ] );
         buffer[ offset ] = aux;
      
         /* on ajuste les curseurs */
         INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + number;
      }
      else
      {
         long len_aux;
         
         /* on commence par remplir le buffer */
         INPUT_PORT( port ).forward = INPUT_PORT( port ).forward + 1;
         input_port_fill_buffer( port );

         /* On regarde la taille de ce qu'on vient de lire */
         if( (len_aux = strlen( (const char *)(buffer) )) > len )
         {
            /* on construit la nouvelle chaine */
            aux = buffer[ len ];
            buffer[ len ] = '\0';
            res = string_to_bstring( BUFFER( port ) );
            buffer[ len ] = aux;
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len;
         }
         else
         {
            res = string_to_bstring( BUFFER( port ) );
      
            /* on ajuste les curseurs */
            INPUT_PORT( port ).forward = INPUT_PORT( port ).backward + len_aux;
         }   
      }

   INPUT_PORT_REMEMBER_REF( port );
   INPUT_PORT_AJUST_CURSOR( port );

   return res;
}

/*---------------------------------------------------------------------*/
/*    input_port_display_error ...                                     */
/*---------------------------------------------------------------------*/
obj_t
input_port_display_error( obj_t ip, obj_t op )
{
   FILE *output        = OUTPUT_PORT( op ).file;
   long forward        = INPUT_PORT( ip ).forward - 1;
   long end_of_print   = forward;
   long start_of_print = INPUT_PORT( ip ).backward - 40;
   long count;
   char bck;
   
   /* on calcule l'endroit ou on va commencer a afficher */
   if( (long)start_of_print < 0 )
      start_of_print = 0;

   count = end_of_print - start_of_print;

   /* on calcule la chaine et la longueur de ce qu'on va afficher */
   while( (BUFFER( ip )[ end_of_print ] != '\0') &&
          (BUFFER( ip )[ end_of_print ] != '\n') &&
          (count < 80) )
      end_of_print++, count++;

   /* on trippote le buffer */
   bck = BUFFER( ip )[ end_of_print ];
   BUFFER( ip )[ end_of_print ] = '\0';

   /* on affiche la chaine */ 
   fprintf( output, "%s\n", &BUFFER( ip )[ start_of_print ] );
   
   /* on restore le buffer */
   BUFFER( ip )[ end_of_print ] = bck;

   return ip;
}

/*---------------------------------------------------------------------*/
/*     init_io ...                                                     */
/*---------------------------------------------------------------------*/
void init_io()
{
#if( !defined( _SBFSIZ ) )
#   define _SBFSIZ 8
#endif

   default_io_bufsiz = BUFSIZ * _SBFSIZ;
   
   current_output_port = make_output_port( "stdout", stdout, KINDOF_CONSOLE );
   current_error_port  = make_output_port( "stderr", stderr, KINDOF_CONSOLE );
   current_input_port  = open_input_console();
}

/*---------------------------------------------------------------------*/
/*    fexists ...                                                      */
/*---------------------------------------------------------------------*/
bool_t
fexists( char *name )
{
   return !access( name, R_OK );
}

/*---------------------------------------------------------------------*/
/*    bool_t                                                           */
/*    reset_eof ...                                                    */
/*    -------------------------------------------------------------    */
/*    Quand le port pointe sur la console, cette fonction annule la    */
/*    lecture du `^D'. Cela permet une reprise de lecture.             */
/*---------------------------------------------------------------------*/
bool_t
reset_eof( obj_t port )
{
   if( INPUT_PORT( port ).kindof == KINDOF_CONSOLE )
   {
      long bufsiz = INPUT_PORT( port ).bufsiz;
      
      /* on annule le `eof' */
      INPUT_PORT( port ).eof = 0;
      
      /* on nettoie le buffer */
      memset( (char *)BUFFER( port ), 0, bufsiz );

      /* on recale les curseurs */
      INPUT_PORT( port ).forward  = default_io_bufsiz;
      INPUT_PORT( port ).remember = default_io_bufsiz;
      INPUT_PORT( port ).backward = default_io_bufsiz;

      /* on flush , et on clear */
      fflush( stdin );
      clearerr( stdin );

      return 1;
   }
   else
      return 0;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_flush ...                                                */
/*    -------------------------------------------------------------    */
/*    On flush un string port.                                         */
/*---------------------------------------------------------------------*/
obj_t
strport_flush( obj_t port )
{
   obj_t res;

   res = string_to_bstring( OUTPUT_STRING_PORT( port ).buffer );

   OUTPUT_STRING_PORT( port ).offset = 0;
   OUTPUT_STRING_PORT( port ).buffer[ 0 ] = '\0';

   return res;
}

/*---------------------------------------------------------------------*/
/*    obj_t                                                            */
/*    strport_grow ...                                                 */
/*---------------------------------------------------------------------*/
obj_t
strport_grow( obj_t p )
{
   long  old_size, new_size;
   char *old_buffer, *new_buffer;
	
   old_buffer = OUTPUT_STRING_PORT( p ).buffer;
   old_size   = OUTPUT_STRING_PORT( p ).size;

   new_size   = old_size * 2;
   new_buffer = (char *)( GC_MALLOC( new_size + 1 ) );

   strcpy( new_buffer, old_buffer );

   OUTPUT_STRING_PORT( p ).buffer = new_buffer;
   OUTPUT_STRING_PORT( p ).size   = new_size;

   return p;
}

/*---------------------------------------------------------------------*/
/*    unsigned char *                                                  */
/*    input_port_get_annexe ...                                        */
/*    -------------------------------------------------------------    */
/*    This function is never used in the system. It is here just for   */
/*    debug purposes.                                                  */
/*---------------------------------------------------------------------*/
unsigned char *
input_port_get_annexe( obj_t p )
{
   return INPUT_PORT( p ).annexe;
}

/*---------------------------------------------------------------------*/
/*    static bool_t                                                    */
/*    pipe_name_p ...                                                  */
/*    -------------------------------------------------------------    */
/*    Is a file name a pipe name ? A pipe name start by the            */
/*    sequence "| ".                                                   */
/*---------------------------------------------------------------------*/
static bool_t
pipe_name_p( char *name )
{
   return ((name[ 0 ] == '|') && (name[ 1 ] == ' '));
}

/*---------------------------------------------------------------------*/
/*    char *                                                           */
/*    pipe_name ...                                                    */
/*    -------------------------------------------------------------    */
/*    Pipe name to name translation.                                   */
/*---------------------------------------------------------------------*/
static char *
pipe_name( char *pipe_name )
{
   return (pipe_name + 1);
}


