;* --------------------------------------------------------------------*/
;*    Copyright (c) 1992-1998 by Manuel Serrano. All rights reserved.  */
;*                                                                     */
;*                                     ,--^,                           */
;*                               _ ___/ /|/                            */
;*                           ,;'( )__, ) '                             */
;*                          ;;  //   L__.                              */
;*                          '   \   /  '                               */
;*                               ^   ^                                 */
;*                                                                     */
;*                                                                     */
;*    This program is distributed in the hope that it will be useful.  */
;*    Use and copying of this software and preparation of derivative   */
;*    works based upon this software are permitted, so long as the     */
;*    following conditions are met:                                    */
;*           o credit to the authors is acknowledged following         */
;*             current academic behaviour                              */
;*           o no fees or compensation are charged for use, copies,    */
;*             or access to this software                              */
;*           o this copyright notice is included intact.               */
;*      This software is made available AS IS, and no warranty is made */
;*      about the software or its performance.                         */
;*                                                                     */
;*      Bug descriptions, use reports, comments or suggestions are     */
;*      welcome. Send them to                                          */
;*        Manuel Serrano -- Manuel.Serrano@unice.fr                    */
;*-------------------------------------------------------------------- */
;*=====================================================================*/
;*    serrano/prgm/project/bigloo/runtime/Rgc/rules.scm                */
;*    -------------------------------------------------------------    */
;*    Author      :  Manuel Serrano                                    */
;*    Creation    :  Tue Mar 31 16:59:43 1992                          */
;*    Last change :  Sat Mar 21 17:00:43 1998 (serrano)                */
;*    -------------------------------------------------------------    */
;*    Rgc rule expansions (many thanks to John Gerard Malecki          */
;*    for his suggestions and code improvements).                      */
;*=====================================================================*/

;*---------------------------------------------------------------------*/
;*    Le module                                                        */
;*---------------------------------------------------------------------*/
(module __rgc_rule

   (import  (__error                   "Llib/error.scm"))

   (use     (__type                    "Llib/type.scm")
	    (__bigloo                  "Llib/bigloo.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__structure               "Llib/struct.scm")
	    (__tvector                 "Llib/tvector.scm")
	    (__r4_numbers_6_5          "Ieee/number.scm")
	    (__r4_numbers_6_5_fixnum   "Ieee/fixnum.scm")
	    (__r4_numbers_6_5_flonum   "Ieee/flonum.scm")
	    (__r4_characters_6_6       "Ieee/char.scm")
	    (__r4_equivalence_6_2      "Ieee/equiv.scm")
	    (__r4_booleans_6_1         "Ieee/boolean.scm")
	    (__r4_symbols_6_4          "Ieee/symbol.scm")
	    (__r4_strings_6_7          "Ieee/string.scm")
	    (__r4_pairs_and_lists_6_3  "Ieee/pair-list.scm")
	    (__r4_input_6_10_2         "Ieee/input.scm")
	    (__r4_control_features_6_9 "Ieee/control.scm")
	    (__r4_vectors_6_8          "Ieee/vector.scm")
	    (__r4_ports_6_10_1         "Ieee/port.scm")
	    (__r4_output_6_10_3        "Ieee/output.scm")
	    (__rgc                     "Rgc/runtime.scm")
	    (__evenv                   "Eval/evenv.scm"))

   (export (expand-rule rule env)
	   (mark-rule rule mark)))


;*---------------------------------------------------------------------*/
;*    *marker* ...                                                     */
;*---------------------------------------------------------------------*/
(define *marker* (gensym 'mark))

;*---------------------------------------------------------------------*/
;*    mark-rule ...                                                    */
;*---------------------------------------------------------------------*/
(define (mark-rule rule mark)
   `(,rule (,*marker* ,mark)))

;*---------------------------------------------------------------------*/
;*    lookup ...                                                       */
;*---------------------------------------------------------------------*/
(define-inline (lookup var env)
   (assq var env))

;*---------------------------------------------------------------------*/
;*    expanded? ...                                                    */
;*---------------------------------------------------------------------*/
(define-inline  (expanded? b)
   (eq? (cadr b) #t))

;*---------------------------------------------------------------------*/
;*    binding-ref ...                                                  */
;*---------------------------------------------------------------------*/
(define-inline (binding-ref b)
   (caddr b))

;*---------------------------------------------------------------------*/
;*    expand-binding! ...                                              */
;*---------------------------------------------------------------------*/
(define-inline  (expand-binding! b env)
   (set-cdr! b (list #t (expand-rule (cadr b) env))))

;*---------------------------------------------------------------------*/
;*    expand-rule ...                                                  */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-rule rule Renv)
   (if (not (pair? rule))
       (expand-atom rule Renv)
       (case (car rule)
	  ((eof bol eol)  (expand-trap rule Renv))
	  ((all)          (expand-all Renv))
	  ((uncase)       (expand-uncase-string rule Renv))
	  ((...)          (expand-... rule Renv))
	  ((*)            (expand-* rule Renv))
	  ((+)            (expand-+ rule Renv))
	  ((?)            (expand-? rule Renv))
	  ((! or)         (expand-! rule Renv))
	  ((>-< inside)   (expand->-< rule Renv))
	  ((<-> outside)  (expand-<-> rule Renv))
	  ((in)           (expand-in rule Renv))
	  ((out)          (expand-out rule Renv))
	  ((context)      (expand-context rule Renv))
	  (else
	   (if (eq? (car rule) *marker*)
	       (expand-marker rule Renv)
	       (expand-. rule Renv))))))

;*---------------------------------------------------------------------*/
;*    check-arity? ...                                                 */
;*    rule x int --> { t, f }                                          */
;*---------------------------------------------------------------------*/
(define (check-arity? rule arity)
   (=fx (length rule) (+fx arity 1)))

;*---------------------------------------------------------------------*/
;*    expand-trap ...                                                  */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-trap rule Renv)
   (if (check-arity? rule 1)
       `(trap ,(car rule) ,(expand-rule (cadr rule) Renv))
       (error #f "Rgc:wrong number of arguments" rule)))

;*---------------------------------------------------------------------*/
;*    expand-distribute ...                                            */
;*    operator x rule * x Renv --> rexp                                */
;*---------------------------------------------------------------------*/
(define (expand-distribute operator rules Renv)
   (let loop ((rules rules))
      (if (null? rules)
	  '()
	  (if (null? (cdr rules))
	      (expand-rule (car rules) Renv)
	      (list operator (expand-rule (car rules) Renv)
		    (loop (cdr rules)))))))
    
;*---------------------------------------------------------------------*/
;*    expand-atom ...                                                  */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-atom rule Renv)
   (cond
      ((char? rule)
       `(char ,rule))
      ((string? rule)
       (expand-string rule Renv))
      (else
       (let ((b (lookup rule Renv)))
	  (if b
	      (begin
		 (if (not (expanded? b))
		     (expand-binding! b Renv))
		 (binding-ref b))
	      (error #f "Rgc:unbound regular variable" rule))))))

;*---------------------------------------------------------------------*/
;*    expand-string ...                                                */
;*    string x Renv --> rexp                                           */
;*---------------------------------------------------------------------*/
(define (expand-string string Renv)
   (expand-distribute 'concat-char
		      (let ((len (string-length string)))
			 (let loop ((i    0)
				    (acc '()))
			    (if (=fx i len)
				(reverse! acc)
				(loop (+fx i 1)
				      (cons (string-ref string i) acc)))))
		      Renv))

;*---------------------------------------------------------------------*/
;*    expand-all ...                                                   */
;*    Renv --> rexp                                                    */
;*---------------------------------------------------------------------*/
(define (expand-all Renv)
   (expand-<-> '(<-> #\Newline) Renv))

;*---------------------------------------------------------------------*/
;*    expand-uncase-string ...                                         */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-uncase-string rule Renv)
   (if (check-arity? rule 1)
       (if (string? (cadr rule))
	   (expand-distribute
	    'concat-char
	    (let* ((str (cadr rule))
		   (len (string-length str)))
	       (let loop ((i    0)
			  (acc '()))
		  (if (=fx i len)
		      (reverse! acc)
		      (let ((char (string-ref str i)))
			 (if (and (char>=? char #\A)
				  (char<=? char #\z))
			     (loop (+fx i 1) (cons `(in ,(char-upcase char)
						      ,(char-downcase char))
						 acc))
			     (loop (+fx 1 i) (cons char acc)))))))
	    Renv)
	   (error #f "Rgc:uncase, argument not a string" rule))
       (error #f "Rgc:uncase, wrong number of arguments" rule)))

;*---------------------------------------------------------------------*/
;*    expand-... ...                                                   */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-... rule Renv)
   (if (check-arity? rule 2)
       (cond
	  ((not (integer? (cadr rule)))
	   (error #f "Rgc:..., the first args has to be a number" rule))
	  ((pair? (caddr rule))
	   (if (and (eq? (car (caddr rule)) 'uncase)
		    (string? (cadr (caddr rule))))
	       (expand-! `(! ,@(explode-uncase-string (cadr rule)
						      (caddr rule)))
			 Renv)
	       (error #f "Rgc:badly formed regular-expression" rule)))
	  ((string? (caddr rule))
	   (if (>fx (cadr rule) (string-length (caddr rule)))
	       (error #f "Rgc:number bigger than string's length" rule)
	       (expand-! `(! ,@(explode-string (cadr rule) (caddr rule)))
			 Renv)))
	  (else
	   (error #f "Rgc:badly formed regular-expression" rule)))))

;*---------------------------------------------------------------------*/
;*    explode-string ...                                               */
;*    int x string --> rexp                                            */
;*---------------------------------------------------------------------*/
(define (explode-string number string)
   (let ((len (string-length string)))
      (let loop ((j number))
	 (if (>fx j len)
	     '()
	     (cons (substring string 0 j) (loop (+fx j 1)))))))

;*---------------------------------------------------------------------*/
;*    explode-uncase-string ...                                        */
;*    int x string --> rexp                                            */
;*---------------------------------------------------------------------*/
(define (explode-uncase-string number string)
   (let ((len (string-length string)))
      (let loop ((j number))
	 (if (>fx j len)
	     '()
	     (cons `(uncase ,(substring string 0 j)) (loop (+fx j 1)))))))

;*---------------------------------------------------------------------*/
;*    expand-* ...                                                     */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-* rule Renv)
   (if (integer? (cadr rule))
       (if (check-arity? rule 2)
	   `(or (epsilon) ,(expand-to-n-times rule Renv))
	   (error #f "Rgc:wrong number of arguments" rule))
       (if (check-arity? rule 1)
	   `(* ,(expand-rule (cadr rule) Renv))
	   (error #f "Rgc:wrong number of arguments" rule))))

;*---------------------------------------------------------------------*/
;*    expand-to-n-times ...                                            */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-to-n-times rule Renv)
   (let ((rexp  (caddr rule))
	 (times (cadr  rule)))
      (if (<fx times 1)
	  (error #f "Rgc:illegal construction" rule)
	  (expand-! `(! ,rexp ,@(let loop ((i 2))
				   (if (>fx i times)
				       '()
				       (cons `(+ ,i ,rexp)
					     (loop (+fx i 1))))))
		    Renv))))

;*---------------------------------------------------------------------*/
;*    expand-+ ...                                                     */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-+ rule Renv)
   (if (integer? (cadr rule))
       (if (check-arity? rule 2)
	   (expand-n-times rule Renv)
	   (error #f "Rgc:wrong number of arguments" rule))
       (if (check-arity? rule 1)
	   `(+ ,(expand-rule (cadr rule) Renv))
	   (error #f "Rgc:wrong number of arguments" rule))))

;*---------------------------------------------------------------------*/
;*    expand-n-times ...                                               */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-n-times rule Renv)
   (let ((rexp  (caddr rule))
         (times (cadr  rule)))
      (if (<fx times 1)
	  (error #f "Rgc:illegal construction" rule)
	  (expand-rule (let loop ((i 0))
				   (if (=fx i times)
				       '()
				       (cons rexp (loop (+fx i 1)))))
				Renv))))

;*---------------------------------------------------------------------*/
;*    expand-? ...                                                     */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-? rule Renv)
   (if (check-arity? rule 1)
       `(or ,(expand-rule (cadr rule) Renv) (epsilon))
       (error #f "Rgc:wrong number of arguments" rule)))

;*---------------------------------------------------------------------*/
;*    expand-! ...                                                     */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-! rule Renv)
   (expand-distribute 'or (cdr rule) Renv))
 
;*---------------------------------------------------------------------*/
;*    expand->-< ...                                                   */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand->-< rule Renv)
   (if (odd? (length (cdr rule)))
       (error #f "Rgc:wrong number of arguments" rule)
       (let loop ((interval (build-interval (cdr rule)))
		  (acc      '()))
	  (if (null? interval)
	      `(in ,(reverse! acc))
	      (loop (cdr interval)
		    (cons (integer->char (car interval)) acc))))))

;*---------------------------------------------------------------------*/
;*    expand-<-> ...                                                   */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-<-> rule Renv)
   (let ((vexecpt 'dummy)
	 (fexecpt 'dummy)
	 (acc     '()))
      (cond
	 ((null? (cdr rule))
	  (error #f "Rgc:wrong number of arguments" rule))
	 ((null? (cddr rule))
	  (set! vexecpt (safe-char->integer (cadr rule)))
	  (set! fexecpt eq?))
	 ((odd? (length (cdr rule)))
	  (error #f "Rgc:wrong number of arguments" rule))
	 (else
	  (set! vexecpt (build-interval (cdr rule)))
	  (set! fexecpt memq)))
      (let loop ((i *rgc-first-char*))
	 (if (=fx i *rgc-last-char*)
	     `(in ,(reverse! acc))
	     (begin
		(if (not (fexecpt i vexecpt))
		    (set! acc (cons (integer->char i) acc)))
		(loop (+fx i 1)))))))
    
;*---------------------------------------------------------------------*/
;*    build-interval ...                                               */
;*    [ min, max ] * --> integer *                                     */
;*    -------------------------------------------------------------    */
;*    Cette fonction est utilisee par expand-<-> et expand->-<.        */
;*    Elle retourne une liste d'INTEGER.                               */
;*---------------------------------------------------------------------*/
(define (build-interval b*)
   (labels ((inter (min max)
		   (if (eq? min max)
		       (list min)
		       (if (eq? min (+fx 1 *rgc-last-char*))
			   (error #f "Rgc:incorrect interval" b*)
			   (cons min (inter (+fx 1 min) max))))))
      (let loop ((b* b*))
	 (if (null? b*)
	     '()
	     (append (inter (safe-char->integer (car b*))
			    (safe-char->integer (cadr b*)))
		     (loop (cddr b*)))))))
    
;*---------------------------------------------------------------------*/
;*    expand-chars ...                                                 */
;*    char_or_interval list --> char list                              */
;*---------------------------------------------------------------------*/
(define (expand-chars rule)
  (labels ((expand (chars acc)
		   (if (null? chars)
		       acc
		       (match-case (car chars)
			 ((and ?c (? char?))
			  (expand (cdr chars) (cons c acc)))
			 ((?- ?-)
			  (let ((inter (build-interval (car chars))))
			     (let loop ((i inter))
				(if (null? i)
				    (expand (cdr chars) (append inter acc))
				    (begin
				       (set-car! i (integer->char (car i)))
				       (loop (cdr i)))))))
			 (else
			  (error #f "Rgc:invalid regexp chars" chars))))))
    (let ((chars (cdr rule)))
      (if (null? chars)
	  (error #f "Rgc:incomplete regexp" rule)
	  (expand chars '())))))

;*---------------------------------------------------------------------*/
;*    expand-in ...                                                    */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-in rule Renv)
   `(in ,(expand-chars rule)))

;*---------------------------------------------------------------------*/
;*    expand-out ...                                                   */
;*---------------------------------------------------------------------*/
(define (expand-out rule Renv)
   (let ((out-list (map safe-char->integer (expand-chars rule))))
      (let loop ((i   *rgc-first-char*)
		 (acc '()))
	 (if (=fx i *rgc-last-char*)
	     `(in ,acc)
	     (begin
		(if (not (memq i out-list))
		    (loop (+fx i 1)
			  (cons (integer->char i) acc))
		    (loop (+fx i 1)
			  acc)))))))
 
;*---------------------------------------------------------------------*/
;*    expand-context ...                                               */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-context rule Renv)
   (if (check-arity? rule 2)
       (if (not (pair? (cadr rule)))
	   (error #f "Rgc:illegal `context' form" rule)
	   `(trap (context ,(cadr rule))
		  ,(expand-rule (caddr rule) Renv)))
       (error #f "Rgc:wrong number of argument" rule)))

;*---------------------------------------------------------------------*/
;*    expand-marker ...                                                */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-marker rule Renv)
   (if (check-arity? rule 1)
       `(end ,(cadr rule))
       (error #f "Rgc:wrong number of argument" rule)))

;*---------------------------------------------------------------------*/
;*    expand-. ...                                                     */
;*    rule x Renv --> rexp                                             */
;*---------------------------------------------------------------------*/
(define (expand-. rule Renv)
   (if (null? (cdr rule))
       (expand-rule (car rule) Renv)
       (expand-distribute 'concat rule Renv)))
    
;*---------------------------------------------------------------------*/
;*    safe-char->integer ...                                           */
;*---------------------------------------------------------------------*/
(define-inline (safe-char->integer obj)
   (if (char? obj)
       (char->integer obj)
       (error #f
	      (string-append "Rgc:type error: `CHAR' expected, `"
			     (find-runtime-type obj)
			     "' provided")
	      obj)))
	      
