/*
 * X-Mame system independent sound code
 */

#define __SOUND_C_

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#ifdef USE_TIMER
#include <sys/time.h>
#include <signal.h>
static int bytes_per_timer_alarm;
#endif

#include "xmame.h"
#include "sound.h"

/* #define SOUND_DEBUG */

static float master_volume_divider = 512;
static int sound_active = 0;
VOICE_T voices[AUDIO_NUM_VOICES];
static unsigned char buf[AUDIO_BUFF_SIZE];

/*********** code to quick alloc/free SAMPLE_T items **************
 * malloc/free is a too cpu-expensive process. When using a fixed size
 * malloc/free scheme , it's better to pre-allocate space and use this
 * algorithm
 *
 *   In our code SAMPLE_T is ideal for a quick malloc scheme. Of course
 * this is not applicable to data samples ( not fixed size ... )
 **************************************************************************/
 
/* this should be enought ....*/
#define ALLOC_TABLE_SIZE 64

static SAMPLE_T		SampleTTable[ALLOC_TABLE_SIZE];
static int		SampleTIndex[ALLOC_TABLE_SIZE];
static int		SampleTPointer;

int InitSampleTTable(void) {
    int i;
    for (i=0;i<ALLOC_TABLE_SIZE;i++) SampleTIndex[i]=i;
    SampleTPointer=(ALLOC_TABLE_SIZE - 1 );
    return 0;
}

int FreeSampleT(SAMPLE_T *pt) {
#if 1
    int item = (int) ( pt - SampleTTable );
    if (SampleTPointer>=(ALLOC_TABLE_SIZE-1) ){
	fprintf(stderr,"FreeSampleT() consistency error\n");
	exit(1);
    }
    SampleTIndex[++SampleTPointer] = item;
#else
    free(pt);
#endif
    return 0;
}

SAMPLE_T  *AllocSampleT(void) {
#if 1
    SAMPLE_T *pt;
    if (SampleTPointer<0) return (SAMPLE_T *) NULL; /* no items availables */
    pt = &SampleTTable[SampleTIndex[SampleTPointer--]];
    return pt;
#else
    return (SAMPLE_T *)malloc(sizeof(SAMPLE_T));
#endif
}
   
/****************************************************************************
*
* Timer based sound routines
*
* Used in those systems that doesn't allow how many bytes can be send to the
* audio device without blocking
* 
*****************************************************************************/

#ifdef USE_TIMER
struct sigaction sig_action;

int start_timer() {
	struct itimerval timer_value ;
	void audio_timer(int arg);
	/* set and configure timer catcher */
	if (!play_sound) return OSD_OK;
	sig_action.sa_handler = audio_timer;
#ifdef hpux
	sig_action.sa_flags   = SA_RESETHAND;
#else
	sig_action.sa_flags   = SA_RESTART;
#endif

	if(sigaction (SIGALRM, &sig_action, NULL)< 0) {
		fprintf(stderr,"Sigaction Failed \n");
		return OSD_NOT_OK;
	}
	/* set and configure realtime alarm */
  	timer_value.it_interval.tv_sec =
 	timer_value.it_value.tv_sec    = 0L;
  	timer_value.it_interval.tv_usec=
  	timer_value.it_value.tv_usec   = 1000000L / audio_timer_freq;

	if (setitimer (ITIMER_REAL, &timer_value, NULL)) {
 		printf ("Setting the timer failed.\n");
  		return OSD_NOT_OK;
  	}
	return OSD_OK;
}

void audio_timer (int arg) {
        static int  in = FALSE;
        void mix_sound(void);
	if (!play_sound) return;
        if (!in) {
            in = TRUE;
            mix_sound();
            in = FALSE;
        } /* else {
          fprintf(stderr, ".");
        } */
#if defined solaris || defined sgi
	/* rearm alarm timer */
	sigaction (SIGALRM, &sig_action, NULL);
#endif
}

#endif  /* ifdef USE_TIMER */
/*****************************************************************************
* 
* NON-Timer based sound routines
*
* the idea is simple: I inquire audio driver how many bytes can i send, 
* parse data trough the mixer and send it to audio (or if the system is
* timer based, wait for a timer irq to arrive )
*
* Modified and hacked up for mame and sound by
*  Bill Lash - lash@tellabs.com
******************************************************************************/

#ifdef USE_TIMER
void osd_update_audio(void) /* make osd_update_audio do nothing, name it */
{                           /* mix_sound which is called by audio_timer  */
}                           /* a bit dirty but it does what we want.     */

void mix_sound(void)
#else
void osd_update_audio(void)
#endif
{
   int i,j;
   long size;
   SAMPLE_T *new_sample;
   unsigned char *current_data_pt;
   unsigned char *end_data_pt;
   float vol, pos_frac, freq_fac;
   
   if ((play_sound == 0) || (sound_active == 0))  return;
   
#ifdef USE_TIMER
   size=bytes_per_timer_alarm;
#else
   if( (size=sysdep_get_audio_freespace()) <= 256) return;
#endif

   memset(buf,128,size);
   sound_active = 0;   

   for(j=0;j<AUDIO_NUM_VOICES;j++) {
      SAMPLE_T *pt = voices[j].sample;
      if (pt) {
         current_data_pt = voices[j].current_data_pt;
         pos_frac        = voices[j].pos_frac;
         end_data_pt     = pt->end_data_pt;
         vol             = pt->vol;
         freq_fac        = pt->freq_fac;
         sound_active    = 1;
      }
      else continue;   
      
      for (i=0;i<size;i++)
      {
#ifndef FANCY_SOUND	    
         buf[i] += (*current_data_pt + 
	  ((*(current_data_pt + 1) - *current_data_pt) * pos_frac) - 128) * vol;
#else
         buf[i] += (*current_data_pt - 128) * vol;
#endif
         pos_frac += freq_fac;
         while(pos_frac > 1) {
            pos_frac--;
            current_data_pt++;
         }
         if ( current_data_pt > end_data_pt ) {
            if( pt->loop_stream ) {
               if( pt->next_sample ) {
                  new_sample = pt->next_sample;
                  if (pt->data) free(pt->data);
                  FreeSampleT(pt);
                  voices[j].sample = pt = new_sample;
                  end_data_pt = pt->end_data_pt;
                  vol         = pt->vol;
               }
               current_data_pt=pt->data;
            } else {
               osd_stop_sample(j);
               i = size;
            }
         } /* if at end of sample */
      } /* for i=0;i<size;i++) */
      voices[j].current_data_pt = current_data_pt;
      voices[j].pos_frac = pos_frac;
   } /* for j< AUDIO_NUM_VOICES */
#ifdef SOUND_DEBUG
   fprintf(stderr,"osd_update_audio(): %d avail %d used\n",size,i);
#endif
   if (sound_active) sysdep_play_audio(buf,size);
}

/****************************************************************************
* 
* General sound routines
*
*****************************************************************************/

int sysdep_audio_initvars(void)
{
	int i;
	
	if (!play_sound ) return OSD_OK;
	
	sound_active = 0;
	audio_timer_freq  = AUDIO_TIMER_FREQ;
#ifdef USE_TIMER
	bytes_per_timer_alarm=audio_sample_freq/audio_timer_freq;
#endif
	InitSampleTTable();
        for (i = 0; i < AUDIO_NUM_VOICES; i++) {
		voices[i].sample = NULL;
		voices[i].current_data_pt = NULL;
		voices[i].pos_frac = 0;
        }
	return OSD_OK;
}

void osd_set_mastervolume(int volume)
{
	int channel;
	float old_master_volume_divider;
	
	if (! play_sound ) return;
	
        old_master_volume_divider=master_volume_divider;
        master_volume_divider=(float)256 * 100 / volume;
        
	for (channel=0; channel < AUDIO_NUM_VOICES; channel++) {
	   SAMPLE_T *pt=voices[channel].sample;
	   for(;pt;pt=pt->next_sample) 
	      pt->vol *= old_master_volume_divider/master_volume_divider;
	}
}


void osd_play_sample(int channel,unsigned char *data,int len,int freq,int volume,int loop)
{
        if ((play_sound == 0) || (channel >= AUDIO_NUM_VOICES)) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr,"osd_play_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}
        sound_active=1;

	osd_stop_sample(channel);
	if ( !(voices[channel].sample=AllocSampleT()) ) return;
	if ( !(voices[channel].sample->data=(unsigned char *)malloc(len+1)) ) {
		FreeSampleT(voices[channel].sample);
		voices[channel].sample=(SAMPLE_T *)NULL;
		return;
	}
#ifdef FANCY_SOUND
        /* calculate one sample more as the actual length for interpolation */
	if (loop) voices[channel].sample->data[len]=data[0];
	 else voices[channel].sample->data[len]=128;
#endif	 
	voices[channel].current_data_pt = voices[channel].sample->data;
	voices[channel].sample->end_data_pt = voices[channel].sample->data + len - 1;
	voices[channel].pos_frac = 0;
	voices[channel].sample->next_sample = (SAMPLE_T *) NULL;
	voices[channel].sample->loop_stream = loop;
 	voices[channel].sample->vol = (float)volume / master_volume_divider;
	voices[channel].sample->freq_fac = (float)freq / audio_sample_freq;
        memcpy(voices[channel].sample->data,data,len);
#ifdef SOUND_DEBUG
        fprintf(stderr,"play() chan:%d len:%d frec:%d vol:%d loop:%d\n",channel,len,freq,volume,loop); 
#endif
	return;
}

void osd_play_streamed_sample(int channel,unsigned char *data,int len,int freq,int volume)
{
	SAMPLE_T *new_samp, *last_samp=NULL;

        if ((play_sound == 0) || (channel >= AUDIO_NUM_VOICES)) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr,"play_streamed_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}
        sound_active=1;

	if(voices[channel].sample == NULL) {
	    voices[channel].sample = AllocSampleT();
	    new_samp= voices[channel].sample;
	} else {
	    last_samp = voices[channel].sample;
	    while (last_samp->next_sample) last_samp=last_samp->next_sample;
	    last_samp->next_sample = AllocSampleT();
	    new_samp = last_samp->next_sample;
	}
	if (!new_samp) return; /* no resources availables */	
	new_samp->data = (unsigned char *)malloc(len+1);
	if (! new_samp->data) {
	    if ( new_samp == voices[channel].sample ) 
		 voices[channel].sample = NULL;
	    else last_samp->next_sample = NULL;
	    FreeSampleT(new_samp);
	    return; /* cannot malloc data space: free sampleT and return */
	}
#ifdef FANCY_SOUND
        /* calculate one sample more as the actual length for interpolation */
	new_samp->data[len]=data[0];
	if(last_samp) 
	   *(last_samp->end_data_pt + 1)=data[0];
	else
#else
	if(last_samp == NULL)
#endif
	   voices[channel].current_data_pt=new_samp->data;
	new_samp->loop_stream = 1;
	new_samp->next_sample = NULL;
	new_samp->end_data_pt = new_samp->data + len - 1;
	new_samp->vol = (float)volume / master_volume_divider;
	new_samp->freq_fac = (float)freq / audio_sample_freq;
        memcpy(new_samp->data,data,len);
}

void osd_adjust_sample(int channel,int freq,int volume)
{
	SAMPLE_T *next_samp;
        if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;
	if ((freq<10) || (freq> 100000)) 
	{ 
#ifdef SOUND_DEBUG
	fprintf(stderr,"osd_adjust_sample() call Ignored: chan:%d frec:%d vol:%d\n",channel,freq,volume);
#endif
	   return;
	}
	if(voices[channel].sample != NULL) {
	    next_samp = voices[channel].sample;
	    while(next_samp != NULL) {
	    	next_samp->vol = (float)volume / master_volume_divider;
	    	next_samp->freq_fac = (float)freq / audio_sample_freq;
	    	next_samp = next_samp->next_sample;
	    }
	}
}

void osd_stop_sample(int channel)
{
	SAMPLE_T *next_samp;
	SAMPLE_T *curr_samp;
        if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;

	if(voices[channel].sample != NULL) {
	    next_samp = voices[channel].sample;
	    voices[channel].sample = NULL;
	    voices[channel].current_data_pt = NULL;
	    voices[channel].pos_frac = 0;
	    while(next_samp != NULL) {
	    	curr_samp = next_samp;
	    	next_samp = next_samp->next_sample;
	    	if (curr_samp->data != NULL) free(curr_samp->data);
	    	FreeSampleT(curr_samp);
	    }
	}
}

void osd_restart_sample(int channel)
{
	if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return;
	voices[channel].current_data_pt = voices[channel].sample->data;
}

int osd_get_sample_status(int channel)
{
        int stopped=0;
        
	if (play_sound == 0 || channel >= AUDIO_NUM_VOICES) return -1;
	if (voices[channel].sample == NULL) stopped=1;

        return stopped;
}
