/*
 * (C) Copyright Keith Visco 1998, 1999  All rights reserved.
 *
 * The contents of this file are released under an Open Source 
 * Definition (OSD) compliant license; you may not use this file 
 * execpt in compliance with the license. Please see license.txt, 
 * distributed with this file. You may also obtain a copy of the
 * license at http://www.clc-marketing.com/xslp/license.txt
 *
 * The program is provided "as is" without any warranty express or
 * implied, including the warranty of non-infringement and the implied
 * warranties of merchantibility and fitness for a particular purpose.
 * The Copyright owner will not be liable for any damages suffered by
 * you as a result of using the Program. In no event will the Copyright
 * owner be liable for any special, indirect or consequential damages or
 * lost profits even if the Copyright owner has been advised of the
 * possibility of their occurrence.
 *
 * 
 */

package com.kvisco.xsl;

import com.kvisco.net.URIUtils;
import com.kvisco.xsl.util.*;
import com.kvisco.xml.*;
import com.kvisco.xml.XMLUtil;
import com.kvisco.util.QuickStack;

import java.io.*;
import java.util.Enumeration;
import java.util.Hashtable;
import java.util.Properties;

import org.w3c.dom.*;

/**
 * A class for reading an XSL Stylesheet from a stream or file.
 * @author <a href="mailto:kvisco@ziplink.net">Keith Visco</a>
 * <PRE>
 *  Modifcations
 *    19990804: Mike Los (comments with MEL)
 *        - modified #readDocument to close InputStream
 * </PRE>
**/
public class XSLReader {
    
    private static final String _MISSING_DOCUMENT_ELEMENT_ERR
        = "invalid stylesheet, missing document element.";
          
    /**
     * XSL Namespace property
    **/
    private static final String XSLNS_PREFIX = "xslns-prefix";
    
    
    /**
     * PrintWriter to write error messages to
    **/
	private PrintWriter errorWriter                = null;
    
    /**
     * The DOM Reader to use when reading an XSLStylesheet
    **/
	private DOMReader domReader                = null;

    private Hashtable  scriptTypeMap           = null;
    
      //----------------/
     //- Constructors -/
    //----------------/
    
    /**
     * Creates a new XSLReader using the Default DOMReader
     * @exception when this XSLReader cannot load the Default DOMReader
    **/
    public XSLReader() throws Exception {
        
	    // Read in properties for XSL:P
	    Properties props = new Properties();
	    try {
	        InputStream is = null;
	        // look in file system
	        File propsFile = new File(XSLProcessor.PROPERTIES_FILE);
	        if (propsFile.exists()) {
	            is = new FileInputStream(propsFile);
	        }
	        else is = getClass().getResourceAsStream(XSLProcessor.PROPERTIES_FILE);
	        if (is != null) props.load(is);
	    }
	    catch(IOException ioEx) { 
	        //-- use defaults 
	    }
	    // Create DOMReader 
	    String domPackage = props.getProperty(XSLProcessor.DOM_PACKAGE);
	    DOMReader dr = null;
	    if (domPackage != null) 
	        dr = new DOMReader(props.getProperty(domPackage));
	    else dr = new DOMReader();
	    init(dr);
        
    } //-- XSLReader
    
    /**
     * Creates a new XSLReader using the given DOMReader 
    **/
	public XSLReader(DOMReader domReader) {
	    super();
	    init(domReader);
    } //--XSLReader

   private void init(DOMReader domReader) {
	    this.domReader = domReader;
	    errorWriter = new PrintWriter(System.out, true);
	    //-- temporary mapping for xsl:functions type attr to
	    //-- xsl:script language attr
        scriptTypeMap = new Hashtable(3);
        scriptTypeMap.put("text/javascript", "ECMAScript");
        scriptTypeMap.put("text/ecmascript", "ECMAScript");
        scriptTypeMap.put("text/jpython", "JPython");
   } //-- init
   
      //------------------/
     //- Public Methods -/
    //------------------/

    /**
     * Parses a DOM Document into an XSL stylesheet
     * @return the XSLStylesheet
     * @exception XSLException
    **/
    public XSLStylesheet readStylesheet (Document document, String filename) 
        throws XSLException {

        
        XSLStylesheet stylesheet = new XSLStylesheet();
        stylesheet.setHref(filename);
        stylesheet.setDocumentBase(URIUtils.getDocumentBase(filename));
        if (document == null) return stylesheet;
        NodeList nl = document.getChildNodes();
        Node node;
        // copy PIs
        for (int i = 0; i < nl.getLength(); i++) {
            node = nl.item(i);
            if (node.getNodeType() == Node.PROCESSING_INSTRUCTION_NODE) {
                stylesheet.addPI((ProcessingInstruction)node);
            }
        }
        
        // get document element and make sure it's
        // the xsl:stylesheet element
        Element root = document.getDocumentElement();
        if (root == null)
            throw new XSLException(_MISSING_DOCUMENT_ELEMENT_ERR); 

        // copy attributes
        stylesheet.copyAttributes(root);
        
        //-- check xsl namespace
        String tagName = root.getNodeName();
        String nsPrefix = XMLUtil.getNameSpace(tagName);
        if (!stylesheet.getXSLNSPrefix().equals(nsPrefix))
            throw new XSLException("Invalid stylesheet. root element must be '" + 
                Names.STYLESHEET + "' element belonging to the XSL Namespace.");
        
        tagName = XMLUtil.getLocalPart(tagName);
        
        if ( (!Names.STYLESHEET.equals(tagName)) &&
             (!Names.TRANSFORM.equals(tagName)) ) 
            throw new XSLException("Invalid stylesheet. missing root '" + 
                Names.STYLESHEET + "' or '" + Names.TRANSFORM + "' element.");
        
        parseStylesheetElement(root, stylesheet);
        return stylesheet;
    } // readStylesheet
    
    
    
    /**
     * Reads an XSL stylesheet from the given filename
     * @param filename the file name of the XSL Stylesheet
     * @return the new XSLStylesheet
     * @exception XSLException
    **/
	public XSLStylesheet readStylesheet(String filename) 
	    throws XSLException 
    {
        XSLStylesheet xsl = null;
        if (filename != null) {
            InputStream xslInput = null;
            try {
                xslInput = URIUtils.getInputStream(filename,null);
                xsl = readStylesheet(xslInput, filename);
            }
            catch(Exception ex) {
                errorWriter.println(ex.getMessage());
                return new XSLStylesheet();
            }
            finally { //-- added by MEL to close InputStream
                try {
                    if (xslInput != null) xslInput.close();
                }
                catch(java.io.IOException ioException) {};
            }
        }
        else {
            xsl = new XSLStylesheet();
        }
        
        return xsl;
	} //-- readStylesheet

    /**
     * Reads an XSLStylesheet from the given InputStream. 
     * @param xslInput the InputStream to read the Stylesheet from.
     * @param filename the file name to use for the href of the Stylesheet,
     * the document base of this Stylesheet for imports, and for
     * error reporting.
     * @return the new XSLStylesheet
     * @exception XSLException
    **/
	public XSLStylesheet readStylesheet(InputStream xslInput, String filename) 
	    throws XSLException {
        XSLStylesheet xsl = null;
        String mFilename = filename;
        if (xslInput != null) {
            if (mFilename == null) mFilename = "XSL InputStream";
            Document xslDocument = domReader.readDocument(xslInput, mFilename, false, errorWriter);
            xsl = readStylesheet(xslDocument,filename);
        }
        return xsl;
	} //-- readStylesheet

      //------------------------------------------/
     //- Methods for Creating XSLObject Objects -/
    //------------------------------------------/
    
    /**
     * Creates the corresponding XSLObject for the given element
     * @param parent the XSLStylesheet in while the XSLObject will
     * be inserted
     * @param element the Element to create the XSLObject from
    **/
    public XSLObject createXSLObject(XSLStylesheet parent, Element element) 
        throws XSLException
    {
        
        if (element == null) //-- just in case
            throw new XSLException("null element encountered");
            
        
        String xslNSPrefix       = parent.getXSLNSPrefix();
        XSLObject  xslObj    = null;
        String     tagName   = element.getNodeName();
        String     nsPrefix  = XMLUtil.getNameSpace(tagName);
        short      xslType   = XSLObject.LITERAL;
            
        if (xslNSPrefix.equals(nsPrefix)){
            if (nsPrefix.length() > 0)
                tagName = tagName.substring(nsPrefix.length()+1);
            xslType = XSLObject.getTypeFromName(tagName);
        }
        boolean copyAtts = true;
        boolean processChildren = true;

        switch (xslType) {
            case XSLObject.APPLY_TEMPLATES:
            case XSLObject.APPLY_IMPORTS:
            case XSLObject.FOR_EACH:
                xslObj = new Selection(parent,xslType);
                break;
            case XSLObject.ATTRIBUTE_SET:
                String setName = element.getAttribute(Names.NAME_ATTR);
                xslObj = new AttributeSet(parent, setName);
                copyAtts = false;
                break;
            case XSLObject.CALL_TEMPLATE:
                String templateName = element.getAttribute(Names.NAME_ATTR);
                xslObj = new XSLCallTemplate(parent, templateName);
                processChildren = true;
                break;
            case XSLObject.CDATA:   //-- proprietary
                XSLCData cdata = new XSLCData(parent);
                cdata.appendData(XSLObject.getText(element));
                xslObj = cdata;
                copyAtts = false;
                processChildren = false;
                break;
            case XSLObject.COPY_OF:
                xslObj = new CopyOf(parent);
                processChildren = false;
                break;
            case XSLObject.ID:
                xslObj = new Id(parent);
                break;
            case XSLObject.IF:
            case XSLObject.WHEN:
                xslObj = new XSLIf(parent);
                break;
            case XSLObject.IMPORT:
                xslObj = new XSLImport(parent);
                processChildren = false;
                break;
            case XSLObject.INCLUDE:
            case XSLObject.PRESERVE_SPACE:
            case XSLObject.STRIP_SPACE:
                xslObj = new EmptyXSLObject(parent,xslType);
                processChildren = false;
                break;            
            case XSLObject.NUMBER:
                xslObj = new XSLNumber(parent);
                break;
            case XSLObject.OTHERWISE:
                xslObj = new XSLOtherwise(parent);
                copyAtts = false;
                break;
            case XSLObject.OUTPUT:
                xslObj = new Output(parent);
                break;
            case XSLObject.FUNCTIONS:
                //-- change later to XSLFunctions
                xslObj = new XSLScript(parent, XSLObject.getText(element));
                processChildren = false;
                break;
            case XSLObject.SCRIPT:
                xslObj = new XSLScript(parent,XSLObject.getText(element));
                processChildren = false;
                break;
            case XSLObject.SORT:
                xslObj = new XSLSort(parent);
                processChildren = false;
                break;
            case XSLObject.TEMPLATE:
                xslObj = new TemplateRule(parent);
                break;
            case XSLObject.TEXT:
                XSLText xslText = new XSLText(parent);
                xslText.appendData(XSLObject.getText(element));
                xslObj = xslText;
                processChildren = false;
                copyAtts = false;
                break;
            case XSLObject.USE:
                xslObj = new XSLUse(parent);
                processChildren = false;
                break;
            case XSLObject.VALUE_OF:
                xslObj = new ValueOf(parent);
                processChildren = false;
                break;
            case XSLObject.VARIABLE:
            {
                // since the name of a variable can't be changed,
                // we need to set it at creation time
                String name = element.getAttribute(Names.NAME_ATTR);
                Attr exprAttr = element.getAttributeNode(Names.EXPR_ATTR);
                xslObj = new Variable(parent, name);
                if (exprAttr != null) 
                    xslObj.setAttribute(Names.EXPR_ATTR, exprAttr.getValue());
                copyAtts = false;
                break;
            }
            case XSLObject.PARAM_VARIABLE:
            {
                // since the name of a variable can't be changed,
                // we need to set it at creation time
                String name = element.getAttribute(Names.NAME_ATTR);
                Attr exprAttr = element.getAttributeNode(Names.EXPR_ATTR);
                xslObj = new ParamVariable(parent, name);
                if (exprAttr != null) 
                    xslObj.setAttribute(Names.EXPR_ATTR, exprAttr.getValue());
                copyAtts = false;
                break;
            }
            case XSLObject.PARAM:
            {
                // since the name of a variable can't be changed,
                // we need to set it at creation time
                String name = element.getAttribute(Names.NAME_ATTR);
                Attr exprAttr = element.getAttributeNode(Names.EXPR_ATTR);
                xslObj = new Param(parent, name);
                if (exprAttr != null) 
                    xslObj.setAttribute(Names.EXPR_ATTR, exprAttr.getValue());
                copyAtts = false;
                break;
            }
            case XSLObject.ENTITY_REF: //--proprietary
                processChildren = false;
            case XSLObject.ATTRIBUTE:
            case XSLObject.CHOOSE:
            case XSLObject.COMMENT:
            case XSLObject.COPY:
            case XSLObject.ELEMENT:
            case XSLObject.MESSAGE:
            case XSLObject.PI:
                xslObj = new XSLObject(parent,xslType);
                break;
            default: //* unsupported XSL element */
                xslObj = new XSLObject(parent, XSLObject.LITERAL);
                xslObj.setTypeName(tagName);
                break;
        }
        
        

        if (copyAtts) xslObj.copyAttributes(element);
        if (!processChildren) return xslObj;
            
        NodeList nl = element.getChildNodes();
        
        for (int i = 0; i < nl.getLength(); i++) {
            Node node = nl.item(i);
            switch (node.getNodeType()) {
                case Node.ELEMENT_NODE:
                    try {
                        XSLObject child = createXSLObject(
                            parent, (Element)node);
                        xslObj.appendAction(child);
                    }
                    catch(XSLException xslException) {
                        //-- handle error
                        errorWriter.println(xslException.getMessage());
                    }
                    break;
                 // Text Nodes
		        case Node.TEXT_NODE:
		        {
		            String text = ((Text)node).getData();
		            //-- set strip leading whitespace flag
		            boolean stripLWS = (node.getPreviousSibling() == null);
		            //-- set strip trailing whitespace flag
		            boolean stripTWS = (node.getNextSibling() == null);
		            text = Whitespace.stripSpace(text,stripLWS,stripTWS);
				    if (text.length() > 0)
			            xslObj.appendAction(new XSLText(parent,text));
			        break;
				}
		        // CDATA Sections
				case Node.CDATA_SECTION_NODE:
				{
				    String text = ((CDATASection)node).getData();
			        xslObj.appendAction(new XSLCData(parent,text));
			        break;
			    }
			    // Default
				default:
				    // Strip comment and PI nodes -- add later?
			        //xslObject.appendChild(copyNode(node, stylesheet, true));
			        break;
			} //-- select
        }
        xslObj.freeUnusedMemory(false);
        return xslObj;
    } //-- createXSLObject
    
    /**
     * Sets the Writer to print all errors to.
     * @param errWriter the Writer to use for error reporting
    **/
    public void setErrorStream(Writer errWriter) {
        this.errorWriter = new PrintWriter(errWriter, true);
    } //-- setErrorStream
    
    /**
     * Sets the PrintStream to print all errors to.
     * @param errStream the PrintStream to use for error reporting
    **/
    public void setErrorStream(PrintStream errStream) {
        this.errorWriter = new PrintWriter(errStream, true);
    } //-- setErrorStream
    
      //-------------------/
     //- Private Methods -/
    //-------------------/

    /**
     * Imports an XSL Stylesheet from the given XML Document
     * @param document the Stylesheet document to import from.
     * @param stylesheet the XSLStylesheet to import to.
     * @exception XSLException
    **/
	private void importStyle
	    (Document document, XSLStylesheet stylesheet, String filename) 
	    throws XSLException
	{
	    XSLStylesheet xsl = readStylesheet(document, filename);
		stylesheet.importFrom(xsl);
	} //-- importStylesheet

    /**
     * Imports an XSL stylesheet
     * @exception XSLException
    **/
    private void importStyle(String filename, XSLStylesheet stylesheet)
        throws XSLException {
        
        if (stylesheet == null) return;
        
        InputStream xslStream = null;
        filename = URIUtils.resolveHref(filename,stylesheet.getDocumentBase());
        // Make sure we aren't trying to import stylesheet already 
        // imported
        if (!stylesheet.isAllowableImport(filename)) {
            errorWriter.println("error including stylesheet: '" + filename + "'.");
            errorWriter.println(" - Stylesheet already directly or indirectly included.");
            errorWriter.println(" - continuing processing without the include.");
            return;
        }
        try {
            xslStream = URIUtils.getInputStream(filename,null);
        }
        catch(Exception ex) {
            errorWriter.println("error importing stylesheet: '" + filename + "'.");
            errorWriter.println(" - " + ex.getMessage());
            errorWriter.println(" - continuing processing without the import.");
            return;
        }
        XSLStylesheet xsl = readStylesheet(xslStream,filename);
        stylesheet.importWithoutVerify(xsl);
    } //-- importStylesheet

    /**
     * Includes an XSLStylesheet into the given XSLStylesheet
     * @param filename the name of the stylesheet to include 
     * @param stylesheet the XSLStylesheet to add the inclusion to
     * @exception XSLException
    **/
    private  void includeStyle
        (String filename, XSLStylesheet stylesheet) 
        throws XSLException 
    {
        if (stylesheet == null) return;
        
        InputStream xslStream = null;
        filename = URIUtils.resolveHref(filename,stylesheet.getDocumentBase());
        // Make sure we aren't trying to import stylesheet already 
        // imported
        if (!stylesheet.isAllowableImport(filename)) {
            errorWriter.println("error including stylesheet: '" + filename + "'.");
            errorWriter.println(" - Stylesheet already directly or indirectly included.");
            errorWriter.println(" - continuing processing without the include.");
            return;
        }
        try {
            xslStream = URIUtils.getInputStream(filename,null);
        }
        catch(Exception ex) {
            errorWriter.println("error importing stylesheet: '" + filename + "'.");
            errorWriter.println(" - " + ex.getMessage());
            errorWriter.println(" - continuing processing without the import.");
            try {
                xslStream.close();
            }
            catch(java.io.IOException ioException) {};
            return;
        }
        XSLStylesheet xsl = readStylesheet(xslStream,filename);
        try {
            xslStream.close();
        }
        catch(java.io.IOException ioException) {};
        stylesheet.includeFrom(xsl);
    } //-- include


    /**
     * Handles processing of the xsl:stylesheet element
    **/
    private void parseStylesheetElement
        (Element xmlElement, XSLStylesheet stylesheet) 
        throws XSLException
    {
        
        
        //-- look for XSL Namespace attribute
        //Properties xslProps = new Properties();
        String namespace = stylesheet.getXSLNSPrefix();
        //xslProps.put(XSLNS_PREFIX, namespace);
        
        NodeList nl = xmlElement.getChildNodes();

        for (int i = 0; i < nl.getLength(); i++) {

            Node node = nl.item(i);
            if (node.getNodeType() != Node.ELEMENT_NODE) continue;
            Element element = (Element)node;
            String tagName = element.getNodeName();
            
            XSLObject xslObject = createXSLObject(stylesheet, element);
            switch(xslObject.getType()) {
                
                // xsl:attribute-set
                case XSLObject.ATTRIBUTE_SET:
                    stylesheet.addAttributeSet((AttributeSet)xslObject);
                    break;
                // xsl:variable
                case XSLObject.VARIABLE:
                    try {
                        stylesheet.addVariable((Variable)xslObject);
                    }
                    catch(XSLException xslException) {
                        errorWriter.println(xslException.getMessage());
                        errorWriter.println(" -- processing without variable");
                    }
                    break;
                // xsl:import
                case XSLObject.IMPORT:
                    XSLImport xslImport = (XSLImport)xslObject;
                    try {
                        importStyle(xslImport.getHref(), stylesheet);
                    }
                    catch(XSLException xslException) {
                        errorWriter.println(xslException.getMessage());
                        errorWriter.println(" -- processing without import");
                    }
                    break;
                // xsl:include
                case XSLObject.INCLUDE:
                    String href = element.getAttribute(Names.HREF_ATTR);
                    try {
                        includeStyle(href, stylesheet);
                    }
                    catch(XSLException xslException) {
                        errorWriter.println(xslException);
                        errorWriter.println(" -- processing without include");
                    }
                    break;
                // <xsl:id>
                case XSLObject.ID:
                    stylesheet.addId((Id)xslObject);
                    break;
                // <xsl:output>
                case XSLObject.OUTPUT:
                    stylesheet.appendAction(xslObject);
                    break;
                //<xsl:preserve-space>
                //<xsl:strip-space>
                case XSLObject.PRESERVE_SPACE:
                case XSLObject.STRIP_SPACE:
                    stylesheet.appendAction(xslObject);
                    break;
                // xsl:template
                case XSLObject.TEMPLATE:
                    stylesheet.addTemplate((TemplateRule)xslObject);
                    break;
                // xsl:script  (proprietary)
                case XSLObject.SCRIPT:
                    stylesheet.addScript((XSLScript)xslObject);
                    break;
                default:
                    break;
                   
            } //-- switch
        } //-- for
    } //--
    
} //-- XSLReader
