/* 
 * Apple // emulator for Linux: X-Windows graphics support
 *
 * Copyright 1994 Alexander Jean-Claude Bottema
 * Copyright 1995 Stephen Lee
 * Copyright 1997, 1998 Aaron Culliney
 * Copyright 1998, 1999 Michael Deutschmann
 *
 * This software package is subject to the GNU General Public License
 * version 2 or later (your choice) as published by the Free Software 
 * Foundation.
 *
 * THERE ARE NO WARRANTIES WHATSOEVER. 
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <signal.h>
#include <string.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/shm.h>
#include <sys/time.h>

#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/keysym.h>
#include <X11/extensions/XShm.h>/* MITSHM! */


#include "video.h"
#include "apple2.h"
#include "misc.h"
#include "keys.h"

/* copied from svideo.c (didn't bother to rename!) */
static unsigned char	*svga_GM;
static unsigned char	vga_mem_page_0[SCANWIDTH*SCANHEIGHT];		/* page0 framebuffer */
static unsigned char	vga_mem_page_1[SCANWIDTH*SCANHEIGHT];		/* page1 framebuffer */
unsigned char	odd_colors[2] = {COLOR_LIGHT_PURPLE, COLOR_LIGHT_BLUE};
unsigned char	even_colors[2] = {COLOR_LIGHT_GREEN, COLOR_LIGHT_RED};

static Display *display;
static Window win;
static GC gc;
static unsigned int width, height;	/* window size */

static int screen_num;
static Visual* visual;
static XVisualInfo visualinfo;
static XColor colors[256];
XImage *image;
static Colormap cmap;
XEvent xevent;

int		doShm = 1;/* assume true */
XShmSegmentInfo	xshminfo;
int		xshmeventtype;



/* -------------------------------------------------------------------------
    c_setpage(p):    Set SVGA 64k page and update GM
                     (Graph Memory base address)
   ------------------------------------------------------------------------- */
void c_setpage(int p)
{
    svga_GM = image->data;
    GM = (crnt_visual_svga_page == p) ? svga_GM :
	((p == 0) ? vga_mem_page_0 : vga_mem_page_1);
    vmode_active = p;
}

/* -------------------------------------------------------------------------
    c_setscreen(p):    Switch to screen page p
   ------------------------------------------------------------------------- */
void c_setscreen(int m)
{
    if (m != crnt_visual_svga_page) {
	memcpy( ((crnt_visual_svga_page == 0) ?
		vga_mem_page_0 : vga_mem_page_1),
		svga_GM,
		SCANWIDTH*SCANHEIGHT
		);
	memcpy( svga_GM, ((m == 0) ? vga_mem_page_0 : vga_mem_page_1),
		SCANWIDTH*SCANHEIGHT
		);
	GM = (vmode_active == m) ? svga_GM :
	    ((vmode_active == 0) ?
	     vga_mem_page_0 : vga_mem_page_1);
	crnt_visual_svga_page = m;
    }
}


/*
 * XShm code influenced from the DOOM source code.
 * This tries to find some shared memory to use.  It checks for stale segments
 * (maybe due to an emulator crash) and tries to remove them or use them.
 */
static void getshm(int size) {
    int			key = ('a'<<24) | ('p'<<16) | ('p'<<8) | 'l';
    struct shmid_ds	shminfo;
    int			minsize = SCANWIDTH*SCANHEIGHT;
    int			id;
    int			rc;
    int			counter=5;

    /* Try to allocate shared memory segment.  Use stale segments if any are
     * found.
     */
    do
    {
	id = shmget((key_t) key, minsize, 0777);
	if (id != -1)
	{
	    /* we got someone else's ID. check if it's stale. */
	    printf("Found shared memory key=`%c%c%c%c', id=%d\n",
		    (key & 0xff000000)>>24,
		    (key & 0xff0000)>>16,
		    (key & 0xff00)>>8,
		    (key & 0xff),
		    id
		    );
	    rc=shmctl(id, IPC_STAT, &shminfo); /* get stats */
	    if (!rc) {
		/* someone's using the emulator */
		if (shminfo.shm_nattch) {
		    printf( "User uid=%d, key=`%c%c%c%c' appears to be running "
			    "the emulator.\n",
			    shminfo.shm_perm.cuid,
			    (key & 0xff000000)>>24,
			    (key & 0xff0000)>>16,
			    (key & 0xff00)>>8,
			    (key & 0xff)
			    );
		    ++key; /* increase the key count */
		}

		/* found a stale ID. */
		else {
		    /* it's my stale ID */
		    if (getuid() == shminfo.shm_perm.cuid) {
			rc = shmctl(id, IPC_RMID, 0);
			if (!rc)
			    printf("Was able to kill my old shared memory\n");
			else {
			    perror("shmctl");
			    printf("Was NOT able to kill my old shared memory\n");
			}

			id = shmget((key_t)key, size, IPC_CREAT|0777);
			if (id == -1) {
			    perror("shmget");
			    printf("Could not get shared memory\n");
			}

			rc=shmctl(id, IPC_STAT, &shminfo);
			if (rc) {
			    perror("shmctl");
			}

			break;

		    }
		    /* not my ID, but maybe we can use it */
		    if (size == shminfo.shm_segsz) {
			printf( "Will use stale shared memory of uid=%d\n",
				shminfo.shm_perm.cuid);
			break;
		    }
		    /* not my ID, and we can't use it */
		    else {
			printf( "Can't use stale shared memory belonging to uid=%d, "
				"key=`%c%c%c%c', id=%d\n",
				shminfo.shm_perm.cuid,
				(key & 0xff000000)>>24,
				(key & 0xff0000)>>16,
				(key & 0xff00)>>8,
				(key & 0xff),
				id
				);
			++key;
		    }
		}
	    }
	    else
	    {
		/* oops.  what to do now? */
		perror("shmctl");
		printf( "Could not get stats on key=`%c%c%c%c', id=%d\n",
			(key & 0xff000000)>>24,
			(key & 0xff0000)>>16,
			(key & 0xff00)>>8,
			(key & 0xff),
			id
			);
		++key;
	    }
	}
	else
	{
	    /* no stale ID's */
	    id = shmget((key_t)key, size, IPC_CREAT|0777);
	    if (id == -1) {
		perror("shmget");
		printf("Could not get shared memory\n");
		++key;
	    }
	    break;
	}
    } while (--counter);

    if (!counter)
    {
	printf( "System has too many stale/used "
		"shared memory segments!\n");
    }	

    xshminfo.shmid = id;

    /* attach to the shared memory segment */
    image->data = xshminfo.shmaddr = shmat(id, 0, 0);

    if (!image->data) {
	perror("shmat");
	printf("Could not attach to shared memory\n");
	exit(1);
    }

    printf( "Using shared memory key=`%c%c%c%c', id=%d, addr=0x%x\n",
	    (key & 0xff000000)>>24,
	    (key & 0xff0000)>>16,
	    (key & 0xff00)>>8,
	    (key & 0xff),
	    id,
	    (int) (image->data));
}


static void c_initialize_colors() {
    static unsigned char col2[ 3 ] = {255,255,255};
    static int		firstcall = 1;
    int			c,i,j;

    if (visualinfo.class == PseudoColor && visualinfo.depth == 8)
    {
	/* initialize the colormap */
	if (firstcall) {
	    firstcall = 0;
	    for (i=0; i<256; i++) {
		colors[i].pixel = i;
		colors[i].flags = DoRed|DoGreen|DoBlue;
	    }
	}

	/* align the palette for hires graphics */
	for (i = 0; i < 8; i++) {
	    for (j = 0; j < 3; j++) {
		c = (i & 1) ? col2[ j ] : 0;
		colors[ j+i*3+32].red = (c<<8) + c;
		c = (i & 2) ? col2[ j ] : 0;
		colors[ j+i*3+32].green = (c<<8) + c;
		c = (i & 4) ? col2[ j ] : 0;
		colors[ j+i*3+32].blue = (c<<8) + c;
	    }
	}
	colors[ COLOR_FLASHING_BLACK].red = 0;
	colors[ COLOR_FLASHING_BLACK].green = 0;
	colors[ COLOR_FLASHING_BLACK].blue = 0;

	colors[ COLOR_LIGHT_WHITE].red = (255<<8)|255;
	colors[ COLOR_LIGHT_WHITE].green = (255<<8)|255;
	colors[ COLOR_LIGHT_WHITE].blue = (255<<8)|255;

	colors[ COLOR_FLASHING_WHITE].red = (255<<8)|255;
	colors[ COLOR_FLASHING_WHITE].green = (255<<8)|255;
	colors[ COLOR_FLASHING_WHITE].blue = (255<<8)|255;

	/* dhires colors are bass-ackwards because it's easier for me to
	   deal with the endianness. */
	colors[0].red = 0;   colors[0].green = 0;   colors[0].blue = 0;
	colors[1].red = 0;   colors[1].green = 0;   colors[1].blue = 160;/*Magenta <med blue>*/
	colors[2].red = 0;   colors[2].green = 100; colors[2].blue = 0;/*Brown <dark green>*/
	colors[3].red = 28;  colors[3].green = 134; colors[3].blue = 255;/*Orange <dodger blue>*/
	colors[4].red = 165; colors[4].green = 21;  colors[4].blue = 10;/*Dark Green <brown>*/
	colors[5].red = 165; colors[5].green = 42;  colors[5].blue = 42;/*Gray2 <light gray>*/
	colors[6].red = 0;   colors[6].green = 255; colors[6].blue = 0;/*Green*/
	colors[7].red = 160; colors[7].green = 255; colors[7].blue = 160;/*Yellow <Aqua>*/
	colors[8].red = 112; colors[8].green = 192; colors[8].blue = 140;/*Dark Blue <magenta>*/
	colors[9].red = 105; colors[9].green = 205; colors[9].blue = 170;/*Purple <Purple>*/
	colors[0xa].red= 90; colors[0xa].green= 90; colors[0xa].blue = 90;/*Gray1 <dark gray>*/
	colors[0xb].red=131; colors[0xb].green=111; colors[0xb].blue = 250;/*Pink <slateblue>*/
	colors[0xc].red=255; colors[0xc].green=198; colors[0xc].blue = 203;/*Med Blue <Orange>*/
	colors[0xd].red=255; colors[0xd].green=167; colors[0xd].blue = 165;/*Light Blue <Pink>*/
	colors[0xe].red=255; colors[0xe].green=255; colors[0xe].blue = 0;/*Aqua <Yellow>*/
	colors[0xf].red=255; colors[0xf].green=255; colors[0xf].blue = 255;/*White*/

	/* lores colors (<<4) */
	colors[0x00].red=0;	colors[0x00].green=0;   colors[0x00].blue = 0;/*Black*/
	colors[0x10].red=165;	colors[0x10].green=42;  colors[0x10].blue = 42;/*Magenta*/
	colors[0x20].red=0;	colors[0x20].green=0;   colors[0x20].blue = 160;/*Dark Blue*/
	colors[0x30].red=138;	colors[0x30].green=0;   colors[0x30].blue = 138;/*Purple*/
	colors[0x40].red=0;	colors[0x40].green=100; colors[0x40].blue = 0;/*Dark Green*/
	colors[0x50].red=128;	colors[0x50].green=128;	colors[0x50].blue = 128;/*Gray*/
	colors[0x60].red=28;	colors[0x60].green=134; colors[0x60].blue = 255;/*Med Blue*/
	colors[0x70].red=65;	colors[0x70].green=255; colors[0x70].blue = 255;/*Light Blue*/
	colors[0x80].red=138;	colors[0x80].green=138; colors[0x80].blue = 0;/*Brown*/
	colors[0x90].red=255;	colors[0x90].green= 66; colors[0x90].blue = 0;/*Orange*/
	colors[0xa0].red=90;	colors[0xa0].green= 90;	colors[0xa0].blue = 90;/*Gray2*/
	colors[0xb0].red=255;	colors[0xb0].green=198; colors[0xb0].blue = 203;/*Pink*/
	colors[0xc0].red=0;	colors[0xc0].green=255; colors[0xc0].blue = 0;/*Green*/
	colors[0xd0].red=255;	colors[0xd0].green=255; colors[0xd0].blue = 0;/*Yello*/
	colors[0xe0].red=160;	colors[0xe0].green=255; colors[0xe0].blue = 160;/*Aqua*/
	colors[0xf0].red=255;	colors[0xf0].green=255; colors[0xf0].blue = 255;/*White*/

	for (i=0; i<16; i++) {
	    colors[i].red = (colors[i].red<<8) | colors[i].red;
	    colors[i].green = (colors[i].green<<8) | colors[i].green;
	    colors[i].blue = (colors[i].blue<<8) | colors[i].blue;

	    colors[i<<4].red = (colors[i<<4].red<<8) | colors[i<<4].red;
	    colors[i<<4].green = (colors[i<<4].green<<8) | colors[i<<4].green;
	    colors[i<<4].blue = (colors[i<<4].blue<<8) | colors[i<<4].blue;
	}
	// store the colors to the current colormap
	XStoreColors(display, cmap, colors, 256);
    }
}


/* HACK: this is incredibly broken.
 * Map the X keysyms back into scancodes so the routines in keys.c can deal
 * with it.  We do this to be compatible with what the SVGAlib version does.
 */
static int keysym_to_scancode(int doRaw) {
    static char buffer;
    static KeySym keysym;
    static XComposeStatus compose;
    static int rc = 0xFF;

    switch(rc = XKeycodeToKeysym(display, xevent.xkey.keycode, 0))
    {
	case XK_F1:
	    rc = 59; break;
	case XK_F2:
	    rc = 60; break;
	case XK_F3:
	    rc = 61; break;
	case XK_F4:
	    rc = 62; break;
	case XK_F5:
	    rc = 63; break;
	case XK_F6:
	    rc = 64; break;
	case XK_F7:
	    rc = 65; break;
	case XK_F8:
	    rc = 66; break;
	case XK_F9:
	    rc = 67; break;
	case XK_F10:
	    rc = 68; break;
	case XK_Left:
	    rc = doRaw ? 105 : kLEFT; break;
	case XK_Right:
	    rc = doRaw ? 106 : kRIGHT; break;
	case XK_Down:
	    rc = doRaw ? 108 : kDOWN; break;
	case XK_Up:
	    rc = doRaw ? 103 : kUP; break;
	case XK_Escape:
	    rc = doRaw ? 1   : kESC; break;
	case XK_Return:
	    rc = doRaw ? 28  : 13; break;
	case XK_Tab:
	    rc = doRaw ? 15  : 9; break;
	case XK_Shift_L:
	    rc = doRaw ? SCODE_L_SHIFT : -1; break;
	case XK_Shift_R:
	    rc = doRaw ? SCODE_R_SHIFT : -1; break;
	case XK_Control_L:
	    rc = SCODE_L_CTRL; break;
	case XK_Control_R:
	    rc = SCODE_R_CTRL; break;
	case XK_Caps_Lock:
	    rc = SCODE_CAPS; break;
	case XK_BackSpace:
	    rc = doRaw ? 14 : 8; break;
	case XK_Insert: /* HACK - acts as DEL */
	    rc = 110; break;
	case XK_Pause:
	    rc = 119; break;
	case XK_Print:
	    rc = 99; break;
	case XK_Delete: /* raw joystick button 0 */
	    rc = 111; break;
	case XK_End: /* raw joystick button 1 */
	    rc = doRaw ? 107 : kEND; break;
	case XK_Home:
	    rc = kHOME; break;
	case XK_Page_Down: /* raw joystick button 2 */
	    rc = doRaw ? 109 : kPGDN; break;
	case XK_Page_Up:
	    rc = kPGUP; break;

	// keypad joystick movement 
	case XK_KP_5:
	case XK_KP_Begin:
	    rc = SCODE_J_C; break;
	case XK_KP_4:
	case XK_KP_Left:
	    rc = SCODE_J_L; break;
	case XK_KP_8:
	case XK_KP_Up:
	    rc = SCODE_J_U; break;
	case XK_KP_6:
	case XK_KP_Right:
	    rc = SCODE_J_R; break;
	case XK_KP_2:
	case XK_KP_Down:
	    rc = SCODE_J_D; break;


	default:
	    if (doRaw && (rc >= XK_space) && (rc <= XK_asciitilde))
		rc = xevent.xkey.keycode - 8;
	    else if (!doRaw && (XLookupString(
		    (XKeyEvent*)&xevent, &buffer, 1, &keysym, &compose) == 1))
		rc = (int)keysym;
	    break;
    }

    return rc & 0xFF;/* normalize */
}

static void post_image() {
    if (doShm) {
	if (!XShmPutImage(
		display,
		win,
		gc,
		image,
		0, 0,
		0, 0,
		SCANWIDTH, SCANHEIGHT,
		True))
	    fprintf(stderr, "XShmPutImage() failed\n");
    } else {
	if (XPutImage(
		display,
		win,
		gc,
		image,
		0, 0,
		0, 0,
		SCANWIDTH, SCANHEIGHT
		))
	    fprintf(stderr, "XPutImage() failed\n");
    }
}

static void c_poll_keyboard_input() {
    c_keyboard_off();
    // post the image and loop waiting for it to finish and
    // also process other input events
    post_image();
LOOP:
	if (doShm)
	    XNextEvent(
		    display,
		    &xevent);
	else if (!XCheckMaskEvent(
		display,
		KeyPressMask|KeyReleaseMask,
		&xevent))
	    goto POLL_FINISHED;
	switch (xevent.type) {
	    case KeyPress:
		c_read_raw_key(keysym_to_scancode(/*raw*/1), 1);
		break;
	    case KeyRelease:
		c_read_raw_key(keysym_to_scancode(/*raw*/1), 0);
		break;
	    default:
		if (xevent.type == xshmeventtype)
		    goto POLL_FINISHED;
		break;
	}
	goto LOOP;

POLL_FINISHED:
    c_periodic_update();
    signal(SIGVTALRM, c_poll_keyboard_input);/* reinstall */
    c_keyboard_on();
}

/* -------------------------------------------------------------------------
    c_initialize_keyboard()
   ------------------------------------------------------------------------- */

static void c_initialize_keyboard()
{
    signal( SIGVTALRM, c_poll_keyboard_input );
    c_keyboard_on();
}

void c_keyboard_on() {
    static struct itimerval tval, old;

    tval.it_interval.tv_sec = 0L;
    tval.it_interval.tv_usec = TIMER_DELAY;
    tval.it_value.tv_sec = 0L;
    tval.it_value.tv_usec = TIMER_DELAY;

    setitimer( ITIMER_VIRTUAL, &tval, &old );/*set up virtual timer*/
}

void c_keyboard_off() {
    static struct itimerval tval, old;

    tval.it_interval.tv_sec = 0L;
    tval.it_interval.tv_usec = 0L;
    tval.it_value.tv_sec = 0L;
    tval.it_value.tv_usec = 0L;

    setitimer( ITIMER_VIRTUAL, &tval, &old );
}

static Cursor hidecursor() {
    Pixmap cursormask;
    XGCValues xgc;
    XColor dummycolour;
    Cursor cursor;

    cursormask = XCreatePixmap(display, win, 1, 1, 1/*depth*/);
    xgc.function = GXclear;
    gc =  XCreateGC(display, cursormask, GCFunction, &xgc);
    XFillRectangle(display, cursormask, gc, 0, 0, 1, 1);
    dummycolour.pixel = 0;
    dummycolour.red = 0;
    dummycolour.flags = 04;
    cursor = XCreatePixmapCursor(display, cursormask, cursormask,
	    &dummycolour,&dummycolour, 0,0);
    XFreePixmap(display,cursormask);
    XFreeGC(display,gc);
    return cursor;
}

static void parseArgs() {
    int i;
    for (i=0; i<argc; i++) {
	if (strstr(argv[i], "-noshm"))
	    doShm=0;
    }
}

void c_initialize_video() {
	XSetWindowAttributes attribs;
	unsigned long	attribmask;
	int x, y; 	/* window position */
	int drawingok;
	unsigned int display_width, display_height;
	XGCValues xgcvalues;
	int valuemask;
	char *window_name = "Apple ][";
	char *icon_name = window_name;
	XSizeHints *size_hints;
	XWMHints *wm_hints;
	XClassHint *class_hints;
	XTextProperty windowName, iconName;
	GC gc;
	char *progname;/* name this program was invoked by */
	char *displayname = NULL;

	progname = argv[0];

	/* give up root privileges. equivalent of vga_init() */
	seteuid(getuid());
	setegid(getgid());

	parseArgs();

	if (!(size_hints = XAllocSizeHints())) {
	    fprintf(stderr, "cannot allocate memory for SizeHints\n");
	    exit(1);
	}
	if (!(wm_hints = XAllocWMHints())) {
	    fprintf(stderr, "cannot allocate memory for WMHints\n");
	    exit(1);
	}
	if (!(class_hints = XAllocClassHint())) {
	    fprintf(stderr, "cannot allocate memory for ClassHints\n");
	    exit(1);
	}

	/* connect to X server */
	if ( (display=XOpenDisplay(displayname)) == NULL )
	{
		fprintf(stderr, "cannot connect to X server \"%s\"\n", 
			XDisplayName(displayname));
		exit(1);
	}

	/* get screen size from display structure macro */
	screen_num = DefaultScreen(display);
	if (!XMatchVisualInfo(display, screen_num, 8, PseudoColor, &visualinfo)) {
	    fprintf(stderr,
		    "Sorry bud, xapple2 only supports "
		    "8bit PseudoColor displays.\n"
		    "Maybe you can fix this!\n");
	    exit(1);
	}
	visual = visualinfo.visual;
	display_width = DisplayWidth(display, screen_num);
	display_height = DisplayHeight(display, screen_num);

	/* Note that in a real application, x and y would default to 0
	 * but would be settable from the command line or resource database.  
	 */
	x = y = 0;
	width = SCANWIDTH, height = SCANHEIGHT;

	/* init MITSHM if we're doing it */
	if (doShm) {
	    /* make sure we have it */
	    doShm = XShmQueryExtension(display);

	    /* and it's a local connection */
	    if (!displayname)
		displayname = getenv("DISPLAY");
	    if (displayname)
	    {
		if ((*displayname != ':') || !doShm) {
		    printf("Cannot run MITSHM version of emulator "
			   "with display \"%s\"\n"
			   "Try something like \":0.0\".  "
			   "Reverting to regular X with no sound.\n",
			   displayname);
		    doShm = 0;
		    soundAllowed=0;
		    c_initialize_sound();
		}
	    }
	} else {
	    /* and it's a local connection */
	    if (!displayname)
		displayname = getenv("DISPLAY");
	    if (displayname)
	    {
		if (*displayname != ':') {
		    printf("Sound not allowed for remote display \"%s\".\n",
			   displayname);
		    soundAllowed=0;
		    c_initialize_sound();
		}
	    }
	}

	/* initialize colors */
	cmap = XCreateColormap(display, RootWindow(display, screen_num),
		visual, AllocAll);
	c_initialize_colors();
	attribs.colormap = cmap;
	attribs.border_pixel = 0;

	/* select event types wanted */
	attribmask = CWEventMask | CWColormap | CWBorderPixel;/* HACK CWBorderPixel? */
	attribs.event_mask =
	    KeyPressMask
	    | KeyReleaseMask
	    | ExposureMask;

	/* create opaque window */
	win = XCreateWindow(display, RootWindow(display,screen_num), 
			x, y, width, height,
			0,/* border_width */
			8,/* depth */
			InputOutput,
			visual,
			attribmask,
			&attribs);


	/* set size hints for window manager.  We don't want the user to
	 * dynamically allocate window size since we won't do the right
	 * scaling in response.  Whaddya want, performance or a snazzy gui?
	 */
	size_hints->flags = PPosition | PSize | PMinSize | PMaxSize;
	size_hints->min_width = width;
	size_hints->min_height = height;
	size_hints->max_width = width;
	size_hints->max_height = height;

	/* store window_name and icon_name for niceity. */
	if (XStringListToTextProperty(&window_name, 1, &windowName) == 0) {
	    fprintf(stderr, "structure allocation for windowName failed.\n");
	    exit(1);
	}
		
	if (XStringListToTextProperty(&icon_name, 1, &iconName) == 0) {
	    fprintf(stderr, "structure allocation for iconName failed.\n");
	    exit(1);
	}

	wm_hints->initial_state = NormalState;
	wm_hints->input = True;
	wm_hints->flags = StateHint | IconPixmapHint/* | InputHint*/;

	class_hints->res_name = progname;
	class_hints->res_class = "Apple2";

	XSetWMProperties(display, win, &windowName, &iconName, 
			argv, argc, size_hints, wm_hints, 
			class_hints);

	XDefineCursor(display, win, hidecursor(display, win));

	/* create the GC */
	valuemask = GCGraphicsExposures;
	xgcvalues.graphics_exposures = False;
	gc = XCreateGC(
		display,
		win,
		valuemask,
		&xgcvalues);

	/* display window */
	XMapWindow(display, win);

	/* wait until it is OK to draw */
	drawingok = 0;
	while (!drawingok)
	{
	    XNextEvent(display, &xevent);
	    if ((xevent.type == Expose) && !xevent.xexpose.count)
	    {
		drawingok = 1;
	    }
	}

	/* safety */
	signal(SIGINT, c_shutdown_video);
	signal(SIGHUP, c_shutdown_video);
	signal(SIGTERM, c_shutdown_video);

	xshmeventtype = XShmGetEventBase(display) + ShmCompletion;

	/* create the image */
	if (doShm) {
	    image = XShmCreateImage(
		    display,
		    visual,
		    8,
		    ZPixmap,
		    0,
		    &xshminfo,
		    SCANWIDTH,
		    SCANHEIGHT);

	    printf("Allocating shared memory %dx%d region\n",
		    image->bytes_per_line, image->height);

	    getshm(image->bytes_per_line * image->height);

	    /* get the X server to attach to it */
	    if (!XShmAttach(display, &xshminfo)) {
		fprintf(stderr, "XShmAttach() failed in InitGraphics()\n");
		exit(1);
	    }
	} else {
	    char *data = malloc(SCANWIDTH*SCANHEIGHT*sizeof(unsigned char));
	    if (!data) {
		fprintf(stderr, "no memory for image data!\n");
		exit(1);
	    }
	    printf("Creating regular XImage\n");
	    image = XCreateImage(
		    display,
		    visual,
		    8,
		    ZPixmap,
		    0,
		    data,
		    SCANWIDTH,
		    SCANHEIGHT,
		    8/*bitmap_pad*/,
		    SCANWIDTH/*bytes_per_line*/);
	}

	/* finally, init the keyboard */
	c_initialize_keyboard();
}

void c_shutdown_video(void)
{
    if (doShm) {
	// Detach from X server
	if (!XShmDetach(display, &xshminfo))
	    fprintf(stderr,"XShmDetach() failed in c_shutdown_video()\n");

	// Release shared memory.
	shmdt(xshminfo.shmaddr);
	shmctl(xshminfo.shmid, IPC_RMID, 0);

	// Paranoia.
	image->data = NULL;
    } else {
	free(image->data);
    }

    exit(0);
}

#define my_pixel(x, y, c) GM[(y)*SCANWIDTH+(x)]=(c)
void c_clear_video() {
    int		x,y,i;

    c_setpage( 1 );
    for (i=0; i<2; i++) {
	c_setpage(i);
	for (y = 0; y < SCANHEIGHT; y++)
	    for (x = 0; x < SCANWIDTH; x++)
		my_pixel( x, y, 0 );
    }
}

/* -------------------------------------------------------------------------
 * c_mygetch():
 * 	get next menu character
 * 	block:	1=block getting input, 0=don't block (this is called by the
 * 		debugger routines when stepping through code to check for a
 * 		user-input break)
 * 	return: -1 (no input) or input character
 * ------------------------------------------------------------------------- */

int c_mygetch(int block) {
    int c = -1, posts = 0;

    // post the image and wait for it to finish and
    // also process other input events
    if (block) {
	post_image();
	if (doShm)
	    ++posts;
    }
    do
    {
	if (block/* && doShm*/)
	    XNextEvent(display, &xevent);
	else if (!XCheckMaskEvent(
		display,
		KeyPressMask/*|KeyReleaseMask*/,
		&xevent))
	    return c;
	switch (xevent.type) {
	    case KeyPress:
		c = keysym_to_scancode(/*raw*/0);
		break;
	    case Expose:
		post_image();
		if (doShm)
		    ++posts;
		break;
	    default:
		if (xevent.type == xshmeventtype) {
		    --posts;
		    break;
		}
		break;
	}
    } while (posts || (block && (c==-1)));
    return c;
}

void c_flash_cursor(int on) {
    // flash only if it's text or mixed modes.
    if (vmode_text || vmode_mixed) {
	if (!on) {
	    colors[ COLOR_FLASHING_BLACK].red = 0;
	    colors[ COLOR_FLASHING_BLACK].green = 0;
	    colors[ COLOR_FLASHING_BLACK].blue = 0;

	    colors[ COLOR_FLASHING_WHITE].red = 0xffff;
	    colors[ COLOR_FLASHING_WHITE].green = 0xffff;
	    colors[ COLOR_FLASHING_WHITE].blue = 0xffff;
	} else {
	    colors[ COLOR_FLASHING_WHITE].red = 0;
	    colors[ COLOR_FLASHING_WHITE].green = 0;
	    colors[ COLOR_FLASHING_WHITE].blue = 0;

	    colors[ COLOR_FLASHING_BLACK].red = 0xffff;
	    colors[ COLOR_FLASHING_BLACK].green = 0xffff;
	    colors[ COLOR_FLASHING_BLACK].blue = 0xffff;
	}

	// store the colors to the current colormap
	XStoreColors(display, cmap, colors, 256);
    }
}

void c_video_refresh() {
    int posts = 0;

    // post the image and wait for it to finish and
    // also process other input events
    post_image();
    if (!doShm)
	return;
    ++posts;
    do
    {
	XNextEvent(display, &xevent);
	switch (xevent.type) {
	    case Expose:
		post_image();
		if (doShm)
		    ++posts;
		break;
	    default:
		if (xevent.type == xshmeventtype)
		    --posts;
		break;
	}
    } while (posts);
}
